/*
 * Decompiled with CFR 0.152.
 */
package org.apache.flink.runtime.jobmaster.event;

import java.io.DataInputStream;
import java.io.EOFException;
import java.io.IOException;
import java.io.InputStream;
import java.time.Duration;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ThreadFactory;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;
import org.apache.flink.annotation.VisibleForTesting;
import org.apache.flink.api.common.JobID;
import org.apache.flink.configuration.Configuration;
import org.apache.flink.configuration.JobEventStoreOptions;
import org.apache.flink.configuration.MemorySize;
import org.apache.flink.core.fs.FileSystem;
import org.apache.flink.core.fs.Path;
import org.apache.flink.core.io.SimpleVersionedSerializer;
import org.apache.flink.runtime.highavailability.HighAvailabilityServicesUtils;
import org.apache.flink.runtime.jobmaster.event.ExecutionJobVertexInitializedEvent;
import org.apache.flink.runtime.jobmaster.event.ExecutionVertexFinishedEvent;
import org.apache.flink.runtime.jobmaster.event.ExecutionVertexResetEvent;
import org.apache.flink.runtime.jobmaster.event.FsBatchFlushOutputStream;
import org.apache.flink.runtime.jobmaster.event.GenericJobEventSerializer;
import org.apache.flink.runtime.jobmaster.event.JobEvent;
import org.apache.flink.runtime.jobmaster.event.JobEventStore;
import org.apache.flink.runtime.jobmaster.event.JobEvents;
import org.apache.flink.util.Preconditions;
import org.apache.flink.util.concurrent.ExecutorThreadFactory;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class FileSystemJobEventStore
implements JobEventStore {
    private static final Logger LOG = LoggerFactory.getLogger(FileSystemJobEventStore.class);
    private static final String FILE_PREFIX = "events.";
    private static final int INITIAL_FILE_INDEX = 0;
    private final FileSystem fileSystem;
    private final Path workingDir;
    private FsBatchFlushOutputStream outputStream;
    private Path writeFile;
    private int writeIndex;
    private ScheduledExecutorService eventWriterExecutor;
    private DataInputStream inputStream;
    private int readIndex;
    private List<Path> readFiles;
    private volatile boolean corrupted = false;
    private final long flushIntervalInMs;
    private final int writeBufferSize;
    private final Map<Integer, SimpleVersionedSerializer<JobEvent>> jobEventSerializers = new HashMap<Integer, SimpleVersionedSerializer<JobEvent>>();

    public FileSystemJobEventStore(JobID jobID, Configuration configuration) throws IOException {
        this(new Path(HighAvailabilityServicesUtils.getClusterHighAvailableStoragePath(configuration), jobID.toString() + "/job-events"), configuration);
    }

    @VisibleForTesting
    public FileSystemJobEventStore(Path workingDir, Configuration configuration) throws IOException {
        this.workingDir = (Path)Preconditions.checkNotNull((Object)workingDir);
        this.fileSystem = workingDir.getFileSystem();
        this.flushIntervalInMs = ((Duration)configuration.get(JobEventStoreOptions.FLUSH_INTERVAL)).toMillis();
        this.writeBufferSize = (int)((MemorySize)configuration.get(JobEventStoreOptions.WRITE_BUFFER_SIZE)).getBytes();
        this.registerJobEventSerializers();
    }

    void registerJobEventSerializer(int eventType, SimpleVersionedSerializer<JobEvent> simpleVersionedSerializer) {
        Preconditions.checkState((!this.jobEventSerializers.containsKey(eventType) ? 1 : 0) != 0);
        this.jobEventSerializers.put(eventType, simpleVersionedSerializer);
    }

    private void registerJobEventSerializers() {
        this.registerJobEventSerializer(JobEvents.getTypeID(ExecutionVertexFinishedEvent.class), new ExecutionVertexFinishedEvent.Serializer());
        this.registerJobEventSerializer(JobEvents.getTypeID(ExecutionVertexResetEvent.class), new GenericJobEventSerializer());
        this.registerJobEventSerializer(JobEvents.getTypeID(ExecutionJobVertexInitializedEvent.class), new GenericJobEventSerializer());
    }

    @VisibleForTesting
    Path getWorkingDir() {
        return this.workingDir;
    }

    @VisibleForTesting
    ScheduledExecutorService getEventWriterExecutor() {
        return this.eventWriterExecutor;
    }

    @VisibleForTesting
    FsBatchFlushOutputStream getOutputStream() {
        return this.outputStream;
    }

    @Override
    public void start() throws IOException {
        if (!this.fileSystem.exists(this.workingDir)) {
            this.fileSystem.mkdirs(this.workingDir);
            LOG.info("Create job event dir {}.", (Object)this.workingDir);
        }
        try {
            this.readIndex = 0;
            this.readFiles = this.getAllJobEventFiles();
        }
        catch (IOException e) {
            throw new IOException("Cannot init filesystem job event store.", e);
        }
        this.writeIndex = this.readFiles.size();
        this.eventWriterExecutor = this.createJobEventWriterExecutor();
        this.eventWriterExecutor.scheduleAtFixedRate(() -> {
            if (this.outputStream != null && !this.corrupted) {
                try {
                    this.outputStream.flush();
                }
                catch (Exception e) {
                    LOG.warn("Error happens when flushing event file {}. Do not record events any more.", (Object)this.writeFile, (Object)e);
                    this.corrupted = true;
                    this.closeOutputStream();
                }
            }
        }, 0L, this.flushIntervalInMs, TimeUnit.MILLISECONDS);
        this.corrupted = false;
    }

    private List<Path> getAllJobEventFiles() throws IOException {
        ArrayList<Path> paths = new ArrayList<Path>();
        int index = 0;
        Set fileNames = Arrays.stream(this.fileSystem.listStatus(this.workingDir)).map(fileStatus -> fileStatus.getPath().getName()).filter(name -> name.startsWith(FILE_PREFIX)).collect(Collectors.toSet());
        while (fileNames.contains(FILE_PREFIX + index)) {
            paths.add(new Path(this.workingDir, FILE_PREFIX + index));
            ++index;
        }
        return paths;
    }

    protected ScheduledExecutorService createJobEventWriterExecutor() {
        return Executors.newSingleThreadScheduledExecutor((ThreadFactory)new ExecutorThreadFactory("job-event-writer"));
    }

    @Override
    public void stop(boolean clearEventLogs) {
        try {
            this.eventWriterExecutor.execute(this::closeOutputStream);
            this.closeInputStream();
            if (this.eventWriterExecutor != null) {
                this.eventWriterExecutor.shutdown();
                try {
                    if (!this.eventWriterExecutor.awaitTermination(10L, TimeUnit.SECONDS)) {
                        this.eventWriterExecutor.shutdownNow();
                    }
                }
                catch (InterruptedException ignored) {
                    this.eventWriterExecutor.shutdownNow();
                }
                this.eventWriterExecutor = null;
            }
            if (clearEventLogs) {
                this.fileSystem.delete(this.workingDir, true);
            }
        }
        catch (Exception exception) {
            LOG.warn("Fail to stop filesystem job event store.", (Throwable)exception);
        }
    }

    @Override
    public void writeEvent(JobEvent event, boolean cutBlock) {
        Preconditions.checkNotNull((Object)this.fileSystem);
        Preconditions.checkNotNull((Object)event);
        this.eventWriterExecutor.execute(() -> this.writeEventRunnable(event, cutBlock));
    }

    @VisibleForTesting
    protected void writeEventRunnable(JobEvent event, boolean cutBlock) {
        if (this.corrupted) {
            if (LOG.isDebugEnabled()) {
                LOG.debug("Skip job event {} because write corrupted.", (Object)event);
            }
            return;
        }
        try {
            if (this.outputStream == null) {
                this.openNewOutputStream();
            }
            SimpleVersionedSerializer serializer = (SimpleVersionedSerializer)Preconditions.checkNotNull(this.jobEventSerializers.get(event.getType()), (String)("There is no registered serializer for job event with type " + event.getType()));
            byte[] binaryEvent = serializer.serialize((Object)event);
            this.outputStream.writeInt(event.getType());
            this.outputStream.writeInt(binaryEvent.length);
            this.outputStream.write(binaryEvent);
            if (LOG.isDebugEnabled()) {
                LOG.debug("Write job event {}.", (Object)event);
            }
            if (cutBlock) {
                this.closeOutputStream();
            }
        }
        catch (Throwable throwable) {
            LOG.warn("Error happens when writing event {} into {}. Do not record events any more.", new Object[]{event, this.writeFile, throwable});
            this.corrupted = true;
            this.closeOutputStream();
        }
    }

    @VisibleForTesting
    void writeEvent(JobEvent event) {
        this.writeEvent(event, false);
    }

    @Override
    public JobEvent readEvent() throws Exception {
        try {
            JobEvent event = null;
            while (event == null) {
                try {
                    if (this.inputStream == null && this.tryGetNewInputStream() == null) {
                        return null;
                    }
                    int binaryEventType = this.inputStream.readInt();
                    int binaryEventSize = this.inputStream.readInt();
                    byte[] binaryEvent = new byte[binaryEventSize];
                    this.inputStream.readFully(binaryEvent);
                    SimpleVersionedSerializer serializer = (SimpleVersionedSerializer)Preconditions.checkNotNull(this.jobEventSerializers.get(binaryEventType), (String)("There is no registered serializer for job event with type " + binaryEventType));
                    event = (JobEvent)serializer.deserialize(GenericJobEventSerializer.INSTANCE.getVersion(), binaryEvent);
                }
                catch (EOFException eof) {
                    this.closeInputStream();
                }
            }
            return event;
        }
        catch (Exception e) {
            throw new IOException("Cannot read next event from event store.", e);
        }
    }

    @Override
    public boolean isEmpty() throws Exception {
        return !this.fileSystem.exists(this.workingDir) || this.fileSystem.listStatus(this.workingDir).length == 0;
    }

    private DataInputStream tryGetNewInputStream() throws IOException {
        if (this.inputStream == null && this.readIndex < this.readFiles.size()) {
            Path file = this.readFiles.get(this.readIndex++);
            this.inputStream = new DataInputStream((InputStream)this.fileSystem.open(file));
            LOG.info("Start reading job event file {}", (Object)file.getPath());
        }
        return this.inputStream;
    }

    private void openNewOutputStream() throws IOException {
        this.writeFile = new Path(this.workingDir, FILE_PREFIX + this.writeIndex);
        this.outputStream = new FsBatchFlushOutputStream(this.fileSystem, this.writeFile, FileSystem.WriteMode.NO_OVERWRITE, this.writeBufferSize);
        LOG.info("Job events will be written to {}.", (Object)this.writeFile);
        ++this.writeIndex;
    }

    @VisibleForTesting
    void closeOutputStream() {
        if (this.outputStream != null) {
            try {
                this.outputStream.close();
            }
            catch (IOException exception) {
                LOG.warn("Error happens when closing the output stream for {}. Do not record events any more.", (Object)this.writeFile, (Object)exception);
                this.corrupted = true;
            }
            finally {
                this.outputStream = null;
            }
        }
    }

    private void closeInputStream() throws IOException {
        if (this.inputStream != null) {
            this.inputStream.close();
            this.inputStream = null;
        }
    }
}

