/*
 * Decompiled with CFR 0.152.
 */
package software.amazon.jdbc;

import java.sql.SQLException;
import java.util.ArrayList;
import java.util.Comparator;
import java.util.HashMap;
import java.util.List;
import java.util.Properties;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.locks.ReentrantLock;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.stream.Collectors;
import org.checkerframework.checker.nullness.qual.NonNull;
import org.checkerframework.checker.nullness.qual.Nullable;
import software.amazon.jdbc.AwsWrapperProperty;
import software.amazon.jdbc.HostRole;
import software.amazon.jdbc.HostSelector;
import software.amazon.jdbc.HostSpec;
import software.amazon.jdbc.PropertyDefinition;
import software.amazon.jdbc.hostavailability.HostAvailability;
import software.amazon.jdbc.util.Messages;
import software.amazon.jdbc.util.StringUtils;
import software.amazon.jdbc.util.storage.CacheMap;

public class RoundRobinHostSelector
implements HostSelector {
    public static final AwsWrapperProperty ROUND_ROBIN_HOST_WEIGHT_PAIRS = new AwsWrapperProperty("roundRobinHostWeightPairs", null, "Comma separated list of database host-weight pairs in the format of `<host>:<weight>`.");
    public static final AwsWrapperProperty ROUND_ROBIN_DEFAULT_WEIGHT = new AwsWrapperProperty("roundRobinDefaultWeight", "1", "The default weight for any hosts that have not been configured with the `roundRobinHostWeightPairs` parameter.");
    public static final String STRATEGY_ROUND_ROBIN = "roundRobin";
    private static final int DEFAULT_WEIGHT = 1;
    private static final long DEFAULT_ROUND_ROBIN_CACHE_EXPIRE_NANO = TimeUnit.MINUTES.toNanos(10L);
    static final Pattern HOST_WEIGHT_PAIRS_PATTERN = Pattern.compile("((?<host>[^:/?#]*):(?<weight>[0-9]*))");
    protected static final CacheMap<String, RoundRobinClusterInfo> roundRobinCache = new CacheMap();
    protected static final ReentrantLock lock = new ReentrantLock();

    public static void setRoundRobinHostWeightPairsProperty(@NonNull Properties properties, @NonNull List<HostSpec> hosts) {
        StringBuilder builder = new StringBuilder();
        for (int i = 0; i < hosts.size(); ++i) {
            builder.append(hosts.get(i).getHostId()).append(":").append(hosts.get(i).getWeight());
            if (i >= hosts.size() - 1) continue;
            builder.append(",");
        }
        String roundRobinHostWeightPairsString = builder.toString();
        properties.setProperty(RoundRobinHostSelector.ROUND_ROBIN_HOST_WEIGHT_PAIRS.name, roundRobinHostWeightPairsString);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public HostSpec getHost(@NonNull List<HostSpec> hosts, @NonNull HostRole role, @Nullable Properties props) throws SQLException {
        lock.lock();
        try {
            int targetHostIndex;
            List<HostSpec> eligibleHosts = hosts.stream().filter(hostSpec -> role.equals((Object)hostSpec.getRole()) && hostSpec.getAvailability().equals((Object)HostAvailability.AVAILABLE)).sorted(Comparator.comparing(HostSpec::getHost)).collect(Collectors.toList());
            if (eligibleHosts.isEmpty()) {
                throw new SQLException(Messages.get("HostSelector.noHostsMatchingRole", new Object[]{role}));
            }
            this.createCacheEntryForHosts(eligibleHosts, props);
            String currentClusterInfoKey = eligibleHosts.get(0).getHost();
            RoundRobinClusterInfo clusterInfo = roundRobinCache.get(currentClusterInfoKey);
            HostSpec lastHost = clusterInfo.lastHost;
            int lastHostIndex = -1;
            if (lastHost != null) {
                for (int i = 0; i < eligibleHosts.size(); ++i) {
                    if (!eligibleHosts.get(i).getHost().equals(lastHost.getHost())) continue;
                    lastHostIndex = i;
                }
            }
            if (clusterInfo.weightCounter > 0 && lastHostIndex != -1) {
                targetHostIndex = lastHostIndex;
            } else {
                targetHostIndex = lastHostIndex != -1 && lastHostIndex != eligibleHosts.size() - 1 ? lastHostIndex + 1 : 0;
                Integer weight = clusterInfo.clusterWeightsMap.get(eligibleHosts.get(targetHostIndex).getHostId());
                clusterInfo.weightCounter = weight == null ? clusterInfo.defaultWeight : weight;
            }
            --clusterInfo.weightCounter;
            clusterInfo.lastHost = eligibleHosts.get(targetHostIndex);
            HostSpec hostSpec2 = eligibleHosts.get(targetHostIndex);
            return hostSpec2;
        }
        finally {
            lock.unlock();
        }
    }

    private void createCacheEntryForHosts(@NonNull List<HostSpec> hosts, @Nullable Properties props) throws SQLException {
        RoundRobinClusterInfo roundRobinClusterInfo;
        ArrayList<HostSpec> hostsWithCacheEntry = new ArrayList<HostSpec>();
        for (HostSpec host : hosts) {
            if (roundRobinCache.get(host.getHost()) == null) continue;
            hostsWithCacheEntry.add(host);
        }
        if (!hostsWithCacheEntry.isEmpty()) {
            roundRobinClusterInfo = roundRobinCache.get(((HostSpec)hostsWithCacheEntry.get(0)).getHost());
            if (this.hasPropertyChanged(roundRobinClusterInfo.lastClusterHostWeightPairPropertyValue, ROUND_ROBIN_HOST_WEIGHT_PAIRS, props)) {
                roundRobinClusterInfo.lastHost = null;
                roundRobinClusterInfo.weightCounter = 0;
                this.updateCachedHostWeightPairsPropertiesForRoundRobinClusterInfo(roundRobinClusterInfo, props);
            }
            if (this.hasPropertyChanged(roundRobinClusterInfo.lastClusterDefaultWeightPropertyValue, ROUND_ROBIN_DEFAULT_WEIGHT, props)) {
                roundRobinClusterInfo.defaultWeight = 1;
                this.updateCachedDefaultWeightPropertiesForRoundRobinClusterInfo(roundRobinClusterInfo, props);
            }
            for (HostSpec host : hosts) {
                roundRobinCache.put(host.getHost(), roundRobinClusterInfo, DEFAULT_ROUND_ROBIN_CACHE_EXPIRE_NANO);
            }
        } else {
            roundRobinClusterInfo = new RoundRobinClusterInfo();
            this.updateCachePropertiesForRoundRobinClusterInfo(roundRobinClusterInfo, props);
            for (HostSpec host : hosts) {
                roundRobinCache.put(host.getHost(), roundRobinClusterInfo, DEFAULT_ROUND_ROBIN_CACHE_EXPIRE_NANO);
            }
        }
    }

    private boolean hasPropertyChanged(String lastClusterHostWeightPairPropertyValue, AwsWrapperProperty wrapperProperty, Properties props) {
        if (props == null || wrapperProperty.getString(props) == null) {
            return false;
        }
        String propValue = wrapperProperty.getString(props);
        return !propValue.equals(lastClusterHostWeightPairPropertyValue);
    }

    private void updateCachePropertiesForRoundRobinClusterInfo(@NonNull RoundRobinClusterInfo roundRobinClusterInfo, @Nullable Properties props) throws SQLException {
        this.updateCachedDefaultWeightPropertiesForRoundRobinClusterInfo(roundRobinClusterInfo, props);
        this.updateCachedHostWeightPairsPropertiesForRoundRobinClusterInfo(roundRobinClusterInfo, props);
    }

    private void updateCachedDefaultWeightPropertiesForRoundRobinClusterInfo(@NonNull RoundRobinClusterInfo roundRobinClusterInfo, @Nullable Properties props) throws SQLException {
        String defaultWeightString;
        int defaultWeight = 1;
        if (props != null && !StringUtils.isNullOrEmpty(defaultWeightString = ROUND_ROBIN_DEFAULT_WEIGHT.getString(props))) {
            try {
                int parsedWeight = Integer.parseInt(defaultWeightString);
                if (parsedWeight < 1) {
                    throw new SQLException(Messages.get("HostSelector.roundRobinInvalidDefaultWeight"));
                }
                defaultWeight = parsedWeight;
            }
            catch (NumberFormatException e) {
                throw new SQLException(Messages.get("HostSelector.roundRobinInvalidDefaultWeight"));
            }
            roundRobinClusterInfo.lastClusterDefaultWeightPropertyValue = ROUND_ROBIN_DEFAULT_WEIGHT.getString(props);
        }
        roundRobinClusterInfo.defaultWeight = defaultWeight;
    }

    private void updateCachedHostWeightPairsPropertiesForRoundRobinClusterInfo(@NonNull RoundRobinClusterInfo roundRobinClusterInfo, @Nullable Properties props) throws SQLException {
        if (props != null) {
            String hostWeights = ROUND_ROBIN_HOST_WEIGHT_PAIRS.getString(props);
            if (!StringUtils.isNullOrEmpty(hostWeights)) {
                String[] hostWeightPairs;
                for (String pair : hostWeightPairs = hostWeights.split(",")) {
                    Matcher matcher = HOST_WEIGHT_PAIRS_PATTERN.matcher(pair);
                    if (!matcher.matches()) {
                        throw new SQLException(Messages.get("HostSelector.roundRobinInvalidHostWeightPairs"));
                    }
                    String hostName = matcher.group("host").trim();
                    String hostWeight = matcher.group("weight").trim();
                    if (hostName.isEmpty() || hostWeight.isEmpty()) {
                        throw new SQLException(Messages.get("HostSelector.roundRobinInvalidHostWeightPairs"));
                    }
                    try {
                        int weight = Integer.parseInt(hostWeight);
                        if (weight < 1) {
                            throw new SQLException(Messages.get("HostSelector.roundRobinInvalidHostWeightPairs"));
                        }
                        roundRobinClusterInfo.clusterWeightsMap.put(hostName, weight);
                    }
                    catch (NumberFormatException e) {
                        throw new SQLException(Messages.get("HostSelector.roundRobinInvalidHostWeightPairs"));
                    }
                }
                roundRobinClusterInfo.lastClusterHostWeightPairPropertyValue = ROUND_ROBIN_HOST_WEIGHT_PAIRS.getString(props);
            } else if (hostWeights != null && hostWeights.isEmpty()) {
                roundRobinClusterInfo.clusterWeightsMap.clear();
                roundRobinClusterInfo.lastClusterHostWeightPairPropertyValue = ROUND_ROBIN_HOST_WEIGHT_PAIRS.getString(props);
            }
        }
    }

    public static void clearCache() {
        roundRobinCache.clear();
    }

    static {
        PropertyDefinition.registerPluginProperties(RoundRobinHostSelector.class);
    }

    public static class RoundRobinClusterInfo {
        public HostSpec lastHost;
        public HashMap<String, Integer> clusterWeightsMap = new HashMap();
        public int defaultWeight = 1;
        public int weightCounter = 0;
        public String lastClusterHostWeightPairPropertyValue = "";
        public String lastClusterDefaultWeightPropertyValue = "";
    }
}

