/*
 * Decompiled with CFR 0.152.
 */
package software.amazon.jdbc.plugin;

import java.lang.ref.WeakReference;
import java.sql.Connection;
import java.sql.SQLException;
import java.util.Arrays;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Queue;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentLinkedQueue;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.logging.Logger;
import software.amazon.jdbc.HostSpec;
import software.amazon.jdbc.util.Messages;
import software.amazon.jdbc.util.RdsUtils;
import software.amazon.jdbc.util.StringUtils;

public class OpenedConnectionTracker {
    static final Map<String, Queue<WeakReference<Connection>>> openedConnections = new ConcurrentHashMap<String, Queue<WeakReference<Connection>>>();
    private static final ExecutorService invalidateConnectionsExecutorService = Executors.newCachedThreadPool(r -> {
        Thread invalidateThread = new Thread(r);
        invalidateThread.setDaemon(true);
        return invalidateThread;
    });
    private static final ExecutorService abortConnectionExecutorService = Executors.newCachedThreadPool(r -> {
        Thread abortThread = new Thread(r);
        abortThread.setDaemon(true);
        return abortThread;
    });
    private static final Logger LOGGER = Logger.getLogger(OpenedConnectionTracker.class.getName());
    private static final RdsUtils rdsUtils = new RdsUtils();

    public void populateOpenedConnectionQueue(HostSpec hostSpec, Connection conn) {
        Set<String> aliases = hostSpec.asAliases();
        String host = hostSpec.getHost();
        if (rdsUtils.isRdsInstance(host)) {
            this.trackConnection(host, conn);
            return;
        }
        Optional<String> instanceEndpoint = aliases.stream().filter(rdsUtils::isRdsInstance).findFirst();
        if (!instanceEndpoint.isPresent()) {
            LOGGER.finest(Messages.get("OpenedConnectionTracker.unableToPopulateOpenedConnectionQueue"));
            return;
        }
        this.trackConnection(instanceEndpoint.get(), conn);
    }

    public void invalidateAllConnections(HostSpec hostSpec) {
        this.invalidateAllConnections(hostSpec.getAliases().toArray(new String[0]));
    }

    public void invalidateAllConnections(String ... node) {
        Optional<String> instanceEndpoint = Arrays.stream(node).filter(rdsUtils::isRdsInstance).findFirst();
        if (!instanceEndpoint.isPresent()) {
            return;
        }
        Queue<WeakReference<Connection>> connectionQueue = openedConnections.get(instanceEndpoint.get());
        this.logConnectionQueue(instanceEndpoint.get(), connectionQueue);
        this.invalidateConnections(openedConnections.get(instanceEndpoint.get()));
    }

    public void invalidateCurrentConnection(HostSpec hostSpec, Connection connection) {
        String host;
        String string = host = rdsUtils.isRdsInstance(hostSpec.getHost()) ? hostSpec.getHost() : hostSpec.getAliases().stream().filter(rdsUtils::isRdsInstance).findFirst().get();
        if (StringUtils.isNullOrEmpty(host)) {
            return;
        }
        Queue<WeakReference<Connection>> connectionQueue = openedConnections.get(host);
        this.logConnectionQueue(host, connectionQueue);
        connectionQueue.removeIf(connectionWeakReference -> Objects.equals(connectionWeakReference.get(), connection));
    }

    private void trackConnection(String instanceEndpoint, Connection connection) {
        Queue connectionQueue = openedConnections.computeIfAbsent(instanceEndpoint, k -> new ConcurrentLinkedQueue());
        connectionQueue.add(new WeakReference<Connection>(connection));
        this.logOpenedConnections();
    }

    private void invalidateConnections(Queue<WeakReference<Connection>> connectionQueue) {
        invalidateConnectionsExecutorService.submit(() -> {
            WeakReference connReference;
            while ((connReference = (WeakReference)connectionQueue.poll()) != null) {
                Connection conn = (Connection)connReference.get();
                if (conn == null) continue;
                try {
                    conn.abort(abortConnectionExecutorService);
                }
                catch (SQLException sQLException) {}
            }
        });
    }

    public void logOpenedConnections() {
        LOGGER.finest(() -> {
            StringBuilder builder = new StringBuilder();
            openedConnections.forEach((key, queue) -> {
                if (!queue.isEmpty()) {
                    builder.append("\t[ ");
                    builder.append((String)key).append(":");
                    builder.append("\n\t {");
                    for (WeakReference connection : queue) {
                        builder.append("\n\t\t").append(connection.get());
                    }
                    builder.append("\n\t }\n");
                    builder.append("\t");
                }
            });
            return String.format("Opened Connections Tracked: \n[\n%s\n]", builder);
        });
    }

    private void logConnectionQueue(String host, Queue<WeakReference<Connection>> queue) {
        if (queue == null || queue.isEmpty()) {
            return;
        }
        StringBuilder builder = new StringBuilder();
        builder.append(host).append("[");
        for (WeakReference weakReference : queue) {
            builder.append("\n\t").append(weakReference.get());
        }
        builder.append("\n]");
        LOGGER.finest(Messages.get("OpenedConnectionTracker.invalidatingConnections", new Object[]{builder.toString()}));
    }
}

