/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.wafv2.model;

import java.io.Serializable;
import java.nio.ByteBuffer;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkBytes;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * A rule statement that defines a string match search for WAF to apply to web requests. The byte match statement
 * provides the bytes to search for, the location in requests that you want WAF to search, and other settings. The bytes
 * to search for are typically a string that corresponds with ASCII characters. In the WAF console and the developer
 * guide, this is called a string match statement.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class ByteMatchStatement implements SdkPojo, Serializable,
        ToCopyableBuilder<ByteMatchStatement.Builder, ByteMatchStatement> {
    private static final SdkField<SdkBytes> SEARCH_STRING_FIELD = SdkField.<SdkBytes> builder(MarshallingType.SDK_BYTES)
            .memberName("SearchString").getter(getter(ByteMatchStatement::searchString)).setter(setter(Builder::searchString))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SearchString").build()).build();

    private static final SdkField<FieldToMatch> FIELD_TO_MATCH_FIELD = SdkField.<FieldToMatch> builder(MarshallingType.SDK_POJO)
            .memberName("FieldToMatch").getter(getter(ByteMatchStatement::fieldToMatch)).setter(setter(Builder::fieldToMatch))
            .constructor(FieldToMatch::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("FieldToMatch").build()).build();

    private static final SdkField<List<TextTransformation>> TEXT_TRANSFORMATIONS_FIELD = SdkField
            .<List<TextTransformation>> builder(MarshallingType.LIST)
            .memberName("TextTransformations")
            .getter(getter(ByteMatchStatement::textTransformations))
            .setter(setter(Builder::textTransformations))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("TextTransformations").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<TextTransformation> builder(MarshallingType.SDK_POJO)
                                            .constructor(TextTransformation::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> POSITIONAL_CONSTRAINT_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("PositionalConstraint").getter(getter(ByteMatchStatement::positionalConstraintAsString))
            .setter(setter(Builder::positionalConstraint))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("PositionalConstraint").build())
            .build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(SEARCH_STRING_FIELD,
            FIELD_TO_MATCH_FIELD, TEXT_TRANSFORMATIONS_FIELD, POSITIONAL_CONSTRAINT_FIELD));

    private static final long serialVersionUID = 1L;

    private final SdkBytes searchString;

    private final FieldToMatch fieldToMatch;

    private final List<TextTransformation> textTransformations;

    private final String positionalConstraint;

    private ByteMatchStatement(BuilderImpl builder) {
        this.searchString = builder.searchString;
        this.fieldToMatch = builder.fieldToMatch;
        this.textTransformations = builder.textTransformations;
        this.positionalConstraint = builder.positionalConstraint;
    }

    /**
     * <p>
     * A string value that you want WAF to search for. WAF searches only in the part of web requests that you designate
     * for inspection in <a>FieldToMatch</a>. The maximum length of the value is 200 bytes.
     * </p>
     * <p>
     * Valid values depend on the component that you specify for inspection in <code>FieldToMatch</code>:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>Method</code>: The HTTP method that you want WAF to search for. This indicates the type of operation
     * specified in the request.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>UriPath</code>: The value that you want WAF to search for in the URI path, for example,
     * <code>/images/daily-ad.jpg</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>HeaderOrder</code>: The comma-separated list of header names to match for. WAF creates a string that
     * contains the ordered list of header names, from the headers in the web request, and then matches against that
     * string.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If <code>SearchString</code> includes alphabetic characters A-Z and a-z, note that the value is case sensitive.
     * </p>
     * <p>
     * <b>If you're using the WAF API</b>
     * </p>
     * <p>
     * Specify a base64-encoded version of the value. The maximum length of the value before you base64-encode it is 200
     * bytes.
     * </p>
     * <p>
     * For example, suppose the value of <code>Type</code> is <code>HEADER</code> and the value of <code>Data</code> is
     * <code>User-Agent</code>. If you want to search the <code>User-Agent</code> header for the value
     * <code>BadBot</code>, you base64-encode <code>BadBot</code> using MIME base64-encoding and include the resulting
     * value, <code>QmFkQm90</code>, in the value of <code>SearchString</code>.
     * </p>
     * <p>
     * <b>If you're using the CLI or one of the Amazon Web Services SDKs</b>
     * </p>
     * <p>
     * The value that you want WAF to search for. The SDK automatically base64 encodes the value.
     * </p>
     * 
     * @return A string value that you want WAF to search for. WAF searches only in the part of web requests that you
     *         designate for inspection in <a>FieldToMatch</a>. The maximum length of the value is 200 bytes.</p>
     *         <p>
     *         Valid values depend on the component that you specify for inspection in <code>FieldToMatch</code>:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>Method</code>: The HTTP method that you want WAF to search for. This indicates the type of
     *         operation specified in the request.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>UriPath</code>: The value that you want WAF to search for in the URI path, for example,
     *         <code>/images/daily-ad.jpg</code>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>HeaderOrder</code>: The comma-separated list of header names to match for. WAF creates a string
     *         that contains the ordered list of header names, from the headers in the web request, and then matches
     *         against that string.
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         If <code>SearchString</code> includes alphabetic characters A-Z and a-z, note that the value is case
     *         sensitive.
     *         </p>
     *         <p>
     *         <b>If you're using the WAF API</b>
     *         </p>
     *         <p>
     *         Specify a base64-encoded version of the value. The maximum length of the value before you base64-encode
     *         it is 200 bytes.
     *         </p>
     *         <p>
     *         For example, suppose the value of <code>Type</code> is <code>HEADER</code> and the value of
     *         <code>Data</code> is <code>User-Agent</code>. If you want to search the <code>User-Agent</code> header
     *         for the value <code>BadBot</code>, you base64-encode <code>BadBot</code> using MIME base64-encoding and
     *         include the resulting value, <code>QmFkQm90</code>, in the value of <code>SearchString</code>.
     *         </p>
     *         <p>
     *         <b>If you're using the CLI or one of the Amazon Web Services SDKs</b>
     *         </p>
     *         <p>
     *         The value that you want WAF to search for. The SDK automatically base64 encodes the value.
     */
    public final SdkBytes searchString() {
        return searchString;
    }

    /**
     * <p>
     * The part of the web request that you want WAF to inspect.
     * </p>
     * 
     * @return The part of the web request that you want WAF to inspect.
     */
    public final FieldToMatch fieldToMatch() {
        return fieldToMatch;
    }

    /**
     * For responses, this returns true if the service returned a value for the TextTransformations property. This DOES
     * NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasTextTransformations() {
        return textTransformations != null && !(textTransformations instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * Text transformations eliminate some of the unusual formatting that attackers use in web requests in an effort to
     * bypass detection. Text transformations are used in rule match statements, to transform the
     * <code>FieldToMatch</code> request component before inspecting it, and they're used in rate-based rule statements,
     * to transform request components before using them as custom aggregation keys. If you specify one or more
     * transformations to apply, WAF performs all transformations on the specified content, starting from the lowest
     * priority setting, and then uses the component contents.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasTextTransformations} method.
     * </p>
     * 
     * @return Text transformations eliminate some of the unusual formatting that attackers use in web requests in an
     *         effort to bypass detection. Text transformations are used in rule match statements, to transform the
     *         <code>FieldToMatch</code> request component before inspecting it, and they're used in rate-based rule
     *         statements, to transform request components before using them as custom aggregation keys. If you specify
     *         one or more transformations to apply, WAF performs all transformations on the specified content, starting
     *         from the lowest priority setting, and then uses the component contents.
     */
    public final List<TextTransformation> textTransformations() {
        return textTransformations;
    }

    /**
     * <p>
     * The area within the portion of the web request that you want WAF to search for <code>SearchString</code>. Valid
     * values include the following:
     * </p>
     * <p>
     * <b>CONTAINS</b>
     * </p>
     * <p>
     * The specified part of the web request must include the value of <code>SearchString</code>, but the location
     * doesn't matter.
     * </p>
     * <p>
     * <b>CONTAINS_WORD</b>
     * </p>
     * <p>
     * The specified part of the web request must include the value of <code>SearchString</code>, and
     * <code>SearchString</code> must contain only alphanumeric characters or underscore (A-Z, a-z, 0-9, or _). In
     * addition, <code>SearchString</code> must be a word, which means that both of the following are true:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>SearchString</code> is at the beginning of the specified part of the web request or is preceded by a
     * character other than an alphanumeric character or underscore (_). Examples include the value of a header and
     * <code>;BadBot</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>SearchString</code> is at the end of the specified part of the web request or is followed by a character
     * other than an alphanumeric character or underscore (_), for example, <code>BadBot;</code> and
     * <code>-BadBot;</code>.
     * </p>
     * </li>
     * </ul>
     * <p>
     * <b>EXACTLY</b>
     * </p>
     * <p>
     * The value of the specified part of the web request must exactly match the value of <code>SearchString</code>.
     * </p>
     * <p>
     * <b>STARTS_WITH</b>
     * </p>
     * <p>
     * The value of <code>SearchString</code> must appear at the beginning of the specified part of the web request.
     * </p>
     * <p>
     * <b>ENDS_WITH</b>
     * </p>
     * <p>
     * The value of <code>SearchString</code> must appear at the end of the specified part of the web request.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #positionalConstraint} will return {@link PositionalConstraint#UNKNOWN_TO_SDK_VERSION}. The raw value
     * returned by the service is available from {@link #positionalConstraintAsString}.
     * </p>
     * 
     * @return The area within the portion of the web request that you want WAF to search for <code>SearchString</code>.
     *         Valid values include the following:</p>
     *         <p>
     *         <b>CONTAINS</b>
     *         </p>
     *         <p>
     *         The specified part of the web request must include the value of <code>SearchString</code>, but the
     *         location doesn't matter.
     *         </p>
     *         <p>
     *         <b>CONTAINS_WORD</b>
     *         </p>
     *         <p>
     *         The specified part of the web request must include the value of <code>SearchString</code>, and
     *         <code>SearchString</code> must contain only alphanumeric characters or underscore (A-Z, a-z, 0-9, or _).
     *         In addition, <code>SearchString</code> must be a word, which means that both of the following are true:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>SearchString</code> is at the beginning of the specified part of the web request or is preceded by
     *         a character other than an alphanumeric character or underscore (_). Examples include the value of a
     *         header and <code>;BadBot</code>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>SearchString</code> is at the end of the specified part of the web request or is followed by a
     *         character other than an alphanumeric character or underscore (_), for example, <code>BadBot;</code> and
     *         <code>-BadBot;</code>.
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         <b>EXACTLY</b>
     *         </p>
     *         <p>
     *         The value of the specified part of the web request must exactly match the value of
     *         <code>SearchString</code>.
     *         </p>
     *         <p>
     *         <b>STARTS_WITH</b>
     *         </p>
     *         <p>
     *         The value of <code>SearchString</code> must appear at the beginning of the specified part of the web
     *         request.
     *         </p>
     *         <p>
     *         <b>ENDS_WITH</b>
     *         </p>
     *         <p>
     *         The value of <code>SearchString</code> must appear at the end of the specified part of the web request.
     * @see PositionalConstraint
     */
    public final PositionalConstraint positionalConstraint() {
        return PositionalConstraint.fromValue(positionalConstraint);
    }

    /**
     * <p>
     * The area within the portion of the web request that you want WAF to search for <code>SearchString</code>. Valid
     * values include the following:
     * </p>
     * <p>
     * <b>CONTAINS</b>
     * </p>
     * <p>
     * The specified part of the web request must include the value of <code>SearchString</code>, but the location
     * doesn't matter.
     * </p>
     * <p>
     * <b>CONTAINS_WORD</b>
     * </p>
     * <p>
     * The specified part of the web request must include the value of <code>SearchString</code>, and
     * <code>SearchString</code> must contain only alphanumeric characters or underscore (A-Z, a-z, 0-9, or _). In
     * addition, <code>SearchString</code> must be a word, which means that both of the following are true:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>SearchString</code> is at the beginning of the specified part of the web request or is preceded by a
     * character other than an alphanumeric character or underscore (_). Examples include the value of a header and
     * <code>;BadBot</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>SearchString</code> is at the end of the specified part of the web request or is followed by a character
     * other than an alphanumeric character or underscore (_), for example, <code>BadBot;</code> and
     * <code>-BadBot;</code>.
     * </p>
     * </li>
     * </ul>
     * <p>
     * <b>EXACTLY</b>
     * </p>
     * <p>
     * The value of the specified part of the web request must exactly match the value of <code>SearchString</code>.
     * </p>
     * <p>
     * <b>STARTS_WITH</b>
     * </p>
     * <p>
     * The value of <code>SearchString</code> must appear at the beginning of the specified part of the web request.
     * </p>
     * <p>
     * <b>ENDS_WITH</b>
     * </p>
     * <p>
     * The value of <code>SearchString</code> must appear at the end of the specified part of the web request.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #positionalConstraint} will return {@link PositionalConstraint#UNKNOWN_TO_SDK_VERSION}. The raw value
     * returned by the service is available from {@link #positionalConstraintAsString}.
     * </p>
     * 
     * @return The area within the portion of the web request that you want WAF to search for <code>SearchString</code>.
     *         Valid values include the following:</p>
     *         <p>
     *         <b>CONTAINS</b>
     *         </p>
     *         <p>
     *         The specified part of the web request must include the value of <code>SearchString</code>, but the
     *         location doesn't matter.
     *         </p>
     *         <p>
     *         <b>CONTAINS_WORD</b>
     *         </p>
     *         <p>
     *         The specified part of the web request must include the value of <code>SearchString</code>, and
     *         <code>SearchString</code> must contain only alphanumeric characters or underscore (A-Z, a-z, 0-9, or _).
     *         In addition, <code>SearchString</code> must be a word, which means that both of the following are true:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>SearchString</code> is at the beginning of the specified part of the web request or is preceded by
     *         a character other than an alphanumeric character or underscore (_). Examples include the value of a
     *         header and <code>;BadBot</code>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>SearchString</code> is at the end of the specified part of the web request or is followed by a
     *         character other than an alphanumeric character or underscore (_), for example, <code>BadBot;</code> and
     *         <code>-BadBot;</code>.
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         <b>EXACTLY</b>
     *         </p>
     *         <p>
     *         The value of the specified part of the web request must exactly match the value of
     *         <code>SearchString</code>.
     *         </p>
     *         <p>
     *         <b>STARTS_WITH</b>
     *         </p>
     *         <p>
     *         The value of <code>SearchString</code> must appear at the beginning of the specified part of the web
     *         request.
     *         </p>
     *         <p>
     *         <b>ENDS_WITH</b>
     *         </p>
     *         <p>
     *         The value of <code>SearchString</code> must appear at the end of the specified part of the web request.
     * @see PositionalConstraint
     */
    public final String positionalConstraintAsString() {
        return positionalConstraint;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(searchString());
        hashCode = 31 * hashCode + Objects.hashCode(fieldToMatch());
        hashCode = 31 * hashCode + Objects.hashCode(hasTextTransformations() ? textTransformations() : null);
        hashCode = 31 * hashCode + Objects.hashCode(positionalConstraintAsString());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof ByteMatchStatement)) {
            return false;
        }
        ByteMatchStatement other = (ByteMatchStatement) obj;
        return Objects.equals(searchString(), other.searchString()) && Objects.equals(fieldToMatch(), other.fieldToMatch())
                && hasTextTransformations() == other.hasTextTransformations()
                && Objects.equals(textTransformations(), other.textTransformations())
                && Objects.equals(positionalConstraintAsString(), other.positionalConstraintAsString());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("ByteMatchStatement").add("SearchString", searchString()).add("FieldToMatch", fieldToMatch())
                .add("TextTransformations", hasTextTransformations() ? textTransformations() : null)
                .add("PositionalConstraint", positionalConstraintAsString()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "SearchString":
            return Optional.ofNullable(clazz.cast(searchString()));
        case "FieldToMatch":
            return Optional.ofNullable(clazz.cast(fieldToMatch()));
        case "TextTransformations":
            return Optional.ofNullable(clazz.cast(textTransformations()));
        case "PositionalConstraint":
            return Optional.ofNullable(clazz.cast(positionalConstraintAsString()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<ByteMatchStatement, T> g) {
        return obj -> g.apply((ByteMatchStatement) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, ByteMatchStatement> {
        /**
         * <p>
         * A string value that you want WAF to search for. WAF searches only in the part of web requests that you
         * designate for inspection in <a>FieldToMatch</a>. The maximum length of the value is 200 bytes.
         * </p>
         * <p>
         * Valid values depend on the component that you specify for inspection in <code>FieldToMatch</code>:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>Method</code>: The HTTP method that you want WAF to search for. This indicates the type of operation
         * specified in the request.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>UriPath</code>: The value that you want WAF to search for in the URI path, for example,
         * <code>/images/daily-ad.jpg</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>HeaderOrder</code>: The comma-separated list of header names to match for. WAF creates a string that
         * contains the ordered list of header names, from the headers in the web request, and then matches against that
         * string.
         * </p>
         * </li>
         * </ul>
         * <p>
         * If <code>SearchString</code> includes alphabetic characters A-Z and a-z, note that the value is case
         * sensitive.
         * </p>
         * <p>
         * <b>If you're using the WAF API</b>
         * </p>
         * <p>
         * Specify a base64-encoded version of the value. The maximum length of the value before you base64-encode it is
         * 200 bytes.
         * </p>
         * <p>
         * For example, suppose the value of <code>Type</code> is <code>HEADER</code> and the value of <code>Data</code>
         * is <code>User-Agent</code>. If you want to search the <code>User-Agent</code> header for the value
         * <code>BadBot</code>, you base64-encode <code>BadBot</code> using MIME base64-encoding and include the
         * resulting value, <code>QmFkQm90</code>, in the value of <code>SearchString</code>.
         * </p>
         * <p>
         * <b>If you're using the CLI or one of the Amazon Web Services SDKs</b>
         * </p>
         * <p>
         * The value that you want WAF to search for. The SDK automatically base64 encodes the value.
         * </p>
         * 
         * @param searchString
         *        A string value that you want WAF to search for. WAF searches only in the part of web requests that you
         *        designate for inspection in <a>FieldToMatch</a>. The maximum length of the value is 200 bytes.</p>
         *        <p>
         *        Valid values depend on the component that you specify for inspection in <code>FieldToMatch</code>:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>Method</code>: The HTTP method that you want WAF to search for. This indicates the type of
         *        operation specified in the request.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>UriPath</code>: The value that you want WAF to search for in the URI path, for example,
         *        <code>/images/daily-ad.jpg</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>HeaderOrder</code>: The comma-separated list of header names to match for. WAF creates a string
         *        that contains the ordered list of header names, from the headers in the web request, and then matches
         *        against that string.
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        If <code>SearchString</code> includes alphabetic characters A-Z and a-z, note that the value is case
         *        sensitive.
         *        </p>
         *        <p>
         *        <b>If you're using the WAF API</b>
         *        </p>
         *        <p>
         *        Specify a base64-encoded version of the value. The maximum length of the value before you
         *        base64-encode it is 200 bytes.
         *        </p>
         *        <p>
         *        For example, suppose the value of <code>Type</code> is <code>HEADER</code> and the value of
         *        <code>Data</code> is <code>User-Agent</code>. If you want to search the <code>User-Agent</code> header
         *        for the value <code>BadBot</code>, you base64-encode <code>BadBot</code> using MIME base64-encoding
         *        and include the resulting value, <code>QmFkQm90</code>, in the value of <code>SearchString</code>.
         *        </p>
         *        <p>
         *        <b>If you're using the CLI or one of the Amazon Web Services SDKs</b>
         *        </p>
         *        <p>
         *        The value that you want WAF to search for. The SDK automatically base64 encodes the value.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder searchString(SdkBytes searchString);

        /**
         * <p>
         * The part of the web request that you want WAF to inspect.
         * </p>
         * 
         * @param fieldToMatch
         *        The part of the web request that you want WAF to inspect.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder fieldToMatch(FieldToMatch fieldToMatch);

        /**
         * <p>
         * The part of the web request that you want WAF to inspect.
         * </p>
         * This is a convenience method that creates an instance of the {@link FieldToMatch.Builder} avoiding the need
         * to create one manually via {@link FieldToMatch#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link FieldToMatch.Builder#build()} is called immediately and its
         * result is passed to {@link #fieldToMatch(FieldToMatch)}.
         * 
         * @param fieldToMatch
         *        a consumer that will call methods on {@link FieldToMatch.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #fieldToMatch(FieldToMatch)
         */
        default Builder fieldToMatch(Consumer<FieldToMatch.Builder> fieldToMatch) {
            return fieldToMatch(FieldToMatch.builder().applyMutation(fieldToMatch).build());
        }

        /**
         * <p>
         * Text transformations eliminate some of the unusual formatting that attackers use in web requests in an effort
         * to bypass detection. Text transformations are used in rule match statements, to transform the
         * <code>FieldToMatch</code> request component before inspecting it, and they're used in rate-based rule
         * statements, to transform request components before using them as custom aggregation keys. If you specify one
         * or more transformations to apply, WAF performs all transformations on the specified content, starting from
         * the lowest priority setting, and then uses the component contents.
         * </p>
         * 
         * @param textTransformations
         *        Text transformations eliminate some of the unusual formatting that attackers use in web requests in an
         *        effort to bypass detection. Text transformations are used in rule match statements, to transform the
         *        <code>FieldToMatch</code> request component before inspecting it, and they're used in rate-based rule
         *        statements, to transform request components before using them as custom aggregation keys. If you
         *        specify one or more transformations to apply, WAF performs all transformations on the specified
         *        content, starting from the lowest priority setting, and then uses the component contents.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder textTransformations(Collection<TextTransformation> textTransformations);

        /**
         * <p>
         * Text transformations eliminate some of the unusual formatting that attackers use in web requests in an effort
         * to bypass detection. Text transformations are used in rule match statements, to transform the
         * <code>FieldToMatch</code> request component before inspecting it, and they're used in rate-based rule
         * statements, to transform request components before using them as custom aggregation keys. If you specify one
         * or more transformations to apply, WAF performs all transformations on the specified content, starting from
         * the lowest priority setting, and then uses the component contents.
         * </p>
         * 
         * @param textTransformations
         *        Text transformations eliminate some of the unusual formatting that attackers use in web requests in an
         *        effort to bypass detection. Text transformations are used in rule match statements, to transform the
         *        <code>FieldToMatch</code> request component before inspecting it, and they're used in rate-based rule
         *        statements, to transform request components before using them as custom aggregation keys. If you
         *        specify one or more transformations to apply, WAF performs all transformations on the specified
         *        content, starting from the lowest priority setting, and then uses the component contents.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder textTransformations(TextTransformation... textTransformations);

        /**
         * <p>
         * Text transformations eliminate some of the unusual formatting that attackers use in web requests in an effort
         * to bypass detection. Text transformations are used in rule match statements, to transform the
         * <code>FieldToMatch</code> request component before inspecting it, and they're used in rate-based rule
         * statements, to transform request components before using them as custom aggregation keys. If you specify one
         * or more transformations to apply, WAF performs all transformations on the specified content, starting from
         * the lowest priority setting, and then uses the component contents.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.wafv2.model.TextTransformation.Builder} avoiding the need to create
         * one manually via {@link software.amazon.awssdk.services.wafv2.model.TextTransformation#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.wafv2.model.TextTransformation.Builder#build()} is called immediately
         * and its result is passed to {@link #textTransformations(List<TextTransformation>)}.
         * 
         * @param textTransformations
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.wafv2.model.TextTransformation.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #textTransformations(java.util.Collection<TextTransformation>)
         */
        Builder textTransformations(Consumer<TextTransformation.Builder>... textTransformations);

        /**
         * <p>
         * The area within the portion of the web request that you want WAF to search for <code>SearchString</code>.
         * Valid values include the following:
         * </p>
         * <p>
         * <b>CONTAINS</b>
         * </p>
         * <p>
         * The specified part of the web request must include the value of <code>SearchString</code>, but the location
         * doesn't matter.
         * </p>
         * <p>
         * <b>CONTAINS_WORD</b>
         * </p>
         * <p>
         * The specified part of the web request must include the value of <code>SearchString</code>, and
         * <code>SearchString</code> must contain only alphanumeric characters or underscore (A-Z, a-z, 0-9, or _). In
         * addition, <code>SearchString</code> must be a word, which means that both of the following are true:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>SearchString</code> is at the beginning of the specified part of the web request or is preceded by a
         * character other than an alphanumeric character or underscore (_). Examples include the value of a header and
         * <code>;BadBot</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>SearchString</code> is at the end of the specified part of the web request or is followed by a
         * character other than an alphanumeric character or underscore (_), for example, <code>BadBot;</code> and
         * <code>-BadBot;</code>.
         * </p>
         * </li>
         * </ul>
         * <p>
         * <b>EXACTLY</b>
         * </p>
         * <p>
         * The value of the specified part of the web request must exactly match the value of <code>SearchString</code>.
         * </p>
         * <p>
         * <b>STARTS_WITH</b>
         * </p>
         * <p>
         * The value of <code>SearchString</code> must appear at the beginning of the specified part of the web request.
         * </p>
         * <p>
         * <b>ENDS_WITH</b>
         * </p>
         * <p>
         * The value of <code>SearchString</code> must appear at the end of the specified part of the web request.
         * </p>
         * 
         * @param positionalConstraint
         *        The area within the portion of the web request that you want WAF to search for
         *        <code>SearchString</code>. Valid values include the following:</p>
         *        <p>
         *        <b>CONTAINS</b>
         *        </p>
         *        <p>
         *        The specified part of the web request must include the value of <code>SearchString</code>, but the
         *        location doesn't matter.
         *        </p>
         *        <p>
         *        <b>CONTAINS_WORD</b>
         *        </p>
         *        <p>
         *        The specified part of the web request must include the value of <code>SearchString</code>, and
         *        <code>SearchString</code> must contain only alphanumeric characters or underscore (A-Z, a-z, 0-9, or
         *        _). In addition, <code>SearchString</code> must be a word, which means that both of the following are
         *        true:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>SearchString</code> is at the beginning of the specified part of the web request or is preceded
         *        by a character other than an alphanumeric character or underscore (_). Examples include the value of a
         *        header and <code>;BadBot</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>SearchString</code> is at the end of the specified part of the web request or is followed by a
         *        character other than an alphanumeric character or underscore (_), for example, <code>BadBot;</code>
         *        and <code>-BadBot;</code>.
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        <b>EXACTLY</b>
         *        </p>
         *        <p>
         *        The value of the specified part of the web request must exactly match the value of
         *        <code>SearchString</code>.
         *        </p>
         *        <p>
         *        <b>STARTS_WITH</b>
         *        </p>
         *        <p>
         *        The value of <code>SearchString</code> must appear at the beginning of the specified part of the web
         *        request.
         *        </p>
         *        <p>
         *        <b>ENDS_WITH</b>
         *        </p>
         *        <p>
         *        The value of <code>SearchString</code> must appear at the end of the specified part of the web
         *        request.
         * @see PositionalConstraint
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see PositionalConstraint
         */
        Builder positionalConstraint(String positionalConstraint);

        /**
         * <p>
         * The area within the portion of the web request that you want WAF to search for <code>SearchString</code>.
         * Valid values include the following:
         * </p>
         * <p>
         * <b>CONTAINS</b>
         * </p>
         * <p>
         * The specified part of the web request must include the value of <code>SearchString</code>, but the location
         * doesn't matter.
         * </p>
         * <p>
         * <b>CONTAINS_WORD</b>
         * </p>
         * <p>
         * The specified part of the web request must include the value of <code>SearchString</code>, and
         * <code>SearchString</code> must contain only alphanumeric characters or underscore (A-Z, a-z, 0-9, or _). In
         * addition, <code>SearchString</code> must be a word, which means that both of the following are true:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>SearchString</code> is at the beginning of the specified part of the web request or is preceded by a
         * character other than an alphanumeric character or underscore (_). Examples include the value of a header and
         * <code>;BadBot</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>SearchString</code> is at the end of the specified part of the web request or is followed by a
         * character other than an alphanumeric character or underscore (_), for example, <code>BadBot;</code> and
         * <code>-BadBot;</code>.
         * </p>
         * </li>
         * </ul>
         * <p>
         * <b>EXACTLY</b>
         * </p>
         * <p>
         * The value of the specified part of the web request must exactly match the value of <code>SearchString</code>.
         * </p>
         * <p>
         * <b>STARTS_WITH</b>
         * </p>
         * <p>
         * The value of <code>SearchString</code> must appear at the beginning of the specified part of the web request.
         * </p>
         * <p>
         * <b>ENDS_WITH</b>
         * </p>
         * <p>
         * The value of <code>SearchString</code> must appear at the end of the specified part of the web request.
         * </p>
         * 
         * @param positionalConstraint
         *        The area within the portion of the web request that you want WAF to search for
         *        <code>SearchString</code>. Valid values include the following:</p>
         *        <p>
         *        <b>CONTAINS</b>
         *        </p>
         *        <p>
         *        The specified part of the web request must include the value of <code>SearchString</code>, but the
         *        location doesn't matter.
         *        </p>
         *        <p>
         *        <b>CONTAINS_WORD</b>
         *        </p>
         *        <p>
         *        The specified part of the web request must include the value of <code>SearchString</code>, and
         *        <code>SearchString</code> must contain only alphanumeric characters or underscore (A-Z, a-z, 0-9, or
         *        _). In addition, <code>SearchString</code> must be a word, which means that both of the following are
         *        true:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>SearchString</code> is at the beginning of the specified part of the web request or is preceded
         *        by a character other than an alphanumeric character or underscore (_). Examples include the value of a
         *        header and <code>;BadBot</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>SearchString</code> is at the end of the specified part of the web request or is followed by a
         *        character other than an alphanumeric character or underscore (_), for example, <code>BadBot;</code>
         *        and <code>-BadBot;</code>.
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        <b>EXACTLY</b>
         *        </p>
         *        <p>
         *        The value of the specified part of the web request must exactly match the value of
         *        <code>SearchString</code>.
         *        </p>
         *        <p>
         *        <b>STARTS_WITH</b>
         *        </p>
         *        <p>
         *        The value of <code>SearchString</code> must appear at the beginning of the specified part of the web
         *        request.
         *        </p>
         *        <p>
         *        <b>ENDS_WITH</b>
         *        </p>
         *        <p>
         *        The value of <code>SearchString</code> must appear at the end of the specified part of the web
         *        request.
         * @see PositionalConstraint
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see PositionalConstraint
         */
        Builder positionalConstraint(PositionalConstraint positionalConstraint);
    }

    static final class BuilderImpl implements Builder {
        private SdkBytes searchString;

        private FieldToMatch fieldToMatch;

        private List<TextTransformation> textTransformations = DefaultSdkAutoConstructList.getInstance();

        private String positionalConstraint;

        private BuilderImpl() {
        }

        private BuilderImpl(ByteMatchStatement model) {
            searchString(model.searchString);
            fieldToMatch(model.fieldToMatch);
            textTransformations(model.textTransformations);
            positionalConstraint(model.positionalConstraint);
        }

        public final ByteBuffer getSearchString() {
            return searchString == null ? null : searchString.asByteBuffer();
        }

        public final void setSearchString(ByteBuffer searchString) {
            searchString(searchString == null ? null : SdkBytes.fromByteBuffer(searchString));
        }

        @Override
        public final Builder searchString(SdkBytes searchString) {
            this.searchString = searchString;
            return this;
        }

        public final FieldToMatch.Builder getFieldToMatch() {
            return fieldToMatch != null ? fieldToMatch.toBuilder() : null;
        }

        public final void setFieldToMatch(FieldToMatch.BuilderImpl fieldToMatch) {
            this.fieldToMatch = fieldToMatch != null ? fieldToMatch.build() : null;
        }

        @Override
        public final Builder fieldToMatch(FieldToMatch fieldToMatch) {
            this.fieldToMatch = fieldToMatch;
            return this;
        }

        public final List<TextTransformation.Builder> getTextTransformations() {
            List<TextTransformation.Builder> result = TextTransformationsCopier.copyToBuilder(this.textTransformations);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setTextTransformations(Collection<TextTransformation.BuilderImpl> textTransformations) {
            this.textTransformations = TextTransformationsCopier.copyFromBuilder(textTransformations);
        }

        @Override
        public final Builder textTransformations(Collection<TextTransformation> textTransformations) {
            this.textTransformations = TextTransformationsCopier.copy(textTransformations);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder textTransformations(TextTransformation... textTransformations) {
            textTransformations(Arrays.asList(textTransformations));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder textTransformations(Consumer<TextTransformation.Builder>... textTransformations) {
            textTransformations(Stream.of(textTransformations).map(c -> TextTransformation.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final String getPositionalConstraint() {
            return positionalConstraint;
        }

        public final void setPositionalConstraint(String positionalConstraint) {
            this.positionalConstraint = positionalConstraint;
        }

        @Override
        public final Builder positionalConstraint(String positionalConstraint) {
            this.positionalConstraint = positionalConstraint;
            return this;
        }

        @Override
        public final Builder positionalConstraint(PositionalConstraint positionalConstraint) {
            this.positionalConstraint(positionalConstraint == null ? null : positionalConstraint.toString());
            return this;
        }

        @Override
        public ByteMatchStatement build() {
            return new ByteMatchStatement(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
