/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.timestreamwrite;

import java.net.URI;
import java.util.Collections;
import java.util.List;
import java.util.function.Consumer;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.awscore.client.config.AwsClientOption;
import software.amazon.awssdk.awscore.client.handler.AwsSyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.ApiName;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.client.handler.SyncClientHandler;
import software.amazon.awssdk.core.endpointdiscovery.EndpointDiscoveryRefreshCache;
import software.amazon.awssdk.core.endpointdiscovery.EndpointDiscoveryRequest;
import software.amazon.awssdk.core.exception.SdkClientException;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.core.util.VersionInfo;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.timestreamwrite.model.AccessDeniedException;
import software.amazon.awssdk.services.timestreamwrite.model.ConflictException;
import software.amazon.awssdk.services.timestreamwrite.model.CreateBatchLoadTaskRequest;
import software.amazon.awssdk.services.timestreamwrite.model.CreateBatchLoadTaskResponse;
import software.amazon.awssdk.services.timestreamwrite.model.CreateDatabaseRequest;
import software.amazon.awssdk.services.timestreamwrite.model.CreateDatabaseResponse;
import software.amazon.awssdk.services.timestreamwrite.model.CreateTableRequest;
import software.amazon.awssdk.services.timestreamwrite.model.CreateTableResponse;
import software.amazon.awssdk.services.timestreamwrite.model.DeleteDatabaseRequest;
import software.amazon.awssdk.services.timestreamwrite.model.DeleteDatabaseResponse;
import software.amazon.awssdk.services.timestreamwrite.model.DeleteTableRequest;
import software.amazon.awssdk.services.timestreamwrite.model.DeleteTableResponse;
import software.amazon.awssdk.services.timestreamwrite.model.DescribeBatchLoadTaskRequest;
import software.amazon.awssdk.services.timestreamwrite.model.DescribeBatchLoadTaskResponse;
import software.amazon.awssdk.services.timestreamwrite.model.DescribeDatabaseRequest;
import software.amazon.awssdk.services.timestreamwrite.model.DescribeDatabaseResponse;
import software.amazon.awssdk.services.timestreamwrite.model.DescribeEndpointsRequest;
import software.amazon.awssdk.services.timestreamwrite.model.DescribeEndpointsResponse;
import software.amazon.awssdk.services.timestreamwrite.model.DescribeTableRequest;
import software.amazon.awssdk.services.timestreamwrite.model.DescribeTableResponse;
import software.amazon.awssdk.services.timestreamwrite.model.InternalServerException;
import software.amazon.awssdk.services.timestreamwrite.model.InvalidEndpointException;
import software.amazon.awssdk.services.timestreamwrite.model.ListBatchLoadTasksRequest;
import software.amazon.awssdk.services.timestreamwrite.model.ListBatchLoadTasksResponse;
import software.amazon.awssdk.services.timestreamwrite.model.ListDatabasesRequest;
import software.amazon.awssdk.services.timestreamwrite.model.ListDatabasesResponse;
import software.amazon.awssdk.services.timestreamwrite.model.ListTablesRequest;
import software.amazon.awssdk.services.timestreamwrite.model.ListTablesResponse;
import software.amazon.awssdk.services.timestreamwrite.model.ListTagsForResourceRequest;
import software.amazon.awssdk.services.timestreamwrite.model.ListTagsForResourceResponse;
import software.amazon.awssdk.services.timestreamwrite.model.RejectedRecordsException;
import software.amazon.awssdk.services.timestreamwrite.model.ResourceNotFoundException;
import software.amazon.awssdk.services.timestreamwrite.model.ResumeBatchLoadTaskRequest;
import software.amazon.awssdk.services.timestreamwrite.model.ResumeBatchLoadTaskResponse;
import software.amazon.awssdk.services.timestreamwrite.model.ServiceQuotaExceededException;
import software.amazon.awssdk.services.timestreamwrite.model.TagResourceRequest;
import software.amazon.awssdk.services.timestreamwrite.model.TagResourceResponse;
import software.amazon.awssdk.services.timestreamwrite.model.ThrottlingException;
import software.amazon.awssdk.services.timestreamwrite.model.TimestreamWriteException;
import software.amazon.awssdk.services.timestreamwrite.model.TimestreamWriteRequest;
import software.amazon.awssdk.services.timestreamwrite.model.UntagResourceRequest;
import software.amazon.awssdk.services.timestreamwrite.model.UntagResourceResponse;
import software.amazon.awssdk.services.timestreamwrite.model.UpdateDatabaseRequest;
import software.amazon.awssdk.services.timestreamwrite.model.UpdateDatabaseResponse;
import software.amazon.awssdk.services.timestreamwrite.model.UpdateTableRequest;
import software.amazon.awssdk.services.timestreamwrite.model.UpdateTableResponse;
import software.amazon.awssdk.services.timestreamwrite.model.ValidationException;
import software.amazon.awssdk.services.timestreamwrite.model.WriteRecordsRequest;
import software.amazon.awssdk.services.timestreamwrite.model.WriteRecordsResponse;
import software.amazon.awssdk.services.timestreamwrite.paginators.ListBatchLoadTasksIterable;
import software.amazon.awssdk.services.timestreamwrite.paginators.ListDatabasesIterable;
import software.amazon.awssdk.services.timestreamwrite.paginators.ListTablesIterable;
import software.amazon.awssdk.services.timestreamwrite.transform.CreateBatchLoadTaskRequestMarshaller;
import software.amazon.awssdk.services.timestreamwrite.transform.CreateDatabaseRequestMarshaller;
import software.amazon.awssdk.services.timestreamwrite.transform.CreateTableRequestMarshaller;
import software.amazon.awssdk.services.timestreamwrite.transform.DeleteDatabaseRequestMarshaller;
import software.amazon.awssdk.services.timestreamwrite.transform.DeleteTableRequestMarshaller;
import software.amazon.awssdk.services.timestreamwrite.transform.DescribeBatchLoadTaskRequestMarshaller;
import software.amazon.awssdk.services.timestreamwrite.transform.DescribeDatabaseRequestMarshaller;
import software.amazon.awssdk.services.timestreamwrite.transform.DescribeEndpointsRequestMarshaller;
import software.amazon.awssdk.services.timestreamwrite.transform.DescribeTableRequestMarshaller;
import software.amazon.awssdk.services.timestreamwrite.transform.ListBatchLoadTasksRequestMarshaller;
import software.amazon.awssdk.services.timestreamwrite.transform.ListDatabasesRequestMarshaller;
import software.amazon.awssdk.services.timestreamwrite.transform.ListTablesRequestMarshaller;
import software.amazon.awssdk.services.timestreamwrite.transform.ListTagsForResourceRequestMarshaller;
import software.amazon.awssdk.services.timestreamwrite.transform.ResumeBatchLoadTaskRequestMarshaller;
import software.amazon.awssdk.services.timestreamwrite.transform.TagResourceRequestMarshaller;
import software.amazon.awssdk.services.timestreamwrite.transform.UntagResourceRequestMarshaller;
import software.amazon.awssdk.services.timestreamwrite.transform.UpdateDatabaseRequestMarshaller;
import software.amazon.awssdk.services.timestreamwrite.transform.UpdateTableRequestMarshaller;
import software.amazon.awssdk.services.timestreamwrite.transform.WriteRecordsRequestMarshaller;
import software.amazon.awssdk.utils.Logger;

/**
 * Internal implementation of {@link TimestreamWriteClient}.
 *
 * @see TimestreamWriteClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultTimestreamWriteClient implements TimestreamWriteClient {
    private static final Logger log = Logger.loggerFor(DefaultTimestreamWriteClient.class);

    private final SyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    private final TimestreamWriteServiceClientConfiguration serviceClientConfiguration;

    private EndpointDiscoveryRefreshCache endpointDiscoveryCache;

    protected DefaultTimestreamWriteClient(TimestreamWriteServiceClientConfiguration serviceClientConfiguration,
            SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsSyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.serviceClientConfiguration = serviceClientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
        if (clientConfiguration.option(SdkClientOption.ENDPOINT_DISCOVERY_ENABLED)) {
            this.endpointDiscoveryCache = EndpointDiscoveryRefreshCache.create(TimestreamWriteEndpointDiscoveryCacheLoader
                    .create(this));
            if (clientConfiguration.option(SdkClientOption.ENDPOINT_OVERRIDDEN) == Boolean.TRUE) {
                log.warn(() -> "Endpoint discovery is enabled for this client, and an endpoint override was also specified. This will disable endpoint discovery for methods that require it, instead using the specified endpoint override. This may or may not be what you intended.");
            }
        }
    }

    /**
     * <p>
     * Creates a new Timestream batch load task. A batch load task processes data from a CSV source in an S3 location
     * and writes to a Timestream table. A mapping from source to target is defined in a batch load task. Errors and
     * events are written to a report at an S3 location. For the report, if the KMS key is not specified, the report
     * will be encrypted with an S3 managed key when <code>SSE_S3</code> is the option. Otherwise an error is thrown.
     * For more information, see <a
     * href="https://docs.aws.amazon.com/kms/latest/developerguide/concepts.html#aws-managed-cmk">Amazon Web Services
     * managed keys</a>. <a href="https://docs.aws.amazon.com/timestream/latest/developerguide/ts-limits.html">Service
     * quotas apply</a>. For details, see <a
     * href="https://docs.aws.amazon.com/timestream/latest/developerguide/code-samples.create-batch-load.html">code
     * sample</a>.
     * </p>
     *
     * @param createBatchLoadTaskRequest
     * @return Result of the CreateBatchLoadTask operation returned by the service.
     * @throws InternalServerException
     *         Timestream was unable to fully process this request because of an internal server error.
     * @throws ThrottlingException
     *         Too many requests were made by a user and they exceeded the service quotas. The request was throttled.
     * @throws AccessDeniedException
     *         You are not authorized to perform this action.
     * @throws ValidationException
     *         An invalid or malformed request.
     * @throws ConflictException
     *         Timestream was unable to process this request because it contains resource that already exists.
     * @throws ResourceNotFoundException
     *         The operation tried to access a nonexistent resource. The resource might not be specified correctly, or
     *         its status might not be ACTIVE.
     * @throws ServiceQuotaExceededException
     *         The instance quota of resource exceeded for this account.
     * @throws InvalidEndpointException
     *         The requested endpoint was not valid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws TimestreamWriteException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample TimestreamWriteClient.CreateBatchLoadTask
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/timestream-write-2018-11-01/CreateBatchLoadTask"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CreateBatchLoadTaskResponse createBatchLoadTask(CreateBatchLoadTaskRequest createBatchLoadTaskRequest)
            throws InternalServerException, ThrottlingException, AccessDeniedException, ValidationException, ConflictException,
            ResourceNotFoundException, ServiceQuotaExceededException, InvalidEndpointException, AwsServiceException,
            SdkClientException, TimestreamWriteException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateBatchLoadTaskResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, CreateBatchLoadTaskResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        boolean endpointDiscoveryEnabled = clientConfiguration.option(SdkClientOption.ENDPOINT_DISCOVERY_ENABLED);
        boolean endpointOverridden = clientConfiguration.option(SdkClientOption.ENDPOINT_OVERRIDDEN) == Boolean.TRUE;
        if (endpointOverridden) {
            endpointDiscoveryEnabled = false;
        } else if (!endpointDiscoveryEnabled) {
            throw new IllegalStateException(
                    "This operation requires endpoint discovery to be enabled, or for you to specify an endpoint override when the client is created.");
        }
        URI cachedEndpoint = null;
        if (endpointDiscoveryEnabled) {
            String key = createBatchLoadTaskRequest.overrideConfiguration()
                    .flatMap(AwsRequestOverrideConfiguration::credentialsProvider)
                    .orElseGet(() -> clientConfiguration.option(AwsClientOption.CREDENTIALS_PROVIDER)).resolveCredentials()
                    .accessKeyId();
            EndpointDiscoveryRequest endpointDiscoveryRequest = EndpointDiscoveryRequest.builder().required(true)
                    .defaultEndpoint(clientConfiguration.option(SdkClientOption.ENDPOINT))
                    .overrideConfiguration(createBatchLoadTaskRequest.overrideConfiguration().orElse(null)).build();
            cachedEndpoint = endpointDiscoveryCache.get(key, endpointDiscoveryRequest);
        }
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createBatchLoadTaskRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Timestream Write");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateBatchLoadTask");

            return clientHandler.execute(new ClientExecutionParams<CreateBatchLoadTaskRequest, CreateBatchLoadTaskResponse>()
                    .withOperationName("CreateBatchLoadTask").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).discoveredEndpoint(cachedEndpoint)
                    .withInput(createBatchLoadTaskRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new CreateBatchLoadTaskRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Creates a new Timestream database. If the KMS key is not specified, the database will be encrypted with a
     * Timestream managed KMS key located in your account. For more information, see <a
     * href="https://docs.aws.amazon.com/kms/latest/developerguide/concepts.html#aws-managed-cmk">Amazon Web Services
     * managed keys</a>. <a href="https://docs.aws.amazon.com/timestream/latest/developerguide/ts-limits.html">Service
     * quotas apply</a>. For details, see <a
     * href="https://docs.aws.amazon.com/timestream/latest/developerguide/code-samples.create-db.html">code sample</a>.
     * </p>
     *
     * @param createDatabaseRequest
     * @return Result of the CreateDatabase operation returned by the service.
     * @throws ConflictException
     *         Timestream was unable to process this request because it contains resource that already exists.
     * @throws ValidationException
     *         An invalid or malformed request.
     * @throws AccessDeniedException
     *         You are not authorized to perform this action.
     * @throws ServiceQuotaExceededException
     *         The instance quota of resource exceeded for this account.
     * @throws ThrottlingException
     *         Too many requests were made by a user and they exceeded the service quotas. The request was throttled.
     * @throws InvalidEndpointException
     *         The requested endpoint was not valid.
     * @throws InternalServerException
     *         Timestream was unable to fully process this request because of an internal server error.
     * @throws InvalidEndpointException
     *         The requested endpoint was not valid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws TimestreamWriteException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample TimestreamWriteClient.CreateDatabase
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/timestream-write-2018-11-01/CreateDatabase"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CreateDatabaseResponse createDatabase(CreateDatabaseRequest createDatabaseRequest) throws ConflictException,
            ValidationException, AccessDeniedException, ServiceQuotaExceededException, ThrottlingException,
            InvalidEndpointException, InternalServerException, AwsServiceException, SdkClientException, TimestreamWriteException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateDatabaseResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                CreateDatabaseResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        boolean endpointDiscoveryEnabled = clientConfiguration.option(SdkClientOption.ENDPOINT_DISCOVERY_ENABLED);
        boolean endpointOverridden = clientConfiguration.option(SdkClientOption.ENDPOINT_OVERRIDDEN) == Boolean.TRUE;
        if (endpointOverridden) {
            endpointDiscoveryEnabled = false;
        } else if (!endpointDiscoveryEnabled) {
            throw new IllegalStateException(
                    "This operation requires endpoint discovery to be enabled, or for you to specify an endpoint override when the client is created.");
        }
        URI cachedEndpoint = null;
        if (endpointDiscoveryEnabled) {
            String key = createDatabaseRequest.overrideConfiguration()
                    .flatMap(AwsRequestOverrideConfiguration::credentialsProvider)
                    .orElseGet(() -> clientConfiguration.option(AwsClientOption.CREDENTIALS_PROVIDER)).resolveCredentials()
                    .accessKeyId();
            EndpointDiscoveryRequest endpointDiscoveryRequest = EndpointDiscoveryRequest.builder().required(true)
                    .defaultEndpoint(clientConfiguration.option(SdkClientOption.ENDPOINT))
                    .overrideConfiguration(createDatabaseRequest.overrideConfiguration().orElse(null)).build();
            cachedEndpoint = endpointDiscoveryCache.get(key, endpointDiscoveryRequest);
        }
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createDatabaseRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Timestream Write");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateDatabase");

            return clientHandler.execute(new ClientExecutionParams<CreateDatabaseRequest, CreateDatabaseResponse>()
                    .withOperationName("CreateDatabase").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).discoveredEndpoint(cachedEndpoint)
                    .withInput(createDatabaseRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new CreateDatabaseRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Adds a new table to an existing database in your account. In an Amazon Web Services account, table names must be
     * at least unique within each Region if they are in the same database. You might have identical table names in the
     * same Region if the tables are in separate databases. While creating the table, you must specify the table name,
     * database name, and the retention properties. <a
     * href="https://docs.aws.amazon.com/timestream/latest/developerguide/ts-limits.html">Service quotas apply</a>. See
     * <a href="https://docs.aws.amazon.com/timestream/latest/developerguide/code-samples.create-table.html">code
     * sample</a> for details.
     * </p>
     *
     * @param createTableRequest
     * @return Result of the CreateTable operation returned by the service.
     * @throws ConflictException
     *         Timestream was unable to process this request because it contains resource that already exists.
     * @throws ValidationException
     *         An invalid or malformed request.
     * @throws AccessDeniedException
     *         You are not authorized to perform this action.
     * @throws ResourceNotFoundException
     *         The operation tried to access a nonexistent resource. The resource might not be specified correctly, or
     *         its status might not be ACTIVE.
     * @throws ServiceQuotaExceededException
     *         The instance quota of resource exceeded for this account.
     * @throws ThrottlingException
     *         Too many requests were made by a user and they exceeded the service quotas. The request was throttled.
     * @throws InvalidEndpointException
     *         The requested endpoint was not valid.
     * @throws InternalServerException
     *         Timestream was unable to fully process this request because of an internal server error.
     * @throws InvalidEndpointException
     *         The requested endpoint was not valid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws TimestreamWriteException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample TimestreamWriteClient.CreateTable
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/timestream-write-2018-11-01/CreateTable" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CreateTableResponse createTable(CreateTableRequest createTableRequest) throws ConflictException, ValidationException,
            AccessDeniedException, ResourceNotFoundException, ServiceQuotaExceededException, ThrottlingException,
            InvalidEndpointException, InternalServerException, AwsServiceException, SdkClientException, TimestreamWriteException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateTableResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                CreateTableResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        boolean endpointDiscoveryEnabled = clientConfiguration.option(SdkClientOption.ENDPOINT_DISCOVERY_ENABLED);
        boolean endpointOverridden = clientConfiguration.option(SdkClientOption.ENDPOINT_OVERRIDDEN) == Boolean.TRUE;
        if (endpointOverridden) {
            endpointDiscoveryEnabled = false;
        } else if (!endpointDiscoveryEnabled) {
            throw new IllegalStateException(
                    "This operation requires endpoint discovery to be enabled, or for you to specify an endpoint override when the client is created.");
        }
        URI cachedEndpoint = null;
        if (endpointDiscoveryEnabled) {
            String key = createTableRequest.overrideConfiguration().flatMap(AwsRequestOverrideConfiguration::credentialsProvider)
                    .orElseGet(() -> clientConfiguration.option(AwsClientOption.CREDENTIALS_PROVIDER)).resolveCredentials()
                    .accessKeyId();
            EndpointDiscoveryRequest endpointDiscoveryRequest = EndpointDiscoveryRequest.builder().required(true)
                    .defaultEndpoint(clientConfiguration.option(SdkClientOption.ENDPOINT))
                    .overrideConfiguration(createTableRequest.overrideConfiguration().orElse(null)).build();
            cachedEndpoint = endpointDiscoveryCache.get(key, endpointDiscoveryRequest);
        }
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createTableRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Timestream Write");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateTable");

            return clientHandler.execute(new ClientExecutionParams<CreateTableRequest, CreateTableResponse>()
                    .withOperationName("CreateTable").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).discoveredEndpoint(cachedEndpoint)
                    .withInput(createTableRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new CreateTableRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deletes a given Timestream database. <i>This is an irreversible operation. After a database is deleted, the
     * time-series data from its tables cannot be recovered.</i>
     * </p>
     * <note>
     * <p>
     * All tables in the database must be deleted first, or a ValidationException error will be thrown.
     * </p>
     * <p>
     * Due to the nature of distributed retries, the operation can return either success or a ResourceNotFoundException.
     * Clients should consider them equivalent.
     * </p>
     * </note>
     * <p>
     * See <a href="https://docs.aws.amazon.com/timestream/latest/developerguide/code-samples.delete-db.html">code
     * sample</a> for details.
     * </p>
     *
     * @param deleteDatabaseRequest
     * @return Result of the DeleteDatabase operation returned by the service.
     * @throws InternalServerException
     *         Timestream was unable to fully process this request because of an internal server error.
     * @throws ThrottlingException
     *         Too many requests were made by a user and they exceeded the service quotas. The request was throttled.
     * @throws ResourceNotFoundException
     *         The operation tried to access a nonexistent resource. The resource might not be specified correctly, or
     *         its status might not be ACTIVE.
     * @throws ValidationException
     *         An invalid or malformed request.
     * @throws AccessDeniedException
     *         You are not authorized to perform this action.
     * @throws InvalidEndpointException
     *         The requested endpoint was not valid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws TimestreamWriteException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample TimestreamWriteClient.DeleteDatabase
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/timestream-write-2018-11-01/DeleteDatabase"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DeleteDatabaseResponse deleteDatabase(DeleteDatabaseRequest deleteDatabaseRequest) throws InternalServerException,
            ThrottlingException, ResourceNotFoundException, ValidationException, AccessDeniedException, InvalidEndpointException,
            AwsServiceException, SdkClientException, TimestreamWriteException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteDatabaseResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DeleteDatabaseResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        boolean endpointDiscoveryEnabled = clientConfiguration.option(SdkClientOption.ENDPOINT_DISCOVERY_ENABLED);
        boolean endpointOverridden = clientConfiguration.option(SdkClientOption.ENDPOINT_OVERRIDDEN) == Boolean.TRUE;
        if (endpointOverridden) {
            endpointDiscoveryEnabled = false;
        } else if (!endpointDiscoveryEnabled) {
            throw new IllegalStateException(
                    "This operation requires endpoint discovery to be enabled, or for you to specify an endpoint override when the client is created.");
        }
        URI cachedEndpoint = null;
        if (endpointDiscoveryEnabled) {
            String key = deleteDatabaseRequest.overrideConfiguration()
                    .flatMap(AwsRequestOverrideConfiguration::credentialsProvider)
                    .orElseGet(() -> clientConfiguration.option(AwsClientOption.CREDENTIALS_PROVIDER)).resolveCredentials()
                    .accessKeyId();
            EndpointDiscoveryRequest endpointDiscoveryRequest = EndpointDiscoveryRequest.builder().required(true)
                    .defaultEndpoint(clientConfiguration.option(SdkClientOption.ENDPOINT))
                    .overrideConfiguration(deleteDatabaseRequest.overrideConfiguration().orElse(null)).build();
            cachedEndpoint = endpointDiscoveryCache.get(key, endpointDiscoveryRequest);
        }
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteDatabaseRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Timestream Write");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteDatabase");

            return clientHandler.execute(new ClientExecutionParams<DeleteDatabaseRequest, DeleteDatabaseResponse>()
                    .withOperationName("DeleteDatabase").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).discoveredEndpoint(cachedEndpoint)
                    .withInput(deleteDatabaseRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DeleteDatabaseRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deletes a given Timestream table. This is an irreversible operation. After a Timestream database table is
     * deleted, the time-series data stored in the table cannot be recovered.
     * </p>
     * <note>
     * <p>
     * Due to the nature of distributed retries, the operation can return either success or a ResourceNotFoundException.
     * Clients should consider them equivalent.
     * </p>
     * </note>
     * <p>
     * See <a href="https://docs.aws.amazon.com/timestream/latest/developerguide/code-samples.delete-table.html">code
     * sample</a> for details.
     * </p>
     *
     * @param deleteTableRequest
     * @return Result of the DeleteTable operation returned by the service.
     * @throws InternalServerException
     *         Timestream was unable to fully process this request because of an internal server error.
     * @throws ThrottlingException
     *         Too many requests were made by a user and they exceeded the service quotas. The request was throttled.
     * @throws ValidationException
     *         An invalid or malformed request.
     * @throws ResourceNotFoundException
     *         The operation tried to access a nonexistent resource. The resource might not be specified correctly, or
     *         its status might not be ACTIVE.
     * @throws AccessDeniedException
     *         You are not authorized to perform this action.
     * @throws InvalidEndpointException
     *         The requested endpoint was not valid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws TimestreamWriteException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample TimestreamWriteClient.DeleteTable
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/timestream-write-2018-11-01/DeleteTable" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public DeleteTableResponse deleteTable(DeleteTableRequest deleteTableRequest) throws InternalServerException,
            ThrottlingException, ValidationException, ResourceNotFoundException, AccessDeniedException, InvalidEndpointException,
            AwsServiceException, SdkClientException, TimestreamWriteException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteTableResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DeleteTableResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        boolean endpointDiscoveryEnabled = clientConfiguration.option(SdkClientOption.ENDPOINT_DISCOVERY_ENABLED);
        boolean endpointOverridden = clientConfiguration.option(SdkClientOption.ENDPOINT_OVERRIDDEN) == Boolean.TRUE;
        if (endpointOverridden) {
            endpointDiscoveryEnabled = false;
        } else if (!endpointDiscoveryEnabled) {
            throw new IllegalStateException(
                    "This operation requires endpoint discovery to be enabled, or for you to specify an endpoint override when the client is created.");
        }
        URI cachedEndpoint = null;
        if (endpointDiscoveryEnabled) {
            String key = deleteTableRequest.overrideConfiguration().flatMap(AwsRequestOverrideConfiguration::credentialsProvider)
                    .orElseGet(() -> clientConfiguration.option(AwsClientOption.CREDENTIALS_PROVIDER)).resolveCredentials()
                    .accessKeyId();
            EndpointDiscoveryRequest endpointDiscoveryRequest = EndpointDiscoveryRequest.builder().required(true)
                    .defaultEndpoint(clientConfiguration.option(SdkClientOption.ENDPOINT))
                    .overrideConfiguration(deleteTableRequest.overrideConfiguration().orElse(null)).build();
            cachedEndpoint = endpointDiscoveryCache.get(key, endpointDiscoveryRequest);
        }
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteTableRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Timestream Write");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteTable");

            return clientHandler.execute(new ClientExecutionParams<DeleteTableRequest, DeleteTableResponse>()
                    .withOperationName("DeleteTable").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).discoveredEndpoint(cachedEndpoint)
                    .withInput(deleteTableRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DeleteTableRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns information about the batch load task, including configurations, mappings, progress, and other details.
     * <a href="https://docs.aws.amazon.com/timestream/latest/developerguide/ts-limits.html">Service quotas apply</a>.
     * See <a
     * href="https://docs.aws.amazon.com/timestream/latest/developerguide/code-samples.describe-batch-load.html">code
     * sample</a> for details.
     * </p>
     *
     * @param describeBatchLoadTaskRequest
     * @return Result of the DescribeBatchLoadTask operation returned by the service.
     * @throws InternalServerException
     *         Timestream was unable to fully process this request because of an internal server error.
     * @throws ThrottlingException
     *         Too many requests were made by a user and they exceeded the service quotas. The request was throttled.
     * @throws AccessDeniedException
     *         You are not authorized to perform this action.
     * @throws ResourceNotFoundException
     *         The operation tried to access a nonexistent resource. The resource might not be specified correctly, or
     *         its status might not be ACTIVE.
     * @throws InvalidEndpointException
     *         The requested endpoint was not valid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws TimestreamWriteException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample TimestreamWriteClient.DescribeBatchLoadTask
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/timestream-write-2018-11-01/DescribeBatchLoadTask"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DescribeBatchLoadTaskResponse describeBatchLoadTask(DescribeBatchLoadTaskRequest describeBatchLoadTaskRequest)
            throws InternalServerException, ThrottlingException, AccessDeniedException, ResourceNotFoundException,
            InvalidEndpointException, AwsServiceException, SdkClientException, TimestreamWriteException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeBatchLoadTaskResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DescribeBatchLoadTaskResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        boolean endpointDiscoveryEnabled = clientConfiguration.option(SdkClientOption.ENDPOINT_DISCOVERY_ENABLED);
        boolean endpointOverridden = clientConfiguration.option(SdkClientOption.ENDPOINT_OVERRIDDEN) == Boolean.TRUE;
        if (endpointOverridden) {
            endpointDiscoveryEnabled = false;
        } else if (!endpointDiscoveryEnabled) {
            throw new IllegalStateException(
                    "This operation requires endpoint discovery to be enabled, or for you to specify an endpoint override when the client is created.");
        }
        URI cachedEndpoint = null;
        if (endpointDiscoveryEnabled) {
            String key = describeBatchLoadTaskRequest.overrideConfiguration()
                    .flatMap(AwsRequestOverrideConfiguration::credentialsProvider)
                    .orElseGet(() -> clientConfiguration.option(AwsClientOption.CREDENTIALS_PROVIDER)).resolveCredentials()
                    .accessKeyId();
            EndpointDiscoveryRequest endpointDiscoveryRequest = EndpointDiscoveryRequest.builder().required(true)
                    .defaultEndpoint(clientConfiguration.option(SdkClientOption.ENDPOINT))
                    .overrideConfiguration(describeBatchLoadTaskRequest.overrideConfiguration().orElse(null)).build();
            cachedEndpoint = endpointDiscoveryCache.get(key, endpointDiscoveryRequest);
        }
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeBatchLoadTaskRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Timestream Write");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeBatchLoadTask");

            return clientHandler.execute(new ClientExecutionParams<DescribeBatchLoadTaskRequest, DescribeBatchLoadTaskResponse>()
                    .withOperationName("DescribeBatchLoadTask").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).discoveredEndpoint(cachedEndpoint)
                    .withInput(describeBatchLoadTaskRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DescribeBatchLoadTaskRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns information about the database, including the database name, time that the database was created, and the
     * total number of tables found within the database. <a
     * href="https://docs.aws.amazon.com/timestream/latest/developerguide/ts-limits.html">Service quotas apply</a>. See
     * <a href="https://docs.aws.amazon.com/timestream/latest/developerguide/code-samples.describe-db.html">code
     * sample</a> for details.
     * </p>
     *
     * @param describeDatabaseRequest
     * @return Result of the DescribeDatabase operation returned by the service.
     * @throws ResourceNotFoundException
     *         The operation tried to access a nonexistent resource. The resource might not be specified correctly, or
     *         its status might not be ACTIVE.
     * @throws ValidationException
     *         An invalid or malformed request.
     * @throws AccessDeniedException
     *         You are not authorized to perform this action.
     * @throws ThrottlingException
     *         Too many requests were made by a user and they exceeded the service quotas. The request was throttled.
     * @throws InternalServerException
     *         Timestream was unable to fully process this request because of an internal server error.
     * @throws InvalidEndpointException
     *         The requested endpoint was not valid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws TimestreamWriteException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample TimestreamWriteClient.DescribeDatabase
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/timestream-write-2018-11-01/DescribeDatabase"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DescribeDatabaseResponse describeDatabase(DescribeDatabaseRequest describeDatabaseRequest)
            throws ResourceNotFoundException, ValidationException, AccessDeniedException, ThrottlingException,
            InternalServerException, InvalidEndpointException, AwsServiceException, SdkClientException, TimestreamWriteException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeDatabaseResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DescribeDatabaseResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        boolean endpointDiscoveryEnabled = clientConfiguration.option(SdkClientOption.ENDPOINT_DISCOVERY_ENABLED);
        boolean endpointOverridden = clientConfiguration.option(SdkClientOption.ENDPOINT_OVERRIDDEN) == Boolean.TRUE;
        if (endpointOverridden) {
            endpointDiscoveryEnabled = false;
        } else if (!endpointDiscoveryEnabled) {
            throw new IllegalStateException(
                    "This operation requires endpoint discovery to be enabled, or for you to specify an endpoint override when the client is created.");
        }
        URI cachedEndpoint = null;
        if (endpointDiscoveryEnabled) {
            String key = describeDatabaseRequest.overrideConfiguration()
                    .flatMap(AwsRequestOverrideConfiguration::credentialsProvider)
                    .orElseGet(() -> clientConfiguration.option(AwsClientOption.CREDENTIALS_PROVIDER)).resolveCredentials()
                    .accessKeyId();
            EndpointDiscoveryRequest endpointDiscoveryRequest = EndpointDiscoveryRequest.builder().required(true)
                    .defaultEndpoint(clientConfiguration.option(SdkClientOption.ENDPOINT))
                    .overrideConfiguration(describeDatabaseRequest.overrideConfiguration().orElse(null)).build();
            cachedEndpoint = endpointDiscoveryCache.get(key, endpointDiscoveryRequest);
        }
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeDatabaseRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Timestream Write");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeDatabase");

            return clientHandler.execute(new ClientExecutionParams<DescribeDatabaseRequest, DescribeDatabaseResponse>()
                    .withOperationName("DescribeDatabase").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).discoveredEndpoint(cachedEndpoint)
                    .withInput(describeDatabaseRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DescribeDatabaseRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns a list of available endpoints to make Timestream API calls against. This API operation is available
     * through both the Write and Query APIs.
     * </p>
     * <p>
     * Because the Timestream SDKs are designed to transparently work with the service’s architecture, including the
     * management and mapping of the service endpoints, <i>we don't recommend that you use this API operation
     * unless</i>:
     * </p>
     * <ul>
     * <li>
     * <p>
     * You are using <a href="https://docs.aws.amazon.com/timestream/latest/developerguide/VPCEndpoints">VPC endpoints
     * (Amazon Web Services PrivateLink) with Timestream</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * Your application uses a programming language that does not yet have SDK support
     * </p>
     * </li>
     * <li>
     * <p>
     * You require better control over the client-side implementation
     * </p>
     * </li>
     * </ul>
     * <p>
     * For detailed information on how and when to use and implement DescribeEndpoints, see <a
     * href="https://docs.aws.amazon.com/timestream/latest/developerguide/Using.API.html#Using-API.endpoint-discovery"
     * >The Endpoint Discovery Pattern</a>.
     * </p>
     *
     * @param describeEndpointsRequest
     * @return Result of the DescribeEndpoints operation returned by the service.
     * @throws InternalServerException
     *         Timestream was unable to fully process this request because of an internal server error.
     * @throws ValidationException
     *         An invalid or malformed request.
     * @throws ThrottlingException
     *         Too many requests were made by a user and they exceeded the service quotas. The request was throttled.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws TimestreamWriteException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample TimestreamWriteClient.DescribeEndpoints
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/timestream-write-2018-11-01/DescribeEndpoints"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DescribeEndpointsResponse describeEndpoints(DescribeEndpointsRequest describeEndpointsRequest)
            throws InternalServerException, ValidationException, ThrottlingException, AwsServiceException, SdkClientException,
            TimestreamWriteException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeEndpointsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DescribeEndpointsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeEndpointsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Timestream Write");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeEndpoints");

            return clientHandler.execute(new ClientExecutionParams<DescribeEndpointsRequest, DescribeEndpointsResponse>()
                    .withOperationName("DescribeEndpoints").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(describeEndpointsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DescribeEndpointsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns information about the table, including the table name, database name, retention duration of the memory
     * store and the magnetic store. <a
     * href="https://docs.aws.amazon.com/timestream/latest/developerguide/ts-limits.html">Service quotas apply</a>. See
     * <a href="https://docs.aws.amazon.com/timestream/latest/developerguide/code-samples.describe-table.html">code
     * sample</a> for details.
     * </p>
     *
     * @param describeTableRequest
     * @return Result of the DescribeTable operation returned by the service.
     * @throws ResourceNotFoundException
     *         The operation tried to access a nonexistent resource. The resource might not be specified correctly, or
     *         its status might not be ACTIVE.
     * @throws ValidationException
     *         An invalid or malformed request.
     * @throws AccessDeniedException
     *         You are not authorized to perform this action.
     * @throws ThrottlingException
     *         Too many requests were made by a user and they exceeded the service quotas. The request was throttled.
     * @throws InternalServerException
     *         Timestream was unable to fully process this request because of an internal server error.
     * @throws InvalidEndpointException
     *         The requested endpoint was not valid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws TimestreamWriteException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample TimestreamWriteClient.DescribeTable
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/timestream-write-2018-11-01/DescribeTable"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DescribeTableResponse describeTable(DescribeTableRequest describeTableRequest) throws ResourceNotFoundException,
            ValidationException, AccessDeniedException, ThrottlingException, InternalServerException, InvalidEndpointException,
            AwsServiceException, SdkClientException, TimestreamWriteException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeTableResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DescribeTableResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        boolean endpointDiscoveryEnabled = clientConfiguration.option(SdkClientOption.ENDPOINT_DISCOVERY_ENABLED);
        boolean endpointOverridden = clientConfiguration.option(SdkClientOption.ENDPOINT_OVERRIDDEN) == Boolean.TRUE;
        if (endpointOverridden) {
            endpointDiscoveryEnabled = false;
        } else if (!endpointDiscoveryEnabled) {
            throw new IllegalStateException(
                    "This operation requires endpoint discovery to be enabled, or for you to specify an endpoint override when the client is created.");
        }
        URI cachedEndpoint = null;
        if (endpointDiscoveryEnabled) {
            String key = describeTableRequest.overrideConfiguration()
                    .flatMap(AwsRequestOverrideConfiguration::credentialsProvider)
                    .orElseGet(() -> clientConfiguration.option(AwsClientOption.CREDENTIALS_PROVIDER)).resolveCredentials()
                    .accessKeyId();
            EndpointDiscoveryRequest endpointDiscoveryRequest = EndpointDiscoveryRequest.builder().required(true)
                    .defaultEndpoint(clientConfiguration.option(SdkClientOption.ENDPOINT))
                    .overrideConfiguration(describeTableRequest.overrideConfiguration().orElse(null)).build();
            cachedEndpoint = endpointDiscoveryCache.get(key, endpointDiscoveryRequest);
        }
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeTableRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Timestream Write");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeTable");

            return clientHandler.execute(new ClientExecutionParams<DescribeTableRequest, DescribeTableResponse>()
                    .withOperationName("DescribeTable").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).discoveredEndpoint(cachedEndpoint)
                    .withInput(describeTableRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DescribeTableRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Provides a list of batch load tasks, along with the name, status, when the task is resumable until, and other
     * details. See <a
     * href="https://docs.aws.amazon.com/timestream/latest/developerguide/code-samples.list-batch-load-tasks.html">code
     * sample</a> for details.
     * </p>
     *
     * @param listBatchLoadTasksRequest
     * @return Result of the ListBatchLoadTasks operation returned by the service.
     * @throws InternalServerException
     *         Timestream was unable to fully process this request because of an internal server error.
     * @throws AccessDeniedException
     *         You are not authorized to perform this action.
     * @throws ThrottlingException
     *         Too many requests were made by a user and they exceeded the service quotas. The request was throttled.
     * @throws ValidationException
     *         An invalid or malformed request.
     * @throws InvalidEndpointException
     *         The requested endpoint was not valid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws TimestreamWriteException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample TimestreamWriteClient.ListBatchLoadTasks
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/timestream-write-2018-11-01/ListBatchLoadTasks"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListBatchLoadTasksResponse listBatchLoadTasks(ListBatchLoadTasksRequest listBatchLoadTasksRequest)
            throws InternalServerException, AccessDeniedException, ThrottlingException, ValidationException,
            InvalidEndpointException, AwsServiceException, SdkClientException, TimestreamWriteException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListBatchLoadTasksResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListBatchLoadTasksResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        boolean endpointDiscoveryEnabled = clientConfiguration.option(SdkClientOption.ENDPOINT_DISCOVERY_ENABLED);
        boolean endpointOverridden = clientConfiguration.option(SdkClientOption.ENDPOINT_OVERRIDDEN) == Boolean.TRUE;
        if (endpointOverridden) {
            endpointDiscoveryEnabled = false;
        } else if (!endpointDiscoveryEnabled) {
            throw new IllegalStateException(
                    "This operation requires endpoint discovery to be enabled, or for you to specify an endpoint override when the client is created.");
        }
        URI cachedEndpoint = null;
        if (endpointDiscoveryEnabled) {
            String key = listBatchLoadTasksRequest.overrideConfiguration()
                    .flatMap(AwsRequestOverrideConfiguration::credentialsProvider)
                    .orElseGet(() -> clientConfiguration.option(AwsClientOption.CREDENTIALS_PROVIDER)).resolveCredentials()
                    .accessKeyId();
            EndpointDiscoveryRequest endpointDiscoveryRequest = EndpointDiscoveryRequest.builder().required(true)
                    .defaultEndpoint(clientConfiguration.option(SdkClientOption.ENDPOINT))
                    .overrideConfiguration(listBatchLoadTasksRequest.overrideConfiguration().orElse(null)).build();
            cachedEndpoint = endpointDiscoveryCache.get(key, endpointDiscoveryRequest);
        }
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listBatchLoadTasksRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Timestream Write");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListBatchLoadTasks");

            return clientHandler.execute(new ClientExecutionParams<ListBatchLoadTasksRequest, ListBatchLoadTasksResponse>()
                    .withOperationName("ListBatchLoadTasks").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).discoveredEndpoint(cachedEndpoint)
                    .withInput(listBatchLoadTasksRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListBatchLoadTasksRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Provides a list of batch load tasks, along with the name, status, when the task is resumable until, and other
     * details. See <a
     * href="https://docs.aws.amazon.com/timestream/latest/developerguide/code-samples.list-batch-load-tasks.html">code
     * sample</a> for details.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listBatchLoadTasks(software.amazon.awssdk.services.timestreamwrite.model.ListBatchLoadTasksRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.timestreamwrite.paginators.ListBatchLoadTasksIterable responses = client.listBatchLoadTasksPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.timestreamwrite.paginators.ListBatchLoadTasksIterable responses = client
     *             .listBatchLoadTasksPaginator(request);
     *     for (software.amazon.awssdk.services.timestreamwrite.model.ListBatchLoadTasksResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.timestreamwrite.paginators.ListBatchLoadTasksIterable responses = client.listBatchLoadTasksPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listBatchLoadTasks(software.amazon.awssdk.services.timestreamwrite.model.ListBatchLoadTasksRequest)}
     * operation.</b>
     * </p>
     *
     * @param listBatchLoadTasksRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws InternalServerException
     *         Timestream was unable to fully process this request because of an internal server error.
     * @throws AccessDeniedException
     *         You are not authorized to perform this action.
     * @throws ThrottlingException
     *         Too many requests were made by a user and they exceeded the service quotas. The request was throttled.
     * @throws ValidationException
     *         An invalid or malformed request.
     * @throws InvalidEndpointException
     *         The requested endpoint was not valid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws TimestreamWriteException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample TimestreamWriteClient.ListBatchLoadTasks
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/timestream-write-2018-11-01/ListBatchLoadTasks"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListBatchLoadTasksIterable listBatchLoadTasksPaginator(ListBatchLoadTasksRequest listBatchLoadTasksRequest)
            throws InternalServerException, AccessDeniedException, ThrottlingException, ValidationException,
            InvalidEndpointException, AwsServiceException, SdkClientException, TimestreamWriteException {
        return new ListBatchLoadTasksIterable(this, applyPaginatorUserAgent(listBatchLoadTasksRequest));
    }

    /**
     * <p>
     * Returns a list of your Timestream databases. <a
     * href="https://docs.aws.amazon.com/timestream/latest/developerguide/ts-limits.html">Service quotas apply</a>. See
     * <a href="https://docs.aws.amazon.com/timestream/latest/developerguide/code-samples.list-db.html">code sample</a>
     * for details.
     * </p>
     *
     * @param listDatabasesRequest
     * @return Result of the ListDatabases operation returned by the service.
     * @throws InternalServerException
     *         Timestream was unable to fully process this request because of an internal server error.
     * @throws ThrottlingException
     *         Too many requests were made by a user and they exceeded the service quotas. The request was throttled.
     * @throws ValidationException
     *         An invalid or malformed request.
     * @throws AccessDeniedException
     *         You are not authorized to perform this action.
     * @throws InvalidEndpointException
     *         The requested endpoint was not valid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws TimestreamWriteException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample TimestreamWriteClient.ListDatabases
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/timestream-write-2018-11-01/ListDatabases"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListDatabasesResponse listDatabases(ListDatabasesRequest listDatabasesRequest) throws InternalServerException,
            ThrottlingException, ValidationException, AccessDeniedException, InvalidEndpointException, AwsServiceException,
            SdkClientException, TimestreamWriteException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListDatabasesResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListDatabasesResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        boolean endpointDiscoveryEnabled = clientConfiguration.option(SdkClientOption.ENDPOINT_DISCOVERY_ENABLED);
        boolean endpointOverridden = clientConfiguration.option(SdkClientOption.ENDPOINT_OVERRIDDEN) == Boolean.TRUE;
        if (endpointOverridden) {
            endpointDiscoveryEnabled = false;
        } else if (!endpointDiscoveryEnabled) {
            throw new IllegalStateException(
                    "This operation requires endpoint discovery to be enabled, or for you to specify an endpoint override when the client is created.");
        }
        URI cachedEndpoint = null;
        if (endpointDiscoveryEnabled) {
            String key = listDatabasesRequest.overrideConfiguration()
                    .flatMap(AwsRequestOverrideConfiguration::credentialsProvider)
                    .orElseGet(() -> clientConfiguration.option(AwsClientOption.CREDENTIALS_PROVIDER)).resolveCredentials()
                    .accessKeyId();
            EndpointDiscoveryRequest endpointDiscoveryRequest = EndpointDiscoveryRequest.builder().required(true)
                    .defaultEndpoint(clientConfiguration.option(SdkClientOption.ENDPOINT))
                    .overrideConfiguration(listDatabasesRequest.overrideConfiguration().orElse(null)).build();
            cachedEndpoint = endpointDiscoveryCache.get(key, endpointDiscoveryRequest);
        }
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listDatabasesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Timestream Write");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListDatabases");

            return clientHandler.execute(new ClientExecutionParams<ListDatabasesRequest, ListDatabasesResponse>()
                    .withOperationName("ListDatabases").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).discoveredEndpoint(cachedEndpoint)
                    .withInput(listDatabasesRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListDatabasesRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns a list of your Timestream databases. <a
     * href="https://docs.aws.amazon.com/timestream/latest/developerguide/ts-limits.html">Service quotas apply</a>. See
     * <a href="https://docs.aws.amazon.com/timestream/latest/developerguide/code-samples.list-db.html">code sample</a>
     * for details.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listDatabases(software.amazon.awssdk.services.timestreamwrite.model.ListDatabasesRequest)} operation. The
     * return type is a custom iterable that can be used to iterate through all the pages. SDK will internally handle
     * making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.timestreamwrite.paginators.ListDatabasesIterable responses = client.listDatabasesPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.timestreamwrite.paginators.ListDatabasesIterable responses = client
     *             .listDatabasesPaginator(request);
     *     for (software.amazon.awssdk.services.timestreamwrite.model.ListDatabasesResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.timestreamwrite.paginators.ListDatabasesIterable responses = client.listDatabasesPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listDatabases(software.amazon.awssdk.services.timestreamwrite.model.ListDatabasesRequest)} operation.</b>
     * </p>
     *
     * @param listDatabasesRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws InternalServerException
     *         Timestream was unable to fully process this request because of an internal server error.
     * @throws ThrottlingException
     *         Too many requests were made by a user and they exceeded the service quotas. The request was throttled.
     * @throws ValidationException
     *         An invalid or malformed request.
     * @throws AccessDeniedException
     *         You are not authorized to perform this action.
     * @throws InvalidEndpointException
     *         The requested endpoint was not valid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws TimestreamWriteException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample TimestreamWriteClient.ListDatabases
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/timestream-write-2018-11-01/ListDatabases"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListDatabasesIterable listDatabasesPaginator(ListDatabasesRequest listDatabasesRequest)
            throws InternalServerException, ThrottlingException, ValidationException, AccessDeniedException,
            InvalidEndpointException, AwsServiceException, SdkClientException, TimestreamWriteException {
        return new ListDatabasesIterable(this, applyPaginatorUserAgent(listDatabasesRequest));
    }

    /**
     * <p>
     * Provides a list of tables, along with the name, status, and retention properties of each table. See <a
     * href="https://docs.aws.amazon.com/timestream/latest/developerguide/code-samples.list-table.html">code sample</a>
     * for details.
     * </p>
     *
     * @param listTablesRequest
     * @return Result of the ListTables operation returned by the service.
     * @throws InternalServerException
     *         Timestream was unable to fully process this request because of an internal server error.
     * @throws ThrottlingException
     *         Too many requests were made by a user and they exceeded the service quotas. The request was throttled.
     * @throws ValidationException
     *         An invalid or malformed request.
     * @throws ResourceNotFoundException
     *         The operation tried to access a nonexistent resource. The resource might not be specified correctly, or
     *         its status might not be ACTIVE.
     * @throws AccessDeniedException
     *         You are not authorized to perform this action.
     * @throws InvalidEndpointException
     *         The requested endpoint was not valid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws TimestreamWriteException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample TimestreamWriteClient.ListTables
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/timestream-write-2018-11-01/ListTables" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public ListTablesResponse listTables(ListTablesRequest listTablesRequest) throws InternalServerException,
            ThrottlingException, ValidationException, ResourceNotFoundException, AccessDeniedException, InvalidEndpointException,
            AwsServiceException, SdkClientException, TimestreamWriteException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListTablesResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListTablesResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        boolean endpointDiscoveryEnabled = clientConfiguration.option(SdkClientOption.ENDPOINT_DISCOVERY_ENABLED);
        boolean endpointOverridden = clientConfiguration.option(SdkClientOption.ENDPOINT_OVERRIDDEN) == Boolean.TRUE;
        if (endpointOverridden) {
            endpointDiscoveryEnabled = false;
        } else if (!endpointDiscoveryEnabled) {
            throw new IllegalStateException(
                    "This operation requires endpoint discovery to be enabled, or for you to specify an endpoint override when the client is created.");
        }
        URI cachedEndpoint = null;
        if (endpointDiscoveryEnabled) {
            String key = listTablesRequest.overrideConfiguration().flatMap(AwsRequestOverrideConfiguration::credentialsProvider)
                    .orElseGet(() -> clientConfiguration.option(AwsClientOption.CREDENTIALS_PROVIDER)).resolveCredentials()
                    .accessKeyId();
            EndpointDiscoveryRequest endpointDiscoveryRequest = EndpointDiscoveryRequest.builder().required(true)
                    .defaultEndpoint(clientConfiguration.option(SdkClientOption.ENDPOINT))
                    .overrideConfiguration(listTablesRequest.overrideConfiguration().orElse(null)).build();
            cachedEndpoint = endpointDiscoveryCache.get(key, endpointDiscoveryRequest);
        }
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTablesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Timestream Write");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTables");

            return clientHandler.execute(new ClientExecutionParams<ListTablesRequest, ListTablesResponse>()
                    .withOperationName("ListTables").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).discoveredEndpoint(cachedEndpoint)
                    .withInput(listTablesRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListTablesRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Provides a list of tables, along with the name, status, and retention properties of each table. See <a
     * href="https://docs.aws.amazon.com/timestream/latest/developerguide/code-samples.list-table.html">code sample</a>
     * for details.
     * </p>
     * <br/>
     * <p>
     * This is a variant of {@link #listTables(software.amazon.awssdk.services.timestreamwrite.model.ListTablesRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.timestreamwrite.paginators.ListTablesIterable responses = client.listTablesPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.timestreamwrite.paginators.ListTablesIterable responses = client.listTablesPaginator(request);
     *     for (software.amazon.awssdk.services.timestreamwrite.model.ListTablesResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.timestreamwrite.paginators.ListTablesIterable responses = client.listTablesPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listTables(software.amazon.awssdk.services.timestreamwrite.model.ListTablesRequest)} operation.</b>
     * </p>
     *
     * @param listTablesRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws InternalServerException
     *         Timestream was unable to fully process this request because of an internal server error.
     * @throws ThrottlingException
     *         Too many requests were made by a user and they exceeded the service quotas. The request was throttled.
     * @throws ValidationException
     *         An invalid or malformed request.
     * @throws ResourceNotFoundException
     *         The operation tried to access a nonexistent resource. The resource might not be specified correctly, or
     *         its status might not be ACTIVE.
     * @throws AccessDeniedException
     *         You are not authorized to perform this action.
     * @throws InvalidEndpointException
     *         The requested endpoint was not valid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws TimestreamWriteException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample TimestreamWriteClient.ListTables
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/timestream-write-2018-11-01/ListTables" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public ListTablesIterable listTablesPaginator(ListTablesRequest listTablesRequest) throws InternalServerException,
            ThrottlingException, ValidationException, ResourceNotFoundException, AccessDeniedException, InvalidEndpointException,
            AwsServiceException, SdkClientException, TimestreamWriteException {
        return new ListTablesIterable(this, applyPaginatorUserAgent(listTablesRequest));
    }

    /**
     * <p>
     * Lists all tags on a Timestream resource.
     * </p>
     *
     * @param listTagsForResourceRequest
     * @return Result of the ListTagsForResource operation returned by the service.
     * @throws ResourceNotFoundException
     *         The operation tried to access a nonexistent resource. The resource might not be specified correctly, or
     *         its status might not be ACTIVE.
     * @throws ThrottlingException
     *         Too many requests were made by a user and they exceeded the service quotas. The request was throttled.
     * @throws ValidationException
     *         An invalid or malformed request.
     * @throws InvalidEndpointException
     *         The requested endpoint was not valid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws TimestreamWriteException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample TimestreamWriteClient.ListTagsForResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/timestream-write-2018-11-01/ListTagsForResource"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListTagsForResourceResponse listTagsForResource(ListTagsForResourceRequest listTagsForResourceRequest)
            throws ResourceNotFoundException, ThrottlingException, ValidationException, InvalidEndpointException,
            AwsServiceException, SdkClientException, TimestreamWriteException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListTagsForResourceResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListTagsForResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        boolean endpointDiscoveryEnabled = clientConfiguration.option(SdkClientOption.ENDPOINT_DISCOVERY_ENABLED);
        boolean endpointOverridden = clientConfiguration.option(SdkClientOption.ENDPOINT_OVERRIDDEN) == Boolean.TRUE;
        if (endpointOverridden) {
            endpointDiscoveryEnabled = false;
        } else if (!endpointDiscoveryEnabled) {
            throw new IllegalStateException(
                    "This operation requires endpoint discovery to be enabled, or for you to specify an endpoint override when the client is created.");
        }
        URI cachedEndpoint = null;
        if (endpointDiscoveryEnabled) {
            String key = listTagsForResourceRequest.overrideConfiguration()
                    .flatMap(AwsRequestOverrideConfiguration::credentialsProvider)
                    .orElseGet(() -> clientConfiguration.option(AwsClientOption.CREDENTIALS_PROVIDER)).resolveCredentials()
                    .accessKeyId();
            EndpointDiscoveryRequest endpointDiscoveryRequest = EndpointDiscoveryRequest.builder().required(true)
                    .defaultEndpoint(clientConfiguration.option(SdkClientOption.ENDPOINT))
                    .overrideConfiguration(listTagsForResourceRequest.overrideConfiguration().orElse(null)).build();
            cachedEndpoint = endpointDiscoveryCache.get(key, endpointDiscoveryRequest);
        }
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTagsForResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Timestream Write");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTagsForResource");

            return clientHandler.execute(new ClientExecutionParams<ListTagsForResourceRequest, ListTagsForResourceResponse>()
                    .withOperationName("ListTagsForResource").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).discoveredEndpoint(cachedEndpoint)
                    .withInput(listTagsForResourceRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListTagsForResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * </p>
     *
     * @param resumeBatchLoadTaskRequest
     * @return Result of the ResumeBatchLoadTask operation returned by the service.
     * @throws InternalServerException
     *         Timestream was unable to fully process this request because of an internal server error.
     * @throws ThrottlingException
     *         Too many requests were made by a user and they exceeded the service quotas. The request was throttled.
     * @throws AccessDeniedException
     *         You are not authorized to perform this action.
     * @throws ValidationException
     *         An invalid or malformed request.
     * @throws ResourceNotFoundException
     *         The operation tried to access a nonexistent resource. The resource might not be specified correctly, or
     *         its status might not be ACTIVE.
     * @throws InvalidEndpointException
     *         The requested endpoint was not valid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws TimestreamWriteException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample TimestreamWriteClient.ResumeBatchLoadTask
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/timestream-write-2018-11-01/ResumeBatchLoadTask"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ResumeBatchLoadTaskResponse resumeBatchLoadTask(ResumeBatchLoadTaskRequest resumeBatchLoadTaskRequest)
            throws InternalServerException, ThrottlingException, AccessDeniedException, ValidationException,
            ResourceNotFoundException, InvalidEndpointException, AwsServiceException, SdkClientException,
            TimestreamWriteException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ResumeBatchLoadTaskResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ResumeBatchLoadTaskResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        boolean endpointDiscoveryEnabled = clientConfiguration.option(SdkClientOption.ENDPOINT_DISCOVERY_ENABLED);
        boolean endpointOverridden = clientConfiguration.option(SdkClientOption.ENDPOINT_OVERRIDDEN) == Boolean.TRUE;
        if (endpointOverridden) {
            endpointDiscoveryEnabled = false;
        } else if (!endpointDiscoveryEnabled) {
            throw new IllegalStateException(
                    "This operation requires endpoint discovery to be enabled, or for you to specify an endpoint override when the client is created.");
        }
        URI cachedEndpoint = null;
        if (endpointDiscoveryEnabled) {
            String key = resumeBatchLoadTaskRequest.overrideConfiguration()
                    .flatMap(AwsRequestOverrideConfiguration::credentialsProvider)
                    .orElseGet(() -> clientConfiguration.option(AwsClientOption.CREDENTIALS_PROVIDER)).resolveCredentials()
                    .accessKeyId();
            EndpointDiscoveryRequest endpointDiscoveryRequest = EndpointDiscoveryRequest.builder().required(true)
                    .defaultEndpoint(clientConfiguration.option(SdkClientOption.ENDPOINT))
                    .overrideConfiguration(resumeBatchLoadTaskRequest.overrideConfiguration().orElse(null)).build();
            cachedEndpoint = endpointDiscoveryCache.get(key, endpointDiscoveryRequest);
        }
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, resumeBatchLoadTaskRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Timestream Write");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ResumeBatchLoadTask");

            return clientHandler.execute(new ClientExecutionParams<ResumeBatchLoadTaskRequest, ResumeBatchLoadTaskResponse>()
                    .withOperationName("ResumeBatchLoadTask").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).discoveredEndpoint(cachedEndpoint)
                    .withInput(resumeBatchLoadTaskRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ResumeBatchLoadTaskRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Associates a set of tags with a Timestream resource. You can then activate these user-defined tags so that they
     * appear on the Billing and Cost Management console for cost allocation tracking.
     * </p>
     *
     * @param tagResourceRequest
     * @return Result of the TagResource operation returned by the service.
     * @throws ResourceNotFoundException
     *         The operation tried to access a nonexistent resource. The resource might not be specified correctly, or
     *         its status might not be ACTIVE.
     * @throws ServiceQuotaExceededException
     *         The instance quota of resource exceeded for this account.
     * @throws ThrottlingException
     *         Too many requests were made by a user and they exceeded the service quotas. The request was throttled.
     * @throws ValidationException
     *         An invalid or malformed request.
     * @throws InvalidEndpointException
     *         The requested endpoint was not valid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws TimestreamWriteException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample TimestreamWriteClient.TagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/timestream-write-2018-11-01/TagResource" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public TagResourceResponse tagResource(TagResourceRequest tagResourceRequest) throws ResourceNotFoundException,
            ServiceQuotaExceededException, ThrottlingException, ValidationException, InvalidEndpointException,
            AwsServiceException, SdkClientException, TimestreamWriteException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<TagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                TagResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        boolean endpointDiscoveryEnabled = clientConfiguration.option(SdkClientOption.ENDPOINT_DISCOVERY_ENABLED);
        boolean endpointOverridden = clientConfiguration.option(SdkClientOption.ENDPOINT_OVERRIDDEN) == Boolean.TRUE;
        if (endpointOverridden) {
            endpointDiscoveryEnabled = false;
        } else if (!endpointDiscoveryEnabled) {
            throw new IllegalStateException(
                    "This operation requires endpoint discovery to be enabled, or for you to specify an endpoint override when the client is created.");
        }
        URI cachedEndpoint = null;
        if (endpointDiscoveryEnabled) {
            String key = tagResourceRequest.overrideConfiguration().flatMap(AwsRequestOverrideConfiguration::credentialsProvider)
                    .orElseGet(() -> clientConfiguration.option(AwsClientOption.CREDENTIALS_PROVIDER)).resolveCredentials()
                    .accessKeyId();
            EndpointDiscoveryRequest endpointDiscoveryRequest = EndpointDiscoveryRequest.builder().required(true)
                    .defaultEndpoint(clientConfiguration.option(SdkClientOption.ENDPOINT))
                    .overrideConfiguration(tagResourceRequest.overrideConfiguration().orElse(null)).build();
            cachedEndpoint = endpointDiscoveryCache.get(key, endpointDiscoveryRequest);
        }
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, tagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Timestream Write");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "TagResource");

            return clientHandler.execute(new ClientExecutionParams<TagResourceRequest, TagResourceResponse>()
                    .withOperationName("TagResource").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).discoveredEndpoint(cachedEndpoint)
                    .withInput(tagResourceRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new TagResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Removes the association of tags from a Timestream resource.
     * </p>
     *
     * @param untagResourceRequest
     * @return Result of the UntagResource operation returned by the service.
     * @throws ValidationException
     *         An invalid or malformed request.
     * @throws ServiceQuotaExceededException
     *         The instance quota of resource exceeded for this account.
     * @throws ThrottlingException
     *         Too many requests were made by a user and they exceeded the service quotas. The request was throttled.
     * @throws ResourceNotFoundException
     *         The operation tried to access a nonexistent resource. The resource might not be specified correctly, or
     *         its status might not be ACTIVE.
     * @throws InvalidEndpointException
     *         The requested endpoint was not valid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws TimestreamWriteException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample TimestreamWriteClient.UntagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/timestream-write-2018-11-01/UntagResource"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public UntagResourceResponse untagResource(UntagResourceRequest untagResourceRequest) throws ValidationException,
            ServiceQuotaExceededException, ThrottlingException, ResourceNotFoundException, InvalidEndpointException,
            AwsServiceException, SdkClientException, TimestreamWriteException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UntagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                UntagResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        boolean endpointDiscoveryEnabled = clientConfiguration.option(SdkClientOption.ENDPOINT_DISCOVERY_ENABLED);
        boolean endpointOverridden = clientConfiguration.option(SdkClientOption.ENDPOINT_OVERRIDDEN) == Boolean.TRUE;
        if (endpointOverridden) {
            endpointDiscoveryEnabled = false;
        } else if (!endpointDiscoveryEnabled) {
            throw new IllegalStateException(
                    "This operation requires endpoint discovery to be enabled, or for you to specify an endpoint override when the client is created.");
        }
        URI cachedEndpoint = null;
        if (endpointDiscoveryEnabled) {
            String key = untagResourceRequest.overrideConfiguration()
                    .flatMap(AwsRequestOverrideConfiguration::credentialsProvider)
                    .orElseGet(() -> clientConfiguration.option(AwsClientOption.CREDENTIALS_PROVIDER)).resolveCredentials()
                    .accessKeyId();
            EndpointDiscoveryRequest endpointDiscoveryRequest = EndpointDiscoveryRequest.builder().required(true)
                    .defaultEndpoint(clientConfiguration.option(SdkClientOption.ENDPOINT))
                    .overrideConfiguration(untagResourceRequest.overrideConfiguration().orElse(null)).build();
            cachedEndpoint = endpointDiscoveryCache.get(key, endpointDiscoveryRequest);
        }
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, untagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Timestream Write");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UntagResource");

            return clientHandler.execute(new ClientExecutionParams<UntagResourceRequest, UntagResourceResponse>()
                    .withOperationName("UntagResource").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).discoveredEndpoint(cachedEndpoint)
                    .withInput(untagResourceRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UntagResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Modifies the KMS key for an existing database. While updating the database, you must specify the database name
     * and the identifier of the new KMS key to be used (<code>KmsKeyId</code>). If there are any concurrent
     * <code>UpdateDatabase</code> requests, first writer wins.
     * </p>
     * <p>
     * See <a href="https://docs.aws.amazon.com/timestream/latest/developerguide/code-samples.update-db.html">code
     * sample</a> for details.
     * </p>
     *
     * @param updateDatabaseRequest
     * @return Result of the UpdateDatabase operation returned by the service.
     * @throws ValidationException
     *         An invalid or malformed request.
     * @throws AccessDeniedException
     *         You are not authorized to perform this action.
     * @throws ResourceNotFoundException
     *         The operation tried to access a nonexistent resource. The resource might not be specified correctly, or
     *         its status might not be ACTIVE.
     * @throws ServiceQuotaExceededException
     *         The instance quota of resource exceeded for this account.
     * @throws ThrottlingException
     *         Too many requests were made by a user and they exceeded the service quotas. The request was throttled.
     * @throws InternalServerException
     *         Timestream was unable to fully process this request because of an internal server error.
     * @throws InvalidEndpointException
     *         The requested endpoint was not valid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws TimestreamWriteException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample TimestreamWriteClient.UpdateDatabase
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/timestream-write-2018-11-01/UpdateDatabase"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public UpdateDatabaseResponse updateDatabase(UpdateDatabaseRequest updateDatabaseRequest) throws ValidationException,
            AccessDeniedException, ResourceNotFoundException, ServiceQuotaExceededException, ThrottlingException,
            InternalServerException, InvalidEndpointException, AwsServiceException, SdkClientException, TimestreamWriteException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateDatabaseResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                UpdateDatabaseResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        boolean endpointDiscoveryEnabled = clientConfiguration.option(SdkClientOption.ENDPOINT_DISCOVERY_ENABLED);
        boolean endpointOverridden = clientConfiguration.option(SdkClientOption.ENDPOINT_OVERRIDDEN) == Boolean.TRUE;
        if (endpointOverridden) {
            endpointDiscoveryEnabled = false;
        } else if (!endpointDiscoveryEnabled) {
            throw new IllegalStateException(
                    "This operation requires endpoint discovery to be enabled, or for you to specify an endpoint override when the client is created.");
        }
        URI cachedEndpoint = null;
        if (endpointDiscoveryEnabled) {
            String key = updateDatabaseRequest.overrideConfiguration()
                    .flatMap(AwsRequestOverrideConfiguration::credentialsProvider)
                    .orElseGet(() -> clientConfiguration.option(AwsClientOption.CREDENTIALS_PROVIDER)).resolveCredentials()
                    .accessKeyId();
            EndpointDiscoveryRequest endpointDiscoveryRequest = EndpointDiscoveryRequest.builder().required(true)
                    .defaultEndpoint(clientConfiguration.option(SdkClientOption.ENDPOINT))
                    .overrideConfiguration(updateDatabaseRequest.overrideConfiguration().orElse(null)).build();
            cachedEndpoint = endpointDiscoveryCache.get(key, endpointDiscoveryRequest);
        }
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateDatabaseRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Timestream Write");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateDatabase");

            return clientHandler.execute(new ClientExecutionParams<UpdateDatabaseRequest, UpdateDatabaseResponse>()
                    .withOperationName("UpdateDatabase").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).discoveredEndpoint(cachedEndpoint)
                    .withInput(updateDatabaseRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UpdateDatabaseRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Modifies the retention duration of the memory store and magnetic store for your Timestream table. Note that the
     * change in retention duration takes effect immediately. For example, if the retention period of the memory store
     * was initially set to 2 hours and then changed to 24 hours, the memory store will be capable of holding 24 hours
     * of data, but will be populated with 24 hours of data 22 hours after this change was made. Timestream does not
     * retrieve data from the magnetic store to populate the memory store.
     * </p>
     * <p>
     * See <a href="https://docs.aws.amazon.com/timestream/latest/developerguide/code-samples.update-table.html">code
     * sample</a> for details.
     * </p>
     *
     * @param updateTableRequest
     * @return Result of the UpdateTable operation returned by the service.
     * @throws InternalServerException
     *         Timestream was unable to fully process this request because of an internal server error.
     * @throws ThrottlingException
     *         Too many requests were made by a user and they exceeded the service quotas. The request was throttled.
     * @throws ValidationException
     *         An invalid or malformed request.
     * @throws ResourceNotFoundException
     *         The operation tried to access a nonexistent resource. The resource might not be specified correctly, or
     *         its status might not be ACTIVE.
     * @throws AccessDeniedException
     *         You are not authorized to perform this action.
     * @throws InvalidEndpointException
     *         The requested endpoint was not valid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws TimestreamWriteException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample TimestreamWriteClient.UpdateTable
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/timestream-write-2018-11-01/UpdateTable" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public UpdateTableResponse updateTable(UpdateTableRequest updateTableRequest) throws InternalServerException,
            ThrottlingException, ValidationException, ResourceNotFoundException, AccessDeniedException, InvalidEndpointException,
            AwsServiceException, SdkClientException, TimestreamWriteException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateTableResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                UpdateTableResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        boolean endpointDiscoveryEnabled = clientConfiguration.option(SdkClientOption.ENDPOINT_DISCOVERY_ENABLED);
        boolean endpointOverridden = clientConfiguration.option(SdkClientOption.ENDPOINT_OVERRIDDEN) == Boolean.TRUE;
        if (endpointOverridden) {
            endpointDiscoveryEnabled = false;
        } else if (!endpointDiscoveryEnabled) {
            throw new IllegalStateException(
                    "This operation requires endpoint discovery to be enabled, or for you to specify an endpoint override when the client is created.");
        }
        URI cachedEndpoint = null;
        if (endpointDiscoveryEnabled) {
            String key = updateTableRequest.overrideConfiguration().flatMap(AwsRequestOverrideConfiguration::credentialsProvider)
                    .orElseGet(() -> clientConfiguration.option(AwsClientOption.CREDENTIALS_PROVIDER)).resolveCredentials()
                    .accessKeyId();
            EndpointDiscoveryRequest endpointDiscoveryRequest = EndpointDiscoveryRequest.builder().required(true)
                    .defaultEndpoint(clientConfiguration.option(SdkClientOption.ENDPOINT))
                    .overrideConfiguration(updateTableRequest.overrideConfiguration().orElse(null)).build();
            cachedEndpoint = endpointDiscoveryCache.get(key, endpointDiscoveryRequest);
        }
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateTableRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Timestream Write");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateTable");

            return clientHandler.execute(new ClientExecutionParams<UpdateTableRequest, UpdateTableResponse>()
                    .withOperationName("UpdateTable").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).discoveredEndpoint(cachedEndpoint)
                    .withInput(updateTableRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UpdateTableRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Enables you to write your time-series data into Timestream. You can specify a single data point or a batch of
     * data points to be inserted into the system. Timestream offers you a flexible schema that auto detects the column
     * names and data types for your Timestream tables based on the dimension names and data types of the data points
     * you specify when invoking writes into the database.
     * </p>
     * <p>
     * Timestream supports eventual consistency read semantics. This means that when you query data immediately after
     * writing a batch of data into Timestream, the query results might not reflect the results of a recently completed
     * write operation. The results may also include some stale data. If you repeat the query request after a short
     * time, the results should return the latest data. <a
     * href="https://docs.aws.amazon.com/timestream/latest/developerguide/ts-limits.html">Service quotas apply</a>.
     * </p>
     * <p>
     * See <a href="https://docs.aws.amazon.com/timestream/latest/developerguide/code-samples.write.html">code
     * sample</a> for details.
     * </p>
     * <p>
     * <b>Upserts</b>
     * </p>
     * <p>
     * You can use the <code>Version</code> parameter in a <code>WriteRecords</code> request to update data points.
     * Timestream tracks a version number with each record. <code>Version</code> defaults to <code>1</code> when it's
     * not specified for the record in the request. Timestream updates an existing record’s measure value along with its
     * <code>Version</code> when it receives a write request with a higher <code>Version</code> number for that record.
     * When it receives an update request where the measure value is the same as that of the existing record, Timestream
     * still updates <code>Version</code>, if it is greater than the existing value of <code>Version</code>. You can
     * update a data point as many times as desired, as long as the value of <code>Version</code> continuously
     * increases.
     * </p>
     * <p>
     * For example, suppose you write a new record without indicating <code>Version</code> in the request. Timestream
     * stores this record, and set <code>Version</code> to <code>1</code>. Now, suppose you try to update this record
     * with a <code>WriteRecords</code> request of the same record with a different measure value but, like before, do
     * not provide <code>Version</code>. In this case, Timestream will reject this update with a
     * <code>RejectedRecordsException</code> since the updated record’s version is not greater than the existing value
     * of Version.
     * </p>
     * <p>
     * However, if you were to resend the update request with <code>Version</code> set to <code>2</code>, Timestream
     * would then succeed in updating the record’s value, and the <code>Version</code> would be set to <code>2</code>.
     * Next, suppose you sent a <code>WriteRecords</code> request with this same record and an identical measure value,
     * but with <code>Version</code> set to <code>3</code>. In this case, Timestream would only update
     * <code>Version</code> to <code>3</code>. Any further updates would need to send a version number greater than
     * <code>3</code>, or the update requests would receive a <code>RejectedRecordsException</code>.
     * </p>
     *
     * @param writeRecordsRequest
     * @return Result of the WriteRecords operation returned by the service.
     * @throws InternalServerException
     *         Timestream was unable to fully process this request because of an internal server error.
     * @throws ThrottlingException
     *         Too many requests were made by a user and they exceeded the service quotas. The request was throttled.
     * @throws ValidationException
     *         An invalid or malformed request.
     * @throws ResourceNotFoundException
     *         The operation tried to access a nonexistent resource. The resource might not be specified correctly, or
     *         its status might not be ACTIVE.
     * @throws AccessDeniedException
     *         You are not authorized to perform this action.
     * @throws RejectedRecordsException
     *         WriteRecords would throw this exception in the following cases: </p>
     *         <ul>
     *         <li>
     *         <p>
     *         Records with duplicate data where there are multiple records with the same dimensions, timestamps, and
     *         measure names but:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         Measure values are different
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Version is not present in the request <i>or</i> the value of version in the new record is equal to or
     *         lower than the existing value
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         In this case, if Timestream rejects data, the <code>ExistingVersion</code> field in the
     *         <code>RejectedRecords</code> response will indicate the current record’s version. To force an update, you
     *         can resend the request with a version for the record set to a value greater than the
     *         <code>ExistingVersion</code>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Records with timestamps that lie outside the retention duration of the memory store.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Records with dimensions or measures that exceed the Timestream defined limits.
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         For more information, see <a
     *         href="https://docs.aws.amazon.com/timestream/latest/developerguide/ts-limits.html">Quotas</a> in the
     *         Amazon Timestream Developer Guide.
     * @throws InvalidEndpointException
     *         The requested endpoint was not valid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws TimestreamWriteException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample TimestreamWriteClient.WriteRecords
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/timestream-write-2018-11-01/WriteRecords" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public WriteRecordsResponse writeRecords(WriteRecordsRequest writeRecordsRequest) throws InternalServerException,
            ThrottlingException, ValidationException, ResourceNotFoundException, AccessDeniedException, RejectedRecordsException,
            InvalidEndpointException, AwsServiceException, SdkClientException, TimestreamWriteException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<WriteRecordsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                WriteRecordsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        boolean endpointDiscoveryEnabled = clientConfiguration.option(SdkClientOption.ENDPOINT_DISCOVERY_ENABLED);
        boolean endpointOverridden = clientConfiguration.option(SdkClientOption.ENDPOINT_OVERRIDDEN) == Boolean.TRUE;
        if (endpointOverridden) {
            endpointDiscoveryEnabled = false;
        } else if (!endpointDiscoveryEnabled) {
            throw new IllegalStateException(
                    "This operation requires endpoint discovery to be enabled, or for you to specify an endpoint override when the client is created.");
        }
        URI cachedEndpoint = null;
        if (endpointDiscoveryEnabled) {
            String key = writeRecordsRequest.overrideConfiguration()
                    .flatMap(AwsRequestOverrideConfiguration::credentialsProvider)
                    .orElseGet(() -> clientConfiguration.option(AwsClientOption.CREDENTIALS_PROVIDER)).resolveCredentials()
                    .accessKeyId();
            EndpointDiscoveryRequest endpointDiscoveryRequest = EndpointDiscoveryRequest.builder().required(true)
                    .defaultEndpoint(clientConfiguration.option(SdkClientOption.ENDPOINT))
                    .overrideConfiguration(writeRecordsRequest.overrideConfiguration().orElse(null)).build();
            cachedEndpoint = endpointDiscoveryCache.get(key, endpointDiscoveryRequest);
        }
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, writeRecordsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Timestream Write");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "WriteRecords");

            return clientHandler.execute(new ClientExecutionParams<WriteRecordsRequest, WriteRecordsResponse>()
                    .withOperationName("WriteRecords").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).discoveredEndpoint(cachedEndpoint)
                    .withInput(writeRecordsRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new WriteRecordsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    private <T extends TimestreamWriteRequest> T applyPaginatorUserAgent(T request) {
        Consumer<AwsRequestOverrideConfiguration.Builder> userAgentApplier = b -> b.addApiName(ApiName.builder()
                .version(VersionInfo.SDK_VERSION).name("PAGINATED").build());
        AwsRequestOverrideConfiguration overrideConfiguration = request.overrideConfiguration()
                .map(c -> c.toBuilder().applyMutation(userAgentApplier).build())
                .orElse((AwsRequestOverrideConfiguration.builder().applyMutation(userAgentApplier).build()));
        return (T) request.toBuilder().overrideConfiguration(overrideConfiguration).build();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(TimestreamWriteException::builder)
                .protocol(AwsJsonProtocol.AWS_JSON)
                .protocolVersion("1.0")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("AccessDeniedException")
                                .exceptionBuilderSupplier(AccessDeniedException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ConflictException")
                                .exceptionBuilderSupplier(ConflictException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidEndpointException")
                                .exceptionBuilderSupplier(InvalidEndpointException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceNotFoundException")
                                .exceptionBuilderSupplier(ResourceNotFoundException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ThrottlingException")
                                .exceptionBuilderSupplier(ThrottlingException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ValidationException")
                                .exceptionBuilderSupplier(ValidationException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException")
                                .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalServerException")
                                .exceptionBuilderSupplier(InternalServerException::builder).httpStatusCode(500).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("RejectedRecordsException")
                                .exceptionBuilderSupplier(RejectedRecordsException::builder).httpStatusCode(400).build());
    }

    @Override
    public final TimestreamWriteServiceClientConfiguration serviceClientConfiguration() {
        return this.serviceClientConfiguration;
    }

    @Override
    public void close() {
        clientHandler.close();
    }
}
