/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.secretsmanager;

import java.util.Collections;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.function.Consumer;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.awscore.client.handler.AwsAsyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.ApiName;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.AsyncClientHandler;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.core.util.VersionInfo;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.secretsmanager.model.CancelRotateSecretRequest;
import software.amazon.awssdk.services.secretsmanager.model.CancelRotateSecretResponse;
import software.amazon.awssdk.services.secretsmanager.model.CreateSecretRequest;
import software.amazon.awssdk.services.secretsmanager.model.CreateSecretResponse;
import software.amazon.awssdk.services.secretsmanager.model.DecryptionFailureException;
import software.amazon.awssdk.services.secretsmanager.model.DeleteResourcePolicyRequest;
import software.amazon.awssdk.services.secretsmanager.model.DeleteResourcePolicyResponse;
import software.amazon.awssdk.services.secretsmanager.model.DeleteSecretRequest;
import software.amazon.awssdk.services.secretsmanager.model.DeleteSecretResponse;
import software.amazon.awssdk.services.secretsmanager.model.DescribeSecretRequest;
import software.amazon.awssdk.services.secretsmanager.model.DescribeSecretResponse;
import software.amazon.awssdk.services.secretsmanager.model.EncryptionFailureException;
import software.amazon.awssdk.services.secretsmanager.model.GetRandomPasswordRequest;
import software.amazon.awssdk.services.secretsmanager.model.GetRandomPasswordResponse;
import software.amazon.awssdk.services.secretsmanager.model.GetResourcePolicyRequest;
import software.amazon.awssdk.services.secretsmanager.model.GetResourcePolicyResponse;
import software.amazon.awssdk.services.secretsmanager.model.GetSecretValueRequest;
import software.amazon.awssdk.services.secretsmanager.model.GetSecretValueResponse;
import software.amazon.awssdk.services.secretsmanager.model.InternalServiceErrorException;
import software.amazon.awssdk.services.secretsmanager.model.InvalidNextTokenException;
import software.amazon.awssdk.services.secretsmanager.model.InvalidParameterException;
import software.amazon.awssdk.services.secretsmanager.model.InvalidRequestException;
import software.amazon.awssdk.services.secretsmanager.model.LimitExceededException;
import software.amazon.awssdk.services.secretsmanager.model.ListSecretVersionIdsRequest;
import software.amazon.awssdk.services.secretsmanager.model.ListSecretVersionIdsResponse;
import software.amazon.awssdk.services.secretsmanager.model.ListSecretsRequest;
import software.amazon.awssdk.services.secretsmanager.model.ListSecretsResponse;
import software.amazon.awssdk.services.secretsmanager.model.MalformedPolicyDocumentException;
import software.amazon.awssdk.services.secretsmanager.model.PreconditionNotMetException;
import software.amazon.awssdk.services.secretsmanager.model.PublicPolicyException;
import software.amazon.awssdk.services.secretsmanager.model.PutResourcePolicyRequest;
import software.amazon.awssdk.services.secretsmanager.model.PutResourcePolicyResponse;
import software.amazon.awssdk.services.secretsmanager.model.PutSecretValueRequest;
import software.amazon.awssdk.services.secretsmanager.model.PutSecretValueResponse;
import software.amazon.awssdk.services.secretsmanager.model.RemoveRegionsFromReplicationRequest;
import software.amazon.awssdk.services.secretsmanager.model.RemoveRegionsFromReplicationResponse;
import software.amazon.awssdk.services.secretsmanager.model.ReplicateSecretToRegionsRequest;
import software.amazon.awssdk.services.secretsmanager.model.ReplicateSecretToRegionsResponse;
import software.amazon.awssdk.services.secretsmanager.model.ResourceExistsException;
import software.amazon.awssdk.services.secretsmanager.model.ResourceNotFoundException;
import software.amazon.awssdk.services.secretsmanager.model.RestoreSecretRequest;
import software.amazon.awssdk.services.secretsmanager.model.RestoreSecretResponse;
import software.amazon.awssdk.services.secretsmanager.model.RotateSecretRequest;
import software.amazon.awssdk.services.secretsmanager.model.RotateSecretResponse;
import software.amazon.awssdk.services.secretsmanager.model.SecretsManagerException;
import software.amazon.awssdk.services.secretsmanager.model.SecretsManagerRequest;
import software.amazon.awssdk.services.secretsmanager.model.StopReplicationToReplicaRequest;
import software.amazon.awssdk.services.secretsmanager.model.StopReplicationToReplicaResponse;
import software.amazon.awssdk.services.secretsmanager.model.TagResourceRequest;
import software.amazon.awssdk.services.secretsmanager.model.TagResourceResponse;
import software.amazon.awssdk.services.secretsmanager.model.UntagResourceRequest;
import software.amazon.awssdk.services.secretsmanager.model.UntagResourceResponse;
import software.amazon.awssdk.services.secretsmanager.model.UpdateSecretRequest;
import software.amazon.awssdk.services.secretsmanager.model.UpdateSecretResponse;
import software.amazon.awssdk.services.secretsmanager.model.UpdateSecretVersionStageRequest;
import software.amazon.awssdk.services.secretsmanager.model.UpdateSecretVersionStageResponse;
import software.amazon.awssdk.services.secretsmanager.model.ValidateResourcePolicyRequest;
import software.amazon.awssdk.services.secretsmanager.model.ValidateResourcePolicyResponse;
import software.amazon.awssdk.services.secretsmanager.paginators.ListSecretVersionIdsPublisher;
import software.amazon.awssdk.services.secretsmanager.paginators.ListSecretsPublisher;
import software.amazon.awssdk.services.secretsmanager.transform.CancelRotateSecretRequestMarshaller;
import software.amazon.awssdk.services.secretsmanager.transform.CreateSecretRequestMarshaller;
import software.amazon.awssdk.services.secretsmanager.transform.DeleteResourcePolicyRequestMarshaller;
import software.amazon.awssdk.services.secretsmanager.transform.DeleteSecretRequestMarshaller;
import software.amazon.awssdk.services.secretsmanager.transform.DescribeSecretRequestMarshaller;
import software.amazon.awssdk.services.secretsmanager.transform.GetRandomPasswordRequestMarshaller;
import software.amazon.awssdk.services.secretsmanager.transform.GetResourcePolicyRequestMarshaller;
import software.amazon.awssdk.services.secretsmanager.transform.GetSecretValueRequestMarshaller;
import software.amazon.awssdk.services.secretsmanager.transform.ListSecretVersionIdsRequestMarshaller;
import software.amazon.awssdk.services.secretsmanager.transform.ListSecretsRequestMarshaller;
import software.amazon.awssdk.services.secretsmanager.transform.PutResourcePolicyRequestMarshaller;
import software.amazon.awssdk.services.secretsmanager.transform.PutSecretValueRequestMarshaller;
import software.amazon.awssdk.services.secretsmanager.transform.RemoveRegionsFromReplicationRequestMarshaller;
import software.amazon.awssdk.services.secretsmanager.transform.ReplicateSecretToRegionsRequestMarshaller;
import software.amazon.awssdk.services.secretsmanager.transform.RestoreSecretRequestMarshaller;
import software.amazon.awssdk.services.secretsmanager.transform.RotateSecretRequestMarshaller;
import software.amazon.awssdk.services.secretsmanager.transform.StopReplicationToReplicaRequestMarshaller;
import software.amazon.awssdk.services.secretsmanager.transform.TagResourceRequestMarshaller;
import software.amazon.awssdk.services.secretsmanager.transform.UntagResourceRequestMarshaller;
import software.amazon.awssdk.services.secretsmanager.transform.UpdateSecretRequestMarshaller;
import software.amazon.awssdk.services.secretsmanager.transform.UpdateSecretVersionStageRequestMarshaller;
import software.amazon.awssdk.services.secretsmanager.transform.ValidateResourcePolicyRequestMarshaller;
import software.amazon.awssdk.utils.CompletableFutureUtils;

/**
 * Internal implementation of {@link SecretsManagerAsyncClient}.
 *
 * @see SecretsManagerAsyncClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultSecretsManagerAsyncClient implements SecretsManagerAsyncClient {
    private static final Logger log = LoggerFactory.getLogger(DefaultSecretsManagerAsyncClient.class);

    private final AsyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultSecretsManagerAsyncClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsAsyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    /**
     * <p>
     * Turns off automatic rotation, and if a rotation is currently in progress, cancels the rotation.
     * </p>
     * <p>
     * If you cancel a rotation in progress, it can leave the <code>VersionStage</code> labels in an unexpected state.
     * You might need to remove the staging label <code>AWSPENDING</code> from the partially created version. You also
     * need to determine whether to roll back to the previous version of the secret by moving the staging label
     * <code>AWSCURRENT</code> to the version that has <code>AWSPENDING</code>. To determine which version has a
     * specific staging label, call <a>ListSecretVersionIds</a>. Then use <a>UpdateSecretVersionStage</a> to change
     * staging labels. For more information, see <a
     * href="https://docs.aws.amazon.com/secretsmanager/latest/userguide/rotate-secrets_how.html">How rotation
     * works</a>.
     * </p>
     * <p>
     * To turn on automatic rotation again, call <a>RotateSecret</a>.
     * </p>
     * <p>
     * <b>Required permissions: </b> <code>secretsmanager:CancelRotateSecret</code>. For more information, see <a href=
     * "https://docs.aws.amazon.com/secretsmanager/latest/userguide/reference_iam-permissions.html#reference_iam-permissions_actions"
     * > IAM policy actions for Secrets Manager</a> and <a
     * href="https://docs.aws.amazon.com/secretsmanager/latest/userguide/auth-and-access.html">Authentication and access
     * control in Secrets Manager</a>.
     * </p>
     *
     * @param cancelRotateSecretRequest
     * @return A Java Future containing the result of the CancelRotateSecret operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException Secrets Manager can't find the resource that you asked for.</li>
     *         <li>InvalidParameterException The parameter name or value is invalid.</li>
     *         <li>InternalServiceErrorException An error occurred on the server side.</li>
     *         <li>InvalidRequestException A parameter value is not valid for the current state of the resource.</p>
     *         <p>
     *         Possible causes:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         The secret is scheduled for deletion.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         You tried to enable rotation on a secret that doesn't already have a Lambda function ARN configured and
     *         you didn't include such an ARN as a parameter in this call.
     *         </p>
     *         </li></li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SecretsManagerException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample SecretsManagerAsyncClient.CancelRotateSecret
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/secretsmanager-2017-10-17/CancelRotateSecret"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<CancelRotateSecretResponse> cancelRotateSecret(CancelRotateSecretRequest cancelRotateSecretRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, cancelRotateSecretRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Secrets Manager");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CancelRotateSecret");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CancelRotateSecretResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CancelRotateSecretResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CancelRotateSecretResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CancelRotateSecretRequest, CancelRotateSecretResponse>()
                            .withOperationName("CancelRotateSecret")
                            .withMarshaller(new CancelRotateSecretRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(cancelRotateSecretRequest));
            CompletableFuture<CancelRotateSecretResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a new secret. A <i>secret</i> can be a password, a set of credentials such as a user name and password,
     * an OAuth token, or other secret information that you store in an encrypted form in Secrets Manager. The secret
     * also includes the connection information to access a database or other service, which Secrets Manager doesn't
     * encrypt. A secret in Secrets Manager consists of both the protected secret data and the important information
     * needed to manage the secret.
     * </p>
     * <p>
     * For information about creating a secret in the console, see <a
     * href="https://docs.aws.amazon.com/secretsmanager/latest/userguide/manage_create-basic-secret.html">Create a
     * secret</a>.
     * </p>
     * <p>
     * To create a secret, you can provide the secret value to be encrypted in either the <code>SecretString</code>
     * parameter or the <code>SecretBinary</code> parameter, but not both. If you include <code>SecretString</code> or
     * <code>SecretBinary</code> then Secrets Manager creates an initial secret version and automatically attaches the
     * staging label <code>AWSCURRENT</code> to it.
     * </p>
     * <p>
     * For database credentials you want to rotate, for Secrets Manager to be able to rotate the secret, you must make
     * sure the JSON you store in the <code>SecretString</code> matches the <a
     * href="https://docs.aws.amazon.com/secretsmanager/latest/userguide/reference_secret_json_structure.html">JSON
     * structure of a database secret</a>.
     * </p>
     * <p>
     * If you don't specify an KMS encryption key, Secrets Manager uses the Amazon Web Services managed key
     * <code>aws/secretsmanager</code>. If this key doesn't already exist in your account, then Secrets Manager creates
     * it for you automatically. All users and roles in the Amazon Web Services account automatically have access to use
     * <code>aws/secretsmanager</code>. Creating <code>aws/secretsmanager</code> can result in a one-time significant
     * delay in returning the result.
     * </p>
     * <p>
     * If the secret is in a different Amazon Web Services account from the credentials calling the API, then you can't
     * use <code>aws/secretsmanager</code> to encrypt the secret, and you must create and use a customer managed KMS
     * key.
     * </p>
     * <p>
     * <b>Required permissions: </b> <code>secretsmanager:CreateSecret</code>. If you include tags in the secret, you
     * also need <code>secretsmanager:TagResource</code>. For more information, see <a href=
     * "https://docs.aws.amazon.com/secretsmanager/latest/userguide/reference_iam-permissions.html#reference_iam-permissions_actions"
     * > IAM policy actions for Secrets Manager</a> and <a
     * href="https://docs.aws.amazon.com/secretsmanager/latest/userguide/auth-and-access.html">Authentication and access
     * control in Secrets Manager</a>.
     * </p>
     * <p>
     * To encrypt the secret with a KMS key other than <code>aws/secretsmanager</code>, you need
     * <code>kms:GenerateDataKey</code> and <code>kms:Decrypt</code> permission to the key.
     * </p>
     *
     * @param createSecretRequest
     * @return A Java Future containing the result of the CreateSecret operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidParameterException The parameter name or value is invalid.</li>
     *         <li>InvalidRequestException A parameter value is not valid for the current state of the resource.</p>
     *         <p>
     *         Possible causes:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         The secret is scheduled for deletion.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         You tried to enable rotation on a secret that doesn't already have a Lambda function ARN configured and
     *         you didn't include such an ARN as a parameter in this call.
     *         </p>
     *         </li></li>
     *         <li>LimitExceededException The request failed because it would exceed one of the Secrets Manager quotas.</li>
     *         <li>EncryptionFailureException Secrets Manager can't encrypt the protected secret text using the provided
     *         KMS key. Check that the KMS key is available, enabled, and not in an invalid state. For more information,
     *         see <a href="https://docs.aws.amazon.com/kms/latest/developerguide/key-state.html">Key state: Effect on
     *         your KMS key</a>.</li>
     *         <li>ResourceExistsException A resource with the ID you requested already exists.</li>
     *         <li>ResourceNotFoundException Secrets Manager can't find the resource that you asked for.</li>
     *         <li>MalformedPolicyDocumentException The resource policy has syntax errors.</li>
     *         <li>InternalServiceErrorException An error occurred on the server side.</li>
     *         <li>PreconditionNotMetException The request failed because you did not complete all the prerequisite
     *         steps.</li>
     *         <li>DecryptionFailureException Secrets Manager can't decrypt the protected secret text using the provided
     *         KMS key.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SecretsManagerException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample SecretsManagerAsyncClient.CreateSecret
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/secretsmanager-2017-10-17/CreateSecret" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateSecretResponse> createSecret(CreateSecretRequest createSecretRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createSecretRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Secrets Manager");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateSecret");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateSecretResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    CreateSecretResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateSecretResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateSecretRequest, CreateSecretResponse>()
                            .withOperationName("CreateSecret").withMarshaller(new CreateSecretRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(createSecretRequest));
            CompletableFuture<CreateSecretResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes the resource-based permission policy attached to the secret. To attach a policy to a secret, use
     * <a>PutResourcePolicy</a>.
     * </p>
     * <p>
     * <b>Required permissions: </b> <code>secretsmanager:DeleteResourcePolicy</code>. For more information, see <a
     * href=
     * "https://docs.aws.amazon.com/secretsmanager/latest/userguide/reference_iam-permissions.html#reference_iam-permissions_actions"
     * > IAM policy actions for Secrets Manager</a> and <a
     * href="https://docs.aws.amazon.com/secretsmanager/latest/userguide/auth-and-access.html">Authentication and access
     * control in Secrets Manager</a>.
     * </p>
     *
     * @param deleteResourcePolicyRequest
     * @return A Java Future containing the result of the DeleteResourcePolicy operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException Secrets Manager can't find the resource that you asked for.</li>
     *         <li>InternalServiceErrorException An error occurred on the server side.</li>
     *         <li>InvalidRequestException A parameter value is not valid for the current state of the resource.</p>
     *         <p>
     *         Possible causes:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         The secret is scheduled for deletion.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         You tried to enable rotation on a secret that doesn't already have a Lambda function ARN configured and
     *         you didn't include such an ARN as a parameter in this call.
     *         </p>
     *         </li></li>
     *         <li>InvalidParameterException The parameter name or value is invalid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SecretsManagerException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample SecretsManagerAsyncClient.DeleteResourcePolicy
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/secretsmanager-2017-10-17/DeleteResourcePolicy"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteResourcePolicyResponse> deleteResourcePolicy(
            DeleteResourcePolicyRequest deleteResourcePolicyRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteResourcePolicyRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Secrets Manager");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteResourcePolicy");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteResourcePolicyResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteResourcePolicyResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteResourcePolicyResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteResourcePolicyRequest, DeleteResourcePolicyResponse>()
                            .withOperationName("DeleteResourcePolicy")
                            .withMarshaller(new DeleteResourcePolicyRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(deleteResourcePolicyRequest));
            CompletableFuture<DeleteResourcePolicyResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes a secret and all of its versions. You can specify a recovery window during which you can restore the
     * secret. The minimum recovery window is 7 days. The default recovery window is 30 days. Secrets Manager attaches a
     * <code>DeletionDate</code> stamp to the secret that specifies the end of the recovery window. At the end of the
     * recovery window, Secrets Manager deletes the secret permanently.
     * </p>
     * <p>
     * You can't delete a primary secret that is replicated to other Regions. You must first delete the replicas using
     * <a>RemoveRegionsFromReplication</a>, and then delete the primary secret. When you delete a replica, it is deleted
     * immediately.
     * </p>
     * <p>
     * You can't directly delete a version of a secret. Instead, you remove all staging labels from the version using
     * <a>UpdateSecretVersionStage</a>. This marks the version as deprecated, and then Secrets Manager can automatically
     * delete the version in the background.
     * </p>
     * <p>
     * To determine whether an application still uses a secret, you can create an Amazon CloudWatch alarm to alert you
     * to any attempts to access a secret during the recovery window. For more information, see <a
     * href="https://docs.aws.amazon.com/secretsmanager/latest/userguide/monitoring_cloudwatch_deleted-secrets.html">
     * Monitor secrets scheduled for deletion</a>.
     * </p>
     * <p>
     * Secrets Manager performs the permanent secret deletion at the end of the waiting period as a background task with
     * low priority. There is no guarantee of a specific time after the recovery window for the permanent delete to
     * occur.
     * </p>
     * <p>
     * At any time before recovery window ends, you can use <a>RestoreSecret</a> to remove the <code>DeletionDate</code>
     * and cancel the deletion of the secret.
     * </p>
     * <p>
     * When a secret is scheduled for deletion, you cannot retrieve the secret value. You must first cancel the deletion
     * with <a>RestoreSecret</a> and then you can retrieve the secret.
     * </p>
     * <p>
     * <b>Required permissions: </b> <code>secretsmanager:DeleteSecret</code>. For more information, see <a href=
     * "https://docs.aws.amazon.com/secretsmanager/latest/userguide/reference_iam-permissions.html#reference_iam-permissions_actions"
     * > IAM policy actions for Secrets Manager</a> and <a
     * href="https://docs.aws.amazon.com/secretsmanager/latest/userguide/auth-and-access.html">Authentication and access
     * control in Secrets Manager</a>.
     * </p>
     *
     * @param deleteSecretRequest
     * @return A Java Future containing the result of the DeleteSecret operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException Secrets Manager can't find the resource that you asked for.</li>
     *         <li>InvalidParameterException The parameter name or value is invalid.</li>
     *         <li>InvalidRequestException A parameter value is not valid for the current state of the resource.</p>
     *         <p>
     *         Possible causes:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         The secret is scheduled for deletion.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         You tried to enable rotation on a secret that doesn't already have a Lambda function ARN configured and
     *         you didn't include such an ARN as a parameter in this call.
     *         </p>
     *         </li></li>
     *         <li>InternalServiceErrorException An error occurred on the server side.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SecretsManagerException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample SecretsManagerAsyncClient.DeleteSecret
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/secretsmanager-2017-10-17/DeleteSecret" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteSecretResponse> deleteSecret(DeleteSecretRequest deleteSecretRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteSecretRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Secrets Manager");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteSecret");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteSecretResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    DeleteSecretResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteSecretResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteSecretRequest, DeleteSecretResponse>()
                            .withOperationName("DeleteSecret").withMarshaller(new DeleteSecretRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(deleteSecretRequest));
            CompletableFuture<DeleteSecretResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves the details of a secret. It does not include the encrypted secret value. Secrets Manager only returns
     * fields that have a value in the response.
     * </p>
     * <p>
     * <b>Required permissions: </b> <code>secretsmanager:DescribeSecret</code>. For more information, see <a href=
     * "https://docs.aws.amazon.com/secretsmanager/latest/userguide/reference_iam-permissions.html#reference_iam-permissions_actions"
     * > IAM policy actions for Secrets Manager</a> and <a
     * href="https://docs.aws.amazon.com/secretsmanager/latest/userguide/auth-and-access.html">Authentication and access
     * control in Secrets Manager</a>.
     * </p>
     *
     * @param describeSecretRequest
     * @return A Java Future containing the result of the DescribeSecret operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException Secrets Manager can't find the resource that you asked for.</li>
     *         <li>InternalServiceErrorException An error occurred on the server side.</li>
     *         <li>InvalidParameterException The parameter name or value is invalid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SecretsManagerException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample SecretsManagerAsyncClient.DescribeSecret
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/secretsmanager-2017-10-17/DescribeSecret" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeSecretResponse> describeSecret(DescribeSecretRequest describeSecretRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeSecretRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Secrets Manager");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeSecret");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeSecretResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeSecretResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeSecretResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeSecretRequest, DescribeSecretResponse>()
                            .withOperationName("DescribeSecret")
                            .withMarshaller(new DescribeSecretRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(describeSecretRequest));
            CompletableFuture<DescribeSecretResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Generates a random password. We recommend that you specify the maximum length and include every character type
     * that the system you are generating a password for can support.
     * </p>
     * <p>
     * <b>Required permissions: </b> <code>secretsmanager:GetRandomPassword</code>. For more information, see <a href=
     * "https://docs.aws.amazon.com/secretsmanager/latest/userguide/reference_iam-permissions.html#reference_iam-permissions_actions"
     * > IAM policy actions for Secrets Manager</a> and <a
     * href="https://docs.aws.amazon.com/secretsmanager/latest/userguide/auth-and-access.html">Authentication and access
     * control in Secrets Manager</a>.
     * </p>
     *
     * @param getRandomPasswordRequest
     * @return A Java Future containing the result of the GetRandomPassword operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidParameterException The parameter name or value is invalid.</li>
     *         <li>InvalidRequestException A parameter value is not valid for the current state of the resource.</p>
     *         <p>
     *         Possible causes:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         The secret is scheduled for deletion.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         You tried to enable rotation on a secret that doesn't already have a Lambda function ARN configured and
     *         you didn't include such an ARN as a parameter in this call.
     *         </p>
     *         </li></li>
     *         <li>InternalServiceErrorException An error occurred on the server side.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SecretsManagerException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample SecretsManagerAsyncClient.GetRandomPassword
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/secretsmanager-2017-10-17/GetRandomPassword"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetRandomPasswordResponse> getRandomPassword(GetRandomPasswordRequest getRandomPasswordRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getRandomPasswordRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Secrets Manager");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetRandomPassword");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetRandomPasswordResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetRandomPasswordResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetRandomPasswordResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetRandomPasswordRequest, GetRandomPasswordResponse>()
                            .withOperationName("GetRandomPassword")
                            .withMarshaller(new GetRandomPasswordRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(getRandomPasswordRequest));
            CompletableFuture<GetRandomPasswordResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves the JSON text of the resource-based policy document attached to the secret. For more information about
     * permissions policies attached to a secret, see <a
     * href="https://docs.aws.amazon.com/secretsmanager/latest/userguide/auth-and-access_resource-policies.html"
     * >Permissions policies attached to a secret</a>.
     * </p>
     * <p>
     * <b>Required permissions: </b> <code>secretsmanager:GetResourcePolicy</code>. For more information, see <a href=
     * "https://docs.aws.amazon.com/secretsmanager/latest/userguide/reference_iam-permissions.html#reference_iam-permissions_actions"
     * > IAM policy actions for Secrets Manager</a> and <a
     * href="https://docs.aws.amazon.com/secretsmanager/latest/userguide/auth-and-access.html">Authentication and access
     * control in Secrets Manager</a>.
     * </p>
     *
     * @param getResourcePolicyRequest
     * @return A Java Future containing the result of the GetResourcePolicy operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException Secrets Manager can't find the resource that you asked for.</li>
     *         <li>InternalServiceErrorException An error occurred on the server side.</li>
     *         <li>InvalidRequestException A parameter value is not valid for the current state of the resource.</p>
     *         <p>
     *         Possible causes:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         The secret is scheduled for deletion.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         You tried to enable rotation on a secret that doesn't already have a Lambda function ARN configured and
     *         you didn't include such an ARN as a parameter in this call.
     *         </p>
     *         </li></li>
     *         <li>InvalidParameterException The parameter name or value is invalid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SecretsManagerException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample SecretsManagerAsyncClient.GetResourcePolicy
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/secretsmanager-2017-10-17/GetResourcePolicy"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetResourcePolicyResponse> getResourcePolicy(GetResourcePolicyRequest getResourcePolicyRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getResourcePolicyRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Secrets Manager");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetResourcePolicy");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetResourcePolicyResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetResourcePolicyResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetResourcePolicyResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetResourcePolicyRequest, GetResourcePolicyResponse>()
                            .withOperationName("GetResourcePolicy")
                            .withMarshaller(new GetResourcePolicyRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(getResourcePolicyRequest));
            CompletableFuture<GetResourcePolicyResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves the contents of the encrypted fields <code>SecretString</code> or <code>SecretBinary</code> from the
     * specified version of a secret, whichever contains content.
     * </p>
     * <p>
     * We recommend that you cache your secret values by using client-side caching. Caching secrets improves speed and
     * reduces your costs. For more information, see <a
     * href="https://docs.aws.amazon.com/secretsmanager/latest/userguide/retrieving-secrets.html">Cache secrets for your
     * applications</a>.
     * </p>
     * <p>
     * <b>Required permissions: </b> <code>secretsmanager:GetSecretValue</code>. If the secret is encrypted using a
     * customer-managed key instead of the Amazon Web Services managed key <code>aws/secretsmanager</code>, then you
     * also need <code>kms:Decrypt</code> permissions for that key. For more information, see <a href=
     * "https://docs.aws.amazon.com/secretsmanager/latest/userguide/reference_iam-permissions.html#reference_iam-permissions_actions"
     * > IAM policy actions for Secrets Manager</a> and <a
     * href="https://docs.aws.amazon.com/secretsmanager/latest/userguide/auth-and-access.html">Authentication and access
     * control in Secrets Manager</a>.
     * </p>
     *
     * @param getSecretValueRequest
     * @return A Java Future containing the result of the GetSecretValue operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException Secrets Manager can't find the resource that you asked for.</li>
     *         <li>InvalidParameterException The parameter name or value is invalid.</li>
     *         <li>InvalidRequestException A parameter value is not valid for the current state of the resource.</p>
     *         <p>
     *         Possible causes:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         The secret is scheduled for deletion.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         You tried to enable rotation on a secret that doesn't already have a Lambda function ARN configured and
     *         you didn't include such an ARN as a parameter in this call.
     *         </p>
     *         </li></li>
     *         <li>DecryptionFailureException Secrets Manager can't decrypt the protected secret text using the provided
     *         KMS key.</li>
     *         <li>InternalServiceErrorException An error occurred on the server side.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SecretsManagerException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample SecretsManagerAsyncClient.GetSecretValue
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/secretsmanager-2017-10-17/GetSecretValue" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<GetSecretValueResponse> getSecretValue(GetSecretValueRequest getSecretValueRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getSecretValueRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Secrets Manager");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetSecretValue");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetSecretValueResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetSecretValueResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetSecretValueResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetSecretValueRequest, GetSecretValueResponse>()
                            .withOperationName("GetSecretValue")
                            .withMarshaller(new GetSecretValueRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(getSecretValueRequest));
            CompletableFuture<GetSecretValueResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists the versions of a secret. Secrets Manager uses staging labels to indicate the different versions of a
     * secret. For more information, see <a
     * href="https://docs.aws.amazon.com/secretsmanager/latest/userguide/getting-started.html#term_version"> Secrets
     * Manager concepts: Versions</a>.
     * </p>
     * <p>
     * To list the secrets in the account, use <a>ListSecrets</a>.
     * </p>
     * <p>
     * <b>Required permissions: </b> <code>secretsmanager:ListSecretVersionIds</code>. For more information, see <a
     * href=
     * "https://docs.aws.amazon.com/secretsmanager/latest/userguide/reference_iam-permissions.html#reference_iam-permissions_actions"
     * > IAM policy actions for Secrets Manager</a> and <a
     * href="https://docs.aws.amazon.com/secretsmanager/latest/userguide/auth-and-access.html">Authentication and access
     * control in Secrets Manager</a>.
     * </p>
     *
     * @param listSecretVersionIdsRequest
     * @return A Java Future containing the result of the ListSecretVersionIds operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidNextTokenException The <code>NextToken</code> value is invalid.</li>
     *         <li>ResourceNotFoundException Secrets Manager can't find the resource that you asked for.</li>
     *         <li>InternalServiceErrorException An error occurred on the server side.</li>
     *         <li>InvalidParameterException The parameter name or value is invalid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SecretsManagerException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample SecretsManagerAsyncClient.ListSecretVersionIds
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/secretsmanager-2017-10-17/ListSecretVersionIds"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListSecretVersionIdsResponse> listSecretVersionIds(
            ListSecretVersionIdsRequest listSecretVersionIdsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listSecretVersionIdsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Secrets Manager");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListSecretVersionIds");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListSecretVersionIdsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListSecretVersionIdsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListSecretVersionIdsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListSecretVersionIdsRequest, ListSecretVersionIdsResponse>()
                            .withOperationName("ListSecretVersionIds")
                            .withMarshaller(new ListSecretVersionIdsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listSecretVersionIdsRequest));
            CompletableFuture<ListSecretVersionIdsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists the versions of a secret. Secrets Manager uses staging labels to indicate the different versions of a
     * secret. For more information, see <a
     * href="https://docs.aws.amazon.com/secretsmanager/latest/userguide/getting-started.html#term_version"> Secrets
     * Manager concepts: Versions</a>.
     * </p>
     * <p>
     * To list the secrets in the account, use <a>ListSecrets</a>.
     * </p>
     * <p>
     * <b>Required permissions: </b> <code>secretsmanager:ListSecretVersionIds</code>. For more information, see <a
     * href=
     * "https://docs.aws.amazon.com/secretsmanager/latest/userguide/reference_iam-permissions.html#reference_iam-permissions_actions"
     * > IAM policy actions for Secrets Manager</a> and <a
     * href="https://docs.aws.amazon.com/secretsmanager/latest/userguide/auth-and-access.html">Authentication and access
     * control in Secrets Manager</a>.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listSecretVersionIds(software.amazon.awssdk.services.secretsmanager.model.ListSecretVersionIdsRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.secretsmanager.paginators.ListSecretVersionIdsPublisher publisher = client.listSecretVersionIdsPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.secretsmanager.paginators.ListSecretVersionIdsPublisher publisher = client.listSecretVersionIdsPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.secretsmanager.model.ListSecretVersionIdsResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.secretsmanager.model.ListSecretVersionIdsResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listSecretVersionIds(software.amazon.awssdk.services.secretsmanager.model.ListSecretVersionIdsRequest)}
     * operation.</b>
     * </p>
     *
     * @param listSecretVersionIdsRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidNextTokenException The <code>NextToken</code> value is invalid.</li>
     *         <li>ResourceNotFoundException Secrets Manager can't find the resource that you asked for.</li>
     *         <li>InternalServiceErrorException An error occurred on the server side.</li>
     *         <li>InvalidParameterException The parameter name or value is invalid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SecretsManagerException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample SecretsManagerAsyncClient.ListSecretVersionIds
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/secretsmanager-2017-10-17/ListSecretVersionIds"
     *      target="_top">AWS API Documentation</a>
     */
    public ListSecretVersionIdsPublisher listSecretVersionIdsPaginator(ListSecretVersionIdsRequest listSecretVersionIdsRequest) {
        return new ListSecretVersionIdsPublisher(this, applyPaginatorUserAgent(listSecretVersionIdsRequest));
    }

    /**
     * <p>
     * Lists the secrets that are stored by Secrets Manager in the Amazon Web Services account, not including secrets
     * that are marked for deletion. To see secrets marked for deletion, use the Secrets Manager console.
     * </p>
     * <p>
     * To list the versions of a secret, use <a>ListSecretVersionIds</a>.
     * </p>
     * <p>
     * To get the secret value from <code>SecretString</code> or <code>SecretBinary</code>, call <a>GetSecretValue</a>.
     * </p>
     * <p>
     * For information about finding secrets in the console, see <a
     * href="https://docs.aws.amazon.com/secretsmanager/latest/userguide/manage_search-secret.html">Find secrets in
     * Secrets Manager</a>.
     * </p>
     * <p>
     * <b>Required permissions: </b> <code>secretsmanager:ListSecrets</code>. For more information, see <a href=
     * "https://docs.aws.amazon.com/secretsmanager/latest/userguide/reference_iam-permissions.html#reference_iam-permissions_actions"
     * > IAM policy actions for Secrets Manager</a> and <a
     * href="https://docs.aws.amazon.com/secretsmanager/latest/userguide/auth-and-access.html">Authentication and access
     * control in Secrets Manager</a>.
     * </p>
     *
     * @param listSecretsRequest
     * @return A Java Future containing the result of the ListSecrets operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidParameterException The parameter name or value is invalid.</li>
     *         <li>InvalidNextTokenException The <code>NextToken</code> value is invalid.</li>
     *         <li>InternalServiceErrorException An error occurred on the server side.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SecretsManagerException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample SecretsManagerAsyncClient.ListSecrets
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/secretsmanager-2017-10-17/ListSecrets" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<ListSecretsResponse> listSecrets(ListSecretsRequest listSecretsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listSecretsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Secrets Manager");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListSecrets");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListSecretsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    ListSecretsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListSecretsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListSecretsRequest, ListSecretsResponse>()
                            .withOperationName("ListSecrets").withMarshaller(new ListSecretsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listSecretsRequest));
            CompletableFuture<ListSecretsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists the secrets that are stored by Secrets Manager in the Amazon Web Services account, not including secrets
     * that are marked for deletion. To see secrets marked for deletion, use the Secrets Manager console.
     * </p>
     * <p>
     * To list the versions of a secret, use <a>ListSecretVersionIds</a>.
     * </p>
     * <p>
     * To get the secret value from <code>SecretString</code> or <code>SecretBinary</code>, call <a>GetSecretValue</a>.
     * </p>
     * <p>
     * For information about finding secrets in the console, see <a
     * href="https://docs.aws.amazon.com/secretsmanager/latest/userguide/manage_search-secret.html">Find secrets in
     * Secrets Manager</a>.
     * </p>
     * <p>
     * <b>Required permissions: </b> <code>secretsmanager:ListSecrets</code>. For more information, see <a href=
     * "https://docs.aws.amazon.com/secretsmanager/latest/userguide/reference_iam-permissions.html#reference_iam-permissions_actions"
     * > IAM policy actions for Secrets Manager</a> and <a
     * href="https://docs.aws.amazon.com/secretsmanager/latest/userguide/auth-and-access.html">Authentication and access
     * control in Secrets Manager</a>.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listSecrets(software.amazon.awssdk.services.secretsmanager.model.ListSecretsRequest)} operation. The
     * return type is a custom publisher that can be subscribed to request a stream of response pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.secretsmanager.paginators.ListSecretsPublisher publisher = client.listSecretsPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.secretsmanager.paginators.ListSecretsPublisher publisher = client.listSecretsPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.secretsmanager.model.ListSecretsResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.secretsmanager.model.ListSecretsResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listSecrets(software.amazon.awssdk.services.secretsmanager.model.ListSecretsRequest)} operation.</b>
     * </p>
     *
     * @param listSecretsRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidParameterException The parameter name or value is invalid.</li>
     *         <li>InvalidNextTokenException The <code>NextToken</code> value is invalid.</li>
     *         <li>InternalServiceErrorException An error occurred on the server side.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SecretsManagerException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample SecretsManagerAsyncClient.ListSecrets
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/secretsmanager-2017-10-17/ListSecrets" target="_top">AWS
     *      API Documentation</a>
     */
    public ListSecretsPublisher listSecretsPaginator(ListSecretsRequest listSecretsRequest) {
        return new ListSecretsPublisher(this, applyPaginatorUserAgent(listSecretsRequest));
    }

    /**
     * <p>
     * Attaches a resource-based permission policy to a secret. A resource-based policy is optional. For more
     * information, see <a
     * href="https://docs.aws.amazon.com/secretsmanager/latest/userguide/auth-and-access.html">Authentication and access
     * control for Secrets Manager</a>
     * </p>
     * <p>
     * For information about attaching a policy in the console, see <a
     * href="https://docs.aws.amazon.com/secretsmanager/latest/userguide/auth-and-access_resource-based-policies.html"
     * >Attach a permissions policy to a secret</a>.
     * </p>
     * <p>
     * <b>Required permissions: </b> <code>secretsmanager:PutResourcePolicy</code>. For more information, see <a href=
     * "https://docs.aws.amazon.com/secretsmanager/latest/userguide/reference_iam-permissions.html#reference_iam-permissions_actions"
     * > IAM policy actions for Secrets Manager</a> and <a
     * href="https://docs.aws.amazon.com/secretsmanager/latest/userguide/auth-and-access.html">Authentication and access
     * control in Secrets Manager</a>.
     * </p>
     *
     * @param putResourcePolicyRequest
     * @return A Java Future containing the result of the PutResourcePolicy operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>MalformedPolicyDocumentException The resource policy has syntax errors.</li>
     *         <li>ResourceNotFoundException Secrets Manager can't find the resource that you asked for.</li>
     *         <li>InvalidParameterException The parameter name or value is invalid.</li>
     *         <li>InternalServiceErrorException An error occurred on the server side.</li>
     *         <li>InvalidRequestException A parameter value is not valid for the current state of the resource.</p>
     *         <p>
     *         Possible causes:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         The secret is scheduled for deletion.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         You tried to enable rotation on a secret that doesn't already have a Lambda function ARN configured and
     *         you didn't include such an ARN as a parameter in this call.
     *         </p>
     *         </li></li>
     *         <li>PublicPolicyException The <code>BlockPublicPolicy</code> parameter is set to true, and the resource
     *         policy did not prevent broad access to the secret.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SecretsManagerException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample SecretsManagerAsyncClient.PutResourcePolicy
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/secretsmanager-2017-10-17/PutResourcePolicy"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<PutResourcePolicyResponse> putResourcePolicy(PutResourcePolicyRequest putResourcePolicyRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, putResourcePolicyRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Secrets Manager");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "PutResourcePolicy");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<PutResourcePolicyResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, PutResourcePolicyResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<PutResourcePolicyResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<PutResourcePolicyRequest, PutResourcePolicyResponse>()
                            .withOperationName("PutResourcePolicy")
                            .withMarshaller(new PutResourcePolicyRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(putResourcePolicyRequest));
            CompletableFuture<PutResourcePolicyResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a new version with a new encrypted secret value and attaches it to the secret. The version can contain a
     * new <code>SecretString</code> value or a new <code>SecretBinary</code> value.
     * </p>
     * <p>
     * We recommend you avoid calling <code>PutSecretValue</code> at a sustained rate of more than once every 10
     * minutes. When you update the secret value, Secrets Manager creates a new version of the secret. Secrets Manager
     * removes outdated versions when there are more than 100, but it does not remove versions created less than 24
     * hours ago. If you call <code>PutSecretValue</code> more than once every 10 minutes, you create more versions than
     * Secrets Manager removes, and you will reach the quota for secret versions.
     * </p>
     * <p>
     * You can specify the staging labels to attach to the new version in <code>VersionStages</code>. If you don't
     * include <code>VersionStages</code>, then Secrets Manager automatically moves the staging label
     * <code>AWSCURRENT</code> to this version. If this operation creates the first version for the secret, then Secrets
     * Manager automatically attaches the staging label <code>AWSCURRENT</code> to it .
     * </p>
     * <p>
     * If this operation moves the staging label <code>AWSCURRENT</code> from another version to this version, then
     * Secrets Manager also automatically moves the staging label <code>AWSPREVIOUS</code> to the version that
     * <code>AWSCURRENT</code> was removed from.
     * </p>
     * <p>
     * This operation is idempotent. If a version with a <code>VersionId</code> with the same value as the
     * <code>ClientRequestToken</code> parameter already exists, and you specify the same secret data, the operation
     * succeeds but does nothing. However, if the secret data is different, then the operation fails because you can't
     * modify an existing version; you can only create new ones.
     * </p>
     * <p>
     * <b>Required permissions: </b> <code>secretsmanager:PutSecretValue</code>. For more information, see <a href=
     * "https://docs.aws.amazon.com/secretsmanager/latest/userguide/reference_iam-permissions.html#reference_iam-permissions_actions"
     * > IAM policy actions for Secrets Manager</a> and <a
     * href="https://docs.aws.amazon.com/secretsmanager/latest/userguide/auth-and-access.html">Authentication and access
     * control in Secrets Manager</a>.
     * </p>
     *
     * @param putSecretValueRequest
     * @return A Java Future containing the result of the PutSecretValue operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidParameterException The parameter name or value is invalid.</li>
     *         <li>InvalidRequestException A parameter value is not valid for the current state of the resource.</p>
     *         <p>
     *         Possible causes:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         The secret is scheduled for deletion.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         You tried to enable rotation on a secret that doesn't already have a Lambda function ARN configured and
     *         you didn't include such an ARN as a parameter in this call.
     *         </p>
     *         </li></li>
     *         <li>LimitExceededException The request failed because it would exceed one of the Secrets Manager quotas.</li>
     *         <li>EncryptionFailureException Secrets Manager can't encrypt the protected secret text using the provided
     *         KMS key. Check that the KMS key is available, enabled, and not in an invalid state. For more information,
     *         see <a href="https://docs.aws.amazon.com/kms/latest/developerguide/key-state.html">Key state: Effect on
     *         your KMS key</a>.</li>
     *         <li>ResourceExistsException A resource with the ID you requested already exists.</li>
     *         <li>ResourceNotFoundException Secrets Manager can't find the resource that you asked for.</li>
     *         <li>InternalServiceErrorException An error occurred on the server side.</li>
     *         <li>DecryptionFailureException Secrets Manager can't decrypt the protected secret text using the provided
     *         KMS key.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SecretsManagerException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample SecretsManagerAsyncClient.PutSecretValue
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/secretsmanager-2017-10-17/PutSecretValue" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<PutSecretValueResponse> putSecretValue(PutSecretValueRequest putSecretValueRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, putSecretValueRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Secrets Manager");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "PutSecretValue");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<PutSecretValueResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, PutSecretValueResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<PutSecretValueResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<PutSecretValueRequest, PutSecretValueResponse>()
                            .withOperationName("PutSecretValue")
                            .withMarshaller(new PutSecretValueRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(putSecretValueRequest));
            CompletableFuture<PutSecretValueResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * For a secret that is replicated to other Regions, deletes the secret replicas from the Regions you specify.
     * </p>
     * <p>
     * <b>Required permissions: </b> <code>secretsmanager:RemoveRegionsFromReplication</code>. For more information, see
     * <a href=
     * "https://docs.aws.amazon.com/secretsmanager/latest/userguide/reference_iam-permissions.html#reference_iam-permissions_actions"
     * > IAM policy actions for Secrets Manager</a> and <a
     * href="https://docs.aws.amazon.com/secretsmanager/latest/userguide/auth-and-access.html">Authentication and access
     * control in Secrets Manager</a>.
     * </p>
     *
     * @param removeRegionsFromReplicationRequest
     * @return A Java Future containing the result of the RemoveRegionsFromReplication operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException Secrets Manager can't find the resource that you asked for.</li>
     *         <li>InvalidRequestException A parameter value is not valid for the current state of the resource.</p>
     *         <p>
     *         Possible causes:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         The secret is scheduled for deletion.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         You tried to enable rotation on a secret that doesn't already have a Lambda function ARN configured and
     *         you didn't include such an ARN as a parameter in this call.
     *         </p>
     *         </li></li>
     *         <li>InvalidParameterException The parameter name or value is invalid.</li>
     *         <li>InternalServiceErrorException An error occurred on the server side.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SecretsManagerException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample SecretsManagerAsyncClient.RemoveRegionsFromReplication
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/secretsmanager-2017-10-17/RemoveRegionsFromReplication"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<RemoveRegionsFromReplicationResponse> removeRegionsFromReplication(
            RemoveRegionsFromReplicationRequest removeRegionsFromReplicationRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, removeRegionsFromReplicationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Secrets Manager");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "RemoveRegionsFromReplication");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<RemoveRegionsFromReplicationResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, RemoveRegionsFromReplicationResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<RemoveRegionsFromReplicationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<RemoveRegionsFromReplicationRequest, RemoveRegionsFromReplicationResponse>()
                            .withOperationName("RemoveRegionsFromReplication")
                            .withMarshaller(new RemoveRegionsFromReplicationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(removeRegionsFromReplicationRequest));
            CompletableFuture<RemoveRegionsFromReplicationResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Replicates the secret to a new Regions. See <a
     * href="https://docs.aws.amazon.com/secretsmanager/latest/userguide/create-manage-multi-region-secrets.html"
     * >Multi-Region secrets</a>.
     * </p>
     * <p>
     * <b>Required permissions: </b> <code>secretsmanager:ReplicateSecretToRegions</code>. For more information, see <a
     * href=
     * "https://docs.aws.amazon.com/secretsmanager/latest/userguide/reference_iam-permissions.html#reference_iam-permissions_actions"
     * > IAM policy actions for Secrets Manager</a> and <a
     * href="https://docs.aws.amazon.com/secretsmanager/latest/userguide/auth-and-access.html">Authentication and access
     * control in Secrets Manager</a>.
     * </p>
     *
     * @param replicateSecretToRegionsRequest
     * @return A Java Future containing the result of the ReplicateSecretToRegions operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException Secrets Manager can't find the resource that you asked for.</li>
     *         <li>InvalidRequestException A parameter value is not valid for the current state of the resource.</p>
     *         <p>
     *         Possible causes:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         The secret is scheduled for deletion.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         You tried to enable rotation on a secret that doesn't already have a Lambda function ARN configured and
     *         you didn't include such an ARN as a parameter in this call.
     *         </p>
     *         </li></li>
     *         <li>InvalidParameterException The parameter name or value is invalid.</li>
     *         <li>InternalServiceErrorException An error occurred on the server side.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SecretsManagerException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample SecretsManagerAsyncClient.ReplicateSecretToRegions
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/secretsmanager-2017-10-17/ReplicateSecretToRegions"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ReplicateSecretToRegionsResponse> replicateSecretToRegions(
            ReplicateSecretToRegionsRequest replicateSecretToRegionsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, replicateSecretToRegionsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Secrets Manager");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ReplicateSecretToRegions");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ReplicateSecretToRegionsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ReplicateSecretToRegionsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ReplicateSecretToRegionsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ReplicateSecretToRegionsRequest, ReplicateSecretToRegionsResponse>()
                            .withOperationName("ReplicateSecretToRegions")
                            .withMarshaller(new ReplicateSecretToRegionsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(replicateSecretToRegionsRequest));
            CompletableFuture<ReplicateSecretToRegionsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Cancels the scheduled deletion of a secret by removing the <code>DeletedDate</code> time stamp. You can access a
     * secret again after it has been restored.
     * </p>
     * <p>
     * <b>Required permissions: </b> <code>secretsmanager:RestoreSecret</code>. For more information, see <a href=
     * "https://docs.aws.amazon.com/secretsmanager/latest/userguide/reference_iam-permissions.html#reference_iam-permissions_actions"
     * > IAM policy actions for Secrets Manager</a> and <a
     * href="https://docs.aws.amazon.com/secretsmanager/latest/userguide/auth-and-access.html">Authentication and access
     * control in Secrets Manager</a>.
     * </p>
     *
     * @param restoreSecretRequest
     * @return A Java Future containing the result of the RestoreSecret operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException Secrets Manager can't find the resource that you asked for.</li>
     *         <li>InvalidParameterException The parameter name or value is invalid.</li>
     *         <li>InvalidRequestException A parameter value is not valid for the current state of the resource.</p>
     *         <p>
     *         Possible causes:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         The secret is scheduled for deletion.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         You tried to enable rotation on a secret that doesn't already have a Lambda function ARN configured and
     *         you didn't include such an ARN as a parameter in this call.
     *         </p>
     *         </li></li>
     *         <li>InternalServiceErrorException An error occurred on the server side.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SecretsManagerException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample SecretsManagerAsyncClient.RestoreSecret
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/secretsmanager-2017-10-17/RestoreSecret" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<RestoreSecretResponse> restoreSecret(RestoreSecretRequest restoreSecretRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, restoreSecretRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Secrets Manager");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "RestoreSecret");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<RestoreSecretResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    RestoreSecretResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<RestoreSecretResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<RestoreSecretRequest, RestoreSecretResponse>()
                            .withOperationName("RestoreSecret")
                            .withMarshaller(new RestoreSecretRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(restoreSecretRequest));
            CompletableFuture<RestoreSecretResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Configures and starts the asynchronous process of rotating the secret. For more information about rotation, see
     * <a href="https://docs.aws.amazon.com/secretsmanager/latest/userguide/rotating-secrets.html">Rotate secrets</a>.
     * </p>
     * <p>
     * If you include the configuration parameters, the operation sets the values for the secret and then immediately
     * starts a rotation. If you don't include the configuration parameters, the operation starts a rotation with the
     * values already stored in the secret.
     * </p>
     * <p>
     * For database credentials you want to rotate, for Secrets Manager to be able to rotate the secret, you must make
     * sure the secret value is in the <a
     * href="https://docs.aws.amazon.com/secretsmanager/latest/userguide/reference_secret_json_structure.html"> JSON
     * structure of a database secret</a>. In particular, if you want to use the <a href=
     * "https://docs.aws.amazon.com/secretsmanager/latest/userguide/rotating-secrets_strategies.html#rotating-secrets-two-users"
     * > alternating users strategy</a>, your secret must contain the ARN of a superuser secret.
     * </p>
     * <p>
     * To configure rotation, you also need the ARN of an Amazon Web Services Lambda function and the schedule for the
     * rotation. The Lambda rotation function creates a new version of the secret and creates or updates the credentials
     * on the database or service to match. After testing the new credentials, the function marks the new secret version
     * with the staging label <code>AWSCURRENT</code>. Then anyone who retrieves the secret gets the new version. For
     * more information, see <a
     * href="https://docs.aws.amazon.com/secretsmanager/latest/userguide/rotate-secrets_how.html">How rotation
     * works</a>.
     * </p>
     * <p>
     * You can create the Lambda rotation function based on the <a
     * href="https://docs.aws.amazon.com/secretsmanager/latest/userguide/reference_available-rotation-templates.html"
     * >rotation function templates</a> that Secrets Manager provides. Choose a template that matches your <a
     * href="https://docs.aws.amazon.com/secretsmanager/latest/userguide/rotating-secrets_strategies.html">Rotation
     * strategy</a>.
     * </p>
     * <p>
     * When rotation is successful, the <code>AWSPENDING</code> staging label might be attached to the same version as
     * the <code>AWSCURRENT</code> version, or it might not be attached to any version. If the <code>AWSPENDING</code>
     * staging label is present but not attached to the same version as <code>AWSCURRENT</code>, then any later
     * invocation of <code>RotateSecret</code> assumes that a previous rotation request is still in progress and returns
     * an error.
     * </p>
     * <p>
     * <b>Required permissions: </b> <code>secretsmanager:RotateSecret</code>. For more information, see <a href=
     * "https://docs.aws.amazon.com/secretsmanager/latest/userguide/reference_iam-permissions.html#reference_iam-permissions_actions"
     * > IAM policy actions for Secrets Manager</a> and <a
     * href="https://docs.aws.amazon.com/secretsmanager/latest/userguide/auth-and-access.html">Authentication and access
     * control in Secrets Manager</a>. You also need <code>lambda:InvokeFunction</code> permissions on the rotation
     * function. For more information, see <a href=
     * "https://docs.aws.amazon.com/secretsmanager/latest/userguide/rotating-secrets-required-permissions-function.html"
     * > Permissions for rotation</a>.
     * </p>
     *
     * @param rotateSecretRequest
     * @return A Java Future containing the result of the RotateSecret operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException Secrets Manager can't find the resource that you asked for.</li>
     *         <li>InvalidParameterException The parameter name or value is invalid.</li>
     *         <li>InternalServiceErrorException An error occurred on the server side.</li>
     *         <li>InvalidRequestException A parameter value is not valid for the current state of the resource.</p>
     *         <p>
     *         Possible causes:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         The secret is scheduled for deletion.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         You tried to enable rotation on a secret that doesn't already have a Lambda function ARN configured and
     *         you didn't include such an ARN as a parameter in this call.
     *         </p>
     *         </li></li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SecretsManagerException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample SecretsManagerAsyncClient.RotateSecret
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/secretsmanager-2017-10-17/RotateSecret" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<RotateSecretResponse> rotateSecret(RotateSecretRequest rotateSecretRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, rotateSecretRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Secrets Manager");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "RotateSecret");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<RotateSecretResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    RotateSecretResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<RotateSecretResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<RotateSecretRequest, RotateSecretResponse>()
                            .withOperationName("RotateSecret").withMarshaller(new RotateSecretRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(rotateSecretRequest));
            CompletableFuture<RotateSecretResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Removes the link between the replica secret and the primary secret and promotes the replica to a primary secret
     * in the replica Region.
     * </p>
     * <p>
     * You must call this operation from the Region in which you want to promote the replica to a primary secret.
     * </p>
     * <p>
     * <b>Required permissions: </b> <code>secretsmanager:StopReplicationToReplica</code>. For more information, see <a
     * href=
     * "https://docs.aws.amazon.com/secretsmanager/latest/userguide/reference_iam-permissions.html#reference_iam-permissions_actions"
     * > IAM policy actions for Secrets Manager</a> and <a
     * href="https://docs.aws.amazon.com/secretsmanager/latest/userguide/auth-and-access.html">Authentication and access
     * control in Secrets Manager</a>.
     * </p>
     *
     * @param stopReplicationToReplicaRequest
     * @return A Java Future containing the result of the StopReplicationToReplica operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException Secrets Manager can't find the resource that you asked for.</li>
     *         <li>InvalidRequestException A parameter value is not valid for the current state of the resource.</p>
     *         <p>
     *         Possible causes:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         The secret is scheduled for deletion.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         You tried to enable rotation on a secret that doesn't already have a Lambda function ARN configured and
     *         you didn't include such an ARN as a parameter in this call.
     *         </p>
     *         </li></li>
     *         <li>InvalidParameterException The parameter name or value is invalid.</li>
     *         <li>InternalServiceErrorException An error occurred on the server side.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SecretsManagerException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample SecretsManagerAsyncClient.StopReplicationToReplica
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/secretsmanager-2017-10-17/StopReplicationToReplica"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<StopReplicationToReplicaResponse> stopReplicationToReplica(
            StopReplicationToReplicaRequest stopReplicationToReplicaRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, stopReplicationToReplicaRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Secrets Manager");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "StopReplicationToReplica");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<StopReplicationToReplicaResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, StopReplicationToReplicaResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<StopReplicationToReplicaResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<StopReplicationToReplicaRequest, StopReplicationToReplicaResponse>()
                            .withOperationName("StopReplicationToReplica")
                            .withMarshaller(new StopReplicationToReplicaRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(stopReplicationToReplicaRequest));
            CompletableFuture<StopReplicationToReplicaResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Attaches tags to a secret. Tags consist of a key name and a value. Tags are part of the secret's metadata. They
     * are not associated with specific versions of the secret. This operation appends tags to the existing list of
     * tags.
     * </p>
     * <p>
     * The following restrictions apply to tags:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Maximum number of tags per secret: 50
     * </p>
     * </li>
     * <li>
     * <p>
     * Maximum key length: 127 Unicode characters in UTF-8
     * </p>
     * </li>
     * <li>
     * <p>
     * Maximum value length: 255 Unicode characters in UTF-8
     * </p>
     * </li>
     * <li>
     * <p>
     * Tag keys and values are case sensitive.
     * </p>
     * </li>
     * <li>
     * <p>
     * Do not use the <code>aws:</code> prefix in your tag names or values because Amazon Web Services reserves it for
     * Amazon Web Services use. You can't edit or delete tag names or values with this prefix. Tags with this prefix do
     * not count against your tags per secret limit.
     * </p>
     * </li>
     * <li>
     * <p>
     * If you use your tagging schema across multiple services and resources, other services might have restrictions on
     * allowed characters. Generally allowed characters: letters, spaces, and numbers representable in UTF-8, plus the
     * following special characters: + - = . _ : / @.
     * </p>
     * </li>
     * </ul>
     * <important>
     * <p>
     * If you use tags as part of your security strategy, then adding or removing a tag can change permissions. If
     * successfully completing this operation would result in you losing your permissions for this secret, then the
     * operation is blocked and returns an Access Denied error.
     * </p>
     * </important>
     * <p>
     * <b>Required permissions: </b> <code>secretsmanager:TagResource</code>. For more information, see <a href=
     * "https://docs.aws.amazon.com/secretsmanager/latest/userguide/reference_iam-permissions.html#reference_iam-permissions_actions"
     * > IAM policy actions for Secrets Manager</a> and <a
     * href="https://docs.aws.amazon.com/secretsmanager/latest/userguide/auth-and-access.html">Authentication and access
     * control in Secrets Manager</a>.
     * </p>
     *
     * @param tagResourceRequest
     * @return A Java Future containing the result of the TagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException Secrets Manager can't find the resource that you asked for.</li>
     *         <li>InvalidRequestException A parameter value is not valid for the current state of the resource.</p>
     *         <p>
     *         Possible causes:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         The secret is scheduled for deletion.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         You tried to enable rotation on a secret that doesn't already have a Lambda function ARN configured and
     *         you didn't include such an ARN as a parameter in this call.
     *         </p>
     *         </li></li>
     *         <li>InvalidParameterException The parameter name or value is invalid.</li>
     *         <li>InternalServiceErrorException An error occurred on the server side.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SecretsManagerException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample SecretsManagerAsyncClient.TagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/secretsmanager-2017-10-17/TagResource" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<TagResourceResponse> tagResource(TagResourceRequest tagResourceRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, tagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Secrets Manager");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "TagResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<TagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    TagResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<TagResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<TagResourceRequest, TagResourceResponse>()
                            .withOperationName("TagResource").withMarshaller(new TagResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(tagResourceRequest));
            CompletableFuture<TagResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Removes specific tags from a secret.
     * </p>
     * <p>
     * This operation is idempotent. If a requested tag is not attached to the secret, no error is returned and the
     * secret metadata is unchanged.
     * </p>
     * <important>
     * <p>
     * If you use tags as part of your security strategy, then removing a tag can change permissions. If successfully
     * completing this operation would result in you losing your permissions for this secret, then the operation is
     * blocked and returns an Access Denied error.
     * </p>
     * </important>
     * <p>
     * <b>Required permissions: </b> <code>secretsmanager:UntagResource</code>. For more information, see <a href=
     * "https://docs.aws.amazon.com/secretsmanager/latest/userguide/reference_iam-permissions.html#reference_iam-permissions_actions"
     * > IAM policy actions for Secrets Manager</a> and <a
     * href="https://docs.aws.amazon.com/secretsmanager/latest/userguide/auth-and-access.html">Authentication and access
     * control in Secrets Manager</a>.
     * </p>
     *
     * @param untagResourceRequest
     * @return A Java Future containing the result of the UntagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException Secrets Manager can't find the resource that you asked for.</li>
     *         <li>InvalidRequestException A parameter value is not valid for the current state of the resource.</p>
     *         <p>
     *         Possible causes:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         The secret is scheduled for deletion.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         You tried to enable rotation on a secret that doesn't already have a Lambda function ARN configured and
     *         you didn't include such an ARN as a parameter in this call.
     *         </p>
     *         </li></li>
     *         <li>InvalidParameterException The parameter name or value is invalid.</li>
     *         <li>InternalServiceErrorException An error occurred on the server side.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SecretsManagerException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample SecretsManagerAsyncClient.UntagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/secretsmanager-2017-10-17/UntagResource" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<UntagResourceResponse> untagResource(UntagResourceRequest untagResourceRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, untagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Secrets Manager");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UntagResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UntagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    UntagResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UntagResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UntagResourceRequest, UntagResourceResponse>()
                            .withOperationName("UntagResource")
                            .withMarshaller(new UntagResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(untagResourceRequest));
            CompletableFuture<UntagResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Modifies the details of a secret, including metadata and the secret value. To change the secret value, you can
     * also use <a>PutSecretValue</a>.
     * </p>
     * <p>
     * To change the rotation configuration of a secret, use <a>RotateSecret</a> instead.
     * </p>
     * <p>
     * We recommend you avoid calling <code>UpdateSecret</code> at a sustained rate of more than once every 10 minutes.
     * When you call <code>UpdateSecret</code> to update the secret value, Secrets Manager creates a new version of the
     * secret. Secrets Manager removes outdated versions when there are more than 100, but it does not remove versions
     * created less than 24 hours ago. If you update the secret value more than once every 10 minutes, you create more
     * versions than Secrets Manager removes, and you will reach the quota for secret versions.
     * </p>
     * <p>
     * If you include <code>SecretString</code> or <code>SecretBinary</code> to create a new secret version, Secrets
     * Manager automatically attaches the staging label <code>AWSCURRENT</code> to the new version.
     * </p>
     * <p>
     * If you call this operation with a <code>VersionId</code> that matches an existing version's
     * <code>ClientRequestToken</code>, the operation results in an error. You can't modify an existing version, you can
     * only create a new version. To remove a version, remove all staging labels from it. See
     * <a>UpdateSecretVersionStage</a>.
     * </p>
     * <p>
     * If you don't specify an KMS encryption key, Secrets Manager uses the Amazon Web Services managed key
     * <code>aws/secretsmanager</code>. If this key doesn't already exist in your account, then Secrets Manager creates
     * it for you automatically. All users and roles in the Amazon Web Services account automatically have access to use
     * <code>aws/secretsmanager</code>. Creating <code>aws/secretsmanager</code> can result in a one-time significant
     * delay in returning the result.
     * </p>
     * <p>
     * If the secret is in a different Amazon Web Services account from the credentials calling the API, then you can't
     * use <code>aws/secretsmanager</code> to encrypt the secret, and you must create and use a customer managed key.
     * </p>
     * <p>
     * <b>Required permissions: </b> <code>secretsmanager:UpdateSecret</code>. For more information, see <a href=
     * "https://docs.aws.amazon.com/secretsmanager/latest/userguide/reference_iam-permissions.html#reference_iam-permissions_actions"
     * > IAM policy actions for Secrets Manager</a> and <a
     * href="https://docs.aws.amazon.com/secretsmanager/latest/userguide/auth-and-access.html">Authentication and access
     * control in Secrets Manager</a>. If you use a customer managed key, you must also have
     * <code>kms:GenerateDataKey</code> and <code>kms:Decrypt</code> permissions on the key. For more information, see
     * <a href="https://docs.aws.amazon.com/secretsmanager/latest/userguide/security-encryption.html"> Secret encryption
     * and decryption</a>.
     * </p>
     *
     * @param updateSecretRequest
     * @return A Java Future containing the result of the UpdateSecret operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidParameterException The parameter name or value is invalid.</li>
     *         <li>InvalidRequestException A parameter value is not valid for the current state of the resource.</p>
     *         <p>
     *         Possible causes:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         The secret is scheduled for deletion.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         You tried to enable rotation on a secret that doesn't already have a Lambda function ARN configured and
     *         you didn't include such an ARN as a parameter in this call.
     *         </p>
     *         </li></li>
     *         <li>LimitExceededException The request failed because it would exceed one of the Secrets Manager quotas.</li>
     *         <li>EncryptionFailureException Secrets Manager can't encrypt the protected secret text using the provided
     *         KMS key. Check that the KMS key is available, enabled, and not in an invalid state. For more information,
     *         see <a href="https://docs.aws.amazon.com/kms/latest/developerguide/key-state.html">Key state: Effect on
     *         your KMS key</a>.</li>
     *         <li>ResourceExistsException A resource with the ID you requested already exists.</li>
     *         <li>ResourceNotFoundException Secrets Manager can't find the resource that you asked for.</li>
     *         <li>MalformedPolicyDocumentException The resource policy has syntax errors.</li>
     *         <li>InternalServiceErrorException An error occurred on the server side.</li>
     *         <li>PreconditionNotMetException The request failed because you did not complete all the prerequisite
     *         steps.</li>
     *         <li>DecryptionFailureException Secrets Manager can't decrypt the protected secret text using the provided
     *         KMS key.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SecretsManagerException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample SecretsManagerAsyncClient.UpdateSecret
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/secretsmanager-2017-10-17/UpdateSecret" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateSecretResponse> updateSecret(UpdateSecretRequest updateSecretRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateSecretRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Secrets Manager");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateSecret");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateSecretResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    UpdateSecretResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateSecretResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateSecretRequest, UpdateSecretResponse>()
                            .withOperationName("UpdateSecret").withMarshaller(new UpdateSecretRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(updateSecretRequest));
            CompletableFuture<UpdateSecretResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Modifies the staging labels attached to a version of a secret. Secrets Manager uses staging labels to track a
     * version as it progresses through the secret rotation process. Each staging label can be attached to only one
     * version at a time. To add a staging label to a version when it is already attached to another version, Secrets
     * Manager first removes it from the other version first and then attaches it to this one. For more information
     * about versions and staging labels, see <a
     * href="https://docs.aws.amazon.com/secretsmanager/latest/userguide/getting-started.html#term_version">Concepts:
     * Version</a>.
     * </p>
     * <p>
     * The staging labels that you specify in the <code>VersionStage</code> parameter are added to the existing list of
     * staging labels for the version.
     * </p>
     * <p>
     * You can move the <code>AWSCURRENT</code> staging label to this version by including it in this call.
     * </p>
     * <note>
     * <p>
     * Whenever you move <code>AWSCURRENT</code>, Secrets Manager automatically moves the label <code>AWSPREVIOUS</code>
     * to the version that <code>AWSCURRENT</code> was removed from.
     * </p>
     * </note>
     * <p>
     * If this action results in the last label being removed from a version, then the version is considered to be
     * 'deprecated' and can be deleted by Secrets Manager.
     * </p>
     * <p>
     * <b>Required permissions: </b> <code>secretsmanager:UpdateSecretVersionStage</code>. For more information, see <a
     * href=
     * "https://docs.aws.amazon.com/secretsmanager/latest/userguide/reference_iam-permissions.html#reference_iam-permissions_actions"
     * > IAM policy actions for Secrets Manager</a> and <a
     * href="https://docs.aws.amazon.com/secretsmanager/latest/userguide/auth-and-access.html">Authentication and access
     * control in Secrets Manager</a>.
     * </p>
     *
     * @param updateSecretVersionStageRequest
     * @return A Java Future containing the result of the UpdateSecretVersionStage operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException Secrets Manager can't find the resource that you asked for.</li>
     *         <li>InvalidParameterException The parameter name or value is invalid.</li>
     *         <li>InvalidRequestException A parameter value is not valid for the current state of the resource.</p>
     *         <p>
     *         Possible causes:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         The secret is scheduled for deletion.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         You tried to enable rotation on a secret that doesn't already have a Lambda function ARN configured and
     *         you didn't include such an ARN as a parameter in this call.
     *         </p>
     *         </li></li>
     *         <li>LimitExceededException The request failed because it would exceed one of the Secrets Manager quotas.</li>
     *         <li>InternalServiceErrorException An error occurred on the server side.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SecretsManagerException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample SecretsManagerAsyncClient.UpdateSecretVersionStage
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/secretsmanager-2017-10-17/UpdateSecretVersionStage"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateSecretVersionStageResponse> updateSecretVersionStage(
            UpdateSecretVersionStageRequest updateSecretVersionStageRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateSecretVersionStageRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Secrets Manager");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateSecretVersionStage");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateSecretVersionStageResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateSecretVersionStageResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateSecretVersionStageResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateSecretVersionStageRequest, UpdateSecretVersionStageResponse>()
                            .withOperationName("UpdateSecretVersionStage")
                            .withMarshaller(new UpdateSecretVersionStageRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(updateSecretVersionStageRequest));
            CompletableFuture<UpdateSecretVersionStageResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Validates that a resource policy does not grant a wide range of principals access to your secret. A
     * resource-based policy is optional for secrets.
     * </p>
     * <p>
     * The API performs three checks when validating the policy:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Sends a call to <a href=
     * "https://aws.amazon.com/blogs/security/protect-sensitive-data-in-the-cloud-with-automated-reasoning-zelkova/"
     * >Zelkova</a>, an automated reasoning engine, to ensure your resource policy does not allow broad access to your
     * secret, for example policies that use a wildcard for the principal.
     * </p>
     * </li>
     * <li>
     * <p>
     * Checks for correct syntax in a policy.
     * </p>
     * </li>
     * <li>
     * <p>
     * Verifies the policy does not lock out a caller.
     * </p>
     * </li>
     * </ul>
     * <p>
     * <b>Required permissions: </b> <code>secretsmanager:ValidateResourcePolicy</code>. For more information, see <a
     * href=
     * "https://docs.aws.amazon.com/secretsmanager/latest/userguide/reference_iam-permissions.html#reference_iam-permissions_actions"
     * > IAM policy actions for Secrets Manager</a> and <a
     * href="https://docs.aws.amazon.com/secretsmanager/latest/userguide/auth-and-access.html">Authentication and access
     * control in Secrets Manager</a>.
     * </p>
     *
     * @param validateResourcePolicyRequest
     * @return A Java Future containing the result of the ValidateResourcePolicy operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>MalformedPolicyDocumentException The resource policy has syntax errors.</li>
     *         <li>ResourceNotFoundException Secrets Manager can't find the resource that you asked for.</li>
     *         <li>InvalidParameterException The parameter name or value is invalid.</li>
     *         <li>InternalServiceErrorException An error occurred on the server side.</li>
     *         <li>InvalidRequestException A parameter value is not valid for the current state of the resource.</p>
     *         <p>
     *         Possible causes:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         The secret is scheduled for deletion.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         You tried to enable rotation on a secret that doesn't already have a Lambda function ARN configured and
     *         you didn't include such an ARN as a parameter in this call.
     *         </p>
     *         </li></li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>SecretsManagerException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample SecretsManagerAsyncClient.ValidateResourcePolicy
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/secretsmanager-2017-10-17/ValidateResourcePolicy"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ValidateResourcePolicyResponse> validateResourcePolicy(
            ValidateResourcePolicyRequest validateResourcePolicyRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, validateResourcePolicyRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Secrets Manager");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ValidateResourcePolicy");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ValidateResourcePolicyResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ValidateResourcePolicyResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ValidateResourcePolicyResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ValidateResourcePolicyRequest, ValidateResourcePolicyResponse>()
                            .withOperationName("ValidateResourcePolicy")
                            .withMarshaller(new ValidateResourcePolicyRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(validateResourcePolicyRequest));
            CompletableFuture<ValidateResourcePolicyResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    @Override
    public void close() {
        clientHandler.close();
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(SecretsManagerException::builder)
                .protocol(AwsJsonProtocol.AWS_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("EncryptionFailure")
                                .exceptionBuilderSupplier(EncryptionFailureException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidParameterException")
                                .exceptionBuilderSupplier(InvalidParameterException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("PublicPolicyException")
                                .exceptionBuilderSupplier(PublicPolicyException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("MalformedPolicyDocumentException")
                                .exceptionBuilderSupplier(MalformedPolicyDocumentException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("DecryptionFailure")
                                .exceptionBuilderSupplier(DecryptionFailureException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidRequestException")
                                .exceptionBuilderSupplier(InvalidRequestException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceNotFoundException")
                                .exceptionBuilderSupplier(ResourceNotFoundException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalServiceError")
                                .exceptionBuilderSupplier(InternalServiceErrorException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceExistsException")
                                .exceptionBuilderSupplier(ResourceExistsException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidNextTokenException")
                                .exceptionBuilderSupplier(InvalidNextTokenException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("LimitExceededException")
                                .exceptionBuilderSupplier(LimitExceededException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("PreconditionNotMetException")
                                .exceptionBuilderSupplier(PreconditionNotMetException::builder).build());
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private <T extends SecretsManagerRequest> T applyPaginatorUserAgent(T request) {
        Consumer<AwsRequestOverrideConfiguration.Builder> userAgentApplier = b -> b.addApiName(ApiName.builder()
                .version(VersionInfo.SDK_VERSION).name("PAGINATED").build());
        AwsRequestOverrideConfiguration overrideConfiguration = request.overrideConfiguration()
                .map(c -> c.toBuilder().applyMutation(userAgentApplier).build())
                .orElse((AwsRequestOverrideConfiguration.builder().applyMutation(userAgentApplier).build()));
        return (T) request.toBuilder().overrideConfiguration(overrideConfiguration).build();
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }
}
