/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.sagemaker.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * The VPC object you use to create or update a workforce.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class WorkforceVpcConfigRequest implements SdkPojo, Serializable,
        ToCopyableBuilder<WorkforceVpcConfigRequest.Builder, WorkforceVpcConfigRequest> {
    private static final SdkField<String> VPC_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("VpcId")
            .getter(getter(WorkforceVpcConfigRequest::vpcId)).setter(setter(Builder::vpcId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("VpcId").build()).build();

    private static final SdkField<List<String>> SECURITY_GROUP_IDS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("SecurityGroupIds")
            .getter(getter(WorkforceVpcConfigRequest::securityGroupIds))
            .setter(setter(Builder::securityGroupIds))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SecurityGroupIds").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<String>> SUBNETS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("Subnets")
            .getter(getter(WorkforceVpcConfigRequest::subnets))
            .setter(setter(Builder::subnets))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Subnets").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(VPC_ID_FIELD,
            SECURITY_GROUP_IDS_FIELD, SUBNETS_FIELD));

    private static final long serialVersionUID = 1L;

    private final String vpcId;

    private final List<String> securityGroupIds;

    private final List<String> subnets;

    private WorkforceVpcConfigRequest(BuilderImpl builder) {
        this.vpcId = builder.vpcId;
        this.securityGroupIds = builder.securityGroupIds;
        this.subnets = builder.subnets;
    }

    /**
     * <p>
     * The ID of the VPC that the workforce uses for communication.
     * </p>
     * 
     * @return The ID of the VPC that the workforce uses for communication.
     */
    public final String vpcId() {
        return vpcId;
    }

    /**
     * For responses, this returns true if the service returned a value for the SecurityGroupIds property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasSecurityGroupIds() {
        return securityGroupIds != null && !(securityGroupIds instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The VPC security group IDs, in the form sg-xxxxxxxx. The security groups must be for the same VPC as specified in
     * the subnet.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasSecurityGroupIds} method.
     * </p>
     * 
     * @return The VPC security group IDs, in the form sg-xxxxxxxx. The security groups must be for the same VPC as
     *         specified in the subnet.
     */
    public final List<String> securityGroupIds() {
        return securityGroupIds;
    }

    /**
     * For responses, this returns true if the service returned a value for the Subnets property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasSubnets() {
        return subnets != null && !(subnets instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The ID of the subnets in the VPC that you want to connect.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasSubnets} method.
     * </p>
     * 
     * @return The ID of the subnets in the VPC that you want to connect.
     */
    public final List<String> subnets() {
        return subnets;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(vpcId());
        hashCode = 31 * hashCode + Objects.hashCode(hasSecurityGroupIds() ? securityGroupIds() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasSubnets() ? subnets() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof WorkforceVpcConfigRequest)) {
            return false;
        }
        WorkforceVpcConfigRequest other = (WorkforceVpcConfigRequest) obj;
        return Objects.equals(vpcId(), other.vpcId()) && hasSecurityGroupIds() == other.hasSecurityGroupIds()
                && Objects.equals(securityGroupIds(), other.securityGroupIds()) && hasSubnets() == other.hasSubnets()
                && Objects.equals(subnets(), other.subnets());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("WorkforceVpcConfigRequest").add("VpcId", vpcId())
                .add("SecurityGroupIds", hasSecurityGroupIds() ? securityGroupIds() : null)
                .add("Subnets", hasSubnets() ? subnets() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "VpcId":
            return Optional.ofNullable(clazz.cast(vpcId()));
        case "SecurityGroupIds":
            return Optional.ofNullable(clazz.cast(securityGroupIds()));
        case "Subnets":
            return Optional.ofNullable(clazz.cast(subnets()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<WorkforceVpcConfigRequest, T> g) {
        return obj -> g.apply((WorkforceVpcConfigRequest) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, WorkforceVpcConfigRequest> {
        /**
         * <p>
         * The ID of the VPC that the workforce uses for communication.
         * </p>
         * 
         * @param vpcId
         *        The ID of the VPC that the workforce uses for communication.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder vpcId(String vpcId);

        /**
         * <p>
         * The VPC security group IDs, in the form sg-xxxxxxxx. The security groups must be for the same VPC as
         * specified in the subnet.
         * </p>
         * 
         * @param securityGroupIds
         *        The VPC security group IDs, in the form sg-xxxxxxxx. The security groups must be for the same VPC as
         *        specified in the subnet.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder securityGroupIds(Collection<String> securityGroupIds);

        /**
         * <p>
         * The VPC security group IDs, in the form sg-xxxxxxxx. The security groups must be for the same VPC as
         * specified in the subnet.
         * </p>
         * 
         * @param securityGroupIds
         *        The VPC security group IDs, in the form sg-xxxxxxxx. The security groups must be for the same VPC as
         *        specified in the subnet.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder securityGroupIds(String... securityGroupIds);

        /**
         * <p>
         * The ID of the subnets in the VPC that you want to connect.
         * </p>
         * 
         * @param subnets
         *        The ID of the subnets in the VPC that you want to connect.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder subnets(Collection<String> subnets);

        /**
         * <p>
         * The ID of the subnets in the VPC that you want to connect.
         * </p>
         * 
         * @param subnets
         *        The ID of the subnets in the VPC that you want to connect.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder subnets(String... subnets);
    }

    static final class BuilderImpl implements Builder {
        private String vpcId;

        private List<String> securityGroupIds = DefaultSdkAutoConstructList.getInstance();

        private List<String> subnets = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(WorkforceVpcConfigRequest model) {
            vpcId(model.vpcId);
            securityGroupIds(model.securityGroupIds);
            subnets(model.subnets);
        }

        public final String getVpcId() {
            return vpcId;
        }

        public final void setVpcId(String vpcId) {
            this.vpcId = vpcId;
        }

        @Override
        public final Builder vpcId(String vpcId) {
            this.vpcId = vpcId;
            return this;
        }

        public final Collection<String> getSecurityGroupIds() {
            if (securityGroupIds instanceof SdkAutoConstructList) {
                return null;
            }
            return securityGroupIds;
        }

        public final void setSecurityGroupIds(Collection<String> securityGroupIds) {
            this.securityGroupIds = WorkforceSecurityGroupIdsCopier.copy(securityGroupIds);
        }

        @Override
        public final Builder securityGroupIds(Collection<String> securityGroupIds) {
            this.securityGroupIds = WorkforceSecurityGroupIdsCopier.copy(securityGroupIds);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder securityGroupIds(String... securityGroupIds) {
            securityGroupIds(Arrays.asList(securityGroupIds));
            return this;
        }

        public final Collection<String> getSubnets() {
            if (subnets instanceof SdkAutoConstructList) {
                return null;
            }
            return subnets;
        }

        public final void setSubnets(Collection<String> subnets) {
            this.subnets = WorkforceSubnetsCopier.copy(subnets);
        }

        @Override
        public final Builder subnets(Collection<String> subnets) {
            this.subnets = WorkforceSubnetsCopier.copy(subnets);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder subnets(String... subnets) {
            subnets(Arrays.asList(subnets));
            return this;
        }

        @Override
        public WorkforceVpcConfigRequest build() {
            return new WorkforceVpcConfigRequest(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
