/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.privatenetworks;

import java.util.Collections;
import java.util.List;
import java.util.function.Consumer;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.awscore.client.handler.AwsSyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.ApiName;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.client.handler.SyncClientHandler;
import software.amazon.awssdk.core.exception.SdkClientException;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.core.util.VersionInfo;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.privatenetworks.model.AccessDeniedException;
import software.amazon.awssdk.services.privatenetworks.model.AcknowledgeOrderReceiptRequest;
import software.amazon.awssdk.services.privatenetworks.model.AcknowledgeOrderReceiptResponse;
import software.amazon.awssdk.services.privatenetworks.model.ActivateDeviceIdentifierRequest;
import software.amazon.awssdk.services.privatenetworks.model.ActivateDeviceIdentifierResponse;
import software.amazon.awssdk.services.privatenetworks.model.ActivateNetworkSiteRequest;
import software.amazon.awssdk.services.privatenetworks.model.ActivateNetworkSiteResponse;
import software.amazon.awssdk.services.privatenetworks.model.ConfigureAccessPointRequest;
import software.amazon.awssdk.services.privatenetworks.model.ConfigureAccessPointResponse;
import software.amazon.awssdk.services.privatenetworks.model.CreateNetworkRequest;
import software.amazon.awssdk.services.privatenetworks.model.CreateNetworkResponse;
import software.amazon.awssdk.services.privatenetworks.model.CreateNetworkSiteRequest;
import software.amazon.awssdk.services.privatenetworks.model.CreateNetworkSiteResponse;
import software.amazon.awssdk.services.privatenetworks.model.DeactivateDeviceIdentifierRequest;
import software.amazon.awssdk.services.privatenetworks.model.DeactivateDeviceIdentifierResponse;
import software.amazon.awssdk.services.privatenetworks.model.DeleteNetworkRequest;
import software.amazon.awssdk.services.privatenetworks.model.DeleteNetworkResponse;
import software.amazon.awssdk.services.privatenetworks.model.DeleteNetworkSiteRequest;
import software.amazon.awssdk.services.privatenetworks.model.DeleteNetworkSiteResponse;
import software.amazon.awssdk.services.privatenetworks.model.GetDeviceIdentifierRequest;
import software.amazon.awssdk.services.privatenetworks.model.GetDeviceIdentifierResponse;
import software.amazon.awssdk.services.privatenetworks.model.GetNetworkRequest;
import software.amazon.awssdk.services.privatenetworks.model.GetNetworkResourceRequest;
import software.amazon.awssdk.services.privatenetworks.model.GetNetworkResourceResponse;
import software.amazon.awssdk.services.privatenetworks.model.GetNetworkResponse;
import software.amazon.awssdk.services.privatenetworks.model.GetNetworkSiteRequest;
import software.amazon.awssdk.services.privatenetworks.model.GetNetworkSiteResponse;
import software.amazon.awssdk.services.privatenetworks.model.GetOrderRequest;
import software.amazon.awssdk.services.privatenetworks.model.GetOrderResponse;
import software.amazon.awssdk.services.privatenetworks.model.InternalServerException;
import software.amazon.awssdk.services.privatenetworks.model.LimitExceededException;
import software.amazon.awssdk.services.privatenetworks.model.ListDeviceIdentifiersRequest;
import software.amazon.awssdk.services.privatenetworks.model.ListDeviceIdentifiersResponse;
import software.amazon.awssdk.services.privatenetworks.model.ListNetworkResourcesRequest;
import software.amazon.awssdk.services.privatenetworks.model.ListNetworkResourcesResponse;
import software.amazon.awssdk.services.privatenetworks.model.ListNetworkSitesRequest;
import software.amazon.awssdk.services.privatenetworks.model.ListNetworkSitesResponse;
import software.amazon.awssdk.services.privatenetworks.model.ListNetworksRequest;
import software.amazon.awssdk.services.privatenetworks.model.ListNetworksResponse;
import software.amazon.awssdk.services.privatenetworks.model.ListOrdersRequest;
import software.amazon.awssdk.services.privatenetworks.model.ListOrdersResponse;
import software.amazon.awssdk.services.privatenetworks.model.ListTagsForResourceRequest;
import software.amazon.awssdk.services.privatenetworks.model.ListTagsForResourceResponse;
import software.amazon.awssdk.services.privatenetworks.model.PingRequest;
import software.amazon.awssdk.services.privatenetworks.model.PingResponse;
import software.amazon.awssdk.services.privatenetworks.model.PrivateNetworksException;
import software.amazon.awssdk.services.privatenetworks.model.PrivateNetworksRequest;
import software.amazon.awssdk.services.privatenetworks.model.ResourceNotFoundException;
import software.amazon.awssdk.services.privatenetworks.model.StartNetworkResourceUpdateRequest;
import software.amazon.awssdk.services.privatenetworks.model.StartNetworkResourceUpdateResponse;
import software.amazon.awssdk.services.privatenetworks.model.TagResourceRequest;
import software.amazon.awssdk.services.privatenetworks.model.TagResourceResponse;
import software.amazon.awssdk.services.privatenetworks.model.ThrottlingException;
import software.amazon.awssdk.services.privatenetworks.model.UntagResourceRequest;
import software.amazon.awssdk.services.privatenetworks.model.UntagResourceResponse;
import software.amazon.awssdk.services.privatenetworks.model.UpdateNetworkSitePlanRequest;
import software.amazon.awssdk.services.privatenetworks.model.UpdateNetworkSitePlanResponse;
import software.amazon.awssdk.services.privatenetworks.model.UpdateNetworkSiteRequest;
import software.amazon.awssdk.services.privatenetworks.model.UpdateNetworkSiteResponse;
import software.amazon.awssdk.services.privatenetworks.model.ValidationException;
import software.amazon.awssdk.services.privatenetworks.paginators.ListDeviceIdentifiersIterable;
import software.amazon.awssdk.services.privatenetworks.paginators.ListNetworkResourcesIterable;
import software.amazon.awssdk.services.privatenetworks.paginators.ListNetworkSitesIterable;
import software.amazon.awssdk.services.privatenetworks.paginators.ListNetworksIterable;
import software.amazon.awssdk.services.privatenetworks.paginators.ListOrdersIterable;
import software.amazon.awssdk.services.privatenetworks.transform.AcknowledgeOrderReceiptRequestMarshaller;
import software.amazon.awssdk.services.privatenetworks.transform.ActivateDeviceIdentifierRequestMarshaller;
import software.amazon.awssdk.services.privatenetworks.transform.ActivateNetworkSiteRequestMarshaller;
import software.amazon.awssdk.services.privatenetworks.transform.ConfigureAccessPointRequestMarshaller;
import software.amazon.awssdk.services.privatenetworks.transform.CreateNetworkRequestMarshaller;
import software.amazon.awssdk.services.privatenetworks.transform.CreateNetworkSiteRequestMarshaller;
import software.amazon.awssdk.services.privatenetworks.transform.DeactivateDeviceIdentifierRequestMarshaller;
import software.amazon.awssdk.services.privatenetworks.transform.DeleteNetworkRequestMarshaller;
import software.amazon.awssdk.services.privatenetworks.transform.DeleteNetworkSiteRequestMarshaller;
import software.amazon.awssdk.services.privatenetworks.transform.GetDeviceIdentifierRequestMarshaller;
import software.amazon.awssdk.services.privatenetworks.transform.GetNetworkRequestMarshaller;
import software.amazon.awssdk.services.privatenetworks.transform.GetNetworkResourceRequestMarshaller;
import software.amazon.awssdk.services.privatenetworks.transform.GetNetworkSiteRequestMarshaller;
import software.amazon.awssdk.services.privatenetworks.transform.GetOrderRequestMarshaller;
import software.amazon.awssdk.services.privatenetworks.transform.ListDeviceIdentifiersRequestMarshaller;
import software.amazon.awssdk.services.privatenetworks.transform.ListNetworkResourcesRequestMarshaller;
import software.amazon.awssdk.services.privatenetworks.transform.ListNetworkSitesRequestMarshaller;
import software.amazon.awssdk.services.privatenetworks.transform.ListNetworksRequestMarshaller;
import software.amazon.awssdk.services.privatenetworks.transform.ListOrdersRequestMarshaller;
import software.amazon.awssdk.services.privatenetworks.transform.ListTagsForResourceRequestMarshaller;
import software.amazon.awssdk.services.privatenetworks.transform.PingRequestMarshaller;
import software.amazon.awssdk.services.privatenetworks.transform.StartNetworkResourceUpdateRequestMarshaller;
import software.amazon.awssdk.services.privatenetworks.transform.TagResourceRequestMarshaller;
import software.amazon.awssdk.services.privatenetworks.transform.UntagResourceRequestMarshaller;
import software.amazon.awssdk.services.privatenetworks.transform.UpdateNetworkSitePlanRequestMarshaller;
import software.amazon.awssdk.services.privatenetworks.transform.UpdateNetworkSiteRequestMarshaller;
import software.amazon.awssdk.utils.Logger;

/**
 * Internal implementation of {@link PrivateNetworksClient}.
 *
 * @see PrivateNetworksClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultPrivateNetworksClient implements PrivateNetworksClient {
    private static final Logger log = Logger.loggerFor(DefaultPrivateNetworksClient.class);

    private final SyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    private final PrivateNetworksServiceClientConfiguration serviceClientConfiguration;

    protected DefaultPrivateNetworksClient(PrivateNetworksServiceClientConfiguration serviceClientConfiguration,
            SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsSyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.serviceClientConfiguration = serviceClientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    /**
     * <p>
     * Acknowledges that the specified network order was received.
     * </p>
     *
     * @param acknowledgeOrderReceiptRequest
     * @return Result of the AcknowledgeOrderReceipt operation returned by the service.
     * @throws ResourceNotFoundException
     *         The resource was not found.
     * @throws ValidationException
     *         The request failed validation.
     * @throws InternalServerException
     *         Information about an internal error.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws PrivateNetworksException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample PrivateNetworksClient.AcknowledgeOrderReceipt
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/privatenetworks-2021-12-03/AcknowledgeOrderReceipt"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public AcknowledgeOrderReceiptResponse acknowledgeOrderReceipt(AcknowledgeOrderReceiptRequest acknowledgeOrderReceiptRequest)
            throws ResourceNotFoundException, ValidationException, InternalServerException, AwsServiceException,
            SdkClientException, PrivateNetworksException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<AcknowledgeOrderReceiptResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, AcknowledgeOrderReceiptResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, acknowledgeOrderReceiptRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "PrivateNetworks");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "AcknowledgeOrderReceipt");

            return clientHandler
                    .execute(new ClientExecutionParams<AcknowledgeOrderReceiptRequest, AcknowledgeOrderReceiptResponse>()
                            .withOperationName("AcknowledgeOrderReceipt").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(acknowledgeOrderReceiptRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new AcknowledgeOrderReceiptRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Activates the specified device identifier.
     * </p>
     *
     * @param activateDeviceIdentifierRequest
     * @return Result of the ActivateDeviceIdentifier operation returned by the service.
     * @throws ResourceNotFoundException
     *         The resource was not found.
     * @throws ValidationException
     *         The request failed validation.
     * @throws InternalServerException
     *         Information about an internal error.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws PrivateNetworksException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample PrivateNetworksClient.ActivateDeviceIdentifier
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/privatenetworks-2021-12-03/ActivateDeviceIdentifier"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ActivateDeviceIdentifierResponse activateDeviceIdentifier(
            ActivateDeviceIdentifierRequest activateDeviceIdentifierRequest) throws ResourceNotFoundException,
            ValidationException, InternalServerException, AwsServiceException, SdkClientException, PrivateNetworksException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ActivateDeviceIdentifierResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ActivateDeviceIdentifierResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, activateDeviceIdentifierRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "PrivateNetworks");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ActivateDeviceIdentifier");

            return clientHandler
                    .execute(new ClientExecutionParams<ActivateDeviceIdentifierRequest, ActivateDeviceIdentifierResponse>()
                            .withOperationName("ActivateDeviceIdentifier").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(activateDeviceIdentifierRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new ActivateDeviceIdentifierRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Activates the specified network site.
     * </p>
     *
     * @param activateNetworkSiteRequest
     * @return Result of the ActivateNetworkSite operation returned by the service.
     * @throws ResourceNotFoundException
     *         The resource was not found.
     * @throws ValidationException
     *         The request failed validation.
     * @throws InternalServerException
     *         Information about an internal error.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws PrivateNetworksException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample PrivateNetworksClient.ActivateNetworkSite
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/privatenetworks-2021-12-03/ActivateNetworkSite"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ActivateNetworkSiteResponse activateNetworkSite(ActivateNetworkSiteRequest activateNetworkSiteRequest)
            throws ResourceNotFoundException, ValidationException, InternalServerException, AwsServiceException,
            SdkClientException, PrivateNetworksException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ActivateNetworkSiteResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ActivateNetworkSiteResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, activateNetworkSiteRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "PrivateNetworks");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ActivateNetworkSite");

            return clientHandler.execute(new ClientExecutionParams<ActivateNetworkSiteRequest, ActivateNetworkSiteResponse>()
                    .withOperationName("ActivateNetworkSite").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(activateNetworkSiteRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ActivateNetworkSiteRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Configures the specified network resource.
     * </p>
     * <p>
     * Use this action to specify the geographic position of the hardware. You must provide Certified Professional
     * Installer (CPI) credentials in the request so that we can obtain spectrum grants. For more information, see <a
     * href="https://docs.aws.amazon.com/private-networks/latest/userguide/radio-units.html">Radio units</a> in the
     * <i>Amazon Web Services Private 5G User Guide</i>.
     * </p>
     *
     * @param configureAccessPointRequest
     * @return Result of the ConfigureAccessPoint operation returned by the service.
     * @throws ResourceNotFoundException
     *         The resource was not found.
     * @throws ValidationException
     *         The request failed validation.
     * @throws InternalServerException
     *         Information about an internal error.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws PrivateNetworksException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample PrivateNetworksClient.ConfigureAccessPoint
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/privatenetworks-2021-12-03/ConfigureAccessPoint"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ConfigureAccessPointResponse configureAccessPoint(ConfigureAccessPointRequest configureAccessPointRequest)
            throws ResourceNotFoundException, ValidationException, InternalServerException, AwsServiceException,
            SdkClientException, PrivateNetworksException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ConfigureAccessPointResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ConfigureAccessPointResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, configureAccessPointRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "PrivateNetworks");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ConfigureAccessPoint");

            return clientHandler.execute(new ClientExecutionParams<ConfigureAccessPointRequest, ConfigureAccessPointResponse>()
                    .withOperationName("ConfigureAccessPoint").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(configureAccessPointRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ConfigureAccessPointRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Creates a network.
     * </p>
     *
     * @param createNetworkRequest
     * @return Result of the CreateNetwork operation returned by the service.
     * @throws LimitExceededException
     *         The limit was exceeded.
     * @throws ValidationException
     *         The request failed validation.
     * @throws InternalServerException
     *         Information about an internal error.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws PrivateNetworksException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample PrivateNetworksClient.CreateNetwork
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/privatenetworks-2021-12-03/CreateNetwork" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CreateNetworkResponse createNetwork(CreateNetworkRequest createNetworkRequest) throws LimitExceededException,
            ValidationException, InternalServerException, AwsServiceException, SdkClientException, PrivateNetworksException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateNetworkResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                CreateNetworkResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createNetworkRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "PrivateNetworks");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateNetwork");

            return clientHandler.execute(new ClientExecutionParams<CreateNetworkRequest, CreateNetworkResponse>()
                    .withOperationName("CreateNetwork").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(createNetworkRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new CreateNetworkRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Creates a network site.
     * </p>
     *
     * @param createNetworkSiteRequest
     * @return Result of the CreateNetworkSite operation returned by the service.
     * @throws ResourceNotFoundException
     *         The resource was not found.
     * @throws ValidationException
     *         The request failed validation.
     * @throws InternalServerException
     *         Information about an internal error.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws PrivateNetworksException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample PrivateNetworksClient.CreateNetworkSite
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/privatenetworks-2021-12-03/CreateNetworkSite"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CreateNetworkSiteResponse createNetworkSite(CreateNetworkSiteRequest createNetworkSiteRequest)
            throws ResourceNotFoundException, ValidationException, InternalServerException, AwsServiceException,
            SdkClientException, PrivateNetworksException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateNetworkSiteResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                CreateNetworkSiteResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createNetworkSiteRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "PrivateNetworks");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateNetworkSite");

            return clientHandler.execute(new ClientExecutionParams<CreateNetworkSiteRequest, CreateNetworkSiteResponse>()
                    .withOperationName("CreateNetworkSite").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(createNetworkSiteRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new CreateNetworkSiteRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deactivates the specified device identifier.
     * </p>
     *
     * @param deactivateDeviceIdentifierRequest
     * @return Result of the DeactivateDeviceIdentifier operation returned by the service.
     * @throws ResourceNotFoundException
     *         The resource was not found.
     * @throws ValidationException
     *         The request failed validation.
     * @throws InternalServerException
     *         Information about an internal error.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws PrivateNetworksException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample PrivateNetworksClient.DeactivateDeviceIdentifier
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/privatenetworks-2021-12-03/DeactivateDeviceIdentifier"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DeactivateDeviceIdentifierResponse deactivateDeviceIdentifier(
            DeactivateDeviceIdentifierRequest deactivateDeviceIdentifierRequest) throws ResourceNotFoundException,
            ValidationException, InternalServerException, AwsServiceException, SdkClientException, PrivateNetworksException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeactivateDeviceIdentifierResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DeactivateDeviceIdentifierResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deactivateDeviceIdentifierRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "PrivateNetworks");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeactivateDeviceIdentifier");

            return clientHandler
                    .execute(new ClientExecutionParams<DeactivateDeviceIdentifierRequest, DeactivateDeviceIdentifierResponse>()
                            .withOperationName("DeactivateDeviceIdentifier").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(deactivateDeviceIdentifierRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new DeactivateDeviceIdentifierRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deletes the specified network. You must delete network sites before you delete the network. For more information,
     * see <a href="https://docs.aws.amazon.com/private-networks/latest/APIReference/API_DeleteNetworkSite.html">
     * DeleteNetworkSite</a> in the <i>API Reference for Amazon Web Services Private 5G</i>.
     * </p>
     *
     * @param deleteNetworkRequest
     * @return Result of the DeleteNetwork operation returned by the service.
     * @throws ResourceNotFoundException
     *         The resource was not found.
     * @throws AccessDeniedException
     *         You do not have permission to perform this operation.
     * @throws ValidationException
     *         The request failed validation.
     * @throws InternalServerException
     *         Information about an internal error.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws PrivateNetworksException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample PrivateNetworksClient.DeleteNetwork
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/privatenetworks-2021-12-03/DeleteNetwork" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public DeleteNetworkResponse deleteNetwork(DeleteNetworkRequest deleteNetworkRequest) throws ResourceNotFoundException,
            AccessDeniedException, ValidationException, InternalServerException, AwsServiceException, SdkClientException,
            PrivateNetworksException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteNetworkResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DeleteNetworkResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteNetworkRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "PrivateNetworks");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteNetwork");

            return clientHandler.execute(new ClientExecutionParams<DeleteNetworkRequest, DeleteNetworkResponse>()
                    .withOperationName("DeleteNetwork").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(deleteNetworkRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DeleteNetworkRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deletes the specified network site. Return the hardware after you delete the network site. You are responsible
     * for minimum charges. For more information, see <a
     * href="https://docs.aws.amazon.com/private-networks/latest/userguide/hardware-maintenance.html">Hardware
     * returns</a> in the <i>Amazon Web Services Private 5G User Guide</i>.
     * </p>
     *
     * @param deleteNetworkSiteRequest
     * @return Result of the DeleteNetworkSite operation returned by the service.
     * @throws ResourceNotFoundException
     *         The resource was not found.
     * @throws AccessDeniedException
     *         You do not have permission to perform this operation.
     * @throws ValidationException
     *         The request failed validation.
     * @throws InternalServerException
     *         Information about an internal error.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws PrivateNetworksException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample PrivateNetworksClient.DeleteNetworkSite
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/privatenetworks-2021-12-03/DeleteNetworkSite"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DeleteNetworkSiteResponse deleteNetworkSite(DeleteNetworkSiteRequest deleteNetworkSiteRequest)
            throws ResourceNotFoundException, AccessDeniedException, ValidationException, InternalServerException,
            AwsServiceException, SdkClientException, PrivateNetworksException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteNetworkSiteResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DeleteNetworkSiteResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteNetworkSiteRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "PrivateNetworks");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteNetworkSite");

            return clientHandler.execute(new ClientExecutionParams<DeleteNetworkSiteRequest, DeleteNetworkSiteResponse>()
                    .withOperationName("DeleteNetworkSite").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(deleteNetworkSiteRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DeleteNetworkSiteRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Gets the specified device identifier.
     * </p>
     *
     * @param getDeviceIdentifierRequest
     * @return Result of the GetDeviceIdentifier operation returned by the service.
     * @throws ResourceNotFoundException
     *         The resource was not found.
     * @throws ValidationException
     *         The request failed validation.
     * @throws InternalServerException
     *         Information about an internal error.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws PrivateNetworksException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample PrivateNetworksClient.GetDeviceIdentifier
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/privatenetworks-2021-12-03/GetDeviceIdentifier"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public GetDeviceIdentifierResponse getDeviceIdentifier(GetDeviceIdentifierRequest getDeviceIdentifierRequest)
            throws ResourceNotFoundException, ValidationException, InternalServerException, AwsServiceException,
            SdkClientException, PrivateNetworksException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetDeviceIdentifierResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, GetDeviceIdentifierResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getDeviceIdentifierRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "PrivateNetworks");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetDeviceIdentifier");

            return clientHandler.execute(new ClientExecutionParams<GetDeviceIdentifierRequest, GetDeviceIdentifierResponse>()
                    .withOperationName("GetDeviceIdentifier").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(getDeviceIdentifierRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetDeviceIdentifierRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Gets the specified network.
     * </p>
     *
     * @param getNetworkRequest
     * @return Result of the GetNetwork operation returned by the service.
     * @throws ResourceNotFoundException
     *         The resource was not found.
     * @throws ValidationException
     *         The request failed validation.
     * @throws InternalServerException
     *         Information about an internal error.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws PrivateNetworksException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample PrivateNetworksClient.GetNetwork
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/privatenetworks-2021-12-03/GetNetwork" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public GetNetworkResponse getNetwork(GetNetworkRequest getNetworkRequest) throws ResourceNotFoundException,
            ValidationException, InternalServerException, AwsServiceException, SdkClientException, PrivateNetworksException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetNetworkResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetNetworkResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getNetworkRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "PrivateNetworks");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetNetwork");

            return clientHandler
                    .execute(new ClientExecutionParams<GetNetworkRequest, GetNetworkResponse>().withOperationName("GetNetwork")
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(getNetworkRequest).withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new GetNetworkRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Gets the specified network resource.
     * </p>
     *
     * @param getNetworkResourceRequest
     * @return Result of the GetNetworkResource operation returned by the service.
     * @throws ResourceNotFoundException
     *         The resource was not found.
     * @throws ValidationException
     *         The request failed validation.
     * @throws InternalServerException
     *         Information about an internal error.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws PrivateNetworksException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample PrivateNetworksClient.GetNetworkResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/privatenetworks-2021-12-03/GetNetworkResource"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public GetNetworkResourceResponse getNetworkResource(GetNetworkResourceRequest getNetworkResourceRequest)
            throws ResourceNotFoundException, ValidationException, InternalServerException, AwsServiceException,
            SdkClientException, PrivateNetworksException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetNetworkResourceResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, GetNetworkResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getNetworkResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "PrivateNetworks");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetNetworkResource");

            return clientHandler.execute(new ClientExecutionParams<GetNetworkResourceRequest, GetNetworkResourceResponse>()
                    .withOperationName("GetNetworkResource").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(getNetworkResourceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetNetworkResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Gets the specified network site.
     * </p>
     *
     * @param getNetworkSiteRequest
     * @return Result of the GetNetworkSite operation returned by the service.
     * @throws ResourceNotFoundException
     *         The resource was not found.
     * @throws ValidationException
     *         The request failed validation.
     * @throws InternalServerException
     *         Information about an internal error.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws PrivateNetworksException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample PrivateNetworksClient.GetNetworkSite
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/privatenetworks-2021-12-03/GetNetworkSite"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public GetNetworkSiteResponse getNetworkSite(GetNetworkSiteRequest getNetworkSiteRequest) throws ResourceNotFoundException,
            ValidationException, InternalServerException, AwsServiceException, SdkClientException, PrivateNetworksException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetNetworkSiteResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetNetworkSiteResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getNetworkSiteRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "PrivateNetworks");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetNetworkSite");

            return clientHandler.execute(new ClientExecutionParams<GetNetworkSiteRequest, GetNetworkSiteResponse>()
                    .withOperationName("GetNetworkSite").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(getNetworkSiteRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetNetworkSiteRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Gets the specified order.
     * </p>
     *
     * @param getOrderRequest
     * @return Result of the GetOrder operation returned by the service.
     * @throws ResourceNotFoundException
     *         The resource was not found.
     * @throws ValidationException
     *         The request failed validation.
     * @throws InternalServerException
     *         Information about an internal error.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws PrivateNetworksException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample PrivateNetworksClient.GetOrder
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/privatenetworks-2021-12-03/GetOrder" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public GetOrderResponse getOrder(GetOrderRequest getOrderRequest) throws ResourceNotFoundException, ValidationException,
            InternalServerException, AwsServiceException, SdkClientException, PrivateNetworksException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetOrderResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetOrderResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getOrderRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "PrivateNetworks");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetOrder");

            return clientHandler.execute(new ClientExecutionParams<GetOrderRequest, GetOrderResponse>()
                    .withOperationName("GetOrder").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(getOrderRequest)
                    .withMetricCollector(apiCallMetricCollector).withMarshaller(new GetOrderRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Lists device identifiers. Add filters to your request to return a more specific list of results. Use filters to
     * match the Amazon Resource Name (ARN) of an order, the status of device identifiers, or the ARN of the traffic
     * group.
     * </p>
     * <p>
     * If you specify multiple filters, filters are joined with an OR, and the request returns results that match all of
     * the specified filters.
     * </p>
     *
     * @param listDeviceIdentifiersRequest
     * @return Result of the ListDeviceIdentifiers operation returned by the service.
     * @throws ResourceNotFoundException
     *         The resource was not found.
     * @throws ValidationException
     *         The request failed validation.
     * @throws InternalServerException
     *         Information about an internal error.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws PrivateNetworksException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample PrivateNetworksClient.ListDeviceIdentifiers
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/privatenetworks-2021-12-03/ListDeviceIdentifiers"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListDeviceIdentifiersResponse listDeviceIdentifiers(ListDeviceIdentifiersRequest listDeviceIdentifiersRequest)
            throws ResourceNotFoundException, ValidationException, InternalServerException, AwsServiceException,
            SdkClientException, PrivateNetworksException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListDeviceIdentifiersResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListDeviceIdentifiersResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listDeviceIdentifiersRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "PrivateNetworks");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListDeviceIdentifiers");

            return clientHandler.execute(new ClientExecutionParams<ListDeviceIdentifiersRequest, ListDeviceIdentifiersResponse>()
                    .withOperationName("ListDeviceIdentifiers").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(listDeviceIdentifiersRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListDeviceIdentifiersRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Lists device identifiers. Add filters to your request to return a more specific list of results. Use filters to
     * match the Amazon Resource Name (ARN) of an order, the status of device identifiers, or the ARN of the traffic
     * group.
     * </p>
     * <p>
     * If you specify multiple filters, filters are joined with an OR, and the request returns results that match all of
     * the specified filters.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listDeviceIdentifiers(software.amazon.awssdk.services.privatenetworks.model.ListDeviceIdentifiersRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.privatenetworks.paginators.ListDeviceIdentifiersIterable responses = client.listDeviceIdentifiersPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.privatenetworks.paginators.ListDeviceIdentifiersIterable responses = client
     *             .listDeviceIdentifiersPaginator(request);
     *     for (software.amazon.awssdk.services.privatenetworks.model.ListDeviceIdentifiersResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.privatenetworks.paginators.ListDeviceIdentifiersIterable responses = client.listDeviceIdentifiersPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of maxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listDeviceIdentifiers(software.amazon.awssdk.services.privatenetworks.model.ListDeviceIdentifiersRequest)}
     * operation.</b>
     * </p>
     *
     * @param listDeviceIdentifiersRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws ResourceNotFoundException
     *         The resource was not found.
     * @throws ValidationException
     *         The request failed validation.
     * @throws InternalServerException
     *         Information about an internal error.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws PrivateNetworksException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample PrivateNetworksClient.ListDeviceIdentifiers
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/privatenetworks-2021-12-03/ListDeviceIdentifiers"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListDeviceIdentifiersIterable listDeviceIdentifiersPaginator(ListDeviceIdentifiersRequest listDeviceIdentifiersRequest)
            throws ResourceNotFoundException, ValidationException, InternalServerException, AwsServiceException,
            SdkClientException, PrivateNetworksException {
        return new ListDeviceIdentifiersIterable(this, applyPaginatorUserAgent(listDeviceIdentifiersRequest));
    }

    /**
     * <p>
     * Lists network resources. Add filters to your request to return a more specific list of results. Use filters to
     * match the Amazon Resource Name (ARN) of an order or the status of network resources.
     * </p>
     * <p>
     * If you specify multiple filters, filters are joined with an OR, and the request returns results that match all of
     * the specified filters.
     * </p>
     *
     * @param listNetworkResourcesRequest
     * @return Result of the ListNetworkResources operation returned by the service.
     * @throws ResourceNotFoundException
     *         The resource was not found.
     * @throws ValidationException
     *         The request failed validation.
     * @throws InternalServerException
     *         Information about an internal error.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws PrivateNetworksException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample PrivateNetworksClient.ListNetworkResources
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/privatenetworks-2021-12-03/ListNetworkResources"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListNetworkResourcesResponse listNetworkResources(ListNetworkResourcesRequest listNetworkResourcesRequest)
            throws ResourceNotFoundException, ValidationException, InternalServerException, AwsServiceException,
            SdkClientException, PrivateNetworksException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListNetworkResourcesResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListNetworkResourcesResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listNetworkResourcesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "PrivateNetworks");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListNetworkResources");

            return clientHandler.execute(new ClientExecutionParams<ListNetworkResourcesRequest, ListNetworkResourcesResponse>()
                    .withOperationName("ListNetworkResources").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(listNetworkResourcesRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListNetworkResourcesRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Lists network resources. Add filters to your request to return a more specific list of results. Use filters to
     * match the Amazon Resource Name (ARN) of an order or the status of network resources.
     * </p>
     * <p>
     * If you specify multiple filters, filters are joined with an OR, and the request returns results that match all of
     * the specified filters.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listNetworkResources(software.amazon.awssdk.services.privatenetworks.model.ListNetworkResourcesRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.privatenetworks.paginators.ListNetworkResourcesIterable responses = client.listNetworkResourcesPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.privatenetworks.paginators.ListNetworkResourcesIterable responses = client
     *             .listNetworkResourcesPaginator(request);
     *     for (software.amazon.awssdk.services.privatenetworks.model.ListNetworkResourcesResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.privatenetworks.paginators.ListNetworkResourcesIterable responses = client.listNetworkResourcesPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of maxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listNetworkResources(software.amazon.awssdk.services.privatenetworks.model.ListNetworkResourcesRequest)}
     * operation.</b>
     * </p>
     *
     * @param listNetworkResourcesRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws ResourceNotFoundException
     *         The resource was not found.
     * @throws ValidationException
     *         The request failed validation.
     * @throws InternalServerException
     *         Information about an internal error.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws PrivateNetworksException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample PrivateNetworksClient.ListNetworkResources
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/privatenetworks-2021-12-03/ListNetworkResources"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListNetworkResourcesIterable listNetworkResourcesPaginator(ListNetworkResourcesRequest listNetworkResourcesRequest)
            throws ResourceNotFoundException, ValidationException, InternalServerException, AwsServiceException,
            SdkClientException, PrivateNetworksException {
        return new ListNetworkResourcesIterable(this, applyPaginatorUserAgent(listNetworkResourcesRequest));
    }

    /**
     * <p>
     * Lists network sites. Add filters to your request to return a more specific list of results. Use filters to match
     * the status of the network site.
     * </p>
     *
     * @param listNetworkSitesRequest
     * @return Result of the ListNetworkSites operation returned by the service.
     * @throws ResourceNotFoundException
     *         The resource was not found.
     * @throws ValidationException
     *         The request failed validation.
     * @throws InternalServerException
     *         Information about an internal error.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws PrivateNetworksException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample PrivateNetworksClient.ListNetworkSites
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/privatenetworks-2021-12-03/ListNetworkSites"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListNetworkSitesResponse listNetworkSites(ListNetworkSitesRequest listNetworkSitesRequest)
            throws ResourceNotFoundException, ValidationException, InternalServerException, AwsServiceException,
            SdkClientException, PrivateNetworksException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListNetworkSitesResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListNetworkSitesResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listNetworkSitesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "PrivateNetworks");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListNetworkSites");

            return clientHandler.execute(new ClientExecutionParams<ListNetworkSitesRequest, ListNetworkSitesResponse>()
                    .withOperationName("ListNetworkSites").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(listNetworkSitesRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListNetworkSitesRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Lists network sites. Add filters to your request to return a more specific list of results. Use filters to match
     * the status of the network site.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listNetworkSites(software.amazon.awssdk.services.privatenetworks.model.ListNetworkSitesRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.privatenetworks.paginators.ListNetworkSitesIterable responses = client.listNetworkSitesPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.privatenetworks.paginators.ListNetworkSitesIterable responses = client
     *             .listNetworkSitesPaginator(request);
     *     for (software.amazon.awssdk.services.privatenetworks.model.ListNetworkSitesResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.privatenetworks.paginators.ListNetworkSitesIterable responses = client.listNetworkSitesPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of maxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listNetworkSites(software.amazon.awssdk.services.privatenetworks.model.ListNetworkSitesRequest)}
     * operation.</b>
     * </p>
     *
     * @param listNetworkSitesRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws ResourceNotFoundException
     *         The resource was not found.
     * @throws ValidationException
     *         The request failed validation.
     * @throws InternalServerException
     *         Information about an internal error.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws PrivateNetworksException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample PrivateNetworksClient.ListNetworkSites
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/privatenetworks-2021-12-03/ListNetworkSites"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListNetworkSitesIterable listNetworkSitesPaginator(ListNetworkSitesRequest listNetworkSitesRequest)
            throws ResourceNotFoundException, ValidationException, InternalServerException, AwsServiceException,
            SdkClientException, PrivateNetworksException {
        return new ListNetworkSitesIterable(this, applyPaginatorUserAgent(listNetworkSitesRequest));
    }

    /**
     * <p>
     * Lists networks. Add filters to your request to return a more specific list of results. Use filters to match the
     * status of the network.
     * </p>
     *
     * @param listNetworksRequest
     * @return Result of the ListNetworks operation returned by the service.
     * @throws ResourceNotFoundException
     *         The resource was not found.
     * @throws ValidationException
     *         The request failed validation.
     * @throws InternalServerException
     *         Information about an internal error.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws PrivateNetworksException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample PrivateNetworksClient.ListNetworks
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/privatenetworks-2021-12-03/ListNetworks" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public ListNetworksResponse listNetworks(ListNetworksRequest listNetworksRequest) throws ResourceNotFoundException,
            ValidationException, InternalServerException, AwsServiceException, SdkClientException, PrivateNetworksException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListNetworksResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListNetworksResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listNetworksRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "PrivateNetworks");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListNetworks");

            return clientHandler.execute(new ClientExecutionParams<ListNetworksRequest, ListNetworksResponse>()
                    .withOperationName("ListNetworks").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(listNetworksRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListNetworksRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Lists networks. Add filters to your request to return a more specific list of results. Use filters to match the
     * status of the network.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listNetworks(software.amazon.awssdk.services.privatenetworks.model.ListNetworksRequest)} operation. The
     * return type is a custom iterable that can be used to iterate through all the pages. SDK will internally handle
     * making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.privatenetworks.paginators.ListNetworksIterable responses = client.listNetworksPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.privatenetworks.paginators.ListNetworksIterable responses = client
     *             .listNetworksPaginator(request);
     *     for (software.amazon.awssdk.services.privatenetworks.model.ListNetworksResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.privatenetworks.paginators.ListNetworksIterable responses = client.listNetworksPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of maxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listNetworks(software.amazon.awssdk.services.privatenetworks.model.ListNetworksRequest)} operation.</b>
     * </p>
     *
     * @param listNetworksRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws ResourceNotFoundException
     *         The resource was not found.
     * @throws ValidationException
     *         The request failed validation.
     * @throws InternalServerException
     *         Information about an internal error.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws PrivateNetworksException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample PrivateNetworksClient.ListNetworks
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/privatenetworks-2021-12-03/ListNetworks" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public ListNetworksIterable listNetworksPaginator(ListNetworksRequest listNetworksRequest) throws ResourceNotFoundException,
            ValidationException, InternalServerException, AwsServiceException, SdkClientException, PrivateNetworksException {
        return new ListNetworksIterable(this, applyPaginatorUserAgent(listNetworksRequest));
    }

    /**
     * <p>
     * Lists orders. Add filters to your request to return a more specific list of results. Use filters to match the
     * Amazon Resource Name (ARN) of the network site or the status of the order.
     * </p>
     * <p>
     * If you specify multiple filters, filters are joined with an OR, and the request returns results that match all of
     * the specified filters.
     * </p>
     *
     * @param listOrdersRequest
     * @return Result of the ListOrders operation returned by the service.
     * @throws ResourceNotFoundException
     *         The resource was not found.
     * @throws ValidationException
     *         The request failed validation.
     * @throws InternalServerException
     *         Information about an internal error.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws PrivateNetworksException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample PrivateNetworksClient.ListOrders
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/privatenetworks-2021-12-03/ListOrders" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public ListOrdersResponse listOrders(ListOrdersRequest listOrdersRequest) throws ResourceNotFoundException,
            ValidationException, InternalServerException, AwsServiceException, SdkClientException, PrivateNetworksException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListOrdersResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListOrdersResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listOrdersRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "PrivateNetworks");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListOrders");

            return clientHandler
                    .execute(new ClientExecutionParams<ListOrdersRequest, ListOrdersResponse>().withOperationName("ListOrders")
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(listOrdersRequest).withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new ListOrdersRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Lists orders. Add filters to your request to return a more specific list of results. Use filters to match the
     * Amazon Resource Name (ARN) of the network site or the status of the order.
     * </p>
     * <p>
     * If you specify multiple filters, filters are joined with an OR, and the request returns results that match all of
     * the specified filters.
     * </p>
     * <br/>
     * <p>
     * This is a variant of {@link #listOrders(software.amazon.awssdk.services.privatenetworks.model.ListOrdersRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.privatenetworks.paginators.ListOrdersIterable responses = client.listOrdersPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.privatenetworks.paginators.ListOrdersIterable responses = client.listOrdersPaginator(request);
     *     for (software.amazon.awssdk.services.privatenetworks.model.ListOrdersResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.privatenetworks.paginators.ListOrdersIterable responses = client.listOrdersPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of maxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listOrders(software.amazon.awssdk.services.privatenetworks.model.ListOrdersRequest)} operation.</b>
     * </p>
     *
     * @param listOrdersRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws ResourceNotFoundException
     *         The resource was not found.
     * @throws ValidationException
     *         The request failed validation.
     * @throws InternalServerException
     *         Information about an internal error.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws PrivateNetworksException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample PrivateNetworksClient.ListOrders
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/privatenetworks-2021-12-03/ListOrders" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public ListOrdersIterable listOrdersPaginator(ListOrdersRequest listOrdersRequest) throws ResourceNotFoundException,
            ValidationException, InternalServerException, AwsServiceException, SdkClientException, PrivateNetworksException {
        return new ListOrdersIterable(this, applyPaginatorUserAgent(listOrdersRequest));
    }

    /**
     * <p>
     * Lists the tags for the specified resource.
     * </p>
     *
     * @param listTagsForResourceRequest
     * @return Result of the ListTagsForResource operation returned by the service.
     * @throws ResourceNotFoundException
     *         The resource was not found.
     * @throws AccessDeniedException
     *         You do not have permission to perform this operation.
     * @throws ValidationException
     *         The request failed validation.
     * @throws InternalServerException
     *         Information about an internal error.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws PrivateNetworksException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample PrivateNetworksClient.ListTagsForResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/privatenetworks-2021-12-03/ListTagsForResource"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListTagsForResourceResponse listTagsForResource(ListTagsForResourceRequest listTagsForResourceRequest)
            throws ResourceNotFoundException, AccessDeniedException, ValidationException, InternalServerException,
            ThrottlingException, AwsServiceException, SdkClientException, PrivateNetworksException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListTagsForResourceResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListTagsForResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTagsForResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "PrivateNetworks");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTagsForResource");

            return clientHandler.execute(new ClientExecutionParams<ListTagsForResourceRequest, ListTagsForResourceResponse>()
                    .withOperationName("ListTagsForResource").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(listTagsForResourceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListTagsForResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Checks the health of the service.
     * </p>
     *
     * @param pingRequest
     * @return Result of the Ping operation returned by the service.
     * @throws InternalServerException
     *         Information about an internal error.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws PrivateNetworksException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample PrivateNetworksClient.Ping
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/privatenetworks-2021-12-03/Ping" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public PingResponse ping(PingRequest pingRequest) throws InternalServerException, AwsServiceException, SdkClientException,
            PrivateNetworksException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<PingResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                PingResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, pingRequest.overrideConfiguration()
                .orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "PrivateNetworks");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "Ping");

            return clientHandler.execute(new ClientExecutionParams<PingRequest, PingResponse>().withOperationName("Ping")
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler).withInput(pingRequest)
                    .withMetricCollector(apiCallMetricCollector).withMarshaller(new PingRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Starts an update of the specified network resource.
     * </p>
     * <p>
     * After you submit a request to replace or return a network resource, the status of the network resource is
     * <code>CREATING_SHIPPING_LABEL</code>. The shipping label is available when the status of the network resource is
     * <code>PENDING_RETURN</code>. After the network resource is successfully returned, its status is
     * <code>DELETED</code>. For more information, see <a
     * href="https://docs.aws.amazon.com/private-networks/latest/userguide/radio-units.html#return-radio-unit">Return a
     * radio unit</a>.
     * </p>
     *
     * @param startNetworkResourceUpdateRequest
     * @return Result of the StartNetworkResourceUpdate operation returned by the service.
     * @throws ResourceNotFoundException
     *         The resource was not found.
     * @throws ValidationException
     *         The request failed validation.
     * @throws InternalServerException
     *         Information about an internal error.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws PrivateNetworksException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample PrivateNetworksClient.StartNetworkResourceUpdate
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/privatenetworks-2021-12-03/StartNetworkResourceUpdate"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public StartNetworkResourceUpdateResponse startNetworkResourceUpdate(
            StartNetworkResourceUpdateRequest startNetworkResourceUpdateRequest) throws ResourceNotFoundException,
            ValidationException, InternalServerException, AwsServiceException, SdkClientException, PrivateNetworksException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<StartNetworkResourceUpdateResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, StartNetworkResourceUpdateResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, startNetworkResourceUpdateRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "PrivateNetworks");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "StartNetworkResourceUpdate");

            return clientHandler
                    .execute(new ClientExecutionParams<StartNetworkResourceUpdateRequest, StartNetworkResourceUpdateResponse>()
                            .withOperationName("StartNetworkResourceUpdate").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(startNetworkResourceUpdateRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new StartNetworkResourceUpdateRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Adds tags to the specified resource.
     * </p>
     *
     * @param tagResourceRequest
     * @return Result of the TagResource operation returned by the service.
     * @throws ResourceNotFoundException
     *         The resource was not found.
     * @throws AccessDeniedException
     *         You do not have permission to perform this operation.
     * @throws ValidationException
     *         The request failed validation.
     * @throws InternalServerException
     *         Information about an internal error.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws PrivateNetworksException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample PrivateNetworksClient.TagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/privatenetworks-2021-12-03/TagResource" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public TagResourceResponse tagResource(TagResourceRequest tagResourceRequest) throws ResourceNotFoundException,
            AccessDeniedException, ValidationException, InternalServerException, ThrottlingException, AwsServiceException,
            SdkClientException, PrivateNetworksException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<TagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                TagResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, tagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "PrivateNetworks");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "TagResource");

            return clientHandler.execute(new ClientExecutionParams<TagResourceRequest, TagResourceResponse>()
                    .withOperationName("TagResource").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(tagResourceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new TagResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Removes tags from the specified resource.
     * </p>
     *
     * @param untagResourceRequest
     * @return Result of the UntagResource operation returned by the service.
     * @throws ResourceNotFoundException
     *         The resource was not found.
     * @throws AccessDeniedException
     *         You do not have permission to perform this operation.
     * @throws ValidationException
     *         The request failed validation.
     * @throws InternalServerException
     *         Information about an internal error.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws PrivateNetworksException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample PrivateNetworksClient.UntagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/privatenetworks-2021-12-03/UntagResource" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public UntagResourceResponse untagResource(UntagResourceRequest untagResourceRequest) throws ResourceNotFoundException,
            AccessDeniedException, ValidationException, InternalServerException, ThrottlingException, AwsServiceException,
            SdkClientException, PrivateNetworksException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UntagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                UntagResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, untagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "PrivateNetworks");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UntagResource");

            return clientHandler.execute(new ClientExecutionParams<UntagResourceRequest, UntagResourceResponse>()
                    .withOperationName("UntagResource").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(untagResourceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UntagResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Updates the specified network site.
     * </p>
     *
     * @param updateNetworkSiteRequest
     * @return Result of the UpdateNetworkSite operation returned by the service.
     * @throws ResourceNotFoundException
     *         The resource was not found.
     * @throws ValidationException
     *         The request failed validation.
     * @throws InternalServerException
     *         Information about an internal error.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws PrivateNetworksException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample PrivateNetworksClient.UpdateNetworkSite
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/privatenetworks-2021-12-03/UpdateNetworkSite"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public UpdateNetworkSiteResponse updateNetworkSite(UpdateNetworkSiteRequest updateNetworkSiteRequest)
            throws ResourceNotFoundException, ValidationException, InternalServerException, AwsServiceException,
            SdkClientException, PrivateNetworksException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateNetworkSiteResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                UpdateNetworkSiteResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateNetworkSiteRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "PrivateNetworks");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateNetworkSite");

            return clientHandler.execute(new ClientExecutionParams<UpdateNetworkSiteRequest, UpdateNetworkSiteResponse>()
                    .withOperationName("UpdateNetworkSite").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(updateNetworkSiteRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UpdateNetworkSiteRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Updates the specified network site plan.
     * </p>
     *
     * @param updateNetworkSitePlanRequest
     * @return Result of the UpdateNetworkSitePlan operation returned by the service.
     * @throws ResourceNotFoundException
     *         The resource was not found.
     * @throws ValidationException
     *         The request failed validation.
     * @throws InternalServerException
     *         Information about an internal error.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws PrivateNetworksException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample PrivateNetworksClient.UpdateNetworkSitePlan
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/privatenetworks-2021-12-03/UpdateNetworkSitePlan"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public UpdateNetworkSitePlanResponse updateNetworkSitePlan(UpdateNetworkSitePlanRequest updateNetworkSitePlanRequest)
            throws ResourceNotFoundException, ValidationException, InternalServerException, AwsServiceException,
            SdkClientException, PrivateNetworksException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateNetworkSitePlanResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, UpdateNetworkSitePlanResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateNetworkSitePlanRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "PrivateNetworks");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateNetworkSitePlan");

            return clientHandler.execute(new ClientExecutionParams<UpdateNetworkSitePlanRequest, UpdateNetworkSitePlanResponse>()
                    .withOperationName("UpdateNetworkSitePlan").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(updateNetworkSitePlanRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UpdateNetworkSitePlanRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    private <T extends PrivateNetworksRequest> T applyPaginatorUserAgent(T request) {
        Consumer<AwsRequestOverrideConfiguration.Builder> userAgentApplier = b -> b.addApiName(ApiName.builder()
                .version(VersionInfo.SDK_VERSION).name("PAGINATED").build());
        AwsRequestOverrideConfiguration overrideConfiguration = request.overrideConfiguration()
                .map(c -> c.toBuilder().applyMutation(userAgentApplier).build())
                .orElse((AwsRequestOverrideConfiguration.builder().applyMutation(userAgentApplier).build()));
        return (T) request.toBuilder().overrideConfiguration(overrideConfiguration).build();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(PrivateNetworksException::builder)
                .protocol(AwsJsonProtocol.REST_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("AccessDeniedException")
                                .exceptionBuilderSupplier(AccessDeniedException::builder).httpStatusCode(403).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceNotFoundException")
                                .exceptionBuilderSupplier(ResourceNotFoundException::builder).httpStatusCode(404).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ThrottlingException")
                                .exceptionBuilderSupplier(ThrottlingException::builder).httpStatusCode(429).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ValidationException")
                                .exceptionBuilderSupplier(ValidationException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalServerException")
                                .exceptionBuilderSupplier(InternalServerException::builder).httpStatusCode(500).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("LimitExceededException")
                                .exceptionBuilderSupplier(LimitExceededException::builder).httpStatusCode(400).build());
    }

    @Override
    public final PrivateNetworksServiceClientConfiguration serviceClientConfiguration() {
        return this.serviceClientConfiguration;
    }

    @Override
    public void close() {
        clientHandler.close();
    }
}
