/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.osis;

import java.util.Collections;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.client.handler.AwsAsyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.awscore.internal.AwsProtocolMetadata;
import software.amazon.awssdk.awscore.internal.AwsServiceProtocol;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkPlugin;
import software.amazon.awssdk.core.SdkRequest;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.AsyncClientHandler;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.osis.internal.OsisServiceClientConfigurationBuilder;
import software.amazon.awssdk.services.osis.model.AccessDeniedException;
import software.amazon.awssdk.services.osis.model.ConflictException;
import software.amazon.awssdk.services.osis.model.CreatePipelineRequest;
import software.amazon.awssdk.services.osis.model.CreatePipelineResponse;
import software.amazon.awssdk.services.osis.model.DeletePipelineRequest;
import software.amazon.awssdk.services.osis.model.DeletePipelineResponse;
import software.amazon.awssdk.services.osis.model.GetPipelineBlueprintRequest;
import software.amazon.awssdk.services.osis.model.GetPipelineBlueprintResponse;
import software.amazon.awssdk.services.osis.model.GetPipelineChangeProgressRequest;
import software.amazon.awssdk.services.osis.model.GetPipelineChangeProgressResponse;
import software.amazon.awssdk.services.osis.model.GetPipelineRequest;
import software.amazon.awssdk.services.osis.model.GetPipelineResponse;
import software.amazon.awssdk.services.osis.model.InternalException;
import software.amazon.awssdk.services.osis.model.InvalidPaginationTokenException;
import software.amazon.awssdk.services.osis.model.LimitExceededException;
import software.amazon.awssdk.services.osis.model.ListPipelineBlueprintsRequest;
import software.amazon.awssdk.services.osis.model.ListPipelineBlueprintsResponse;
import software.amazon.awssdk.services.osis.model.ListPipelinesRequest;
import software.amazon.awssdk.services.osis.model.ListPipelinesResponse;
import software.amazon.awssdk.services.osis.model.ListTagsForResourceRequest;
import software.amazon.awssdk.services.osis.model.ListTagsForResourceResponse;
import software.amazon.awssdk.services.osis.model.OsisException;
import software.amazon.awssdk.services.osis.model.ResourceAlreadyExistsException;
import software.amazon.awssdk.services.osis.model.ResourceNotFoundException;
import software.amazon.awssdk.services.osis.model.StartPipelineRequest;
import software.amazon.awssdk.services.osis.model.StartPipelineResponse;
import software.amazon.awssdk.services.osis.model.StopPipelineRequest;
import software.amazon.awssdk.services.osis.model.StopPipelineResponse;
import software.amazon.awssdk.services.osis.model.TagResourceRequest;
import software.amazon.awssdk.services.osis.model.TagResourceResponse;
import software.amazon.awssdk.services.osis.model.UntagResourceRequest;
import software.amazon.awssdk.services.osis.model.UntagResourceResponse;
import software.amazon.awssdk.services.osis.model.UpdatePipelineRequest;
import software.amazon.awssdk.services.osis.model.UpdatePipelineResponse;
import software.amazon.awssdk.services.osis.model.ValidatePipelineRequest;
import software.amazon.awssdk.services.osis.model.ValidatePipelineResponse;
import software.amazon.awssdk.services.osis.model.ValidationException;
import software.amazon.awssdk.services.osis.transform.CreatePipelineRequestMarshaller;
import software.amazon.awssdk.services.osis.transform.DeletePipelineRequestMarshaller;
import software.amazon.awssdk.services.osis.transform.GetPipelineBlueprintRequestMarshaller;
import software.amazon.awssdk.services.osis.transform.GetPipelineChangeProgressRequestMarshaller;
import software.amazon.awssdk.services.osis.transform.GetPipelineRequestMarshaller;
import software.amazon.awssdk.services.osis.transform.ListPipelineBlueprintsRequestMarshaller;
import software.amazon.awssdk.services.osis.transform.ListPipelinesRequestMarshaller;
import software.amazon.awssdk.services.osis.transform.ListTagsForResourceRequestMarshaller;
import software.amazon.awssdk.services.osis.transform.StartPipelineRequestMarshaller;
import software.amazon.awssdk.services.osis.transform.StopPipelineRequestMarshaller;
import software.amazon.awssdk.services.osis.transform.TagResourceRequestMarshaller;
import software.amazon.awssdk.services.osis.transform.UntagResourceRequestMarshaller;
import software.amazon.awssdk.services.osis.transform.UpdatePipelineRequestMarshaller;
import software.amazon.awssdk.services.osis.transform.ValidatePipelineRequestMarshaller;
import software.amazon.awssdk.utils.CompletableFutureUtils;

/**
 * Internal implementation of {@link OsisAsyncClient}.
 *
 * @see OsisAsyncClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultOsisAsyncClient implements OsisAsyncClient {
    private static final Logger log = LoggerFactory.getLogger(DefaultOsisAsyncClient.class);

    private static final AwsProtocolMetadata protocolMetadata = AwsProtocolMetadata.builder()
            .serviceProtocol(AwsServiceProtocol.REST_JSON).build();

    private final AsyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    private final OsisServiceClientConfiguration serviceClientConfiguration;

    protected DefaultOsisAsyncClient(OsisServiceClientConfiguration serviceClientConfiguration,
            SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsAsyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.serviceClientConfiguration = serviceClientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    /**
     * <p>
     * Creates an OpenSearch Ingestion pipeline. For more information, see <a
     * href="https://docs.aws.amazon.com/opensearch-service/latest/developerguide/creating-pipeline.html">Creating
     * Amazon OpenSearch Ingestion pipelines</a>.
     * </p>
     *
     * @param createPipelineRequest
     * @return A Java Future containing the result of the CreatePipeline operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>LimitExceededException You attempted to create more than the allowed number of tags.</li>
     *         <li>ValidationException An exception for missing or invalid input fields.</li>
     *         <li>InternalException The request failed because of an unknown error, exception, or failure (the failure
     *         is internal to the service).</li>
     *         <li>AccessDeniedException You don't have permissions to access the resource.</li>
     *         <li>ResourceAlreadyExistsException You attempted to create a resource that already exists.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>OsisException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample OsisAsyncClient.CreatePipeline
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/osis-2022-01-01/CreatePipeline" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<CreatePipelineResponse> createPipeline(CreatePipelineRequest createPipelineRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createPipelineRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createPipelineRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "OSIS");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreatePipeline");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreatePipelineResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreatePipelineResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreatePipelineResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreatePipelineRequest, CreatePipelineResponse>()
                            .withOperationName("CreatePipeline").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new CreatePipelineRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(createPipelineRequest));
            CompletableFuture<CreatePipelineResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes an OpenSearch Ingestion pipeline. For more information, see <a
     * href="https://docs.aws.amazon.com/opensearch-service/latest/developerguide/delete-pipeline.html">Deleting Amazon
     * OpenSearch Ingestion pipelines</a>.
     * </p>
     *
     * @param deletePipelineRequest
     * @return A Java Future containing the result of the DeletePipeline operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException An exception for missing or invalid input fields.</li>
     *         <li>InternalException The request failed because of an unknown error, exception, or failure (the failure
     *         is internal to the service).</li>
     *         <li>AccessDeniedException You don't have permissions to access the resource.</li>
     *         <li>ResourceNotFoundException You attempted to access or delete a resource that does not exist.</li>
     *         <li>ConflictException The client attempted to remove a resource that is currently in use.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>OsisException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample OsisAsyncClient.DeletePipeline
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/osis-2022-01-01/DeletePipeline" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DeletePipelineResponse> deletePipeline(DeletePipelineRequest deletePipelineRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deletePipelineRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deletePipelineRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "OSIS");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeletePipeline");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeletePipelineResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeletePipelineResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeletePipelineResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeletePipelineRequest, DeletePipelineResponse>()
                            .withOperationName("DeletePipeline").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DeletePipelineRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deletePipelineRequest));
            CompletableFuture<DeletePipelineResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves information about an OpenSearch Ingestion pipeline.
     * </p>
     *
     * @param getPipelineRequest
     * @return A Java Future containing the result of the GetPipeline operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException An exception for missing or invalid input fields.</li>
     *         <li>InternalException The request failed because of an unknown error, exception, or failure (the failure
     *         is internal to the service).</li>
     *         <li>AccessDeniedException You don't have permissions to access the resource.</li>
     *         <li>ResourceNotFoundException You attempted to access or delete a resource that does not exist.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>OsisException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample OsisAsyncClient.GetPipeline
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/osis-2022-01-01/GetPipeline" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<GetPipelineResponse> getPipeline(GetPipelineRequest getPipelineRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getPipelineRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getPipelineRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "OSIS");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetPipeline");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetPipelineResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    GetPipelineResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetPipelineResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetPipelineRequest, GetPipelineResponse>()
                            .withOperationName("GetPipeline").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetPipelineRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getPipelineRequest));
            CompletableFuture<GetPipelineResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves information about a specific blueprint for OpenSearch Ingestion. Blueprints are templates for the
     * configuration needed for a <code>CreatePipeline</code> request. For more information, see <a href=
     * "https://docs.aws.amazon.com/opensearch-service/latest/developerguide/creating-pipeline.html#pipeline-blueprint"
     * >Using blueprints to create a pipeline</a>.
     * </p>
     *
     * @param getPipelineBlueprintRequest
     * @return A Java Future containing the result of the GetPipelineBlueprint operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException You don't have permissions to access the resource.</li>
     *         <li>InternalException The request failed because of an unknown error, exception, or failure (the failure
     *         is internal to the service).</li>
     *         <li>ValidationException An exception for missing or invalid input fields.</li>
     *         <li>ResourceNotFoundException You attempted to access or delete a resource that does not exist.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>OsisException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample OsisAsyncClient.GetPipelineBlueprint
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/osis-2022-01-01/GetPipelineBlueprint" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<GetPipelineBlueprintResponse> getPipelineBlueprint(
            GetPipelineBlueprintRequest getPipelineBlueprintRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getPipelineBlueprintRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getPipelineBlueprintRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "OSIS");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetPipelineBlueprint");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetPipelineBlueprintResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetPipelineBlueprintResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetPipelineBlueprintResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetPipelineBlueprintRequest, GetPipelineBlueprintResponse>()
                            .withOperationName("GetPipelineBlueprint").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetPipelineBlueprintRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getPipelineBlueprintRequest));
            CompletableFuture<GetPipelineBlueprintResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns progress information for the current change happening on an OpenSearch Ingestion pipeline. Currently,
     * this operation only returns information when a pipeline is being created.
     * </p>
     * <p>
     * For more information, see <a href=
     * "https://docs.aws.amazon.com/opensearch-service/latest/developerguide/creating-pipeline.html#get-pipeline-progress"
     * >Tracking the status of pipeline creation</a>.
     * </p>
     *
     * @param getPipelineChangeProgressRequest
     * @return A Java Future containing the result of the GetPipelineChangeProgress operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException An exception for missing or invalid input fields.</li>
     *         <li>InternalException The request failed because of an unknown error, exception, or failure (the failure
     *         is internal to the service).</li>
     *         <li>AccessDeniedException You don't have permissions to access the resource.</li>
     *         <li>ResourceNotFoundException You attempted to access or delete a resource that does not exist.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>OsisException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample OsisAsyncClient.GetPipelineChangeProgress
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/osis-2022-01-01/GetPipelineChangeProgress"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetPipelineChangeProgressResponse> getPipelineChangeProgress(
            GetPipelineChangeProgressRequest getPipelineChangeProgressRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getPipelineChangeProgressRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getPipelineChangeProgressRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "OSIS");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetPipelineChangeProgress");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetPipelineChangeProgressResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetPipelineChangeProgressResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetPipelineChangeProgressResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetPipelineChangeProgressRequest, GetPipelineChangeProgressResponse>()
                            .withOperationName("GetPipelineChangeProgress").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetPipelineChangeProgressRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getPipelineChangeProgressRequest));
            CompletableFuture<GetPipelineChangeProgressResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves a list of all available blueprints for Data Prepper. For more information, see <a href=
     * "https://docs.aws.amazon.com/opensearch-service/latest/developerguide/creating-pipeline.html#pipeline-blueprint"
     * >Using blueprints to create a pipeline</a>.
     * </p>
     *
     * @param listPipelineBlueprintsRequest
     * @return A Java Future containing the result of the ListPipelineBlueprints operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException An exception for missing or invalid input fields.</li>
     *         <li>InternalException The request failed because of an unknown error, exception, or failure (the failure
     *         is internal to the service).</li>
     *         <li>AccessDeniedException You don't have permissions to access the resource.</li>
     *         <li>InvalidPaginationTokenException An invalid pagination token provided in the request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>OsisException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample OsisAsyncClient.ListPipelineBlueprints
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/osis-2022-01-01/ListPipelineBlueprints" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<ListPipelineBlueprintsResponse> listPipelineBlueprints(
            ListPipelineBlueprintsRequest listPipelineBlueprintsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listPipelineBlueprintsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listPipelineBlueprintsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "OSIS");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListPipelineBlueprints");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListPipelineBlueprintsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListPipelineBlueprintsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListPipelineBlueprintsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListPipelineBlueprintsRequest, ListPipelineBlueprintsResponse>()
                            .withOperationName("ListPipelineBlueprints").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListPipelineBlueprintsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listPipelineBlueprintsRequest));
            CompletableFuture<ListPipelineBlueprintsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists all OpenSearch Ingestion pipelines in the current Amazon Web Services account and Region. For more
     * information, see <a
     * href="https://docs.aws.amazon.com/opensearch-service/latest/developerguide/list-pipeline.html">Viewing Amazon
     * OpenSearch Ingestion pipelines</a>.
     * </p>
     *
     * @param listPipelinesRequest
     * @return A Java Future containing the result of the ListPipelines operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException An exception for missing or invalid input fields.</li>
     *         <li>InternalException The request failed because of an unknown error, exception, or failure (the failure
     *         is internal to the service).</li>
     *         <li>AccessDeniedException You don't have permissions to access the resource.</li>
     *         <li>InvalidPaginationTokenException An invalid pagination token provided in the request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>OsisException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample OsisAsyncClient.ListPipelines
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/osis-2022-01-01/ListPipelines" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListPipelinesResponse> listPipelines(ListPipelinesRequest listPipelinesRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listPipelinesRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listPipelinesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "OSIS");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListPipelines");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListPipelinesResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    ListPipelinesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListPipelinesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListPipelinesRequest, ListPipelinesResponse>()
                            .withOperationName("ListPipelines").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListPipelinesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listPipelinesRequest));
            CompletableFuture<ListPipelinesResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists all resource tags associated with an OpenSearch Ingestion pipeline. For more information, see <a
     * href="https://docs.aws.amazon.com/opensearch-service/latest/developerguide/tag-pipeline.html">Tagging Amazon
     * OpenSearch Ingestion pipelines</a>.
     * </p>
     *
     * @param listTagsForResourceRequest
     * @return A Java Future containing the result of the ListTagsForResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException An exception for missing or invalid input fields.</li>
     *         <li>ResourceNotFoundException You attempted to access or delete a resource that does not exist.</li>
     *         <li>InternalException The request failed because of an unknown error, exception, or failure (the failure
     *         is internal to the service).</li>
     *         <li>AccessDeniedException You don't have permissions to access the resource.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>OsisException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample OsisAsyncClient.ListTagsForResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/osis-2022-01-01/ListTagsForResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListTagsForResourceResponse> listTagsForResource(
            ListTagsForResourceRequest listTagsForResourceRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listTagsForResourceRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTagsForResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "OSIS");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTagsForResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListTagsForResourceResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListTagsForResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListTagsForResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListTagsForResourceRequest, ListTagsForResourceResponse>()
                            .withOperationName("ListTagsForResource").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListTagsForResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listTagsForResourceRequest));
            CompletableFuture<ListTagsForResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Starts an OpenSearch Ingestion pipeline. For more information, see <a href=
     * "https://docs.aws.amazon.com/opensearch-service/latest/developerguide/pipeline--stop-start.html#pipeline--start"
     * >Starting an OpenSearch Ingestion pipeline</a>.
     * </p>
     *
     * @param startPipelineRequest
     * @return A Java Future containing the result of the StartPipeline operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException You don't have permissions to access the resource.</li>
     *         <li>ConflictException The client attempted to remove a resource that is currently in use.</li>
     *         <li>InternalException The request failed because of an unknown error, exception, or failure (the failure
     *         is internal to the service).</li>
     *         <li>ResourceNotFoundException You attempted to access or delete a resource that does not exist.</li>
     *         <li>ValidationException An exception for missing or invalid input fields.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>OsisException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample OsisAsyncClient.StartPipeline
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/osis-2022-01-01/StartPipeline" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<StartPipelineResponse> startPipeline(StartPipelineRequest startPipelineRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(startPipelineRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, startPipelineRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "OSIS");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "StartPipeline");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<StartPipelineResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    StartPipelineResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<StartPipelineResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<StartPipelineRequest, StartPipelineResponse>()
                            .withOperationName("StartPipeline").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new StartPipelineRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(startPipelineRequest));
            CompletableFuture<StartPipelineResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Stops an OpenSearch Ingestion pipeline. For more information, see <a href=
     * "https://docs.aws.amazon.com/opensearch-service/latest/developerguide/pipeline--stop-start.html#pipeline--stop"
     * >Stopping an OpenSearch Ingestion pipeline</a>.
     * </p>
     *
     * @param stopPipelineRequest
     * @return A Java Future containing the result of the StopPipeline operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException You don't have permissions to access the resource.</li>
     *         <li>ConflictException The client attempted to remove a resource that is currently in use.</li>
     *         <li>InternalException The request failed because of an unknown error, exception, or failure (the failure
     *         is internal to the service).</li>
     *         <li>ResourceNotFoundException You attempted to access or delete a resource that does not exist.</li>
     *         <li>ValidationException An exception for missing or invalid input fields.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>OsisException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample OsisAsyncClient.StopPipeline
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/osis-2022-01-01/StopPipeline" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<StopPipelineResponse> stopPipeline(StopPipelineRequest stopPipelineRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(stopPipelineRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, stopPipelineRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "OSIS");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "StopPipeline");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<StopPipelineResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    StopPipelineResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<StopPipelineResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<StopPipelineRequest, StopPipelineResponse>()
                            .withOperationName("StopPipeline").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new StopPipelineRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(stopPipelineRequest));
            CompletableFuture<StopPipelineResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Tags an OpenSearch Ingestion pipeline. For more information, see <a
     * href="https://docs.aws.amazon.com/opensearch-service/latest/developerguide/tag-pipeline.html">Tagging Amazon
     * OpenSearch Ingestion pipelines</a>.
     * </p>
     *
     * @param tagResourceRequest
     * @return A Java Future containing the result of the TagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>LimitExceededException You attempted to create more than the allowed number of tags.</li>
     *         <li>ValidationException An exception for missing or invalid input fields.</li>
     *         <li>InternalException The request failed because of an unknown error, exception, or failure (the failure
     *         is internal to the service).</li>
     *         <li>AccessDeniedException You don't have permissions to access the resource.</li>
     *         <li>ResourceNotFoundException You attempted to access or delete a resource that does not exist.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>OsisException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample OsisAsyncClient.TagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/osis-2022-01-01/TagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<TagResourceResponse> tagResource(TagResourceRequest tagResourceRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(tagResourceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, tagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "OSIS");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "TagResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<TagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    TagResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<TagResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<TagResourceRequest, TagResourceResponse>()
                            .withOperationName("TagResource").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new TagResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(tagResourceRequest));
            CompletableFuture<TagResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Removes one or more tags from an OpenSearch Ingestion pipeline. For more information, see <a
     * href="https://docs.aws.amazon.com/opensearch-service/latest/developerguide/tag-pipeline.html">Tagging Amazon
     * OpenSearch Ingestion pipelines</a>.
     * </p>
     *
     * @param untagResourceRequest
     * @return A Java Future containing the result of the UntagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException An exception for missing or invalid input fields.</li>
     *         <li>ResourceNotFoundException You attempted to access or delete a resource that does not exist.</li>
     *         <li>InternalException The request failed because of an unknown error, exception, or failure (the failure
     *         is internal to the service).</li>
     *         <li>AccessDeniedException You don't have permissions to access the resource.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>OsisException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample OsisAsyncClient.UntagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/osis-2022-01-01/UntagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<UntagResourceResponse> untagResource(UntagResourceRequest untagResourceRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(untagResourceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, untagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "OSIS");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UntagResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UntagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    UntagResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UntagResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UntagResourceRequest, UntagResourceResponse>()
                            .withOperationName("UntagResource").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new UntagResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(untagResourceRequest));
            CompletableFuture<UntagResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates an OpenSearch Ingestion pipeline. For more information, see <a
     * href="https://docs.aws.amazon.com/opensearch-service/latest/developerguide/update-pipeline.html">Updating Amazon
     * OpenSearch Ingestion pipelines</a>.
     * </p>
     *
     * @param updatePipelineRequest
     * @return A Java Future containing the result of the UpdatePipeline operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException An exception for missing or invalid input fields.</li>
     *         <li>InternalException The request failed because of an unknown error, exception, or failure (the failure
     *         is internal to the service).</li>
     *         <li>AccessDeniedException You don't have permissions to access the resource.</li>
     *         <li>ResourceNotFoundException You attempted to access or delete a resource that does not exist.</li>
     *         <li>ConflictException The client attempted to remove a resource that is currently in use.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>OsisException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample OsisAsyncClient.UpdatePipeline
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/osis-2022-01-01/UpdatePipeline" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<UpdatePipelineResponse> updatePipeline(UpdatePipelineRequest updatePipelineRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updatePipelineRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updatePipelineRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "OSIS");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdatePipeline");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdatePipelineResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdatePipelineResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdatePipelineResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdatePipelineRequest, UpdatePipelineResponse>()
                            .withOperationName("UpdatePipeline").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new UpdatePipelineRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(updatePipelineRequest));
            CompletableFuture<UpdatePipelineResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Checks whether an OpenSearch Ingestion pipeline configuration is valid prior to creation. For more information,
     * see <a
     * href="https://docs.aws.amazon.com/opensearch-service/latest/developerguide/creating-pipeline.html">Creating
     * Amazon OpenSearch Ingestion pipelines</a>.
     * </p>
     *
     * @param validatePipelineRequest
     * @return A Java Future containing the result of the ValidatePipeline operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException You don't have permissions to access the resource.</li>
     *         <li>InternalException The request failed because of an unknown error, exception, or failure (the failure
     *         is internal to the service).</li>
     *         <li>ValidationException An exception for missing or invalid input fields.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>OsisException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample OsisAsyncClient.ValidatePipeline
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/osis-2022-01-01/ValidatePipeline" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ValidatePipelineResponse> validatePipeline(ValidatePipelineRequest validatePipelineRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(validatePipelineRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, validatePipelineRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "OSIS");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ValidatePipeline");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ValidatePipelineResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ValidatePipelineResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ValidatePipelineResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ValidatePipelineRequest, ValidatePipelineResponse>()
                            .withOperationName("ValidatePipeline").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ValidatePipelineRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(validatePipelineRequest));
            CompletableFuture<ValidatePipelineResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    @Override
    public final OsisServiceClientConfiguration serviceClientConfiguration() {
        return this.serviceClientConfiguration;
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(OsisException::builder)
                .protocol(AwsJsonProtocol.REST_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("AccessDeniedException")
                                .exceptionBuilderSupplier(AccessDeniedException::builder).httpStatusCode(403).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ConflictException")
                                .exceptionBuilderSupplier(ConflictException::builder).httpStatusCode(409).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceNotFoundException")
                                .exceptionBuilderSupplier(ResourceNotFoundException::builder).httpStatusCode(404).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ValidationException")
                                .exceptionBuilderSupplier(ValidationException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceAlreadyExistsException")
                                .exceptionBuilderSupplier(ResourceAlreadyExistsException::builder).httpStatusCode(409).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalException")
                                .exceptionBuilderSupplier(InternalException::builder).httpStatusCode(500).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidPaginationTokenException")
                                .exceptionBuilderSupplier(InvalidPaginationTokenException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("LimitExceededException")
                                .exceptionBuilderSupplier(LimitExceededException::builder).httpStatusCode(409).build());
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private SdkClientConfiguration updateSdkClientConfiguration(SdkRequest request, SdkClientConfiguration clientConfiguration) {
        List<SdkPlugin> plugins = request.overrideConfiguration().map(c -> c.plugins()).orElse(Collections.emptyList());
        if (plugins.isEmpty()) {
            return clientConfiguration;
        }
        OsisServiceClientConfigurationBuilder.BuilderInternal serviceConfigBuilder = OsisServiceClientConfigurationBuilder
                .builder(clientConfiguration.toBuilder());
        serviceConfigBuilder.overrideConfiguration(serviceClientConfiguration.overrideConfiguration());
        for (SdkPlugin plugin : plugins) {
            plugin.configureClient(serviceConfigBuilder);
        }
        return serviceConfigBuilder.buildSdkClientConfiguration();
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    @Override
    public void close() {
        clientHandler.close();
    }
}
