/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.managedblockchainquery;

import java.util.Collections;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.client.handler.AwsAsyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.AsyncClientHandler;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.managedblockchainquery.model.AccessDeniedException;
import software.amazon.awssdk.services.managedblockchainquery.model.BatchGetTokenBalanceRequest;
import software.amazon.awssdk.services.managedblockchainquery.model.BatchGetTokenBalanceResponse;
import software.amazon.awssdk.services.managedblockchainquery.model.GetTokenBalanceRequest;
import software.amazon.awssdk.services.managedblockchainquery.model.GetTokenBalanceResponse;
import software.amazon.awssdk.services.managedblockchainquery.model.GetTransactionRequest;
import software.amazon.awssdk.services.managedblockchainquery.model.GetTransactionResponse;
import software.amazon.awssdk.services.managedblockchainquery.model.InternalServerException;
import software.amazon.awssdk.services.managedblockchainquery.model.ListTokenBalancesRequest;
import software.amazon.awssdk.services.managedblockchainquery.model.ListTokenBalancesResponse;
import software.amazon.awssdk.services.managedblockchainquery.model.ListTransactionEventsRequest;
import software.amazon.awssdk.services.managedblockchainquery.model.ListTransactionEventsResponse;
import software.amazon.awssdk.services.managedblockchainquery.model.ListTransactionsRequest;
import software.amazon.awssdk.services.managedblockchainquery.model.ListTransactionsResponse;
import software.amazon.awssdk.services.managedblockchainquery.model.ManagedBlockchainQueryException;
import software.amazon.awssdk.services.managedblockchainquery.model.ResourceNotFoundException;
import software.amazon.awssdk.services.managedblockchainquery.model.ServiceQuotaExceededException;
import software.amazon.awssdk.services.managedblockchainquery.model.ThrottlingException;
import software.amazon.awssdk.services.managedblockchainquery.model.ValidationException;
import software.amazon.awssdk.services.managedblockchainquery.transform.BatchGetTokenBalanceRequestMarshaller;
import software.amazon.awssdk.services.managedblockchainquery.transform.GetTokenBalanceRequestMarshaller;
import software.amazon.awssdk.services.managedblockchainquery.transform.GetTransactionRequestMarshaller;
import software.amazon.awssdk.services.managedblockchainquery.transform.ListTokenBalancesRequestMarshaller;
import software.amazon.awssdk.services.managedblockchainquery.transform.ListTransactionEventsRequestMarshaller;
import software.amazon.awssdk.services.managedblockchainquery.transform.ListTransactionsRequestMarshaller;
import software.amazon.awssdk.utils.CompletableFutureUtils;

/**
 * Internal implementation of {@link ManagedBlockchainQueryAsyncClient}.
 *
 * @see ManagedBlockchainQueryAsyncClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultManagedBlockchainQueryAsyncClient implements ManagedBlockchainQueryAsyncClient {
    private static final Logger log = LoggerFactory.getLogger(DefaultManagedBlockchainQueryAsyncClient.class);

    private final AsyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    private final ManagedBlockchainQueryServiceClientConfiguration serviceClientConfiguration;

    protected DefaultManagedBlockchainQueryAsyncClient(
            ManagedBlockchainQueryServiceClientConfiguration serviceClientConfiguration,
            SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsAsyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.serviceClientConfiguration = serviceClientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    /**
     * <p>
     * Gets the token balance for a batch of tokens by using the <code>GetTokenBalance</code> action for every token in
     * the request.
     * </p>
     * <note>
     * <p>
     * Only the native tokens BTC,ETH, and the ERC-20, ERC-721, and ERC 1155 token standards are supported.
     * </p>
     * </note>
     *
     * @param batchGetTokenBalanceRequest
     * @return A Java Future containing the result of the BatchGetTokenBalance operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ThrottlingException The request or operation couldn't be performed because a service is throttling
     *         requests. The most common source of throttling errors is when you create resources that exceed your
     *         service limit for this resource type. Request a limit increase or delete unused resources, if possible.</li>
     *         <li>ValidationException The resource passed is invalid.</li>
     *         <li>ResourceNotFoundException The resource was not found.</li>
     *         <li>AccessDeniedException The Amazon Web Services account doesn’t have access to this resource.</li>
     *         <li>InternalServerException The request processing has failed because of an internal error in the
     *         service.</li>
     *         <li>ServiceQuotaExceededException The service quota has been exceeded for this resource.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ManagedBlockchainQueryException Base class for all service exceptions. Unknown exceptions will be
     *         thrown as an instance of this type.</li>
     *         </ul>
     * @sample ManagedBlockchainQueryAsyncClient.BatchGetTokenBalance
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/managedblockchain-query-2023-05-04/BatchGetTokenBalance"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<BatchGetTokenBalanceResponse> batchGetTokenBalance(
            BatchGetTokenBalanceRequest batchGetTokenBalanceRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, batchGetTokenBalanceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ManagedBlockchain Query");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "BatchGetTokenBalance");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<BatchGetTokenBalanceResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, BatchGetTokenBalanceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<BatchGetTokenBalanceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<BatchGetTokenBalanceRequest, BatchGetTokenBalanceResponse>()
                            .withOperationName("BatchGetTokenBalance")
                            .withMarshaller(new BatchGetTokenBalanceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(batchGetTokenBalanceRequest));
            CompletableFuture<BatchGetTokenBalanceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Gets the balance of a specific token, including native tokens, for a given address (wallet or contract) on the
     * blockchain.
     * </p>
     * <note>
     * <p>
     * Only the native tokens BTC,ETH, and the ERC-20, ERC-721, and ERC 1155 token standards are supported.
     * </p>
     * </note>
     *
     * @param getTokenBalanceRequest
     * @return A Java Future containing the result of the GetTokenBalance operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ThrottlingException The request or operation couldn't be performed because a service is throttling
     *         requests. The most common source of throttling errors is when you create resources that exceed your
     *         service limit for this resource type. Request a limit increase or delete unused resources, if possible.</li>
     *         <li>ValidationException The resource passed is invalid.</li>
     *         <li>ResourceNotFoundException The resource was not found.</li>
     *         <li>AccessDeniedException The Amazon Web Services account doesn’t have access to this resource.</li>
     *         <li>InternalServerException The request processing has failed because of an internal error in the
     *         service.</li>
     *         <li>ServiceQuotaExceededException The service quota has been exceeded for this resource.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ManagedBlockchainQueryException Base class for all service exceptions. Unknown exceptions will be
     *         thrown as an instance of this type.</li>
     *         </ul>
     * @sample ManagedBlockchainQueryAsyncClient.GetTokenBalance
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/managedblockchain-query-2023-05-04/GetTokenBalance"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetTokenBalanceResponse> getTokenBalance(GetTokenBalanceRequest getTokenBalanceRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getTokenBalanceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ManagedBlockchain Query");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetTokenBalance");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetTokenBalanceResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetTokenBalanceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetTokenBalanceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetTokenBalanceRequest, GetTokenBalanceResponse>()
                            .withOperationName("GetTokenBalance")
                            .withMarshaller(new GetTokenBalanceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(getTokenBalanceRequest));
            CompletableFuture<GetTokenBalanceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Get the details of a transaction.
     * </p>
     *
     * @param getTransactionRequest
     * @return A Java Future containing the result of the GetTransaction operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ThrottlingException The request or operation couldn't be performed because a service is throttling
     *         requests. The most common source of throttling errors is when you create resources that exceed your
     *         service limit for this resource type. Request a limit increase or delete unused resources, if possible.</li>
     *         <li>ValidationException The resource passed is invalid.</li>
     *         <li>ResourceNotFoundException The resource was not found.</li>
     *         <li>AccessDeniedException The Amazon Web Services account doesn’t have access to this resource.</li>
     *         <li>InternalServerException The request processing has failed because of an internal error in the
     *         service.</li>
     *         <li>ServiceQuotaExceededException The service quota has been exceeded for this resource.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ManagedBlockchainQueryException Base class for all service exceptions. Unknown exceptions will be
     *         thrown as an instance of this type.</li>
     *         </ul>
     * @sample ManagedBlockchainQueryAsyncClient.GetTransaction
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/managedblockchain-query-2023-05-04/GetTransaction"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetTransactionResponse> getTransaction(GetTransactionRequest getTransactionRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getTransactionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ManagedBlockchain Query");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetTransaction");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetTransactionResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetTransactionResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetTransactionResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetTransactionRequest, GetTransactionResponse>()
                            .withOperationName("GetTransaction")
                            .withMarshaller(new GetTransactionRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(getTransactionRequest));
            CompletableFuture<GetTransactionResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * This action returns the following for a given a blockchain network:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Lists all token balances owned by an address (either a contact address or a wallet address).
     * </p>
     * </li>
     * <li>
     * <p>
     * Lists all token balances for all tokens created by a contract.
     * </p>
     * </li>
     * <li>
     * <p>
     * Lists all token balances for a given token.
     * </p>
     * </li>
     * </ul>
     * <note>
     * <p>
     * You must always specify the network property of the <code>tokenFilter</code> when using this operation.
     * </p>
     * </note>
     *
     * @param listTokenBalancesRequest
     * @return A Java Future containing the result of the ListTokenBalances operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ThrottlingException The request or operation couldn't be performed because a service is throttling
     *         requests. The most common source of throttling errors is when you create resources that exceed your
     *         service limit for this resource type. Request a limit increase or delete unused resources, if possible.</li>
     *         <li>ValidationException The resource passed is invalid.</li>
     *         <li>AccessDeniedException The Amazon Web Services account doesn’t have access to this resource.</li>
     *         <li>InternalServerException The request processing has failed because of an internal error in the
     *         service.</li>
     *         <li>ServiceQuotaExceededException The service quota has been exceeded for this resource.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ManagedBlockchainQueryException Base class for all service exceptions. Unknown exceptions will be
     *         thrown as an instance of this type.</li>
     *         </ul>
     * @sample ManagedBlockchainQueryAsyncClient.ListTokenBalances
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/managedblockchain-query-2023-05-04/ListTokenBalances"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListTokenBalancesResponse> listTokenBalances(ListTokenBalancesRequest listTokenBalancesRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTokenBalancesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ManagedBlockchain Query");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTokenBalances");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListTokenBalancesResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListTokenBalancesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListTokenBalancesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListTokenBalancesRequest, ListTokenBalancesResponse>()
                            .withOperationName("ListTokenBalances")
                            .withMarshaller(new ListTokenBalancesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listTokenBalancesRequest));
            CompletableFuture<ListTokenBalancesResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * An array of <code>TransactionEvent</code> objects. Each object contains details about the transaction event.
     * </p>
     *
     * @param listTransactionEventsRequest
     * @return A Java Future containing the result of the ListTransactionEvents operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ThrottlingException The request or operation couldn't be performed because a service is throttling
     *         requests. The most common source of throttling errors is when you create resources that exceed your
     *         service limit for this resource type. Request a limit increase or delete unused resources, if possible.</li>
     *         <li>ValidationException The resource passed is invalid.</li>
     *         <li>AccessDeniedException The Amazon Web Services account doesn’t have access to this resource.</li>
     *         <li>InternalServerException The request processing has failed because of an internal error in the
     *         service.</li>
     *         <li>ServiceQuotaExceededException The service quota has been exceeded for this resource.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ManagedBlockchainQueryException Base class for all service exceptions. Unknown exceptions will be
     *         thrown as an instance of this type.</li>
     *         </ul>
     * @sample ManagedBlockchainQueryAsyncClient.ListTransactionEvents
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/managedblockchain-query-2023-05-04/ListTransactionEvents"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListTransactionEventsResponse> listTransactionEvents(
            ListTransactionEventsRequest listTransactionEventsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTransactionEventsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ManagedBlockchain Query");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTransactionEvents");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListTransactionEventsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListTransactionEventsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListTransactionEventsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListTransactionEventsRequest, ListTransactionEventsResponse>()
                            .withOperationName("ListTransactionEvents")
                            .withMarshaller(new ListTransactionEventsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listTransactionEventsRequest));
            CompletableFuture<ListTransactionEventsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists all of the transactions on a given wallet address or to a specific contract.
     * </p>
     *
     * @param listTransactionsRequest
     * @return A Java Future containing the result of the ListTransactions operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ThrottlingException The request or operation couldn't be performed because a service is throttling
     *         requests. The most common source of throttling errors is when you create resources that exceed your
     *         service limit for this resource type. Request a limit increase or delete unused resources, if possible.</li>
     *         <li>ValidationException The resource passed is invalid.</li>
     *         <li>AccessDeniedException The Amazon Web Services account doesn’t have access to this resource.</li>
     *         <li>InternalServerException The request processing has failed because of an internal error in the
     *         service.</li>
     *         <li>ServiceQuotaExceededException The service quota has been exceeded for this resource.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ManagedBlockchainQueryException Base class for all service exceptions. Unknown exceptions will be
     *         thrown as an instance of this type.</li>
     *         </ul>
     * @sample ManagedBlockchainQueryAsyncClient.ListTransactions
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/managedblockchain-query-2023-05-04/ListTransactions"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListTransactionsResponse> listTransactions(ListTransactionsRequest listTransactionsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTransactionsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ManagedBlockchain Query");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTransactions");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListTransactionsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListTransactionsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListTransactionsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListTransactionsRequest, ListTransactionsResponse>()
                            .withOperationName("ListTransactions")
                            .withMarshaller(new ListTransactionsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listTransactionsRequest));
            CompletableFuture<ListTransactionsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    @Override
    public final ManagedBlockchainQueryServiceClientConfiguration serviceClientConfiguration() {
        return this.serviceClientConfiguration;
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(ManagedBlockchainQueryException::builder)
                .protocol(AwsJsonProtocol.REST_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("AccessDeniedException")
                                .exceptionBuilderSupplier(AccessDeniedException::builder).httpStatusCode(403).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ValidationException")
                                .exceptionBuilderSupplier(ValidationException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceNotFoundException")
                                .exceptionBuilderSupplier(ResourceNotFoundException::builder).httpStatusCode(404).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException")
                                .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).httpStatusCode(402).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalServerException")
                                .exceptionBuilderSupplier(InternalServerException::builder).httpStatusCode(500).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ThrottlingException")
                                .exceptionBuilderSupplier(ThrottlingException::builder).httpStatusCode(429).build());
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    @Override
    public void close() {
        clientHandler.close();
    }
}
