/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.lexmodelsv2.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * An object containing information about a specific utterance.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class UtteranceSpecification implements SdkPojo, Serializable,
        ToCopyableBuilder<UtteranceSpecification.Builder, UtteranceSpecification> {
    private static final SdkField<String> BOT_ALIAS_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("botAliasId").getter(getter(UtteranceSpecification::botAliasId)).setter(setter(Builder::botAliasId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("botAliasId").build()).build();

    private static final SdkField<String> BOT_VERSION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("botVersion").getter(getter(UtteranceSpecification::botVersion)).setter(setter(Builder::botVersion))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("botVersion").build()).build();

    private static final SdkField<String> LOCALE_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("localeId").getter(getter(UtteranceSpecification::localeId)).setter(setter(Builder::localeId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("localeId").build()).build();

    private static final SdkField<String> SESSION_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("sessionId").getter(getter(UtteranceSpecification::sessionId)).setter(setter(Builder::sessionId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("sessionId").build()).build();

    private static final SdkField<String> CHANNEL_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("channel")
            .getter(getter(UtteranceSpecification::channel)).setter(setter(Builder::channel))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("channel").build()).build();

    private static final SdkField<String> MODE_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("mode")
            .getter(getter(UtteranceSpecification::modeAsString)).setter(setter(Builder::mode))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("mode").build()).build();

    private static final SdkField<Instant> CONVERSATION_START_TIME_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("conversationStartTime").getter(getter(UtteranceSpecification::conversationStartTime))
            .setter(setter(Builder::conversationStartTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("conversationStartTime").build())
            .build();

    private static final SdkField<Instant> CONVERSATION_END_TIME_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("conversationEndTime").getter(getter(UtteranceSpecification::conversationEndTime))
            .setter(setter(Builder::conversationEndTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("conversationEndTime").build())
            .build();

    private static final SdkField<String> UTTERANCE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("utterance").getter(getter(UtteranceSpecification::utterance)).setter(setter(Builder::utterance))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("utterance").build()).build();

    private static final SdkField<Instant> UTTERANCE_TIMESTAMP_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("utteranceTimestamp").getter(getter(UtteranceSpecification::utteranceTimestamp))
            .setter(setter(Builder::utteranceTimestamp))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("utteranceTimestamp").build())
            .build();

    private static final SdkField<Long> AUDIO_VOICE_DURATION_MILLIS_FIELD = SdkField.<Long> builder(MarshallingType.LONG)
            .memberName("audioVoiceDurationMillis").getter(getter(UtteranceSpecification::audioVoiceDurationMillis))
            .setter(setter(Builder::audioVoiceDurationMillis))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("audioVoiceDurationMillis").build())
            .build();

    private static final SdkField<Boolean> UTTERANCE_UNDERSTOOD_FIELD = SdkField.<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("utteranceUnderstood").getter(getter(UtteranceSpecification::utteranceUnderstood))
            .setter(setter(Builder::utteranceUnderstood))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("utteranceUnderstood").build())
            .build();

    private static final SdkField<String> INPUT_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("inputType").getter(getter(UtteranceSpecification::inputType)).setter(setter(Builder::inputType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("inputType").build()).build();

    private static final SdkField<String> OUTPUT_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("outputType").getter(getter(UtteranceSpecification::outputType)).setter(setter(Builder::outputType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("outputType").build()).build();

    private static final SdkField<String> ASSOCIATED_INTENT_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("associatedIntentName").getter(getter(UtteranceSpecification::associatedIntentName))
            .setter(setter(Builder::associatedIntentName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("associatedIntentName").build())
            .build();

    private static final SdkField<String> ASSOCIATED_SLOT_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("associatedSlotName").getter(getter(UtteranceSpecification::associatedSlotName))
            .setter(setter(Builder::associatedSlotName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("associatedSlotName").build())
            .build();

    private static final SdkField<String> INTENT_STATE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("intentState").getter(getter(UtteranceSpecification::intentStateAsString))
            .setter(setter(Builder::intentState))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("intentState").build()).build();

    private static final SdkField<String> DIALOG_ACTION_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("dialogActionType").getter(getter(UtteranceSpecification::dialogActionType))
            .setter(setter(Builder::dialogActionType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("dialogActionType").build()).build();

    private static final SdkField<String> BOT_RESPONSE_AUDIO_VOICE_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("botResponseAudioVoiceId").getter(getter(UtteranceSpecification::botResponseAudioVoiceId))
            .setter(setter(Builder::botResponseAudioVoiceId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("botResponseAudioVoiceId").build())
            .build();

    private static final SdkField<String> SLOTS_FILLED_IN_SESSION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("slotsFilledInSession").getter(getter(UtteranceSpecification::slotsFilledInSession))
            .setter(setter(Builder::slotsFilledInSession))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("slotsFilledInSession").build())
            .build();

    private static final SdkField<String> UTTERANCE_REQUEST_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("utteranceRequestId").getter(getter(UtteranceSpecification::utteranceRequestId))
            .setter(setter(Builder::utteranceRequestId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("utteranceRequestId").build())
            .build();

    private static final SdkField<List<UtteranceBotResponse>> BOT_RESPONSES_FIELD = SdkField
            .<List<UtteranceBotResponse>> builder(MarshallingType.LIST)
            .memberName("botResponses")
            .getter(getter(UtteranceSpecification::botResponses))
            .setter(setter(Builder::botResponses))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("botResponses").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<UtteranceBotResponse> builder(MarshallingType.SDK_POJO)
                                            .constructor(UtteranceBotResponse::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(BOT_ALIAS_ID_FIELD,
            BOT_VERSION_FIELD, LOCALE_ID_FIELD, SESSION_ID_FIELD, CHANNEL_FIELD, MODE_FIELD, CONVERSATION_START_TIME_FIELD,
            CONVERSATION_END_TIME_FIELD, UTTERANCE_FIELD, UTTERANCE_TIMESTAMP_FIELD, AUDIO_VOICE_DURATION_MILLIS_FIELD,
            UTTERANCE_UNDERSTOOD_FIELD, INPUT_TYPE_FIELD, OUTPUT_TYPE_FIELD, ASSOCIATED_INTENT_NAME_FIELD,
            ASSOCIATED_SLOT_NAME_FIELD, INTENT_STATE_FIELD, DIALOG_ACTION_TYPE_FIELD, BOT_RESPONSE_AUDIO_VOICE_ID_FIELD,
            SLOTS_FILLED_IN_SESSION_FIELD, UTTERANCE_REQUEST_ID_FIELD, BOT_RESPONSES_FIELD));

    private static final long serialVersionUID = 1L;

    private final String botAliasId;

    private final String botVersion;

    private final String localeId;

    private final String sessionId;

    private final String channel;

    private final String mode;

    private final Instant conversationStartTime;

    private final Instant conversationEndTime;

    private final String utterance;

    private final Instant utteranceTimestamp;

    private final Long audioVoiceDurationMillis;

    private final Boolean utteranceUnderstood;

    private final String inputType;

    private final String outputType;

    private final String associatedIntentName;

    private final String associatedSlotName;

    private final String intentState;

    private final String dialogActionType;

    private final String botResponseAudioVoiceId;

    private final String slotsFilledInSession;

    private final String utteranceRequestId;

    private final List<UtteranceBotResponse> botResponses;

    private UtteranceSpecification(BuilderImpl builder) {
        this.botAliasId = builder.botAliasId;
        this.botVersion = builder.botVersion;
        this.localeId = builder.localeId;
        this.sessionId = builder.sessionId;
        this.channel = builder.channel;
        this.mode = builder.mode;
        this.conversationStartTime = builder.conversationStartTime;
        this.conversationEndTime = builder.conversationEndTime;
        this.utterance = builder.utterance;
        this.utteranceTimestamp = builder.utteranceTimestamp;
        this.audioVoiceDurationMillis = builder.audioVoiceDurationMillis;
        this.utteranceUnderstood = builder.utteranceUnderstood;
        this.inputType = builder.inputType;
        this.outputType = builder.outputType;
        this.associatedIntentName = builder.associatedIntentName;
        this.associatedSlotName = builder.associatedSlotName;
        this.intentState = builder.intentState;
        this.dialogActionType = builder.dialogActionType;
        this.botResponseAudioVoiceId = builder.botResponseAudioVoiceId;
        this.slotsFilledInSession = builder.slotsFilledInSession;
        this.utteranceRequestId = builder.utteranceRequestId;
        this.botResponses = builder.botResponses;
    }

    /**
     * <p>
     * The identifier of the alias of the bot that the utterance was made to.
     * </p>
     * 
     * @return The identifier of the alias of the bot that the utterance was made to.
     */
    public final String botAliasId() {
        return botAliasId;
    }

    /**
     * <p>
     * The version of the bot that the utterance was made to.
     * </p>
     * 
     * @return The version of the bot that the utterance was made to.
     */
    public final String botVersion() {
        return botVersion;
    }

    /**
     * <p>
     * The locale of the bot that the utterance was made to.
     * </p>
     * 
     * @return The locale of the bot that the utterance was made to.
     */
    public final String localeId() {
        return localeId;
    }

    /**
     * <p>
     * The identifier of the session that the utterance was made in.
     * </p>
     * 
     * @return The identifier of the session that the utterance was made in.
     */
    public final String sessionId() {
        return sessionId;
    }

    /**
     * <p>
     * The channel that is integrated with the bot that the utterance was made to.
     * </p>
     * 
     * @return The channel that is integrated with the bot that the utterance was made to.
     */
    public final String channel() {
        return channel;
    }

    /**
     * <p>
     * The mode of the session. The possible values are as follows:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>Speech</code> – The session consisted of spoken dialogue.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>Text</code> – The session consisted of written dialogue.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>DTMF</code> – The session consisted of touch-tone keypad (Dual Tone Multi-Frequency) key presses.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>MultiMode</code> – The session consisted of multiple modes.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #mode} will return
     * {@link AnalyticsModality#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #modeAsString}.
     * </p>
     * 
     * @return The mode of the session. The possible values are as follows:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>Speech</code> – The session consisted of spoken dialogue.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>Text</code> – The session consisted of written dialogue.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>DTMF</code> – The session consisted of touch-tone keypad (Dual Tone Multi-Frequency) key presses.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>MultiMode</code> – The session consisted of multiple modes.
     *         </p>
     *         </li>
     * @see AnalyticsModality
     */
    public final AnalyticsModality mode() {
        return AnalyticsModality.fromValue(mode);
    }

    /**
     * <p>
     * The mode of the session. The possible values are as follows:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>Speech</code> – The session consisted of spoken dialogue.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>Text</code> – The session consisted of written dialogue.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>DTMF</code> – The session consisted of touch-tone keypad (Dual Tone Multi-Frequency) key presses.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>MultiMode</code> – The session consisted of multiple modes.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #mode} will return
     * {@link AnalyticsModality#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #modeAsString}.
     * </p>
     * 
     * @return The mode of the session. The possible values are as follows:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>Speech</code> – The session consisted of spoken dialogue.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>Text</code> – The session consisted of written dialogue.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>DTMF</code> – The session consisted of touch-tone keypad (Dual Tone Multi-Frequency) key presses.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>MultiMode</code> – The session consisted of multiple modes.
     *         </p>
     *         </li>
     * @see AnalyticsModality
     */
    public final String modeAsString() {
        return mode;
    }

    /**
     * <p>
     * The date and time when the conversation in which the utterance took place began. A conversation is defined as a
     * unique combination of a <code>sessionId</code> and an <code>originatingRequestId</code>.
     * </p>
     * 
     * @return The date and time when the conversation in which the utterance took place began. A conversation is
     *         defined as a unique combination of a <code>sessionId</code> and an <code>originatingRequestId</code>.
     */
    public final Instant conversationStartTime() {
        return conversationStartTime;
    }

    /**
     * <p>
     * The date and time when the conversation in which the utterance took place ended. A conversation is defined as a
     * unique combination of a <code>sessionId</code> and an <code>originatingRequestId</code>.
     * </p>
     * 
     * @return The date and time when the conversation in which the utterance took place ended. A conversation is
     *         defined as a unique combination of a <code>sessionId</code> and an <code>originatingRequestId</code>.
     */
    public final Instant conversationEndTime() {
        return conversationEndTime;
    }

    /**
     * <p>
     * The text of the utterance.
     * </p>
     * 
     * @return The text of the utterance.
     */
    public final String utterance() {
        return utterance;
    }

    /**
     * <p>
     * The date and time when the utterance took place.
     * </p>
     * 
     * @return The date and time when the utterance took place.
     */
    public final Instant utteranceTimestamp() {
        return utteranceTimestamp;
    }

    /**
     * <p>
     * The duration in milliseconds of the audio associated with the utterance.
     * </p>
     * 
     * @return The duration in milliseconds of the audio associated with the utterance.
     */
    public final Long audioVoiceDurationMillis() {
        return audioVoiceDurationMillis;
    }

    /**
     * <p>
     * Specifies whether the bot understood the utterance or not.
     * </p>
     * 
     * @return Specifies whether the bot understood the utterance or not.
     */
    public final Boolean utteranceUnderstood() {
        return utteranceUnderstood;
    }

    /**
     * <p>
     * The input type of the utterance. The possible values are as follows:
     * </p>
     * <ul>
     * <li>
     * <p>
     * PCM format: audio data must be in little-endian byte order.
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>audio/l16; rate=16000; channels=1</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>audio/x-l16; sample-rate=16000; channel-count=1</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>audio/lpcm; sample-rate=8000; sample-size-bits=16; channel-count=1; is-big-endian=false</code>
     * </p>
     * </li>
     * </ul>
     * </li>
     * <li>
     * <p>
     * Opus format
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>audio/x-cbr-opus-with-preamble;preamble-size=0;bit-rate=256000;frame-size-milliseconds=4</code>
     * </p>
     * </li>
     * </ul>
     * </li>
     * <li>
     * <p>
     * Text format
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>text/plain; charset=utf-8</code>
     * </p>
     * </li>
     * </ul>
     * </li>
     * </ul>
     * 
     * @return The input type of the utterance. The possible values are as follows:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         PCM format: audio data must be in little-endian byte order.
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>audio/l16; rate=16000; channels=1</code>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>audio/x-l16; sample-rate=16000; channel-count=1</code>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>audio/lpcm; sample-rate=8000; sample-size-bits=16; channel-count=1; is-big-endian=false</code>
     *         </p>
     *         </li>
     *         </ul>
     *         </li>
     *         <li>
     *         <p>
     *         Opus format
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>audio/x-cbr-opus-with-preamble;preamble-size=0;bit-rate=256000;frame-size-milliseconds=4</code>
     *         </p>
     *         </li>
     *         </ul>
     *         </li>
     *         <li>
     *         <p>
     *         Text format
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>text/plain; charset=utf-8</code>
     *         </p>
     *         </li>
     *         </ul>
     *         </li>
     */
    public final String inputType() {
        return inputType;
    }

    /**
     * <p>
     * The output type of the utterance. The possible values are as follows:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>audio/mpeg</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>audio/ogg</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>audio/pcm (16 KHz)</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>audio/</code> (defaults to <code>mpeg</code>)
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>text/plain; charset=utf-8</code>
     * </p>
     * </li>
     * </ul>
     * 
     * @return The output type of the utterance. The possible values are as follows:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>audio/mpeg</code>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>audio/ogg</code>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>audio/pcm (16 KHz)</code>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>audio/</code> (defaults to <code>mpeg</code>)
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>text/plain; charset=utf-8</code>
     *         </p>
     *         </li>
     */
    public final String outputType() {
        return outputType;
    }

    /**
     * <p>
     * The name of the intent that the utterance is associated to.
     * </p>
     * 
     * @return The name of the intent that the utterance is associated to.
     */
    public final String associatedIntentName() {
        return associatedIntentName;
    }

    /**
     * <p>
     * The name of the slot that the utterance is associated to.
     * </p>
     * 
     * @return The name of the slot that the utterance is associated to.
     */
    public final String associatedSlotName() {
        return associatedSlotName;
    }

    /**
     * <p>
     * The state of the intent that the utterance is associated to.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #intentState} will
     * return {@link IntentState#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #intentStateAsString}.
     * </p>
     * 
     * @return The state of the intent that the utterance is associated to.
     * @see IntentState
     */
    public final IntentState intentState() {
        return IntentState.fromValue(intentState);
    }

    /**
     * <p>
     * The state of the intent that the utterance is associated to.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #intentState} will
     * return {@link IntentState#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #intentStateAsString}.
     * </p>
     * 
     * @return The state of the intent that the utterance is associated to.
     * @see IntentState
     */
    public final String intentStateAsString() {
        return intentState;
    }

    /**
     * <p>
     * The type of dialog action that the utterance is associated to. See the <code>type</code> field in <a
     * href="https://docs.aws.amazon.com/lexv2/latest/APIReference/API_runtime_DialogAction.html">DialogAction</a> for
     * more information.
     * </p>
     * 
     * @return The type of dialog action that the utterance is associated to. See the <code>type</code> field in <a
     *         href=
     *         "https://docs.aws.amazon.com/lexv2/latest/APIReference/API_runtime_DialogAction.html">DialogAction</a>
     *         for more information.
     */
    public final String dialogActionType() {
        return dialogActionType;
    }

    /**
     * <p>
     * The identifier for the audio of the bot response.
     * </p>
     * 
     * @return The identifier for the audio of the bot response.
     */
    public final String botResponseAudioVoiceId() {
        return botResponseAudioVoiceId;
    }

    /**
     * <p>
     * The slots that have been filled in the session by the time of the utterance.
     * </p>
     * 
     * @return The slots that have been filled in the session by the time of the utterance.
     */
    public final String slotsFilledInSession() {
        return slotsFilledInSession;
    }

    /**
     * <p>
     * The identifier of the request associated with the utterance.
     * </p>
     * 
     * @return The identifier of the request associated with the utterance.
     */
    public final String utteranceRequestId() {
        return utteranceRequestId;
    }

    /**
     * For responses, this returns true if the service returned a value for the BotResponses property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasBotResponses() {
        return botResponses != null && !(botResponses instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A list of objects containing information about the bot response to the utterance.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasBotResponses} method.
     * </p>
     * 
     * @return A list of objects containing information about the bot response to the utterance.
     */
    public final List<UtteranceBotResponse> botResponses() {
        return botResponses;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(botAliasId());
        hashCode = 31 * hashCode + Objects.hashCode(botVersion());
        hashCode = 31 * hashCode + Objects.hashCode(localeId());
        hashCode = 31 * hashCode + Objects.hashCode(sessionId());
        hashCode = 31 * hashCode + Objects.hashCode(channel());
        hashCode = 31 * hashCode + Objects.hashCode(modeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(conversationStartTime());
        hashCode = 31 * hashCode + Objects.hashCode(conversationEndTime());
        hashCode = 31 * hashCode + Objects.hashCode(utterance());
        hashCode = 31 * hashCode + Objects.hashCode(utteranceTimestamp());
        hashCode = 31 * hashCode + Objects.hashCode(audioVoiceDurationMillis());
        hashCode = 31 * hashCode + Objects.hashCode(utteranceUnderstood());
        hashCode = 31 * hashCode + Objects.hashCode(inputType());
        hashCode = 31 * hashCode + Objects.hashCode(outputType());
        hashCode = 31 * hashCode + Objects.hashCode(associatedIntentName());
        hashCode = 31 * hashCode + Objects.hashCode(associatedSlotName());
        hashCode = 31 * hashCode + Objects.hashCode(intentStateAsString());
        hashCode = 31 * hashCode + Objects.hashCode(dialogActionType());
        hashCode = 31 * hashCode + Objects.hashCode(botResponseAudioVoiceId());
        hashCode = 31 * hashCode + Objects.hashCode(slotsFilledInSession());
        hashCode = 31 * hashCode + Objects.hashCode(utteranceRequestId());
        hashCode = 31 * hashCode + Objects.hashCode(hasBotResponses() ? botResponses() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof UtteranceSpecification)) {
            return false;
        }
        UtteranceSpecification other = (UtteranceSpecification) obj;
        return Objects.equals(botAliasId(), other.botAliasId()) && Objects.equals(botVersion(), other.botVersion())
                && Objects.equals(localeId(), other.localeId()) && Objects.equals(sessionId(), other.sessionId())
                && Objects.equals(channel(), other.channel()) && Objects.equals(modeAsString(), other.modeAsString())
                && Objects.equals(conversationStartTime(), other.conversationStartTime())
                && Objects.equals(conversationEndTime(), other.conversationEndTime())
                && Objects.equals(utterance(), other.utterance())
                && Objects.equals(utteranceTimestamp(), other.utteranceTimestamp())
                && Objects.equals(audioVoiceDurationMillis(), other.audioVoiceDurationMillis())
                && Objects.equals(utteranceUnderstood(), other.utteranceUnderstood())
                && Objects.equals(inputType(), other.inputType()) && Objects.equals(outputType(), other.outputType())
                && Objects.equals(associatedIntentName(), other.associatedIntentName())
                && Objects.equals(associatedSlotName(), other.associatedSlotName())
                && Objects.equals(intentStateAsString(), other.intentStateAsString())
                && Objects.equals(dialogActionType(), other.dialogActionType())
                && Objects.equals(botResponseAudioVoiceId(), other.botResponseAudioVoiceId())
                && Objects.equals(slotsFilledInSession(), other.slotsFilledInSession())
                && Objects.equals(utteranceRequestId(), other.utteranceRequestId())
                && hasBotResponses() == other.hasBotResponses() && Objects.equals(botResponses(), other.botResponses());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("UtteranceSpecification").add("BotAliasId", botAliasId()).add("BotVersion", botVersion())
                .add("LocaleId", localeId()).add("SessionId", sessionId()).add("Channel", channel()).add("Mode", modeAsString())
                .add("ConversationStartTime", conversationStartTime()).add("ConversationEndTime", conversationEndTime())
                .add("Utterance", utterance()).add("UtteranceTimestamp", utteranceTimestamp())
                .add("AudioVoiceDurationMillis", audioVoiceDurationMillis()).add("UtteranceUnderstood", utteranceUnderstood())
                .add("InputType", inputType()).add("OutputType", outputType())
                .add("AssociatedIntentName", associatedIntentName()).add("AssociatedSlotName", associatedSlotName())
                .add("IntentState", intentStateAsString()).add("DialogActionType", dialogActionType())
                .add("BotResponseAudioVoiceId", botResponseAudioVoiceId()).add("SlotsFilledInSession", slotsFilledInSession())
                .add("UtteranceRequestId", utteranceRequestId()).add("BotResponses", hasBotResponses() ? botResponses() : null)
                .build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "botAliasId":
            return Optional.ofNullable(clazz.cast(botAliasId()));
        case "botVersion":
            return Optional.ofNullable(clazz.cast(botVersion()));
        case "localeId":
            return Optional.ofNullable(clazz.cast(localeId()));
        case "sessionId":
            return Optional.ofNullable(clazz.cast(sessionId()));
        case "channel":
            return Optional.ofNullable(clazz.cast(channel()));
        case "mode":
            return Optional.ofNullable(clazz.cast(modeAsString()));
        case "conversationStartTime":
            return Optional.ofNullable(clazz.cast(conversationStartTime()));
        case "conversationEndTime":
            return Optional.ofNullable(clazz.cast(conversationEndTime()));
        case "utterance":
            return Optional.ofNullable(clazz.cast(utterance()));
        case "utteranceTimestamp":
            return Optional.ofNullable(clazz.cast(utteranceTimestamp()));
        case "audioVoiceDurationMillis":
            return Optional.ofNullable(clazz.cast(audioVoiceDurationMillis()));
        case "utteranceUnderstood":
            return Optional.ofNullable(clazz.cast(utteranceUnderstood()));
        case "inputType":
            return Optional.ofNullable(clazz.cast(inputType()));
        case "outputType":
            return Optional.ofNullable(clazz.cast(outputType()));
        case "associatedIntentName":
            return Optional.ofNullable(clazz.cast(associatedIntentName()));
        case "associatedSlotName":
            return Optional.ofNullable(clazz.cast(associatedSlotName()));
        case "intentState":
            return Optional.ofNullable(clazz.cast(intentStateAsString()));
        case "dialogActionType":
            return Optional.ofNullable(clazz.cast(dialogActionType()));
        case "botResponseAudioVoiceId":
            return Optional.ofNullable(clazz.cast(botResponseAudioVoiceId()));
        case "slotsFilledInSession":
            return Optional.ofNullable(clazz.cast(slotsFilledInSession()));
        case "utteranceRequestId":
            return Optional.ofNullable(clazz.cast(utteranceRequestId()));
        case "botResponses":
            return Optional.ofNullable(clazz.cast(botResponses()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<UtteranceSpecification, T> g) {
        return obj -> g.apply((UtteranceSpecification) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, UtteranceSpecification> {
        /**
         * <p>
         * The identifier of the alias of the bot that the utterance was made to.
         * </p>
         * 
         * @param botAliasId
         *        The identifier of the alias of the bot that the utterance was made to.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder botAliasId(String botAliasId);

        /**
         * <p>
         * The version of the bot that the utterance was made to.
         * </p>
         * 
         * @param botVersion
         *        The version of the bot that the utterance was made to.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder botVersion(String botVersion);

        /**
         * <p>
         * The locale of the bot that the utterance was made to.
         * </p>
         * 
         * @param localeId
         *        The locale of the bot that the utterance was made to.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder localeId(String localeId);

        /**
         * <p>
         * The identifier of the session that the utterance was made in.
         * </p>
         * 
         * @param sessionId
         *        The identifier of the session that the utterance was made in.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder sessionId(String sessionId);

        /**
         * <p>
         * The channel that is integrated with the bot that the utterance was made to.
         * </p>
         * 
         * @param channel
         *        The channel that is integrated with the bot that the utterance was made to.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder channel(String channel);

        /**
         * <p>
         * The mode of the session. The possible values are as follows:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>Speech</code> – The session consisted of spoken dialogue.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>Text</code> – The session consisted of written dialogue.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>DTMF</code> – The session consisted of touch-tone keypad (Dual Tone Multi-Frequency) key presses.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>MultiMode</code> – The session consisted of multiple modes.
         * </p>
         * </li>
         * </ul>
         * 
         * @param mode
         *        The mode of the session. The possible values are as follows:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>Speech</code> – The session consisted of spoken dialogue.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>Text</code> – The session consisted of written dialogue.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>DTMF</code> – The session consisted of touch-tone keypad (Dual Tone Multi-Frequency) key
         *        presses.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>MultiMode</code> – The session consisted of multiple modes.
         *        </p>
         *        </li>
         * @see AnalyticsModality
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see AnalyticsModality
         */
        Builder mode(String mode);

        /**
         * <p>
         * The mode of the session. The possible values are as follows:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>Speech</code> – The session consisted of spoken dialogue.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>Text</code> – The session consisted of written dialogue.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>DTMF</code> – The session consisted of touch-tone keypad (Dual Tone Multi-Frequency) key presses.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>MultiMode</code> – The session consisted of multiple modes.
         * </p>
         * </li>
         * </ul>
         * 
         * @param mode
         *        The mode of the session. The possible values are as follows:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>Speech</code> – The session consisted of spoken dialogue.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>Text</code> – The session consisted of written dialogue.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>DTMF</code> – The session consisted of touch-tone keypad (Dual Tone Multi-Frequency) key
         *        presses.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>MultiMode</code> – The session consisted of multiple modes.
         *        </p>
         *        </li>
         * @see AnalyticsModality
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see AnalyticsModality
         */
        Builder mode(AnalyticsModality mode);

        /**
         * <p>
         * The date and time when the conversation in which the utterance took place began. A conversation is defined as
         * a unique combination of a <code>sessionId</code> and an <code>originatingRequestId</code>.
         * </p>
         * 
         * @param conversationStartTime
         *        The date and time when the conversation in which the utterance took place began. A conversation is
         *        defined as a unique combination of a <code>sessionId</code> and an <code>originatingRequestId</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder conversationStartTime(Instant conversationStartTime);

        /**
         * <p>
         * The date and time when the conversation in which the utterance took place ended. A conversation is defined as
         * a unique combination of a <code>sessionId</code> and an <code>originatingRequestId</code>.
         * </p>
         * 
         * @param conversationEndTime
         *        The date and time when the conversation in which the utterance took place ended. A conversation is
         *        defined as a unique combination of a <code>sessionId</code> and an <code>originatingRequestId</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder conversationEndTime(Instant conversationEndTime);

        /**
         * <p>
         * The text of the utterance.
         * </p>
         * 
         * @param utterance
         *        The text of the utterance.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder utterance(String utterance);

        /**
         * <p>
         * The date and time when the utterance took place.
         * </p>
         * 
         * @param utteranceTimestamp
         *        The date and time when the utterance took place.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder utteranceTimestamp(Instant utteranceTimestamp);

        /**
         * <p>
         * The duration in milliseconds of the audio associated with the utterance.
         * </p>
         * 
         * @param audioVoiceDurationMillis
         *        The duration in milliseconds of the audio associated with the utterance.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder audioVoiceDurationMillis(Long audioVoiceDurationMillis);

        /**
         * <p>
         * Specifies whether the bot understood the utterance or not.
         * </p>
         * 
         * @param utteranceUnderstood
         *        Specifies whether the bot understood the utterance or not.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder utteranceUnderstood(Boolean utteranceUnderstood);

        /**
         * <p>
         * The input type of the utterance. The possible values are as follows:
         * </p>
         * <ul>
         * <li>
         * <p>
         * PCM format: audio data must be in little-endian byte order.
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>audio/l16; rate=16000; channels=1</code>
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>audio/x-l16; sample-rate=16000; channel-count=1</code>
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>audio/lpcm; sample-rate=8000; sample-size-bits=16; channel-count=1; is-big-endian=false</code>
         * </p>
         * </li>
         * </ul>
         * </li>
         * <li>
         * <p>
         * Opus format
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>audio/x-cbr-opus-with-preamble;preamble-size=0;bit-rate=256000;frame-size-milliseconds=4</code>
         * </p>
         * </li>
         * </ul>
         * </li>
         * <li>
         * <p>
         * Text format
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>text/plain; charset=utf-8</code>
         * </p>
         * </li>
         * </ul>
         * </li>
         * </ul>
         * 
         * @param inputType
         *        The input type of the utterance. The possible values are as follows:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        PCM format: audio data must be in little-endian byte order.
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>audio/l16; rate=16000; channels=1</code>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>audio/x-l16; sample-rate=16000; channel-count=1</code>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>audio/lpcm; sample-rate=8000; sample-size-bits=16; channel-count=1; is-big-endian=false</code>
         *        </p>
         *        </li>
         *        </ul>
         *        </li>
         *        <li>
         *        <p>
         *        Opus format
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>audio/x-cbr-opus-with-preamble;preamble-size=0;bit-rate=256000;frame-size-milliseconds=4</code>
         *        </p>
         *        </li>
         *        </ul>
         *        </li>
         *        <li>
         *        <p>
         *        Text format
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>text/plain; charset=utf-8</code>
         *        </p>
         *        </li>
         *        </ul>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder inputType(String inputType);

        /**
         * <p>
         * The output type of the utterance. The possible values are as follows:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>audio/mpeg</code>
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>audio/ogg</code>
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>audio/pcm (16 KHz)</code>
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>audio/</code> (defaults to <code>mpeg</code>)
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>text/plain; charset=utf-8</code>
         * </p>
         * </li>
         * </ul>
         * 
         * @param outputType
         *        The output type of the utterance. The possible values are as follows:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>audio/mpeg</code>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>audio/ogg</code>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>audio/pcm (16 KHz)</code>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>audio/</code> (defaults to <code>mpeg</code>)
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>text/plain; charset=utf-8</code>
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder outputType(String outputType);

        /**
         * <p>
         * The name of the intent that the utterance is associated to.
         * </p>
         * 
         * @param associatedIntentName
         *        The name of the intent that the utterance is associated to.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder associatedIntentName(String associatedIntentName);

        /**
         * <p>
         * The name of the slot that the utterance is associated to.
         * </p>
         * 
         * @param associatedSlotName
         *        The name of the slot that the utterance is associated to.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder associatedSlotName(String associatedSlotName);

        /**
         * <p>
         * The state of the intent that the utterance is associated to.
         * </p>
         * 
         * @param intentState
         *        The state of the intent that the utterance is associated to.
         * @see IntentState
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see IntentState
         */
        Builder intentState(String intentState);

        /**
         * <p>
         * The state of the intent that the utterance is associated to.
         * </p>
         * 
         * @param intentState
         *        The state of the intent that the utterance is associated to.
         * @see IntentState
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see IntentState
         */
        Builder intentState(IntentState intentState);

        /**
         * <p>
         * The type of dialog action that the utterance is associated to. See the <code>type</code> field in <a
         * href="https://docs.aws.amazon.com/lexv2/latest/APIReference/API_runtime_DialogAction.html">DialogAction</a>
         * for more information.
         * </p>
         * 
         * @param dialogActionType
         *        The type of dialog action that the utterance is associated to. See the <code>type</code> field in <a
         *        href
         *        ="https://docs.aws.amazon.com/lexv2/latest/APIReference/API_runtime_DialogAction.html">DialogAction
         *        </a> for more information.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder dialogActionType(String dialogActionType);

        /**
         * <p>
         * The identifier for the audio of the bot response.
         * </p>
         * 
         * @param botResponseAudioVoiceId
         *        The identifier for the audio of the bot response.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder botResponseAudioVoiceId(String botResponseAudioVoiceId);

        /**
         * <p>
         * The slots that have been filled in the session by the time of the utterance.
         * </p>
         * 
         * @param slotsFilledInSession
         *        The slots that have been filled in the session by the time of the utterance.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder slotsFilledInSession(String slotsFilledInSession);

        /**
         * <p>
         * The identifier of the request associated with the utterance.
         * </p>
         * 
         * @param utteranceRequestId
         *        The identifier of the request associated with the utterance.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder utteranceRequestId(String utteranceRequestId);

        /**
         * <p>
         * A list of objects containing information about the bot response to the utterance.
         * </p>
         * 
         * @param botResponses
         *        A list of objects containing information about the bot response to the utterance.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder botResponses(Collection<UtteranceBotResponse> botResponses);

        /**
         * <p>
         * A list of objects containing information about the bot response to the utterance.
         * </p>
         * 
         * @param botResponses
         *        A list of objects containing information about the bot response to the utterance.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder botResponses(UtteranceBotResponse... botResponses);

        /**
         * <p>
         * A list of objects containing information about the bot response to the utterance.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.lexmodelsv2.model.UtteranceBotResponse.Builder} avoiding the need to
         * create one manually via
         * {@link software.amazon.awssdk.services.lexmodelsv2.model.UtteranceBotResponse#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.lexmodelsv2.model.UtteranceBotResponse.Builder#build()} is called
         * immediately and its result is passed to {@link #botResponses(List<UtteranceBotResponse>)}.
         * 
         * @param botResponses
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.lexmodelsv2.model.UtteranceBotResponse.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #botResponses(java.util.Collection<UtteranceBotResponse>)
         */
        Builder botResponses(Consumer<UtteranceBotResponse.Builder>... botResponses);
    }

    static final class BuilderImpl implements Builder {
        private String botAliasId;

        private String botVersion;

        private String localeId;

        private String sessionId;

        private String channel;

        private String mode;

        private Instant conversationStartTime;

        private Instant conversationEndTime;

        private String utterance;

        private Instant utteranceTimestamp;

        private Long audioVoiceDurationMillis;

        private Boolean utteranceUnderstood;

        private String inputType;

        private String outputType;

        private String associatedIntentName;

        private String associatedSlotName;

        private String intentState;

        private String dialogActionType;

        private String botResponseAudioVoiceId;

        private String slotsFilledInSession;

        private String utteranceRequestId;

        private List<UtteranceBotResponse> botResponses = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(UtteranceSpecification model) {
            botAliasId(model.botAliasId);
            botVersion(model.botVersion);
            localeId(model.localeId);
            sessionId(model.sessionId);
            channel(model.channel);
            mode(model.mode);
            conversationStartTime(model.conversationStartTime);
            conversationEndTime(model.conversationEndTime);
            utterance(model.utterance);
            utteranceTimestamp(model.utteranceTimestamp);
            audioVoiceDurationMillis(model.audioVoiceDurationMillis);
            utteranceUnderstood(model.utteranceUnderstood);
            inputType(model.inputType);
            outputType(model.outputType);
            associatedIntentName(model.associatedIntentName);
            associatedSlotName(model.associatedSlotName);
            intentState(model.intentState);
            dialogActionType(model.dialogActionType);
            botResponseAudioVoiceId(model.botResponseAudioVoiceId);
            slotsFilledInSession(model.slotsFilledInSession);
            utteranceRequestId(model.utteranceRequestId);
            botResponses(model.botResponses);
        }

        public final String getBotAliasId() {
            return botAliasId;
        }

        public final void setBotAliasId(String botAliasId) {
            this.botAliasId = botAliasId;
        }

        @Override
        public final Builder botAliasId(String botAliasId) {
            this.botAliasId = botAliasId;
            return this;
        }

        public final String getBotVersion() {
            return botVersion;
        }

        public final void setBotVersion(String botVersion) {
            this.botVersion = botVersion;
        }

        @Override
        public final Builder botVersion(String botVersion) {
            this.botVersion = botVersion;
            return this;
        }

        public final String getLocaleId() {
            return localeId;
        }

        public final void setLocaleId(String localeId) {
            this.localeId = localeId;
        }

        @Override
        public final Builder localeId(String localeId) {
            this.localeId = localeId;
            return this;
        }

        public final String getSessionId() {
            return sessionId;
        }

        public final void setSessionId(String sessionId) {
            this.sessionId = sessionId;
        }

        @Override
        public final Builder sessionId(String sessionId) {
            this.sessionId = sessionId;
            return this;
        }

        public final String getChannel() {
            return channel;
        }

        public final void setChannel(String channel) {
            this.channel = channel;
        }

        @Override
        public final Builder channel(String channel) {
            this.channel = channel;
            return this;
        }

        public final String getMode() {
            return mode;
        }

        public final void setMode(String mode) {
            this.mode = mode;
        }

        @Override
        public final Builder mode(String mode) {
            this.mode = mode;
            return this;
        }

        @Override
        public final Builder mode(AnalyticsModality mode) {
            this.mode(mode == null ? null : mode.toString());
            return this;
        }

        public final Instant getConversationStartTime() {
            return conversationStartTime;
        }

        public final void setConversationStartTime(Instant conversationStartTime) {
            this.conversationStartTime = conversationStartTime;
        }

        @Override
        public final Builder conversationStartTime(Instant conversationStartTime) {
            this.conversationStartTime = conversationStartTime;
            return this;
        }

        public final Instant getConversationEndTime() {
            return conversationEndTime;
        }

        public final void setConversationEndTime(Instant conversationEndTime) {
            this.conversationEndTime = conversationEndTime;
        }

        @Override
        public final Builder conversationEndTime(Instant conversationEndTime) {
            this.conversationEndTime = conversationEndTime;
            return this;
        }

        public final String getUtterance() {
            return utterance;
        }

        public final void setUtterance(String utterance) {
            this.utterance = utterance;
        }

        @Override
        public final Builder utterance(String utterance) {
            this.utterance = utterance;
            return this;
        }

        public final Instant getUtteranceTimestamp() {
            return utteranceTimestamp;
        }

        public final void setUtteranceTimestamp(Instant utteranceTimestamp) {
            this.utteranceTimestamp = utteranceTimestamp;
        }

        @Override
        public final Builder utteranceTimestamp(Instant utteranceTimestamp) {
            this.utteranceTimestamp = utteranceTimestamp;
            return this;
        }

        public final Long getAudioVoiceDurationMillis() {
            return audioVoiceDurationMillis;
        }

        public final void setAudioVoiceDurationMillis(Long audioVoiceDurationMillis) {
            this.audioVoiceDurationMillis = audioVoiceDurationMillis;
        }

        @Override
        public final Builder audioVoiceDurationMillis(Long audioVoiceDurationMillis) {
            this.audioVoiceDurationMillis = audioVoiceDurationMillis;
            return this;
        }

        public final Boolean getUtteranceUnderstood() {
            return utteranceUnderstood;
        }

        public final void setUtteranceUnderstood(Boolean utteranceUnderstood) {
            this.utteranceUnderstood = utteranceUnderstood;
        }

        @Override
        public final Builder utteranceUnderstood(Boolean utteranceUnderstood) {
            this.utteranceUnderstood = utteranceUnderstood;
            return this;
        }

        public final String getInputType() {
            return inputType;
        }

        public final void setInputType(String inputType) {
            this.inputType = inputType;
        }

        @Override
        public final Builder inputType(String inputType) {
            this.inputType = inputType;
            return this;
        }

        public final String getOutputType() {
            return outputType;
        }

        public final void setOutputType(String outputType) {
            this.outputType = outputType;
        }

        @Override
        public final Builder outputType(String outputType) {
            this.outputType = outputType;
            return this;
        }

        public final String getAssociatedIntentName() {
            return associatedIntentName;
        }

        public final void setAssociatedIntentName(String associatedIntentName) {
            this.associatedIntentName = associatedIntentName;
        }

        @Override
        public final Builder associatedIntentName(String associatedIntentName) {
            this.associatedIntentName = associatedIntentName;
            return this;
        }

        public final String getAssociatedSlotName() {
            return associatedSlotName;
        }

        public final void setAssociatedSlotName(String associatedSlotName) {
            this.associatedSlotName = associatedSlotName;
        }

        @Override
        public final Builder associatedSlotName(String associatedSlotName) {
            this.associatedSlotName = associatedSlotName;
            return this;
        }

        public final String getIntentState() {
            return intentState;
        }

        public final void setIntentState(String intentState) {
            this.intentState = intentState;
        }

        @Override
        public final Builder intentState(String intentState) {
            this.intentState = intentState;
            return this;
        }

        @Override
        public final Builder intentState(IntentState intentState) {
            this.intentState(intentState == null ? null : intentState.toString());
            return this;
        }

        public final String getDialogActionType() {
            return dialogActionType;
        }

        public final void setDialogActionType(String dialogActionType) {
            this.dialogActionType = dialogActionType;
        }

        @Override
        public final Builder dialogActionType(String dialogActionType) {
            this.dialogActionType = dialogActionType;
            return this;
        }

        public final String getBotResponseAudioVoiceId() {
            return botResponseAudioVoiceId;
        }

        public final void setBotResponseAudioVoiceId(String botResponseAudioVoiceId) {
            this.botResponseAudioVoiceId = botResponseAudioVoiceId;
        }

        @Override
        public final Builder botResponseAudioVoiceId(String botResponseAudioVoiceId) {
            this.botResponseAudioVoiceId = botResponseAudioVoiceId;
            return this;
        }

        public final String getSlotsFilledInSession() {
            return slotsFilledInSession;
        }

        public final void setSlotsFilledInSession(String slotsFilledInSession) {
            this.slotsFilledInSession = slotsFilledInSession;
        }

        @Override
        public final Builder slotsFilledInSession(String slotsFilledInSession) {
            this.slotsFilledInSession = slotsFilledInSession;
            return this;
        }

        public final String getUtteranceRequestId() {
            return utteranceRequestId;
        }

        public final void setUtteranceRequestId(String utteranceRequestId) {
            this.utteranceRequestId = utteranceRequestId;
        }

        @Override
        public final Builder utteranceRequestId(String utteranceRequestId) {
            this.utteranceRequestId = utteranceRequestId;
            return this;
        }

        public final List<UtteranceBotResponse.Builder> getBotResponses() {
            List<UtteranceBotResponse.Builder> result = UtteranceBotResponsesCopier.copyToBuilder(this.botResponses);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setBotResponses(Collection<UtteranceBotResponse.BuilderImpl> botResponses) {
            this.botResponses = UtteranceBotResponsesCopier.copyFromBuilder(botResponses);
        }

        @Override
        public final Builder botResponses(Collection<UtteranceBotResponse> botResponses) {
            this.botResponses = UtteranceBotResponsesCopier.copy(botResponses);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder botResponses(UtteranceBotResponse... botResponses) {
            botResponses(Arrays.asList(botResponses));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder botResponses(Consumer<UtteranceBotResponse.Builder>... botResponses) {
            botResponses(Stream.of(botResponses).map(c -> UtteranceBotResponse.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        @Override
        public UtteranceSpecification build() {
            return new UtteranceSpecification(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
