/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.kendraranking;

import java.util.Collections;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.client.handler.AwsAsyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.AsyncClientHandler;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.kendraranking.model.AccessDeniedException;
import software.amazon.awssdk.services.kendraranking.model.ConflictException;
import software.amazon.awssdk.services.kendraranking.model.CreateRescoreExecutionPlanRequest;
import software.amazon.awssdk.services.kendraranking.model.CreateRescoreExecutionPlanResponse;
import software.amazon.awssdk.services.kendraranking.model.DeleteRescoreExecutionPlanRequest;
import software.amazon.awssdk.services.kendraranking.model.DeleteRescoreExecutionPlanResponse;
import software.amazon.awssdk.services.kendraranking.model.DescribeRescoreExecutionPlanRequest;
import software.amazon.awssdk.services.kendraranking.model.DescribeRescoreExecutionPlanResponse;
import software.amazon.awssdk.services.kendraranking.model.InternalServerException;
import software.amazon.awssdk.services.kendraranking.model.KendraRankingException;
import software.amazon.awssdk.services.kendraranking.model.ListRescoreExecutionPlansRequest;
import software.amazon.awssdk.services.kendraranking.model.ListRescoreExecutionPlansResponse;
import software.amazon.awssdk.services.kendraranking.model.ListTagsForResourceRequest;
import software.amazon.awssdk.services.kendraranking.model.ListTagsForResourceResponse;
import software.amazon.awssdk.services.kendraranking.model.RescoreRequest;
import software.amazon.awssdk.services.kendraranking.model.RescoreResponse;
import software.amazon.awssdk.services.kendraranking.model.ResourceNotFoundException;
import software.amazon.awssdk.services.kendraranking.model.ResourceUnavailableException;
import software.amazon.awssdk.services.kendraranking.model.ServiceQuotaExceededException;
import software.amazon.awssdk.services.kendraranking.model.TagResourceRequest;
import software.amazon.awssdk.services.kendraranking.model.TagResourceResponse;
import software.amazon.awssdk.services.kendraranking.model.ThrottlingException;
import software.amazon.awssdk.services.kendraranking.model.UntagResourceRequest;
import software.amazon.awssdk.services.kendraranking.model.UntagResourceResponse;
import software.amazon.awssdk.services.kendraranking.model.UpdateRescoreExecutionPlanRequest;
import software.amazon.awssdk.services.kendraranking.model.UpdateRescoreExecutionPlanResponse;
import software.amazon.awssdk.services.kendraranking.model.ValidationException;
import software.amazon.awssdk.services.kendraranking.transform.CreateRescoreExecutionPlanRequestMarshaller;
import software.amazon.awssdk.services.kendraranking.transform.DeleteRescoreExecutionPlanRequestMarshaller;
import software.amazon.awssdk.services.kendraranking.transform.DescribeRescoreExecutionPlanRequestMarshaller;
import software.amazon.awssdk.services.kendraranking.transform.ListRescoreExecutionPlansRequestMarshaller;
import software.amazon.awssdk.services.kendraranking.transform.ListTagsForResourceRequestMarshaller;
import software.amazon.awssdk.services.kendraranking.transform.RescoreRequestMarshaller;
import software.amazon.awssdk.services.kendraranking.transform.TagResourceRequestMarshaller;
import software.amazon.awssdk.services.kendraranking.transform.UntagResourceRequestMarshaller;
import software.amazon.awssdk.services.kendraranking.transform.UpdateRescoreExecutionPlanRequestMarshaller;
import software.amazon.awssdk.utils.CompletableFutureUtils;

/**
 * Internal implementation of {@link KendraRankingAsyncClient}.
 *
 * @see KendraRankingAsyncClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultKendraRankingAsyncClient implements KendraRankingAsyncClient {
    private static final Logger log = LoggerFactory.getLogger(DefaultKendraRankingAsyncClient.class);

    private final AsyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    private final KendraRankingServiceClientConfiguration serviceClientConfiguration;

    protected DefaultKendraRankingAsyncClient(KendraRankingServiceClientConfiguration serviceClientConfiguration,
            SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsAsyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.serviceClientConfiguration = serviceClientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    /**
     * <p>
     * Creates a rescore execution plan. A rescore execution plan is an Amazon Kendra Intelligent Ranking resource used
     * for provisioning the <code>Rescore</code> API. You set the number of capacity units that you require for Amazon
     * Kendra Intelligent Ranking to rescore or re-rank a search service's results.
     * </p>
     * <p>
     * For an example of using the <code>CreateRescoreExecutionPlan</code> API, including using the Python and Java
     * SDKs, see <a href="https://docs.aws.amazon.com/kendra/latest/dg/search-service-rerank.html">Semantically ranking
     * a search service's results</a>.
     * </p>
     *
     * @param createRescoreExecutionPlanRequest
     * @return A Java Future containing the result of the CreateRescoreExecutionPlan operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException You don’t have sufficient access to perform this action. Please ensure you have
     *         the required permission policies and user accounts and try again.</li>
     *         <li>ConflictException A conflict occurred with the request. Please fix any inconsistencies with your
     *         resources and try again.</li>
     *         <li>ServiceQuotaExceededException You have exceeded the set limits for your Amazon Kendra Intelligent
     *         Ranking service. Please see <a href="https://docs.aws.amazon.com/kendra/latest/dg/quotas.html">Quotas</a>
     *         for more information, or contact <a href="http://aws.amazon.com/contact-us/">Support</a> to inquire about
     *         an increase of limits.</li>
     *         <li>ThrottlingException The request was denied due to request throttling. Please reduce the number of
     *         requests and try again.</li>
     *         <li>ValidationException The input fails to satisfy the constraints set by the Amazon Kendra Intelligent
     *         Ranking service. Please provide the correct input and try again.</li>
     *         <li>InternalServerException An issue occurred with the internal server used for your Amazon Kendra
     *         Intelligent Ranking service. Please wait a few minutes and try again, or contact <a
     *         href="http://aws.amazon.com/contact-us/">Support</a> for help.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>KendraRankingException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample KendraRankingAsyncClient.CreateRescoreExecutionPlan
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/kendra-ranking-2022-10-19/CreateRescoreExecutionPlan"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateRescoreExecutionPlanResponse> createRescoreExecutionPlan(
            CreateRescoreExecutionPlanRequest createRescoreExecutionPlanRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createRescoreExecutionPlanRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Kendra Ranking");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateRescoreExecutionPlan");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateRescoreExecutionPlanResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateRescoreExecutionPlanResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateRescoreExecutionPlanResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateRescoreExecutionPlanRequest, CreateRescoreExecutionPlanResponse>()
                            .withOperationName("CreateRescoreExecutionPlan")
                            .withMarshaller(new CreateRescoreExecutionPlanRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(createRescoreExecutionPlanRequest));
            CompletableFuture<CreateRescoreExecutionPlanResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes a rescore execution plan. A rescore execution plan is an Amazon Kendra Intelligent Ranking resource used
     * for provisioning the <code>Rescore</code> API.
     * </p>
     *
     * @param deleteRescoreExecutionPlanRequest
     * @return A Java Future containing the result of the DeleteRescoreExecutionPlan operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException You don’t have sufficient access to perform this action. Please ensure you have
     *         the required permission policies and user accounts and try again.</li>
     *         <li>ValidationException The input fails to satisfy the constraints set by the Amazon Kendra Intelligent
     *         Ranking service. Please provide the correct input and try again.</li>
     *         <li>ConflictException A conflict occurred with the request. Please fix any inconsistencies with your
     *         resources and try again.</li>
     *         <li>ResourceNotFoundException The resource you want to use doesn't exist. Please check you have provided
     *         the correct resource and try again.</li>
     *         <li>ThrottlingException The request was denied due to request throttling. Please reduce the number of
     *         requests and try again.</li>
     *         <li>InternalServerException An issue occurred with the internal server used for your Amazon Kendra
     *         Intelligent Ranking service. Please wait a few minutes and try again, or contact <a
     *         href="http://aws.amazon.com/contact-us/">Support</a> for help.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>KendraRankingException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample KendraRankingAsyncClient.DeleteRescoreExecutionPlan
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/kendra-ranking-2022-10-19/DeleteRescoreExecutionPlan"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteRescoreExecutionPlanResponse> deleteRescoreExecutionPlan(
            DeleteRescoreExecutionPlanRequest deleteRescoreExecutionPlanRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteRescoreExecutionPlanRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Kendra Ranking");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteRescoreExecutionPlan");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteRescoreExecutionPlanResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteRescoreExecutionPlanResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteRescoreExecutionPlanResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteRescoreExecutionPlanRequest, DeleteRescoreExecutionPlanResponse>()
                            .withOperationName("DeleteRescoreExecutionPlan")
                            .withMarshaller(new DeleteRescoreExecutionPlanRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(deleteRescoreExecutionPlanRequest));
            CompletableFuture<DeleteRescoreExecutionPlanResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Gets information about a rescore execution plan. A rescore execution plan is an Amazon Kendra Intelligent Ranking
     * resource used for provisioning the <code>Rescore</code> API.
     * </p>
     *
     * @param describeRescoreExecutionPlanRequest
     * @return A Java Future containing the result of the DescribeRescoreExecutionPlan operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The input fails to satisfy the constraints set by the Amazon Kendra Intelligent
     *         Ranking service. Please provide the correct input and try again.</li>
     *         <li>ResourceNotFoundException The resource you want to use doesn't exist. Please check you have provided
     *         the correct resource and try again.</li>
     *         <li>ThrottlingException The request was denied due to request throttling. Please reduce the number of
     *         requests and try again.</li>
     *         <li>AccessDeniedException You don’t have sufficient access to perform this action. Please ensure you have
     *         the required permission policies and user accounts and try again.</li>
     *         <li>InternalServerException An issue occurred with the internal server used for your Amazon Kendra
     *         Intelligent Ranking service. Please wait a few minutes and try again, or contact <a
     *         href="http://aws.amazon.com/contact-us/">Support</a> for help.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>KendraRankingException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample KendraRankingAsyncClient.DescribeRescoreExecutionPlan
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/kendra-ranking-2022-10-19/DescribeRescoreExecutionPlan"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeRescoreExecutionPlanResponse> describeRescoreExecutionPlan(
            DescribeRescoreExecutionPlanRequest describeRescoreExecutionPlanRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeRescoreExecutionPlanRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Kendra Ranking");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeRescoreExecutionPlan");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeRescoreExecutionPlanResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeRescoreExecutionPlanResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeRescoreExecutionPlanResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeRescoreExecutionPlanRequest, DescribeRescoreExecutionPlanResponse>()
                            .withOperationName("DescribeRescoreExecutionPlan")
                            .withMarshaller(new DescribeRescoreExecutionPlanRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(describeRescoreExecutionPlanRequest));
            CompletableFuture<DescribeRescoreExecutionPlanResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists your rescore execution plans. A rescore execution plan is an Amazon Kendra Intelligent Ranking resource
     * used for provisioning the <code>Rescore</code> API.
     * </p>
     *
     * @param listRescoreExecutionPlansRequest
     * @return A Java Future containing the result of the ListRescoreExecutionPlans operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The input fails to satisfy the constraints set by the Amazon Kendra Intelligent
     *         Ranking service. Please provide the correct input and try again.</li>
     *         <li>AccessDeniedException You don’t have sufficient access to perform this action. Please ensure you have
     *         the required permission policies and user accounts and try again.</li>
     *         <li>ThrottlingException The request was denied due to request throttling. Please reduce the number of
     *         requests and try again.</li>
     *         <li>InternalServerException An issue occurred with the internal server used for your Amazon Kendra
     *         Intelligent Ranking service. Please wait a few minutes and try again, or contact <a
     *         href="http://aws.amazon.com/contact-us/">Support</a> for help.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>KendraRankingException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample KendraRankingAsyncClient.ListRescoreExecutionPlans
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/kendra-ranking-2022-10-19/ListRescoreExecutionPlans"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListRescoreExecutionPlansResponse> listRescoreExecutionPlans(
            ListRescoreExecutionPlansRequest listRescoreExecutionPlansRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listRescoreExecutionPlansRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Kendra Ranking");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListRescoreExecutionPlans");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListRescoreExecutionPlansResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListRescoreExecutionPlansResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListRescoreExecutionPlansResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListRescoreExecutionPlansRequest, ListRescoreExecutionPlansResponse>()
                            .withOperationName("ListRescoreExecutionPlans")
                            .withMarshaller(new ListRescoreExecutionPlansRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listRescoreExecutionPlansRequest));
            CompletableFuture<ListRescoreExecutionPlansResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Gets a list of tags associated with a specified resource. A rescore execution plan is an example of a resource
     * that can have tags associated with it.
     * </p>
     *
     * @param listTagsForResourceRequest
     *        The request information for listing tags associated with a rescore execution plan. A rescore execution
     *        plan is an Amazon Kendra Intelligent Ranking resource used for provisioning the <code>Rescore</code> API.
     * @return A Java Future containing the result of the ListTagsForResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The input fails to satisfy the constraints set by the Amazon Kendra Intelligent
     *         Ranking service. Please provide the correct input and try again.</li>
     *         <li>ResourceUnavailableException The resource you want to use is unavailable. Please check you have
     *         provided the correct resource information and try again.</li>
     *         <li>ThrottlingException The request was denied due to request throttling. Please reduce the number of
     *         requests and try again.</li>
     *         <li>AccessDeniedException You don’t have sufficient access to perform this action. Please ensure you have
     *         the required permission policies and user accounts and try again.</li>
     *         <li>InternalServerException An issue occurred with the internal server used for your Amazon Kendra
     *         Intelligent Ranking service. Please wait a few minutes and try again, or contact <a
     *         href="http://aws.amazon.com/contact-us/">Support</a> for help.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>KendraRankingException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample KendraRankingAsyncClient.ListTagsForResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/kendra-ranking-2022-10-19/ListTagsForResource"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListTagsForResourceResponse> listTagsForResource(
            ListTagsForResourceRequest listTagsForResourceRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTagsForResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Kendra Ranking");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTagsForResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListTagsForResourceResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListTagsForResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListTagsForResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListTagsForResourceRequest, ListTagsForResourceResponse>()
                            .withOperationName("ListTagsForResource")
                            .withMarshaller(new ListTagsForResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listTagsForResourceRequest));
            CompletableFuture<ListTagsForResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Rescores or re-ranks search results from a search service such as OpenSearch (self managed). You use the semantic
     * search capabilities of Amazon Kendra Intelligent Ranking to improve the search service's results.
     * </p>
     *
     * @param rescoreRequest
     * @return A Java Future containing the result of the Rescore operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The input fails to satisfy the constraints set by the Amazon Kendra Intelligent
     *         Ranking service. Please provide the correct input and try again.</li>
     *         <li>ConflictException A conflict occurred with the request. Please fix any inconsistencies with your
     *         resources and try again.</li>
     *         <li>AccessDeniedException You don’t have sufficient access to perform this action. Please ensure you have
     *         the required permission policies and user accounts and try again.</li>
     *         <li>ResourceNotFoundException The resource you want to use doesn't exist. Please check you have provided
     *         the correct resource and try again.</li>
     *         <li>ThrottlingException The request was denied due to request throttling. Please reduce the number of
     *         requests and try again.</li>
     *         <li>InternalServerException An issue occurred with the internal server used for your Amazon Kendra
     *         Intelligent Ranking service. Please wait a few minutes and try again, or contact <a
     *         href="http://aws.amazon.com/contact-us/">Support</a> for help.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>KendraRankingException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample KendraRankingAsyncClient.Rescore
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/kendra-ranking-2022-10-19/Rescore" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<RescoreResponse> rescore(RescoreRequest rescoreRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, rescoreRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Kendra Ranking");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "Rescore");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<RescoreResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    RescoreResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<RescoreResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<RescoreRequest, RescoreResponse>().withOperationName("Rescore")
                            .withMarshaller(new RescoreRequestMarshaller(protocolFactory)).withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withMetricCollector(apiCallMetricCollector)
                            .withInput(rescoreRequest));
            CompletableFuture<RescoreResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Adds a specified tag to a specified rescore execution plan. A rescore execution plan is an Amazon Kendra
     * Intelligent Ranking resource used for provisioning the <code>Rescore</code> API. If the tag already exists, the
     * existing value is replaced with the new value.
     * </p>
     *
     * @param tagResourceRequest
     *        The request information for tagging a rescore execution plan. A rescore execution plan is an Amazon Kendra
     *        Intelligent Ranking resource used for provisioning the <code>Rescore</code> API.
     * @return A Java Future containing the result of the TagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The input fails to satisfy the constraints set by the Amazon Kendra Intelligent
     *         Ranking service. Please provide the correct input and try again.</li>
     *         <li>ResourceUnavailableException The resource you want to use is unavailable. Please check you have
     *         provided the correct resource information and try again.</li>
     *         <li>ThrottlingException The request was denied due to request throttling. Please reduce the number of
     *         requests and try again.</li>
     *         <li>AccessDeniedException You don’t have sufficient access to perform this action. Please ensure you have
     *         the required permission policies and user accounts and try again.</li>
     *         <li>InternalServerException An issue occurred with the internal server used for your Amazon Kendra
     *         Intelligent Ranking service. Please wait a few minutes and try again, or contact <a
     *         href="http://aws.amazon.com/contact-us/">Support</a> for help.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>KendraRankingException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample KendraRankingAsyncClient.TagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/kendra-ranking-2022-10-19/TagResource" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<TagResourceResponse> tagResource(TagResourceRequest tagResourceRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, tagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Kendra Ranking");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "TagResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<TagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    TagResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<TagResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<TagResourceRequest, TagResourceResponse>()
                            .withOperationName("TagResource").withMarshaller(new TagResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(tagResourceRequest));
            CompletableFuture<TagResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Removes a tag from a rescore execution plan. A rescore execution plan is an Amazon Kendra Intelligent Ranking
     * resource used for provisioning the <code>Rescore</code> operation.
     * </p>
     *
     * @param untagResourceRequest
     *        The request information to remove a tag from a rescore execution plan. A rescore execution plan is an
     *        Amazon Kendra Intelligent Ranking resource used for provisioning the <code>Rescore</code> API.
     * @return A Java Future containing the result of the UntagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The input fails to satisfy the constraints set by the Amazon Kendra Intelligent
     *         Ranking service. Please provide the correct input and try again.</li>
     *         <li>ResourceUnavailableException The resource you want to use is unavailable. Please check you have
     *         provided the correct resource information and try again.</li>
     *         <li>ThrottlingException The request was denied due to request throttling. Please reduce the number of
     *         requests and try again.</li>
     *         <li>AccessDeniedException You don’t have sufficient access to perform this action. Please ensure you have
     *         the required permission policies and user accounts and try again.</li>
     *         <li>InternalServerException An issue occurred with the internal server used for your Amazon Kendra
     *         Intelligent Ranking service. Please wait a few minutes and try again, or contact <a
     *         href="http://aws.amazon.com/contact-us/">Support</a> for help.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>KendraRankingException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample KendraRankingAsyncClient.UntagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/kendra-ranking-2022-10-19/UntagResource" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<UntagResourceResponse> untagResource(UntagResourceRequest untagResourceRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, untagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Kendra Ranking");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UntagResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UntagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    UntagResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UntagResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UntagResourceRequest, UntagResourceResponse>()
                            .withOperationName("UntagResource")
                            .withMarshaller(new UntagResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(untagResourceRequest));
            CompletableFuture<UntagResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates a rescore execution plan. A rescore execution plan is an Amazon Kendra Intelligent Ranking resource used
     * for provisioning the <code>Rescore</code> API. You can update the number of capacity units you require for Amazon
     * Kendra Intelligent Ranking to rescore or re-rank a search service's results.
     * </p>
     *
     * @param updateRescoreExecutionPlanRequest
     * @return A Java Future containing the result of the UpdateRescoreExecutionPlan operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The input fails to satisfy the constraints set by the Amazon Kendra Intelligent
     *         Ranking service. Please provide the correct input and try again.</li>
     *         <li>ResourceNotFoundException The resource you want to use doesn't exist. Please check you have provided
     *         the correct resource and try again.</li>
     *         <li>ThrottlingException The request was denied due to request throttling. Please reduce the number of
     *         requests and try again.</li>
     *         <li>AccessDeniedException You don’t have sufficient access to perform this action. Please ensure you have
     *         the required permission policies and user accounts and try again.</li>
     *         <li>ServiceQuotaExceededException You have exceeded the set limits for your Amazon Kendra Intelligent
     *         Ranking service. Please see <a href="https://docs.aws.amazon.com/kendra/latest/dg/quotas.html">Quotas</a>
     *         for more information, or contact <a href="http://aws.amazon.com/contact-us/">Support</a> to inquire about
     *         an increase of limits.</li>
     *         <li>ConflictException A conflict occurred with the request. Please fix any inconsistencies with your
     *         resources and try again.</li>
     *         <li>InternalServerException An issue occurred with the internal server used for your Amazon Kendra
     *         Intelligent Ranking service. Please wait a few minutes and try again, or contact <a
     *         href="http://aws.amazon.com/contact-us/">Support</a> for help.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>KendraRankingException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample KendraRankingAsyncClient.UpdateRescoreExecutionPlan
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/kendra-ranking-2022-10-19/UpdateRescoreExecutionPlan"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateRescoreExecutionPlanResponse> updateRescoreExecutionPlan(
            UpdateRescoreExecutionPlanRequest updateRescoreExecutionPlanRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateRescoreExecutionPlanRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Kendra Ranking");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateRescoreExecutionPlan");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateRescoreExecutionPlanResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateRescoreExecutionPlanResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateRescoreExecutionPlanResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateRescoreExecutionPlanRequest, UpdateRescoreExecutionPlanResponse>()
                            .withOperationName("UpdateRescoreExecutionPlan")
                            .withMarshaller(new UpdateRescoreExecutionPlanRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(updateRescoreExecutionPlanRequest));
            CompletableFuture<UpdateRescoreExecutionPlanResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    @Override
    public final KendraRankingServiceClientConfiguration serviceClientConfiguration() {
        return this.serviceClientConfiguration;
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(KendraRankingException::builder)
                .protocol(AwsJsonProtocol.AWS_JSON)
                .protocolVersion("1.0")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("AccessDeniedException")
                                .exceptionBuilderSupplier(AccessDeniedException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ValidationException")
                                .exceptionBuilderSupplier(ValidationException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ConflictException")
                                .exceptionBuilderSupplier(ConflictException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException")
                                .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceNotFoundException")
                                .exceptionBuilderSupplier(ResourceNotFoundException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalServerException")
                                .exceptionBuilderSupplier(InternalServerException::builder).httpStatusCode(500).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceUnavailableException")
                                .exceptionBuilderSupplier(ResourceUnavailableException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ThrottlingException")
                                .exceptionBuilderSupplier(ThrottlingException::builder).httpStatusCode(400).build());
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    @Override
    public void close() {
        clientHandler.close();
    }
}
