/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.ivs;

import java.util.Collections;
import java.util.List;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.client.handler.AwsSyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.awscore.internal.AwsProtocolMetadata;
import software.amazon.awssdk.awscore.internal.AwsServiceProtocol;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkPlugin;
import software.amazon.awssdk.core.SdkRequest;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.client.handler.SyncClientHandler;
import software.amazon.awssdk.core.exception.SdkClientException;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.ivs.internal.IvsServiceClientConfigurationBuilder;
import software.amazon.awssdk.services.ivs.model.AccessDeniedException;
import software.amazon.awssdk.services.ivs.model.BatchGetChannelRequest;
import software.amazon.awssdk.services.ivs.model.BatchGetChannelResponse;
import software.amazon.awssdk.services.ivs.model.BatchGetStreamKeyRequest;
import software.amazon.awssdk.services.ivs.model.BatchGetStreamKeyResponse;
import software.amazon.awssdk.services.ivs.model.BatchStartViewerSessionRevocationRequest;
import software.amazon.awssdk.services.ivs.model.BatchStartViewerSessionRevocationResponse;
import software.amazon.awssdk.services.ivs.model.ChannelNotBroadcastingException;
import software.amazon.awssdk.services.ivs.model.ConflictException;
import software.amazon.awssdk.services.ivs.model.CreateChannelRequest;
import software.amazon.awssdk.services.ivs.model.CreateChannelResponse;
import software.amazon.awssdk.services.ivs.model.CreateRecordingConfigurationRequest;
import software.amazon.awssdk.services.ivs.model.CreateRecordingConfigurationResponse;
import software.amazon.awssdk.services.ivs.model.CreateStreamKeyRequest;
import software.amazon.awssdk.services.ivs.model.CreateStreamKeyResponse;
import software.amazon.awssdk.services.ivs.model.DeleteChannelRequest;
import software.amazon.awssdk.services.ivs.model.DeleteChannelResponse;
import software.amazon.awssdk.services.ivs.model.DeletePlaybackKeyPairRequest;
import software.amazon.awssdk.services.ivs.model.DeletePlaybackKeyPairResponse;
import software.amazon.awssdk.services.ivs.model.DeleteRecordingConfigurationRequest;
import software.amazon.awssdk.services.ivs.model.DeleteRecordingConfigurationResponse;
import software.amazon.awssdk.services.ivs.model.DeleteStreamKeyRequest;
import software.amazon.awssdk.services.ivs.model.DeleteStreamKeyResponse;
import software.amazon.awssdk.services.ivs.model.GetChannelRequest;
import software.amazon.awssdk.services.ivs.model.GetChannelResponse;
import software.amazon.awssdk.services.ivs.model.GetPlaybackKeyPairRequest;
import software.amazon.awssdk.services.ivs.model.GetPlaybackKeyPairResponse;
import software.amazon.awssdk.services.ivs.model.GetRecordingConfigurationRequest;
import software.amazon.awssdk.services.ivs.model.GetRecordingConfigurationResponse;
import software.amazon.awssdk.services.ivs.model.GetStreamKeyRequest;
import software.amazon.awssdk.services.ivs.model.GetStreamKeyResponse;
import software.amazon.awssdk.services.ivs.model.GetStreamRequest;
import software.amazon.awssdk.services.ivs.model.GetStreamResponse;
import software.amazon.awssdk.services.ivs.model.GetStreamSessionRequest;
import software.amazon.awssdk.services.ivs.model.GetStreamSessionResponse;
import software.amazon.awssdk.services.ivs.model.ImportPlaybackKeyPairRequest;
import software.amazon.awssdk.services.ivs.model.ImportPlaybackKeyPairResponse;
import software.amazon.awssdk.services.ivs.model.InternalServerException;
import software.amazon.awssdk.services.ivs.model.IvsException;
import software.amazon.awssdk.services.ivs.model.ListChannelsRequest;
import software.amazon.awssdk.services.ivs.model.ListChannelsResponse;
import software.amazon.awssdk.services.ivs.model.ListPlaybackKeyPairsRequest;
import software.amazon.awssdk.services.ivs.model.ListPlaybackKeyPairsResponse;
import software.amazon.awssdk.services.ivs.model.ListRecordingConfigurationsRequest;
import software.amazon.awssdk.services.ivs.model.ListRecordingConfigurationsResponse;
import software.amazon.awssdk.services.ivs.model.ListStreamKeysRequest;
import software.amazon.awssdk.services.ivs.model.ListStreamKeysResponse;
import software.amazon.awssdk.services.ivs.model.ListStreamSessionsRequest;
import software.amazon.awssdk.services.ivs.model.ListStreamSessionsResponse;
import software.amazon.awssdk.services.ivs.model.ListStreamsRequest;
import software.amazon.awssdk.services.ivs.model.ListStreamsResponse;
import software.amazon.awssdk.services.ivs.model.ListTagsForResourceRequest;
import software.amazon.awssdk.services.ivs.model.ListTagsForResourceResponse;
import software.amazon.awssdk.services.ivs.model.PendingVerificationException;
import software.amazon.awssdk.services.ivs.model.PutMetadataRequest;
import software.amazon.awssdk.services.ivs.model.PutMetadataResponse;
import software.amazon.awssdk.services.ivs.model.ResourceNotFoundException;
import software.amazon.awssdk.services.ivs.model.ServiceQuotaExceededException;
import software.amazon.awssdk.services.ivs.model.StartViewerSessionRevocationRequest;
import software.amazon.awssdk.services.ivs.model.StartViewerSessionRevocationResponse;
import software.amazon.awssdk.services.ivs.model.StopStreamRequest;
import software.amazon.awssdk.services.ivs.model.StopStreamResponse;
import software.amazon.awssdk.services.ivs.model.StreamUnavailableException;
import software.amazon.awssdk.services.ivs.model.TagResourceRequest;
import software.amazon.awssdk.services.ivs.model.TagResourceResponse;
import software.amazon.awssdk.services.ivs.model.ThrottlingException;
import software.amazon.awssdk.services.ivs.model.UntagResourceRequest;
import software.amazon.awssdk.services.ivs.model.UntagResourceResponse;
import software.amazon.awssdk.services.ivs.model.UpdateChannelRequest;
import software.amazon.awssdk.services.ivs.model.UpdateChannelResponse;
import software.amazon.awssdk.services.ivs.model.ValidationException;
import software.amazon.awssdk.services.ivs.transform.BatchGetChannelRequestMarshaller;
import software.amazon.awssdk.services.ivs.transform.BatchGetStreamKeyRequestMarshaller;
import software.amazon.awssdk.services.ivs.transform.BatchStartViewerSessionRevocationRequestMarshaller;
import software.amazon.awssdk.services.ivs.transform.CreateChannelRequestMarshaller;
import software.amazon.awssdk.services.ivs.transform.CreateRecordingConfigurationRequestMarshaller;
import software.amazon.awssdk.services.ivs.transform.CreateStreamKeyRequestMarshaller;
import software.amazon.awssdk.services.ivs.transform.DeleteChannelRequestMarshaller;
import software.amazon.awssdk.services.ivs.transform.DeletePlaybackKeyPairRequestMarshaller;
import software.amazon.awssdk.services.ivs.transform.DeleteRecordingConfigurationRequestMarshaller;
import software.amazon.awssdk.services.ivs.transform.DeleteStreamKeyRequestMarshaller;
import software.amazon.awssdk.services.ivs.transform.GetChannelRequestMarshaller;
import software.amazon.awssdk.services.ivs.transform.GetPlaybackKeyPairRequestMarshaller;
import software.amazon.awssdk.services.ivs.transform.GetRecordingConfigurationRequestMarshaller;
import software.amazon.awssdk.services.ivs.transform.GetStreamKeyRequestMarshaller;
import software.amazon.awssdk.services.ivs.transform.GetStreamRequestMarshaller;
import software.amazon.awssdk.services.ivs.transform.GetStreamSessionRequestMarshaller;
import software.amazon.awssdk.services.ivs.transform.ImportPlaybackKeyPairRequestMarshaller;
import software.amazon.awssdk.services.ivs.transform.ListChannelsRequestMarshaller;
import software.amazon.awssdk.services.ivs.transform.ListPlaybackKeyPairsRequestMarshaller;
import software.amazon.awssdk.services.ivs.transform.ListRecordingConfigurationsRequestMarshaller;
import software.amazon.awssdk.services.ivs.transform.ListStreamKeysRequestMarshaller;
import software.amazon.awssdk.services.ivs.transform.ListStreamSessionsRequestMarshaller;
import software.amazon.awssdk.services.ivs.transform.ListStreamsRequestMarshaller;
import software.amazon.awssdk.services.ivs.transform.ListTagsForResourceRequestMarshaller;
import software.amazon.awssdk.services.ivs.transform.PutMetadataRequestMarshaller;
import software.amazon.awssdk.services.ivs.transform.StartViewerSessionRevocationRequestMarshaller;
import software.amazon.awssdk.services.ivs.transform.StopStreamRequestMarshaller;
import software.amazon.awssdk.services.ivs.transform.TagResourceRequestMarshaller;
import software.amazon.awssdk.services.ivs.transform.UntagResourceRequestMarshaller;
import software.amazon.awssdk.services.ivs.transform.UpdateChannelRequestMarshaller;
import software.amazon.awssdk.utils.Logger;

/**
 * Internal implementation of {@link IvsClient}.
 *
 * @see IvsClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultIvsClient implements IvsClient {
    private static final Logger log = Logger.loggerFor(DefaultIvsClient.class);

    private static final AwsProtocolMetadata protocolMetadata = AwsProtocolMetadata.builder()
            .serviceProtocol(AwsServiceProtocol.REST_JSON).build();

    private final SyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultIvsClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsSyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    /**
     * <p>
     * Performs <a>GetChannel</a> on multiple ARNs simultaneously.
     * </p>
     *
     * @param batchGetChannelRequest
     * @return Result of the BatchGetChannel operation returned by the service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws IvsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample IvsClient.BatchGetChannel
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ivs-2020-07-14/BatchGetChannel" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public BatchGetChannelResponse batchGetChannel(BatchGetChannelRequest batchGetChannelRequest) throws AwsServiceException,
            SdkClientException, IvsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<BatchGetChannelResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                BatchGetChannelResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(batchGetChannelRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, batchGetChannelRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ivs");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "BatchGetChannel");

            return clientHandler.execute(new ClientExecutionParams<BatchGetChannelRequest, BatchGetChannelResponse>()
                    .withOperationName("BatchGetChannel").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(batchGetChannelRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new BatchGetChannelRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Performs <a>GetStreamKey</a> on multiple ARNs simultaneously.
     * </p>
     *
     * @param batchGetStreamKeyRequest
     * @return Result of the BatchGetStreamKey operation returned by the service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws IvsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample IvsClient.BatchGetStreamKey
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ivs-2020-07-14/BatchGetStreamKey" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public BatchGetStreamKeyResponse batchGetStreamKey(BatchGetStreamKeyRequest batchGetStreamKeyRequest)
            throws AwsServiceException, SdkClientException, IvsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<BatchGetStreamKeyResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                BatchGetStreamKeyResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(batchGetStreamKeyRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, batchGetStreamKeyRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ivs");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "BatchGetStreamKey");

            return clientHandler.execute(new ClientExecutionParams<BatchGetStreamKeyRequest, BatchGetStreamKeyResponse>()
                    .withOperationName("BatchGetStreamKey").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(batchGetStreamKeyRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new BatchGetStreamKeyRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Performs <a>StartViewerSessionRevocation</a> on multiple channel ARN and viewer ID pairs simultaneously.
     * </p>
     *
     * @param batchStartViewerSessionRevocationRequest
     * @return Result of the BatchStartViewerSessionRevocation operation returned by the service.
     * @throws AccessDeniedException
     * @throws ValidationException
     * @throws PendingVerificationException
     * @throws ThrottlingException
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws IvsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample IvsClient.BatchStartViewerSessionRevocation
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ivs-2020-07-14/BatchStartViewerSessionRevocation"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public BatchStartViewerSessionRevocationResponse batchStartViewerSessionRevocation(
            BatchStartViewerSessionRevocationRequest batchStartViewerSessionRevocationRequest) throws AccessDeniedException,
            ValidationException, PendingVerificationException, ThrottlingException, AwsServiceException, SdkClientException,
            IvsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<BatchStartViewerSessionRevocationResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, BatchStartViewerSessionRevocationResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(batchStartViewerSessionRevocationRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                batchStartViewerSessionRevocationRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ivs");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "BatchStartViewerSessionRevocation");

            return clientHandler
                    .execute(new ClientExecutionParams<BatchStartViewerSessionRevocationRequest, BatchStartViewerSessionRevocationResponse>()
                            .withOperationName("BatchStartViewerSessionRevocation").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(batchStartViewerSessionRevocationRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new BatchStartViewerSessionRevocationRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Creates a new channel and an associated stream key to start streaming.
     * </p>
     *
     * @param createChannelRequest
     * @return Result of the CreateChannel operation returned by the service.
     * @throws ResourceNotFoundException
     * @throws AccessDeniedException
     * @throws ValidationException
     * @throws PendingVerificationException
     * @throws ServiceQuotaExceededException
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws IvsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample IvsClient.CreateChannel
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ivs-2020-07-14/CreateChannel" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CreateChannelResponse createChannel(CreateChannelRequest createChannelRequest) throws ResourceNotFoundException,
            AccessDeniedException, ValidationException, PendingVerificationException, ServiceQuotaExceededException,
            AwsServiceException, SdkClientException, IvsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateChannelResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                CreateChannelResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createChannelRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createChannelRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ivs");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateChannel");

            return clientHandler.execute(new ClientExecutionParams<CreateChannelRequest, CreateChannelResponse>()
                    .withOperationName("CreateChannel").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(createChannelRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new CreateChannelRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Creates a new recording configuration, used to enable recording to Amazon S3.
     * </p>
     * <p>
     * <b>Known issue:</b> In the us-east-1 region, if you use the Amazon Web Services CLI to create a recording
     * configuration, it returns success even if the S3 bucket is in a different region. In this case, the
     * <code>state</code> of the recording configuration is <code>CREATE_FAILED</code> (instead of <code>ACTIVE</code>).
     * (In other regions, the CLI correctly returns failure if the bucket is in a different region.)
     * </p>
     * <p>
     * <b>Workaround:</b> Ensure that your S3 bucket is in the same region as the recording configuration. If you create
     * a recording configuration in a different region as your S3 bucket, delete that recording configuration and create
     * a new one with an S3 bucket from the correct region.
     * </p>
     *
     * @param createRecordingConfigurationRequest
     * @return Result of the CreateRecordingConfiguration operation returned by the service.
     * @throws InternalServerException
     * @throws AccessDeniedException
     * @throws ValidationException
     * @throws PendingVerificationException
     * @throws ConflictException
     * @throws ServiceQuotaExceededException
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws IvsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample IvsClient.CreateRecordingConfiguration
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ivs-2020-07-14/CreateRecordingConfiguration"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CreateRecordingConfigurationResponse createRecordingConfiguration(
            CreateRecordingConfigurationRequest createRecordingConfigurationRequest) throws InternalServerException,
            AccessDeniedException, ValidationException, PendingVerificationException, ConflictException,
            ServiceQuotaExceededException, AwsServiceException, SdkClientException, IvsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateRecordingConfigurationResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, CreateRecordingConfigurationResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createRecordingConfigurationRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createRecordingConfigurationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ivs");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateRecordingConfiguration");

            return clientHandler
                    .execute(new ClientExecutionParams<CreateRecordingConfigurationRequest, CreateRecordingConfigurationResponse>()
                            .withOperationName("CreateRecordingConfiguration").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(createRecordingConfigurationRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new CreateRecordingConfigurationRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Creates a stream key, used to initiate a stream, for the specified channel ARN.
     * </p>
     * <p>
     * Note that <a>CreateChannel</a> creates a stream key. If you subsequently use CreateStreamKey on the same channel,
     * it will fail because a stream key already exists and there is a limit of 1 stream key per channel. To reset the
     * stream key on a channel, use <a>DeleteStreamKey</a> and then CreateStreamKey.
     * </p>
     *
     * @param createStreamKeyRequest
     * @return Result of the CreateStreamKey operation returned by the service.
     * @throws ResourceNotFoundException
     * @throws AccessDeniedException
     * @throws ValidationException
     * @throws PendingVerificationException
     * @throws ServiceQuotaExceededException
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws IvsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample IvsClient.CreateStreamKey
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ivs-2020-07-14/CreateStreamKey" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CreateStreamKeyResponse createStreamKey(CreateStreamKeyRequest createStreamKeyRequest)
            throws ResourceNotFoundException, AccessDeniedException, ValidationException, PendingVerificationException,
            ServiceQuotaExceededException, AwsServiceException, SdkClientException, IvsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateStreamKeyResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                CreateStreamKeyResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createStreamKeyRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createStreamKeyRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ivs");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateStreamKey");

            return clientHandler.execute(new ClientExecutionParams<CreateStreamKeyRequest, CreateStreamKeyResponse>()
                    .withOperationName("CreateStreamKey").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(createStreamKeyRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new CreateStreamKeyRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deletes the specified channel and its associated stream keys.
     * </p>
     * <p>
     * If you try to delete a live channel, you will get an error (409 ConflictException). To delete a channel that is
     * live, call <a>StopStream</a>, wait for the Amazon EventBridge "Stream End" event (to verify that the stream's
     * state is no longer Live), then call DeleteChannel. (See <a
     * href="https://docs.aws.amazon.com/ivs/latest/userguide/eventbridge.html"> Using EventBridge with Amazon IVS</a>.)
     * </p>
     *
     * @param deleteChannelRequest
     * @return Result of the DeleteChannel operation returned by the service.
     * @throws ResourceNotFoundException
     * @throws AccessDeniedException
     * @throws ValidationException
     * @throws PendingVerificationException
     * @throws ConflictException
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws IvsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample IvsClient.DeleteChannel
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ivs-2020-07-14/DeleteChannel" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public DeleteChannelResponse deleteChannel(DeleteChannelRequest deleteChannelRequest) throws ResourceNotFoundException,
            AccessDeniedException, ValidationException, PendingVerificationException, ConflictException, AwsServiceException,
            SdkClientException, IvsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteChannelResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DeleteChannelResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteChannelRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteChannelRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ivs");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteChannel");

            return clientHandler.execute(new ClientExecutionParams<DeleteChannelRequest, DeleteChannelResponse>()
                    .withOperationName("DeleteChannel").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(deleteChannelRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DeleteChannelRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deletes a specified authorization key pair. This invalidates future viewer tokens generated using the key pair’s
     * <code>privateKey</code>. For more information, see <a
     * href="https://docs.aws.amazon.com/ivs/latest/userguide/private-channels.html">Setting Up Private Channels</a> in
     * the <i>Amazon IVS User Guide</i>.
     * </p>
     *
     * @param deletePlaybackKeyPairRequest
     * @return Result of the DeletePlaybackKeyPair operation returned by the service.
     * @throws ResourceNotFoundException
     * @throws AccessDeniedException
     * @throws ValidationException
     * @throws PendingVerificationException
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws IvsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample IvsClient.DeletePlaybackKeyPair
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ivs-2020-07-14/DeletePlaybackKeyPair" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public DeletePlaybackKeyPairResponse deletePlaybackKeyPair(DeletePlaybackKeyPairRequest deletePlaybackKeyPairRequest)
            throws ResourceNotFoundException, AccessDeniedException, ValidationException, PendingVerificationException,
            AwsServiceException, SdkClientException, IvsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeletePlaybackKeyPairResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DeletePlaybackKeyPairResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deletePlaybackKeyPairRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deletePlaybackKeyPairRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ivs");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeletePlaybackKeyPair");

            return clientHandler.execute(new ClientExecutionParams<DeletePlaybackKeyPairRequest, DeletePlaybackKeyPairResponse>()
                    .withOperationName("DeletePlaybackKeyPair").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(deletePlaybackKeyPairRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DeletePlaybackKeyPairRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deletes the recording configuration for the specified ARN.
     * </p>
     * <p>
     * If you try to delete a recording configuration that is associated with a channel, you will get an error (409
     * ConflictException). To avoid this, for all channels that reference the recording configuration, first use
     * <a>UpdateChannel</a> to set the <code>recordingConfigurationArn</code> field to an empty string, then use
     * DeleteRecordingConfiguration.
     * </p>
     *
     * @param deleteRecordingConfigurationRequest
     * @return Result of the DeleteRecordingConfiguration operation returned by the service.
     * @throws ResourceNotFoundException
     * @throws InternalServerException
     * @throws AccessDeniedException
     * @throws ValidationException
     * @throws ConflictException
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws IvsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample IvsClient.DeleteRecordingConfiguration
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ivs-2020-07-14/DeleteRecordingConfiguration"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DeleteRecordingConfigurationResponse deleteRecordingConfiguration(
            DeleteRecordingConfigurationRequest deleteRecordingConfigurationRequest) throws ResourceNotFoundException,
            InternalServerException, AccessDeniedException, ValidationException, ConflictException, AwsServiceException,
            SdkClientException, IvsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteRecordingConfigurationResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DeleteRecordingConfigurationResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteRecordingConfigurationRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteRecordingConfigurationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ivs");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteRecordingConfiguration");

            return clientHandler
                    .execute(new ClientExecutionParams<DeleteRecordingConfigurationRequest, DeleteRecordingConfigurationResponse>()
                            .withOperationName("DeleteRecordingConfiguration").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(deleteRecordingConfigurationRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new DeleteRecordingConfigurationRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deletes the stream key for the specified ARN, so it can no longer be used to stream.
     * </p>
     *
     * @param deleteStreamKeyRequest
     * @return Result of the DeleteStreamKey operation returned by the service.
     * @throws ResourceNotFoundException
     * @throws AccessDeniedException
     * @throws ValidationException
     * @throws PendingVerificationException
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws IvsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample IvsClient.DeleteStreamKey
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ivs-2020-07-14/DeleteStreamKey" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public DeleteStreamKeyResponse deleteStreamKey(DeleteStreamKeyRequest deleteStreamKeyRequest)
            throws ResourceNotFoundException, AccessDeniedException, ValidationException, PendingVerificationException,
            AwsServiceException, SdkClientException, IvsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteStreamKeyResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DeleteStreamKeyResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteStreamKeyRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteStreamKeyRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ivs");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteStreamKey");

            return clientHandler.execute(new ClientExecutionParams<DeleteStreamKeyRequest, DeleteStreamKeyResponse>()
                    .withOperationName("DeleteStreamKey").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(deleteStreamKeyRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DeleteStreamKeyRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Gets the channel configuration for the specified channel ARN. See also <a>BatchGetChannel</a>.
     * </p>
     *
     * @param getChannelRequest
     * @return Result of the GetChannel operation returned by the service.
     * @throws ResourceNotFoundException
     * @throws AccessDeniedException
     * @throws ValidationException
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws IvsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample IvsClient.GetChannel
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ivs-2020-07-14/GetChannel" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public GetChannelResponse getChannel(GetChannelRequest getChannelRequest) throws ResourceNotFoundException,
            AccessDeniedException, ValidationException, AwsServiceException, SdkClientException, IvsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetChannelResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetChannelResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getChannelRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getChannelRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ivs");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetChannel");

            return clientHandler.execute(new ClientExecutionParams<GetChannelRequest, GetChannelResponse>()
                    .withOperationName("GetChannel").withProtocolMetadata(protocolMetadata).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(getChannelRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetChannelRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Gets a specified playback authorization key pair and returns the <code>arn</code> and <code>fingerprint</code>.
     * The <code>privateKey</code> held by the caller can be used to generate viewer authorization tokens, to grant
     * viewers access to private channels. For more information, see <a
     * href="https://docs.aws.amazon.com/ivs/latest/userguide/private-channels.html">Setting Up Private Channels</a> in
     * the <i>Amazon IVS User Guide</i>.
     * </p>
     *
     * @param getPlaybackKeyPairRequest
     * @return Result of the GetPlaybackKeyPair operation returned by the service.
     * @throws ResourceNotFoundException
     * @throws AccessDeniedException
     * @throws ValidationException
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws IvsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample IvsClient.GetPlaybackKeyPair
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ivs-2020-07-14/GetPlaybackKeyPair" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public GetPlaybackKeyPairResponse getPlaybackKeyPair(GetPlaybackKeyPairRequest getPlaybackKeyPairRequest)
            throws ResourceNotFoundException, AccessDeniedException, ValidationException, AwsServiceException,
            SdkClientException, IvsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetPlaybackKeyPairResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, GetPlaybackKeyPairResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getPlaybackKeyPairRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getPlaybackKeyPairRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ivs");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetPlaybackKeyPair");

            return clientHandler.execute(new ClientExecutionParams<GetPlaybackKeyPairRequest, GetPlaybackKeyPairResponse>()
                    .withOperationName("GetPlaybackKeyPair").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(getPlaybackKeyPairRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetPlaybackKeyPairRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Gets the recording configuration for the specified ARN.
     * </p>
     *
     * @param getRecordingConfigurationRequest
     * @return Result of the GetRecordingConfiguration operation returned by the service.
     * @throws ResourceNotFoundException
     * @throws InternalServerException
     * @throws AccessDeniedException
     * @throws ValidationException
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws IvsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample IvsClient.GetRecordingConfiguration
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ivs-2020-07-14/GetRecordingConfiguration" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public GetRecordingConfigurationResponse getRecordingConfiguration(
            GetRecordingConfigurationRequest getRecordingConfigurationRequest) throws ResourceNotFoundException,
            InternalServerException, AccessDeniedException, ValidationException, AwsServiceException, SdkClientException,
            IvsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetRecordingConfigurationResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, GetRecordingConfigurationResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getRecordingConfigurationRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getRecordingConfigurationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ivs");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetRecordingConfiguration");

            return clientHandler
                    .execute(new ClientExecutionParams<GetRecordingConfigurationRequest, GetRecordingConfigurationResponse>()
                            .withOperationName("GetRecordingConfiguration").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(getRecordingConfigurationRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new GetRecordingConfigurationRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Gets information about the active (live) stream on a specified channel.
     * </p>
     *
     * @param getStreamRequest
     * @return Result of the GetStream operation returned by the service.
     * @throws ResourceNotFoundException
     * @throws AccessDeniedException
     * @throws ValidationException
     * @throws ChannelNotBroadcastingException
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws IvsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample IvsClient.GetStream
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ivs-2020-07-14/GetStream" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public GetStreamResponse getStream(GetStreamRequest getStreamRequest) throws ResourceNotFoundException,
            AccessDeniedException, ValidationException, ChannelNotBroadcastingException, AwsServiceException, SdkClientException,
            IvsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetStreamResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetStreamResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getStreamRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getStreamRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ivs");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetStream");

            return clientHandler.execute(new ClientExecutionParams<GetStreamRequest, GetStreamResponse>()
                    .withOperationName("GetStream").withProtocolMetadata(protocolMetadata).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(getStreamRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetStreamRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Gets stream-key information for a specified ARN.
     * </p>
     *
     * @param getStreamKeyRequest
     * @return Result of the GetStreamKey operation returned by the service.
     * @throws ResourceNotFoundException
     * @throws AccessDeniedException
     * @throws ValidationException
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws IvsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample IvsClient.GetStreamKey
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ivs-2020-07-14/GetStreamKey" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public GetStreamKeyResponse getStreamKey(GetStreamKeyRequest getStreamKeyRequest) throws ResourceNotFoundException,
            AccessDeniedException, ValidationException, AwsServiceException, SdkClientException, IvsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetStreamKeyResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetStreamKeyResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getStreamKeyRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getStreamKeyRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ivs");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetStreamKey");

            return clientHandler.execute(new ClientExecutionParams<GetStreamKeyRequest, GetStreamKeyResponse>()
                    .withOperationName("GetStreamKey").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(getStreamKeyRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetStreamKeyRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Gets metadata on a specified stream.
     * </p>
     *
     * @param getStreamSessionRequest
     * @return Result of the GetStreamSession operation returned by the service.
     * @throws ResourceNotFoundException
     * @throws AccessDeniedException
     * @throws ValidationException
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws IvsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample IvsClient.GetStreamSession
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ivs-2020-07-14/GetStreamSession" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public GetStreamSessionResponse getStreamSession(GetStreamSessionRequest getStreamSessionRequest)
            throws ResourceNotFoundException, AccessDeniedException, ValidationException, AwsServiceException,
            SdkClientException, IvsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetStreamSessionResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetStreamSessionResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getStreamSessionRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getStreamSessionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ivs");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetStreamSession");

            return clientHandler.execute(new ClientExecutionParams<GetStreamSessionRequest, GetStreamSessionResponse>()
                    .withOperationName("GetStreamSession").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(getStreamSessionRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetStreamSessionRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Imports the public portion of a new key pair and returns its <code>arn</code> and <code>fingerprint</code>. The
     * <code>privateKey</code> can then be used to generate viewer authorization tokens, to grant viewers access to
     * private channels. For more information, see <a
     * href="https://docs.aws.amazon.com/ivs/latest/userguide/private-channels.html">Setting Up Private Channels</a> in
     * the <i>Amazon IVS User Guide</i>.
     * </p>
     *
     * @param importPlaybackKeyPairRequest
     * @return Result of the ImportPlaybackKeyPair operation returned by the service.
     * @throws AccessDeniedException
     * @throws ValidationException
     * @throws PendingVerificationException
     * @throws ConflictException
     * @throws ServiceQuotaExceededException
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws IvsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample IvsClient.ImportPlaybackKeyPair
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ivs-2020-07-14/ImportPlaybackKeyPair" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ImportPlaybackKeyPairResponse importPlaybackKeyPair(ImportPlaybackKeyPairRequest importPlaybackKeyPairRequest)
            throws AccessDeniedException, ValidationException, PendingVerificationException, ConflictException,
            ServiceQuotaExceededException, AwsServiceException, SdkClientException, IvsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ImportPlaybackKeyPairResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ImportPlaybackKeyPairResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(importPlaybackKeyPairRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, importPlaybackKeyPairRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ivs");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ImportPlaybackKeyPair");

            return clientHandler.execute(new ClientExecutionParams<ImportPlaybackKeyPairRequest, ImportPlaybackKeyPairResponse>()
                    .withOperationName("ImportPlaybackKeyPair").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(importPlaybackKeyPairRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ImportPlaybackKeyPairRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Gets summary information about all channels in your account, in the Amazon Web Services region where the API
     * request is processed. This list can be filtered to match a specified name or recording-configuration ARN. Filters
     * are mutually exclusive and cannot be used together. If you try to use both filters, you will get an error (409
     * ConflictException).
     * </p>
     *
     * @param listChannelsRequest
     * @return Result of the ListChannels operation returned by the service.
     * @throws AccessDeniedException
     * @throws ValidationException
     * @throws ConflictException
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws IvsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample IvsClient.ListChannels
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ivs-2020-07-14/ListChannels" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListChannelsResponse listChannels(ListChannelsRequest listChannelsRequest) throws AccessDeniedException,
            ValidationException, ConflictException, AwsServiceException, SdkClientException, IvsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListChannelsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListChannelsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listChannelsRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listChannelsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ivs");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListChannels");

            return clientHandler.execute(new ClientExecutionParams<ListChannelsRequest, ListChannelsResponse>()
                    .withOperationName("ListChannels").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(listChannelsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListChannelsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Gets summary information about playback key pairs. For more information, see <a
     * href="https://docs.aws.amazon.com/ivs/latest/userguide/private-channels.html">Setting Up Private Channels</a> in
     * the <i>Amazon IVS User Guide</i>.
     * </p>
     *
     * @param listPlaybackKeyPairsRequest
     * @return Result of the ListPlaybackKeyPairs operation returned by the service.
     * @throws AccessDeniedException
     * @throws ValidationException
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws IvsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample IvsClient.ListPlaybackKeyPairs
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ivs-2020-07-14/ListPlaybackKeyPairs" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListPlaybackKeyPairsResponse listPlaybackKeyPairs(ListPlaybackKeyPairsRequest listPlaybackKeyPairsRequest)
            throws AccessDeniedException, ValidationException, AwsServiceException, SdkClientException, IvsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListPlaybackKeyPairsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListPlaybackKeyPairsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listPlaybackKeyPairsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listPlaybackKeyPairsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ivs");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListPlaybackKeyPairs");

            return clientHandler.execute(new ClientExecutionParams<ListPlaybackKeyPairsRequest, ListPlaybackKeyPairsResponse>()
                    .withOperationName("ListPlaybackKeyPairs").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(listPlaybackKeyPairsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListPlaybackKeyPairsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Gets summary information about all recording configurations in your account, in the Amazon Web Services region
     * where the API request is processed.
     * </p>
     *
     * @param listRecordingConfigurationsRequest
     * @return Result of the ListRecordingConfigurations operation returned by the service.
     * @throws InternalServerException
     * @throws AccessDeniedException
     * @throws ValidationException
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws IvsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample IvsClient.ListRecordingConfigurations
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ivs-2020-07-14/ListRecordingConfigurations"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListRecordingConfigurationsResponse listRecordingConfigurations(
            ListRecordingConfigurationsRequest listRecordingConfigurationsRequest) throws InternalServerException,
            AccessDeniedException, ValidationException, AwsServiceException, SdkClientException, IvsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListRecordingConfigurationsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListRecordingConfigurationsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listRecordingConfigurationsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listRecordingConfigurationsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ivs");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListRecordingConfigurations");

            return clientHandler
                    .execute(new ClientExecutionParams<ListRecordingConfigurationsRequest, ListRecordingConfigurationsResponse>()
                            .withOperationName("ListRecordingConfigurations").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(listRecordingConfigurationsRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new ListRecordingConfigurationsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Gets summary information about stream keys for the specified channel.
     * </p>
     *
     * @param listStreamKeysRequest
     * @return Result of the ListStreamKeys operation returned by the service.
     * @throws ResourceNotFoundException
     * @throws AccessDeniedException
     * @throws ValidationException
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws IvsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample IvsClient.ListStreamKeys
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ivs-2020-07-14/ListStreamKeys" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListStreamKeysResponse listStreamKeys(ListStreamKeysRequest listStreamKeysRequest) throws ResourceNotFoundException,
            AccessDeniedException, ValidationException, AwsServiceException, SdkClientException, IvsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListStreamKeysResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListStreamKeysResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listStreamKeysRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listStreamKeysRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ivs");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListStreamKeys");

            return clientHandler.execute(new ClientExecutionParams<ListStreamKeysRequest, ListStreamKeysResponse>()
                    .withOperationName("ListStreamKeys").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(listStreamKeysRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListStreamKeysRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Gets a summary of current and previous streams for a specified channel in your account, in the AWS region where
     * the API request is processed.
     * </p>
     *
     * @param listStreamSessionsRequest
     * @return Result of the ListStreamSessions operation returned by the service.
     * @throws ResourceNotFoundException
     * @throws AccessDeniedException
     * @throws ValidationException
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws IvsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample IvsClient.ListStreamSessions
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ivs-2020-07-14/ListStreamSessions" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListStreamSessionsResponse listStreamSessions(ListStreamSessionsRequest listStreamSessionsRequest)
            throws ResourceNotFoundException, AccessDeniedException, ValidationException, AwsServiceException,
            SdkClientException, IvsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListStreamSessionsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListStreamSessionsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listStreamSessionsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listStreamSessionsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ivs");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListStreamSessions");

            return clientHandler.execute(new ClientExecutionParams<ListStreamSessionsRequest, ListStreamSessionsResponse>()
                    .withOperationName("ListStreamSessions").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(listStreamSessionsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListStreamSessionsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Gets summary information about live streams in your account, in the Amazon Web Services region where the API
     * request is processed.
     * </p>
     *
     * @param listStreamsRequest
     * @return Result of the ListStreams operation returned by the service.
     * @throws AccessDeniedException
     * @throws ValidationException
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws IvsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample IvsClient.ListStreams
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ivs-2020-07-14/ListStreams" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListStreamsResponse listStreams(ListStreamsRequest listStreamsRequest) throws AccessDeniedException,
            ValidationException, AwsServiceException, SdkClientException, IvsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListStreamsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListStreamsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listStreamsRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listStreamsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ivs");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListStreams");

            return clientHandler.execute(new ClientExecutionParams<ListStreamsRequest, ListStreamsResponse>()
                    .withOperationName("ListStreams").withProtocolMetadata(protocolMetadata).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(listStreamsRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListStreamsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Gets information about Amazon Web Services tags for the specified ARN.
     * </p>
     *
     * @param listTagsForResourceRequest
     * @return Result of the ListTagsForResource operation returned by the service.
     * @throws ResourceNotFoundException
     * @throws InternalServerException
     * @throws ValidationException
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws IvsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample IvsClient.ListTagsForResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ivs-2020-07-14/ListTagsForResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListTagsForResourceResponse listTagsForResource(ListTagsForResourceRequest listTagsForResourceRequest)
            throws ResourceNotFoundException, InternalServerException, ValidationException, AwsServiceException,
            SdkClientException, IvsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListTagsForResourceResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListTagsForResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listTagsForResourceRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTagsForResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ivs");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTagsForResource");

            return clientHandler.execute(new ClientExecutionParams<ListTagsForResourceRequest, ListTagsForResourceResponse>()
                    .withOperationName("ListTagsForResource").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(listTagsForResourceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListTagsForResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Inserts metadata into the active stream of the specified channel. At most 5 requests per second per channel are
     * allowed, each with a maximum 1 KB payload. (If 5 TPS is not sufficient for your needs, we recommend batching your
     * data into a single PutMetadata call.) At most 155 requests per second per account are allowed. Also see <a
     * href="https://docs.aws.amazon.com/ivs/latest/userguide/metadata.html">Embedding Metadata within a Video
     * Stream</a> in the <i>Amazon IVS User Guide</i>.
     * </p>
     *
     * @param putMetadataRequest
     * @return Result of the PutMetadata operation returned by the service.
     * @throws ResourceNotFoundException
     * @throws AccessDeniedException
     * @throws ValidationException
     * @throws ChannelNotBroadcastingException
     * @throws ThrottlingException
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws IvsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample IvsClient.PutMetadata
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ivs-2020-07-14/PutMetadata" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public PutMetadataResponse putMetadata(PutMetadataRequest putMetadataRequest) throws ResourceNotFoundException,
            AccessDeniedException, ValidationException, ChannelNotBroadcastingException, ThrottlingException,
            AwsServiceException, SdkClientException, IvsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<PutMetadataResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                PutMetadataResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(putMetadataRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, putMetadataRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ivs");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "PutMetadata");

            return clientHandler.execute(new ClientExecutionParams<PutMetadataRequest, PutMetadataResponse>()
                    .withOperationName("PutMetadata").withProtocolMetadata(protocolMetadata).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(putMetadataRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new PutMetadataRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Starts the process of revoking the viewer session associated with a specified channel ARN and viewer ID.
     * Optionally, you can provide a version to revoke viewer sessions less than and including that version. For
     * instructions on associating a viewer ID with a viewer session, see <a
     * href="https://docs.aws.amazon.com/ivs/latest/userguide/private-channels.html">Setting Up Private Channels</a>.
     * </p>
     *
     * @param startViewerSessionRevocationRequest
     * @return Result of the StartViewerSessionRevocation operation returned by the service.
     * @throws ResourceNotFoundException
     * @throws InternalServerException
     * @throws AccessDeniedException
     * @throws ValidationException
     * @throws PendingVerificationException
     * @throws ThrottlingException
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws IvsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample IvsClient.StartViewerSessionRevocation
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ivs-2020-07-14/StartViewerSessionRevocation"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public StartViewerSessionRevocationResponse startViewerSessionRevocation(
            StartViewerSessionRevocationRequest startViewerSessionRevocationRequest) throws ResourceNotFoundException,
            InternalServerException, AccessDeniedException, ValidationException, PendingVerificationException,
            ThrottlingException, AwsServiceException, SdkClientException, IvsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<StartViewerSessionRevocationResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, StartViewerSessionRevocationResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(startViewerSessionRevocationRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, startViewerSessionRevocationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ivs");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "StartViewerSessionRevocation");

            return clientHandler
                    .execute(new ClientExecutionParams<StartViewerSessionRevocationRequest, StartViewerSessionRevocationResponse>()
                            .withOperationName("StartViewerSessionRevocation").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(startViewerSessionRevocationRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new StartViewerSessionRevocationRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Disconnects the incoming RTMPS stream for the specified channel. Can be used in conjunction with
     * <a>DeleteStreamKey</a> to prevent further streaming to a channel.
     * </p>
     * <note>
     * <p>
     * Many streaming client-software libraries automatically reconnect a dropped RTMPS session, so to stop the stream
     * permanently, you may want to first revoke the <code>streamKey</code> attached to the channel.
     * </p>
     * </note>
     *
     * @param stopStreamRequest
     * @return Result of the StopStream operation returned by the service.
     * @throws ResourceNotFoundException
     * @throws AccessDeniedException
     * @throws ValidationException
     * @throws ChannelNotBroadcastingException
     * @throws StreamUnavailableException
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws IvsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample IvsClient.StopStream
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ivs-2020-07-14/StopStream" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public StopStreamResponse stopStream(StopStreamRequest stopStreamRequest) throws ResourceNotFoundException,
            AccessDeniedException, ValidationException, ChannelNotBroadcastingException, StreamUnavailableException,
            AwsServiceException, SdkClientException, IvsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<StopStreamResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                StopStreamResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(stopStreamRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, stopStreamRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ivs");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "StopStream");

            return clientHandler.execute(new ClientExecutionParams<StopStreamRequest, StopStreamResponse>()
                    .withOperationName("StopStream").withProtocolMetadata(protocolMetadata).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(stopStreamRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new StopStreamRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Adds or updates tags for the Amazon Web Services resource with the specified ARN.
     * </p>
     *
     * @param tagResourceRequest
     * @return Result of the TagResource operation returned by the service.
     * @throws ResourceNotFoundException
     * @throws InternalServerException
     * @throws ValidationException
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws IvsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample IvsClient.TagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ivs-2020-07-14/TagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public TagResourceResponse tagResource(TagResourceRequest tagResourceRequest) throws ResourceNotFoundException,
            InternalServerException, ValidationException, AwsServiceException, SdkClientException, IvsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<TagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                TagResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(tagResourceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, tagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ivs");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "TagResource");

            return clientHandler.execute(new ClientExecutionParams<TagResourceRequest, TagResourceResponse>()
                    .withOperationName("TagResource").withProtocolMetadata(protocolMetadata).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(tagResourceRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new TagResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Removes tags from the resource with the specified ARN.
     * </p>
     *
     * @param untagResourceRequest
     * @return Result of the UntagResource operation returned by the service.
     * @throws ResourceNotFoundException
     * @throws InternalServerException
     * @throws ValidationException
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws IvsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample IvsClient.UntagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ivs-2020-07-14/UntagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public UntagResourceResponse untagResource(UntagResourceRequest untagResourceRequest) throws ResourceNotFoundException,
            InternalServerException, ValidationException, AwsServiceException, SdkClientException, IvsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UntagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                UntagResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(untagResourceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, untagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ivs");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UntagResource");

            return clientHandler.execute(new ClientExecutionParams<UntagResourceRequest, UntagResourceResponse>()
                    .withOperationName("UntagResource").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(untagResourceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UntagResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Updates a channel's configuration. Live channels cannot be updated. You must stop the ongoing stream, update the
     * channel, and restart the stream for the changes to take effect.
     * </p>
     *
     * @param updateChannelRequest
     * @return Result of the UpdateChannel operation returned by the service.
     * @throws ResourceNotFoundException
     * @throws AccessDeniedException
     * @throws ValidationException
     * @throws PendingVerificationException
     * @throws ConflictException
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws IvsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample IvsClient.UpdateChannel
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ivs-2020-07-14/UpdateChannel" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public UpdateChannelResponse updateChannel(UpdateChannelRequest updateChannelRequest) throws ResourceNotFoundException,
            AccessDeniedException, ValidationException, PendingVerificationException, ConflictException, AwsServiceException,
            SdkClientException, IvsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateChannelResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                UpdateChannelResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateChannelRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateChannelRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ivs");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateChannel");

            return clientHandler.execute(new ClientExecutionParams<UpdateChannelRequest, UpdateChannelResponse>()
                    .withOperationName("UpdateChannel").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(updateChannelRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UpdateChannelRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    private SdkClientConfiguration updateSdkClientConfiguration(SdkRequest request, SdkClientConfiguration clientConfiguration) {
        List<SdkPlugin> plugins = request.overrideConfiguration().map(c -> c.plugins()).orElse(Collections.emptyList());
        if (plugins.isEmpty()) {
            return clientConfiguration;
        }
        SdkClientConfiguration.Builder configuration = clientConfiguration.toBuilder();
        IvsServiceClientConfigurationBuilder serviceConfigBuilder = new IvsServiceClientConfigurationBuilder(configuration);
        for (SdkPlugin plugin : plugins) {
            plugin.configureClient(serviceConfigBuilder);
        }
        return configuration.build();
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(IvsException::builder)
                .protocol(AwsJsonProtocol.REST_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("StreamUnavailable")
                                .exceptionBuilderSupplier(StreamUnavailableException::builder).httpStatusCode(503).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ChannelNotBroadcasting")
                                .exceptionBuilderSupplier(ChannelNotBroadcastingException::builder).httpStatusCode(404).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ThrottlingException")
                                .exceptionBuilderSupplier(ThrottlingException::builder).httpStatusCode(429).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException")
                                .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).httpStatusCode(402).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("PendingVerification")
                                .exceptionBuilderSupplier(PendingVerificationException::builder).httpStatusCode(403).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalServerException")
                                .exceptionBuilderSupplier(InternalServerException::builder).httpStatusCode(500).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("AccessDeniedException")
                                .exceptionBuilderSupplier(AccessDeniedException::builder).httpStatusCode(403).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ConflictException")
                                .exceptionBuilderSupplier(ConflictException::builder).httpStatusCode(409).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceNotFoundException")
                                .exceptionBuilderSupplier(ResourceNotFoundException::builder).httpStatusCode(404).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ValidationException")
                                .exceptionBuilderSupplier(ValidationException::builder).httpStatusCode(400).build());
    }

    @Override
    public final IvsServiceClientConfiguration serviceClientConfiguration() {
        return new IvsServiceClientConfigurationBuilder(this.clientConfiguration.toBuilder()).build();
    }

    @Override
    public void close() {
        clientHandler.close();
    }
}
