/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.glue.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Specifies a JDBC data store to crawl.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class JdbcTarget implements SdkPojo, Serializable, ToCopyableBuilder<JdbcTarget.Builder, JdbcTarget> {
    private static final SdkField<String> CONNECTION_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ConnectionName").getter(getter(JdbcTarget::connectionName)).setter(setter(Builder::connectionName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ConnectionName").build()).build();

    private static final SdkField<String> PATH_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Path")
            .getter(getter(JdbcTarget::path)).setter(setter(Builder::path))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Path").build()).build();

    private static final SdkField<List<String>> EXCLUSIONS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("Exclusions")
            .getter(getter(JdbcTarget::exclusions))
            .setter(setter(Builder::exclusions))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Exclusions").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(CONNECTION_NAME_FIELD,
            PATH_FIELD, EXCLUSIONS_FIELD));

    private static final long serialVersionUID = 1L;

    private final String connectionName;

    private final String path;

    private final List<String> exclusions;

    private JdbcTarget(BuilderImpl builder) {
        this.connectionName = builder.connectionName;
        this.path = builder.path;
        this.exclusions = builder.exclusions;
    }

    /**
     * <p>
     * The name of the connection to use to connect to the JDBC target.
     * </p>
     * 
     * @return The name of the connection to use to connect to the JDBC target.
     */
    public final String connectionName() {
        return connectionName;
    }

    /**
     * <p>
     * The path of the JDBC target.
     * </p>
     * 
     * @return The path of the JDBC target.
     */
    public final String path() {
        return path;
    }

    /**
     * For responses, this returns true if the service returned a value for the Exclusions property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasExclusions() {
        return exclusions != null && !(exclusions instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A list of glob patterns used to exclude from the crawl. For more information, see <a
     * href="https://docs.aws.amazon.com/glue/latest/dg/add-crawler.html">Catalog Tables with a Crawler</a>.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasExclusions} method.
     * </p>
     * 
     * @return A list of glob patterns used to exclude from the crawl. For more information, see <a
     *         href="https://docs.aws.amazon.com/glue/latest/dg/add-crawler.html">Catalog Tables with a Crawler</a>.
     */
    public final List<String> exclusions() {
        return exclusions;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(connectionName());
        hashCode = 31 * hashCode + Objects.hashCode(path());
        hashCode = 31 * hashCode + Objects.hashCode(hasExclusions() ? exclusions() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof JdbcTarget)) {
            return false;
        }
        JdbcTarget other = (JdbcTarget) obj;
        return Objects.equals(connectionName(), other.connectionName()) && Objects.equals(path(), other.path())
                && hasExclusions() == other.hasExclusions() && Objects.equals(exclusions(), other.exclusions());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("JdbcTarget").add("ConnectionName", connectionName()).add("Path", path())
                .add("Exclusions", hasExclusions() ? exclusions() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "ConnectionName":
            return Optional.ofNullable(clazz.cast(connectionName()));
        case "Path":
            return Optional.ofNullable(clazz.cast(path()));
        case "Exclusions":
            return Optional.ofNullable(clazz.cast(exclusions()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<JdbcTarget, T> g) {
        return obj -> g.apply((JdbcTarget) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, JdbcTarget> {
        /**
         * <p>
         * The name of the connection to use to connect to the JDBC target.
         * </p>
         * 
         * @param connectionName
         *        The name of the connection to use to connect to the JDBC target.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder connectionName(String connectionName);

        /**
         * <p>
         * The path of the JDBC target.
         * </p>
         * 
         * @param path
         *        The path of the JDBC target.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder path(String path);

        /**
         * <p>
         * A list of glob patterns used to exclude from the crawl. For more information, see <a
         * href="https://docs.aws.amazon.com/glue/latest/dg/add-crawler.html">Catalog Tables with a Crawler</a>.
         * </p>
         * 
         * @param exclusions
         *        A list of glob patterns used to exclude from the crawl. For more information, see <a
         *        href="https://docs.aws.amazon.com/glue/latest/dg/add-crawler.html">Catalog Tables with a Crawler</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder exclusions(Collection<String> exclusions);

        /**
         * <p>
         * A list of glob patterns used to exclude from the crawl. For more information, see <a
         * href="https://docs.aws.amazon.com/glue/latest/dg/add-crawler.html">Catalog Tables with a Crawler</a>.
         * </p>
         * 
         * @param exclusions
         *        A list of glob patterns used to exclude from the crawl. For more information, see <a
         *        href="https://docs.aws.amazon.com/glue/latest/dg/add-crawler.html">Catalog Tables with a Crawler</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder exclusions(String... exclusions);
    }

    static final class BuilderImpl implements Builder {
        private String connectionName;

        private String path;

        private List<String> exclusions = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(JdbcTarget model) {
            connectionName(model.connectionName);
            path(model.path);
            exclusions(model.exclusions);
        }

        public final String getConnectionName() {
            return connectionName;
        }

        public final void setConnectionName(String connectionName) {
            this.connectionName = connectionName;
        }

        @Override
        public final Builder connectionName(String connectionName) {
            this.connectionName = connectionName;
            return this;
        }

        public final String getPath() {
            return path;
        }

        public final void setPath(String path) {
            this.path = path;
        }

        @Override
        public final Builder path(String path) {
            this.path = path;
            return this;
        }

        public final Collection<String> getExclusions() {
            if (exclusions instanceof SdkAutoConstructList) {
                return null;
            }
            return exclusions;
        }

        public final void setExclusions(Collection<String> exclusions) {
            this.exclusions = PathListCopier.copy(exclusions);
        }

        @Override
        public final Builder exclusions(Collection<String> exclusions) {
            this.exclusions = PathListCopier.copy(exclusions);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder exclusions(String... exclusions) {
            exclusions(Arrays.asList(exclusions));
            return this;
        }

        @Override
        public JdbcTarget build() {
            return new JdbcTarget(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
