/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.gamelift.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Represents an EC2 instance of virtual computing resources that hosts one or more game servers. In GameLift, a fleet
 * can contain zero or more instances.
 * </p>
 * <p>
 * <b>Related actions</b>
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class Instance implements SdkPojo, Serializable, ToCopyableBuilder<Instance.Builder, Instance> {
    private static final SdkField<String> FLEET_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("FleetId").getter(getter(Instance::fleetId)).setter(setter(Builder::fleetId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("FleetId").build()).build();

    private static final SdkField<String> FLEET_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("FleetArn").getter(getter(Instance::fleetArn)).setter(setter(Builder::fleetArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("FleetArn").build()).build();

    private static final SdkField<String> INSTANCE_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("InstanceId").getter(getter(Instance::instanceId)).setter(setter(Builder::instanceId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("InstanceId").build()).build();

    private static final SdkField<String> IP_ADDRESS_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("IpAddress").getter(getter(Instance::ipAddress)).setter(setter(Builder::ipAddress))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("IpAddress").build()).build();

    private static final SdkField<String> DNS_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("DnsName").getter(getter(Instance::dnsName)).setter(setter(Builder::dnsName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DnsName").build()).build();

    private static final SdkField<String> OPERATING_SYSTEM_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("OperatingSystem").getter(getter(Instance::operatingSystemAsString))
            .setter(setter(Builder::operatingSystem))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("OperatingSystem").build()).build();

    private static final SdkField<String> TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Type")
            .getter(getter(Instance::typeAsString)).setter(setter(Builder::type))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Type").build()).build();

    private static final SdkField<String> STATUS_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Status")
            .getter(getter(Instance::statusAsString)).setter(setter(Builder::status))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Status").build()).build();

    private static final SdkField<Instant> CREATION_TIME_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("CreationTime").getter(getter(Instance::creationTime)).setter(setter(Builder::creationTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CreationTime").build()).build();

    private static final SdkField<String> LOCATION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("Location").getter(getter(Instance::location)).setter(setter(Builder::location))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Location").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(FLEET_ID_FIELD,
            FLEET_ARN_FIELD, INSTANCE_ID_FIELD, IP_ADDRESS_FIELD, DNS_NAME_FIELD, OPERATING_SYSTEM_FIELD, TYPE_FIELD,
            STATUS_FIELD, CREATION_TIME_FIELD, LOCATION_FIELD));

    private static final long serialVersionUID = 1L;

    private final String fleetId;

    private final String fleetArn;

    private final String instanceId;

    private final String ipAddress;

    private final String dnsName;

    private final String operatingSystem;

    private final String type;

    private final String status;

    private final Instant creationTime;

    private final String location;

    private Instance(BuilderImpl builder) {
        this.fleetId = builder.fleetId;
        this.fleetArn = builder.fleetArn;
        this.instanceId = builder.instanceId;
        this.ipAddress = builder.ipAddress;
        this.dnsName = builder.dnsName;
        this.operatingSystem = builder.operatingSystem;
        this.type = builder.type;
        this.status = builder.status;
        this.creationTime = builder.creationTime;
        this.location = builder.location;
    }

    /**
     * <p>
     * A unique identifier for the fleet that the instance is in.
     * </p>
     * 
     * @return A unique identifier for the fleet that the instance is in.
     */
    public final String fleetId() {
        return fleetId;
    }

    /**
     * <p>
     * The Amazon Resource Name (<a href="https://docs.aws.amazon.com/AmazonS3/latest/dev/s3-arn-format.html">ARN</a>)
     * that is assigned to a GameLift fleet resource and uniquely identifies it. ARNs are unique across all Regions.
     * Format is <code>arn:aws:gamelift:&lt;region&gt;::fleet/fleet-a1234567-b8c9-0d1e-2fa3-b45c6d7e8912</code>.
     * </p>
     * 
     * @return The Amazon Resource Name (<a
     *         href="https://docs.aws.amazon.com/AmazonS3/latest/dev/s3-arn-format.html">ARN</a>) that is assigned to a
     *         GameLift fleet resource and uniquely identifies it. ARNs are unique across all Regions. Format is
     *         <code>arn:aws:gamelift:&lt;region&gt;::fleet/fleet-a1234567-b8c9-0d1e-2fa3-b45c6d7e8912</code>.
     */
    public final String fleetArn() {
        return fleetArn;
    }

    /**
     * <p>
     * A unique identifier for the instance.
     * </p>
     * 
     * @return A unique identifier for the instance.
     */
    public final String instanceId() {
        return instanceId;
    }

    /**
     * <p>
     * IP address that is assigned to the instance.
     * </p>
     * 
     * @return IP address that is assigned to the instance.
     */
    public final String ipAddress() {
        return ipAddress;
    }

    /**
     * <p>
     * The DNS identifier assigned to the instance that is running the game session. Values have the following format:
     * </p>
     * <ul>
     * <li>
     * <p>
     * TLS-enabled fleets: <code>&lt;unique identifier&gt;.&lt;region identifier&gt;.amazongamelift.com</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * Non-TLS-enabled fleets: <code>ec2-&lt;unique identifier&gt;.compute.amazonaws.com</code>. (See <a href=
     * "https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/using-instance-addressing.html#concepts-public-addresses"
     * >Amazon EC2 Instance IP Addressing</a>.)
     * </p>
     * </li>
     * </ul>
     * <p>
     * When connecting to a game session that is running on a TLS-enabled fleet, you must use the DNS name, not the IP
     * address.
     * </p>
     * 
     * @return The DNS identifier assigned to the instance that is running the game session. Values have the following
     *         format:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         TLS-enabled fleets: <code>&lt;unique identifier&gt;.&lt;region identifier&gt;.amazongamelift.com</code>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Non-TLS-enabled fleets: <code>ec2-&lt;unique identifier&gt;.compute.amazonaws.com</code>. (See <a href=
     *         "https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/using-instance-addressing.html#concepts-public-addresses"
     *         >Amazon EC2 Instance IP Addressing</a>.)
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         When connecting to a game session that is running on a TLS-enabled fleet, you must use the DNS name, not
     *         the IP address.
     */
    public final String dnsName() {
        return dnsName;
    }

    /**
     * <p>
     * Operating system that is running on this instance.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #operatingSystem}
     * will return {@link OperatingSystem#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #operatingSystemAsString}.
     * </p>
     * 
     * @return Operating system that is running on this instance.
     * @see OperatingSystem
     */
    public final OperatingSystem operatingSystem() {
        return OperatingSystem.fromValue(operatingSystem);
    }

    /**
     * <p>
     * Operating system that is running on this instance.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #operatingSystem}
     * will return {@link OperatingSystem#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #operatingSystemAsString}.
     * </p>
     * 
     * @return Operating system that is running on this instance.
     * @see OperatingSystem
     */
    public final String operatingSystemAsString() {
        return operatingSystem;
    }

    /**
     * <p>
     * Amazon EC2 instance type that defines the computing resources of this instance.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #type} will return
     * {@link EC2InstanceType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #typeAsString}.
     * </p>
     * 
     * @return Amazon EC2 instance type that defines the computing resources of this instance.
     * @see EC2InstanceType
     */
    public final EC2InstanceType type() {
        return EC2InstanceType.fromValue(type);
    }

    /**
     * <p>
     * Amazon EC2 instance type that defines the computing resources of this instance.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #type} will return
     * {@link EC2InstanceType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #typeAsString}.
     * </p>
     * 
     * @return Amazon EC2 instance type that defines the computing resources of this instance.
     * @see EC2InstanceType
     */
    public final String typeAsString() {
        return type;
    }

    /**
     * <p>
     * Current status of the instance. Possible statuses include the following:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <b>PENDING</b> -- The instance is in the process of being created and launching server processes as defined in
     * the fleet's run-time configuration.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>ACTIVE</b> -- The instance has been successfully created and at least one server process has successfully
     * launched and reported back to GameLift that it is ready to host a game session. The instance is now considered
     * ready to host game sessions.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>TERMINATING</b> -- The instance is in the process of shutting down. This may happen to reduce capacity during
     * a scaling down event or to recycle resources in the event of a problem.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link InstanceStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #statusAsString}.
     * </p>
     * 
     * @return Current status of the instance. Possible statuses include the following:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <b>PENDING</b> -- The instance is in the process of being created and launching server processes as
     *         defined in the fleet's run-time configuration.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>ACTIVE</b> -- The instance has been successfully created and at least one server process has
     *         successfully launched and reported back to GameLift that it is ready to host a game session. The instance
     *         is now considered ready to host game sessions.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>TERMINATING</b> -- The instance is in the process of shutting down. This may happen to reduce capacity
     *         during a scaling down event or to recycle resources in the event of a problem.
     *         </p>
     *         </li>
     * @see InstanceStatus
     */
    public final InstanceStatus status() {
        return InstanceStatus.fromValue(status);
    }

    /**
     * <p>
     * Current status of the instance. Possible statuses include the following:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <b>PENDING</b> -- The instance is in the process of being created and launching server processes as defined in
     * the fleet's run-time configuration.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>ACTIVE</b> -- The instance has been successfully created and at least one server process has successfully
     * launched and reported back to GameLift that it is ready to host a game session. The instance is now considered
     * ready to host game sessions.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>TERMINATING</b> -- The instance is in the process of shutting down. This may happen to reduce capacity during
     * a scaling down event or to recycle resources in the event of a problem.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link InstanceStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #statusAsString}.
     * </p>
     * 
     * @return Current status of the instance. Possible statuses include the following:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <b>PENDING</b> -- The instance is in the process of being created and launching server processes as
     *         defined in the fleet's run-time configuration.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>ACTIVE</b> -- The instance has been successfully created and at least one server process has
     *         successfully launched and reported back to GameLift that it is ready to host a game session. The instance
     *         is now considered ready to host game sessions.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>TERMINATING</b> -- The instance is in the process of shutting down. This may happen to reduce capacity
     *         during a scaling down event or to recycle resources in the event of a problem.
     *         </p>
     *         </li>
     * @see InstanceStatus
     */
    public final String statusAsString() {
        return status;
    }

    /**
     * <p>
     * A time stamp indicating when this data object was created. Format is a number expressed in Unix time as
     * milliseconds (for example <code>"1469498468.057"</code>).
     * </p>
     * 
     * @return A time stamp indicating when this data object was created. Format is a number expressed in Unix time as
     *         milliseconds (for example <code>"1469498468.057"</code>).
     */
    public final Instant creationTime() {
        return creationTime;
    }

    /**
     * <p>
     * The fleet location of the instance, expressed as an Amazon Web Services Region code, such as
     * <code>us-west-2</code>.
     * </p>
     * 
     * @return The fleet location of the instance, expressed as an Amazon Web Services Region code, such as
     *         <code>us-west-2</code>.
     */
    public final String location() {
        return location;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(fleetId());
        hashCode = 31 * hashCode + Objects.hashCode(fleetArn());
        hashCode = 31 * hashCode + Objects.hashCode(instanceId());
        hashCode = 31 * hashCode + Objects.hashCode(ipAddress());
        hashCode = 31 * hashCode + Objects.hashCode(dnsName());
        hashCode = 31 * hashCode + Objects.hashCode(operatingSystemAsString());
        hashCode = 31 * hashCode + Objects.hashCode(typeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(statusAsString());
        hashCode = 31 * hashCode + Objects.hashCode(creationTime());
        hashCode = 31 * hashCode + Objects.hashCode(location());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof Instance)) {
            return false;
        }
        Instance other = (Instance) obj;
        return Objects.equals(fleetId(), other.fleetId()) && Objects.equals(fleetArn(), other.fleetArn())
                && Objects.equals(instanceId(), other.instanceId()) && Objects.equals(ipAddress(), other.ipAddress())
                && Objects.equals(dnsName(), other.dnsName())
                && Objects.equals(operatingSystemAsString(), other.operatingSystemAsString())
                && Objects.equals(typeAsString(), other.typeAsString())
                && Objects.equals(statusAsString(), other.statusAsString())
                && Objects.equals(creationTime(), other.creationTime()) && Objects.equals(location(), other.location());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("Instance").add("FleetId", fleetId()).add("FleetArn", fleetArn()).add("InstanceId", instanceId())
                .add("IpAddress", ipAddress()).add("DnsName", dnsName()).add("OperatingSystem", operatingSystemAsString())
                .add("Type", typeAsString()).add("Status", statusAsString()).add("CreationTime", creationTime())
                .add("Location", location()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "FleetId":
            return Optional.ofNullable(clazz.cast(fleetId()));
        case "FleetArn":
            return Optional.ofNullable(clazz.cast(fleetArn()));
        case "InstanceId":
            return Optional.ofNullable(clazz.cast(instanceId()));
        case "IpAddress":
            return Optional.ofNullable(clazz.cast(ipAddress()));
        case "DnsName":
            return Optional.ofNullable(clazz.cast(dnsName()));
        case "OperatingSystem":
            return Optional.ofNullable(clazz.cast(operatingSystemAsString()));
        case "Type":
            return Optional.ofNullable(clazz.cast(typeAsString()));
        case "Status":
            return Optional.ofNullable(clazz.cast(statusAsString()));
        case "CreationTime":
            return Optional.ofNullable(clazz.cast(creationTime()));
        case "Location":
            return Optional.ofNullable(clazz.cast(location()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<Instance, T> g) {
        return obj -> g.apply((Instance) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, Instance> {
        /**
         * <p>
         * A unique identifier for the fleet that the instance is in.
         * </p>
         * 
         * @param fleetId
         *        A unique identifier for the fleet that the instance is in.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder fleetId(String fleetId);

        /**
         * <p>
         * The Amazon Resource Name (<a
         * href="https://docs.aws.amazon.com/AmazonS3/latest/dev/s3-arn-format.html">ARN</a>) that is assigned to a
         * GameLift fleet resource and uniquely identifies it. ARNs are unique across all Regions. Format is
         * <code>arn:aws:gamelift:&lt;region&gt;::fleet/fleet-a1234567-b8c9-0d1e-2fa3-b45c6d7e8912</code>.
         * </p>
         * 
         * @param fleetArn
         *        The Amazon Resource Name (<a
         *        href="https://docs.aws.amazon.com/AmazonS3/latest/dev/s3-arn-format.html">ARN</a>) that is assigned to
         *        a GameLift fleet resource and uniquely identifies it. ARNs are unique across all Regions. Format is
         *        <code>arn:aws:gamelift:&lt;region&gt;::fleet/fleet-a1234567-b8c9-0d1e-2fa3-b45c6d7e8912</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder fleetArn(String fleetArn);

        /**
         * <p>
         * A unique identifier for the instance.
         * </p>
         * 
         * @param instanceId
         *        A unique identifier for the instance.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder instanceId(String instanceId);

        /**
         * <p>
         * IP address that is assigned to the instance.
         * </p>
         * 
         * @param ipAddress
         *        IP address that is assigned to the instance.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder ipAddress(String ipAddress);

        /**
         * <p>
         * The DNS identifier assigned to the instance that is running the game session. Values have the following
         * format:
         * </p>
         * <ul>
         * <li>
         * <p>
         * TLS-enabled fleets: <code>&lt;unique identifier&gt;.&lt;region identifier&gt;.amazongamelift.com</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * Non-TLS-enabled fleets: <code>ec2-&lt;unique identifier&gt;.compute.amazonaws.com</code>. (See <a href=
         * "https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/using-instance-addressing.html#concepts-public-addresses"
         * >Amazon EC2 Instance IP Addressing</a>.)
         * </p>
         * </li>
         * </ul>
         * <p>
         * When connecting to a game session that is running on a TLS-enabled fleet, you must use the DNS name, not the
         * IP address.
         * </p>
         * 
         * @param dnsName
         *        The DNS identifier assigned to the instance that is running the game session. Values have the
         *        following format:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        TLS-enabled fleets:
         *        <code>&lt;unique identifier&gt;.&lt;region identifier&gt;.amazongamelift.com</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Non-TLS-enabled fleets: <code>ec2-&lt;unique identifier&gt;.compute.amazonaws.com</code>. (See <a
         *        href=
         *        "https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/using-instance-addressing.html#concepts-public-addresses"
         *        >Amazon EC2 Instance IP Addressing</a>.)
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        When connecting to a game session that is running on a TLS-enabled fleet, you must use the DNS name,
         *        not the IP address.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder dnsName(String dnsName);

        /**
         * <p>
         * Operating system that is running on this instance.
         * </p>
         * 
         * @param operatingSystem
         *        Operating system that is running on this instance.
         * @see OperatingSystem
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see OperatingSystem
         */
        Builder operatingSystem(String operatingSystem);

        /**
         * <p>
         * Operating system that is running on this instance.
         * </p>
         * 
         * @param operatingSystem
         *        Operating system that is running on this instance.
         * @see OperatingSystem
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see OperatingSystem
         */
        Builder operatingSystem(OperatingSystem operatingSystem);

        /**
         * <p>
         * Amazon EC2 instance type that defines the computing resources of this instance.
         * </p>
         * 
         * @param type
         *        Amazon EC2 instance type that defines the computing resources of this instance.
         * @see EC2InstanceType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see EC2InstanceType
         */
        Builder type(String type);

        /**
         * <p>
         * Amazon EC2 instance type that defines the computing resources of this instance.
         * </p>
         * 
         * @param type
         *        Amazon EC2 instance type that defines the computing resources of this instance.
         * @see EC2InstanceType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see EC2InstanceType
         */
        Builder type(EC2InstanceType type);

        /**
         * <p>
         * Current status of the instance. Possible statuses include the following:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <b>PENDING</b> -- The instance is in the process of being created and launching server processes as defined
         * in the fleet's run-time configuration.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b>ACTIVE</b> -- The instance has been successfully created and at least one server process has successfully
         * launched and reported back to GameLift that it is ready to host a game session. The instance is now
         * considered ready to host game sessions.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b>TERMINATING</b> -- The instance is in the process of shutting down. This may happen to reduce capacity
         * during a scaling down event or to recycle resources in the event of a problem.
         * </p>
         * </li>
         * </ul>
         * 
         * @param status
         *        Current status of the instance. Possible statuses include the following:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <b>PENDING</b> -- The instance is in the process of being created and launching server processes as
         *        defined in the fleet's run-time configuration.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b>ACTIVE</b> -- The instance has been successfully created and at least one server process has
         *        successfully launched and reported back to GameLift that it is ready to host a game session. The
         *        instance is now considered ready to host game sessions.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b>TERMINATING</b> -- The instance is in the process of shutting down. This may happen to reduce
         *        capacity during a scaling down event or to recycle resources in the event of a problem.
         *        </p>
         *        </li>
         * @see InstanceStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see InstanceStatus
         */
        Builder status(String status);

        /**
         * <p>
         * Current status of the instance. Possible statuses include the following:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <b>PENDING</b> -- The instance is in the process of being created and launching server processes as defined
         * in the fleet's run-time configuration.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b>ACTIVE</b> -- The instance has been successfully created and at least one server process has successfully
         * launched and reported back to GameLift that it is ready to host a game session. The instance is now
         * considered ready to host game sessions.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b>TERMINATING</b> -- The instance is in the process of shutting down. This may happen to reduce capacity
         * during a scaling down event or to recycle resources in the event of a problem.
         * </p>
         * </li>
         * </ul>
         * 
         * @param status
         *        Current status of the instance. Possible statuses include the following:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <b>PENDING</b> -- The instance is in the process of being created and launching server processes as
         *        defined in the fleet's run-time configuration.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b>ACTIVE</b> -- The instance has been successfully created and at least one server process has
         *        successfully launched and reported back to GameLift that it is ready to host a game session. The
         *        instance is now considered ready to host game sessions.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b>TERMINATING</b> -- The instance is in the process of shutting down. This may happen to reduce
         *        capacity during a scaling down event or to recycle resources in the event of a problem.
         *        </p>
         *        </li>
         * @see InstanceStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see InstanceStatus
         */
        Builder status(InstanceStatus status);

        /**
         * <p>
         * A time stamp indicating when this data object was created. Format is a number expressed in Unix time as
         * milliseconds (for example <code>"1469498468.057"</code>).
         * </p>
         * 
         * @param creationTime
         *        A time stamp indicating when this data object was created. Format is a number expressed in Unix time
         *        as milliseconds (for example <code>"1469498468.057"</code>).
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder creationTime(Instant creationTime);

        /**
         * <p>
         * The fleet location of the instance, expressed as an Amazon Web Services Region code, such as
         * <code>us-west-2</code>.
         * </p>
         * 
         * @param location
         *        The fleet location of the instance, expressed as an Amazon Web Services Region code, such as
         *        <code>us-west-2</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder location(String location);
    }

    static final class BuilderImpl implements Builder {
        private String fleetId;

        private String fleetArn;

        private String instanceId;

        private String ipAddress;

        private String dnsName;

        private String operatingSystem;

        private String type;

        private String status;

        private Instant creationTime;

        private String location;

        private BuilderImpl() {
        }

        private BuilderImpl(Instance model) {
            fleetId(model.fleetId);
            fleetArn(model.fleetArn);
            instanceId(model.instanceId);
            ipAddress(model.ipAddress);
            dnsName(model.dnsName);
            operatingSystem(model.operatingSystem);
            type(model.type);
            status(model.status);
            creationTime(model.creationTime);
            location(model.location);
        }

        public final String getFleetId() {
            return fleetId;
        }

        public final void setFleetId(String fleetId) {
            this.fleetId = fleetId;
        }

        @Override
        public final Builder fleetId(String fleetId) {
            this.fleetId = fleetId;
            return this;
        }

        public final String getFleetArn() {
            return fleetArn;
        }

        public final void setFleetArn(String fleetArn) {
            this.fleetArn = fleetArn;
        }

        @Override
        public final Builder fleetArn(String fleetArn) {
            this.fleetArn = fleetArn;
            return this;
        }

        public final String getInstanceId() {
            return instanceId;
        }

        public final void setInstanceId(String instanceId) {
            this.instanceId = instanceId;
        }

        @Override
        public final Builder instanceId(String instanceId) {
            this.instanceId = instanceId;
            return this;
        }

        public final String getIpAddress() {
            return ipAddress;
        }

        public final void setIpAddress(String ipAddress) {
            this.ipAddress = ipAddress;
        }

        @Override
        public final Builder ipAddress(String ipAddress) {
            this.ipAddress = ipAddress;
            return this;
        }

        public final String getDnsName() {
            return dnsName;
        }

        public final void setDnsName(String dnsName) {
            this.dnsName = dnsName;
        }

        @Override
        public final Builder dnsName(String dnsName) {
            this.dnsName = dnsName;
            return this;
        }

        public final String getOperatingSystem() {
            return operatingSystem;
        }

        public final void setOperatingSystem(String operatingSystem) {
            this.operatingSystem = operatingSystem;
        }

        @Override
        public final Builder operatingSystem(String operatingSystem) {
            this.operatingSystem = operatingSystem;
            return this;
        }

        @Override
        public final Builder operatingSystem(OperatingSystem operatingSystem) {
            this.operatingSystem(operatingSystem == null ? null : operatingSystem.toString());
            return this;
        }

        public final String getType() {
            return type;
        }

        public final void setType(String type) {
            this.type = type;
        }

        @Override
        public final Builder type(String type) {
            this.type = type;
            return this;
        }

        @Override
        public final Builder type(EC2InstanceType type) {
            this.type(type == null ? null : type.toString());
            return this;
        }

        public final String getStatus() {
            return status;
        }

        public final void setStatus(String status) {
            this.status = status;
        }

        @Override
        public final Builder status(String status) {
            this.status = status;
            return this;
        }

        @Override
        public final Builder status(InstanceStatus status) {
            this.status(status == null ? null : status.toString());
            return this;
        }

        public final Instant getCreationTime() {
            return creationTime;
        }

        public final void setCreationTime(Instant creationTime) {
            this.creationTime = creationTime;
        }

        @Override
        public final Builder creationTime(Instant creationTime) {
            this.creationTime = creationTime;
            return this;
        }

        public final String getLocation() {
            return location;
        }

        public final void setLocation(String location) {
            this.location = location;
        }

        @Override
        public final Builder location(String location) {
            this.location = location;
            return this;
        }

        @Override
        public Instance build() {
            return new Instance(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
