/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.fis;

import java.util.Collections;
import java.util.List;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.client.handler.AwsSyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.awscore.internal.AwsProtocolMetadata;
import software.amazon.awssdk.awscore.internal.AwsServiceProtocol;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkPlugin;
import software.amazon.awssdk.core.SdkRequest;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.client.handler.SyncClientHandler;
import software.amazon.awssdk.core.exception.SdkClientException;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.fis.internal.FisServiceClientConfigurationBuilder;
import software.amazon.awssdk.services.fis.model.ConflictException;
import software.amazon.awssdk.services.fis.model.CreateExperimentTemplateRequest;
import software.amazon.awssdk.services.fis.model.CreateExperimentTemplateResponse;
import software.amazon.awssdk.services.fis.model.DeleteExperimentTemplateRequest;
import software.amazon.awssdk.services.fis.model.DeleteExperimentTemplateResponse;
import software.amazon.awssdk.services.fis.model.FisException;
import software.amazon.awssdk.services.fis.model.GetActionRequest;
import software.amazon.awssdk.services.fis.model.GetActionResponse;
import software.amazon.awssdk.services.fis.model.GetExperimentRequest;
import software.amazon.awssdk.services.fis.model.GetExperimentResponse;
import software.amazon.awssdk.services.fis.model.GetExperimentTemplateRequest;
import software.amazon.awssdk.services.fis.model.GetExperimentTemplateResponse;
import software.amazon.awssdk.services.fis.model.GetTargetResourceTypeRequest;
import software.amazon.awssdk.services.fis.model.GetTargetResourceTypeResponse;
import software.amazon.awssdk.services.fis.model.ListActionsRequest;
import software.amazon.awssdk.services.fis.model.ListActionsResponse;
import software.amazon.awssdk.services.fis.model.ListExperimentTemplatesRequest;
import software.amazon.awssdk.services.fis.model.ListExperimentTemplatesResponse;
import software.amazon.awssdk.services.fis.model.ListExperimentsRequest;
import software.amazon.awssdk.services.fis.model.ListExperimentsResponse;
import software.amazon.awssdk.services.fis.model.ListTagsForResourceRequest;
import software.amazon.awssdk.services.fis.model.ListTagsForResourceResponse;
import software.amazon.awssdk.services.fis.model.ListTargetResourceTypesRequest;
import software.amazon.awssdk.services.fis.model.ListTargetResourceTypesResponse;
import software.amazon.awssdk.services.fis.model.ResourceNotFoundException;
import software.amazon.awssdk.services.fis.model.ServiceQuotaExceededException;
import software.amazon.awssdk.services.fis.model.StartExperimentRequest;
import software.amazon.awssdk.services.fis.model.StartExperimentResponse;
import software.amazon.awssdk.services.fis.model.StopExperimentRequest;
import software.amazon.awssdk.services.fis.model.StopExperimentResponse;
import software.amazon.awssdk.services.fis.model.TagResourceRequest;
import software.amazon.awssdk.services.fis.model.TagResourceResponse;
import software.amazon.awssdk.services.fis.model.UntagResourceRequest;
import software.amazon.awssdk.services.fis.model.UntagResourceResponse;
import software.amazon.awssdk.services.fis.model.UpdateExperimentTemplateRequest;
import software.amazon.awssdk.services.fis.model.UpdateExperimentTemplateResponse;
import software.amazon.awssdk.services.fis.model.ValidationException;
import software.amazon.awssdk.services.fis.transform.CreateExperimentTemplateRequestMarshaller;
import software.amazon.awssdk.services.fis.transform.DeleteExperimentTemplateRequestMarshaller;
import software.amazon.awssdk.services.fis.transform.GetActionRequestMarshaller;
import software.amazon.awssdk.services.fis.transform.GetExperimentRequestMarshaller;
import software.amazon.awssdk.services.fis.transform.GetExperimentTemplateRequestMarshaller;
import software.amazon.awssdk.services.fis.transform.GetTargetResourceTypeRequestMarshaller;
import software.amazon.awssdk.services.fis.transform.ListActionsRequestMarshaller;
import software.amazon.awssdk.services.fis.transform.ListExperimentTemplatesRequestMarshaller;
import software.amazon.awssdk.services.fis.transform.ListExperimentsRequestMarshaller;
import software.amazon.awssdk.services.fis.transform.ListTagsForResourceRequestMarshaller;
import software.amazon.awssdk.services.fis.transform.ListTargetResourceTypesRequestMarshaller;
import software.amazon.awssdk.services.fis.transform.StartExperimentRequestMarshaller;
import software.amazon.awssdk.services.fis.transform.StopExperimentRequestMarshaller;
import software.amazon.awssdk.services.fis.transform.TagResourceRequestMarshaller;
import software.amazon.awssdk.services.fis.transform.UntagResourceRequestMarshaller;
import software.amazon.awssdk.services.fis.transform.UpdateExperimentTemplateRequestMarshaller;
import software.amazon.awssdk.utils.Logger;

/**
 * Internal implementation of {@link FisClient}.
 *
 * @see FisClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultFisClient implements FisClient {
    private static final Logger log = Logger.loggerFor(DefaultFisClient.class);

    private static final AwsProtocolMetadata protocolMetadata = AwsProtocolMetadata.builder()
            .serviceProtocol(AwsServiceProtocol.REST_JSON).build();

    private final SyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    private final FisServiceClientConfiguration serviceClientConfiguration;

    protected DefaultFisClient(FisServiceClientConfiguration serviceClientConfiguration,
            SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsSyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.serviceClientConfiguration = serviceClientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    /**
     * <p>
     * Creates an experiment template.
     * </p>
     * <p>
     * An experiment template includes the following components:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <b>Targets</b>: A target can be a specific resource in your Amazon Web Services environment, or one or more
     * resources that match criteria that you specify, for example, resources that have specific tags.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>Actions</b>: The actions to carry out on the target. You can specify multiple actions, the duration of each
     * action, and when to start each action during an experiment.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>Stop conditions</b>: If a stop condition is triggered while an experiment is running, the experiment is
     * automatically stopped. You can define a stop condition as a CloudWatch alarm.
     * </p>
     * </li>
     * </ul>
     * <p>
     * For more information, see <a
     * href="https://docs.aws.amazon.com/fis/latest/userguide/experiment-templates.html">Experiment templates</a> in the
     * <i>Fault Injection Simulator User Guide</i>.
     * </p>
     *
     * @param createExperimentTemplateRequest
     * @return Result of the CreateExperimentTemplate operation returned by the service.
     * @throws ValidationException
     *         The specified input is not valid, or fails to satisfy the constraints for the request.
     * @throws ConflictException
     *         The request could not be processed because of a conflict.
     * @throws ResourceNotFoundException
     *         The specified resource cannot be found.
     * @throws ServiceQuotaExceededException
     *         You have exceeded your service quota.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws FisException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample FisClient.CreateExperimentTemplate
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/fis-2020-12-01/CreateExperimentTemplate" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CreateExperimentTemplateResponse createExperimentTemplate(
            CreateExperimentTemplateRequest createExperimentTemplateRequest) throws ValidationException, ConflictException,
            ResourceNotFoundException, ServiceQuotaExceededException, AwsServiceException, SdkClientException, FisException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateExperimentTemplateResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, CreateExperimentTemplateResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createExperimentTemplateRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createExperimentTemplateRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "fis");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateExperimentTemplate");

            return clientHandler
                    .execute(new ClientExecutionParams<CreateExperimentTemplateRequest, CreateExperimentTemplateResponse>()
                            .withOperationName("CreateExperimentTemplate").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(createExperimentTemplateRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new CreateExperimentTemplateRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deletes the specified experiment template.
     * </p>
     *
     * @param deleteExperimentTemplateRequest
     * @return Result of the DeleteExperimentTemplate operation returned by the service.
     * @throws ValidationException
     *         The specified input is not valid, or fails to satisfy the constraints for the request.
     * @throws ResourceNotFoundException
     *         The specified resource cannot be found.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws FisException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample FisClient.DeleteExperimentTemplate
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/fis-2020-12-01/DeleteExperimentTemplate" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public DeleteExperimentTemplateResponse deleteExperimentTemplate(
            DeleteExperimentTemplateRequest deleteExperimentTemplateRequest) throws ValidationException,
            ResourceNotFoundException, AwsServiceException, SdkClientException, FisException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteExperimentTemplateResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DeleteExperimentTemplateResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteExperimentTemplateRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteExperimentTemplateRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "fis");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteExperimentTemplate");

            return clientHandler
                    .execute(new ClientExecutionParams<DeleteExperimentTemplateRequest, DeleteExperimentTemplateResponse>()
                            .withOperationName("DeleteExperimentTemplate").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(deleteExperimentTemplateRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new DeleteExperimentTemplateRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Gets information about the specified FIS action.
     * </p>
     *
     * @param getActionRequest
     * @return Result of the GetAction operation returned by the service.
     * @throws ValidationException
     *         The specified input is not valid, or fails to satisfy the constraints for the request.
     * @throws ResourceNotFoundException
     *         The specified resource cannot be found.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws FisException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample FisClient.GetAction
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/fis-2020-12-01/GetAction" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public GetActionResponse getAction(GetActionRequest getActionRequest) throws ValidationException, ResourceNotFoundException,
            AwsServiceException, SdkClientException, FisException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetActionResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetActionResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getActionRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getActionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "fis");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetAction");

            return clientHandler.execute(new ClientExecutionParams<GetActionRequest, GetActionResponse>()
                    .withOperationName("GetAction").withProtocolMetadata(protocolMetadata).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(getActionRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetActionRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Gets information about the specified experiment.
     * </p>
     *
     * @param getExperimentRequest
     * @return Result of the GetExperiment operation returned by the service.
     * @throws ValidationException
     *         The specified input is not valid, or fails to satisfy the constraints for the request.
     * @throws ResourceNotFoundException
     *         The specified resource cannot be found.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws FisException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample FisClient.GetExperiment
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/fis-2020-12-01/GetExperiment" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public GetExperimentResponse getExperiment(GetExperimentRequest getExperimentRequest) throws ValidationException,
            ResourceNotFoundException, AwsServiceException, SdkClientException, FisException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetExperimentResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetExperimentResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getExperimentRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getExperimentRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "fis");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetExperiment");

            return clientHandler.execute(new ClientExecutionParams<GetExperimentRequest, GetExperimentResponse>()
                    .withOperationName("GetExperiment").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(getExperimentRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetExperimentRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Gets information about the specified experiment template.
     * </p>
     *
     * @param getExperimentTemplateRequest
     * @return Result of the GetExperimentTemplate operation returned by the service.
     * @throws ValidationException
     *         The specified input is not valid, or fails to satisfy the constraints for the request.
     * @throws ResourceNotFoundException
     *         The specified resource cannot be found.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws FisException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample FisClient.GetExperimentTemplate
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/fis-2020-12-01/GetExperimentTemplate" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public GetExperimentTemplateResponse getExperimentTemplate(GetExperimentTemplateRequest getExperimentTemplateRequest)
            throws ValidationException, ResourceNotFoundException, AwsServiceException, SdkClientException, FisException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetExperimentTemplateResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, GetExperimentTemplateResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getExperimentTemplateRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getExperimentTemplateRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "fis");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetExperimentTemplate");

            return clientHandler.execute(new ClientExecutionParams<GetExperimentTemplateRequest, GetExperimentTemplateResponse>()
                    .withOperationName("GetExperimentTemplate").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(getExperimentTemplateRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetExperimentTemplateRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Gets information about the specified resource type.
     * </p>
     *
     * @param getTargetResourceTypeRequest
     * @return Result of the GetTargetResourceType operation returned by the service.
     * @throws ValidationException
     *         The specified input is not valid, or fails to satisfy the constraints for the request.
     * @throws ResourceNotFoundException
     *         The specified resource cannot be found.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws FisException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample FisClient.GetTargetResourceType
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/fis-2020-12-01/GetTargetResourceType" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public GetTargetResourceTypeResponse getTargetResourceType(GetTargetResourceTypeRequest getTargetResourceTypeRequest)
            throws ValidationException, ResourceNotFoundException, AwsServiceException, SdkClientException, FisException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetTargetResourceTypeResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, GetTargetResourceTypeResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getTargetResourceTypeRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getTargetResourceTypeRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "fis");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetTargetResourceType");

            return clientHandler.execute(new ClientExecutionParams<GetTargetResourceTypeRequest, GetTargetResourceTypeResponse>()
                    .withOperationName("GetTargetResourceType").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(getTargetResourceTypeRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetTargetResourceTypeRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Lists the available FIS actions.
     * </p>
     *
     * @param listActionsRequest
     * @return Result of the ListActions operation returned by the service.
     * @throws ValidationException
     *         The specified input is not valid, or fails to satisfy the constraints for the request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws FisException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample FisClient.ListActions
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/fis-2020-12-01/ListActions" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListActionsResponse listActions(ListActionsRequest listActionsRequest) throws ValidationException,
            AwsServiceException, SdkClientException, FisException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListActionsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListActionsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listActionsRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listActionsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "fis");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListActions");

            return clientHandler.execute(new ClientExecutionParams<ListActionsRequest, ListActionsResponse>()
                    .withOperationName("ListActions").withProtocolMetadata(protocolMetadata).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(listActionsRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListActionsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Lists your experiment templates.
     * </p>
     *
     * @param listExperimentTemplatesRequest
     * @return Result of the ListExperimentTemplates operation returned by the service.
     * @throws ValidationException
     *         The specified input is not valid, or fails to satisfy the constraints for the request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws FisException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample FisClient.ListExperimentTemplates
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/fis-2020-12-01/ListExperimentTemplates" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public ListExperimentTemplatesResponse listExperimentTemplates(ListExperimentTemplatesRequest listExperimentTemplatesRequest)
            throws ValidationException, AwsServiceException, SdkClientException, FisException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListExperimentTemplatesResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListExperimentTemplatesResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listExperimentTemplatesRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listExperimentTemplatesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "fis");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListExperimentTemplates");

            return clientHandler
                    .execute(new ClientExecutionParams<ListExperimentTemplatesRequest, ListExperimentTemplatesResponse>()
                            .withOperationName("ListExperimentTemplates").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(listExperimentTemplatesRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new ListExperimentTemplatesRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Lists your experiments.
     * </p>
     *
     * @param listExperimentsRequest
     * @return Result of the ListExperiments operation returned by the service.
     * @throws ValidationException
     *         The specified input is not valid, or fails to satisfy the constraints for the request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws FisException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample FisClient.ListExperiments
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/fis-2020-12-01/ListExperiments" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListExperimentsResponse listExperiments(ListExperimentsRequest listExperimentsRequest) throws ValidationException,
            AwsServiceException, SdkClientException, FisException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListExperimentsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListExperimentsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listExperimentsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listExperimentsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "fis");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListExperiments");

            return clientHandler.execute(new ClientExecutionParams<ListExperimentsRequest, ListExperimentsResponse>()
                    .withOperationName("ListExperiments").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(listExperimentsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListExperimentsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Lists the tags for the specified resource.
     * </p>
     *
     * @param listTagsForResourceRequest
     * @return Result of the ListTagsForResource operation returned by the service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws FisException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample FisClient.ListTagsForResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/fis-2020-12-01/ListTagsForResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListTagsForResourceResponse listTagsForResource(ListTagsForResourceRequest listTagsForResourceRequest)
            throws AwsServiceException, SdkClientException, FisException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListTagsForResourceResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListTagsForResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listTagsForResourceRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTagsForResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "fis");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTagsForResource");

            return clientHandler.execute(new ClientExecutionParams<ListTagsForResourceRequest, ListTagsForResourceResponse>()
                    .withOperationName("ListTagsForResource").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(listTagsForResourceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListTagsForResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Lists the target resource types.
     * </p>
     *
     * @param listTargetResourceTypesRequest
     * @return Result of the ListTargetResourceTypes operation returned by the service.
     * @throws ValidationException
     *         The specified input is not valid, or fails to satisfy the constraints for the request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws FisException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample FisClient.ListTargetResourceTypes
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/fis-2020-12-01/ListTargetResourceTypes" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public ListTargetResourceTypesResponse listTargetResourceTypes(ListTargetResourceTypesRequest listTargetResourceTypesRequest)
            throws ValidationException, AwsServiceException, SdkClientException, FisException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListTargetResourceTypesResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListTargetResourceTypesResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listTargetResourceTypesRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTargetResourceTypesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "fis");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTargetResourceTypes");

            return clientHandler
                    .execute(new ClientExecutionParams<ListTargetResourceTypesRequest, ListTargetResourceTypesResponse>()
                            .withOperationName("ListTargetResourceTypes").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(listTargetResourceTypesRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new ListTargetResourceTypesRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Starts running an experiment from the specified experiment template.
     * </p>
     *
     * @param startExperimentRequest
     * @return Result of the StartExperiment operation returned by the service.
     * @throws ValidationException
     *         The specified input is not valid, or fails to satisfy the constraints for the request.
     * @throws ConflictException
     *         The request could not be processed because of a conflict.
     * @throws ResourceNotFoundException
     *         The specified resource cannot be found.
     * @throws ServiceQuotaExceededException
     *         You have exceeded your service quota.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws FisException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample FisClient.StartExperiment
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/fis-2020-12-01/StartExperiment" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public StartExperimentResponse startExperiment(StartExperimentRequest startExperimentRequest) throws ValidationException,
            ConflictException, ResourceNotFoundException, ServiceQuotaExceededException, AwsServiceException, SdkClientException,
            FisException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<StartExperimentResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                StartExperimentResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(startExperimentRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, startExperimentRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "fis");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "StartExperiment");

            return clientHandler.execute(new ClientExecutionParams<StartExperimentRequest, StartExperimentResponse>()
                    .withOperationName("StartExperiment").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(startExperimentRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new StartExperimentRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Stops the specified experiment.
     * </p>
     *
     * @param stopExperimentRequest
     * @return Result of the StopExperiment operation returned by the service.
     * @throws ValidationException
     *         The specified input is not valid, or fails to satisfy the constraints for the request.
     * @throws ResourceNotFoundException
     *         The specified resource cannot be found.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws FisException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample FisClient.StopExperiment
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/fis-2020-12-01/StopExperiment" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public StopExperimentResponse stopExperiment(StopExperimentRequest stopExperimentRequest) throws ValidationException,
            ResourceNotFoundException, AwsServiceException, SdkClientException, FisException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<StopExperimentResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                StopExperimentResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(stopExperimentRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, stopExperimentRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "fis");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "StopExperiment");

            return clientHandler.execute(new ClientExecutionParams<StopExperimentRequest, StopExperimentResponse>()
                    .withOperationName("StopExperiment").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(stopExperimentRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new StopExperimentRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Applies the specified tags to the specified resource.
     * </p>
     *
     * @param tagResourceRequest
     * @return Result of the TagResource operation returned by the service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws FisException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample FisClient.TagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/fis-2020-12-01/TagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public TagResourceResponse tagResource(TagResourceRequest tagResourceRequest) throws AwsServiceException, SdkClientException,
            FisException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<TagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                TagResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(tagResourceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, tagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "fis");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "TagResource");

            return clientHandler.execute(new ClientExecutionParams<TagResourceRequest, TagResourceResponse>()
                    .withOperationName("TagResource").withProtocolMetadata(protocolMetadata).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(tagResourceRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new TagResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Removes the specified tags from the specified resource.
     * </p>
     *
     * @param untagResourceRequest
     * @return Result of the UntagResource operation returned by the service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws FisException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample FisClient.UntagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/fis-2020-12-01/UntagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public UntagResourceResponse untagResource(UntagResourceRequest untagResourceRequest) throws AwsServiceException,
            SdkClientException, FisException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UntagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                UntagResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(untagResourceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, untagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "fis");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UntagResource");

            return clientHandler.execute(new ClientExecutionParams<UntagResourceRequest, UntagResourceResponse>()
                    .withOperationName("UntagResource").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(untagResourceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UntagResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Updates the specified experiment template.
     * </p>
     *
     * @param updateExperimentTemplateRequest
     * @return Result of the UpdateExperimentTemplate operation returned by the service.
     * @throws ValidationException
     *         The specified input is not valid, or fails to satisfy the constraints for the request.
     * @throws ResourceNotFoundException
     *         The specified resource cannot be found.
     * @throws ServiceQuotaExceededException
     *         You have exceeded your service quota.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws FisException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample FisClient.UpdateExperimentTemplate
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/fis-2020-12-01/UpdateExperimentTemplate" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public UpdateExperimentTemplateResponse updateExperimentTemplate(
            UpdateExperimentTemplateRequest updateExperimentTemplateRequest) throws ValidationException,
            ResourceNotFoundException, ServiceQuotaExceededException, AwsServiceException, SdkClientException, FisException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateExperimentTemplateResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, UpdateExperimentTemplateResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateExperimentTemplateRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateExperimentTemplateRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "fis");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateExperimentTemplate");

            return clientHandler
                    .execute(new ClientExecutionParams<UpdateExperimentTemplateRequest, UpdateExperimentTemplateResponse>()
                            .withOperationName("UpdateExperimentTemplate").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(updateExperimentTemplateRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new UpdateExperimentTemplateRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    private SdkClientConfiguration updateSdkClientConfiguration(SdkRequest request, SdkClientConfiguration clientConfiguration) {
        List<SdkPlugin> plugins = request.overrideConfiguration().map(c -> c.plugins()).orElse(Collections.emptyList());
        if (plugins.isEmpty()) {
            return clientConfiguration;
        }
        FisServiceClientConfigurationBuilder.BuilderInternal serviceConfigBuilder = FisServiceClientConfigurationBuilder
                .builder(clientConfiguration.toBuilder());
        serviceConfigBuilder.overrideConfiguration(serviceClientConfiguration.overrideConfiguration());
        for (SdkPlugin plugin : plugins) {
            plugin.configureClient(serviceConfigBuilder);
        }
        return serviceConfigBuilder.buildSdkClientConfiguration();
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(FisException::builder)
                .protocol(AwsJsonProtocol.REST_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ConflictException")
                                .exceptionBuilderSupplier(ConflictException::builder).httpStatusCode(409).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceNotFoundException")
                                .exceptionBuilderSupplier(ResourceNotFoundException::builder).httpStatusCode(404).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ValidationException")
                                .exceptionBuilderSupplier(ValidationException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException")
                                .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).httpStatusCode(402).build());
    }

    @Override
    public final FisServiceClientConfiguration serviceClientConfiguration() {
        return this.serviceClientConfiguration;
    }

    @Override
    public void close() {
        clientHandler.close();
    }
}
