/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.eks.waiters;

import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ScheduledExecutorService;
import java.util.function.Consumer;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkPublicApi;
import software.amazon.awssdk.core.waiters.WaiterOverrideConfiguration;
import software.amazon.awssdk.core.waiters.WaiterResponse;
import software.amazon.awssdk.services.eks.EksAsyncClient;
import software.amazon.awssdk.services.eks.model.DescribeAddonRequest;
import software.amazon.awssdk.services.eks.model.DescribeAddonResponse;
import software.amazon.awssdk.services.eks.model.DescribeClusterRequest;
import software.amazon.awssdk.services.eks.model.DescribeClusterResponse;
import software.amazon.awssdk.services.eks.model.DescribeFargateProfileRequest;
import software.amazon.awssdk.services.eks.model.DescribeFargateProfileResponse;
import software.amazon.awssdk.services.eks.model.DescribeNodegroupRequest;
import software.amazon.awssdk.services.eks.model.DescribeNodegroupResponse;
import software.amazon.awssdk.utils.SdkAutoCloseable;

/**
 * Waiter utility class that polls a resource until a desired state is reached or until it is determined that the
 * resource will never enter into the desired state. This can be created using the static {@link #builder()} method
 */
@Generated("software.amazon.awssdk:codegen")
@SdkPublicApi
public interface EksAsyncWaiter extends SdkAutoCloseable {
    /**
     * Polls {@link EksAsyncClient#describeAddon} API until the desired condition {@code AddonActive} is met, or until
     * it is determined that the resource will never enter into the desired state
     *
     * @param describeAddonRequest
     *        the request to be used for polling
     * @return CompletableFuture containing the WaiterResponse. It completes successfully when the resource enters into
     *         a desired state or exceptionally when it is determined that the resource will never enter into the
     *         desired state.
     */
    default CompletableFuture<WaiterResponse<DescribeAddonResponse>> waitUntilAddonActive(
            DescribeAddonRequest describeAddonRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * Polls {@link EksAsyncClient#describeAddon} API until the desired condition {@code AddonActive} is met, or until
     * it is determined that the resource will never enter into the desired state.
     * <p>
     * This is a convenience method to create an instance of the request builder without the need to create one manually
     * using {@link DescribeAddonRequest#builder()}
     *
     * @param describeAddonRequest
     *        The consumer that will configure the request to be used for polling
     * @return CompletableFuture of the WaiterResponse containing either a response or an exception that has matched
     *         with the waiter success condition
     */
    default CompletableFuture<WaiterResponse<DescribeAddonResponse>> waitUntilAddonActive(
            Consumer<DescribeAddonRequest.Builder> describeAddonRequest) {
        return waitUntilAddonActive(DescribeAddonRequest.builder().applyMutation(describeAddonRequest).build());
    }

    /**
     * Polls {@link EksAsyncClient#describeAddon} API until the desired condition {@code AddonActive} is met, or until
     * it is determined that the resource will never enter into the desired state
     *
     * @param describeAddonRequest
     *        The request to be used for polling
     * @param overrideConfig
     *        Per request override configuration for waiters
     * @return WaiterResponse containing either a response or an exception that has matched with the waiter success
     *         condition
     */
    default CompletableFuture<WaiterResponse<DescribeAddonResponse>> waitUntilAddonActive(
            DescribeAddonRequest describeAddonRequest, WaiterOverrideConfiguration overrideConfig) {
        throw new UnsupportedOperationException();
    }

    /**
     * Polls {@link EksAsyncClient#describeAddon} API until the desired condition {@code AddonActive} is met, or until
     * it is determined that the resource will never enter into the desired state.
     * <p>
     * This is a convenience method to create an instance of the request builder and instance of the override config
     * builder
     *
     * @param describeAddonRequest
     *        The consumer that will configure the request to be used for polling
     * @param overrideConfig
     *        The consumer that will configure the per request override configuration for waiters
     * @return WaiterResponse containing either a response or an exception that has matched with the waiter success
     *         condition
     */
    default CompletableFuture<WaiterResponse<DescribeAddonResponse>> waitUntilAddonActive(
            Consumer<DescribeAddonRequest.Builder> describeAddonRequest,
            Consumer<WaiterOverrideConfiguration.Builder> overrideConfig) {
        return waitUntilAddonActive(DescribeAddonRequest.builder().applyMutation(describeAddonRequest).build(),
                WaiterOverrideConfiguration.builder().applyMutation(overrideConfig).build());
    }

    /**
     * Polls {@link EksAsyncClient#describeAddon} API until the desired condition {@code AddonDeleted} is met, or until
     * it is determined that the resource will never enter into the desired state
     *
     * @param describeAddonRequest
     *        the request to be used for polling
     * @return CompletableFuture containing the WaiterResponse. It completes successfully when the resource enters into
     *         a desired state or exceptionally when it is determined that the resource will never enter into the
     *         desired state.
     */
    default CompletableFuture<WaiterResponse<DescribeAddonResponse>> waitUntilAddonDeleted(
            DescribeAddonRequest describeAddonRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * Polls {@link EksAsyncClient#describeAddon} API until the desired condition {@code AddonDeleted} is met, or until
     * it is determined that the resource will never enter into the desired state.
     * <p>
     * This is a convenience method to create an instance of the request builder without the need to create one manually
     * using {@link DescribeAddonRequest#builder()}
     *
     * @param describeAddonRequest
     *        The consumer that will configure the request to be used for polling
     * @return CompletableFuture of the WaiterResponse containing either a response or an exception that has matched
     *         with the waiter success condition
     */
    default CompletableFuture<WaiterResponse<DescribeAddonResponse>> waitUntilAddonDeleted(
            Consumer<DescribeAddonRequest.Builder> describeAddonRequest) {
        return waitUntilAddonDeleted(DescribeAddonRequest.builder().applyMutation(describeAddonRequest).build());
    }

    /**
     * Polls {@link EksAsyncClient#describeAddon} API until the desired condition {@code AddonDeleted} is met, or until
     * it is determined that the resource will never enter into the desired state
     *
     * @param describeAddonRequest
     *        The request to be used for polling
     * @param overrideConfig
     *        Per request override configuration for waiters
     * @return WaiterResponse containing either a response or an exception that has matched with the waiter success
     *         condition
     */
    default CompletableFuture<WaiterResponse<DescribeAddonResponse>> waitUntilAddonDeleted(
            DescribeAddonRequest describeAddonRequest, WaiterOverrideConfiguration overrideConfig) {
        throw new UnsupportedOperationException();
    }

    /**
     * Polls {@link EksAsyncClient#describeAddon} API until the desired condition {@code AddonDeleted} is met, or until
     * it is determined that the resource will never enter into the desired state.
     * <p>
     * This is a convenience method to create an instance of the request builder and instance of the override config
     * builder
     *
     * @param describeAddonRequest
     *        The consumer that will configure the request to be used for polling
     * @param overrideConfig
     *        The consumer that will configure the per request override configuration for waiters
     * @return WaiterResponse containing either a response or an exception that has matched with the waiter success
     *         condition
     */
    default CompletableFuture<WaiterResponse<DescribeAddonResponse>> waitUntilAddonDeleted(
            Consumer<DescribeAddonRequest.Builder> describeAddonRequest,
            Consumer<WaiterOverrideConfiguration.Builder> overrideConfig) {
        return waitUntilAddonDeleted(DescribeAddonRequest.builder().applyMutation(describeAddonRequest).build(),
                WaiterOverrideConfiguration.builder().applyMutation(overrideConfig).build());
    }

    /**
     * Polls {@link EksAsyncClient#describeCluster} API until the desired condition {@code ClusterActive} is met, or
     * until it is determined that the resource will never enter into the desired state
     *
     * @param describeClusterRequest
     *        the request to be used for polling
     * @return CompletableFuture containing the WaiterResponse. It completes successfully when the resource enters into
     *         a desired state or exceptionally when it is determined that the resource will never enter into the
     *         desired state.
     */
    default CompletableFuture<WaiterResponse<DescribeClusterResponse>> waitUntilClusterActive(
            DescribeClusterRequest describeClusterRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * Polls {@link EksAsyncClient#describeCluster} API until the desired condition {@code ClusterActive} is met, or
     * until it is determined that the resource will never enter into the desired state.
     * <p>
     * This is a convenience method to create an instance of the request builder without the need to create one manually
     * using {@link DescribeClusterRequest#builder()}
     *
     * @param describeClusterRequest
     *        The consumer that will configure the request to be used for polling
     * @return CompletableFuture of the WaiterResponse containing either a response or an exception that has matched
     *         with the waiter success condition
     */
    default CompletableFuture<WaiterResponse<DescribeClusterResponse>> waitUntilClusterActive(
            Consumer<DescribeClusterRequest.Builder> describeClusterRequest) {
        return waitUntilClusterActive(DescribeClusterRequest.builder().applyMutation(describeClusterRequest).build());
    }

    /**
     * Polls {@link EksAsyncClient#describeCluster} API until the desired condition {@code ClusterActive} is met, or
     * until it is determined that the resource will never enter into the desired state
     *
     * @param describeClusterRequest
     *        The request to be used for polling
     * @param overrideConfig
     *        Per request override configuration for waiters
     * @return WaiterResponse containing either a response or an exception that has matched with the waiter success
     *         condition
     */
    default CompletableFuture<WaiterResponse<DescribeClusterResponse>> waitUntilClusterActive(
            DescribeClusterRequest describeClusterRequest, WaiterOverrideConfiguration overrideConfig) {
        throw new UnsupportedOperationException();
    }

    /**
     * Polls {@link EksAsyncClient#describeCluster} API until the desired condition {@code ClusterActive} is met, or
     * until it is determined that the resource will never enter into the desired state.
     * <p>
     * This is a convenience method to create an instance of the request builder and instance of the override config
     * builder
     *
     * @param describeClusterRequest
     *        The consumer that will configure the request to be used for polling
     * @param overrideConfig
     *        The consumer that will configure the per request override configuration for waiters
     * @return WaiterResponse containing either a response or an exception that has matched with the waiter success
     *         condition
     */
    default CompletableFuture<WaiterResponse<DescribeClusterResponse>> waitUntilClusterActive(
            Consumer<DescribeClusterRequest.Builder> describeClusterRequest,
            Consumer<WaiterOverrideConfiguration.Builder> overrideConfig) {
        return waitUntilClusterActive(DescribeClusterRequest.builder().applyMutation(describeClusterRequest).build(),
                WaiterOverrideConfiguration.builder().applyMutation(overrideConfig).build());
    }

    /**
     * Polls {@link EksAsyncClient#describeCluster} API until the desired condition {@code ClusterDeleted} is met, or
     * until it is determined that the resource will never enter into the desired state
     *
     * @param describeClusterRequest
     *        the request to be used for polling
     * @return CompletableFuture containing the WaiterResponse. It completes successfully when the resource enters into
     *         a desired state or exceptionally when it is determined that the resource will never enter into the
     *         desired state.
     */
    default CompletableFuture<WaiterResponse<DescribeClusterResponse>> waitUntilClusterDeleted(
            DescribeClusterRequest describeClusterRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * Polls {@link EksAsyncClient#describeCluster} API until the desired condition {@code ClusterDeleted} is met, or
     * until it is determined that the resource will never enter into the desired state.
     * <p>
     * This is a convenience method to create an instance of the request builder without the need to create one manually
     * using {@link DescribeClusterRequest#builder()}
     *
     * @param describeClusterRequest
     *        The consumer that will configure the request to be used for polling
     * @return CompletableFuture of the WaiterResponse containing either a response or an exception that has matched
     *         with the waiter success condition
     */
    default CompletableFuture<WaiterResponse<DescribeClusterResponse>> waitUntilClusterDeleted(
            Consumer<DescribeClusterRequest.Builder> describeClusterRequest) {
        return waitUntilClusterDeleted(DescribeClusterRequest.builder().applyMutation(describeClusterRequest).build());
    }

    /**
     * Polls {@link EksAsyncClient#describeCluster} API until the desired condition {@code ClusterDeleted} is met, or
     * until it is determined that the resource will never enter into the desired state
     *
     * @param describeClusterRequest
     *        The request to be used for polling
     * @param overrideConfig
     *        Per request override configuration for waiters
     * @return WaiterResponse containing either a response or an exception that has matched with the waiter success
     *         condition
     */
    default CompletableFuture<WaiterResponse<DescribeClusterResponse>> waitUntilClusterDeleted(
            DescribeClusterRequest describeClusterRequest, WaiterOverrideConfiguration overrideConfig) {
        throw new UnsupportedOperationException();
    }

    /**
     * Polls {@link EksAsyncClient#describeCluster} API until the desired condition {@code ClusterDeleted} is met, or
     * until it is determined that the resource will never enter into the desired state.
     * <p>
     * This is a convenience method to create an instance of the request builder and instance of the override config
     * builder
     *
     * @param describeClusterRequest
     *        The consumer that will configure the request to be used for polling
     * @param overrideConfig
     *        The consumer that will configure the per request override configuration for waiters
     * @return WaiterResponse containing either a response or an exception that has matched with the waiter success
     *         condition
     */
    default CompletableFuture<WaiterResponse<DescribeClusterResponse>> waitUntilClusterDeleted(
            Consumer<DescribeClusterRequest.Builder> describeClusterRequest,
            Consumer<WaiterOverrideConfiguration.Builder> overrideConfig) {
        return waitUntilClusterDeleted(DescribeClusterRequest.builder().applyMutation(describeClusterRequest).build(),
                WaiterOverrideConfiguration.builder().applyMutation(overrideConfig).build());
    }

    /**
     * Polls {@link EksAsyncClient#describeFargateProfile} API until the desired condition {@code FargateProfileActive}
     * is met, or until it is determined that the resource will never enter into the desired state
     *
     * @param describeFargateProfileRequest
     *        the request to be used for polling
     * @return CompletableFuture containing the WaiterResponse. It completes successfully when the resource enters into
     *         a desired state or exceptionally when it is determined that the resource will never enter into the
     *         desired state.
     */
    default CompletableFuture<WaiterResponse<DescribeFargateProfileResponse>> waitUntilFargateProfileActive(
            DescribeFargateProfileRequest describeFargateProfileRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * Polls {@link EksAsyncClient#describeFargateProfile} API until the desired condition {@code FargateProfileActive}
     * is met, or until it is determined that the resource will never enter into the desired state.
     * <p>
     * This is a convenience method to create an instance of the request builder without the need to create one manually
     * using {@link DescribeFargateProfileRequest#builder()}
     *
     * @param describeFargateProfileRequest
     *        The consumer that will configure the request to be used for polling
     * @return CompletableFuture of the WaiterResponse containing either a response or an exception that has matched
     *         with the waiter success condition
     */
    default CompletableFuture<WaiterResponse<DescribeFargateProfileResponse>> waitUntilFargateProfileActive(
            Consumer<DescribeFargateProfileRequest.Builder> describeFargateProfileRequest) {
        return waitUntilFargateProfileActive(DescribeFargateProfileRequest.builder().applyMutation(describeFargateProfileRequest)
                .build());
    }

    /**
     * Polls {@link EksAsyncClient#describeFargateProfile} API until the desired condition {@code FargateProfileActive}
     * is met, or until it is determined that the resource will never enter into the desired state
     *
     * @param describeFargateProfileRequest
     *        The request to be used for polling
     * @param overrideConfig
     *        Per request override configuration for waiters
     * @return WaiterResponse containing either a response or an exception that has matched with the waiter success
     *         condition
     */
    default CompletableFuture<WaiterResponse<DescribeFargateProfileResponse>> waitUntilFargateProfileActive(
            DescribeFargateProfileRequest describeFargateProfileRequest, WaiterOverrideConfiguration overrideConfig) {
        throw new UnsupportedOperationException();
    }

    /**
     * Polls {@link EksAsyncClient#describeFargateProfile} API until the desired condition {@code FargateProfileActive}
     * is met, or until it is determined that the resource will never enter into the desired state.
     * <p>
     * This is a convenience method to create an instance of the request builder and instance of the override config
     * builder
     *
     * @param describeFargateProfileRequest
     *        The consumer that will configure the request to be used for polling
     * @param overrideConfig
     *        The consumer that will configure the per request override configuration for waiters
     * @return WaiterResponse containing either a response or an exception that has matched with the waiter success
     *         condition
     */
    default CompletableFuture<WaiterResponse<DescribeFargateProfileResponse>> waitUntilFargateProfileActive(
            Consumer<DescribeFargateProfileRequest.Builder> describeFargateProfileRequest,
            Consumer<WaiterOverrideConfiguration.Builder> overrideConfig) {
        return waitUntilFargateProfileActive(DescribeFargateProfileRequest.builder().applyMutation(describeFargateProfileRequest)
                .build(), WaiterOverrideConfiguration.builder().applyMutation(overrideConfig).build());
    }

    /**
     * Polls {@link EksAsyncClient#describeFargateProfile} API until the desired condition {@code FargateProfileDeleted}
     * is met, or until it is determined that the resource will never enter into the desired state
     *
     * @param describeFargateProfileRequest
     *        the request to be used for polling
     * @return CompletableFuture containing the WaiterResponse. It completes successfully when the resource enters into
     *         a desired state or exceptionally when it is determined that the resource will never enter into the
     *         desired state.
     */
    default CompletableFuture<WaiterResponse<DescribeFargateProfileResponse>> waitUntilFargateProfileDeleted(
            DescribeFargateProfileRequest describeFargateProfileRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * Polls {@link EksAsyncClient#describeFargateProfile} API until the desired condition {@code FargateProfileDeleted}
     * is met, or until it is determined that the resource will never enter into the desired state.
     * <p>
     * This is a convenience method to create an instance of the request builder without the need to create one manually
     * using {@link DescribeFargateProfileRequest#builder()}
     *
     * @param describeFargateProfileRequest
     *        The consumer that will configure the request to be used for polling
     * @return CompletableFuture of the WaiterResponse containing either a response or an exception that has matched
     *         with the waiter success condition
     */
    default CompletableFuture<WaiterResponse<DescribeFargateProfileResponse>> waitUntilFargateProfileDeleted(
            Consumer<DescribeFargateProfileRequest.Builder> describeFargateProfileRequest) {
        return waitUntilFargateProfileDeleted(DescribeFargateProfileRequest.builder()
                .applyMutation(describeFargateProfileRequest).build());
    }

    /**
     * Polls {@link EksAsyncClient#describeFargateProfile} API until the desired condition {@code FargateProfileDeleted}
     * is met, or until it is determined that the resource will never enter into the desired state
     *
     * @param describeFargateProfileRequest
     *        The request to be used for polling
     * @param overrideConfig
     *        Per request override configuration for waiters
     * @return WaiterResponse containing either a response or an exception that has matched with the waiter success
     *         condition
     */
    default CompletableFuture<WaiterResponse<DescribeFargateProfileResponse>> waitUntilFargateProfileDeleted(
            DescribeFargateProfileRequest describeFargateProfileRequest, WaiterOverrideConfiguration overrideConfig) {
        throw new UnsupportedOperationException();
    }

    /**
     * Polls {@link EksAsyncClient#describeFargateProfile} API until the desired condition {@code FargateProfileDeleted}
     * is met, or until it is determined that the resource will never enter into the desired state.
     * <p>
     * This is a convenience method to create an instance of the request builder and instance of the override config
     * builder
     *
     * @param describeFargateProfileRequest
     *        The consumer that will configure the request to be used for polling
     * @param overrideConfig
     *        The consumer that will configure the per request override configuration for waiters
     * @return WaiterResponse containing either a response or an exception that has matched with the waiter success
     *         condition
     */
    default CompletableFuture<WaiterResponse<DescribeFargateProfileResponse>> waitUntilFargateProfileDeleted(
            Consumer<DescribeFargateProfileRequest.Builder> describeFargateProfileRequest,
            Consumer<WaiterOverrideConfiguration.Builder> overrideConfig) {
        return waitUntilFargateProfileDeleted(DescribeFargateProfileRequest.builder()
                .applyMutation(describeFargateProfileRequest).build(),
                WaiterOverrideConfiguration.builder().applyMutation(overrideConfig).build());
    }

    /**
     * Polls {@link EksAsyncClient#describeNodegroup} API until the desired condition {@code NodegroupActive} is met, or
     * until it is determined that the resource will never enter into the desired state
     *
     * @param describeNodegroupRequest
     *        the request to be used for polling
     * @return CompletableFuture containing the WaiterResponse. It completes successfully when the resource enters into
     *         a desired state or exceptionally when it is determined that the resource will never enter into the
     *         desired state.
     */
    default CompletableFuture<WaiterResponse<DescribeNodegroupResponse>> waitUntilNodegroupActive(
            DescribeNodegroupRequest describeNodegroupRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * Polls {@link EksAsyncClient#describeNodegroup} API until the desired condition {@code NodegroupActive} is met, or
     * until it is determined that the resource will never enter into the desired state.
     * <p>
     * This is a convenience method to create an instance of the request builder without the need to create one manually
     * using {@link DescribeNodegroupRequest#builder()}
     *
     * @param describeNodegroupRequest
     *        The consumer that will configure the request to be used for polling
     * @return CompletableFuture of the WaiterResponse containing either a response or an exception that has matched
     *         with the waiter success condition
     */
    default CompletableFuture<WaiterResponse<DescribeNodegroupResponse>> waitUntilNodegroupActive(
            Consumer<DescribeNodegroupRequest.Builder> describeNodegroupRequest) {
        return waitUntilNodegroupActive(DescribeNodegroupRequest.builder().applyMutation(describeNodegroupRequest).build());
    }

    /**
     * Polls {@link EksAsyncClient#describeNodegroup} API until the desired condition {@code NodegroupActive} is met, or
     * until it is determined that the resource will never enter into the desired state
     *
     * @param describeNodegroupRequest
     *        The request to be used for polling
     * @param overrideConfig
     *        Per request override configuration for waiters
     * @return WaiterResponse containing either a response or an exception that has matched with the waiter success
     *         condition
     */
    default CompletableFuture<WaiterResponse<DescribeNodegroupResponse>> waitUntilNodegroupActive(
            DescribeNodegroupRequest describeNodegroupRequest, WaiterOverrideConfiguration overrideConfig) {
        throw new UnsupportedOperationException();
    }

    /**
     * Polls {@link EksAsyncClient#describeNodegroup} API until the desired condition {@code NodegroupActive} is met, or
     * until it is determined that the resource will never enter into the desired state.
     * <p>
     * This is a convenience method to create an instance of the request builder and instance of the override config
     * builder
     *
     * @param describeNodegroupRequest
     *        The consumer that will configure the request to be used for polling
     * @param overrideConfig
     *        The consumer that will configure the per request override configuration for waiters
     * @return WaiterResponse containing either a response or an exception that has matched with the waiter success
     *         condition
     */
    default CompletableFuture<WaiterResponse<DescribeNodegroupResponse>> waitUntilNodegroupActive(
            Consumer<DescribeNodegroupRequest.Builder> describeNodegroupRequest,
            Consumer<WaiterOverrideConfiguration.Builder> overrideConfig) {
        return waitUntilNodegroupActive(DescribeNodegroupRequest.builder().applyMutation(describeNodegroupRequest).build(),
                WaiterOverrideConfiguration.builder().applyMutation(overrideConfig).build());
    }

    /**
     * Polls {@link EksAsyncClient#describeNodegroup} API until the desired condition {@code NodegroupDeleted} is met,
     * or until it is determined that the resource will never enter into the desired state
     *
     * @param describeNodegroupRequest
     *        the request to be used for polling
     * @return CompletableFuture containing the WaiterResponse. It completes successfully when the resource enters into
     *         a desired state or exceptionally when it is determined that the resource will never enter into the
     *         desired state.
     */
    default CompletableFuture<WaiterResponse<DescribeNodegroupResponse>> waitUntilNodegroupDeleted(
            DescribeNodegroupRequest describeNodegroupRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * Polls {@link EksAsyncClient#describeNodegroup} API until the desired condition {@code NodegroupDeleted} is met,
     * or until it is determined that the resource will never enter into the desired state.
     * <p>
     * This is a convenience method to create an instance of the request builder without the need to create one manually
     * using {@link DescribeNodegroupRequest#builder()}
     *
     * @param describeNodegroupRequest
     *        The consumer that will configure the request to be used for polling
     * @return CompletableFuture of the WaiterResponse containing either a response or an exception that has matched
     *         with the waiter success condition
     */
    default CompletableFuture<WaiterResponse<DescribeNodegroupResponse>> waitUntilNodegroupDeleted(
            Consumer<DescribeNodegroupRequest.Builder> describeNodegroupRequest) {
        return waitUntilNodegroupDeleted(DescribeNodegroupRequest.builder().applyMutation(describeNodegroupRequest).build());
    }

    /**
     * Polls {@link EksAsyncClient#describeNodegroup} API until the desired condition {@code NodegroupDeleted} is met,
     * or until it is determined that the resource will never enter into the desired state
     *
     * @param describeNodegroupRequest
     *        The request to be used for polling
     * @param overrideConfig
     *        Per request override configuration for waiters
     * @return WaiterResponse containing either a response or an exception that has matched with the waiter success
     *         condition
     */
    default CompletableFuture<WaiterResponse<DescribeNodegroupResponse>> waitUntilNodegroupDeleted(
            DescribeNodegroupRequest describeNodegroupRequest, WaiterOverrideConfiguration overrideConfig) {
        throw new UnsupportedOperationException();
    }

    /**
     * Polls {@link EksAsyncClient#describeNodegroup} API until the desired condition {@code NodegroupDeleted} is met,
     * or until it is determined that the resource will never enter into the desired state.
     * <p>
     * This is a convenience method to create an instance of the request builder and instance of the override config
     * builder
     *
     * @param describeNodegroupRequest
     *        The consumer that will configure the request to be used for polling
     * @param overrideConfig
     *        The consumer that will configure the per request override configuration for waiters
     * @return WaiterResponse containing either a response or an exception that has matched with the waiter success
     *         condition
     */
    default CompletableFuture<WaiterResponse<DescribeNodegroupResponse>> waitUntilNodegroupDeleted(
            Consumer<DescribeNodegroupRequest.Builder> describeNodegroupRequest,
            Consumer<WaiterOverrideConfiguration.Builder> overrideConfig) {
        return waitUntilNodegroupDeleted(DescribeNodegroupRequest.builder().applyMutation(describeNodegroupRequest).build(),
                WaiterOverrideConfiguration.builder().applyMutation(overrideConfig).build());
    }

    /**
     * Create a builder that can be used to configure and create a {@link EksAsyncWaiter}.
     *
     * @return a builder
     */
    static Builder builder() {
        return DefaultEksAsyncWaiter.builder();
    }

    /**
     * Create an instance of {@link EksAsyncWaiter} with the default configuration.
     * <p>
     * <b>A default {@link EksAsyncClient} will be created to poll resources. It is recommended to share a single
     * instance of the waiter created via this method. If it is not desirable to share a waiter instance, invoke
     * {@link #close()} to release the resources once the waiter is not needed.</b>
     *
     * @return an instance of {@link EksAsyncWaiter}
     */
    static EksAsyncWaiter create() {
        return DefaultEksAsyncWaiter.builder().build();
    }

    interface Builder {
        /**
         * Sets a custom {@link ScheduledExecutorService} that will be used to schedule async polling attempts
         * <p>
         * This executorService must be closed by the caller when it is ready to be disposed. The SDK will not close the
         * executorService when the waiter is closed
         *
         * @param executorService
         *        the executorService to set
         * @return a reference to this object so that method calls can be chained together.
         */
        Builder scheduledExecutorService(ScheduledExecutorService executorService);

        /**
         * Defines overrides to the default SDK waiter configuration that should be used for waiters created from this
         * builder
         *
         * @param overrideConfiguration
         *        the override configuration to set
         * @return a reference to this object so that method calls can be chained together.
         */
        Builder overrideConfiguration(WaiterOverrideConfiguration overrideConfiguration);

        /**
         * This is a convenient method to pass the override configuration without the need to create an instance
         * manually via {@link WaiterOverrideConfiguration#builder()}
         *
         * @param overrideConfiguration
         *        The consumer that will configure the overrideConfiguration
         * @return a reference to this object so that method calls can be chained together.
         * @see #overrideConfiguration(WaiterOverrideConfiguration)
         */
        default Builder overrideConfiguration(Consumer<WaiterOverrideConfiguration.Builder> overrideConfiguration) {
            WaiterOverrideConfiguration.Builder builder = WaiterOverrideConfiguration.builder();
            overrideConfiguration.accept(builder);
            return overrideConfiguration(builder.build());
        }

        /**
         * Sets a custom {@link EksAsyncClient} that will be used to poll the resource
         * <p>
         * This SDK client must be closed by the caller when it is ready to be disposed. The SDK will not close the
         * client when the waiter is closed
         *
         * @param client
         *        the client to send the request
         * @return a reference to this object so that method calls can be chained together.
         */
        Builder client(EksAsyncClient client);

        /**
         * Builds an instance of {@link EksAsyncWaiter} based on the configurations supplied to this builder
         *
         * @return An initialized {@link EksAsyncWaiter}
         */
        EksAsyncWaiter build();
    }
}
