/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.ec2.model;

import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 */
@Generated("software.amazon.awssdk:codegen")
public final class CreateFlowLogsRequest extends Ec2Request implements
        ToCopyableBuilder<CreateFlowLogsRequest.Builder, CreateFlowLogsRequest> {
    private static final SdkField<String> CLIENT_TOKEN_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("ClientToken")
            .getter(getter(CreateFlowLogsRequest::clientToken))
            .setter(setter(Builder::clientToken))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ClientToken")
                    .unmarshallLocationName("ClientToken").build()).build();

    private static final SdkField<String> DELIVER_LOGS_PERMISSION_ARN_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("DeliverLogsPermissionArn")
            .getter(getter(CreateFlowLogsRequest::deliverLogsPermissionArn))
            .setter(setter(Builder::deliverLogsPermissionArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DeliverLogsPermissionArn")
                    .unmarshallLocationName("DeliverLogsPermissionArn").build()).build();

    private static final SdkField<String> DELIVER_CROSS_ACCOUNT_ROLE_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("DeliverCrossAccountRole")
            .getter(getter(CreateFlowLogsRequest::deliverCrossAccountRole))
            .setter(setter(Builder::deliverCrossAccountRole))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DeliverCrossAccountRole")
                    .unmarshallLocationName("DeliverCrossAccountRole").build()).build();

    private static final SdkField<String> LOG_GROUP_NAME_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("LogGroupName")
            .getter(getter(CreateFlowLogsRequest::logGroupName))
            .setter(setter(Builder::logGroupName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("LogGroupName")
                    .unmarshallLocationName("LogGroupName").build()).build();

    private static final SdkField<List<String>> RESOURCE_IDS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("ResourceIds")
            .getter(getter(CreateFlowLogsRequest::resourceIds))
            .setter(setter(Builder::resourceIds))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ResourceId")
                    .unmarshallLocationName("ResourceId").build(),
                    ListTrait
                            .builder()
                            .memberLocationName("item")
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("Item").unmarshallLocationName("item").build()).build())
                            .build()).build();

    private static final SdkField<String> RESOURCE_TYPE_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("ResourceType")
            .getter(getter(CreateFlowLogsRequest::resourceTypeAsString))
            .setter(setter(Builder::resourceType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ResourceType")
                    .unmarshallLocationName("ResourceType").build()).build();

    private static final SdkField<String> TRAFFIC_TYPE_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("TrafficType")
            .getter(getter(CreateFlowLogsRequest::trafficTypeAsString))
            .setter(setter(Builder::trafficType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("TrafficType")
                    .unmarshallLocationName("TrafficType").build()).build();

    private static final SdkField<String> LOG_DESTINATION_TYPE_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("LogDestinationType")
            .getter(getter(CreateFlowLogsRequest::logDestinationTypeAsString))
            .setter(setter(Builder::logDestinationType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("LogDestinationType")
                    .unmarshallLocationName("LogDestinationType").build()).build();

    private static final SdkField<String> LOG_DESTINATION_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("LogDestination")
            .getter(getter(CreateFlowLogsRequest::logDestination))
            .setter(setter(Builder::logDestination))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("LogDestination")
                    .unmarshallLocationName("LogDestination").build()).build();

    private static final SdkField<String> LOG_FORMAT_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("LogFormat")
            .getter(getter(CreateFlowLogsRequest::logFormat))
            .setter(setter(Builder::logFormat))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("LogFormat")
                    .unmarshallLocationName("LogFormat").build()).build();

    private static final SdkField<List<TagSpecification>> TAG_SPECIFICATIONS_FIELD = SdkField
            .<List<TagSpecification>> builder(MarshallingType.LIST)
            .memberName("TagSpecifications")
            .getter(getter(CreateFlowLogsRequest::tagSpecifications))
            .setter(setter(Builder::tagSpecifications))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("TagSpecification")
                    .unmarshallLocationName("TagSpecification").build(),
                    ListTrait
                            .builder()
                            .memberLocationName("item")
                            .memberFieldInfo(
                                    SdkField.<TagSpecification> builder(MarshallingType.SDK_POJO)
                                            .constructor(TagSpecification::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("Item").unmarshallLocationName("item").build()).build())
                            .build()).build();

    private static final SdkField<Integer> MAX_AGGREGATION_INTERVAL_FIELD = SdkField
            .<Integer> builder(MarshallingType.INTEGER)
            .memberName("MaxAggregationInterval")
            .getter(getter(CreateFlowLogsRequest::maxAggregationInterval))
            .setter(setter(Builder::maxAggregationInterval))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("MaxAggregationInterval")
                    .unmarshallLocationName("MaxAggregationInterval").build()).build();

    private static final SdkField<DestinationOptionsRequest> DESTINATION_OPTIONS_FIELD = SdkField
            .<DestinationOptionsRequest> builder(MarshallingType.SDK_POJO)
            .memberName("DestinationOptions")
            .getter(getter(CreateFlowLogsRequest::destinationOptions))
            .setter(setter(Builder::destinationOptions))
            .constructor(DestinationOptionsRequest::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DestinationOptions")
                    .unmarshallLocationName("DestinationOptions").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(CLIENT_TOKEN_FIELD,
            DELIVER_LOGS_PERMISSION_ARN_FIELD, DELIVER_CROSS_ACCOUNT_ROLE_FIELD, LOG_GROUP_NAME_FIELD, RESOURCE_IDS_FIELD,
            RESOURCE_TYPE_FIELD, TRAFFIC_TYPE_FIELD, LOG_DESTINATION_TYPE_FIELD, LOG_DESTINATION_FIELD, LOG_FORMAT_FIELD,
            TAG_SPECIFICATIONS_FIELD, MAX_AGGREGATION_INTERVAL_FIELD, DESTINATION_OPTIONS_FIELD));

    private final String clientToken;

    private final String deliverLogsPermissionArn;

    private final String deliverCrossAccountRole;

    private final String logGroupName;

    private final List<String> resourceIds;

    private final String resourceType;

    private final String trafficType;

    private final String logDestinationType;

    private final String logDestination;

    private final String logFormat;

    private final List<TagSpecification> tagSpecifications;

    private final Integer maxAggregationInterval;

    private final DestinationOptionsRequest destinationOptions;

    private CreateFlowLogsRequest(BuilderImpl builder) {
        super(builder);
        this.clientToken = builder.clientToken;
        this.deliverLogsPermissionArn = builder.deliverLogsPermissionArn;
        this.deliverCrossAccountRole = builder.deliverCrossAccountRole;
        this.logGroupName = builder.logGroupName;
        this.resourceIds = builder.resourceIds;
        this.resourceType = builder.resourceType;
        this.trafficType = builder.trafficType;
        this.logDestinationType = builder.logDestinationType;
        this.logDestination = builder.logDestination;
        this.logFormat = builder.logFormat;
        this.tagSpecifications = builder.tagSpecifications;
        this.maxAggregationInterval = builder.maxAggregationInterval;
        this.destinationOptions = builder.destinationOptions;
    }

    /**
     * <p>
     * Unique, case-sensitive identifier that you provide to ensure the idempotency of the request. For more
     * information, see <a href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/Run_Instance_Idempotency.html">How
     * to ensure idempotency</a>.
     * </p>
     * 
     * @return Unique, case-sensitive identifier that you provide to ensure the idempotency of the request. For more
     *         information, see <a
     *         href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/Run_Instance_Idempotency.html">How to ensure
     *         idempotency</a>.
     */
    public final String clientToken() {
        return clientToken;
    }

    /**
     * <p>
     * The ARN of the IAM role that allows Amazon EC2 to publish flow logs to a CloudWatch Logs log group in your
     * account.
     * </p>
     * <p>
     * This parameter is required if the destination type is <code>cloud-watch-logs</code> and unsupported otherwise.
     * </p>
     * 
     * @return The ARN of the IAM role that allows Amazon EC2 to publish flow logs to a CloudWatch Logs log group in
     *         your account.</p>
     *         <p>
     *         This parameter is required if the destination type is <code>cloud-watch-logs</code> and unsupported
     *         otherwise.
     */
    public final String deliverLogsPermissionArn() {
        return deliverLogsPermissionArn;
    }

    /**
     * <p>
     * The ARN of the IAM role that allows Amazon EC2 to publish flow logs across accounts.
     * </p>
     * 
     * @return The ARN of the IAM role that allows Amazon EC2 to publish flow logs across accounts.
     */
    public final String deliverCrossAccountRole() {
        return deliverCrossAccountRole;
    }

    /**
     * <p>
     * The name of a new or existing CloudWatch Logs log group where Amazon EC2 publishes your flow logs.
     * </p>
     * <p>
     * This parameter is valid only if the destination type is <code>cloud-watch-logs</code>.
     * </p>
     * 
     * @return The name of a new or existing CloudWatch Logs log group where Amazon EC2 publishes your flow logs.</p>
     *         <p>
     *         This parameter is valid only if the destination type is <code>cloud-watch-logs</code>.
     */
    public final String logGroupName() {
        return logGroupName;
    }

    /**
     * For responses, this returns true if the service returned a value for the ResourceIds property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasResourceIds() {
        return resourceIds != null && !(resourceIds instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The IDs of the resources to monitor. For example, if the resource type is <code>VPC</code>, specify the IDs of
     * the VPCs.
     * </p>
     * <p>
     * Constraints: Maximum of 1000 resources
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasResourceIds} method.
     * </p>
     * 
     * @return The IDs of the resources to monitor. For example, if the resource type is <code>VPC</code>, specify the
     *         IDs of the VPCs.</p>
     *         <p>
     *         Constraints: Maximum of 1000 resources
     */
    public final List<String> resourceIds() {
        return resourceIds;
    }

    /**
     * <p>
     * The type of resource to monitor.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #resourceType} will
     * return {@link FlowLogsResourceType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #resourceTypeAsString}.
     * </p>
     * 
     * @return The type of resource to monitor.
     * @see FlowLogsResourceType
     */
    public final FlowLogsResourceType resourceType() {
        return FlowLogsResourceType.fromValue(resourceType);
    }

    /**
     * <p>
     * The type of resource to monitor.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #resourceType} will
     * return {@link FlowLogsResourceType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #resourceTypeAsString}.
     * </p>
     * 
     * @return The type of resource to monitor.
     * @see FlowLogsResourceType
     */
    public final String resourceTypeAsString() {
        return resourceType;
    }

    /**
     * <p>
     * The type of traffic to monitor (accepted traffic, rejected traffic, or all traffic).
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #trafficType} will
     * return {@link TrafficType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #trafficTypeAsString}.
     * </p>
     * 
     * @return The type of traffic to monitor (accepted traffic, rejected traffic, or all traffic).
     * @see TrafficType
     */
    public final TrafficType trafficType() {
        return TrafficType.fromValue(trafficType);
    }

    /**
     * <p>
     * The type of traffic to monitor (accepted traffic, rejected traffic, or all traffic).
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #trafficType} will
     * return {@link TrafficType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #trafficTypeAsString}.
     * </p>
     * 
     * @return The type of traffic to monitor (accepted traffic, rejected traffic, or all traffic).
     * @see TrafficType
     */
    public final String trafficTypeAsString() {
        return trafficType;
    }

    /**
     * <p>
     * The type of destination for the flow log data.
     * </p>
     * <p>
     * Default: <code>cloud-watch-logs</code>
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #logDestinationType} will return {@link LogDestinationType#UNKNOWN_TO_SDK_VERSION}. The raw value returned
     * by the service is available from {@link #logDestinationTypeAsString}.
     * </p>
     * 
     * @return The type of destination for the flow log data.</p>
     *         <p>
     *         Default: <code>cloud-watch-logs</code>
     * @see LogDestinationType
     */
    public final LogDestinationType logDestinationType() {
        return LogDestinationType.fromValue(logDestinationType);
    }

    /**
     * <p>
     * The type of destination for the flow log data.
     * </p>
     * <p>
     * Default: <code>cloud-watch-logs</code>
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #logDestinationType} will return {@link LogDestinationType#UNKNOWN_TO_SDK_VERSION}. The raw value returned
     * by the service is available from {@link #logDestinationTypeAsString}.
     * </p>
     * 
     * @return The type of destination for the flow log data.</p>
     *         <p>
     *         Default: <code>cloud-watch-logs</code>
     * @see LogDestinationType
     */
    public final String logDestinationTypeAsString() {
        return logDestinationType;
    }

    /**
     * <p>
     * The destination for the flow log data. The meaning of this parameter depends on the destination type.
     * </p>
     * <ul>
     * <li>
     * <p>
     * If the destination type is <code>cloud-watch-logs</code>, specify the ARN of a CloudWatch Logs log group. For
     * example:
     * </p>
     * <p>
     * arn:aws:logs:<i>region</i>:<i>account_id</i>:log-group:<i>my_group</i>
     * </p>
     * <p>
     * Alternatively, use the <code>LogGroupName</code> parameter.
     * </p>
     * </li>
     * <li>
     * <p>
     * If the destination type is <code>s3</code>, specify the ARN of an S3 bucket. For example:
     * </p>
     * <p>
     * arn:aws:s3:::<i>my_bucket</i>/<i>my_subfolder</i>/
     * </p>
     * <p>
     * The subfolder is optional. Note that you can't use <code>AWSLogs</code> as a subfolder name.
     * </p>
     * </li>
     * <li>
     * <p>
     * If the destination type is <code>kinesis-data-firehose</code>, specify the ARN of a Kinesis Data Firehose
     * delivery stream. For example:
     * </p>
     * <p>
     * arn:aws:firehose:<i>region</i>:<i>account_id</i>:deliverystream:<i>my_stream</i>
     * </p>
     * </li>
     * </ul>
     * 
     * @return The destination for the flow log data. The meaning of this parameter depends on the destination type.</p>
     *         <ul>
     *         <li>
     *         <p>
     *         If the destination type is <code>cloud-watch-logs</code>, specify the ARN of a CloudWatch Logs log group.
     *         For example:
     *         </p>
     *         <p>
     *         arn:aws:logs:<i>region</i>:<i>account_id</i>:log-group:<i>my_group</i>
     *         </p>
     *         <p>
     *         Alternatively, use the <code>LogGroupName</code> parameter.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         If the destination type is <code>s3</code>, specify the ARN of an S3 bucket. For example:
     *         </p>
     *         <p>
     *         arn:aws:s3:::<i>my_bucket</i>/<i>my_subfolder</i>/
     *         </p>
     *         <p>
     *         The subfolder is optional. Note that you can't use <code>AWSLogs</code> as a subfolder name.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         If the destination type is <code>kinesis-data-firehose</code>, specify the ARN of a Kinesis Data Firehose
     *         delivery stream. For example:
     *         </p>
     *         <p>
     *         arn:aws:firehose:<i>region</i>:<i>account_id</i>:deliverystream:<i>my_stream</i>
     *         </p>
     *         </li>
     */
    public final String logDestination() {
        return logDestination;
    }

    /**
     * <p>
     * The fields to include in the flow log record. List the fields in the order in which they should appear. For more
     * information about the available fields, see <a
     * href="https://docs.aws.amazon.com/vpc/latest/userguide/flow-logs.html#flow-log-records">Flow log records</a>. If
     * you omit this parameter, the flow log is created using the default format. If you specify this parameter, you
     * must include at least one field.
     * </p>
     * <p>
     * Specify the fields using the <code>${field-id}</code> format, separated by spaces. For the CLI, surround this
     * parameter value with single quotes on Linux or double quotes on Windows.
     * </p>
     * 
     * @return The fields to include in the flow log record. List the fields in the order in which they should appear.
     *         For more information about the available fields, see <a
     *         href="https://docs.aws.amazon.com/vpc/latest/userguide/flow-logs.html#flow-log-records">Flow log
     *         records</a>. If you omit this parameter, the flow log is created using the default format. If you specify
     *         this parameter, you must include at least one field.</p>
     *         <p>
     *         Specify the fields using the <code>${field-id}</code> format, separated by spaces. For the CLI, surround
     *         this parameter value with single quotes on Linux or double quotes on Windows.
     */
    public final String logFormat() {
        return logFormat;
    }

    /**
     * For responses, this returns true if the service returned a value for the TagSpecifications property. This DOES
     * NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasTagSpecifications() {
        return tagSpecifications != null && !(tagSpecifications instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The tags to apply to the flow logs.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasTagSpecifications} method.
     * </p>
     * 
     * @return The tags to apply to the flow logs.
     */
    public final List<TagSpecification> tagSpecifications() {
        return tagSpecifications;
    }

    /**
     * <p>
     * The maximum interval of time during which a flow of packets is captured and aggregated into a flow log record.
     * You can specify 60 seconds (1 minute) or 600 seconds (10 minutes).
     * </p>
     * <p>
     * When a network interface is attached to a <a
     * href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/instance-types.html#ec2-nitro-instances">Nitro-based
     * instance</a>, the aggregation interval is always 60 seconds or less, regardless of the value that you specify.
     * </p>
     * <p>
     * Default: 600
     * </p>
     * 
     * @return The maximum interval of time during which a flow of packets is captured and aggregated into a flow log
     *         record. You can specify 60 seconds (1 minute) or 600 seconds (10 minutes).</p>
     *         <p>
     *         When a network interface is attached to a <a
     *         href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/instance-types.html#ec2-nitro-instances"
     *         >Nitro-based instance</a>, the aggregation interval is always 60 seconds or less, regardless of the value
     *         that you specify.
     *         </p>
     *         <p>
     *         Default: 600
     */
    public final Integer maxAggregationInterval() {
        return maxAggregationInterval;
    }

    /**
     * <p>
     * The destination options.
     * </p>
     * 
     * @return The destination options.
     */
    public final DestinationOptionsRequest destinationOptions() {
        return destinationOptions;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + super.hashCode();
        hashCode = 31 * hashCode + Objects.hashCode(clientToken());
        hashCode = 31 * hashCode + Objects.hashCode(deliverLogsPermissionArn());
        hashCode = 31 * hashCode + Objects.hashCode(deliverCrossAccountRole());
        hashCode = 31 * hashCode + Objects.hashCode(logGroupName());
        hashCode = 31 * hashCode + Objects.hashCode(hasResourceIds() ? resourceIds() : null);
        hashCode = 31 * hashCode + Objects.hashCode(resourceTypeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(trafficTypeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(logDestinationTypeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(logDestination());
        hashCode = 31 * hashCode + Objects.hashCode(logFormat());
        hashCode = 31 * hashCode + Objects.hashCode(hasTagSpecifications() ? tagSpecifications() : null);
        hashCode = 31 * hashCode + Objects.hashCode(maxAggregationInterval());
        hashCode = 31 * hashCode + Objects.hashCode(destinationOptions());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return super.equals(obj) && equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof CreateFlowLogsRequest)) {
            return false;
        }
        CreateFlowLogsRequest other = (CreateFlowLogsRequest) obj;
        return Objects.equals(clientToken(), other.clientToken())
                && Objects.equals(deliverLogsPermissionArn(), other.deliverLogsPermissionArn())
                && Objects.equals(deliverCrossAccountRole(), other.deliverCrossAccountRole())
                && Objects.equals(logGroupName(), other.logGroupName()) && hasResourceIds() == other.hasResourceIds()
                && Objects.equals(resourceIds(), other.resourceIds())
                && Objects.equals(resourceTypeAsString(), other.resourceTypeAsString())
                && Objects.equals(trafficTypeAsString(), other.trafficTypeAsString())
                && Objects.equals(logDestinationTypeAsString(), other.logDestinationTypeAsString())
                && Objects.equals(logDestination(), other.logDestination()) && Objects.equals(logFormat(), other.logFormat())
                && hasTagSpecifications() == other.hasTagSpecifications()
                && Objects.equals(tagSpecifications(), other.tagSpecifications())
                && Objects.equals(maxAggregationInterval(), other.maxAggregationInterval())
                && Objects.equals(destinationOptions(), other.destinationOptions());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("CreateFlowLogsRequest").add("ClientToken", clientToken())
                .add("DeliverLogsPermissionArn", deliverLogsPermissionArn())
                .add("DeliverCrossAccountRole", deliverCrossAccountRole()).add("LogGroupName", logGroupName())
                .add("ResourceIds", hasResourceIds() ? resourceIds() : null).add("ResourceType", resourceTypeAsString())
                .add("TrafficType", trafficTypeAsString()).add("LogDestinationType", logDestinationTypeAsString())
                .add("LogDestination", logDestination()).add("LogFormat", logFormat())
                .add("TagSpecifications", hasTagSpecifications() ? tagSpecifications() : null)
                .add("MaxAggregationInterval", maxAggregationInterval()).add("DestinationOptions", destinationOptions()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "ClientToken":
            return Optional.ofNullable(clazz.cast(clientToken()));
        case "DeliverLogsPermissionArn":
            return Optional.ofNullable(clazz.cast(deliverLogsPermissionArn()));
        case "DeliverCrossAccountRole":
            return Optional.ofNullable(clazz.cast(deliverCrossAccountRole()));
        case "LogGroupName":
            return Optional.ofNullable(clazz.cast(logGroupName()));
        case "ResourceIds":
            return Optional.ofNullable(clazz.cast(resourceIds()));
        case "ResourceType":
            return Optional.ofNullable(clazz.cast(resourceTypeAsString()));
        case "TrafficType":
            return Optional.ofNullable(clazz.cast(trafficTypeAsString()));
        case "LogDestinationType":
            return Optional.ofNullable(clazz.cast(logDestinationTypeAsString()));
        case "LogDestination":
            return Optional.ofNullable(clazz.cast(logDestination()));
        case "LogFormat":
            return Optional.ofNullable(clazz.cast(logFormat()));
        case "TagSpecifications":
            return Optional.ofNullable(clazz.cast(tagSpecifications()));
        case "MaxAggregationInterval":
            return Optional.ofNullable(clazz.cast(maxAggregationInterval()));
        case "DestinationOptions":
            return Optional.ofNullable(clazz.cast(destinationOptions()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<CreateFlowLogsRequest, T> g) {
        return obj -> g.apply((CreateFlowLogsRequest) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends Ec2Request.Builder, SdkPojo, CopyableBuilder<Builder, CreateFlowLogsRequest> {
        /**
         * <p>
         * Unique, case-sensitive identifier that you provide to ensure the idempotency of the request. For more
         * information, see <a
         * href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/Run_Instance_Idempotency.html">How to ensure
         * idempotency</a>.
         * </p>
         * 
         * @param clientToken
         *        Unique, case-sensitive identifier that you provide to ensure the idempotency of the request. For more
         *        information, see <a
         *        href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/Run_Instance_Idempotency.html">How to ensure
         *        idempotency</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder clientToken(String clientToken);

        /**
         * <p>
         * The ARN of the IAM role that allows Amazon EC2 to publish flow logs to a CloudWatch Logs log group in your
         * account.
         * </p>
         * <p>
         * This parameter is required if the destination type is <code>cloud-watch-logs</code> and unsupported
         * otherwise.
         * </p>
         * 
         * @param deliverLogsPermissionArn
         *        The ARN of the IAM role that allows Amazon EC2 to publish flow logs to a CloudWatch Logs log group in
         *        your account.</p>
         *        <p>
         *        This parameter is required if the destination type is <code>cloud-watch-logs</code> and unsupported
         *        otherwise.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder deliverLogsPermissionArn(String deliverLogsPermissionArn);

        /**
         * <p>
         * The ARN of the IAM role that allows Amazon EC2 to publish flow logs across accounts.
         * </p>
         * 
         * @param deliverCrossAccountRole
         *        The ARN of the IAM role that allows Amazon EC2 to publish flow logs across accounts.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder deliverCrossAccountRole(String deliverCrossAccountRole);

        /**
         * <p>
         * The name of a new or existing CloudWatch Logs log group where Amazon EC2 publishes your flow logs.
         * </p>
         * <p>
         * This parameter is valid only if the destination type is <code>cloud-watch-logs</code>.
         * </p>
         * 
         * @param logGroupName
         *        The name of a new or existing CloudWatch Logs log group where Amazon EC2 publishes your flow logs.</p>
         *        <p>
         *        This parameter is valid only if the destination type is <code>cloud-watch-logs</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder logGroupName(String logGroupName);

        /**
         * <p>
         * The IDs of the resources to monitor. For example, if the resource type is <code>VPC</code>, specify the IDs
         * of the VPCs.
         * </p>
         * <p>
         * Constraints: Maximum of 1000 resources
         * </p>
         * 
         * @param resourceIds
         *        The IDs of the resources to monitor. For example, if the resource type is <code>VPC</code>, specify
         *        the IDs of the VPCs.</p>
         *        <p>
         *        Constraints: Maximum of 1000 resources
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder resourceIds(Collection<String> resourceIds);

        /**
         * <p>
         * The IDs of the resources to monitor. For example, if the resource type is <code>VPC</code>, specify the IDs
         * of the VPCs.
         * </p>
         * <p>
         * Constraints: Maximum of 1000 resources
         * </p>
         * 
         * @param resourceIds
         *        The IDs of the resources to monitor. For example, if the resource type is <code>VPC</code>, specify
         *        the IDs of the VPCs.</p>
         *        <p>
         *        Constraints: Maximum of 1000 resources
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder resourceIds(String... resourceIds);

        /**
         * <p>
         * The type of resource to monitor.
         * </p>
         * 
         * @param resourceType
         *        The type of resource to monitor.
         * @see FlowLogsResourceType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see FlowLogsResourceType
         */
        Builder resourceType(String resourceType);

        /**
         * <p>
         * The type of resource to monitor.
         * </p>
         * 
         * @param resourceType
         *        The type of resource to monitor.
         * @see FlowLogsResourceType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see FlowLogsResourceType
         */
        Builder resourceType(FlowLogsResourceType resourceType);

        /**
         * <p>
         * The type of traffic to monitor (accepted traffic, rejected traffic, or all traffic).
         * </p>
         * 
         * @param trafficType
         *        The type of traffic to monitor (accepted traffic, rejected traffic, or all traffic).
         * @see TrafficType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see TrafficType
         */
        Builder trafficType(String trafficType);

        /**
         * <p>
         * The type of traffic to monitor (accepted traffic, rejected traffic, or all traffic).
         * </p>
         * 
         * @param trafficType
         *        The type of traffic to monitor (accepted traffic, rejected traffic, or all traffic).
         * @see TrafficType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see TrafficType
         */
        Builder trafficType(TrafficType trafficType);

        /**
         * <p>
         * The type of destination for the flow log data.
         * </p>
         * <p>
         * Default: <code>cloud-watch-logs</code>
         * </p>
         * 
         * @param logDestinationType
         *        The type of destination for the flow log data.</p>
         *        <p>
         *        Default: <code>cloud-watch-logs</code>
         * @see LogDestinationType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see LogDestinationType
         */
        Builder logDestinationType(String logDestinationType);

        /**
         * <p>
         * The type of destination for the flow log data.
         * </p>
         * <p>
         * Default: <code>cloud-watch-logs</code>
         * </p>
         * 
         * @param logDestinationType
         *        The type of destination for the flow log data.</p>
         *        <p>
         *        Default: <code>cloud-watch-logs</code>
         * @see LogDestinationType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see LogDestinationType
         */
        Builder logDestinationType(LogDestinationType logDestinationType);

        /**
         * <p>
         * The destination for the flow log data. The meaning of this parameter depends on the destination type.
         * </p>
         * <ul>
         * <li>
         * <p>
         * If the destination type is <code>cloud-watch-logs</code>, specify the ARN of a CloudWatch Logs log group. For
         * example:
         * </p>
         * <p>
         * arn:aws:logs:<i>region</i>:<i>account_id</i>:log-group:<i>my_group</i>
         * </p>
         * <p>
         * Alternatively, use the <code>LogGroupName</code> parameter.
         * </p>
         * </li>
         * <li>
         * <p>
         * If the destination type is <code>s3</code>, specify the ARN of an S3 bucket. For example:
         * </p>
         * <p>
         * arn:aws:s3:::<i>my_bucket</i>/<i>my_subfolder</i>/
         * </p>
         * <p>
         * The subfolder is optional. Note that you can't use <code>AWSLogs</code> as a subfolder name.
         * </p>
         * </li>
         * <li>
         * <p>
         * If the destination type is <code>kinesis-data-firehose</code>, specify the ARN of a Kinesis Data Firehose
         * delivery stream. For example:
         * </p>
         * <p>
         * arn:aws:firehose:<i>region</i>:<i>account_id</i>:deliverystream:<i>my_stream</i>
         * </p>
         * </li>
         * </ul>
         * 
         * @param logDestination
         *        The destination for the flow log data. The meaning of this parameter depends on the destination
         *        type.</p>
         *        <ul>
         *        <li>
         *        <p>
         *        If the destination type is <code>cloud-watch-logs</code>, specify the ARN of a CloudWatch Logs log
         *        group. For example:
         *        </p>
         *        <p>
         *        arn:aws:logs:<i>region</i>:<i>account_id</i>:log-group:<i>my_group</i>
         *        </p>
         *        <p>
         *        Alternatively, use the <code>LogGroupName</code> parameter.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        If the destination type is <code>s3</code>, specify the ARN of an S3 bucket. For example:
         *        </p>
         *        <p>
         *        arn:aws:s3:::<i>my_bucket</i>/<i>my_subfolder</i>/
         *        </p>
         *        <p>
         *        The subfolder is optional. Note that you can't use <code>AWSLogs</code> as a subfolder name.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        If the destination type is <code>kinesis-data-firehose</code>, specify the ARN of a Kinesis Data
         *        Firehose delivery stream. For example:
         *        </p>
         *        <p>
         *        arn:aws:firehose:<i>region</i>:<i>account_id</i>:deliverystream:<i>my_stream</i>
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder logDestination(String logDestination);

        /**
         * <p>
         * The fields to include in the flow log record. List the fields in the order in which they should appear. For
         * more information about the available fields, see <a
         * href="https://docs.aws.amazon.com/vpc/latest/userguide/flow-logs.html#flow-log-records">Flow log records</a>.
         * If you omit this parameter, the flow log is created using the default format. If you specify this parameter,
         * you must include at least one field.
         * </p>
         * <p>
         * Specify the fields using the <code>${field-id}</code> format, separated by spaces. For the CLI, surround this
         * parameter value with single quotes on Linux or double quotes on Windows.
         * </p>
         * 
         * @param logFormat
         *        The fields to include in the flow log record. List the fields in the order in which they should
         *        appear. For more information about the available fields, see <a
         *        href="https://docs.aws.amazon.com/vpc/latest/userguide/flow-logs.html#flow-log-records">Flow log
         *        records</a>. If you omit this parameter, the flow log is created using the default format. If you
         *        specify this parameter, you must include at least one field.</p>
         *        <p>
         *        Specify the fields using the <code>${field-id}</code> format, separated by spaces. For the CLI,
         *        surround this parameter value with single quotes on Linux or double quotes on Windows.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder logFormat(String logFormat);

        /**
         * <p>
         * The tags to apply to the flow logs.
         * </p>
         * 
         * @param tagSpecifications
         *        The tags to apply to the flow logs.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tagSpecifications(Collection<TagSpecification> tagSpecifications);

        /**
         * <p>
         * The tags to apply to the flow logs.
         * </p>
         * 
         * @param tagSpecifications
         *        The tags to apply to the flow logs.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tagSpecifications(TagSpecification... tagSpecifications);

        /**
         * <p>
         * The tags to apply to the flow logs.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.ec2.model.TagSpecification.Builder} avoiding the need to create one
         * manually via {@link software.amazon.awssdk.services.ec2.model.TagSpecification#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.ec2.model.TagSpecification.Builder#build()} is called immediately and
         * its result is passed to {@link #tagSpecifications(List<TagSpecification>)}.
         * 
         * @param tagSpecifications
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.ec2.model.TagSpecification.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #tagSpecifications(java.util.Collection<TagSpecification>)
         */
        Builder tagSpecifications(Consumer<TagSpecification.Builder>... tagSpecifications);

        /**
         * <p>
         * The maximum interval of time during which a flow of packets is captured and aggregated into a flow log
         * record. You can specify 60 seconds (1 minute) or 600 seconds (10 minutes).
         * </p>
         * <p>
         * When a network interface is attached to a <a
         * href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/instance-types.html#ec2-nitro-instances"
         * >Nitro-based instance</a>, the aggregation interval is always 60 seconds or less, regardless of the value
         * that you specify.
         * </p>
         * <p>
         * Default: 600
         * </p>
         * 
         * @param maxAggregationInterval
         *        The maximum interval of time during which a flow of packets is captured and aggregated into a flow log
         *        record. You can specify 60 seconds (1 minute) or 600 seconds (10 minutes).</p>
         *        <p>
         *        When a network interface is attached to a <a
         *        href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/instance-types.html#ec2-nitro-instances"
         *        >Nitro-based instance</a>, the aggregation interval is always 60 seconds or less, regardless of the
         *        value that you specify.
         *        </p>
         *        <p>
         *        Default: 600
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder maxAggregationInterval(Integer maxAggregationInterval);

        /**
         * <p>
         * The destination options.
         * </p>
         * 
         * @param destinationOptions
         *        The destination options.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder destinationOptions(DestinationOptionsRequest destinationOptions);

        /**
         * <p>
         * The destination options.
         * </p>
         * This is a convenience method that creates an instance of the {@link DestinationOptionsRequest.Builder}
         * avoiding the need to create one manually via {@link DestinationOptionsRequest#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link DestinationOptionsRequest.Builder#build()} is called immediately
         * and its result is passed to {@link #destinationOptions(DestinationOptionsRequest)}.
         * 
         * @param destinationOptions
         *        a consumer that will call methods on {@link DestinationOptionsRequest.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #destinationOptions(DestinationOptionsRequest)
         */
        default Builder destinationOptions(Consumer<DestinationOptionsRequest.Builder> destinationOptions) {
            return destinationOptions(DestinationOptionsRequest.builder().applyMutation(destinationOptions).build());
        }

        @Override
        Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration);

        @Override
        Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer);
    }

    static final class BuilderImpl extends Ec2Request.BuilderImpl implements Builder {
        private String clientToken;

        private String deliverLogsPermissionArn;

        private String deliverCrossAccountRole;

        private String logGroupName;

        private List<String> resourceIds = DefaultSdkAutoConstructList.getInstance();

        private String resourceType;

        private String trafficType;

        private String logDestinationType;

        private String logDestination;

        private String logFormat;

        private List<TagSpecification> tagSpecifications = DefaultSdkAutoConstructList.getInstance();

        private Integer maxAggregationInterval;

        private DestinationOptionsRequest destinationOptions;

        private BuilderImpl() {
        }

        private BuilderImpl(CreateFlowLogsRequest model) {
            super(model);
            clientToken(model.clientToken);
            deliverLogsPermissionArn(model.deliverLogsPermissionArn);
            deliverCrossAccountRole(model.deliverCrossAccountRole);
            logGroupName(model.logGroupName);
            resourceIds(model.resourceIds);
            resourceType(model.resourceType);
            trafficType(model.trafficType);
            logDestinationType(model.logDestinationType);
            logDestination(model.logDestination);
            logFormat(model.logFormat);
            tagSpecifications(model.tagSpecifications);
            maxAggregationInterval(model.maxAggregationInterval);
            destinationOptions(model.destinationOptions);
        }

        public final String getClientToken() {
            return clientToken;
        }

        public final void setClientToken(String clientToken) {
            this.clientToken = clientToken;
        }

        @Override
        public final Builder clientToken(String clientToken) {
            this.clientToken = clientToken;
            return this;
        }

        public final String getDeliverLogsPermissionArn() {
            return deliverLogsPermissionArn;
        }

        public final void setDeliverLogsPermissionArn(String deliverLogsPermissionArn) {
            this.deliverLogsPermissionArn = deliverLogsPermissionArn;
        }

        @Override
        public final Builder deliverLogsPermissionArn(String deliverLogsPermissionArn) {
            this.deliverLogsPermissionArn = deliverLogsPermissionArn;
            return this;
        }

        public final String getDeliverCrossAccountRole() {
            return deliverCrossAccountRole;
        }

        public final void setDeliverCrossAccountRole(String deliverCrossAccountRole) {
            this.deliverCrossAccountRole = deliverCrossAccountRole;
        }

        @Override
        public final Builder deliverCrossAccountRole(String deliverCrossAccountRole) {
            this.deliverCrossAccountRole = deliverCrossAccountRole;
            return this;
        }

        public final String getLogGroupName() {
            return logGroupName;
        }

        public final void setLogGroupName(String logGroupName) {
            this.logGroupName = logGroupName;
        }

        @Override
        public final Builder logGroupName(String logGroupName) {
            this.logGroupName = logGroupName;
            return this;
        }

        public final Collection<String> getResourceIds() {
            if (resourceIds instanceof SdkAutoConstructList) {
                return null;
            }
            return resourceIds;
        }

        public final void setResourceIds(Collection<String> resourceIds) {
            this.resourceIds = FlowLogResourceIdsCopier.copy(resourceIds);
        }

        @Override
        public final Builder resourceIds(Collection<String> resourceIds) {
            this.resourceIds = FlowLogResourceIdsCopier.copy(resourceIds);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder resourceIds(String... resourceIds) {
            resourceIds(Arrays.asList(resourceIds));
            return this;
        }

        public final String getResourceType() {
            return resourceType;
        }

        public final void setResourceType(String resourceType) {
            this.resourceType = resourceType;
        }

        @Override
        public final Builder resourceType(String resourceType) {
            this.resourceType = resourceType;
            return this;
        }

        @Override
        public final Builder resourceType(FlowLogsResourceType resourceType) {
            this.resourceType(resourceType == null ? null : resourceType.toString());
            return this;
        }

        public final String getTrafficType() {
            return trafficType;
        }

        public final void setTrafficType(String trafficType) {
            this.trafficType = trafficType;
        }

        @Override
        public final Builder trafficType(String trafficType) {
            this.trafficType = trafficType;
            return this;
        }

        @Override
        public final Builder trafficType(TrafficType trafficType) {
            this.trafficType(trafficType == null ? null : trafficType.toString());
            return this;
        }

        public final String getLogDestinationType() {
            return logDestinationType;
        }

        public final void setLogDestinationType(String logDestinationType) {
            this.logDestinationType = logDestinationType;
        }

        @Override
        public final Builder logDestinationType(String logDestinationType) {
            this.logDestinationType = logDestinationType;
            return this;
        }

        @Override
        public final Builder logDestinationType(LogDestinationType logDestinationType) {
            this.logDestinationType(logDestinationType == null ? null : logDestinationType.toString());
            return this;
        }

        public final String getLogDestination() {
            return logDestination;
        }

        public final void setLogDestination(String logDestination) {
            this.logDestination = logDestination;
        }

        @Override
        public final Builder logDestination(String logDestination) {
            this.logDestination = logDestination;
            return this;
        }

        public final String getLogFormat() {
            return logFormat;
        }

        public final void setLogFormat(String logFormat) {
            this.logFormat = logFormat;
        }

        @Override
        public final Builder logFormat(String logFormat) {
            this.logFormat = logFormat;
            return this;
        }

        public final List<TagSpecification.Builder> getTagSpecifications() {
            List<TagSpecification.Builder> result = TagSpecificationListCopier.copyToBuilder(this.tagSpecifications);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setTagSpecifications(Collection<TagSpecification.BuilderImpl> tagSpecifications) {
            this.tagSpecifications = TagSpecificationListCopier.copyFromBuilder(tagSpecifications);
        }

        @Override
        public final Builder tagSpecifications(Collection<TagSpecification> tagSpecifications) {
            this.tagSpecifications = TagSpecificationListCopier.copy(tagSpecifications);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder tagSpecifications(TagSpecification... tagSpecifications) {
            tagSpecifications(Arrays.asList(tagSpecifications));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder tagSpecifications(Consumer<TagSpecification.Builder>... tagSpecifications) {
            tagSpecifications(Stream.of(tagSpecifications).map(c -> TagSpecification.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final Integer getMaxAggregationInterval() {
            return maxAggregationInterval;
        }

        public final void setMaxAggregationInterval(Integer maxAggregationInterval) {
            this.maxAggregationInterval = maxAggregationInterval;
        }

        @Override
        public final Builder maxAggregationInterval(Integer maxAggregationInterval) {
            this.maxAggregationInterval = maxAggregationInterval;
            return this;
        }

        public final DestinationOptionsRequest.Builder getDestinationOptions() {
            return destinationOptions != null ? destinationOptions.toBuilder() : null;
        }

        public final void setDestinationOptions(DestinationOptionsRequest.BuilderImpl destinationOptions) {
            this.destinationOptions = destinationOptions != null ? destinationOptions.build() : null;
        }

        @Override
        public final Builder destinationOptions(DestinationOptionsRequest destinationOptions) {
            this.destinationOptions = destinationOptions;
            return this;
        }

        @Override
        public Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration) {
            super.overrideConfiguration(overrideConfiguration);
            return this;
        }

        @Override
        public Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer) {
            super.overrideConfiguration(builderConsumer);
            return this;
        }

        @Override
        public CreateFlowLogsRequest build() {
            return new CreateFlowLogsRequest(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
