/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.detective.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.MapTrait;
import software.amazon.awssdk.core.traits.TimestampFormatTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructMap;
import software.amazon.awssdk.core.util.SdkAutoConstructMap;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Details about a member account in a behavior graph.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class MemberDetail implements SdkPojo, Serializable, ToCopyableBuilder<MemberDetail.Builder, MemberDetail> {
    private static final SdkField<String> ACCOUNT_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("AccountId").getter(getter(MemberDetail::accountId)).setter(setter(Builder::accountId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AccountId").build()).build();

    private static final SdkField<String> EMAIL_ADDRESS_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("EmailAddress").getter(getter(MemberDetail::emailAddress)).setter(setter(Builder::emailAddress))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("EmailAddress").build()).build();

    private static final SdkField<String> GRAPH_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("GraphArn").getter(getter(MemberDetail::graphArn)).setter(setter(Builder::graphArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("GraphArn").build()).build();

    private static final SdkField<String> MASTER_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("MasterId").getter(getter(MemberDetail::masterId)).setter(setter(Builder::masterId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("MasterId").build()).build();

    private static final SdkField<String> ADMINISTRATOR_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("AdministratorId").getter(getter(MemberDetail::administratorId)).setter(setter(Builder::administratorId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AdministratorId").build()).build();

    private static final SdkField<String> STATUS_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Status")
            .getter(getter(MemberDetail::statusAsString)).setter(setter(Builder::status))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Status").build()).build();

    private static final SdkField<String> DISABLED_REASON_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("DisabledReason").getter(getter(MemberDetail::disabledReasonAsString))
            .setter(setter(Builder::disabledReason))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DisabledReason").build()).build();

    private static final SdkField<Instant> INVITED_TIME_FIELD = SdkField
            .<Instant> builder(MarshallingType.INSTANT)
            .memberName("InvitedTime")
            .getter(getter(MemberDetail::invitedTime))
            .setter(setter(Builder::invitedTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("InvitedTime").build(),
                    TimestampFormatTrait.create(TimestampFormatTrait.Format.ISO_8601)).build();

    private static final SdkField<Instant> UPDATED_TIME_FIELD = SdkField
            .<Instant> builder(MarshallingType.INSTANT)
            .memberName("UpdatedTime")
            .getter(getter(MemberDetail::updatedTime))
            .setter(setter(Builder::updatedTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("UpdatedTime").build(),
                    TimestampFormatTrait.create(TimestampFormatTrait.Format.ISO_8601)).build();

    private static final SdkField<Long> VOLUME_USAGE_IN_BYTES_FIELD = SdkField.<Long> builder(MarshallingType.LONG)
            .memberName("VolumeUsageInBytes").getter(getter(MemberDetail::volumeUsageInBytes))
            .setter(setter(Builder::volumeUsageInBytes))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("VolumeUsageInBytes").build())
            .build();

    private static final SdkField<Instant> VOLUME_USAGE_UPDATED_TIME_FIELD = SdkField
            .<Instant> builder(MarshallingType.INSTANT)
            .memberName("VolumeUsageUpdatedTime")
            .getter(getter(MemberDetail::volumeUsageUpdatedTime))
            .setter(setter(Builder::volumeUsageUpdatedTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("VolumeUsageUpdatedTime").build(),
                    TimestampFormatTrait.create(TimestampFormatTrait.Format.ISO_8601)).build();

    private static final SdkField<Double> PERCENT_OF_GRAPH_UTILIZATION_FIELD = SdkField.<Double> builder(MarshallingType.DOUBLE)
            .memberName("PercentOfGraphUtilization").getter(getter(MemberDetail::percentOfGraphUtilization))
            .setter(setter(Builder::percentOfGraphUtilization))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("PercentOfGraphUtilization").build())
            .build();

    private static final SdkField<Instant> PERCENT_OF_GRAPH_UTILIZATION_UPDATED_TIME_FIELD = SdkField
            .<Instant> builder(MarshallingType.INSTANT)
            .memberName("PercentOfGraphUtilizationUpdatedTime")
            .getter(getter(MemberDetail::percentOfGraphUtilizationUpdatedTime))
            .setter(setter(Builder::percentOfGraphUtilizationUpdatedTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                    .locationName("PercentOfGraphUtilizationUpdatedTime").build(),
                    TimestampFormatTrait.create(TimestampFormatTrait.Format.ISO_8601)).build();

    private static final SdkField<String> INVITATION_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("InvitationType").getter(getter(MemberDetail::invitationTypeAsString))
            .setter(setter(Builder::invitationType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("InvitationType").build()).build();

    private static final SdkField<Map<String, DatasourcePackageUsageInfo>> VOLUME_USAGE_BY_DATASOURCE_PACKAGE_FIELD = SdkField
            .<Map<String, DatasourcePackageUsageInfo>> builder(MarshallingType.MAP)
            .memberName("VolumeUsageByDatasourcePackage")
            .getter(getter(MemberDetail::volumeUsageByDatasourcePackageAsStrings))
            .setter(setter(Builder::volumeUsageByDatasourcePackageWithStrings))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("VolumeUsageByDatasourcePackage")
                    .build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<DatasourcePackageUsageInfo> builder(MarshallingType.SDK_POJO)
                                            .constructor(DatasourcePackageUsageInfo::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build()).build()).build()).build();

    private static final SdkField<Map<String, String>> DATASOURCE_PACKAGE_INGEST_STATES_FIELD = SdkField
            .<Map<String, String>> builder(MarshallingType.MAP)
            .memberName("DatasourcePackageIngestStates")
            .getter(getter(MemberDetail::datasourcePackageIngestStatesAsStrings))
            .setter(setter(Builder::datasourcePackageIngestStatesWithStrings))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DatasourcePackageIngestStates")
                    .build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(ACCOUNT_ID_FIELD,
            EMAIL_ADDRESS_FIELD, GRAPH_ARN_FIELD, MASTER_ID_FIELD, ADMINISTRATOR_ID_FIELD, STATUS_FIELD, DISABLED_REASON_FIELD,
            INVITED_TIME_FIELD, UPDATED_TIME_FIELD, VOLUME_USAGE_IN_BYTES_FIELD, VOLUME_USAGE_UPDATED_TIME_FIELD,
            PERCENT_OF_GRAPH_UTILIZATION_FIELD, PERCENT_OF_GRAPH_UTILIZATION_UPDATED_TIME_FIELD, INVITATION_TYPE_FIELD,
            VOLUME_USAGE_BY_DATASOURCE_PACKAGE_FIELD, DATASOURCE_PACKAGE_INGEST_STATES_FIELD));

    private static final long serialVersionUID = 1L;

    private final String accountId;

    private final String emailAddress;

    private final String graphArn;

    private final String masterId;

    private final String administratorId;

    private final String status;

    private final String disabledReason;

    private final Instant invitedTime;

    private final Instant updatedTime;

    private final Long volumeUsageInBytes;

    private final Instant volumeUsageUpdatedTime;

    private final Double percentOfGraphUtilization;

    private final Instant percentOfGraphUtilizationUpdatedTime;

    private final String invitationType;

    private final Map<String, DatasourcePackageUsageInfo> volumeUsageByDatasourcePackage;

    private final Map<String, String> datasourcePackageIngestStates;

    private MemberDetail(BuilderImpl builder) {
        this.accountId = builder.accountId;
        this.emailAddress = builder.emailAddress;
        this.graphArn = builder.graphArn;
        this.masterId = builder.masterId;
        this.administratorId = builder.administratorId;
        this.status = builder.status;
        this.disabledReason = builder.disabledReason;
        this.invitedTime = builder.invitedTime;
        this.updatedTime = builder.updatedTime;
        this.volumeUsageInBytes = builder.volumeUsageInBytes;
        this.volumeUsageUpdatedTime = builder.volumeUsageUpdatedTime;
        this.percentOfGraphUtilization = builder.percentOfGraphUtilization;
        this.percentOfGraphUtilizationUpdatedTime = builder.percentOfGraphUtilizationUpdatedTime;
        this.invitationType = builder.invitationType;
        this.volumeUsageByDatasourcePackage = builder.volumeUsageByDatasourcePackage;
        this.datasourcePackageIngestStates = builder.datasourcePackageIngestStates;
    }

    /**
     * <p>
     * The Amazon Web Services account identifier for the member account.
     * </p>
     * 
     * @return The Amazon Web Services account identifier for the member account.
     */
    public final String accountId() {
        return accountId;
    }

    /**
     * <p>
     * The Amazon Web Services account root user email address for the member account.
     * </p>
     * 
     * @return The Amazon Web Services account root user email address for the member account.
     */
    public final String emailAddress() {
        return emailAddress;
    }

    /**
     * <p>
     * The ARN of the behavior graph.
     * </p>
     * 
     * @return The ARN of the behavior graph.
     */
    public final String graphArn() {
        return graphArn;
    }

    /**
     * <p>
     * The Amazon Web Services account identifier of the administrator account for the behavior graph.
     * </p>
     * 
     * @return The Amazon Web Services account identifier of the administrator account for the behavior graph.
     * @deprecated This property is deprecated. Use AdministratorId instead.
     */
    @Deprecated
    public final String masterId() {
        return masterId;
    }

    /**
     * <p>
     * The Amazon Web Services account identifier of the administrator account for the behavior graph.
     * </p>
     * 
     * @return The Amazon Web Services account identifier of the administrator account for the behavior graph.
     */
    public final String administratorId() {
        return administratorId;
    }

    /**
     * <p>
     * The current membership status of the member account. The status can have one of the following values:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>INVITED</code> - For invited accounts only. Indicates that the member was sent an invitation but has not
     * yet responded.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>VERIFICATION_IN_PROGRESS</code> - For invited accounts only, indicates that Detective is verifying that the
     * account identifier and email address provided for the member account match. If they do match, then Detective
     * sends the invitation. If the email address and account identifier don't match, then the member cannot be added to
     * the behavior graph.
     * </p>
     * <p>
     * For organization accounts in the organization behavior graph, indicates that Detective is verifying that the
     * account belongs to the organization.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>VERIFICATION_FAILED</code> - For invited accounts only. Indicates that the account and email address
     * provided for the member account do not match, and Detective did not send an invitation to the account.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>ENABLED</code> - Indicates that the member account currently contributes data to the behavior graph. For
     * invited accounts, the member account accepted the invitation. For organization accounts in the organization
     * behavior graph, the Detective administrator account enabled the organization account as a member account.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>ACCEPTED_BUT_DISABLED</code> - The account accepted the invitation, or was enabled by the Detective
     * administrator account, but is prevented from contributing data to the behavior graph. <code>DisabledReason</code>
     * provides the reason why the member account is not enabled.
     * </p>
     * </li>
     * </ul>
     * <p>
     * Invited accounts that declined an invitation or that were removed from the behavior graph are not included. In
     * the organization behavior graph, organization accounts that the Detective administrator account did not enable
     * are not included.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link MemberStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #statusAsString}.
     * </p>
     * 
     * @return The current membership status of the member account. The status can have one of the following values:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>INVITED</code> - For invited accounts only. Indicates that the member was sent an invitation but
     *         has not yet responded.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>VERIFICATION_IN_PROGRESS</code> - For invited accounts only, indicates that Detective is verifying
     *         that the account identifier and email address provided for the member account match. If they do match,
     *         then Detective sends the invitation. If the email address and account identifier don't match, then the
     *         member cannot be added to the behavior graph.
     *         </p>
     *         <p>
     *         For organization accounts in the organization behavior graph, indicates that Detective is verifying that
     *         the account belongs to the organization.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>VERIFICATION_FAILED</code> - For invited accounts only. Indicates that the account and email
     *         address provided for the member account do not match, and Detective did not send an invitation to the
     *         account.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>ENABLED</code> - Indicates that the member account currently contributes data to the behavior
     *         graph. For invited accounts, the member account accepted the invitation. For organization accounts in the
     *         organization behavior graph, the Detective administrator account enabled the organization account as a
     *         member account.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>ACCEPTED_BUT_DISABLED</code> - The account accepted the invitation, or was enabled by the Detective
     *         administrator account, but is prevented from contributing data to the behavior graph.
     *         <code>DisabledReason</code> provides the reason why the member account is not enabled.
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         Invited accounts that declined an invitation or that were removed from the behavior graph are not
     *         included. In the organization behavior graph, organization accounts that the Detective administrator
     *         account did not enable are not included.
     * @see MemberStatus
     */
    public final MemberStatus status() {
        return MemberStatus.fromValue(status);
    }

    /**
     * <p>
     * The current membership status of the member account. The status can have one of the following values:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>INVITED</code> - For invited accounts only. Indicates that the member was sent an invitation but has not
     * yet responded.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>VERIFICATION_IN_PROGRESS</code> - For invited accounts only, indicates that Detective is verifying that the
     * account identifier and email address provided for the member account match. If they do match, then Detective
     * sends the invitation. If the email address and account identifier don't match, then the member cannot be added to
     * the behavior graph.
     * </p>
     * <p>
     * For organization accounts in the organization behavior graph, indicates that Detective is verifying that the
     * account belongs to the organization.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>VERIFICATION_FAILED</code> - For invited accounts only. Indicates that the account and email address
     * provided for the member account do not match, and Detective did not send an invitation to the account.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>ENABLED</code> - Indicates that the member account currently contributes data to the behavior graph. For
     * invited accounts, the member account accepted the invitation. For organization accounts in the organization
     * behavior graph, the Detective administrator account enabled the organization account as a member account.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>ACCEPTED_BUT_DISABLED</code> - The account accepted the invitation, or was enabled by the Detective
     * administrator account, but is prevented from contributing data to the behavior graph. <code>DisabledReason</code>
     * provides the reason why the member account is not enabled.
     * </p>
     * </li>
     * </ul>
     * <p>
     * Invited accounts that declined an invitation or that were removed from the behavior graph are not included. In
     * the organization behavior graph, organization accounts that the Detective administrator account did not enable
     * are not included.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link MemberStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #statusAsString}.
     * </p>
     * 
     * @return The current membership status of the member account. The status can have one of the following values:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>INVITED</code> - For invited accounts only. Indicates that the member was sent an invitation but
     *         has not yet responded.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>VERIFICATION_IN_PROGRESS</code> - For invited accounts only, indicates that Detective is verifying
     *         that the account identifier and email address provided for the member account match. If they do match,
     *         then Detective sends the invitation. If the email address and account identifier don't match, then the
     *         member cannot be added to the behavior graph.
     *         </p>
     *         <p>
     *         For organization accounts in the organization behavior graph, indicates that Detective is verifying that
     *         the account belongs to the organization.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>VERIFICATION_FAILED</code> - For invited accounts only. Indicates that the account and email
     *         address provided for the member account do not match, and Detective did not send an invitation to the
     *         account.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>ENABLED</code> - Indicates that the member account currently contributes data to the behavior
     *         graph. For invited accounts, the member account accepted the invitation. For organization accounts in the
     *         organization behavior graph, the Detective administrator account enabled the organization account as a
     *         member account.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>ACCEPTED_BUT_DISABLED</code> - The account accepted the invitation, or was enabled by the Detective
     *         administrator account, but is prevented from contributing data to the behavior graph.
     *         <code>DisabledReason</code> provides the reason why the member account is not enabled.
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         Invited accounts that declined an invitation or that were removed from the behavior graph are not
     *         included. In the organization behavior graph, organization accounts that the Detective administrator
     *         account did not enable are not included.
     * @see MemberStatus
     */
    public final String statusAsString() {
        return status;
    }

    /**
     * <p>
     * For member accounts with a status of <code>ACCEPTED_BUT_DISABLED</code>, the reason that the member account is
     * not enabled.
     * </p>
     * <p>
     * The reason can have one of the following values:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>VOLUME_TOO_HIGH</code> - Indicates that adding the member account would cause the data volume for the
     * behavior graph to be too high.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>VOLUME_UNKNOWN</code> - Indicates that Detective is unable to verify the data volume for the member
     * account. This is usually because the member account is not enrolled in Amazon GuardDuty.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #disabledReason}
     * will return {@link MemberDisabledReason#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is
     * available from {@link #disabledReasonAsString}.
     * </p>
     * 
     * @return For member accounts with a status of <code>ACCEPTED_BUT_DISABLED</code>, the reason that the member
     *         account is not enabled.</p>
     *         <p>
     *         The reason can have one of the following values:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>VOLUME_TOO_HIGH</code> - Indicates that adding the member account would cause the data volume for
     *         the behavior graph to be too high.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>VOLUME_UNKNOWN</code> - Indicates that Detective is unable to verify the data volume for the member
     *         account. This is usually because the member account is not enrolled in Amazon GuardDuty.
     *         </p>
     *         </li>
     * @see MemberDisabledReason
     */
    public final MemberDisabledReason disabledReason() {
        return MemberDisabledReason.fromValue(disabledReason);
    }

    /**
     * <p>
     * For member accounts with a status of <code>ACCEPTED_BUT_DISABLED</code>, the reason that the member account is
     * not enabled.
     * </p>
     * <p>
     * The reason can have one of the following values:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>VOLUME_TOO_HIGH</code> - Indicates that adding the member account would cause the data volume for the
     * behavior graph to be too high.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>VOLUME_UNKNOWN</code> - Indicates that Detective is unable to verify the data volume for the member
     * account. This is usually because the member account is not enrolled in Amazon GuardDuty.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #disabledReason}
     * will return {@link MemberDisabledReason#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is
     * available from {@link #disabledReasonAsString}.
     * </p>
     * 
     * @return For member accounts with a status of <code>ACCEPTED_BUT_DISABLED</code>, the reason that the member
     *         account is not enabled.</p>
     *         <p>
     *         The reason can have one of the following values:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>VOLUME_TOO_HIGH</code> - Indicates that adding the member account would cause the data volume for
     *         the behavior graph to be too high.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>VOLUME_UNKNOWN</code> - Indicates that Detective is unable to verify the data volume for the member
     *         account. This is usually because the member account is not enrolled in Amazon GuardDuty.
     *         </p>
     *         </li>
     * @see MemberDisabledReason
     */
    public final String disabledReasonAsString() {
        return disabledReason;
    }

    /**
     * <p>
     * For invited accounts, the date and time that Detective sent the invitation to the account. The value is an
     * ISO8601 formatted string. For example, <code>2021-08-18T16:35:56.284Z</code>.
     * </p>
     * 
     * @return For invited accounts, the date and time that Detective sent the invitation to the account. The value is
     *         an ISO8601 formatted string. For example, <code>2021-08-18T16:35:56.284Z</code>.
     */
    public final Instant invitedTime() {
        return invitedTime;
    }

    /**
     * <p>
     * The date and time that the member account was last updated. The value is an ISO8601 formatted string. For
     * example, <code>2021-08-18T16:35:56.284Z</code>.
     * </p>
     * 
     * @return The date and time that the member account was last updated. The value is an ISO8601 formatted string. For
     *         example, <code>2021-08-18T16:35:56.284Z</code>.
     */
    public final Instant updatedTime() {
        return updatedTime;
    }

    /**
     * <p>
     * The data volume in bytes per day for the member account.
     * </p>
     * 
     * @return The data volume in bytes per day for the member account.
     * @deprecated This property is deprecated. Use VolumeUsageByDatasourcePackage instead.
     */
    @Deprecated
    public final Long volumeUsageInBytes() {
        return volumeUsageInBytes;
    }

    /**
     * <p>
     * The data and time when the member account data volume was last updated. The value is an ISO8601 formatted string.
     * For example, <code>2021-08-18T16:35:56.284Z</code>.
     * </p>
     * 
     * @return The data and time when the member account data volume was last updated. The value is an ISO8601 formatted
     *         string. For example, <code>2021-08-18T16:35:56.284Z</code>.
     * @deprecated This property is deprecated. Use VolumeUsageByDatasourcePackage instead.
     */
    @Deprecated
    public final Instant volumeUsageUpdatedTime() {
        return volumeUsageUpdatedTime;
    }

    /**
     * <p>
     * The member account data volume as a percentage of the maximum allowed data volume. 0 indicates 0 percent, and 100
     * indicates 100 percent.
     * </p>
     * <p>
     * Note that this is not the percentage of the behavior graph data volume.
     * </p>
     * <p>
     * For example, the data volume for the behavior graph is 80 GB per day. The maximum data volume is 160 GB per day.
     * If the data volume for the member account is 40 GB per day, then <code>PercentOfGraphUtilization</code> is 25. It
     * represents 25% of the maximum allowed data volume.
     * </p>
     * 
     * @return The member account data volume as a percentage of the maximum allowed data volume. 0 indicates 0 percent,
     *         and 100 indicates 100 percent.</p>
     *         <p>
     *         Note that this is not the percentage of the behavior graph data volume.
     *         </p>
     *         <p>
     *         For example, the data volume for the behavior graph is 80 GB per day. The maximum data volume is 160 GB
     *         per day. If the data volume for the member account is 40 GB per day, then
     *         <code>PercentOfGraphUtilization</code> is 25. It represents 25% of the maximum allowed data volume.
     * @deprecated This property is deprecated. Use VolumeUsageByDatasourcePackage instead.
     */
    @Deprecated
    public final Double percentOfGraphUtilization() {
        return percentOfGraphUtilization;
    }

    /**
     * <p>
     * The date and time when the graph utilization percentage was last updated. The value is an ISO8601 formatted
     * string. For example, <code>2021-08-18T16:35:56.284Z</code>.
     * </p>
     * 
     * @return The date and time when the graph utilization percentage was last updated. The value is an ISO8601
     *         formatted string. For example, <code>2021-08-18T16:35:56.284Z</code>.
     * @deprecated This property is deprecated. Use VolumeUsageByDatasourcePackage instead.
     */
    @Deprecated
    public final Instant percentOfGraphUtilizationUpdatedTime() {
        return percentOfGraphUtilizationUpdatedTime;
    }

    /**
     * <p>
     * The type of behavior graph membership.
     * </p>
     * <p>
     * For an organization account in the organization behavior graph, the type is <code>ORGANIZATION</code>.
     * </p>
     * <p>
     * For an account that was invited to a behavior graph, the type is <code>INVITATION</code>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #invitationType}
     * will return {@link InvitationType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #invitationTypeAsString}.
     * </p>
     * 
     * @return The type of behavior graph membership.</p>
     *         <p>
     *         For an organization account in the organization behavior graph, the type is <code>ORGANIZATION</code>.
     *         </p>
     *         <p>
     *         For an account that was invited to a behavior graph, the type is <code>INVITATION</code>.
     * @see InvitationType
     */
    public final InvitationType invitationType() {
        return InvitationType.fromValue(invitationType);
    }

    /**
     * <p>
     * The type of behavior graph membership.
     * </p>
     * <p>
     * For an organization account in the organization behavior graph, the type is <code>ORGANIZATION</code>.
     * </p>
     * <p>
     * For an account that was invited to a behavior graph, the type is <code>INVITATION</code>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #invitationType}
     * will return {@link InvitationType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #invitationTypeAsString}.
     * </p>
     * 
     * @return The type of behavior graph membership.</p>
     *         <p>
     *         For an organization account in the organization behavior graph, the type is <code>ORGANIZATION</code>.
     *         </p>
     *         <p>
     *         For an account that was invited to a behavior graph, the type is <code>INVITATION</code>.
     * @see InvitationType
     */
    public final String invitationTypeAsString() {
        return invitationType;
    }

    /**
     * <p>
     * Details on the volume of usage for each data source package in a behavior graph.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasVolumeUsageByDatasourcePackage} method.
     * </p>
     * 
     * @return Details on the volume of usage for each data source package in a behavior graph.
     */
    public final Map<DatasourcePackage, DatasourcePackageUsageInfo> volumeUsageByDatasourcePackage() {
        return VolumeUsageByDatasourcePackageCopier.copyStringToEnum(volumeUsageByDatasourcePackage);
    }

    /**
     * For responses, this returns true if the service returned a value for the VolumeUsageByDatasourcePackage property.
     * This DOES NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the
     * property). This is useful because the SDK will never return a null collection or map, but you may need to
     * differentiate between the service returning nothing (or null) and the service returning an empty collection or
     * map. For requests, this returns true if a value for the property was specified in the request builder, and false
     * if a value was not specified.
     */
    public final boolean hasVolumeUsageByDatasourcePackage() {
        return volumeUsageByDatasourcePackage != null && !(volumeUsageByDatasourcePackage instanceof SdkAutoConstructMap);
    }

    /**
     * <p>
     * Details on the volume of usage for each data source package in a behavior graph.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasVolumeUsageByDatasourcePackage} method.
     * </p>
     * 
     * @return Details on the volume of usage for each data source package in a behavior graph.
     */
    public final Map<String, DatasourcePackageUsageInfo> volumeUsageByDatasourcePackageAsStrings() {
        return volumeUsageByDatasourcePackage;
    }

    /**
     * <p>
     * The state of a data source package for the behavior graph.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasDatasourcePackageIngestStates} method.
     * </p>
     * 
     * @return The state of a data source package for the behavior graph.
     */
    public final Map<DatasourcePackage, DatasourcePackageIngestState> datasourcePackageIngestStates() {
        return DatasourcePackageIngestStatesCopier.copyStringToEnum(datasourcePackageIngestStates);
    }

    /**
     * For responses, this returns true if the service returned a value for the DatasourcePackageIngestStates property.
     * This DOES NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the
     * property). This is useful because the SDK will never return a null collection or map, but you may need to
     * differentiate between the service returning nothing (or null) and the service returning an empty collection or
     * map. For requests, this returns true if a value for the property was specified in the request builder, and false
     * if a value was not specified.
     */
    public final boolean hasDatasourcePackageIngestStates() {
        return datasourcePackageIngestStates != null && !(datasourcePackageIngestStates instanceof SdkAutoConstructMap);
    }

    /**
     * <p>
     * The state of a data source package for the behavior graph.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasDatasourcePackageIngestStates} method.
     * </p>
     * 
     * @return The state of a data source package for the behavior graph.
     */
    public final Map<String, String> datasourcePackageIngestStatesAsStrings() {
        return datasourcePackageIngestStates;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(accountId());
        hashCode = 31 * hashCode + Objects.hashCode(emailAddress());
        hashCode = 31 * hashCode + Objects.hashCode(graphArn());
        hashCode = 31 * hashCode + Objects.hashCode(masterId());
        hashCode = 31 * hashCode + Objects.hashCode(administratorId());
        hashCode = 31 * hashCode + Objects.hashCode(statusAsString());
        hashCode = 31 * hashCode + Objects.hashCode(disabledReasonAsString());
        hashCode = 31 * hashCode + Objects.hashCode(invitedTime());
        hashCode = 31 * hashCode + Objects.hashCode(updatedTime());
        hashCode = 31 * hashCode + Objects.hashCode(volumeUsageInBytes());
        hashCode = 31 * hashCode + Objects.hashCode(volumeUsageUpdatedTime());
        hashCode = 31 * hashCode + Objects.hashCode(percentOfGraphUtilization());
        hashCode = 31 * hashCode + Objects.hashCode(percentOfGraphUtilizationUpdatedTime());
        hashCode = 31 * hashCode + Objects.hashCode(invitationTypeAsString());
        hashCode = 31 * hashCode
                + Objects.hashCode(hasVolumeUsageByDatasourcePackage() ? volumeUsageByDatasourcePackageAsStrings() : null);
        hashCode = 31 * hashCode
                + Objects.hashCode(hasDatasourcePackageIngestStates() ? datasourcePackageIngestStatesAsStrings() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof MemberDetail)) {
            return false;
        }
        MemberDetail other = (MemberDetail) obj;
        return Objects.equals(accountId(), other.accountId()) && Objects.equals(emailAddress(), other.emailAddress())
                && Objects.equals(graphArn(), other.graphArn()) && Objects.equals(masterId(), other.masterId())
                && Objects.equals(administratorId(), other.administratorId())
                && Objects.equals(statusAsString(), other.statusAsString())
                && Objects.equals(disabledReasonAsString(), other.disabledReasonAsString())
                && Objects.equals(invitedTime(), other.invitedTime()) && Objects.equals(updatedTime(), other.updatedTime())
                && Objects.equals(volumeUsageInBytes(), other.volumeUsageInBytes())
                && Objects.equals(volumeUsageUpdatedTime(), other.volumeUsageUpdatedTime())
                && Objects.equals(percentOfGraphUtilization(), other.percentOfGraphUtilization())
                && Objects.equals(percentOfGraphUtilizationUpdatedTime(), other.percentOfGraphUtilizationUpdatedTime())
                && Objects.equals(invitationTypeAsString(), other.invitationTypeAsString())
                && hasVolumeUsageByDatasourcePackage() == other.hasVolumeUsageByDatasourcePackage()
                && Objects.equals(volumeUsageByDatasourcePackageAsStrings(), other.volumeUsageByDatasourcePackageAsStrings())
                && hasDatasourcePackageIngestStates() == other.hasDatasourcePackageIngestStates()
                && Objects.equals(datasourcePackageIngestStatesAsStrings(), other.datasourcePackageIngestStatesAsStrings());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString
                .builder("MemberDetail")
                .add("AccountId", accountId())
                .add("EmailAddress", emailAddress())
                .add("GraphArn", graphArn())
                .add("MasterId", masterId())
                .add("AdministratorId", administratorId())
                .add("Status", statusAsString())
                .add("DisabledReason", disabledReasonAsString())
                .add("InvitedTime", invitedTime())
                .add("UpdatedTime", updatedTime())
                .add("VolumeUsageInBytes", volumeUsageInBytes())
                .add("VolumeUsageUpdatedTime", volumeUsageUpdatedTime())
                .add("PercentOfGraphUtilization", percentOfGraphUtilization())
                .add("PercentOfGraphUtilizationUpdatedTime", percentOfGraphUtilizationUpdatedTime())
                .add("InvitationType", invitationTypeAsString())
                .add("VolumeUsageByDatasourcePackage",
                        hasVolumeUsageByDatasourcePackage() ? volumeUsageByDatasourcePackageAsStrings() : null)
                .add("DatasourcePackageIngestStates",
                        hasDatasourcePackageIngestStates() ? datasourcePackageIngestStatesAsStrings() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "AccountId":
            return Optional.ofNullable(clazz.cast(accountId()));
        case "EmailAddress":
            return Optional.ofNullable(clazz.cast(emailAddress()));
        case "GraphArn":
            return Optional.ofNullable(clazz.cast(graphArn()));
        case "MasterId":
            return Optional.ofNullable(clazz.cast(masterId()));
        case "AdministratorId":
            return Optional.ofNullable(clazz.cast(administratorId()));
        case "Status":
            return Optional.ofNullable(clazz.cast(statusAsString()));
        case "DisabledReason":
            return Optional.ofNullable(clazz.cast(disabledReasonAsString()));
        case "InvitedTime":
            return Optional.ofNullable(clazz.cast(invitedTime()));
        case "UpdatedTime":
            return Optional.ofNullable(clazz.cast(updatedTime()));
        case "VolumeUsageInBytes":
            return Optional.ofNullable(clazz.cast(volumeUsageInBytes()));
        case "VolumeUsageUpdatedTime":
            return Optional.ofNullable(clazz.cast(volumeUsageUpdatedTime()));
        case "PercentOfGraphUtilization":
            return Optional.ofNullable(clazz.cast(percentOfGraphUtilization()));
        case "PercentOfGraphUtilizationUpdatedTime":
            return Optional.ofNullable(clazz.cast(percentOfGraphUtilizationUpdatedTime()));
        case "InvitationType":
            return Optional.ofNullable(clazz.cast(invitationTypeAsString()));
        case "VolumeUsageByDatasourcePackage":
            return Optional.ofNullable(clazz.cast(volumeUsageByDatasourcePackageAsStrings()));
        case "DatasourcePackageIngestStates":
            return Optional.ofNullable(clazz.cast(datasourcePackageIngestStatesAsStrings()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<MemberDetail, T> g) {
        return obj -> g.apply((MemberDetail) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, MemberDetail> {
        /**
         * <p>
         * The Amazon Web Services account identifier for the member account.
         * </p>
         * 
         * @param accountId
         *        The Amazon Web Services account identifier for the member account.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder accountId(String accountId);

        /**
         * <p>
         * The Amazon Web Services account root user email address for the member account.
         * </p>
         * 
         * @param emailAddress
         *        The Amazon Web Services account root user email address for the member account.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder emailAddress(String emailAddress);

        /**
         * <p>
         * The ARN of the behavior graph.
         * </p>
         * 
         * @param graphArn
         *        The ARN of the behavior graph.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder graphArn(String graphArn);

        /**
         * <p>
         * The Amazon Web Services account identifier of the administrator account for the behavior graph.
         * </p>
         * 
         * @param masterId
         *        The Amazon Web Services account identifier of the administrator account for the behavior graph.
         * @return Returns a reference to this object so that method calls can be chained together.
         * @deprecated This property is deprecated. Use AdministratorId instead.
         */
        @Deprecated
        Builder masterId(String masterId);

        /**
         * <p>
         * The Amazon Web Services account identifier of the administrator account for the behavior graph.
         * </p>
         * 
         * @param administratorId
         *        The Amazon Web Services account identifier of the administrator account for the behavior graph.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder administratorId(String administratorId);

        /**
         * <p>
         * The current membership status of the member account. The status can have one of the following values:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>INVITED</code> - For invited accounts only. Indicates that the member was sent an invitation but has
         * not yet responded.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>VERIFICATION_IN_PROGRESS</code> - For invited accounts only, indicates that Detective is verifying that
         * the account identifier and email address provided for the member account match. If they do match, then
         * Detective sends the invitation. If the email address and account identifier don't match, then the member
         * cannot be added to the behavior graph.
         * </p>
         * <p>
         * For organization accounts in the organization behavior graph, indicates that Detective is verifying that the
         * account belongs to the organization.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>VERIFICATION_FAILED</code> - For invited accounts only. Indicates that the account and email address
         * provided for the member account do not match, and Detective did not send an invitation to the account.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>ENABLED</code> - Indicates that the member account currently contributes data to the behavior graph.
         * For invited accounts, the member account accepted the invitation. For organization accounts in the
         * organization behavior graph, the Detective administrator account enabled the organization account as a member
         * account.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>ACCEPTED_BUT_DISABLED</code> - The account accepted the invitation, or was enabled by the Detective
         * administrator account, but is prevented from contributing data to the behavior graph.
         * <code>DisabledReason</code> provides the reason why the member account is not enabled.
         * </p>
         * </li>
         * </ul>
         * <p>
         * Invited accounts that declined an invitation or that were removed from the behavior graph are not included.
         * In the organization behavior graph, organization accounts that the Detective administrator account did not
         * enable are not included.
         * </p>
         * 
         * @param status
         *        The current membership status of the member account. The status can have one of the following
         *        values:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>INVITED</code> - For invited accounts only. Indicates that the member was sent an invitation but
         *        has not yet responded.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>VERIFICATION_IN_PROGRESS</code> - For invited accounts only, indicates that Detective is
         *        verifying that the account identifier and email address provided for the member account match. If they
         *        do match, then Detective sends the invitation. If the email address and account identifier don't
         *        match, then the member cannot be added to the behavior graph.
         *        </p>
         *        <p>
         *        For organization accounts in the organization behavior graph, indicates that Detective is verifying
         *        that the account belongs to the organization.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>VERIFICATION_FAILED</code> - For invited accounts only. Indicates that the account and email
         *        address provided for the member account do not match, and Detective did not send an invitation to the
         *        account.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>ENABLED</code> - Indicates that the member account currently contributes data to the behavior
         *        graph. For invited accounts, the member account accepted the invitation. For organization accounts in
         *        the organization behavior graph, the Detective administrator account enabled the organization account
         *        as a member account.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>ACCEPTED_BUT_DISABLED</code> - The account accepted the invitation, or was enabled by the
         *        Detective administrator account, but is prevented from contributing data to the behavior graph.
         *        <code>DisabledReason</code> provides the reason why the member account is not enabled.
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        Invited accounts that declined an invitation or that were removed from the behavior graph are not
         *        included. In the organization behavior graph, organization accounts that the Detective administrator
         *        account did not enable are not included.
         * @see MemberStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see MemberStatus
         */
        Builder status(String status);

        /**
         * <p>
         * The current membership status of the member account. The status can have one of the following values:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>INVITED</code> - For invited accounts only. Indicates that the member was sent an invitation but has
         * not yet responded.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>VERIFICATION_IN_PROGRESS</code> - For invited accounts only, indicates that Detective is verifying that
         * the account identifier and email address provided for the member account match. If they do match, then
         * Detective sends the invitation. If the email address and account identifier don't match, then the member
         * cannot be added to the behavior graph.
         * </p>
         * <p>
         * For organization accounts in the organization behavior graph, indicates that Detective is verifying that the
         * account belongs to the organization.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>VERIFICATION_FAILED</code> - For invited accounts only. Indicates that the account and email address
         * provided for the member account do not match, and Detective did not send an invitation to the account.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>ENABLED</code> - Indicates that the member account currently contributes data to the behavior graph.
         * For invited accounts, the member account accepted the invitation. For organization accounts in the
         * organization behavior graph, the Detective administrator account enabled the organization account as a member
         * account.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>ACCEPTED_BUT_DISABLED</code> - The account accepted the invitation, or was enabled by the Detective
         * administrator account, but is prevented from contributing data to the behavior graph.
         * <code>DisabledReason</code> provides the reason why the member account is not enabled.
         * </p>
         * </li>
         * </ul>
         * <p>
         * Invited accounts that declined an invitation or that were removed from the behavior graph are not included.
         * In the organization behavior graph, organization accounts that the Detective administrator account did not
         * enable are not included.
         * </p>
         * 
         * @param status
         *        The current membership status of the member account. The status can have one of the following
         *        values:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>INVITED</code> - For invited accounts only. Indicates that the member was sent an invitation but
         *        has not yet responded.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>VERIFICATION_IN_PROGRESS</code> - For invited accounts only, indicates that Detective is
         *        verifying that the account identifier and email address provided for the member account match. If they
         *        do match, then Detective sends the invitation. If the email address and account identifier don't
         *        match, then the member cannot be added to the behavior graph.
         *        </p>
         *        <p>
         *        For organization accounts in the organization behavior graph, indicates that Detective is verifying
         *        that the account belongs to the organization.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>VERIFICATION_FAILED</code> - For invited accounts only. Indicates that the account and email
         *        address provided for the member account do not match, and Detective did not send an invitation to the
         *        account.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>ENABLED</code> - Indicates that the member account currently contributes data to the behavior
         *        graph. For invited accounts, the member account accepted the invitation. For organization accounts in
         *        the organization behavior graph, the Detective administrator account enabled the organization account
         *        as a member account.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>ACCEPTED_BUT_DISABLED</code> - The account accepted the invitation, or was enabled by the
         *        Detective administrator account, but is prevented from contributing data to the behavior graph.
         *        <code>DisabledReason</code> provides the reason why the member account is not enabled.
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        Invited accounts that declined an invitation or that were removed from the behavior graph are not
         *        included. In the organization behavior graph, organization accounts that the Detective administrator
         *        account did not enable are not included.
         * @see MemberStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see MemberStatus
         */
        Builder status(MemberStatus status);

        /**
         * <p>
         * For member accounts with a status of <code>ACCEPTED_BUT_DISABLED</code>, the reason that the member account
         * is not enabled.
         * </p>
         * <p>
         * The reason can have one of the following values:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>VOLUME_TOO_HIGH</code> - Indicates that adding the member account would cause the data volume for the
         * behavior graph to be too high.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>VOLUME_UNKNOWN</code> - Indicates that Detective is unable to verify the data volume for the member
         * account. This is usually because the member account is not enrolled in Amazon GuardDuty.
         * </p>
         * </li>
         * </ul>
         * 
         * @param disabledReason
         *        For member accounts with a status of <code>ACCEPTED_BUT_DISABLED</code>, the reason that the member
         *        account is not enabled.</p>
         *        <p>
         *        The reason can have one of the following values:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>VOLUME_TOO_HIGH</code> - Indicates that adding the member account would cause the data volume
         *        for the behavior graph to be too high.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>VOLUME_UNKNOWN</code> - Indicates that Detective is unable to verify the data volume for the
         *        member account. This is usually because the member account is not enrolled in Amazon GuardDuty.
         *        </p>
         *        </li>
         * @see MemberDisabledReason
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see MemberDisabledReason
         */
        Builder disabledReason(String disabledReason);

        /**
         * <p>
         * For member accounts with a status of <code>ACCEPTED_BUT_DISABLED</code>, the reason that the member account
         * is not enabled.
         * </p>
         * <p>
         * The reason can have one of the following values:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>VOLUME_TOO_HIGH</code> - Indicates that adding the member account would cause the data volume for the
         * behavior graph to be too high.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>VOLUME_UNKNOWN</code> - Indicates that Detective is unable to verify the data volume for the member
         * account. This is usually because the member account is not enrolled in Amazon GuardDuty.
         * </p>
         * </li>
         * </ul>
         * 
         * @param disabledReason
         *        For member accounts with a status of <code>ACCEPTED_BUT_DISABLED</code>, the reason that the member
         *        account is not enabled.</p>
         *        <p>
         *        The reason can have one of the following values:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>VOLUME_TOO_HIGH</code> - Indicates that adding the member account would cause the data volume
         *        for the behavior graph to be too high.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>VOLUME_UNKNOWN</code> - Indicates that Detective is unable to verify the data volume for the
         *        member account. This is usually because the member account is not enrolled in Amazon GuardDuty.
         *        </p>
         *        </li>
         * @see MemberDisabledReason
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see MemberDisabledReason
         */
        Builder disabledReason(MemberDisabledReason disabledReason);

        /**
         * <p>
         * For invited accounts, the date and time that Detective sent the invitation to the account. The value is an
         * ISO8601 formatted string. For example, <code>2021-08-18T16:35:56.284Z</code>.
         * </p>
         * 
         * @param invitedTime
         *        For invited accounts, the date and time that Detective sent the invitation to the account. The value
         *        is an ISO8601 formatted string. For example, <code>2021-08-18T16:35:56.284Z</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder invitedTime(Instant invitedTime);

        /**
         * <p>
         * The date and time that the member account was last updated. The value is an ISO8601 formatted string. For
         * example, <code>2021-08-18T16:35:56.284Z</code>.
         * </p>
         * 
         * @param updatedTime
         *        The date and time that the member account was last updated. The value is an ISO8601 formatted string.
         *        For example, <code>2021-08-18T16:35:56.284Z</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder updatedTime(Instant updatedTime);

        /**
         * <p>
         * The data volume in bytes per day for the member account.
         * </p>
         * 
         * @param volumeUsageInBytes
         *        The data volume in bytes per day for the member account.
         * @return Returns a reference to this object so that method calls can be chained together.
         * @deprecated This property is deprecated. Use VolumeUsageByDatasourcePackage instead.
         */
        @Deprecated
        Builder volumeUsageInBytes(Long volumeUsageInBytes);

        /**
         * <p>
         * The data and time when the member account data volume was last updated. The value is an ISO8601 formatted
         * string. For example, <code>2021-08-18T16:35:56.284Z</code>.
         * </p>
         * 
         * @param volumeUsageUpdatedTime
         *        The data and time when the member account data volume was last updated. The value is an ISO8601
         *        formatted string. For example, <code>2021-08-18T16:35:56.284Z</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         * @deprecated This property is deprecated. Use VolumeUsageByDatasourcePackage instead.
         */
        @Deprecated
        Builder volumeUsageUpdatedTime(Instant volumeUsageUpdatedTime);

        /**
         * <p>
         * The member account data volume as a percentage of the maximum allowed data volume. 0 indicates 0 percent, and
         * 100 indicates 100 percent.
         * </p>
         * <p>
         * Note that this is not the percentage of the behavior graph data volume.
         * </p>
         * <p>
         * For example, the data volume for the behavior graph is 80 GB per day. The maximum data volume is 160 GB per
         * day. If the data volume for the member account is 40 GB per day, then <code>PercentOfGraphUtilization</code>
         * is 25. It represents 25% of the maximum allowed data volume.
         * </p>
         * 
         * @param percentOfGraphUtilization
         *        The member account data volume as a percentage of the maximum allowed data volume. 0 indicates 0
         *        percent, and 100 indicates 100 percent.</p>
         *        <p>
         *        Note that this is not the percentage of the behavior graph data volume.
         *        </p>
         *        <p>
         *        For example, the data volume for the behavior graph is 80 GB per day. The maximum data volume is 160
         *        GB per day. If the data volume for the member account is 40 GB per day, then
         *        <code>PercentOfGraphUtilization</code> is 25. It represents 25% of the maximum allowed data volume.
         * @return Returns a reference to this object so that method calls can be chained together.
         * @deprecated This property is deprecated. Use VolumeUsageByDatasourcePackage instead.
         */
        @Deprecated
        Builder percentOfGraphUtilization(Double percentOfGraphUtilization);

        /**
         * <p>
         * The date and time when the graph utilization percentage was last updated. The value is an ISO8601 formatted
         * string. For example, <code>2021-08-18T16:35:56.284Z</code>.
         * </p>
         * 
         * @param percentOfGraphUtilizationUpdatedTime
         *        The date and time when the graph utilization percentage was last updated. The value is an ISO8601
         *        formatted string. For example, <code>2021-08-18T16:35:56.284Z</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         * @deprecated This property is deprecated. Use VolumeUsageByDatasourcePackage instead.
         */
        @Deprecated
        Builder percentOfGraphUtilizationUpdatedTime(Instant percentOfGraphUtilizationUpdatedTime);

        /**
         * <p>
         * The type of behavior graph membership.
         * </p>
         * <p>
         * For an organization account in the organization behavior graph, the type is <code>ORGANIZATION</code>.
         * </p>
         * <p>
         * For an account that was invited to a behavior graph, the type is <code>INVITATION</code>.
         * </p>
         * 
         * @param invitationType
         *        The type of behavior graph membership.</p>
         *        <p>
         *        For an organization account in the organization behavior graph, the type is <code>ORGANIZATION</code>.
         *        </p>
         *        <p>
         *        For an account that was invited to a behavior graph, the type is <code>INVITATION</code>.
         * @see InvitationType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see InvitationType
         */
        Builder invitationType(String invitationType);

        /**
         * <p>
         * The type of behavior graph membership.
         * </p>
         * <p>
         * For an organization account in the organization behavior graph, the type is <code>ORGANIZATION</code>.
         * </p>
         * <p>
         * For an account that was invited to a behavior graph, the type is <code>INVITATION</code>.
         * </p>
         * 
         * @param invitationType
         *        The type of behavior graph membership.</p>
         *        <p>
         *        For an organization account in the organization behavior graph, the type is <code>ORGANIZATION</code>.
         *        </p>
         *        <p>
         *        For an account that was invited to a behavior graph, the type is <code>INVITATION</code>.
         * @see InvitationType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see InvitationType
         */
        Builder invitationType(InvitationType invitationType);

        /**
         * <p>
         * Details on the volume of usage for each data source package in a behavior graph.
         * </p>
         * 
         * @param volumeUsageByDatasourcePackage
         *        Details on the volume of usage for each data source package in a behavior graph.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder volumeUsageByDatasourcePackageWithStrings(Map<String, DatasourcePackageUsageInfo> volumeUsageByDatasourcePackage);

        /**
         * <p>
         * Details on the volume of usage for each data source package in a behavior graph.
         * </p>
         * 
         * @param volumeUsageByDatasourcePackage
         *        Details on the volume of usage for each data source package in a behavior graph.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder volumeUsageByDatasourcePackage(Map<DatasourcePackage, DatasourcePackageUsageInfo> volumeUsageByDatasourcePackage);

        /**
         * <p>
         * The state of a data source package for the behavior graph.
         * </p>
         * 
         * @param datasourcePackageIngestStates
         *        The state of a data source package for the behavior graph.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder datasourcePackageIngestStatesWithStrings(Map<String, String> datasourcePackageIngestStates);

        /**
         * <p>
         * The state of a data source package for the behavior graph.
         * </p>
         * 
         * @param datasourcePackageIngestStates
         *        The state of a data source package for the behavior graph.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder datasourcePackageIngestStates(Map<DatasourcePackage, DatasourcePackageIngestState> datasourcePackageIngestStates);
    }

    static final class BuilderImpl implements Builder {
        private String accountId;

        private String emailAddress;

        private String graphArn;

        private String masterId;

        private String administratorId;

        private String status;

        private String disabledReason;

        private Instant invitedTime;

        private Instant updatedTime;

        private Long volumeUsageInBytes;

        private Instant volumeUsageUpdatedTime;

        private Double percentOfGraphUtilization;

        private Instant percentOfGraphUtilizationUpdatedTime;

        private String invitationType;

        private Map<String, DatasourcePackageUsageInfo> volumeUsageByDatasourcePackage = DefaultSdkAutoConstructMap.getInstance();

        private Map<String, String> datasourcePackageIngestStates = DefaultSdkAutoConstructMap.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(MemberDetail model) {
            accountId(model.accountId);
            emailAddress(model.emailAddress);
            graphArn(model.graphArn);
            masterId(model.masterId);
            administratorId(model.administratorId);
            status(model.status);
            disabledReason(model.disabledReason);
            invitedTime(model.invitedTime);
            updatedTime(model.updatedTime);
            volumeUsageInBytes(model.volumeUsageInBytes);
            volumeUsageUpdatedTime(model.volumeUsageUpdatedTime);
            percentOfGraphUtilization(model.percentOfGraphUtilization);
            percentOfGraphUtilizationUpdatedTime(model.percentOfGraphUtilizationUpdatedTime);
            invitationType(model.invitationType);
            volumeUsageByDatasourcePackageWithStrings(model.volumeUsageByDatasourcePackage);
            datasourcePackageIngestStatesWithStrings(model.datasourcePackageIngestStates);
        }

        public final String getAccountId() {
            return accountId;
        }

        public final void setAccountId(String accountId) {
            this.accountId = accountId;
        }

        @Override
        public final Builder accountId(String accountId) {
            this.accountId = accountId;
            return this;
        }

        public final String getEmailAddress() {
            return emailAddress;
        }

        public final void setEmailAddress(String emailAddress) {
            this.emailAddress = emailAddress;
        }

        @Override
        public final Builder emailAddress(String emailAddress) {
            this.emailAddress = emailAddress;
            return this;
        }

        public final String getGraphArn() {
            return graphArn;
        }

        public final void setGraphArn(String graphArn) {
            this.graphArn = graphArn;
        }

        @Override
        public final Builder graphArn(String graphArn) {
            this.graphArn = graphArn;
            return this;
        }

        @Deprecated
        public final String getMasterId() {
            return masterId;
        }

        @Deprecated
        public final void setMasterId(String masterId) {
            this.masterId = masterId;
        }

        @Override
        @Deprecated
        public final Builder masterId(String masterId) {
            this.masterId = masterId;
            return this;
        }

        public final String getAdministratorId() {
            return administratorId;
        }

        public final void setAdministratorId(String administratorId) {
            this.administratorId = administratorId;
        }

        @Override
        public final Builder administratorId(String administratorId) {
            this.administratorId = administratorId;
            return this;
        }

        public final String getStatus() {
            return status;
        }

        public final void setStatus(String status) {
            this.status = status;
        }

        @Override
        public final Builder status(String status) {
            this.status = status;
            return this;
        }

        @Override
        public final Builder status(MemberStatus status) {
            this.status(status == null ? null : status.toString());
            return this;
        }

        public final String getDisabledReason() {
            return disabledReason;
        }

        public final void setDisabledReason(String disabledReason) {
            this.disabledReason = disabledReason;
        }

        @Override
        public final Builder disabledReason(String disabledReason) {
            this.disabledReason = disabledReason;
            return this;
        }

        @Override
        public final Builder disabledReason(MemberDisabledReason disabledReason) {
            this.disabledReason(disabledReason == null ? null : disabledReason.toString());
            return this;
        }

        public final Instant getInvitedTime() {
            return invitedTime;
        }

        public final void setInvitedTime(Instant invitedTime) {
            this.invitedTime = invitedTime;
        }

        @Override
        public final Builder invitedTime(Instant invitedTime) {
            this.invitedTime = invitedTime;
            return this;
        }

        public final Instant getUpdatedTime() {
            return updatedTime;
        }

        public final void setUpdatedTime(Instant updatedTime) {
            this.updatedTime = updatedTime;
        }

        @Override
        public final Builder updatedTime(Instant updatedTime) {
            this.updatedTime = updatedTime;
            return this;
        }

        @Deprecated
        public final Long getVolumeUsageInBytes() {
            return volumeUsageInBytes;
        }

        @Deprecated
        public final void setVolumeUsageInBytes(Long volumeUsageInBytes) {
            this.volumeUsageInBytes = volumeUsageInBytes;
        }

        @Override
        @Deprecated
        public final Builder volumeUsageInBytes(Long volumeUsageInBytes) {
            this.volumeUsageInBytes = volumeUsageInBytes;
            return this;
        }

        @Deprecated
        public final Instant getVolumeUsageUpdatedTime() {
            return volumeUsageUpdatedTime;
        }

        @Deprecated
        public final void setVolumeUsageUpdatedTime(Instant volumeUsageUpdatedTime) {
            this.volumeUsageUpdatedTime = volumeUsageUpdatedTime;
        }

        @Override
        @Deprecated
        public final Builder volumeUsageUpdatedTime(Instant volumeUsageUpdatedTime) {
            this.volumeUsageUpdatedTime = volumeUsageUpdatedTime;
            return this;
        }

        @Deprecated
        public final Double getPercentOfGraphUtilization() {
            return percentOfGraphUtilization;
        }

        @Deprecated
        public final void setPercentOfGraphUtilization(Double percentOfGraphUtilization) {
            this.percentOfGraphUtilization = percentOfGraphUtilization;
        }

        @Override
        @Deprecated
        public final Builder percentOfGraphUtilization(Double percentOfGraphUtilization) {
            this.percentOfGraphUtilization = percentOfGraphUtilization;
            return this;
        }

        @Deprecated
        public final Instant getPercentOfGraphUtilizationUpdatedTime() {
            return percentOfGraphUtilizationUpdatedTime;
        }

        @Deprecated
        public final void setPercentOfGraphUtilizationUpdatedTime(Instant percentOfGraphUtilizationUpdatedTime) {
            this.percentOfGraphUtilizationUpdatedTime = percentOfGraphUtilizationUpdatedTime;
        }

        @Override
        @Deprecated
        public final Builder percentOfGraphUtilizationUpdatedTime(Instant percentOfGraphUtilizationUpdatedTime) {
            this.percentOfGraphUtilizationUpdatedTime = percentOfGraphUtilizationUpdatedTime;
            return this;
        }

        public final String getInvitationType() {
            return invitationType;
        }

        public final void setInvitationType(String invitationType) {
            this.invitationType = invitationType;
        }

        @Override
        public final Builder invitationType(String invitationType) {
            this.invitationType = invitationType;
            return this;
        }

        @Override
        public final Builder invitationType(InvitationType invitationType) {
            this.invitationType(invitationType == null ? null : invitationType.toString());
            return this;
        }

        public final Map<String, DatasourcePackageUsageInfo.Builder> getVolumeUsageByDatasourcePackage() {
            Map<String, DatasourcePackageUsageInfo.Builder> result = VolumeUsageByDatasourcePackageCopier
                    .copyToBuilder(this.volumeUsageByDatasourcePackage);
            if (result instanceof SdkAutoConstructMap) {
                return null;
            }
            return result;
        }

        public final void setVolumeUsageByDatasourcePackage(
                Map<String, DatasourcePackageUsageInfo.BuilderImpl> volumeUsageByDatasourcePackage) {
            this.volumeUsageByDatasourcePackage = VolumeUsageByDatasourcePackageCopier
                    .copyFromBuilder(volumeUsageByDatasourcePackage);
        }

        @Override
        public final Builder volumeUsageByDatasourcePackageWithStrings(
                Map<String, DatasourcePackageUsageInfo> volumeUsageByDatasourcePackage) {
            this.volumeUsageByDatasourcePackage = VolumeUsageByDatasourcePackageCopier.copy(volumeUsageByDatasourcePackage);
            return this;
        }

        @Override
        public final Builder volumeUsageByDatasourcePackage(
                Map<DatasourcePackage, DatasourcePackageUsageInfo> volumeUsageByDatasourcePackage) {
            this.volumeUsageByDatasourcePackage = VolumeUsageByDatasourcePackageCopier
                    .copyEnumToString(volumeUsageByDatasourcePackage);
            return this;
        }

        public final Map<String, String> getDatasourcePackageIngestStates() {
            if (datasourcePackageIngestStates instanceof SdkAutoConstructMap) {
                return null;
            }
            return datasourcePackageIngestStates;
        }

        public final void setDatasourcePackageIngestStates(Map<String, String> datasourcePackageIngestStates) {
            this.datasourcePackageIngestStates = DatasourcePackageIngestStatesCopier.copy(datasourcePackageIngestStates);
        }

        @Override
        public final Builder datasourcePackageIngestStatesWithStrings(Map<String, String> datasourcePackageIngestStates) {
            this.datasourcePackageIngestStates = DatasourcePackageIngestStatesCopier.copy(datasourcePackageIngestStates);
            return this;
        }

        @Override
        public final Builder datasourcePackageIngestStates(
                Map<DatasourcePackage, DatasourcePackageIngestState> datasourcePackageIngestStates) {
            this.datasourcePackageIngestStates = DatasourcePackageIngestStatesCopier
                    .copyEnumToString(datasourcePackageIngestStates);
            return this;
        }

        @Override
        public MemberDetail build() {
            return new MemberDetail(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
