/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.controltower;

import java.util.Collections;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.client.handler.AwsAsyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.awscore.internal.AwsProtocolMetadata;
import software.amazon.awssdk.awscore.internal.AwsServiceProtocol;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkPlugin;
import software.amazon.awssdk.core.SdkRequest;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.AsyncClientHandler;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.controltower.internal.ControlTowerServiceClientConfigurationBuilder;
import software.amazon.awssdk.services.controltower.model.AccessDeniedException;
import software.amazon.awssdk.services.controltower.model.ConflictException;
import software.amazon.awssdk.services.controltower.model.ControlTowerException;
import software.amazon.awssdk.services.controltower.model.CreateLandingZoneRequest;
import software.amazon.awssdk.services.controltower.model.CreateLandingZoneResponse;
import software.amazon.awssdk.services.controltower.model.DeleteLandingZoneRequest;
import software.amazon.awssdk.services.controltower.model.DeleteLandingZoneResponse;
import software.amazon.awssdk.services.controltower.model.DisableControlRequest;
import software.amazon.awssdk.services.controltower.model.DisableControlResponse;
import software.amazon.awssdk.services.controltower.model.EnableControlRequest;
import software.amazon.awssdk.services.controltower.model.EnableControlResponse;
import software.amazon.awssdk.services.controltower.model.GetControlOperationRequest;
import software.amazon.awssdk.services.controltower.model.GetControlOperationResponse;
import software.amazon.awssdk.services.controltower.model.GetEnabledControlRequest;
import software.amazon.awssdk.services.controltower.model.GetEnabledControlResponse;
import software.amazon.awssdk.services.controltower.model.GetLandingZoneOperationRequest;
import software.amazon.awssdk.services.controltower.model.GetLandingZoneOperationResponse;
import software.amazon.awssdk.services.controltower.model.GetLandingZoneRequest;
import software.amazon.awssdk.services.controltower.model.GetLandingZoneResponse;
import software.amazon.awssdk.services.controltower.model.InternalServerException;
import software.amazon.awssdk.services.controltower.model.ListEnabledControlsRequest;
import software.amazon.awssdk.services.controltower.model.ListEnabledControlsResponse;
import software.amazon.awssdk.services.controltower.model.ListLandingZonesRequest;
import software.amazon.awssdk.services.controltower.model.ListLandingZonesResponse;
import software.amazon.awssdk.services.controltower.model.ListTagsForResourceRequest;
import software.amazon.awssdk.services.controltower.model.ListTagsForResourceResponse;
import software.amazon.awssdk.services.controltower.model.ResetLandingZoneRequest;
import software.amazon.awssdk.services.controltower.model.ResetLandingZoneResponse;
import software.amazon.awssdk.services.controltower.model.ResourceNotFoundException;
import software.amazon.awssdk.services.controltower.model.ServiceQuotaExceededException;
import software.amazon.awssdk.services.controltower.model.TagResourceRequest;
import software.amazon.awssdk.services.controltower.model.TagResourceResponse;
import software.amazon.awssdk.services.controltower.model.ThrottlingException;
import software.amazon.awssdk.services.controltower.model.UntagResourceRequest;
import software.amazon.awssdk.services.controltower.model.UntagResourceResponse;
import software.amazon.awssdk.services.controltower.model.UpdateLandingZoneRequest;
import software.amazon.awssdk.services.controltower.model.UpdateLandingZoneResponse;
import software.amazon.awssdk.services.controltower.model.ValidationException;
import software.amazon.awssdk.services.controltower.transform.CreateLandingZoneRequestMarshaller;
import software.amazon.awssdk.services.controltower.transform.DeleteLandingZoneRequestMarshaller;
import software.amazon.awssdk.services.controltower.transform.DisableControlRequestMarshaller;
import software.amazon.awssdk.services.controltower.transform.EnableControlRequestMarshaller;
import software.amazon.awssdk.services.controltower.transform.GetControlOperationRequestMarshaller;
import software.amazon.awssdk.services.controltower.transform.GetEnabledControlRequestMarshaller;
import software.amazon.awssdk.services.controltower.transform.GetLandingZoneOperationRequestMarshaller;
import software.amazon.awssdk.services.controltower.transform.GetLandingZoneRequestMarshaller;
import software.amazon.awssdk.services.controltower.transform.ListEnabledControlsRequestMarshaller;
import software.amazon.awssdk.services.controltower.transform.ListLandingZonesRequestMarshaller;
import software.amazon.awssdk.services.controltower.transform.ListTagsForResourceRequestMarshaller;
import software.amazon.awssdk.services.controltower.transform.ResetLandingZoneRequestMarshaller;
import software.amazon.awssdk.services.controltower.transform.TagResourceRequestMarshaller;
import software.amazon.awssdk.services.controltower.transform.UntagResourceRequestMarshaller;
import software.amazon.awssdk.services.controltower.transform.UpdateLandingZoneRequestMarshaller;
import software.amazon.awssdk.utils.CompletableFutureUtils;

/**
 * Internal implementation of {@link ControlTowerAsyncClient}.
 *
 * @see ControlTowerAsyncClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultControlTowerAsyncClient implements ControlTowerAsyncClient {
    private static final Logger log = LoggerFactory.getLogger(DefaultControlTowerAsyncClient.class);

    private static final AwsProtocolMetadata protocolMetadata = AwsProtocolMetadata.builder()
            .serviceProtocol(AwsServiceProtocol.REST_JSON).build();

    private final AsyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultControlTowerAsyncClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsAsyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    /**
     * <p>
     * Creates a new landing zone. This starts an asynchronous operation that creates and configures a landing zone
     * based on the parameters specified in the manifest JSON file.
     * </p>
     *
     * @param createLandingZoneRequest
     * @return A Java Future containing the result of the CreateLandingZone operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The input fails to satisfy the constraints specified by an Amazon Web Services
     *         service.</li>
     *         <li>ConflictException Updating or deleting a resource can cause an inconsistent state.</li>
     *         <li>InternalServerException Unexpected error during processing of request.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>ThrottlingException Request was denied due to request throttling.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ControlTowerException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample ControlTowerAsyncClient.CreateLandingZone
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/controltower-2018-05-10/CreateLandingZone"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateLandingZoneResponse> createLandingZone(CreateLandingZoneRequest createLandingZoneRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createLandingZoneRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createLandingZoneRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ControlTower");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateLandingZone");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateLandingZoneResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateLandingZoneResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateLandingZoneResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateLandingZoneRequest, CreateLandingZoneResponse>()
                            .withOperationName("CreateLandingZone").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new CreateLandingZoneRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(createLandingZoneRequest));
            CompletableFuture<CreateLandingZoneResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * This decommissions a landing zone. This starts an asynchronous operation that deletes Amazon Web Services Control
     * Tower resources deployed in Amazon Web Services Control Tower managed accounts.
     * </p>
     *
     * @param deleteLandingZoneRequest
     * @return A Java Future containing the result of the DeleteLandingZone operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The input fails to satisfy the constraints specified by an Amazon Web Services
     *         service.</li>
     *         <li>ConflictException Updating or deleting a resource can cause an inconsistent state.</li>
     *         <li>InternalServerException Unexpected error during processing of request.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>ThrottlingException Request was denied due to request throttling.</li>
     *         <li>ResourceNotFoundException Request references a resource which does not exist.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ControlTowerException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample ControlTowerAsyncClient.DeleteLandingZone
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/controltower-2018-05-10/DeleteLandingZone"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteLandingZoneResponse> deleteLandingZone(DeleteLandingZoneRequest deleteLandingZoneRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteLandingZoneRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteLandingZoneRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ControlTower");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteLandingZone");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteLandingZoneResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteLandingZoneResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteLandingZoneResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteLandingZoneRequest, DeleteLandingZoneResponse>()
                            .withOperationName("DeleteLandingZone").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DeleteLandingZoneRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deleteLandingZoneRequest));
            CompletableFuture<DeleteLandingZoneResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * This API call turns off a control. It starts an asynchronous operation that deletes Amazon Web Services resources
     * on the specified organizational unit and the accounts it contains. The resources will vary according to the
     * control that you specify. For usage examples, see <a
     * href="https://docs.aws.amazon.com/controltower/latest/userguide/control-api-examples-short.html"> <i>the Amazon
     * Web Services Control Tower User Guide</i> </a>.
     * </p>
     *
     * @param disableControlRequest
     * @return A Java Future containing the result of the DisableControl operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The input fails to satisfy the constraints specified by an Amazon Web Services
     *         service.</li>
     *         <li>ConflictException Updating or deleting a resource can cause an inconsistent state.</li>
     *         <li>ServiceQuotaExceededException Request would cause a service quota to be exceeded. The limit is 10
     *         concurrent operations.</li>
     *         <li>InternalServerException Unexpected error during processing of request.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>ThrottlingException Request was denied due to request throttling.</li>
     *         <li>ResourceNotFoundException Request references a resource which does not exist.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ControlTowerException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample ControlTowerAsyncClient.DisableControl
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/controltower-2018-05-10/DisableControl" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<DisableControlResponse> disableControl(DisableControlRequest disableControlRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(disableControlRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, disableControlRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ControlTower");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DisableControl");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DisableControlResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DisableControlResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DisableControlResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DisableControlRequest, DisableControlResponse>()
                            .withOperationName("DisableControl").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DisableControlRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(disableControlRequest));
            CompletableFuture<DisableControlResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * This API call activates a control. It starts an asynchronous operation that creates Amazon Web Services resources
     * on the specified organizational unit and the accounts it contains. The resources created will vary according to
     * the control that you specify. For usage examples, see <a
     * href="https://docs.aws.amazon.com/controltower/latest/userguide/control-api-examples-short.html"> <i>the Amazon
     * Web Services Control Tower User Guide</i> </a>.
     * </p>
     *
     * @param enableControlRequest
     * @return A Java Future containing the result of the EnableControl operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The input fails to satisfy the constraints specified by an Amazon Web Services
     *         service.</li>
     *         <li>ConflictException Updating or deleting a resource can cause an inconsistent state.</li>
     *         <li>ServiceQuotaExceededException Request would cause a service quota to be exceeded. The limit is 10
     *         concurrent operations.</li>
     *         <li>InternalServerException Unexpected error during processing of request.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>ThrottlingException Request was denied due to request throttling.</li>
     *         <li>ResourceNotFoundException Request references a resource which does not exist.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ControlTowerException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample ControlTowerAsyncClient.EnableControl
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/controltower-2018-05-10/EnableControl" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<EnableControlResponse> enableControl(EnableControlRequest enableControlRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(enableControlRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, enableControlRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ControlTower");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "EnableControl");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<EnableControlResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    EnableControlResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<EnableControlResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<EnableControlRequest, EnableControlResponse>()
                            .withOperationName("EnableControl").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new EnableControlRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(enableControlRequest));
            CompletableFuture<EnableControlResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns the status of a particular <code>EnableControl</code> or <code>DisableControl</code> operation. Displays
     * a message in case of error. Details for an operation are available for 90 days. For usage examples, see <a
     * href="https://docs.aws.amazon.com/controltower/latest/userguide/control-api-examples-short.html"> <i>the Amazon
     * Web Services Control Tower User Guide</i> </a>.
     * </p>
     *
     * @param getControlOperationRequest
     * @return A Java Future containing the result of the GetControlOperation operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The input fails to satisfy the constraints specified by an Amazon Web Services
     *         service.</li>
     *         <li>InternalServerException Unexpected error during processing of request.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>ThrottlingException Request was denied due to request throttling.</li>
     *         <li>ResourceNotFoundException Request references a resource which does not exist.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ControlTowerException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample ControlTowerAsyncClient.GetControlOperation
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/controltower-2018-05-10/GetControlOperation"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetControlOperationResponse> getControlOperation(
            GetControlOperationRequest getControlOperationRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getControlOperationRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getControlOperationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ControlTower");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetControlOperation");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetControlOperationResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetControlOperationResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetControlOperationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetControlOperationRequest, GetControlOperationResponse>()
                            .withOperationName("GetControlOperation").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetControlOperationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getControlOperationRequest));
            CompletableFuture<GetControlOperationResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves details about an enabled control. For usage examples, see <a
     * href="https://docs.aws.amazon.com/controltower/latest/userguide/control-api-examples-short.html"> <i>the Amazon
     * Web Services Control Tower User Guide</i> </a>.
     * </p>
     *
     * @param getEnabledControlRequest
     * @return A Java Future containing the result of the GetEnabledControl operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The input fails to satisfy the constraints specified by an Amazon Web Services
     *         service.</li>
     *         <li>InternalServerException Unexpected error during processing of request.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>ThrottlingException Request was denied due to request throttling.</li>
     *         <li>ResourceNotFoundException Request references a resource which does not exist.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ControlTowerException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample ControlTowerAsyncClient.GetEnabledControl
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/controltower-2018-05-10/GetEnabledControl"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetEnabledControlResponse> getEnabledControl(GetEnabledControlRequest getEnabledControlRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getEnabledControlRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getEnabledControlRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ControlTower");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetEnabledControl");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetEnabledControlResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetEnabledControlResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetEnabledControlResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetEnabledControlRequest, GetEnabledControlResponse>()
                            .withOperationName("GetEnabledControl").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetEnabledControlRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getEnabledControlRequest));
            CompletableFuture<GetEnabledControlResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns details about the landing zone. Displays a message in case of error.
     * </p>
     *
     * @param getLandingZoneRequest
     * @return A Java Future containing the result of the GetLandingZone operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The input fails to satisfy the constraints specified by an Amazon Web Services
     *         service.</li>
     *         <li>InternalServerException Unexpected error during processing of request.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>ThrottlingException Request was denied due to request throttling.</li>
     *         <li>ResourceNotFoundException Request references a resource which does not exist.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ControlTowerException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample ControlTowerAsyncClient.GetLandingZone
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/controltower-2018-05-10/GetLandingZone" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<GetLandingZoneResponse> getLandingZone(GetLandingZoneRequest getLandingZoneRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getLandingZoneRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getLandingZoneRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ControlTower");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetLandingZone");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetLandingZoneResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetLandingZoneResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetLandingZoneResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetLandingZoneRequest, GetLandingZoneResponse>()
                            .withOperationName("GetLandingZone").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetLandingZoneRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getLandingZoneRequest));
            CompletableFuture<GetLandingZoneResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns the status of the specified landing zone operation. Details for an operation are available for X days.
     * </p>
     *
     * @param getLandingZoneOperationRequest
     * @return A Java Future containing the result of the GetLandingZoneOperation operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The input fails to satisfy the constraints specified by an Amazon Web Services
     *         service.</li>
     *         <li>InternalServerException Unexpected error during processing of request.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>ThrottlingException Request was denied due to request throttling.</li>
     *         <li>ResourceNotFoundException Request references a resource which does not exist.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ControlTowerException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample ControlTowerAsyncClient.GetLandingZoneOperation
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/controltower-2018-05-10/GetLandingZoneOperation"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetLandingZoneOperationResponse> getLandingZoneOperation(
            GetLandingZoneOperationRequest getLandingZoneOperationRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getLandingZoneOperationRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getLandingZoneOperationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ControlTower");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetLandingZoneOperation");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetLandingZoneOperationResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetLandingZoneOperationResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetLandingZoneOperationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetLandingZoneOperationRequest, GetLandingZoneOperationResponse>()
                            .withOperationName("GetLandingZoneOperation").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetLandingZoneOperationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getLandingZoneOperationRequest));
            CompletableFuture<GetLandingZoneOperationResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists the controls enabled by Amazon Web Services Control Tower on the specified organizational unit and the
     * accounts it contains. For usage examples, see <a
     * href="https://docs.aws.amazon.com/controltower/latest/userguide/control-api-examples-short.html"> <i>the Amazon
     * Web Services Control Tower User Guide</i> </a>.
     * </p>
     *
     * @param listEnabledControlsRequest
     * @return A Java Future containing the result of the ListEnabledControls operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The input fails to satisfy the constraints specified by an Amazon Web Services
     *         service.</li>
     *         <li>InternalServerException Unexpected error during processing of request.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>ThrottlingException Request was denied due to request throttling.</li>
     *         <li>ResourceNotFoundException Request references a resource which does not exist.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ControlTowerException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample ControlTowerAsyncClient.ListEnabledControls
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/controltower-2018-05-10/ListEnabledControls"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListEnabledControlsResponse> listEnabledControls(
            ListEnabledControlsRequest listEnabledControlsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listEnabledControlsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listEnabledControlsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ControlTower");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListEnabledControls");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListEnabledControlsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListEnabledControlsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListEnabledControlsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListEnabledControlsRequest, ListEnabledControlsResponse>()
                            .withOperationName("ListEnabledControls").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListEnabledControlsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listEnabledControlsRequest));
            CompletableFuture<ListEnabledControlsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns the landing zone ARN for the landing zone deployed in your managed account. This API also creates an ARN
     * for existing accounts that do not yet have a landing zone ARN.
     * </p>
     * <p>
     * The return limit is one landing zone ARN.
     * </p>
     *
     * @param listLandingZonesRequest
     * @return A Java Future containing the result of the ListLandingZones operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The input fails to satisfy the constraints specified by an Amazon Web Services
     *         service.</li>
     *         <li>InternalServerException Unexpected error during processing of request.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>ThrottlingException Request was denied due to request throttling.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ControlTowerException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample ControlTowerAsyncClient.ListLandingZones
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/controltower-2018-05-10/ListLandingZones" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<ListLandingZonesResponse> listLandingZones(ListLandingZonesRequest listLandingZonesRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listLandingZonesRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listLandingZonesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ControlTower");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListLandingZones");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListLandingZonesResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListLandingZonesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListLandingZonesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListLandingZonesRequest, ListLandingZonesResponse>()
                            .withOperationName("ListLandingZones").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListLandingZonesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listLandingZonesRequest));
            CompletableFuture<ListLandingZonesResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns a list of tags associated with the resource. For usage examples, see <a
     * href="https://docs.aws.amazon.com/controltower/latest/userguide/control-api-examples-short.html"> <i>the Amazon
     * Web Services Control Tower User Guide</i> </a>.
     * </p>
     *
     * @param listTagsForResourceRequest
     * @return A Java Future containing the result of the ListTagsForResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The input fails to satisfy the constraints specified by an Amazon Web Services
     *         service.</li>
     *         <li>InternalServerException Unexpected error during processing of request.</li>
     *         <li>ResourceNotFoundException Request references a resource which does not exist.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ControlTowerException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample ControlTowerAsyncClient.ListTagsForResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/controltower-2018-05-10/ListTagsForResource"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListTagsForResourceResponse> listTagsForResource(
            ListTagsForResourceRequest listTagsForResourceRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listTagsForResourceRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTagsForResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ControlTower");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTagsForResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListTagsForResourceResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListTagsForResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListTagsForResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListTagsForResourceRequest, ListTagsForResourceResponse>()
                            .withOperationName("ListTagsForResource").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListTagsForResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listTagsForResourceRequest));
            CompletableFuture<ListTagsForResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * This API call resets a landing zone. It starts an asynchronous operation that resets the landing zone to the
     * parameters specified in its original configuration.
     * </p>
     *
     * @param resetLandingZoneRequest
     * @return A Java Future containing the result of the ResetLandingZone operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The input fails to satisfy the constraints specified by an Amazon Web Services
     *         service.</li>
     *         <li>ConflictException Updating or deleting a resource can cause an inconsistent state.</li>
     *         <li>InternalServerException Unexpected error during processing of request.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>ThrottlingException Request was denied due to request throttling.</li>
     *         <li>ResourceNotFoundException Request references a resource which does not exist.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ControlTowerException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample ControlTowerAsyncClient.ResetLandingZone
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/controltower-2018-05-10/ResetLandingZone" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<ResetLandingZoneResponse> resetLandingZone(ResetLandingZoneRequest resetLandingZoneRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(resetLandingZoneRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, resetLandingZoneRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ControlTower");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ResetLandingZone");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ResetLandingZoneResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ResetLandingZoneResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ResetLandingZoneResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ResetLandingZoneRequest, ResetLandingZoneResponse>()
                            .withOperationName("ResetLandingZone").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ResetLandingZoneRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(resetLandingZoneRequest));
            CompletableFuture<ResetLandingZoneResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Applies tags to a resource. For usage examples, see <a
     * href="https://docs.aws.amazon.com/controltower/latest/userguide/control-api-examples-short.html"> <i>the Amazon
     * Web Services Control Tower User Guide</i> </a>.
     * </p>
     *
     * @param tagResourceRequest
     * @return A Java Future containing the result of the TagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The input fails to satisfy the constraints specified by an Amazon Web Services
     *         service.</li>
     *         <li>InternalServerException Unexpected error during processing of request.</li>
     *         <li>ResourceNotFoundException Request references a resource which does not exist.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ControlTowerException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample ControlTowerAsyncClient.TagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/controltower-2018-05-10/TagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<TagResourceResponse> tagResource(TagResourceRequest tagResourceRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(tagResourceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, tagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ControlTower");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "TagResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<TagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    TagResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<TagResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<TagResourceRequest, TagResourceResponse>()
                            .withOperationName("TagResource").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new TagResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(tagResourceRequest));
            CompletableFuture<TagResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Removes tags from a resource. For usage examples, see <a
     * href="https://docs.aws.amazon.com/controltower/latest/userguide/control-api-examples-short.html"> <i>the Amazon
     * Web Services Control Tower User Guide</i> </a>.
     * </p>
     *
     * @param untagResourceRequest
     * @return A Java Future containing the result of the UntagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The input fails to satisfy the constraints specified by an Amazon Web Services
     *         service.</li>
     *         <li>InternalServerException Unexpected error during processing of request.</li>
     *         <li>ResourceNotFoundException Request references a resource which does not exist.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ControlTowerException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample ControlTowerAsyncClient.UntagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/controltower-2018-05-10/UntagResource" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<UntagResourceResponse> untagResource(UntagResourceRequest untagResourceRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(untagResourceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, untagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ControlTower");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UntagResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UntagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    UntagResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UntagResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UntagResourceRequest, UntagResourceResponse>()
                            .withOperationName("UntagResource").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new UntagResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(untagResourceRequest));
            CompletableFuture<UntagResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * This API call updates the landing zone. It starts an asynchronous operation that updates the landing zone based
     * on the new landing zone version or the updated parameters specified in the updated manifest file.
     * </p>
     *
     * @param updateLandingZoneRequest
     * @return A Java Future containing the result of the UpdateLandingZone operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The input fails to satisfy the constraints specified by an Amazon Web Services
     *         service.</li>
     *         <li>ConflictException Updating or deleting a resource can cause an inconsistent state.</li>
     *         <li>InternalServerException Unexpected error during processing of request.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>ThrottlingException Request was denied due to request throttling.</li>
     *         <li>ResourceNotFoundException Request references a resource which does not exist.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ControlTowerException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample ControlTowerAsyncClient.UpdateLandingZone
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/controltower-2018-05-10/UpdateLandingZone"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateLandingZoneResponse> updateLandingZone(UpdateLandingZoneRequest updateLandingZoneRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateLandingZoneRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateLandingZoneRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ControlTower");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateLandingZone");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateLandingZoneResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateLandingZoneResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateLandingZoneResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateLandingZoneRequest, UpdateLandingZoneResponse>()
                            .withOperationName("UpdateLandingZone").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new UpdateLandingZoneRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(updateLandingZoneRequest));
            CompletableFuture<UpdateLandingZoneResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    @Override
    public final ControlTowerServiceClientConfiguration serviceClientConfiguration() {
        return new ControlTowerServiceClientConfigurationBuilder(this.clientConfiguration.toBuilder()).build();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(ControlTowerException::builder)
                .protocol(AwsJsonProtocol.REST_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("AccessDeniedException")
                                .exceptionBuilderSupplier(AccessDeniedException::builder).httpStatusCode(403).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ConflictException")
                                .exceptionBuilderSupplier(ConflictException::builder).httpStatusCode(409).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceNotFoundException")
                                .exceptionBuilderSupplier(ResourceNotFoundException::builder).httpStatusCode(404).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ThrottlingException")
                                .exceptionBuilderSupplier(ThrottlingException::builder).httpStatusCode(429).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ValidationException")
                                .exceptionBuilderSupplier(ValidationException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException")
                                .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).httpStatusCode(402).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalServerException")
                                .exceptionBuilderSupplier(InternalServerException::builder).httpStatusCode(500).build());
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private SdkClientConfiguration updateSdkClientConfiguration(SdkRequest request, SdkClientConfiguration clientConfiguration) {
        List<SdkPlugin> plugins = request.overrideConfiguration().map(c -> c.plugins()).orElse(Collections.emptyList());
        if (plugins.isEmpty()) {
            return clientConfiguration;
        }
        SdkClientConfiguration.Builder configuration = clientConfiguration.toBuilder();
        ControlTowerServiceClientConfigurationBuilder serviceConfigBuilder = new ControlTowerServiceClientConfigurationBuilder(
                configuration);
        for (SdkPlugin plugin : plugins) {
            plugin.configureClient(serviceConfigBuilder);
        }
        return configuration.build();
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    @Override
    public void close() {
        clientHandler.close();
    }
}
