/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.controltower;

import java.util.function.Consumer;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkPublicApi;
import software.amazon.awssdk.annotations.ThreadSafe;
import software.amazon.awssdk.awscore.AwsClient;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.exception.SdkClientException;
import software.amazon.awssdk.regions.ServiceMetadata;
import software.amazon.awssdk.services.controltower.model.AccessDeniedException;
import software.amazon.awssdk.services.controltower.model.ConflictException;
import software.amazon.awssdk.services.controltower.model.ControlTowerException;
import software.amazon.awssdk.services.controltower.model.DisableControlRequest;
import software.amazon.awssdk.services.controltower.model.DisableControlResponse;
import software.amazon.awssdk.services.controltower.model.EnableControlRequest;
import software.amazon.awssdk.services.controltower.model.EnableControlResponse;
import software.amazon.awssdk.services.controltower.model.GetControlOperationRequest;
import software.amazon.awssdk.services.controltower.model.GetControlOperationResponse;
import software.amazon.awssdk.services.controltower.model.GetEnabledControlRequest;
import software.amazon.awssdk.services.controltower.model.GetEnabledControlResponse;
import software.amazon.awssdk.services.controltower.model.InternalServerException;
import software.amazon.awssdk.services.controltower.model.ListEnabledControlsRequest;
import software.amazon.awssdk.services.controltower.model.ListEnabledControlsResponse;
import software.amazon.awssdk.services.controltower.model.ResourceNotFoundException;
import software.amazon.awssdk.services.controltower.model.ServiceQuotaExceededException;
import software.amazon.awssdk.services.controltower.model.ThrottlingException;
import software.amazon.awssdk.services.controltower.model.ValidationException;
import software.amazon.awssdk.services.controltower.paginators.ListEnabledControlsIterable;

/**
 * Service client for accessing AWS Control Tower. This can be created using the static {@link #builder()} method.
 *
 * <p>
 * These interfaces allow you to apply the AWS library of pre-defined <i>controls</i> to your organizational units,
 * programmatically. In AWS Control Tower, the terms "control" and "guardrail" are synonyms. .
 * </p>
 * <p>
 * To call these APIs, you'll need to know:
 * </p>
 * <ul>
 * <li>
 * <p>
 * the <code>controlIdentifier</code> for the control--or guardrail--you are targeting.
 * </p>
 * </li>
 * <li>
 * <p>
 * the ARN associated with the target organizational unit (OU), which we call the <code>targetIdentifier</code>.
 * </p>
 * </li>
 * </ul>
 * <p>
 * <b>To get the <code>controlIdentifier</code> for your AWS Control Tower control:</b>
 * </p>
 * <p>
 * The <code>controlIdentifier</code> is an ARN that is specified for each control. You can view the
 * <code>controlIdentifier</code> in the console on the <b>Control details</b> page, as well as in the documentation.
 * </p>
 * <p>
 * The <code>controlIdentifier</code> is unique in each AWS Region for each control. You can find the
 * <code>controlIdentifier</code> for each Region and control in the <a
 * href="https://docs.aws.amazon.com/controltower/latest/userguide/control-metadata-tables.html">Tables of control
 * metadata</a> in the <i>AWS Control Tower User Guide.</i>
 * </p>
 * <p>
 * A quick-reference list of control identifers for the AWS Control Tower legacy <i>Strongly recommended</i> and
 * <i>Elective</i> controls is given in <a
 * href="https://docs.aws.amazon.com/controltower/latest/userguide/control-identifiers.html.html">Resource identifiers
 * for APIs and guardrails</a> in the <a
 * href="https://docs.aws.amazon.com/controltower/latest/userguide/control-identifiers.html">Controls reference guide
 * section</a> of the <i>AWS Control Tower User Guide</i>. Remember that <i>Mandatory</i> controls cannot be added or
 * removed.
 * </p>
 * <note>
 * <p>
 * <b>ARN format:</b> <code>arn:aws:controltower:{REGION}::control/{CONTROL_NAME}</code>
 * </p>
 * <p>
 * <b>Example:</b>
 * </p>
 * <p>
 * <code>arn:aws:controltower:us-west-2::control/AWS-GR_AUTOSCALING_LAUNCH_CONFIG_PUBLIC_IP_DISABLED</code>
 * </p>
 * </note>
 * <p>
 * <b>To get the <code>targetIdentifier</code>:</b>
 * </p>
 * <p>
 * The <code>targetIdentifier</code> is the ARN for an OU.
 * </p>
 * <p>
 * In the AWS Organizations console, you can find the ARN for the OU on the <b>Organizational unit details</b> page
 * associated with that OU.
 * </p>
 * <note>
 * <p>
 * <b>OU ARN format:</b>
 * </p>
 * <p>
 * <code>arn:${Partition}:organizations::${MasterAccountId}:ou/o-${OrganizationId}/ou-${OrganizationalUnitId}</code>
 * </p>
 * </note>
 * <p class="title">
 * <b>Details and examples</b>
 * </p>
 * <ul>
 * <li>
 * <p>
 * <a href="https://docs.aws.amazon.com/controltower/latest/userguide/control-api-examples-short.html">Control API input
 * and output examples with CLI</a>
 * </p>
 * </li>
 * <li>
 * <p>
 * <a href="https://docs.aws.amazon.com/controltower/latest/userguide/enable-controls.html">Enable controls with
 * CloudFormation</a>
 * </p>
 * </li>
 * <li>
 * <p>
 * <a href="https://docs.aws.amazon.com/controltower/latest/userguide/control-metadata-tables.html">Control metadata
 * tables</a>
 * </p>
 * </li>
 * <li>
 * <p>
 * <a href="https://docs.aws.amazon.com/controltower/latest/userguide/control-identifiers.html">List of identifiers for
 * legacy controls</a>
 * </p>
 * </li>
 * <li>
 * <p>
 * <a href="https://docs.aws.amazon.com/controltower/latest/userguide/controls.html">Controls reference guide</a>
 * </p>
 * </li>
 * <li>
 * <p>
 * <a href="https://docs.aws.amazon.com/controltower/latest/userguide/controls-reference.html">Controls library
 * groupings</a>
 * </p>
 * </li>
 * <li>
 * <p>
 * <a href="https://docs.aws.amazon.com/controltower/latest/userguide/creating-resources-with-cloudformation.html">
 * Creating AWS Control Tower resources with AWS CloudFormation</a>
 * </p>
 * </li>
 * </ul>
 * <p>
 * To view the open source resource repository on GitHub, see <a
 * href="https://github.com/aws-cloudformation/aws-cloudformation-resource-providers-controltower"
 * >aws-cloudformation/aws-cloudformation-resource-providers-controltower</a>
 * </p>
 * <p>
 * <b>Recording API Requests</b>
 * </p>
 * <p>
 * AWS Control Tower supports AWS CloudTrail, a service that records AWS API calls for your AWS account and delivers log
 * files to an Amazon S3 bucket. By using information collected by CloudTrail, you can determine which requests the AWS
 * Control Tower service received, who made the request and when, and so on. For more about AWS Control Tower and its
 * support for CloudTrail, see <a
 * href="https://docs.aws.amazon.com/controltower/latest/userguide/logging-using-cloudtrail.html">Logging AWS Control
 * Tower Actions with AWS CloudTrail</a> in the AWS Control Tower User Guide. To learn more about CloudTrail, including
 * how to turn it on and find your log files, see the AWS CloudTrail User Guide.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
@SdkPublicApi
@ThreadSafe
public interface ControlTowerClient extends AwsClient {
    String SERVICE_NAME = "controltower";

    /**
     * Value for looking up the service's metadata from the
     * {@link software.amazon.awssdk.regions.ServiceMetadataProvider}.
     */
    String SERVICE_METADATA_ID = "controltower";

    /**
     * <p>
     * This API call turns off a control. It starts an asynchronous operation that deletes AWS resources on the
     * specified organizational unit and the accounts it contains. The resources will vary according to the control that
     * you specify. For usage examples, see <a
     * href="https://docs.aws.amazon.com/controltower/latest/userguide/control-api-examples-short.html"> <i>the AWS
     * Control Tower User Guide</i> </a>.
     * </p>
     *
     * @param disableControlRequest
     * @return Result of the DisableControl operation returned by the service.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an AWS service.
     * @throws ConflictException
     *         Updating or deleting a resource can cause an inconsistent state.
     * @throws ServiceQuotaExceededException
     *         Request would cause a service quota to be exceeded. The limit is 10 concurrent operations.
     * @throws InternalServerException
     *         Unexpected error during processing of request.
     * @throws AccessDeniedException
     *         User does not have sufficient access to perform this action.
     * @throws ThrottlingException
     *         Request was denied due to request throttling.
     * @throws ResourceNotFoundException
     *         Request references a resource which does not exist.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ControlTowerException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ControlTowerClient.DisableControl
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/controltower-2018-05-10/DisableControl" target="_top">AWS
     *      API Documentation</a>
     */
    default DisableControlResponse disableControl(DisableControlRequest disableControlRequest) throws ValidationException,
            ConflictException, ServiceQuotaExceededException, InternalServerException, AccessDeniedException,
            ThrottlingException, ResourceNotFoundException, AwsServiceException, SdkClientException, ControlTowerException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * This API call turns off a control. It starts an asynchronous operation that deletes AWS resources on the
     * specified organizational unit and the accounts it contains. The resources will vary according to the control that
     * you specify. For usage examples, see <a
     * href="https://docs.aws.amazon.com/controltower/latest/userguide/control-api-examples-short.html"> <i>the AWS
     * Control Tower User Guide</i> </a>.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link DisableControlRequest.Builder} avoiding the need to
     * create one manually via {@link DisableControlRequest#builder()}
     * </p>
     *
     * @param disableControlRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.controltower.model.DisableControlRequest.Builder} to create a
     *        request.
     * @return Result of the DisableControl operation returned by the service.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an AWS service.
     * @throws ConflictException
     *         Updating or deleting a resource can cause an inconsistent state.
     * @throws ServiceQuotaExceededException
     *         Request would cause a service quota to be exceeded. The limit is 10 concurrent operations.
     * @throws InternalServerException
     *         Unexpected error during processing of request.
     * @throws AccessDeniedException
     *         User does not have sufficient access to perform this action.
     * @throws ThrottlingException
     *         Request was denied due to request throttling.
     * @throws ResourceNotFoundException
     *         Request references a resource which does not exist.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ControlTowerException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ControlTowerClient.DisableControl
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/controltower-2018-05-10/DisableControl" target="_top">AWS
     *      API Documentation</a>
     */
    default DisableControlResponse disableControl(Consumer<DisableControlRequest.Builder> disableControlRequest)
            throws ValidationException, ConflictException, ServiceQuotaExceededException, InternalServerException,
            AccessDeniedException, ThrottlingException, ResourceNotFoundException, AwsServiceException, SdkClientException,
            ControlTowerException {
        return disableControl(DisableControlRequest.builder().applyMutation(disableControlRequest).build());
    }

    /**
     * <p>
     * This API call activates a control. It starts an asynchronous operation that creates AWS resources on the
     * specified organizational unit and the accounts it contains. The resources created will vary according to the
     * control that you specify. For usage examples, see <a
     * href="https://docs.aws.amazon.com/controltower/latest/userguide/control-api-examples-short.html"> <i>the AWS
     * Control Tower User Guide</i> </a>
     * </p>
     *
     * @param enableControlRequest
     * @return Result of the EnableControl operation returned by the service.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an AWS service.
     * @throws ConflictException
     *         Updating or deleting a resource can cause an inconsistent state.
     * @throws ServiceQuotaExceededException
     *         Request would cause a service quota to be exceeded. The limit is 10 concurrent operations.
     * @throws InternalServerException
     *         Unexpected error during processing of request.
     * @throws AccessDeniedException
     *         User does not have sufficient access to perform this action.
     * @throws ThrottlingException
     *         Request was denied due to request throttling.
     * @throws ResourceNotFoundException
     *         Request references a resource which does not exist.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ControlTowerException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ControlTowerClient.EnableControl
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/controltower-2018-05-10/EnableControl" target="_top">AWS
     *      API Documentation</a>
     */
    default EnableControlResponse enableControl(EnableControlRequest enableControlRequest) throws ValidationException,
            ConflictException, ServiceQuotaExceededException, InternalServerException, AccessDeniedException,
            ThrottlingException, ResourceNotFoundException, AwsServiceException, SdkClientException, ControlTowerException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * This API call activates a control. It starts an asynchronous operation that creates AWS resources on the
     * specified organizational unit and the accounts it contains. The resources created will vary according to the
     * control that you specify. For usage examples, see <a
     * href="https://docs.aws.amazon.com/controltower/latest/userguide/control-api-examples-short.html"> <i>the AWS
     * Control Tower User Guide</i> </a>
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link EnableControlRequest.Builder} avoiding the need to
     * create one manually via {@link EnableControlRequest#builder()}
     * </p>
     *
     * @param enableControlRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.controltower.model.EnableControlRequest.Builder} to create a
     *        request.
     * @return Result of the EnableControl operation returned by the service.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an AWS service.
     * @throws ConflictException
     *         Updating or deleting a resource can cause an inconsistent state.
     * @throws ServiceQuotaExceededException
     *         Request would cause a service quota to be exceeded. The limit is 10 concurrent operations.
     * @throws InternalServerException
     *         Unexpected error during processing of request.
     * @throws AccessDeniedException
     *         User does not have sufficient access to perform this action.
     * @throws ThrottlingException
     *         Request was denied due to request throttling.
     * @throws ResourceNotFoundException
     *         Request references a resource which does not exist.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ControlTowerException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ControlTowerClient.EnableControl
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/controltower-2018-05-10/EnableControl" target="_top">AWS
     *      API Documentation</a>
     */
    default EnableControlResponse enableControl(Consumer<EnableControlRequest.Builder> enableControlRequest)
            throws ValidationException, ConflictException, ServiceQuotaExceededException, InternalServerException,
            AccessDeniedException, ThrottlingException, ResourceNotFoundException, AwsServiceException, SdkClientException,
            ControlTowerException {
        return enableControl(EnableControlRequest.builder().applyMutation(enableControlRequest).build());
    }

    /**
     * <p>
     * Returns the status of a particular <code>EnableControl</code> or <code>DisableControl</code> operation. Displays
     * a message in case of error. Details for an operation are available for 90 days. For usage examples, see <a
     * href="https://docs.aws.amazon.com/controltower/latest/userguide/control-api-examples-short.html"> <i>the AWS
     * Control Tower User Guide</i> </a>
     * </p>
     *
     * @param getControlOperationRequest
     * @return Result of the GetControlOperation operation returned by the service.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an AWS service.
     * @throws InternalServerException
     *         Unexpected error during processing of request.
     * @throws AccessDeniedException
     *         User does not have sufficient access to perform this action.
     * @throws ThrottlingException
     *         Request was denied due to request throttling.
     * @throws ResourceNotFoundException
     *         Request references a resource which does not exist.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ControlTowerException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ControlTowerClient.GetControlOperation
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/controltower-2018-05-10/GetControlOperation"
     *      target="_top">AWS API Documentation</a>
     */
    default GetControlOperationResponse getControlOperation(GetControlOperationRequest getControlOperationRequest)
            throws ValidationException, InternalServerException, AccessDeniedException, ThrottlingException,
            ResourceNotFoundException, AwsServiceException, SdkClientException, ControlTowerException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Returns the status of a particular <code>EnableControl</code> or <code>DisableControl</code> operation. Displays
     * a message in case of error. Details for an operation are available for 90 days. For usage examples, see <a
     * href="https://docs.aws.amazon.com/controltower/latest/userguide/control-api-examples-short.html"> <i>the AWS
     * Control Tower User Guide</i> </a>
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link GetControlOperationRequest.Builder} avoiding the
     * need to create one manually via {@link GetControlOperationRequest#builder()}
     * </p>
     *
     * @param getControlOperationRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.controltower.model.GetControlOperationRequest.Builder} to create a
     *        request.
     * @return Result of the GetControlOperation operation returned by the service.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an AWS service.
     * @throws InternalServerException
     *         Unexpected error during processing of request.
     * @throws AccessDeniedException
     *         User does not have sufficient access to perform this action.
     * @throws ThrottlingException
     *         Request was denied due to request throttling.
     * @throws ResourceNotFoundException
     *         Request references a resource which does not exist.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ControlTowerException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ControlTowerClient.GetControlOperation
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/controltower-2018-05-10/GetControlOperation"
     *      target="_top">AWS API Documentation</a>
     */
    default GetControlOperationResponse getControlOperation(
            Consumer<GetControlOperationRequest.Builder> getControlOperationRequest) throws ValidationException,
            InternalServerException, AccessDeniedException, ThrottlingException, ResourceNotFoundException, AwsServiceException,
            SdkClientException, ControlTowerException {
        return getControlOperation(GetControlOperationRequest.builder().applyMutation(getControlOperationRequest).build());
    }

    /**
     * <p>
     * Provides details about the enabled control. For usage examples, see <a
     * href="https://docs.aws.amazon.com/controltower/latest/userguide/control-api-examples-short.html"> <i>the AWS
     * Control Tower User Guide</i> </a>.
     * </p>
     * <p class="title">
     * <b>Returned values</b>
     * </p>
     * <ul>
     * <li>
     * <p>
     * TargetRegions: Shows target AWS Regions where the enabled control is available to be deployed.
     * </p>
     * </li>
     * <li>
     * <p>
     * StatusSummary: Provides a detailed summary of the deployment status.
     * </p>
     * </li>
     * <li>
     * <p>
     * DriftSummary: Provides a detailed summary of the drifted status.
     * </p>
     * </li>
     * </ul>
     *
     * @param getEnabledControlRequest
     * @return Result of the GetEnabledControl operation returned by the service.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an AWS service.
     * @throws InternalServerException
     *         Unexpected error during processing of request.
     * @throws AccessDeniedException
     *         User does not have sufficient access to perform this action.
     * @throws ThrottlingException
     *         Request was denied due to request throttling.
     * @throws ResourceNotFoundException
     *         Request references a resource which does not exist.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ControlTowerException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ControlTowerClient.GetEnabledControl
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/controltower-2018-05-10/GetEnabledControl"
     *      target="_top">AWS API Documentation</a>
     */
    default GetEnabledControlResponse getEnabledControl(GetEnabledControlRequest getEnabledControlRequest)
            throws ValidationException, InternalServerException, AccessDeniedException, ThrottlingException,
            ResourceNotFoundException, AwsServiceException, SdkClientException, ControlTowerException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Provides details about the enabled control. For usage examples, see <a
     * href="https://docs.aws.amazon.com/controltower/latest/userguide/control-api-examples-short.html"> <i>the AWS
     * Control Tower User Guide</i> </a>.
     * </p>
     * <p class="title">
     * <b>Returned values</b>
     * </p>
     * <ul>
     * <li>
     * <p>
     * TargetRegions: Shows target AWS Regions where the enabled control is available to be deployed.
     * </p>
     * </li>
     * <li>
     * <p>
     * StatusSummary: Provides a detailed summary of the deployment status.
     * </p>
     * </li>
     * <li>
     * <p>
     * DriftSummary: Provides a detailed summary of the drifted status.
     * </p>
     * </li>
     * </ul>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link GetEnabledControlRequest.Builder} avoiding the need
     * to create one manually via {@link GetEnabledControlRequest#builder()}
     * </p>
     *
     * @param getEnabledControlRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.controltower.model.GetEnabledControlRequest.Builder} to create a
     *        request.
     * @return Result of the GetEnabledControl operation returned by the service.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an AWS service.
     * @throws InternalServerException
     *         Unexpected error during processing of request.
     * @throws AccessDeniedException
     *         User does not have sufficient access to perform this action.
     * @throws ThrottlingException
     *         Request was denied due to request throttling.
     * @throws ResourceNotFoundException
     *         Request references a resource which does not exist.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ControlTowerException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ControlTowerClient.GetEnabledControl
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/controltower-2018-05-10/GetEnabledControl"
     *      target="_top">AWS API Documentation</a>
     */
    default GetEnabledControlResponse getEnabledControl(Consumer<GetEnabledControlRequest.Builder> getEnabledControlRequest)
            throws ValidationException, InternalServerException, AccessDeniedException, ThrottlingException,
            ResourceNotFoundException, AwsServiceException, SdkClientException, ControlTowerException {
        return getEnabledControl(GetEnabledControlRequest.builder().applyMutation(getEnabledControlRequest).build());
    }

    /**
     * <p>
     * Lists the controls enabled by AWS Control Tower on the specified organizational unit and the accounts it
     * contains. For usage examples, see <a
     * href="https://docs.aws.amazon.com/controltower/latest/userguide/control-api-examples-short.html"> <i>the AWS
     * Control Tower User Guide</i> </a>
     * </p>
     *
     * @param listEnabledControlsRequest
     * @return Result of the ListEnabledControls operation returned by the service.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an AWS service.
     * @throws InternalServerException
     *         Unexpected error during processing of request.
     * @throws AccessDeniedException
     *         User does not have sufficient access to perform this action.
     * @throws ThrottlingException
     *         Request was denied due to request throttling.
     * @throws ResourceNotFoundException
     *         Request references a resource which does not exist.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ControlTowerException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ControlTowerClient.ListEnabledControls
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/controltower-2018-05-10/ListEnabledControls"
     *      target="_top">AWS API Documentation</a>
     */
    default ListEnabledControlsResponse listEnabledControls(ListEnabledControlsRequest listEnabledControlsRequest)
            throws ValidationException, InternalServerException, AccessDeniedException, ThrottlingException,
            ResourceNotFoundException, AwsServiceException, SdkClientException, ControlTowerException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Lists the controls enabled by AWS Control Tower on the specified organizational unit and the accounts it
     * contains. For usage examples, see <a
     * href="https://docs.aws.amazon.com/controltower/latest/userguide/control-api-examples-short.html"> <i>the AWS
     * Control Tower User Guide</i> </a>
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link ListEnabledControlsRequest.Builder} avoiding the
     * need to create one manually via {@link ListEnabledControlsRequest#builder()}
     * </p>
     *
     * @param listEnabledControlsRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.controltower.model.ListEnabledControlsRequest.Builder} to create a
     *        request.
     * @return Result of the ListEnabledControls operation returned by the service.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an AWS service.
     * @throws InternalServerException
     *         Unexpected error during processing of request.
     * @throws AccessDeniedException
     *         User does not have sufficient access to perform this action.
     * @throws ThrottlingException
     *         Request was denied due to request throttling.
     * @throws ResourceNotFoundException
     *         Request references a resource which does not exist.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ControlTowerException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ControlTowerClient.ListEnabledControls
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/controltower-2018-05-10/ListEnabledControls"
     *      target="_top">AWS API Documentation</a>
     */
    default ListEnabledControlsResponse listEnabledControls(
            Consumer<ListEnabledControlsRequest.Builder> listEnabledControlsRequest) throws ValidationException,
            InternalServerException, AccessDeniedException, ThrottlingException, ResourceNotFoundException, AwsServiceException,
            SdkClientException, ControlTowerException {
        return listEnabledControls(ListEnabledControlsRequest.builder().applyMutation(listEnabledControlsRequest).build());
    }

    /**
     * <p>
     * Lists the controls enabled by AWS Control Tower on the specified organizational unit and the accounts it
     * contains. For usage examples, see <a
     * href="https://docs.aws.amazon.com/controltower/latest/userguide/control-api-examples-short.html"> <i>the AWS
     * Control Tower User Guide</i> </a>
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listEnabledControls(software.amazon.awssdk.services.controltower.model.ListEnabledControlsRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.controltower.paginators.ListEnabledControlsIterable responses = client.listEnabledControlsPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.controltower.paginators.ListEnabledControlsIterable responses = client
     *             .listEnabledControlsPaginator(request);
     *     for (software.amazon.awssdk.services.controltower.model.ListEnabledControlsResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.controltower.paginators.ListEnabledControlsIterable responses = client.listEnabledControlsPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of maxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listEnabledControls(software.amazon.awssdk.services.controltower.model.ListEnabledControlsRequest)}
     * operation.</b>
     * </p>
     *
     * @param listEnabledControlsRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an AWS service.
     * @throws InternalServerException
     *         Unexpected error during processing of request.
     * @throws AccessDeniedException
     *         User does not have sufficient access to perform this action.
     * @throws ThrottlingException
     *         Request was denied due to request throttling.
     * @throws ResourceNotFoundException
     *         Request references a resource which does not exist.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ControlTowerException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ControlTowerClient.ListEnabledControls
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/controltower-2018-05-10/ListEnabledControls"
     *      target="_top">AWS API Documentation</a>
     */
    default ListEnabledControlsIterable listEnabledControlsPaginator(ListEnabledControlsRequest listEnabledControlsRequest)
            throws ValidationException, InternalServerException, AccessDeniedException, ThrottlingException,
            ResourceNotFoundException, AwsServiceException, SdkClientException, ControlTowerException {
        return new ListEnabledControlsIterable(this, listEnabledControlsRequest);
    }

    /**
     * <p>
     * Lists the controls enabled by AWS Control Tower on the specified organizational unit and the accounts it
     * contains. For usage examples, see <a
     * href="https://docs.aws.amazon.com/controltower/latest/userguide/control-api-examples-short.html"> <i>the AWS
     * Control Tower User Guide</i> </a>
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listEnabledControls(software.amazon.awssdk.services.controltower.model.ListEnabledControlsRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.controltower.paginators.ListEnabledControlsIterable responses = client.listEnabledControlsPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.controltower.paginators.ListEnabledControlsIterable responses = client
     *             .listEnabledControlsPaginator(request);
     *     for (software.amazon.awssdk.services.controltower.model.ListEnabledControlsResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.controltower.paginators.ListEnabledControlsIterable responses = client.listEnabledControlsPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of maxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listEnabledControls(software.amazon.awssdk.services.controltower.model.ListEnabledControlsRequest)}
     * operation.</b>
     * </p>
     * <p>
     * This is a convenience which creates an instance of the {@link ListEnabledControlsRequest.Builder} avoiding the
     * need to create one manually via {@link ListEnabledControlsRequest#builder()}
     * </p>
     *
     * @param listEnabledControlsRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.controltower.model.ListEnabledControlsRequest.Builder} to create a
     *        request.
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an AWS service.
     * @throws InternalServerException
     *         Unexpected error during processing of request.
     * @throws AccessDeniedException
     *         User does not have sufficient access to perform this action.
     * @throws ThrottlingException
     *         Request was denied due to request throttling.
     * @throws ResourceNotFoundException
     *         Request references a resource which does not exist.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ControlTowerException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ControlTowerClient.ListEnabledControls
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/controltower-2018-05-10/ListEnabledControls"
     *      target="_top">AWS API Documentation</a>
     */
    default ListEnabledControlsIterable listEnabledControlsPaginator(
            Consumer<ListEnabledControlsRequest.Builder> listEnabledControlsRequest) throws ValidationException,
            InternalServerException, AccessDeniedException, ThrottlingException, ResourceNotFoundException, AwsServiceException,
            SdkClientException, ControlTowerException {
        return listEnabledControlsPaginator(ListEnabledControlsRequest.builder().applyMutation(listEnabledControlsRequest)
                .build());
    }

    /**
     * Create a {@link ControlTowerClient} with the region loaded from the
     * {@link software.amazon.awssdk.regions.providers.DefaultAwsRegionProviderChain} and credentials loaded from the
     * {@link software.amazon.awssdk.auth.credentials.DefaultCredentialsProvider}.
     */
    static ControlTowerClient create() {
        return builder().build();
    }

    /**
     * Create a builder that can be used to configure and create a {@link ControlTowerClient}.
     */
    static ControlTowerClientBuilder builder() {
        return new DefaultControlTowerClientBuilder();
    }

    static ServiceMetadata serviceMetadata() {
        return ServiceMetadata.of(SERVICE_METADATA_ID);
    }

    @Override
    default ControlTowerServiceClientConfiguration serviceClientConfiguration() {
        throw new UnsupportedOperationException();
    }
}
