/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.connectcampaigns;

import java.util.Collections;
import java.util.List;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.client.handler.AwsSyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.client.handler.SyncClientHandler;
import software.amazon.awssdk.core.exception.SdkClientException;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.connectcampaigns.model.AccessDeniedException;
import software.amazon.awssdk.services.connectcampaigns.model.ConflictException;
import software.amazon.awssdk.services.connectcampaigns.model.ConnectCampaignsException;
import software.amazon.awssdk.services.connectcampaigns.model.CreateCampaignRequest;
import software.amazon.awssdk.services.connectcampaigns.model.CreateCampaignResponse;
import software.amazon.awssdk.services.connectcampaigns.model.DeleteCampaignRequest;
import software.amazon.awssdk.services.connectcampaigns.model.DeleteCampaignResponse;
import software.amazon.awssdk.services.connectcampaigns.model.DeleteConnectInstanceConfigRequest;
import software.amazon.awssdk.services.connectcampaigns.model.DeleteConnectInstanceConfigResponse;
import software.amazon.awssdk.services.connectcampaigns.model.DeleteInstanceOnboardingJobRequest;
import software.amazon.awssdk.services.connectcampaigns.model.DeleteInstanceOnboardingJobResponse;
import software.amazon.awssdk.services.connectcampaigns.model.DescribeCampaignRequest;
import software.amazon.awssdk.services.connectcampaigns.model.DescribeCampaignResponse;
import software.amazon.awssdk.services.connectcampaigns.model.GetCampaignStateBatchRequest;
import software.amazon.awssdk.services.connectcampaigns.model.GetCampaignStateBatchResponse;
import software.amazon.awssdk.services.connectcampaigns.model.GetCampaignStateRequest;
import software.amazon.awssdk.services.connectcampaigns.model.GetCampaignStateResponse;
import software.amazon.awssdk.services.connectcampaigns.model.GetConnectInstanceConfigRequest;
import software.amazon.awssdk.services.connectcampaigns.model.GetConnectInstanceConfigResponse;
import software.amazon.awssdk.services.connectcampaigns.model.GetInstanceOnboardingJobStatusRequest;
import software.amazon.awssdk.services.connectcampaigns.model.GetInstanceOnboardingJobStatusResponse;
import software.amazon.awssdk.services.connectcampaigns.model.InternalServerException;
import software.amazon.awssdk.services.connectcampaigns.model.InvalidCampaignStateException;
import software.amazon.awssdk.services.connectcampaigns.model.InvalidStateException;
import software.amazon.awssdk.services.connectcampaigns.model.ListCampaignsRequest;
import software.amazon.awssdk.services.connectcampaigns.model.ListCampaignsResponse;
import software.amazon.awssdk.services.connectcampaigns.model.ListTagsForResourceRequest;
import software.amazon.awssdk.services.connectcampaigns.model.ListTagsForResourceResponse;
import software.amazon.awssdk.services.connectcampaigns.model.PauseCampaignRequest;
import software.amazon.awssdk.services.connectcampaigns.model.PauseCampaignResponse;
import software.amazon.awssdk.services.connectcampaigns.model.PutDialRequestBatchRequest;
import software.amazon.awssdk.services.connectcampaigns.model.PutDialRequestBatchResponse;
import software.amazon.awssdk.services.connectcampaigns.model.ResourceNotFoundException;
import software.amazon.awssdk.services.connectcampaigns.model.ResumeCampaignRequest;
import software.amazon.awssdk.services.connectcampaigns.model.ResumeCampaignResponse;
import software.amazon.awssdk.services.connectcampaigns.model.ServiceQuotaExceededException;
import software.amazon.awssdk.services.connectcampaigns.model.StartCampaignRequest;
import software.amazon.awssdk.services.connectcampaigns.model.StartCampaignResponse;
import software.amazon.awssdk.services.connectcampaigns.model.StartInstanceOnboardingJobRequest;
import software.amazon.awssdk.services.connectcampaigns.model.StartInstanceOnboardingJobResponse;
import software.amazon.awssdk.services.connectcampaigns.model.StopCampaignRequest;
import software.amazon.awssdk.services.connectcampaigns.model.StopCampaignResponse;
import software.amazon.awssdk.services.connectcampaigns.model.TagResourceRequest;
import software.amazon.awssdk.services.connectcampaigns.model.TagResourceResponse;
import software.amazon.awssdk.services.connectcampaigns.model.ThrottlingException;
import software.amazon.awssdk.services.connectcampaigns.model.UntagResourceRequest;
import software.amazon.awssdk.services.connectcampaigns.model.UntagResourceResponse;
import software.amazon.awssdk.services.connectcampaigns.model.UpdateCampaignDialerConfigRequest;
import software.amazon.awssdk.services.connectcampaigns.model.UpdateCampaignDialerConfigResponse;
import software.amazon.awssdk.services.connectcampaigns.model.UpdateCampaignNameRequest;
import software.amazon.awssdk.services.connectcampaigns.model.UpdateCampaignNameResponse;
import software.amazon.awssdk.services.connectcampaigns.model.UpdateCampaignOutboundCallConfigRequest;
import software.amazon.awssdk.services.connectcampaigns.model.UpdateCampaignOutboundCallConfigResponse;
import software.amazon.awssdk.services.connectcampaigns.model.ValidationException;
import software.amazon.awssdk.services.connectcampaigns.transform.CreateCampaignRequestMarshaller;
import software.amazon.awssdk.services.connectcampaigns.transform.DeleteCampaignRequestMarshaller;
import software.amazon.awssdk.services.connectcampaigns.transform.DeleteConnectInstanceConfigRequestMarshaller;
import software.amazon.awssdk.services.connectcampaigns.transform.DeleteInstanceOnboardingJobRequestMarshaller;
import software.amazon.awssdk.services.connectcampaigns.transform.DescribeCampaignRequestMarshaller;
import software.amazon.awssdk.services.connectcampaigns.transform.GetCampaignStateBatchRequestMarshaller;
import software.amazon.awssdk.services.connectcampaigns.transform.GetCampaignStateRequestMarshaller;
import software.amazon.awssdk.services.connectcampaigns.transform.GetConnectInstanceConfigRequestMarshaller;
import software.amazon.awssdk.services.connectcampaigns.transform.GetInstanceOnboardingJobStatusRequestMarshaller;
import software.amazon.awssdk.services.connectcampaigns.transform.ListCampaignsRequestMarshaller;
import software.amazon.awssdk.services.connectcampaigns.transform.ListTagsForResourceRequestMarshaller;
import software.amazon.awssdk.services.connectcampaigns.transform.PauseCampaignRequestMarshaller;
import software.amazon.awssdk.services.connectcampaigns.transform.PutDialRequestBatchRequestMarshaller;
import software.amazon.awssdk.services.connectcampaigns.transform.ResumeCampaignRequestMarshaller;
import software.amazon.awssdk.services.connectcampaigns.transform.StartCampaignRequestMarshaller;
import software.amazon.awssdk.services.connectcampaigns.transform.StartInstanceOnboardingJobRequestMarshaller;
import software.amazon.awssdk.services.connectcampaigns.transform.StopCampaignRequestMarshaller;
import software.amazon.awssdk.services.connectcampaigns.transform.TagResourceRequestMarshaller;
import software.amazon.awssdk.services.connectcampaigns.transform.UntagResourceRequestMarshaller;
import software.amazon.awssdk.services.connectcampaigns.transform.UpdateCampaignDialerConfigRequestMarshaller;
import software.amazon.awssdk.services.connectcampaigns.transform.UpdateCampaignNameRequestMarshaller;
import software.amazon.awssdk.services.connectcampaigns.transform.UpdateCampaignOutboundCallConfigRequestMarshaller;
import software.amazon.awssdk.utils.Logger;

/**
 * Internal implementation of {@link ConnectCampaignsClient}.
 *
 * @see ConnectCampaignsClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultConnectCampaignsClient implements ConnectCampaignsClient {
    private static final Logger log = Logger.loggerFor(DefaultConnectCampaignsClient.class);

    private final SyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    private final ConnectCampaignsServiceClientConfiguration serviceClientConfiguration;

    protected DefaultConnectCampaignsClient(ConnectCampaignsServiceClientConfiguration serviceClientConfiguration,
            SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsSyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.serviceClientConfiguration = serviceClientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    /**
     * <p>
     * Creates a campaign for the specified Amazon Connect account. This API is idempotent.
     * </p>
     *
     * @param createCampaignRequest
     *        The request for Create Campaign API.
     * @return Result of the CreateCampaign operation returned by the service.
     * @throws InternalServerException
     *         Request processing failed because of an error or failure with the service.
     * @throws ResourceNotFoundException
     *         The specified resource was not found.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an AWS service.
     * @throws ConflictException
     *         The request could not be processed because of conflict in the current state of the resource.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ServiceQuotaExceededException
     *         Request would cause a service quota to be exceeded.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ConnectCampaignsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ConnectCampaignsClient.CreateCampaign
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/connectcampaigns-2021-01-30/CreateCampaign"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CreateCampaignResponse createCampaign(CreateCampaignRequest createCampaignRequest) throws InternalServerException,
            ResourceNotFoundException, ValidationException, ConflictException, AccessDeniedException,
            ServiceQuotaExceededException, ThrottlingException, AwsServiceException, SdkClientException,
            ConnectCampaignsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateCampaignResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                CreateCampaignResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createCampaignRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ConnectCampaigns");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateCampaign");

            return clientHandler.execute(new ClientExecutionParams<CreateCampaignRequest, CreateCampaignResponse>()
                    .withOperationName("CreateCampaign").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(createCampaignRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new CreateCampaignRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deletes a campaign from the specified Amazon Connect account.
     * </p>
     *
     * @param deleteCampaignRequest
     *        DeleteCampaignRequest
     * @return Result of the DeleteCampaign operation returned by the service.
     * @throws InternalServerException
     *         Request processing failed because of an error or failure with the service.
     * @throws ResourceNotFoundException
     *         The specified resource was not found.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an AWS service.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ConnectCampaignsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ConnectCampaignsClient.DeleteCampaign
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/connectcampaigns-2021-01-30/DeleteCampaign"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DeleteCampaignResponse deleteCampaign(DeleteCampaignRequest deleteCampaignRequest) throws InternalServerException,
            ResourceNotFoundException, ValidationException, AccessDeniedException, AwsServiceException, SdkClientException,
            ConnectCampaignsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteCampaignResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DeleteCampaignResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteCampaignRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ConnectCampaigns");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteCampaign");

            return clientHandler.execute(new ClientExecutionParams<DeleteCampaignRequest, DeleteCampaignResponse>()
                    .withOperationName("DeleteCampaign").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(deleteCampaignRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DeleteCampaignRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deletes a connect instance config from the specified AWS account.
     * </p>
     *
     * @param deleteConnectInstanceConfigRequest
     *        DeleteCampaignRequest
     * @return Result of the DeleteConnectInstanceConfig operation returned by the service.
     * @throws InternalServerException
     *         Request processing failed because of an error or failure with the service.
     * @throws ResourceNotFoundException
     *         The specified resource was not found.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an AWS service.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws InvalidStateException
     *         The request could not be processed because of conflict in the current state.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ConnectCampaignsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ConnectCampaignsClient.DeleteConnectInstanceConfig
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/connectcampaigns-2021-01-30/DeleteConnectInstanceConfig"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DeleteConnectInstanceConfigResponse deleteConnectInstanceConfig(
            DeleteConnectInstanceConfigRequest deleteConnectInstanceConfigRequest) throws InternalServerException,
            ResourceNotFoundException, ValidationException, AccessDeniedException, InvalidStateException, ThrottlingException,
            AwsServiceException, SdkClientException, ConnectCampaignsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteConnectInstanceConfigResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DeleteConnectInstanceConfigResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteConnectInstanceConfigRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ConnectCampaigns");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteConnectInstanceConfig");

            return clientHandler
                    .execute(new ClientExecutionParams<DeleteConnectInstanceConfigRequest, DeleteConnectInstanceConfigResponse>()
                            .withOperationName("DeleteConnectInstanceConfig").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(deleteConnectInstanceConfigRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new DeleteConnectInstanceConfigRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Delete the Connect Campaigns onboarding job for the specified Amazon Connect instance.
     * </p>
     *
     * @param deleteInstanceOnboardingJobRequest
     *        The request for DeleteInstanceOnboardingJob API.
     * @return Result of the DeleteInstanceOnboardingJob operation returned by the service.
     * @throws InternalServerException
     *         Request processing failed because of an error or failure with the service.
     * @throws ResourceNotFoundException
     *         The specified resource was not found.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an AWS service.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws InvalidStateException
     *         The request could not be processed because of conflict in the current state.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ConnectCampaignsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ConnectCampaignsClient.DeleteInstanceOnboardingJob
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/connectcampaigns-2021-01-30/DeleteInstanceOnboardingJob"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DeleteInstanceOnboardingJobResponse deleteInstanceOnboardingJob(
            DeleteInstanceOnboardingJobRequest deleteInstanceOnboardingJobRequest) throws InternalServerException,
            ResourceNotFoundException, ValidationException, AccessDeniedException, InvalidStateException, AwsServiceException,
            SdkClientException, ConnectCampaignsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteInstanceOnboardingJobResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DeleteInstanceOnboardingJobResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteInstanceOnboardingJobRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ConnectCampaigns");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteInstanceOnboardingJob");

            return clientHandler
                    .execute(new ClientExecutionParams<DeleteInstanceOnboardingJobRequest, DeleteInstanceOnboardingJobResponse>()
                            .withOperationName("DeleteInstanceOnboardingJob").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(deleteInstanceOnboardingJobRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new DeleteInstanceOnboardingJobRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Describes the specific campaign.
     * </p>
     *
     * @param describeCampaignRequest
     *        DescribeCampaignRequests
     * @return Result of the DescribeCampaign operation returned by the service.
     * @throws InternalServerException
     *         Request processing failed because of an error or failure with the service.
     * @throws ResourceNotFoundException
     *         The specified resource was not found.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an AWS service.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ConnectCampaignsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ConnectCampaignsClient.DescribeCampaign
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/connectcampaigns-2021-01-30/DescribeCampaign"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DescribeCampaignResponse describeCampaign(DescribeCampaignRequest describeCampaignRequest)
            throws InternalServerException, ResourceNotFoundException, ValidationException, AccessDeniedException,
            AwsServiceException, SdkClientException, ConnectCampaignsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeCampaignResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DescribeCampaignResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeCampaignRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ConnectCampaigns");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeCampaign");

            return clientHandler.execute(new ClientExecutionParams<DescribeCampaignRequest, DescribeCampaignResponse>()
                    .withOperationName("DescribeCampaign").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(describeCampaignRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DescribeCampaignRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Get state of a campaign for the specified Amazon Connect account.
     * </p>
     *
     * @param getCampaignStateRequest
     *        GetCampaignStateRequest
     * @return Result of the GetCampaignState operation returned by the service.
     * @throws InternalServerException
     *         Request processing failed because of an error or failure with the service.
     * @throws ResourceNotFoundException
     *         The specified resource was not found.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an AWS service.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ConnectCampaignsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ConnectCampaignsClient.GetCampaignState
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/connectcampaigns-2021-01-30/GetCampaignState"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public GetCampaignStateResponse getCampaignState(GetCampaignStateRequest getCampaignStateRequest)
            throws InternalServerException, ResourceNotFoundException, ValidationException, AccessDeniedException,
            ThrottlingException, AwsServiceException, SdkClientException, ConnectCampaignsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetCampaignStateResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetCampaignStateResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getCampaignStateRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ConnectCampaigns");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetCampaignState");

            return clientHandler.execute(new ClientExecutionParams<GetCampaignStateRequest, GetCampaignStateResponse>()
                    .withOperationName("GetCampaignState").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(getCampaignStateRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetCampaignStateRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Get state of campaigns for the specified Amazon Connect account.
     * </p>
     *
     * @param getCampaignStateBatchRequest
     *        GetCampaignStateBatchRequest
     * @return Result of the GetCampaignStateBatch operation returned by the service.
     * @throws InternalServerException
     *         Request processing failed because of an error or failure with the service.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an AWS service.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ConnectCampaignsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ConnectCampaignsClient.GetCampaignStateBatch
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/connectcampaigns-2021-01-30/GetCampaignStateBatch"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public GetCampaignStateBatchResponse getCampaignStateBatch(GetCampaignStateBatchRequest getCampaignStateBatchRequest)
            throws InternalServerException, ValidationException, AccessDeniedException, ThrottlingException, AwsServiceException,
            SdkClientException, ConnectCampaignsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetCampaignStateBatchResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, GetCampaignStateBatchResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getCampaignStateBatchRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ConnectCampaigns");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetCampaignStateBatch");

            return clientHandler.execute(new ClientExecutionParams<GetCampaignStateBatchRequest, GetCampaignStateBatchResponse>()
                    .withOperationName("GetCampaignStateBatch").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(getCampaignStateBatchRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetCampaignStateBatchRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Get the specific Connect instance config.
     * </p>
     *
     * @param getConnectInstanceConfigRequest
     *        GetConnectInstanceConfigRequest
     * @return Result of the GetConnectInstanceConfig operation returned by the service.
     * @throws InternalServerException
     *         Request processing failed because of an error or failure with the service.
     * @throws ResourceNotFoundException
     *         The specified resource was not found.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an AWS service.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ConnectCampaignsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ConnectCampaignsClient.GetConnectInstanceConfig
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/connectcampaigns-2021-01-30/GetConnectInstanceConfig"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public GetConnectInstanceConfigResponse getConnectInstanceConfig(
            GetConnectInstanceConfigRequest getConnectInstanceConfigRequest) throws InternalServerException,
            ResourceNotFoundException, ValidationException, AccessDeniedException, AwsServiceException, SdkClientException,
            ConnectCampaignsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetConnectInstanceConfigResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, GetConnectInstanceConfigResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getConnectInstanceConfigRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ConnectCampaigns");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetConnectInstanceConfig");

            return clientHandler
                    .execute(new ClientExecutionParams<GetConnectInstanceConfigRequest, GetConnectInstanceConfigResponse>()
                            .withOperationName("GetConnectInstanceConfig").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(getConnectInstanceConfigRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new GetConnectInstanceConfigRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Get the specific instance onboarding job status.
     * </p>
     *
     * @param getInstanceOnboardingJobStatusRequest
     *        GetInstanceOnboardingJobStatusRequest
     * @return Result of the GetInstanceOnboardingJobStatus operation returned by the service.
     * @throws InternalServerException
     *         Request processing failed because of an error or failure with the service.
     * @throws ResourceNotFoundException
     *         The specified resource was not found.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an AWS service.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ConnectCampaignsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ConnectCampaignsClient.GetInstanceOnboardingJobStatus
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/connectcampaigns-2021-01-30/GetInstanceOnboardingJobStatus"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public GetInstanceOnboardingJobStatusResponse getInstanceOnboardingJobStatus(
            GetInstanceOnboardingJobStatusRequest getInstanceOnboardingJobStatusRequest) throws InternalServerException,
            ResourceNotFoundException, ValidationException, AccessDeniedException, AwsServiceException, SdkClientException,
            ConnectCampaignsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetInstanceOnboardingJobStatusResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, GetInstanceOnboardingJobStatusResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                getInstanceOnboardingJobStatusRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ConnectCampaigns");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetInstanceOnboardingJobStatus");

            return clientHandler
                    .execute(new ClientExecutionParams<GetInstanceOnboardingJobStatusRequest, GetInstanceOnboardingJobStatusResponse>()
                            .withOperationName("GetInstanceOnboardingJobStatus").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(getInstanceOnboardingJobStatusRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new GetInstanceOnboardingJobStatusRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Provides summary information about the campaigns under the specified Amazon Connect account.
     * </p>
     *
     * @param listCampaignsRequest
     *        ListCampaignsRequest
     * @return Result of the ListCampaigns operation returned by the service.
     * @throws InternalServerException
     *         Request processing failed because of an error or failure with the service.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an AWS service.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ConnectCampaignsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ConnectCampaignsClient.ListCampaigns
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/connectcampaigns-2021-01-30/ListCampaigns"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListCampaignsResponse listCampaigns(ListCampaignsRequest listCampaignsRequest) throws InternalServerException,
            ValidationException, AccessDeniedException, AwsServiceException, SdkClientException, ConnectCampaignsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListCampaignsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListCampaignsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listCampaignsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ConnectCampaigns");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListCampaigns");

            return clientHandler.execute(new ClientExecutionParams<ListCampaignsRequest, ListCampaignsResponse>()
                    .withOperationName("ListCampaigns").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(listCampaignsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListCampaignsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * List tags for a resource.
     * </p>
     *
     * @param listTagsForResourceRequest
     *        ListTagsForResource
     * @return Result of the ListTagsForResource operation returned by the service.
     * @throws InternalServerException
     *         Request processing failed because of an error or failure with the service.
     * @throws ResourceNotFoundException
     *         The specified resource was not found.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an AWS service.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ConnectCampaignsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ConnectCampaignsClient.ListTagsForResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/connectcampaigns-2021-01-30/ListTagsForResource"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListTagsForResourceResponse listTagsForResource(ListTagsForResourceRequest listTagsForResourceRequest)
            throws InternalServerException, ResourceNotFoundException, ValidationException, AccessDeniedException,
            ThrottlingException, AwsServiceException, SdkClientException, ConnectCampaignsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListTagsForResourceResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListTagsForResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTagsForResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ConnectCampaigns");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTagsForResource");

            return clientHandler.execute(new ClientExecutionParams<ListTagsForResourceRequest, ListTagsForResourceResponse>()
                    .withOperationName("ListTagsForResource").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(listTagsForResourceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListTagsForResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Pauses a campaign for the specified Amazon Connect account.
     * </p>
     *
     * @param pauseCampaignRequest
     *        PauseCampaignRequest
     * @return Result of the PauseCampaign operation returned by the service.
     * @throws InternalServerException
     *         Request processing failed because of an error or failure with the service.
     * @throws ResourceNotFoundException
     *         The specified resource was not found.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an AWS service.
     * @throws ConflictException
     *         The request could not be processed because of conflict in the current state of the resource.
     * @throws InvalidCampaignStateException
     *         The request could not be processed because of conflict in the current state of the campaign.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ConnectCampaignsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ConnectCampaignsClient.PauseCampaign
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/connectcampaigns-2021-01-30/PauseCampaign"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public PauseCampaignResponse pauseCampaign(PauseCampaignRequest pauseCampaignRequest) throws InternalServerException,
            ResourceNotFoundException, ValidationException, ConflictException, InvalidCampaignStateException,
            AccessDeniedException, ThrottlingException, AwsServiceException, SdkClientException, ConnectCampaignsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<PauseCampaignResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                PauseCampaignResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, pauseCampaignRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ConnectCampaigns");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "PauseCampaign");

            return clientHandler.execute(new ClientExecutionParams<PauseCampaignRequest, PauseCampaignResponse>()
                    .withOperationName("PauseCampaign").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(pauseCampaignRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new PauseCampaignRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Creates dials requests for the specified campaign Amazon Connect account. This API is idempotent.
     * </p>
     *
     * @param putDialRequestBatchRequest
     *        PutDialRequestBatchRequest
     * @return Result of the PutDialRequestBatch operation returned by the service.
     * @throws InternalServerException
     *         Request processing failed because of an error or failure with the service.
     * @throws ResourceNotFoundException
     *         The specified resource was not found.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an AWS service.
     * @throws ConflictException
     *         The request could not be processed because of conflict in the current state of the resource.
     * @throws InvalidCampaignStateException
     *         The request could not be processed because of conflict in the current state of the campaign.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ConnectCampaignsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ConnectCampaignsClient.PutDialRequestBatch
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/connectcampaigns-2021-01-30/PutDialRequestBatch"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public PutDialRequestBatchResponse putDialRequestBatch(PutDialRequestBatchRequest putDialRequestBatchRequest)
            throws InternalServerException, ResourceNotFoundException, ValidationException, ConflictException,
            InvalidCampaignStateException, AccessDeniedException, ThrottlingException, AwsServiceException, SdkClientException,
            ConnectCampaignsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<PutDialRequestBatchResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, PutDialRequestBatchResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, putDialRequestBatchRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ConnectCampaigns");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "PutDialRequestBatch");

            return clientHandler.execute(new ClientExecutionParams<PutDialRequestBatchRequest, PutDialRequestBatchResponse>()
                    .withOperationName("PutDialRequestBatch").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(putDialRequestBatchRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new PutDialRequestBatchRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Stops a campaign for the specified Amazon Connect account.
     * </p>
     *
     * @param resumeCampaignRequest
     *        ResumeCampaignRequest
     * @return Result of the ResumeCampaign operation returned by the service.
     * @throws InternalServerException
     *         Request processing failed because of an error or failure with the service.
     * @throws ResourceNotFoundException
     *         The specified resource was not found.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an AWS service.
     * @throws ConflictException
     *         The request could not be processed because of conflict in the current state of the resource.
     * @throws InvalidCampaignStateException
     *         The request could not be processed because of conflict in the current state of the campaign.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ConnectCampaignsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ConnectCampaignsClient.ResumeCampaign
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/connectcampaigns-2021-01-30/ResumeCampaign"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ResumeCampaignResponse resumeCampaign(ResumeCampaignRequest resumeCampaignRequest) throws InternalServerException,
            ResourceNotFoundException, ValidationException, ConflictException, InvalidCampaignStateException,
            AccessDeniedException, ThrottlingException, AwsServiceException, SdkClientException, ConnectCampaignsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ResumeCampaignResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ResumeCampaignResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, resumeCampaignRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ConnectCampaigns");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ResumeCampaign");

            return clientHandler.execute(new ClientExecutionParams<ResumeCampaignRequest, ResumeCampaignResponse>()
                    .withOperationName("ResumeCampaign").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(resumeCampaignRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ResumeCampaignRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Starts a campaign for the specified Amazon Connect account.
     * </p>
     *
     * @param startCampaignRequest
     *        StartCampaignRequest
     * @return Result of the StartCampaign operation returned by the service.
     * @throws InternalServerException
     *         Request processing failed because of an error or failure with the service.
     * @throws ResourceNotFoundException
     *         The specified resource was not found.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an AWS service.
     * @throws ConflictException
     *         The request could not be processed because of conflict in the current state of the resource.
     * @throws InvalidCampaignStateException
     *         The request could not be processed because of conflict in the current state of the campaign.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ConnectCampaignsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ConnectCampaignsClient.StartCampaign
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/connectcampaigns-2021-01-30/StartCampaign"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public StartCampaignResponse startCampaign(StartCampaignRequest startCampaignRequest) throws InternalServerException,
            ResourceNotFoundException, ValidationException, ConflictException, InvalidCampaignStateException,
            AccessDeniedException, ThrottlingException, AwsServiceException, SdkClientException, ConnectCampaignsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<StartCampaignResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                StartCampaignResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, startCampaignRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ConnectCampaigns");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "StartCampaign");

            return clientHandler.execute(new ClientExecutionParams<StartCampaignRequest, StartCampaignResponse>()
                    .withOperationName("StartCampaign").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(startCampaignRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new StartCampaignRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Onboard the specific Amazon Connect instance to Connect Campaigns.
     * </p>
     *
     * @param startInstanceOnboardingJobRequest
     *        The request for StartInstanceOnboardingJob API.
     * @return Result of the StartInstanceOnboardingJob operation returned by the service.
     * @throws InternalServerException
     *         Request processing failed because of an error or failure with the service.
     * @throws ResourceNotFoundException
     *         The specified resource was not found.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an AWS service.
     * @throws ConflictException
     *         The request could not be processed because of conflict in the current state of the resource.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ConnectCampaignsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ConnectCampaignsClient.StartInstanceOnboardingJob
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/connectcampaigns-2021-01-30/StartInstanceOnboardingJob"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public StartInstanceOnboardingJobResponse startInstanceOnboardingJob(
            StartInstanceOnboardingJobRequest startInstanceOnboardingJobRequest) throws InternalServerException,
            ResourceNotFoundException, ValidationException, ConflictException, AccessDeniedException, ThrottlingException,
            AwsServiceException, SdkClientException, ConnectCampaignsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<StartInstanceOnboardingJobResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, StartInstanceOnboardingJobResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, startInstanceOnboardingJobRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ConnectCampaigns");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "StartInstanceOnboardingJob");

            return clientHandler
                    .execute(new ClientExecutionParams<StartInstanceOnboardingJobRequest, StartInstanceOnboardingJobResponse>()
                            .withOperationName("StartInstanceOnboardingJob").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(startInstanceOnboardingJobRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new StartInstanceOnboardingJobRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Stops a campaign for the specified Amazon Connect account.
     * </p>
     *
     * @param stopCampaignRequest
     *        StopCampaignRequest
     * @return Result of the StopCampaign operation returned by the service.
     * @throws InternalServerException
     *         Request processing failed because of an error or failure with the service.
     * @throws ResourceNotFoundException
     *         The specified resource was not found.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an AWS service.
     * @throws ConflictException
     *         The request could not be processed because of conflict in the current state of the resource.
     * @throws InvalidCampaignStateException
     *         The request could not be processed because of conflict in the current state of the campaign.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ConnectCampaignsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ConnectCampaignsClient.StopCampaign
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/connectcampaigns-2021-01-30/StopCampaign" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public StopCampaignResponse stopCampaign(StopCampaignRequest stopCampaignRequest) throws InternalServerException,
            ResourceNotFoundException, ValidationException, ConflictException, InvalidCampaignStateException,
            AccessDeniedException, ThrottlingException, AwsServiceException, SdkClientException, ConnectCampaignsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<StopCampaignResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                StopCampaignResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, stopCampaignRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ConnectCampaigns");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "StopCampaign");

            return clientHandler.execute(new ClientExecutionParams<StopCampaignRequest, StopCampaignResponse>()
                    .withOperationName("StopCampaign").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(stopCampaignRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new StopCampaignRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Tag a resource.
     * </p>
     *
     * @param tagResourceRequest
     *        TagResourceRequest
     * @return Result of the TagResource operation returned by the service.
     * @throws InternalServerException
     *         Request processing failed because of an error or failure with the service.
     * @throws ResourceNotFoundException
     *         The specified resource was not found.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an AWS service.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ConnectCampaignsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ConnectCampaignsClient.TagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/connectcampaigns-2021-01-30/TagResource" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public TagResourceResponse tagResource(TagResourceRequest tagResourceRequest) throws InternalServerException,
            ResourceNotFoundException, ValidationException, AccessDeniedException, ThrottlingException, AwsServiceException,
            SdkClientException, ConnectCampaignsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<TagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                TagResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, tagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ConnectCampaigns");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "TagResource");

            return clientHandler.execute(new ClientExecutionParams<TagResourceRequest, TagResourceResponse>()
                    .withOperationName("TagResource").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(tagResourceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new TagResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Untag a resource.
     * </p>
     *
     * @param untagResourceRequest
     *        UntagResourceRequest
     * @return Result of the UntagResource operation returned by the service.
     * @throws InternalServerException
     *         Request processing failed because of an error or failure with the service.
     * @throws ResourceNotFoundException
     *         The specified resource was not found.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an AWS service.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ConnectCampaignsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ConnectCampaignsClient.UntagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/connectcampaigns-2021-01-30/UntagResource"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public UntagResourceResponse untagResource(UntagResourceRequest untagResourceRequest) throws InternalServerException,
            ResourceNotFoundException, ValidationException, AccessDeniedException, ThrottlingException, AwsServiceException,
            SdkClientException, ConnectCampaignsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UntagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                UntagResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, untagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ConnectCampaigns");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UntagResource");

            return clientHandler.execute(new ClientExecutionParams<UntagResourceRequest, UntagResourceResponse>()
                    .withOperationName("UntagResource").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(untagResourceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UntagResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Updates the dialer config of a campaign. This API is idempotent.
     * </p>
     *
     * @param updateCampaignDialerConfigRequest
     *        UpdateCampaignDialerConfigRequest
     * @return Result of the UpdateCampaignDialerConfig operation returned by the service.
     * @throws InternalServerException
     *         Request processing failed because of an error or failure with the service.
     * @throws ResourceNotFoundException
     *         The specified resource was not found.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an AWS service.
     * @throws ConflictException
     *         The request could not be processed because of conflict in the current state of the resource.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ConnectCampaignsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ConnectCampaignsClient.UpdateCampaignDialerConfig
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/connectcampaigns-2021-01-30/UpdateCampaignDialerConfig"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public UpdateCampaignDialerConfigResponse updateCampaignDialerConfig(
            UpdateCampaignDialerConfigRequest updateCampaignDialerConfigRequest) throws InternalServerException,
            ResourceNotFoundException, ValidationException, ConflictException, AccessDeniedException, AwsServiceException,
            SdkClientException, ConnectCampaignsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateCampaignDialerConfigResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, UpdateCampaignDialerConfigResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateCampaignDialerConfigRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ConnectCampaigns");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateCampaignDialerConfig");

            return clientHandler
                    .execute(new ClientExecutionParams<UpdateCampaignDialerConfigRequest, UpdateCampaignDialerConfigResponse>()
                            .withOperationName("UpdateCampaignDialerConfig").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(updateCampaignDialerConfigRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new UpdateCampaignDialerConfigRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Updates the name of a campaign. This API is idempotent.
     * </p>
     *
     * @param updateCampaignNameRequest
     *        UpdateCampaignNameRequest
     * @return Result of the UpdateCampaignName operation returned by the service.
     * @throws InternalServerException
     *         Request processing failed because of an error or failure with the service.
     * @throws ResourceNotFoundException
     *         The specified resource was not found.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an AWS service.
     * @throws ConflictException
     *         The request could not be processed because of conflict in the current state of the resource.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ConnectCampaignsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ConnectCampaignsClient.UpdateCampaignName
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/connectcampaigns-2021-01-30/UpdateCampaignName"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public UpdateCampaignNameResponse updateCampaignName(UpdateCampaignNameRequest updateCampaignNameRequest)
            throws InternalServerException, ResourceNotFoundException, ValidationException, ConflictException,
            AccessDeniedException, AwsServiceException, SdkClientException, ConnectCampaignsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateCampaignNameResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, UpdateCampaignNameResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateCampaignNameRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ConnectCampaigns");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateCampaignName");

            return clientHandler.execute(new ClientExecutionParams<UpdateCampaignNameRequest, UpdateCampaignNameResponse>()
                    .withOperationName("UpdateCampaignName").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(updateCampaignNameRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UpdateCampaignNameRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Updates the outbound call config of a campaign. This API is idempotent.
     * </p>
     *
     * @param updateCampaignOutboundCallConfigRequest
     *        UpdateCampaignOutboundCallConfigRequest
     * @return Result of the UpdateCampaignOutboundCallConfig operation returned by the service.
     * @throws InternalServerException
     *         Request processing failed because of an error or failure with the service.
     * @throws ResourceNotFoundException
     *         The specified resource was not found.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an AWS service.
     * @throws ConflictException
     *         The request could not be processed because of conflict in the current state of the resource.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ConnectCampaignsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ConnectCampaignsClient.UpdateCampaignOutboundCallConfig
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/connectcampaigns-2021-01-30/UpdateCampaignOutboundCallConfig"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public UpdateCampaignOutboundCallConfigResponse updateCampaignOutboundCallConfig(
            UpdateCampaignOutboundCallConfigRequest updateCampaignOutboundCallConfigRequest) throws InternalServerException,
            ResourceNotFoundException, ValidationException, ConflictException, AccessDeniedException, ThrottlingException,
            AwsServiceException, SdkClientException, ConnectCampaignsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateCampaignOutboundCallConfigResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, UpdateCampaignOutboundCallConfigResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                updateCampaignOutboundCallConfigRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ConnectCampaigns");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateCampaignOutboundCallConfig");

            return clientHandler
                    .execute(new ClientExecutionParams<UpdateCampaignOutboundCallConfigRequest, UpdateCampaignOutboundCallConfigResponse>()
                            .withOperationName("UpdateCampaignOutboundCallConfig").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(updateCampaignOutboundCallConfigRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new UpdateCampaignOutboundCallConfigRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(ConnectCampaignsException::builder)
                .protocol(AwsJsonProtocol.REST_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("AccessDeniedException")
                                .exceptionBuilderSupplier(AccessDeniedException::builder).httpStatusCode(403).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ConflictException")
                                .exceptionBuilderSupplier(ConflictException::builder).httpStatusCode(409).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceNotFoundException")
                                .exceptionBuilderSupplier(ResourceNotFoundException::builder).httpStatusCode(404).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ThrottlingException")
                                .exceptionBuilderSupplier(ThrottlingException::builder).httpStatusCode(429).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ValidationException")
                                .exceptionBuilderSupplier(ValidationException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException")
                                .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).httpStatusCode(402).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidCampaignStateException")
                                .exceptionBuilderSupplier(InvalidCampaignStateException::builder).httpStatusCode(409).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalServerException")
                                .exceptionBuilderSupplier(InternalServerException::builder).httpStatusCode(500).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidStateException")
                                .exceptionBuilderSupplier(InvalidStateException::builder).httpStatusCode(409).build());
    }

    @Override
    public final ConnectCampaignsServiceClientConfiguration serviceClientConfiguration() {
        return this.serviceClientConfiguration;
    }

    @Override
    public void close() {
        clientHandler.close();
    }
}
