/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.computeoptimizer.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * The estimated monthly savings after you adjust the configurations of your instances running on the inferred workload
 * types to the recommended configurations. If the <code>inferredWorkloadTypes</code> list contains multiple entries,
 * then the savings are the sum of the monthly savings from instances that run the exact combination of the inferred
 * workload types.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class InferredWorkloadSaving implements SdkPojo, Serializable,
        ToCopyableBuilder<InferredWorkloadSaving.Builder, InferredWorkloadSaving> {
    private static final SdkField<List<String>> INFERRED_WORKLOAD_TYPES_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("inferredWorkloadTypes")
            .getter(getter(InferredWorkloadSaving::inferredWorkloadTypesAsStrings))
            .setter(setter(Builder::inferredWorkloadTypesWithStrings))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("inferredWorkloadTypes").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<EstimatedMonthlySavings> ESTIMATED_MONTHLY_SAVINGS_FIELD = SdkField
            .<EstimatedMonthlySavings> builder(MarshallingType.SDK_POJO).memberName("estimatedMonthlySavings")
            .getter(getter(InferredWorkloadSaving::estimatedMonthlySavings)).setter(setter(Builder::estimatedMonthlySavings))
            .constructor(EstimatedMonthlySavings::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("estimatedMonthlySavings").build())
            .build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(INFERRED_WORKLOAD_TYPES_FIELD,
            ESTIMATED_MONTHLY_SAVINGS_FIELD));

    private static final long serialVersionUID = 1L;

    private final List<String> inferredWorkloadTypes;

    private final EstimatedMonthlySavings estimatedMonthlySavings;

    private InferredWorkloadSaving(BuilderImpl builder) {
        this.inferredWorkloadTypes = builder.inferredWorkloadTypes;
        this.estimatedMonthlySavings = builder.estimatedMonthlySavings;
    }

    /**
     * <p>
     * The applications that might be running on the instance as inferred by Compute Optimizer.
     * </p>
     * <p>
     * Compute Optimizer can infer if one of the following applications might be running on the instance:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>AmazonEmr</code> - Infers that Amazon EMR might be running on the instance.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>ApacheCassandra</code> - Infers that Apache Cassandra might be running on the instance.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>ApacheHadoop</code> - Infers that Apache Hadoop might be running on the instance.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>Memcached</code> - Infers that Memcached might be running on the instance.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>NGINX</code> - Infers that NGINX might be running on the instance.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>PostgreSql</code> - Infers that PostgreSQL might be running on the instance.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>Redis</code> - Infers that Redis might be running on the instance.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>Kafka</code> - Infers that Kafka might be running on the instance.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>SQLServer</code> - Infers that SQLServer might be running on the instance.
     * </p>
     * </li>
     * </ul>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasInferredWorkloadTypes} method.
     * </p>
     * 
     * @return The applications that might be running on the instance as inferred by Compute Optimizer.</p>
     *         <p>
     *         Compute Optimizer can infer if one of the following applications might be running on the instance:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>AmazonEmr</code> - Infers that Amazon EMR might be running on the instance.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>ApacheCassandra</code> - Infers that Apache Cassandra might be running on the instance.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>ApacheHadoop</code> - Infers that Apache Hadoop might be running on the instance.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>Memcached</code> - Infers that Memcached might be running on the instance.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>NGINX</code> - Infers that NGINX might be running on the instance.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>PostgreSql</code> - Infers that PostgreSQL might be running on the instance.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>Redis</code> - Infers that Redis might be running on the instance.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>Kafka</code> - Infers that Kafka might be running on the instance.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>SQLServer</code> - Infers that SQLServer might be running on the instance.
     *         </p>
     *         </li>
     */
    public final List<InferredWorkloadType> inferredWorkloadTypes() {
        return InferredWorkloadTypesCopier.copyStringToEnum(inferredWorkloadTypes);
    }

    /**
     * For responses, this returns true if the service returned a value for the InferredWorkloadTypes property. This
     * DOES NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the
     * property). This is useful because the SDK will never return a null collection or map, but you may need to
     * differentiate between the service returning nothing (or null) and the service returning an empty collection or
     * map. For requests, this returns true if a value for the property was specified in the request builder, and false
     * if a value was not specified.
     */
    public final boolean hasInferredWorkloadTypes() {
        return inferredWorkloadTypes != null && !(inferredWorkloadTypes instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The applications that might be running on the instance as inferred by Compute Optimizer.
     * </p>
     * <p>
     * Compute Optimizer can infer if one of the following applications might be running on the instance:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>AmazonEmr</code> - Infers that Amazon EMR might be running on the instance.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>ApacheCassandra</code> - Infers that Apache Cassandra might be running on the instance.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>ApacheHadoop</code> - Infers that Apache Hadoop might be running on the instance.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>Memcached</code> - Infers that Memcached might be running on the instance.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>NGINX</code> - Infers that NGINX might be running on the instance.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>PostgreSql</code> - Infers that PostgreSQL might be running on the instance.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>Redis</code> - Infers that Redis might be running on the instance.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>Kafka</code> - Infers that Kafka might be running on the instance.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>SQLServer</code> - Infers that SQLServer might be running on the instance.
     * </p>
     * </li>
     * </ul>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasInferredWorkloadTypes} method.
     * </p>
     * 
     * @return The applications that might be running on the instance as inferred by Compute Optimizer.</p>
     *         <p>
     *         Compute Optimizer can infer if one of the following applications might be running on the instance:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>AmazonEmr</code> - Infers that Amazon EMR might be running on the instance.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>ApacheCassandra</code> - Infers that Apache Cassandra might be running on the instance.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>ApacheHadoop</code> - Infers that Apache Hadoop might be running on the instance.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>Memcached</code> - Infers that Memcached might be running on the instance.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>NGINX</code> - Infers that NGINX might be running on the instance.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>PostgreSql</code> - Infers that PostgreSQL might be running on the instance.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>Redis</code> - Infers that Redis might be running on the instance.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>Kafka</code> - Infers that Kafka might be running on the instance.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>SQLServer</code> - Infers that SQLServer might be running on the instance.
     *         </p>
     *         </li>
     */
    public final List<String> inferredWorkloadTypesAsStrings() {
        return inferredWorkloadTypes;
    }

    /**
     * <p>
     * An object that describes the estimated monthly savings amount possible by adopting Compute Optimizer
     * recommendations for a given resource. This is based on the On-Demand instance pricing.
     * </p>
     * 
     * @return An object that describes the estimated monthly savings amount possible by adopting Compute Optimizer
     *         recommendations for a given resource. This is based on the On-Demand instance pricing.
     */
    public final EstimatedMonthlySavings estimatedMonthlySavings() {
        return estimatedMonthlySavings;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(hasInferredWorkloadTypes() ? inferredWorkloadTypesAsStrings() : null);
        hashCode = 31 * hashCode + Objects.hashCode(estimatedMonthlySavings());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof InferredWorkloadSaving)) {
            return false;
        }
        InferredWorkloadSaving other = (InferredWorkloadSaving) obj;
        return hasInferredWorkloadTypes() == other.hasInferredWorkloadTypes()
                && Objects.equals(inferredWorkloadTypesAsStrings(), other.inferredWorkloadTypesAsStrings())
                && Objects.equals(estimatedMonthlySavings(), other.estimatedMonthlySavings());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("InferredWorkloadSaving")
                .add("InferredWorkloadTypes", hasInferredWorkloadTypes() ? inferredWorkloadTypesAsStrings() : null)
                .add("EstimatedMonthlySavings", estimatedMonthlySavings()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "inferredWorkloadTypes":
            return Optional.ofNullable(clazz.cast(inferredWorkloadTypesAsStrings()));
        case "estimatedMonthlySavings":
            return Optional.ofNullable(clazz.cast(estimatedMonthlySavings()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<InferredWorkloadSaving, T> g) {
        return obj -> g.apply((InferredWorkloadSaving) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, InferredWorkloadSaving> {
        /**
         * <p>
         * The applications that might be running on the instance as inferred by Compute Optimizer.
         * </p>
         * <p>
         * Compute Optimizer can infer if one of the following applications might be running on the instance:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>AmazonEmr</code> - Infers that Amazon EMR might be running on the instance.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>ApacheCassandra</code> - Infers that Apache Cassandra might be running on the instance.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>ApacheHadoop</code> - Infers that Apache Hadoop might be running on the instance.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>Memcached</code> - Infers that Memcached might be running on the instance.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>NGINX</code> - Infers that NGINX might be running on the instance.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>PostgreSql</code> - Infers that PostgreSQL might be running on the instance.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>Redis</code> - Infers that Redis might be running on the instance.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>Kafka</code> - Infers that Kafka might be running on the instance.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>SQLServer</code> - Infers that SQLServer might be running on the instance.
         * </p>
         * </li>
         * </ul>
         * 
         * @param inferredWorkloadTypes
         *        The applications that might be running on the instance as inferred by Compute Optimizer.</p>
         *        <p>
         *        Compute Optimizer can infer if one of the following applications might be running on the instance:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>AmazonEmr</code> - Infers that Amazon EMR might be running on the instance.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>ApacheCassandra</code> - Infers that Apache Cassandra might be running on the instance.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>ApacheHadoop</code> - Infers that Apache Hadoop might be running on the instance.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>Memcached</code> - Infers that Memcached might be running on the instance.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>NGINX</code> - Infers that NGINX might be running on the instance.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>PostgreSql</code> - Infers that PostgreSQL might be running on the instance.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>Redis</code> - Infers that Redis might be running on the instance.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>Kafka</code> - Infers that Kafka might be running on the instance.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>SQLServer</code> - Infers that SQLServer might be running on the instance.
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder inferredWorkloadTypesWithStrings(Collection<String> inferredWorkloadTypes);

        /**
         * <p>
         * The applications that might be running on the instance as inferred by Compute Optimizer.
         * </p>
         * <p>
         * Compute Optimizer can infer if one of the following applications might be running on the instance:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>AmazonEmr</code> - Infers that Amazon EMR might be running on the instance.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>ApacheCassandra</code> - Infers that Apache Cassandra might be running on the instance.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>ApacheHadoop</code> - Infers that Apache Hadoop might be running on the instance.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>Memcached</code> - Infers that Memcached might be running on the instance.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>NGINX</code> - Infers that NGINX might be running on the instance.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>PostgreSql</code> - Infers that PostgreSQL might be running on the instance.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>Redis</code> - Infers that Redis might be running on the instance.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>Kafka</code> - Infers that Kafka might be running on the instance.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>SQLServer</code> - Infers that SQLServer might be running on the instance.
         * </p>
         * </li>
         * </ul>
         * 
         * @param inferredWorkloadTypes
         *        The applications that might be running on the instance as inferred by Compute Optimizer.</p>
         *        <p>
         *        Compute Optimizer can infer if one of the following applications might be running on the instance:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>AmazonEmr</code> - Infers that Amazon EMR might be running on the instance.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>ApacheCassandra</code> - Infers that Apache Cassandra might be running on the instance.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>ApacheHadoop</code> - Infers that Apache Hadoop might be running on the instance.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>Memcached</code> - Infers that Memcached might be running on the instance.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>NGINX</code> - Infers that NGINX might be running on the instance.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>PostgreSql</code> - Infers that PostgreSQL might be running on the instance.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>Redis</code> - Infers that Redis might be running on the instance.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>Kafka</code> - Infers that Kafka might be running on the instance.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>SQLServer</code> - Infers that SQLServer might be running on the instance.
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder inferredWorkloadTypesWithStrings(String... inferredWorkloadTypes);

        /**
         * <p>
         * The applications that might be running on the instance as inferred by Compute Optimizer.
         * </p>
         * <p>
         * Compute Optimizer can infer if one of the following applications might be running on the instance:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>AmazonEmr</code> - Infers that Amazon EMR might be running on the instance.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>ApacheCassandra</code> - Infers that Apache Cassandra might be running on the instance.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>ApacheHadoop</code> - Infers that Apache Hadoop might be running on the instance.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>Memcached</code> - Infers that Memcached might be running on the instance.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>NGINX</code> - Infers that NGINX might be running on the instance.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>PostgreSql</code> - Infers that PostgreSQL might be running on the instance.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>Redis</code> - Infers that Redis might be running on the instance.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>Kafka</code> - Infers that Kafka might be running on the instance.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>SQLServer</code> - Infers that SQLServer might be running on the instance.
         * </p>
         * </li>
         * </ul>
         * 
         * @param inferredWorkloadTypes
         *        The applications that might be running on the instance as inferred by Compute Optimizer.</p>
         *        <p>
         *        Compute Optimizer can infer if one of the following applications might be running on the instance:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>AmazonEmr</code> - Infers that Amazon EMR might be running on the instance.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>ApacheCassandra</code> - Infers that Apache Cassandra might be running on the instance.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>ApacheHadoop</code> - Infers that Apache Hadoop might be running on the instance.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>Memcached</code> - Infers that Memcached might be running on the instance.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>NGINX</code> - Infers that NGINX might be running on the instance.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>PostgreSql</code> - Infers that PostgreSQL might be running on the instance.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>Redis</code> - Infers that Redis might be running on the instance.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>Kafka</code> - Infers that Kafka might be running on the instance.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>SQLServer</code> - Infers that SQLServer might be running on the instance.
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder inferredWorkloadTypes(Collection<InferredWorkloadType> inferredWorkloadTypes);

        /**
         * <p>
         * The applications that might be running on the instance as inferred by Compute Optimizer.
         * </p>
         * <p>
         * Compute Optimizer can infer if one of the following applications might be running on the instance:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>AmazonEmr</code> - Infers that Amazon EMR might be running on the instance.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>ApacheCassandra</code> - Infers that Apache Cassandra might be running on the instance.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>ApacheHadoop</code> - Infers that Apache Hadoop might be running on the instance.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>Memcached</code> - Infers that Memcached might be running on the instance.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>NGINX</code> - Infers that NGINX might be running on the instance.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>PostgreSql</code> - Infers that PostgreSQL might be running on the instance.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>Redis</code> - Infers that Redis might be running on the instance.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>Kafka</code> - Infers that Kafka might be running on the instance.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>SQLServer</code> - Infers that SQLServer might be running on the instance.
         * </p>
         * </li>
         * </ul>
         * 
         * @param inferredWorkloadTypes
         *        The applications that might be running on the instance as inferred by Compute Optimizer.</p>
         *        <p>
         *        Compute Optimizer can infer if one of the following applications might be running on the instance:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>AmazonEmr</code> - Infers that Amazon EMR might be running on the instance.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>ApacheCassandra</code> - Infers that Apache Cassandra might be running on the instance.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>ApacheHadoop</code> - Infers that Apache Hadoop might be running on the instance.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>Memcached</code> - Infers that Memcached might be running on the instance.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>NGINX</code> - Infers that NGINX might be running on the instance.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>PostgreSql</code> - Infers that PostgreSQL might be running on the instance.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>Redis</code> - Infers that Redis might be running on the instance.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>Kafka</code> - Infers that Kafka might be running on the instance.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>SQLServer</code> - Infers that SQLServer might be running on the instance.
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder inferredWorkloadTypes(InferredWorkloadType... inferredWorkloadTypes);

        /**
         * <p>
         * An object that describes the estimated monthly savings amount possible by adopting Compute Optimizer
         * recommendations for a given resource. This is based on the On-Demand instance pricing.
         * </p>
         * 
         * @param estimatedMonthlySavings
         *        An object that describes the estimated monthly savings amount possible by adopting Compute Optimizer
         *        recommendations for a given resource. This is based on the On-Demand instance pricing.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder estimatedMonthlySavings(EstimatedMonthlySavings estimatedMonthlySavings);

        /**
         * <p>
         * An object that describes the estimated monthly savings amount possible by adopting Compute Optimizer
         * recommendations for a given resource. This is based on the On-Demand instance pricing.
         * </p>
         * This is a convenience method that creates an instance of the {@link EstimatedMonthlySavings.Builder} avoiding
         * the need to create one manually via {@link EstimatedMonthlySavings#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link EstimatedMonthlySavings.Builder#build()} is called immediately
         * and its result is passed to {@link #estimatedMonthlySavings(EstimatedMonthlySavings)}.
         * 
         * @param estimatedMonthlySavings
         *        a consumer that will call methods on {@link EstimatedMonthlySavings.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #estimatedMonthlySavings(EstimatedMonthlySavings)
         */
        default Builder estimatedMonthlySavings(Consumer<EstimatedMonthlySavings.Builder> estimatedMonthlySavings) {
            return estimatedMonthlySavings(EstimatedMonthlySavings.builder().applyMutation(estimatedMonthlySavings).build());
        }
    }

    static final class BuilderImpl implements Builder {
        private List<String> inferredWorkloadTypes = DefaultSdkAutoConstructList.getInstance();

        private EstimatedMonthlySavings estimatedMonthlySavings;

        private BuilderImpl() {
        }

        private BuilderImpl(InferredWorkloadSaving model) {
            inferredWorkloadTypesWithStrings(model.inferredWorkloadTypes);
            estimatedMonthlySavings(model.estimatedMonthlySavings);
        }

        public final Collection<String> getInferredWorkloadTypes() {
            if (inferredWorkloadTypes instanceof SdkAutoConstructList) {
                return null;
            }
            return inferredWorkloadTypes;
        }

        public final void setInferredWorkloadTypes(Collection<String> inferredWorkloadTypes) {
            this.inferredWorkloadTypes = InferredWorkloadTypesCopier.copy(inferredWorkloadTypes);
        }

        @Override
        public final Builder inferredWorkloadTypesWithStrings(Collection<String> inferredWorkloadTypes) {
            this.inferredWorkloadTypes = InferredWorkloadTypesCopier.copy(inferredWorkloadTypes);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder inferredWorkloadTypesWithStrings(String... inferredWorkloadTypes) {
            inferredWorkloadTypesWithStrings(Arrays.asList(inferredWorkloadTypes));
            return this;
        }

        @Override
        public final Builder inferredWorkloadTypes(Collection<InferredWorkloadType> inferredWorkloadTypes) {
            this.inferredWorkloadTypes = InferredWorkloadTypesCopier.copyEnumToString(inferredWorkloadTypes);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder inferredWorkloadTypes(InferredWorkloadType... inferredWorkloadTypes) {
            inferredWorkloadTypes(Arrays.asList(inferredWorkloadTypes));
            return this;
        }

        public final EstimatedMonthlySavings.Builder getEstimatedMonthlySavings() {
            return estimatedMonthlySavings != null ? estimatedMonthlySavings.toBuilder() : null;
        }

        public final void setEstimatedMonthlySavings(EstimatedMonthlySavings.BuilderImpl estimatedMonthlySavings) {
            this.estimatedMonthlySavings = estimatedMonthlySavings != null ? estimatedMonthlySavings.build() : null;
        }

        @Override
        public final Builder estimatedMonthlySavings(EstimatedMonthlySavings estimatedMonthlySavings) {
            this.estimatedMonthlySavings = estimatedMonthlySavings;
            return this;
        }

        @Override
        public InferredWorkloadSaving build() {
            return new InferredWorkloadSaving(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
