/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.cloudsearch;

import java.util.Collections;
import java.util.List;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.client.handler.AwsSyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.client.handler.SyncClientHandler;
import software.amazon.awssdk.core.exception.SdkClientException;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.query.AwsQueryProtocolFactory;
import software.amazon.awssdk.services.cloudsearch.model.BaseException;
import software.amazon.awssdk.services.cloudsearch.model.BuildSuggestersRequest;
import software.amazon.awssdk.services.cloudsearch.model.BuildSuggestersResponse;
import software.amazon.awssdk.services.cloudsearch.model.CloudSearchException;
import software.amazon.awssdk.services.cloudsearch.model.CreateDomainRequest;
import software.amazon.awssdk.services.cloudsearch.model.CreateDomainResponse;
import software.amazon.awssdk.services.cloudsearch.model.DefineAnalysisSchemeRequest;
import software.amazon.awssdk.services.cloudsearch.model.DefineAnalysisSchemeResponse;
import software.amazon.awssdk.services.cloudsearch.model.DefineExpressionRequest;
import software.amazon.awssdk.services.cloudsearch.model.DefineExpressionResponse;
import software.amazon.awssdk.services.cloudsearch.model.DefineIndexFieldRequest;
import software.amazon.awssdk.services.cloudsearch.model.DefineIndexFieldResponse;
import software.amazon.awssdk.services.cloudsearch.model.DefineSuggesterRequest;
import software.amazon.awssdk.services.cloudsearch.model.DefineSuggesterResponse;
import software.amazon.awssdk.services.cloudsearch.model.DeleteAnalysisSchemeRequest;
import software.amazon.awssdk.services.cloudsearch.model.DeleteAnalysisSchemeResponse;
import software.amazon.awssdk.services.cloudsearch.model.DeleteDomainRequest;
import software.amazon.awssdk.services.cloudsearch.model.DeleteDomainResponse;
import software.amazon.awssdk.services.cloudsearch.model.DeleteExpressionRequest;
import software.amazon.awssdk.services.cloudsearch.model.DeleteExpressionResponse;
import software.amazon.awssdk.services.cloudsearch.model.DeleteIndexFieldRequest;
import software.amazon.awssdk.services.cloudsearch.model.DeleteIndexFieldResponse;
import software.amazon.awssdk.services.cloudsearch.model.DeleteSuggesterRequest;
import software.amazon.awssdk.services.cloudsearch.model.DeleteSuggesterResponse;
import software.amazon.awssdk.services.cloudsearch.model.DescribeAnalysisSchemesRequest;
import software.amazon.awssdk.services.cloudsearch.model.DescribeAnalysisSchemesResponse;
import software.amazon.awssdk.services.cloudsearch.model.DescribeAvailabilityOptionsRequest;
import software.amazon.awssdk.services.cloudsearch.model.DescribeAvailabilityOptionsResponse;
import software.amazon.awssdk.services.cloudsearch.model.DescribeDomainEndpointOptionsRequest;
import software.amazon.awssdk.services.cloudsearch.model.DescribeDomainEndpointOptionsResponse;
import software.amazon.awssdk.services.cloudsearch.model.DescribeDomainsRequest;
import software.amazon.awssdk.services.cloudsearch.model.DescribeDomainsResponse;
import software.amazon.awssdk.services.cloudsearch.model.DescribeExpressionsRequest;
import software.amazon.awssdk.services.cloudsearch.model.DescribeExpressionsResponse;
import software.amazon.awssdk.services.cloudsearch.model.DescribeIndexFieldsRequest;
import software.amazon.awssdk.services.cloudsearch.model.DescribeIndexFieldsResponse;
import software.amazon.awssdk.services.cloudsearch.model.DescribeScalingParametersRequest;
import software.amazon.awssdk.services.cloudsearch.model.DescribeScalingParametersResponse;
import software.amazon.awssdk.services.cloudsearch.model.DescribeServiceAccessPoliciesRequest;
import software.amazon.awssdk.services.cloudsearch.model.DescribeServiceAccessPoliciesResponse;
import software.amazon.awssdk.services.cloudsearch.model.DescribeSuggestersRequest;
import software.amazon.awssdk.services.cloudsearch.model.DescribeSuggestersResponse;
import software.amazon.awssdk.services.cloudsearch.model.DisabledOperationException;
import software.amazon.awssdk.services.cloudsearch.model.IndexDocumentsRequest;
import software.amazon.awssdk.services.cloudsearch.model.IndexDocumentsResponse;
import software.amazon.awssdk.services.cloudsearch.model.InternalException;
import software.amazon.awssdk.services.cloudsearch.model.InvalidTypeException;
import software.amazon.awssdk.services.cloudsearch.model.LimitExceededException;
import software.amazon.awssdk.services.cloudsearch.model.ListDomainNamesRequest;
import software.amazon.awssdk.services.cloudsearch.model.ListDomainNamesResponse;
import software.amazon.awssdk.services.cloudsearch.model.ResourceAlreadyExistsException;
import software.amazon.awssdk.services.cloudsearch.model.ResourceNotFoundException;
import software.amazon.awssdk.services.cloudsearch.model.UpdateAvailabilityOptionsRequest;
import software.amazon.awssdk.services.cloudsearch.model.UpdateAvailabilityOptionsResponse;
import software.amazon.awssdk.services.cloudsearch.model.UpdateDomainEndpointOptionsRequest;
import software.amazon.awssdk.services.cloudsearch.model.UpdateDomainEndpointOptionsResponse;
import software.amazon.awssdk.services.cloudsearch.model.UpdateScalingParametersRequest;
import software.amazon.awssdk.services.cloudsearch.model.UpdateScalingParametersResponse;
import software.amazon.awssdk.services.cloudsearch.model.UpdateServiceAccessPoliciesRequest;
import software.amazon.awssdk.services.cloudsearch.model.UpdateServiceAccessPoliciesResponse;
import software.amazon.awssdk.services.cloudsearch.model.ValidationException;
import software.amazon.awssdk.services.cloudsearch.transform.BuildSuggestersRequestMarshaller;
import software.amazon.awssdk.services.cloudsearch.transform.CreateDomainRequestMarshaller;
import software.amazon.awssdk.services.cloudsearch.transform.DefineAnalysisSchemeRequestMarshaller;
import software.amazon.awssdk.services.cloudsearch.transform.DefineExpressionRequestMarshaller;
import software.amazon.awssdk.services.cloudsearch.transform.DefineIndexFieldRequestMarshaller;
import software.amazon.awssdk.services.cloudsearch.transform.DefineSuggesterRequestMarshaller;
import software.amazon.awssdk.services.cloudsearch.transform.DeleteAnalysisSchemeRequestMarshaller;
import software.amazon.awssdk.services.cloudsearch.transform.DeleteDomainRequestMarshaller;
import software.amazon.awssdk.services.cloudsearch.transform.DeleteExpressionRequestMarshaller;
import software.amazon.awssdk.services.cloudsearch.transform.DeleteIndexFieldRequestMarshaller;
import software.amazon.awssdk.services.cloudsearch.transform.DeleteSuggesterRequestMarshaller;
import software.amazon.awssdk.services.cloudsearch.transform.DescribeAnalysisSchemesRequestMarshaller;
import software.amazon.awssdk.services.cloudsearch.transform.DescribeAvailabilityOptionsRequestMarshaller;
import software.amazon.awssdk.services.cloudsearch.transform.DescribeDomainEndpointOptionsRequestMarshaller;
import software.amazon.awssdk.services.cloudsearch.transform.DescribeDomainsRequestMarshaller;
import software.amazon.awssdk.services.cloudsearch.transform.DescribeExpressionsRequestMarshaller;
import software.amazon.awssdk.services.cloudsearch.transform.DescribeIndexFieldsRequestMarshaller;
import software.amazon.awssdk.services.cloudsearch.transform.DescribeScalingParametersRequestMarshaller;
import software.amazon.awssdk.services.cloudsearch.transform.DescribeServiceAccessPoliciesRequestMarshaller;
import software.amazon.awssdk.services.cloudsearch.transform.DescribeSuggestersRequestMarshaller;
import software.amazon.awssdk.services.cloudsearch.transform.IndexDocumentsRequestMarshaller;
import software.amazon.awssdk.services.cloudsearch.transform.ListDomainNamesRequestMarshaller;
import software.amazon.awssdk.services.cloudsearch.transform.UpdateAvailabilityOptionsRequestMarshaller;
import software.amazon.awssdk.services.cloudsearch.transform.UpdateDomainEndpointOptionsRequestMarshaller;
import software.amazon.awssdk.services.cloudsearch.transform.UpdateScalingParametersRequestMarshaller;
import software.amazon.awssdk.services.cloudsearch.transform.UpdateServiceAccessPoliciesRequestMarshaller;
import software.amazon.awssdk.utils.Logger;

/**
 * Internal implementation of {@link CloudSearchClient}.
 *
 * @see CloudSearchClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultCloudSearchClient implements CloudSearchClient {
    private static final Logger log = Logger.loggerFor(DefaultCloudSearchClient.class);

    private final SyncClientHandler clientHandler;

    private final AwsQueryProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    private final CloudSearchServiceClientConfiguration serviceClientConfiguration;

    protected DefaultCloudSearchClient(CloudSearchServiceClientConfiguration serviceClientConfiguration,
            SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsSyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.serviceClientConfiguration = serviceClientConfiguration;
        this.protocolFactory = init();
    }

    /**
     * <p>
     * Indexes the search suggestions. For more information, see <a href=
     * "http://docs.aws.amazon.com/cloudsearch/latest/developerguide/getting-suggestions.html#configuring-suggesters"
     * >Configuring Suggesters</a> in the <i>Amazon CloudSearch Developer Guide</i>.
     * </p>
     *
     * @param buildSuggestersRequest
     *        Container for the parameters to the <code><a>BuildSuggester</a></code> operation. Specifies the name of
     *        the domain you want to update.
     * @return Result of the BuildSuggesters operation returned by the service.
     * @throws BaseException
     *         An error occurred while processing the request.
     * @throws InternalException
     *         An internal error occurred while processing the request. If this problem persists, report an issue from
     *         the <a href="http://status.aws.amazon.com/" target="_blank">Service Health Dashboard</a>.
     * @throws ResourceNotFoundException
     *         The request was rejected because it attempted to reference a resource that does not exist.
     * @throws ValidationException
     *         The request was rejected because it has invalid parameters.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CloudSearchException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CloudSearchClient.BuildSuggesters
     */
    @Override
    public BuildSuggestersResponse buildSuggesters(BuildSuggestersRequest buildSuggestersRequest) throws BaseException,
            InternalException, ResourceNotFoundException, ValidationException, AwsServiceException, SdkClientException,
            CloudSearchException {

        HttpResponseHandler<BuildSuggestersResponse> responseHandler = protocolFactory
                .createResponseHandler(BuildSuggestersResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, buildSuggestersRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CloudSearch");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "BuildSuggesters");

            return clientHandler.execute(new ClientExecutionParams<BuildSuggestersRequest, BuildSuggestersResponse>()
                    .withOperationName("BuildSuggesters").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(buildSuggestersRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new BuildSuggestersRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Creates a new search domain. For more information, see <a
     * href="http://docs.aws.amazon.com/cloudsearch/latest/developerguide/creating-domains.html"
     * target="_blank">Creating a Search Domain</a> in the <i>Amazon CloudSearch Developer Guide</i>.
     * </p>
     *
     * @param createDomainRequest
     *        Container for the parameters to the <code><a>CreateDomain</a></code> operation. Specifies a name for the
     *        new search domain.
     * @return Result of the CreateDomain operation returned by the service.
     * @throws BaseException
     *         An error occurred while processing the request.
     * @throws InternalException
     *         An internal error occurred while processing the request. If this problem persists, report an issue from
     *         the <a href="http://status.aws.amazon.com/" target="_blank">Service Health Dashboard</a>.
     * @throws LimitExceededException
     *         The request was rejected because a resource limit has already been met.
     * @throws ResourceAlreadyExistsException
     *         The request was rejected because it attempted to create a resource that already exists.
     * @throws ValidationException
     *         The request was rejected because it has invalid parameters.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CloudSearchException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CloudSearchClient.CreateDomain
     */
    @Override
    public CreateDomainResponse createDomain(CreateDomainRequest createDomainRequest) throws BaseException, InternalException,
            LimitExceededException, ResourceAlreadyExistsException, ValidationException, AwsServiceException, SdkClientException,
            CloudSearchException {

        HttpResponseHandler<CreateDomainResponse> responseHandler = protocolFactory
                .createResponseHandler(CreateDomainResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createDomainRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CloudSearch");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateDomain");

            return clientHandler.execute(new ClientExecutionParams<CreateDomainRequest, CreateDomainResponse>()
                    .withOperationName("CreateDomain").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(createDomainRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new CreateDomainRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Configures an analysis scheme that can be applied to a <code>text</code> or <code>text-array</code> field to
     * define language-specific text processing options. For more information, see <a
     * href="http://docs.aws.amazon.com/cloudsearch/latest/developerguide/configuring-analysis-schemes.html"
     * target="_blank">Configuring Analysis Schemes</a> in the <i>Amazon CloudSearch Developer Guide</i>.
     * </p>
     *
     * @param defineAnalysisSchemeRequest
     *        Container for the parameters to the <code><a>DefineAnalysisScheme</a></code> operation. Specifies the name
     *        of the domain you want to update and the analysis scheme configuration.
     * @return Result of the DefineAnalysisScheme operation returned by the service.
     * @throws BaseException
     *         An error occurred while processing the request.
     * @throws InternalException
     *         An internal error occurred while processing the request. If this problem persists, report an issue from
     *         the <a href="http://status.aws.amazon.com/" target="_blank">Service Health Dashboard</a>.
     * @throws LimitExceededException
     *         The request was rejected because a resource limit has already been met.
     * @throws InvalidTypeException
     *         The request was rejected because it specified an invalid type definition.
     * @throws ResourceNotFoundException
     *         The request was rejected because it attempted to reference a resource that does not exist.
     * @throws ValidationException
     *         The request was rejected because it has invalid parameters.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CloudSearchException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CloudSearchClient.DefineAnalysisScheme
     */
    @Override
    public DefineAnalysisSchemeResponse defineAnalysisScheme(DefineAnalysisSchemeRequest defineAnalysisSchemeRequest)
            throws BaseException, InternalException, LimitExceededException, InvalidTypeException, ResourceNotFoundException,
            ValidationException, AwsServiceException, SdkClientException, CloudSearchException {

        HttpResponseHandler<DefineAnalysisSchemeResponse> responseHandler = protocolFactory
                .createResponseHandler(DefineAnalysisSchemeResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, defineAnalysisSchemeRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CloudSearch");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DefineAnalysisScheme");

            return clientHandler.execute(new ClientExecutionParams<DefineAnalysisSchemeRequest, DefineAnalysisSchemeResponse>()
                    .withOperationName("DefineAnalysisScheme").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(defineAnalysisSchemeRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DefineAnalysisSchemeRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Configures an <code><a>Expression</a></code> for the search domain. Used to create new expressions and modify
     * existing ones. If the expression exists, the new configuration replaces the old one. For more information, see <a
     * href="http://docs.aws.amazon.com/cloudsearch/latest/developerguide/configuring-expressions.html"
     * target="_blank">Configuring Expressions</a> in the <i>Amazon CloudSearch Developer Guide</i>.
     * </p>
     *
     * @param defineExpressionRequest
     *        Container for the parameters to the <code><a>DefineExpression</a></code> operation. Specifies the name of
     *        the domain you want to update and the expression you want to configure.
     * @return Result of the DefineExpression operation returned by the service.
     * @throws BaseException
     *         An error occurred while processing the request.
     * @throws InternalException
     *         An internal error occurred while processing the request. If this problem persists, report an issue from
     *         the <a href="http://status.aws.amazon.com/" target="_blank">Service Health Dashboard</a>.
     * @throws LimitExceededException
     *         The request was rejected because a resource limit has already been met.
     * @throws InvalidTypeException
     *         The request was rejected because it specified an invalid type definition.
     * @throws ResourceNotFoundException
     *         The request was rejected because it attempted to reference a resource that does not exist.
     * @throws ValidationException
     *         The request was rejected because it has invalid parameters.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CloudSearchException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CloudSearchClient.DefineExpression
     */
    @Override
    public DefineExpressionResponse defineExpression(DefineExpressionRequest defineExpressionRequest) throws BaseException,
            InternalException, LimitExceededException, InvalidTypeException, ResourceNotFoundException, ValidationException,
            AwsServiceException, SdkClientException, CloudSearchException {

        HttpResponseHandler<DefineExpressionResponse> responseHandler = protocolFactory
                .createResponseHandler(DefineExpressionResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, defineExpressionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CloudSearch");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DefineExpression");

            return clientHandler.execute(new ClientExecutionParams<DefineExpressionRequest, DefineExpressionResponse>()
                    .withOperationName("DefineExpression").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(defineExpressionRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DefineExpressionRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Configures an <code><a>IndexField</a></code> for the search domain. Used to create new fields and modify existing
     * ones. You must specify the name of the domain you are configuring and an index field configuration. The index
     * field configuration specifies a unique name, the index field type, and the options you want to configure for the
     * field. The options you can specify depend on the <code><a>IndexFieldType</a></code>. If the field exists, the new
     * configuration replaces the old one. For more information, see <a
     * href="http://docs.aws.amazon.com/cloudsearch/latest/developerguide/configuring-index-fields.html"
     * target="_blank">Configuring Index Fields</a> in the <i>Amazon CloudSearch Developer Guide</i>.
     * </p>
     *
     * @param defineIndexFieldRequest
     *        Container for the parameters to the <code><a>DefineIndexField</a></code> operation. Specifies the name of
     *        the domain you want to update and the index field configuration.
     * @return Result of the DefineIndexField operation returned by the service.
     * @throws BaseException
     *         An error occurred while processing the request.
     * @throws InternalException
     *         An internal error occurred while processing the request. If this problem persists, report an issue from
     *         the <a href="http://status.aws.amazon.com/" target="_blank">Service Health Dashboard</a>.
     * @throws LimitExceededException
     *         The request was rejected because a resource limit has already been met.
     * @throws InvalidTypeException
     *         The request was rejected because it specified an invalid type definition.
     * @throws ResourceNotFoundException
     *         The request was rejected because it attempted to reference a resource that does not exist.
     * @throws ValidationException
     *         The request was rejected because it has invalid parameters.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CloudSearchException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CloudSearchClient.DefineIndexField
     */
    @Override
    public DefineIndexFieldResponse defineIndexField(DefineIndexFieldRequest defineIndexFieldRequest) throws BaseException,
            InternalException, LimitExceededException, InvalidTypeException, ResourceNotFoundException, ValidationException,
            AwsServiceException, SdkClientException, CloudSearchException {

        HttpResponseHandler<DefineIndexFieldResponse> responseHandler = protocolFactory
                .createResponseHandler(DefineIndexFieldResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, defineIndexFieldRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CloudSearch");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DefineIndexField");

            return clientHandler.execute(new ClientExecutionParams<DefineIndexFieldRequest, DefineIndexFieldResponse>()
                    .withOperationName("DefineIndexField").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(defineIndexFieldRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DefineIndexFieldRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Configures a suggester for a domain. A suggester enables you to display possible matches before users finish
     * typing their queries. When you configure a suggester, you must specify the name of the text field you want to
     * search for possible matches and a unique name for the suggester. For more information, see <a
     * href="http://docs.aws.amazon.com/cloudsearch/latest/developerguide/getting-suggestions.html"
     * target="_blank">Getting Search Suggestions</a> in the <i>Amazon CloudSearch Developer Guide</i>.
     * </p>
     *
     * @param defineSuggesterRequest
     *        Container for the parameters to the <code><a>DefineSuggester</a></code> operation. Specifies the name of
     *        the domain you want to update and the suggester configuration.
     * @return Result of the DefineSuggester operation returned by the service.
     * @throws BaseException
     *         An error occurred while processing the request.
     * @throws InternalException
     *         An internal error occurred while processing the request. If this problem persists, report an issue from
     *         the <a href="http://status.aws.amazon.com/" target="_blank">Service Health Dashboard</a>.
     * @throws LimitExceededException
     *         The request was rejected because a resource limit has already been met.
     * @throws InvalidTypeException
     *         The request was rejected because it specified an invalid type definition.
     * @throws ResourceNotFoundException
     *         The request was rejected because it attempted to reference a resource that does not exist.
     * @throws ValidationException
     *         The request was rejected because it has invalid parameters.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CloudSearchException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CloudSearchClient.DefineSuggester
     */
    @Override
    public DefineSuggesterResponse defineSuggester(DefineSuggesterRequest defineSuggesterRequest) throws BaseException,
            InternalException, LimitExceededException, InvalidTypeException, ResourceNotFoundException, ValidationException,
            AwsServiceException, SdkClientException, CloudSearchException {

        HttpResponseHandler<DefineSuggesterResponse> responseHandler = protocolFactory
                .createResponseHandler(DefineSuggesterResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, defineSuggesterRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CloudSearch");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DefineSuggester");

            return clientHandler.execute(new ClientExecutionParams<DefineSuggesterRequest, DefineSuggesterResponse>()
                    .withOperationName("DefineSuggester").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(defineSuggesterRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DefineSuggesterRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deletes an analysis scheme. For more information, see <a
     * href="http://docs.aws.amazon.com/cloudsearch/latest/developerguide/configuring-analysis-schemes.html"
     * target="_blank">Configuring Analysis Schemes</a> in the <i>Amazon CloudSearch Developer Guide</i>.
     * </p>
     *
     * @param deleteAnalysisSchemeRequest
     *        Container for the parameters to the <code><a>DeleteAnalysisScheme</a></code> operation. Specifies the name
     *        of the domain you want to update and the analysis scheme you want to delete.
     * @return Result of the DeleteAnalysisScheme operation returned by the service.
     * @throws BaseException
     *         An error occurred while processing the request.
     * @throws InternalException
     *         An internal error occurred while processing the request. If this problem persists, report an issue from
     *         the <a href="http://status.aws.amazon.com/" target="_blank">Service Health Dashboard</a>.
     * @throws InvalidTypeException
     *         The request was rejected because it specified an invalid type definition.
     * @throws ResourceNotFoundException
     *         The request was rejected because it attempted to reference a resource that does not exist.
     * @throws ValidationException
     *         The request was rejected because it has invalid parameters.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CloudSearchException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CloudSearchClient.DeleteAnalysisScheme
     */
    @Override
    public DeleteAnalysisSchemeResponse deleteAnalysisScheme(DeleteAnalysisSchemeRequest deleteAnalysisSchemeRequest)
            throws BaseException, InternalException, InvalidTypeException, ResourceNotFoundException, ValidationException,
            AwsServiceException, SdkClientException, CloudSearchException {

        HttpResponseHandler<DeleteAnalysisSchemeResponse> responseHandler = protocolFactory
                .createResponseHandler(DeleteAnalysisSchemeResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteAnalysisSchemeRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CloudSearch");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteAnalysisScheme");

            return clientHandler.execute(new ClientExecutionParams<DeleteAnalysisSchemeRequest, DeleteAnalysisSchemeResponse>()
                    .withOperationName("DeleteAnalysisScheme").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(deleteAnalysisSchemeRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DeleteAnalysisSchemeRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Permanently deletes a search domain and all of its data. Once a domain has been deleted, it cannot be recovered.
     * For more information, see <a
     * href="http://docs.aws.amazon.com/cloudsearch/latest/developerguide/deleting-domains.html"
     * target="_blank">Deleting a Search Domain</a> in the <i>Amazon CloudSearch Developer Guide</i>.
     * </p>
     *
     * @param deleteDomainRequest
     *        Container for the parameters to the <code><a>DeleteDomain</a></code> operation. Specifies the name of the
     *        domain you want to delete.
     * @return Result of the DeleteDomain operation returned by the service.
     * @throws BaseException
     *         An error occurred while processing the request.
     * @throws InternalException
     *         An internal error occurred while processing the request. If this problem persists, report an issue from
     *         the <a href="http://status.aws.amazon.com/" target="_blank">Service Health Dashboard</a>.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CloudSearchException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CloudSearchClient.DeleteDomain
     */
    @Override
    public DeleteDomainResponse deleteDomain(DeleteDomainRequest deleteDomainRequest) throws BaseException, InternalException,
            AwsServiceException, SdkClientException, CloudSearchException {

        HttpResponseHandler<DeleteDomainResponse> responseHandler = protocolFactory
                .createResponseHandler(DeleteDomainResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteDomainRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CloudSearch");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteDomain");

            return clientHandler.execute(new ClientExecutionParams<DeleteDomainRequest, DeleteDomainResponse>()
                    .withOperationName("DeleteDomain").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(deleteDomainRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DeleteDomainRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Removes an <code><a>Expression</a></code> from the search domain. For more information, see <a
     * href="http://docs.aws.amazon.com/cloudsearch/latest/developerguide/configuring-expressions.html"
     * target="_blank">Configuring Expressions</a> in the <i>Amazon CloudSearch Developer Guide</i>.
     * </p>
     *
     * @param deleteExpressionRequest
     *        Container for the parameters to the <code><a>DeleteExpression</a></code> operation. Specifies the name of
     *        the domain you want to update and the name of the expression you want to delete.
     * @return Result of the DeleteExpression operation returned by the service.
     * @throws BaseException
     *         An error occurred while processing the request.
     * @throws InternalException
     *         An internal error occurred while processing the request. If this problem persists, report an issue from
     *         the <a href="http://status.aws.amazon.com/" target="_blank">Service Health Dashboard</a>.
     * @throws InvalidTypeException
     *         The request was rejected because it specified an invalid type definition.
     * @throws ResourceNotFoundException
     *         The request was rejected because it attempted to reference a resource that does not exist.
     * @throws ValidationException
     *         The request was rejected because it has invalid parameters.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CloudSearchException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CloudSearchClient.DeleteExpression
     */
    @Override
    public DeleteExpressionResponse deleteExpression(DeleteExpressionRequest deleteExpressionRequest) throws BaseException,
            InternalException, InvalidTypeException, ResourceNotFoundException, ValidationException, AwsServiceException,
            SdkClientException, CloudSearchException {

        HttpResponseHandler<DeleteExpressionResponse> responseHandler = protocolFactory
                .createResponseHandler(DeleteExpressionResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteExpressionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CloudSearch");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteExpression");

            return clientHandler.execute(new ClientExecutionParams<DeleteExpressionRequest, DeleteExpressionResponse>()
                    .withOperationName("DeleteExpression").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(deleteExpressionRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DeleteExpressionRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Removes an <code><a>IndexField</a></code> from the search domain. For more information, see <a
     * href="http://docs.aws.amazon.com/cloudsearch/latest/developerguide/configuring-index-fields.html"
     * target="_blank">Configuring Index Fields</a> in the <i>Amazon CloudSearch Developer Guide</i>.
     * </p>
     *
     * @param deleteIndexFieldRequest
     *        Container for the parameters to the <code><a>DeleteIndexField</a></code> operation. Specifies the name of
     *        the domain you want to update and the name of the index field you want to delete.
     * @return Result of the DeleteIndexField operation returned by the service.
     * @throws BaseException
     *         An error occurred while processing the request.
     * @throws InternalException
     *         An internal error occurred while processing the request. If this problem persists, report an issue from
     *         the <a href="http://status.aws.amazon.com/" target="_blank">Service Health Dashboard</a>.
     * @throws InvalidTypeException
     *         The request was rejected because it specified an invalid type definition.
     * @throws ResourceNotFoundException
     *         The request was rejected because it attempted to reference a resource that does not exist.
     * @throws ValidationException
     *         The request was rejected because it has invalid parameters.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CloudSearchException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CloudSearchClient.DeleteIndexField
     */
    @Override
    public DeleteIndexFieldResponse deleteIndexField(DeleteIndexFieldRequest deleteIndexFieldRequest) throws BaseException,
            InternalException, InvalidTypeException, ResourceNotFoundException, ValidationException, AwsServiceException,
            SdkClientException, CloudSearchException {

        HttpResponseHandler<DeleteIndexFieldResponse> responseHandler = protocolFactory
                .createResponseHandler(DeleteIndexFieldResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteIndexFieldRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CloudSearch");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteIndexField");

            return clientHandler.execute(new ClientExecutionParams<DeleteIndexFieldRequest, DeleteIndexFieldResponse>()
                    .withOperationName("DeleteIndexField").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(deleteIndexFieldRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DeleteIndexFieldRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deletes a suggester. For more information, see <a
     * href="http://docs.aws.amazon.com/cloudsearch/latest/developerguide/getting-suggestions.html"
     * target="_blank">Getting Search Suggestions</a> in the <i>Amazon CloudSearch Developer Guide</i>.
     * </p>
     *
     * @param deleteSuggesterRequest
     *        Container for the parameters to the <code><a>DeleteSuggester</a></code> operation. Specifies the name of
     *        the domain you want to update and name of the suggester you want to delete.
     * @return Result of the DeleteSuggester operation returned by the service.
     * @throws BaseException
     *         An error occurred while processing the request.
     * @throws InternalException
     *         An internal error occurred while processing the request. If this problem persists, report an issue from
     *         the <a href="http://status.aws.amazon.com/" target="_blank">Service Health Dashboard</a>.
     * @throws InvalidTypeException
     *         The request was rejected because it specified an invalid type definition.
     * @throws ResourceNotFoundException
     *         The request was rejected because it attempted to reference a resource that does not exist.
     * @throws ValidationException
     *         The request was rejected because it has invalid parameters.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CloudSearchException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CloudSearchClient.DeleteSuggester
     */
    @Override
    public DeleteSuggesterResponse deleteSuggester(DeleteSuggesterRequest deleteSuggesterRequest) throws BaseException,
            InternalException, InvalidTypeException, ResourceNotFoundException, ValidationException, AwsServiceException,
            SdkClientException, CloudSearchException {

        HttpResponseHandler<DeleteSuggesterResponse> responseHandler = protocolFactory
                .createResponseHandler(DeleteSuggesterResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteSuggesterRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CloudSearch");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteSuggester");

            return clientHandler.execute(new ClientExecutionParams<DeleteSuggesterRequest, DeleteSuggesterResponse>()
                    .withOperationName("DeleteSuggester").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(deleteSuggesterRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DeleteSuggesterRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Gets the analysis schemes configured for a domain. An analysis scheme defines language-specific text processing
     * options for a <code>text</code> field. Can be limited to specific analysis schemes by name. By default, shows all
     * analysis schemes and includes any pending changes to the configuration. Set the <code>Deployed</code> option to
     * <code>true</code> to show the active configuration and exclude pending changes. For more information, see <a
     * href="http://docs.aws.amazon.com/cloudsearch/latest/developerguide/configuring-analysis-schemes.html"
     * target="_blank">Configuring Analysis Schemes</a> in the <i>Amazon CloudSearch Developer Guide</i>.
     * </p>
     *
     * @param describeAnalysisSchemesRequest
     *        Container for the parameters to the <code><a>DescribeAnalysisSchemes</a></code> operation. Specifies the
     *        name of the domain you want to describe. To limit the response to particular analysis schemes, specify the
     *        names of the analysis schemes you want to describe. To show the active configuration and exclude any
     *        pending changes, set the <code>Deployed</code> option to <code>true</code>.
     * @return Result of the DescribeAnalysisSchemes operation returned by the service.
     * @throws BaseException
     *         An error occurred while processing the request.
     * @throws InternalException
     *         An internal error occurred while processing the request. If this problem persists, report an issue from
     *         the <a href="http://status.aws.amazon.com/" target="_blank">Service Health Dashboard</a>.
     * @throws ResourceNotFoundException
     *         The request was rejected because it attempted to reference a resource that does not exist.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CloudSearchException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CloudSearchClient.DescribeAnalysisSchemes
     */
    @Override
    public DescribeAnalysisSchemesResponse describeAnalysisSchemes(DescribeAnalysisSchemesRequest describeAnalysisSchemesRequest)
            throws BaseException, InternalException, ResourceNotFoundException, AwsServiceException, SdkClientException,
            CloudSearchException {

        HttpResponseHandler<DescribeAnalysisSchemesResponse> responseHandler = protocolFactory
                .createResponseHandler(DescribeAnalysisSchemesResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeAnalysisSchemesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CloudSearch");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeAnalysisSchemes");

            return clientHandler
                    .execute(new ClientExecutionParams<DescribeAnalysisSchemesRequest, DescribeAnalysisSchemesResponse>()
                            .withOperationName("DescribeAnalysisSchemes").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(describeAnalysisSchemesRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new DescribeAnalysisSchemesRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Gets the availability options configured for a domain. By default, shows the configuration with any pending
     * changes. Set the <code>Deployed</code> option to <code>true</code> to show the active configuration and exclude
     * pending changes. For more information, see <a
     * href="http://docs.aws.amazon.com/cloudsearch/latest/developerguide/configuring-availability-options.html"
     * target="_blank">Configuring Availability Options</a> in the <i>Amazon CloudSearch Developer Guide</i>.
     * </p>
     *
     * @param describeAvailabilityOptionsRequest
     *        Container for the parameters to the <code><a>DescribeAvailabilityOptions</a></code> operation. Specifies
     *        the name of the domain you want to describe. To show the active configuration and exclude any pending
     *        changes, set the Deployed option to <code>true</code>.
     * @return Result of the DescribeAvailabilityOptions operation returned by the service.
     * @throws BaseException
     *         An error occurred while processing the request.
     * @throws InternalException
     *         An internal error occurred while processing the request. If this problem persists, report an issue from
     *         the <a href="http://status.aws.amazon.com/" target="_blank">Service Health Dashboard</a>.
     * @throws InvalidTypeException
     *         The request was rejected because it specified an invalid type definition.
     * @throws LimitExceededException
     *         The request was rejected because a resource limit has already been met.
     * @throws ResourceNotFoundException
     *         The request was rejected because it attempted to reference a resource that does not exist.
     * @throws DisabledOperationException
     *         The request was rejected because it attempted an operation which is not enabled.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CloudSearchException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CloudSearchClient.DescribeAvailabilityOptions
     */
    @Override
    public DescribeAvailabilityOptionsResponse describeAvailabilityOptions(
            DescribeAvailabilityOptionsRequest describeAvailabilityOptionsRequest) throws BaseException, InternalException,
            InvalidTypeException, LimitExceededException, ResourceNotFoundException, DisabledOperationException,
            AwsServiceException, SdkClientException, CloudSearchException {

        HttpResponseHandler<DescribeAvailabilityOptionsResponse> responseHandler = protocolFactory
                .createResponseHandler(DescribeAvailabilityOptionsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeAvailabilityOptionsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CloudSearch");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeAvailabilityOptions");

            return clientHandler
                    .execute(new ClientExecutionParams<DescribeAvailabilityOptionsRequest, DescribeAvailabilityOptionsResponse>()
                            .withOperationName("DescribeAvailabilityOptions").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(describeAvailabilityOptionsRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new DescribeAvailabilityOptionsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns the domain's endpoint options, specifically whether all requests to the domain must arrive over HTTPS.
     * For more information, see <a
     * href="http://docs.aws.amazon.com/cloudsearch/latest/developerguide/configuring-domain-endpoint-options.html"
     * target="_blank">Configuring Domain Endpoint Options</a> in the <i>Amazon CloudSearch Developer Guide</i>.
     * </p>
     *
     * @param describeDomainEndpointOptionsRequest
     *        Container for the parameters to the <code><a>DescribeDomainEndpointOptions</a></code> operation. Specify
     *        the name of the domain you want to describe. To show the active configuration and exclude any pending
     *        changes, set the Deployed option to <code>true</code>.
     * @return Result of the DescribeDomainEndpointOptions operation returned by the service.
     * @throws BaseException
     *         An error occurred while processing the request.
     * @throws InternalException
     *         An internal error occurred while processing the request. If this problem persists, report an issue from
     *         the <a href="http://status.aws.amazon.com/" target="_blank">Service Health Dashboard</a>.
     * @throws LimitExceededException
     *         The request was rejected because a resource limit has already been met.
     * @throws ResourceNotFoundException
     *         The request was rejected because it attempted to reference a resource that does not exist.
     * @throws DisabledOperationException
     *         The request was rejected because it attempted an operation which is not enabled.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CloudSearchException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CloudSearchClient.DescribeDomainEndpointOptions
     */
    @Override
    public DescribeDomainEndpointOptionsResponse describeDomainEndpointOptions(
            DescribeDomainEndpointOptionsRequest describeDomainEndpointOptionsRequest) throws BaseException, InternalException,
            LimitExceededException, ResourceNotFoundException, DisabledOperationException, AwsServiceException,
            SdkClientException, CloudSearchException {

        HttpResponseHandler<DescribeDomainEndpointOptionsResponse> responseHandler = protocolFactory
                .createResponseHandler(DescribeDomainEndpointOptionsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                describeDomainEndpointOptionsRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CloudSearch");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeDomainEndpointOptions");

            return clientHandler
                    .execute(new ClientExecutionParams<DescribeDomainEndpointOptionsRequest, DescribeDomainEndpointOptionsResponse>()
                            .withOperationName("DescribeDomainEndpointOptions").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(describeDomainEndpointOptionsRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new DescribeDomainEndpointOptionsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Gets information about the search domains owned by this account. Can be limited to specific domains. Shows all
     * domains by default. To get the number of searchable documents in a domain, use the console or submit a
     * <code>matchall</code> request to your domain's search endpoint:
     * <code>q=matchall&amp;amp;q.parser=structured&amp;amp;size=0</code>. For more information, see <a
     * href="http://docs.aws.amazon.com/cloudsearch/latest/developerguide/getting-domain-info.html"
     * target="_blank">Getting Information about a Search Domain</a> in the <i>Amazon CloudSearch Developer Guide</i>.
     * </p>
     *
     * @param describeDomainsRequest
     *        Container for the parameters to the <code><a>DescribeDomains</a></code> operation. By default shows the
     *        status of all domains. To restrict the response to particular domains, specify the names of the domains
     *        you want to describe.
     * @return Result of the DescribeDomains operation returned by the service.
     * @throws BaseException
     *         An error occurred while processing the request.
     * @throws InternalException
     *         An internal error occurred while processing the request. If this problem persists, report an issue from
     *         the <a href="http://status.aws.amazon.com/" target="_blank">Service Health Dashboard</a>.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CloudSearchException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CloudSearchClient.DescribeDomains
     */
    @Override
    public DescribeDomainsResponse describeDomains(DescribeDomainsRequest describeDomainsRequest) throws BaseException,
            InternalException, AwsServiceException, SdkClientException, CloudSearchException {

        HttpResponseHandler<DescribeDomainsResponse> responseHandler = protocolFactory
                .createResponseHandler(DescribeDomainsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeDomainsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CloudSearch");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeDomains");

            return clientHandler.execute(new ClientExecutionParams<DescribeDomainsRequest, DescribeDomainsResponse>()
                    .withOperationName("DescribeDomains").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(describeDomainsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DescribeDomainsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Gets the expressions configured for the search domain. Can be limited to specific expressions by name. By
     * default, shows all expressions and includes any pending changes to the configuration. Set the
     * <code>Deployed</code> option to <code>true</code> to show the active configuration and exclude pending changes.
     * For more information, see <a
     * href="http://docs.aws.amazon.com/cloudsearch/latest/developerguide/configuring-expressions.html"
     * target="_blank">Configuring Expressions</a> in the <i>Amazon CloudSearch Developer Guide</i>.
     * </p>
     *
     * @param describeExpressionsRequest
     *        Container for the parameters to the <code><a>DescribeDomains</a></code> operation. Specifies the name of
     *        the domain you want to describe. To restrict the response to particular expressions, specify the names of
     *        the expressions you want to describe. To show the active configuration and exclude any pending changes,
     *        set the <code>Deployed</code> option to <code>true</code>.
     * @return Result of the DescribeExpressions operation returned by the service.
     * @throws BaseException
     *         An error occurred while processing the request.
     * @throws InternalException
     *         An internal error occurred while processing the request. If this problem persists, report an issue from
     *         the <a href="http://status.aws.amazon.com/" target="_blank">Service Health Dashboard</a>.
     * @throws ResourceNotFoundException
     *         The request was rejected because it attempted to reference a resource that does not exist.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CloudSearchException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CloudSearchClient.DescribeExpressions
     */
    @Override
    public DescribeExpressionsResponse describeExpressions(DescribeExpressionsRequest describeExpressionsRequest)
            throws BaseException, InternalException, ResourceNotFoundException, AwsServiceException, SdkClientException,
            CloudSearchException {

        HttpResponseHandler<DescribeExpressionsResponse> responseHandler = protocolFactory
                .createResponseHandler(DescribeExpressionsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeExpressionsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CloudSearch");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeExpressions");

            return clientHandler.execute(new ClientExecutionParams<DescribeExpressionsRequest, DescribeExpressionsResponse>()
                    .withOperationName("DescribeExpressions").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(describeExpressionsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DescribeExpressionsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Gets information about the index fields configured for the search domain. Can be limited to specific fields by
     * name. By default, shows all fields and includes any pending changes to the configuration. Set the
     * <code>Deployed</code> option to <code>true</code> to show the active configuration and exclude pending changes.
     * For more information, see <a
     * href="http://docs.aws.amazon.com/cloudsearch/latest/developerguide/getting-domain-info.html"
     * target="_blank">Getting Domain Information</a> in the <i>Amazon CloudSearch Developer Guide</i>.
     * </p>
     *
     * @param describeIndexFieldsRequest
     *        Container for the parameters to the <code><a>DescribeIndexFields</a></code> operation. Specifies the name
     *        of the domain you want to describe. To restrict the response to particular index fields, specify the names
     *        of the index fields you want to describe. To show the active configuration and exclude any pending
     *        changes, set the <code>Deployed</code> option to <code>true</code>.
     * @return Result of the DescribeIndexFields operation returned by the service.
     * @throws BaseException
     *         An error occurred while processing the request.
     * @throws InternalException
     *         An internal error occurred while processing the request. If this problem persists, report an issue from
     *         the <a href="http://status.aws.amazon.com/" target="_blank">Service Health Dashboard</a>.
     * @throws ResourceNotFoundException
     *         The request was rejected because it attempted to reference a resource that does not exist.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CloudSearchException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CloudSearchClient.DescribeIndexFields
     */
    @Override
    public DescribeIndexFieldsResponse describeIndexFields(DescribeIndexFieldsRequest describeIndexFieldsRequest)
            throws BaseException, InternalException, ResourceNotFoundException, AwsServiceException, SdkClientException,
            CloudSearchException {

        HttpResponseHandler<DescribeIndexFieldsResponse> responseHandler = protocolFactory
                .createResponseHandler(DescribeIndexFieldsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeIndexFieldsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CloudSearch");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeIndexFields");

            return clientHandler.execute(new ClientExecutionParams<DescribeIndexFieldsRequest, DescribeIndexFieldsResponse>()
                    .withOperationName("DescribeIndexFields").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(describeIndexFieldsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DescribeIndexFieldsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Gets the scaling parameters configured for a domain. A domain's scaling parameters specify the desired search
     * instance type and replication count. For more information, see <a
     * href="http://docs.aws.amazon.com/cloudsearch/latest/developerguide/configuring-scaling-options.html"
     * target="_blank">Configuring Scaling Options</a> in the <i>Amazon CloudSearch Developer Guide</i>.
     * </p>
     *
     * @param describeScalingParametersRequest
     *        Container for the parameters to the <code><a>DescribeScalingParameters</a></code> operation. Specifies the
     *        name of the domain you want to describe.
     * @return Result of the DescribeScalingParameters operation returned by the service.
     * @throws BaseException
     *         An error occurred while processing the request.
     * @throws InternalException
     *         An internal error occurred while processing the request. If this problem persists, report an issue from
     *         the <a href="http://status.aws.amazon.com/" target="_blank">Service Health Dashboard</a>.
     * @throws ResourceNotFoundException
     *         The request was rejected because it attempted to reference a resource that does not exist.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CloudSearchException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CloudSearchClient.DescribeScalingParameters
     */
    @Override
    public DescribeScalingParametersResponse describeScalingParameters(
            DescribeScalingParametersRequest describeScalingParametersRequest) throws BaseException, InternalException,
            ResourceNotFoundException, AwsServiceException, SdkClientException, CloudSearchException {

        HttpResponseHandler<DescribeScalingParametersResponse> responseHandler = protocolFactory
                .createResponseHandler(DescribeScalingParametersResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeScalingParametersRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CloudSearch");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeScalingParameters");

            return clientHandler
                    .execute(new ClientExecutionParams<DescribeScalingParametersRequest, DescribeScalingParametersResponse>()
                            .withOperationName("DescribeScalingParameters").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(describeScalingParametersRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new DescribeScalingParametersRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Gets information about the access policies that control access to the domain's document and search endpoints. By
     * default, shows the configuration with any pending changes. Set the <code>Deployed</code> option to
     * <code>true</code> to show the active configuration and exclude pending changes. For more information, see <a
     * href="http://docs.aws.amazon.com/cloudsearch/latest/developerguide/configuring-access.html"
     * target="_blank">Configuring Access for a Search Domain</a> in the <i>Amazon CloudSearch Developer Guide</i>.
     * </p>
     *
     * @param describeServiceAccessPoliciesRequest
     *        Container for the parameters to the <code><a>DescribeServiceAccessPolicies</a></code> operation. Specifies
     *        the name of the domain you want to describe. To show the active configuration and exclude any pending
     *        changes, set the <code>Deployed</code> option to <code>true</code>.
     * @return Result of the DescribeServiceAccessPolicies operation returned by the service.
     * @throws BaseException
     *         An error occurred while processing the request.
     * @throws InternalException
     *         An internal error occurred while processing the request. If this problem persists, report an issue from
     *         the <a href="http://status.aws.amazon.com/" target="_blank">Service Health Dashboard</a>.
     * @throws ResourceNotFoundException
     *         The request was rejected because it attempted to reference a resource that does not exist.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CloudSearchException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CloudSearchClient.DescribeServiceAccessPolicies
     */
    @Override
    public DescribeServiceAccessPoliciesResponse describeServiceAccessPolicies(
            DescribeServiceAccessPoliciesRequest describeServiceAccessPoliciesRequest) throws BaseException, InternalException,
            ResourceNotFoundException, AwsServiceException, SdkClientException, CloudSearchException {

        HttpResponseHandler<DescribeServiceAccessPoliciesResponse> responseHandler = protocolFactory
                .createResponseHandler(DescribeServiceAccessPoliciesResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                describeServiceAccessPoliciesRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CloudSearch");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeServiceAccessPolicies");

            return clientHandler
                    .execute(new ClientExecutionParams<DescribeServiceAccessPoliciesRequest, DescribeServiceAccessPoliciesResponse>()
                            .withOperationName("DescribeServiceAccessPolicies").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(describeServiceAccessPoliciesRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new DescribeServiceAccessPoliciesRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Gets the suggesters configured for a domain. A suggester enables you to display possible matches before users
     * finish typing their queries. Can be limited to specific suggesters by name. By default, shows all suggesters and
     * includes any pending changes to the configuration. Set the <code>Deployed</code> option to <code>true</code> to
     * show the active configuration and exclude pending changes. For more information, see <a
     * href="http://docs.aws.amazon.com/cloudsearch/latest/developerguide/getting-suggestions.html"
     * target="_blank">Getting Search Suggestions</a> in the <i>Amazon CloudSearch Developer Guide</i>.
     * </p>
     *
     * @param describeSuggestersRequest
     *        Container for the parameters to the <code><a>DescribeSuggester</a></code> operation. Specifies the name of
     *        the domain you want to describe. To restrict the response to particular suggesters, specify the names of
     *        the suggesters you want to describe. To show the active configuration and exclude any pending changes, set
     *        the <code>Deployed</code> option to <code>true</code>.
     * @return Result of the DescribeSuggesters operation returned by the service.
     * @throws BaseException
     *         An error occurred while processing the request.
     * @throws InternalException
     *         An internal error occurred while processing the request. If this problem persists, report an issue from
     *         the <a href="http://status.aws.amazon.com/" target="_blank">Service Health Dashboard</a>.
     * @throws ResourceNotFoundException
     *         The request was rejected because it attempted to reference a resource that does not exist.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CloudSearchException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CloudSearchClient.DescribeSuggesters
     */
    @Override
    public DescribeSuggestersResponse describeSuggesters(DescribeSuggestersRequest describeSuggestersRequest)
            throws BaseException, InternalException, ResourceNotFoundException, AwsServiceException, SdkClientException,
            CloudSearchException {

        HttpResponseHandler<DescribeSuggestersResponse> responseHandler = protocolFactory
                .createResponseHandler(DescribeSuggestersResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeSuggestersRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CloudSearch");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeSuggesters");

            return clientHandler.execute(new ClientExecutionParams<DescribeSuggestersRequest, DescribeSuggestersResponse>()
                    .withOperationName("DescribeSuggesters").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(describeSuggestersRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DescribeSuggestersRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Tells the search domain to start indexing its documents using the latest indexing options. This operation must be
     * invoked to activate options whose <a>OptionStatus</a> is <code>RequiresIndexDocuments</code>.
     * </p>
     *
     * @param indexDocumentsRequest
     *        Container for the parameters to the <code><a>IndexDocuments</a></code> operation. Specifies the name of
     *        the domain you want to re-index.
     * @return Result of the IndexDocuments operation returned by the service.
     * @throws BaseException
     *         An error occurred while processing the request.
     * @throws InternalException
     *         An internal error occurred while processing the request. If this problem persists, report an issue from
     *         the <a href="http://status.aws.amazon.com/" target="_blank">Service Health Dashboard</a>.
     * @throws ResourceNotFoundException
     *         The request was rejected because it attempted to reference a resource that does not exist.
     * @throws ValidationException
     *         The request was rejected because it has invalid parameters.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CloudSearchException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CloudSearchClient.IndexDocuments
     */
    @Override
    public IndexDocumentsResponse indexDocuments(IndexDocumentsRequest indexDocumentsRequest) throws BaseException,
            InternalException, ResourceNotFoundException, ValidationException, AwsServiceException, SdkClientException,
            CloudSearchException {

        HttpResponseHandler<IndexDocumentsResponse> responseHandler = protocolFactory
                .createResponseHandler(IndexDocumentsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, indexDocumentsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CloudSearch");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "IndexDocuments");

            return clientHandler.execute(new ClientExecutionParams<IndexDocumentsRequest, IndexDocumentsResponse>()
                    .withOperationName("IndexDocuments").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(indexDocumentsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new IndexDocumentsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Lists all search domains owned by an account.
     * </p>
     *
     * @param listDomainNamesRequest
     * @return Result of the ListDomainNames operation returned by the service.
     * @throws BaseException
     *         An error occurred while processing the request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CloudSearchException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CloudSearchClient.ListDomainNames
     */
    @Override
    public ListDomainNamesResponse listDomainNames(ListDomainNamesRequest listDomainNamesRequest) throws BaseException,
            AwsServiceException, SdkClientException, CloudSearchException {

        HttpResponseHandler<ListDomainNamesResponse> responseHandler = protocolFactory
                .createResponseHandler(ListDomainNamesResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listDomainNamesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CloudSearch");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListDomainNames");

            return clientHandler.execute(new ClientExecutionParams<ListDomainNamesRequest, ListDomainNamesResponse>()
                    .withOperationName("ListDomainNames").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(listDomainNamesRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListDomainNamesRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Configures the availability options for a domain. Enabling the Multi-AZ option expands an Amazon CloudSearch
     * domain to an additional Availability Zone in the same Region to increase fault tolerance in the event of a
     * service disruption. Changes to the Multi-AZ option can take about half an hour to become active. For more
     * information, see <a
     * href="http://docs.aws.amazon.com/cloudsearch/latest/developerguide/configuring-availability-options.html"
     * target="_blank">Configuring Availability Options</a> in the <i>Amazon CloudSearch Developer Guide</i>.
     * </p>
     *
     * @param updateAvailabilityOptionsRequest
     *        Container for the parameters to the <code><a>UpdateAvailabilityOptions</a></code> operation. Specifies the
     *        name of the domain you want to update and the Multi-AZ availability option.
     * @return Result of the UpdateAvailabilityOptions operation returned by the service.
     * @throws BaseException
     *         An error occurred while processing the request.
     * @throws InternalException
     *         An internal error occurred while processing the request. If this problem persists, report an issue from
     *         the <a href="http://status.aws.amazon.com/" target="_blank">Service Health Dashboard</a>.
     * @throws InvalidTypeException
     *         The request was rejected because it specified an invalid type definition.
     * @throws LimitExceededException
     *         The request was rejected because a resource limit has already been met.
     * @throws ResourceNotFoundException
     *         The request was rejected because it attempted to reference a resource that does not exist.
     * @throws DisabledOperationException
     *         The request was rejected because it attempted an operation which is not enabled.
     * @throws ValidationException
     *         The request was rejected because it has invalid parameters.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CloudSearchException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CloudSearchClient.UpdateAvailabilityOptions
     */
    @Override
    public UpdateAvailabilityOptionsResponse updateAvailabilityOptions(
            UpdateAvailabilityOptionsRequest updateAvailabilityOptionsRequest) throws BaseException, InternalException,
            InvalidTypeException, LimitExceededException, ResourceNotFoundException, DisabledOperationException,
            ValidationException, AwsServiceException, SdkClientException, CloudSearchException {

        HttpResponseHandler<UpdateAvailabilityOptionsResponse> responseHandler = protocolFactory
                .createResponseHandler(UpdateAvailabilityOptionsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateAvailabilityOptionsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CloudSearch");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateAvailabilityOptions");

            return clientHandler
                    .execute(new ClientExecutionParams<UpdateAvailabilityOptionsRequest, UpdateAvailabilityOptionsResponse>()
                            .withOperationName("UpdateAvailabilityOptions").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(updateAvailabilityOptionsRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new UpdateAvailabilityOptionsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Updates the domain's endpoint options, specifically whether all requests to the domain must arrive over HTTPS.
     * For more information, see <a
     * href="http://docs.aws.amazon.com/cloudsearch/latest/developerguide/configuring-domain-endpoint-options.html"
     * target="_blank">Configuring Domain Endpoint Options</a> in the <i>Amazon CloudSearch Developer Guide</i>.
     * </p>
     *
     * @param updateDomainEndpointOptionsRequest
     *        Container for the parameters to the <code><a>UpdateDomainEndpointOptions</a></code> operation. Specifies
     *        the name of the domain you want to update and the domain endpoint options.
     * @return Result of the UpdateDomainEndpointOptions operation returned by the service.
     * @throws BaseException
     *         An error occurred while processing the request.
     * @throws InternalException
     *         An internal error occurred while processing the request. If this problem persists, report an issue from
     *         the <a href="http://status.aws.amazon.com/" target="_blank">Service Health Dashboard</a>.
     * @throws InvalidTypeException
     *         The request was rejected because it specified an invalid type definition.
     * @throws LimitExceededException
     *         The request was rejected because a resource limit has already been met.
     * @throws ResourceNotFoundException
     *         The request was rejected because it attempted to reference a resource that does not exist.
     * @throws DisabledOperationException
     *         The request was rejected because it attempted an operation which is not enabled.
     * @throws ValidationException
     *         The request was rejected because it has invalid parameters.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CloudSearchException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CloudSearchClient.UpdateDomainEndpointOptions
     */
    @Override
    public UpdateDomainEndpointOptionsResponse updateDomainEndpointOptions(
            UpdateDomainEndpointOptionsRequest updateDomainEndpointOptionsRequest) throws BaseException, InternalException,
            InvalidTypeException, LimitExceededException, ResourceNotFoundException, DisabledOperationException,
            ValidationException, AwsServiceException, SdkClientException, CloudSearchException {

        HttpResponseHandler<UpdateDomainEndpointOptionsResponse> responseHandler = protocolFactory
                .createResponseHandler(UpdateDomainEndpointOptionsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateDomainEndpointOptionsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CloudSearch");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateDomainEndpointOptions");

            return clientHandler
                    .execute(new ClientExecutionParams<UpdateDomainEndpointOptionsRequest, UpdateDomainEndpointOptionsResponse>()
                            .withOperationName("UpdateDomainEndpointOptions").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(updateDomainEndpointOptionsRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new UpdateDomainEndpointOptionsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Configures scaling parameters for a domain. A domain's scaling parameters specify the desired search instance
     * type and replication count. Amazon CloudSearch will still automatically scale your domain based on the volume of
     * data and traffic, but not below the desired instance type and replication count. If the Multi-AZ option is
     * enabled, these values control the resources used per Availability Zone. For more information, see <a
     * href="http://docs.aws.amazon.com/cloudsearch/latest/developerguide/configuring-scaling-options.html"
     * target="_blank">Configuring Scaling Options</a> in the <i>Amazon CloudSearch Developer Guide</i>.
     * </p>
     *
     * @param updateScalingParametersRequest
     *        Container for the parameters to the <code><a>UpdateScalingParameters</a></code> operation. Specifies the
     *        name of the domain you want to update and the scaling parameters you want to configure.
     * @return Result of the UpdateScalingParameters operation returned by the service.
     * @throws BaseException
     *         An error occurred while processing the request.
     * @throws InternalException
     *         An internal error occurred while processing the request. If this problem persists, report an issue from
     *         the <a href="http://status.aws.amazon.com/" target="_blank">Service Health Dashboard</a>.
     * @throws LimitExceededException
     *         The request was rejected because a resource limit has already been met.
     * @throws ResourceNotFoundException
     *         The request was rejected because it attempted to reference a resource that does not exist.
     * @throws InvalidTypeException
     *         The request was rejected because it specified an invalid type definition.
     * @throws ValidationException
     *         The request was rejected because it has invalid parameters.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CloudSearchException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CloudSearchClient.UpdateScalingParameters
     */
    @Override
    public UpdateScalingParametersResponse updateScalingParameters(UpdateScalingParametersRequest updateScalingParametersRequest)
            throws BaseException, InternalException, LimitExceededException, ResourceNotFoundException, InvalidTypeException,
            ValidationException, AwsServiceException, SdkClientException, CloudSearchException {

        HttpResponseHandler<UpdateScalingParametersResponse> responseHandler = protocolFactory
                .createResponseHandler(UpdateScalingParametersResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateScalingParametersRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CloudSearch");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateScalingParameters");

            return clientHandler
                    .execute(new ClientExecutionParams<UpdateScalingParametersRequest, UpdateScalingParametersResponse>()
                            .withOperationName("UpdateScalingParameters").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(updateScalingParametersRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new UpdateScalingParametersRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Configures the access rules that control access to the domain's document and search endpoints. For more
     * information, see <a href="http://docs.aws.amazon.com/cloudsearch/latest/developerguide/configuring-access.html"
     * target="_blank"> Configuring Access for an Amazon CloudSearch Domain</a>.
     * </p>
     *
     * @param updateServiceAccessPoliciesRequest
     *        Container for the parameters to the <code><a>UpdateServiceAccessPolicies</a></code> operation. Specifies
     *        the name of the domain you want to update and the access rules you want to configure.
     * @return Result of the UpdateServiceAccessPolicies operation returned by the service.
     * @throws BaseException
     *         An error occurred while processing the request.
     * @throws InternalException
     *         An internal error occurred while processing the request. If this problem persists, report an issue from
     *         the <a href="http://status.aws.amazon.com/" target="_blank">Service Health Dashboard</a>.
     * @throws LimitExceededException
     *         The request was rejected because a resource limit has already been met.
     * @throws ResourceNotFoundException
     *         The request was rejected because it attempted to reference a resource that does not exist.
     * @throws InvalidTypeException
     *         The request was rejected because it specified an invalid type definition.
     * @throws ValidationException
     *         The request was rejected because it has invalid parameters.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CloudSearchException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CloudSearchClient.UpdateServiceAccessPolicies
     */
    @Override
    public UpdateServiceAccessPoliciesResponse updateServiceAccessPolicies(
            UpdateServiceAccessPoliciesRequest updateServiceAccessPoliciesRequest) throws BaseException, InternalException,
            LimitExceededException, ResourceNotFoundException, InvalidTypeException, ValidationException, AwsServiceException,
            SdkClientException, CloudSearchException {

        HttpResponseHandler<UpdateServiceAccessPoliciesResponse> responseHandler = protocolFactory
                .createResponseHandler(UpdateServiceAccessPoliciesResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateServiceAccessPoliciesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CloudSearch");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateServiceAccessPolicies");

            return clientHandler
                    .execute(new ClientExecutionParams<UpdateServiceAccessPoliciesRequest, UpdateServiceAccessPoliciesResponse>()
                            .withOperationName("UpdateServiceAccessPolicies").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(updateServiceAccessPoliciesRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new UpdateServiceAccessPoliciesRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private AwsQueryProtocolFactory init() {
        return AwsQueryProtocolFactory
                .builder()
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("DisabledAction")
                                .exceptionBuilderSupplier(DisabledOperationException::builder).httpStatusCode(409).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("LimitExceeded")
                                .exceptionBuilderSupplier(LimitExceededException::builder).httpStatusCode(409).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidType")
                                .exceptionBuilderSupplier(InvalidTypeException::builder).httpStatusCode(409).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceNotFound")
                                .exceptionBuilderSupplier(ResourceNotFoundException::builder).httpStatusCode(409).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ValidationException")
                                .exceptionBuilderSupplier(ValidationException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("BaseException").exceptionBuilderSupplier(BaseException::builder)
                                .build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceAlreadyExists")
                                .exceptionBuilderSupplier(ResourceAlreadyExistsException::builder).httpStatusCode(409).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalException")
                                .exceptionBuilderSupplier(InternalException::builder).httpStatusCode(500).build())
                .clientConfiguration(clientConfiguration).defaultServiceExceptionSupplier(CloudSearchException::builder).build();
    }

    @Override
    public final CloudSearchServiceClientConfiguration serviceClientConfiguration() {
        return this.serviceClientConfiguration;
    }

    @Override
    public void close() {
        clientHandler.close();
    }
}
