/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.cloudfront.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * A configuration for a set of security-related HTTP response headers. CloudFront adds these headers to HTTP responses
 * that it sends for requests that match a cache behavior associated with this response headers policy.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class ResponseHeadersPolicySecurityHeadersConfig implements SdkPojo, Serializable,
        ToCopyableBuilder<ResponseHeadersPolicySecurityHeadersConfig.Builder, ResponseHeadersPolicySecurityHeadersConfig> {
    private static final SdkField<ResponseHeadersPolicyXSSProtection> XSS_PROTECTION_FIELD = SdkField
            .<ResponseHeadersPolicyXSSProtection> builder(MarshallingType.SDK_POJO)
            .memberName("XSSProtection")
            .getter(getter(ResponseHeadersPolicySecurityHeadersConfig::xssProtection))
            .setter(setter(Builder::xssProtection))
            .constructor(ResponseHeadersPolicyXSSProtection::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("XSSProtection")
                    .unmarshallLocationName("XSSProtection").build()).build();

    private static final SdkField<ResponseHeadersPolicyFrameOptions> FRAME_OPTIONS_FIELD = SdkField
            .<ResponseHeadersPolicyFrameOptions> builder(MarshallingType.SDK_POJO)
            .memberName("FrameOptions")
            .getter(getter(ResponseHeadersPolicySecurityHeadersConfig::frameOptions))
            .setter(setter(Builder::frameOptions))
            .constructor(ResponseHeadersPolicyFrameOptions::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("FrameOptions")
                    .unmarshallLocationName("FrameOptions").build()).build();

    private static final SdkField<ResponseHeadersPolicyReferrerPolicy> REFERRER_POLICY_FIELD = SdkField
            .<ResponseHeadersPolicyReferrerPolicy> builder(MarshallingType.SDK_POJO)
            .memberName("ReferrerPolicy")
            .getter(getter(ResponseHeadersPolicySecurityHeadersConfig::referrerPolicy))
            .setter(setter(Builder::referrerPolicy))
            .constructor(ResponseHeadersPolicyReferrerPolicy::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ReferrerPolicy")
                    .unmarshallLocationName("ReferrerPolicy").build()).build();

    private static final SdkField<ResponseHeadersPolicyContentSecurityPolicy> CONTENT_SECURITY_POLICY_FIELD = SdkField
            .<ResponseHeadersPolicyContentSecurityPolicy> builder(MarshallingType.SDK_POJO)
            .memberName("ContentSecurityPolicy")
            .getter(getter(ResponseHeadersPolicySecurityHeadersConfig::contentSecurityPolicy))
            .setter(setter(Builder::contentSecurityPolicy))
            .constructor(ResponseHeadersPolicyContentSecurityPolicy::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ContentSecurityPolicy")
                    .unmarshallLocationName("ContentSecurityPolicy").build()).build();

    private static final SdkField<ResponseHeadersPolicyContentTypeOptions> CONTENT_TYPE_OPTIONS_FIELD = SdkField
            .<ResponseHeadersPolicyContentTypeOptions> builder(MarshallingType.SDK_POJO)
            .memberName("ContentTypeOptions")
            .getter(getter(ResponseHeadersPolicySecurityHeadersConfig::contentTypeOptions))
            .setter(setter(Builder::contentTypeOptions))
            .constructor(ResponseHeadersPolicyContentTypeOptions::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ContentTypeOptions")
                    .unmarshallLocationName("ContentTypeOptions").build()).build();

    private static final SdkField<ResponseHeadersPolicyStrictTransportSecurity> STRICT_TRANSPORT_SECURITY_FIELD = SdkField
            .<ResponseHeadersPolicyStrictTransportSecurity> builder(MarshallingType.SDK_POJO)
            .memberName("StrictTransportSecurity")
            .getter(getter(ResponseHeadersPolicySecurityHeadersConfig::strictTransportSecurity))
            .setter(setter(Builder::strictTransportSecurity))
            .constructor(ResponseHeadersPolicyStrictTransportSecurity::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("StrictTransportSecurity")
                    .unmarshallLocationName("StrictTransportSecurity").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(XSS_PROTECTION_FIELD,
            FRAME_OPTIONS_FIELD, REFERRER_POLICY_FIELD, CONTENT_SECURITY_POLICY_FIELD, CONTENT_TYPE_OPTIONS_FIELD,
            STRICT_TRANSPORT_SECURITY_FIELD));

    private static final long serialVersionUID = 1L;

    private final ResponseHeadersPolicyXSSProtection xssProtection;

    private final ResponseHeadersPolicyFrameOptions frameOptions;

    private final ResponseHeadersPolicyReferrerPolicy referrerPolicy;

    private final ResponseHeadersPolicyContentSecurityPolicy contentSecurityPolicy;

    private final ResponseHeadersPolicyContentTypeOptions contentTypeOptions;

    private final ResponseHeadersPolicyStrictTransportSecurity strictTransportSecurity;

    private ResponseHeadersPolicySecurityHeadersConfig(BuilderImpl builder) {
        this.xssProtection = builder.xssProtection;
        this.frameOptions = builder.frameOptions;
        this.referrerPolicy = builder.referrerPolicy;
        this.contentSecurityPolicy = builder.contentSecurityPolicy;
        this.contentTypeOptions = builder.contentTypeOptions;
        this.strictTransportSecurity = builder.strictTransportSecurity;
    }

    /**
     * <p>
     * Determines whether CloudFront includes the <code>X-XSS-Protection</code> HTTP response header and the header's
     * value.
     * </p>
     * <p>
     * For more information about the <code>X-XSS-Protection</code> HTTP response header, see <a
     * href="https://developer.mozilla.org/en-US/docs/Web/HTTP/Headers/X-XSS-Protection">X-XSS-Protection</a> in the MDN
     * Web Docs.
     * </p>
     * 
     * @return Determines whether CloudFront includes the <code>X-XSS-Protection</code> HTTP response header and the
     *         header's value.</p>
     *         <p>
     *         For more information about the <code>X-XSS-Protection</code> HTTP response header, see <a
     *         href="https://developer.mozilla.org/en-US/docs/Web/HTTP/Headers/X-XSS-Protection">X-XSS-Protection</a> in
     *         the MDN Web Docs.
     */
    public final ResponseHeadersPolicyXSSProtection xssProtection() {
        return xssProtection;
    }

    /**
     * <p>
     * Determines whether CloudFront includes the <code>X-Frame-Options</code> HTTP response header and the header's
     * value.
     * </p>
     * <p>
     * For more information about the <code>X-Frame-Options</code> HTTP response header, see <a
     * href="https://developer.mozilla.org/en-US/docs/Web/HTTP/Headers/X-Frame-Options">X-Frame-Options</a> in the MDN
     * Web Docs.
     * </p>
     * 
     * @return Determines whether CloudFront includes the <code>X-Frame-Options</code> HTTP response header and the
     *         header's value.</p>
     *         <p>
     *         For more information about the <code>X-Frame-Options</code> HTTP response header, see <a
     *         href="https://developer.mozilla.org/en-US/docs/Web/HTTP/Headers/X-Frame-Options">X-Frame-Options</a> in
     *         the MDN Web Docs.
     */
    public final ResponseHeadersPolicyFrameOptions frameOptions() {
        return frameOptions;
    }

    /**
     * <p>
     * Determines whether CloudFront includes the <code>Referrer-Policy</code> HTTP response header and the header's
     * value.
     * </p>
     * <p>
     * For more information about the <code>Referrer-Policy</code> HTTP response header, see <a
     * href="https://developer.mozilla.org/en-US/docs/Web/HTTP/Headers/Referrer-Policy">Referrer-Policy</a> in the MDN
     * Web Docs.
     * </p>
     * 
     * @return Determines whether CloudFront includes the <code>Referrer-Policy</code> HTTP response header and the
     *         header's value.</p>
     *         <p>
     *         For more information about the <code>Referrer-Policy</code> HTTP response header, see <a
     *         href="https://developer.mozilla.org/en-US/docs/Web/HTTP/Headers/Referrer-Policy">Referrer-Policy</a> in
     *         the MDN Web Docs.
     */
    public final ResponseHeadersPolicyReferrerPolicy referrerPolicy() {
        return referrerPolicy;
    }

    /**
     * <p>
     * The policy directives and their values that CloudFront includes as values for the
     * <code>Content-Security-Policy</code> HTTP response header.
     * </p>
     * <p>
     * For more information about the <code>Content-Security-Policy</code> HTTP response header, see <a
     * href="https://developer.mozilla.org/en-US/docs/Web/HTTP/Headers/Content-Security-Policy"
     * >Content-Security-Policy</a> in the MDN Web Docs.
     * </p>
     * 
     * @return The policy directives and their values that CloudFront includes as values for the
     *         <code>Content-Security-Policy</code> HTTP response header.</p>
     *         <p>
     *         For more information about the <code>Content-Security-Policy</code> HTTP response header, see <a
     *         href="https://developer.mozilla.org/en-US/docs/Web/HTTP/Headers/Content-Security-Policy"
     *         >Content-Security-Policy</a> in the MDN Web Docs.
     */
    public final ResponseHeadersPolicyContentSecurityPolicy contentSecurityPolicy() {
        return contentSecurityPolicy;
    }

    /**
     * <p>
     * Determines whether CloudFront includes the <code>X-Content-Type-Options</code> HTTP response header with its
     * value set to <code>nosniff</code>.
     * </p>
     * <p>
     * For more information about the <code>X-Content-Type-Options</code> HTTP response header, see <a
     * href="https://developer.mozilla.org/en-US/docs/Web/HTTP/Headers/X-Content-Type-Options"
     * >X-Content-Type-Options</a> in the MDN Web Docs.
     * </p>
     * 
     * @return Determines whether CloudFront includes the <code>X-Content-Type-Options</code> HTTP response header with
     *         its value set to <code>nosniff</code>.</p>
     *         <p>
     *         For more information about the <code>X-Content-Type-Options</code> HTTP response header, see <a
     *         href="https://developer.mozilla.org/en-US/docs/Web/HTTP/Headers/X-Content-Type-Options"
     *         >X-Content-Type-Options</a> in the MDN Web Docs.
     */
    public final ResponseHeadersPolicyContentTypeOptions contentTypeOptions() {
        return contentTypeOptions;
    }

    /**
     * <p>
     * Determines whether CloudFront includes the <code>Strict-Transport-Security</code> HTTP response header and the
     * header's value.
     * </p>
     * <p>
     * For more information about the <code>Strict-Transport-Security</code> HTTP response header, see <a
     * href="https://developer.mozilla.org/en-US/docs/Web/HTTP/Headers/Strict-Transport-Security"
     * >Strict-Transport-Security</a> in the MDN Web Docs.
     * </p>
     * 
     * @return Determines whether CloudFront includes the <code>Strict-Transport-Security</code> HTTP response header
     *         and the header's value.</p>
     *         <p>
     *         For more information about the <code>Strict-Transport-Security</code> HTTP response header, see <a
     *         href="https://developer.mozilla.org/en-US/docs/Web/HTTP/Headers/Strict-Transport-Security"
     *         >Strict-Transport-Security</a> in the MDN Web Docs.
     */
    public final ResponseHeadersPolicyStrictTransportSecurity strictTransportSecurity() {
        return strictTransportSecurity;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(xssProtection());
        hashCode = 31 * hashCode + Objects.hashCode(frameOptions());
        hashCode = 31 * hashCode + Objects.hashCode(referrerPolicy());
        hashCode = 31 * hashCode + Objects.hashCode(contentSecurityPolicy());
        hashCode = 31 * hashCode + Objects.hashCode(contentTypeOptions());
        hashCode = 31 * hashCode + Objects.hashCode(strictTransportSecurity());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof ResponseHeadersPolicySecurityHeadersConfig)) {
            return false;
        }
        ResponseHeadersPolicySecurityHeadersConfig other = (ResponseHeadersPolicySecurityHeadersConfig) obj;
        return Objects.equals(xssProtection(), other.xssProtection()) && Objects.equals(frameOptions(), other.frameOptions())
                && Objects.equals(referrerPolicy(), other.referrerPolicy())
                && Objects.equals(contentSecurityPolicy(), other.contentSecurityPolicy())
                && Objects.equals(contentTypeOptions(), other.contentTypeOptions())
                && Objects.equals(strictTransportSecurity(), other.strictTransportSecurity());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("ResponseHeadersPolicySecurityHeadersConfig").add("XSSProtection", xssProtection())
                .add("FrameOptions", frameOptions()).add("ReferrerPolicy", referrerPolicy())
                .add("ContentSecurityPolicy", contentSecurityPolicy()).add("ContentTypeOptions", contentTypeOptions())
                .add("StrictTransportSecurity", strictTransportSecurity()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "XSSProtection":
            return Optional.ofNullable(clazz.cast(xssProtection()));
        case "FrameOptions":
            return Optional.ofNullable(clazz.cast(frameOptions()));
        case "ReferrerPolicy":
            return Optional.ofNullable(clazz.cast(referrerPolicy()));
        case "ContentSecurityPolicy":
            return Optional.ofNullable(clazz.cast(contentSecurityPolicy()));
        case "ContentTypeOptions":
            return Optional.ofNullable(clazz.cast(contentTypeOptions()));
        case "StrictTransportSecurity":
            return Optional.ofNullable(clazz.cast(strictTransportSecurity()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<ResponseHeadersPolicySecurityHeadersConfig, T> g) {
        return obj -> g.apply((ResponseHeadersPolicySecurityHeadersConfig) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, ResponseHeadersPolicySecurityHeadersConfig> {
        /**
         * <p>
         * Determines whether CloudFront includes the <code>X-XSS-Protection</code> HTTP response header and the
         * header's value.
         * </p>
         * <p>
         * For more information about the <code>X-XSS-Protection</code> HTTP response header, see <a
         * href="https://developer.mozilla.org/en-US/docs/Web/HTTP/Headers/X-XSS-Protection">X-XSS-Protection</a> in the
         * MDN Web Docs.
         * </p>
         * 
         * @param xssProtection
         *        Determines whether CloudFront includes the <code>X-XSS-Protection</code> HTTP response header and the
         *        header's value.</p>
         *        <p>
         *        For more information about the <code>X-XSS-Protection</code> HTTP response header, see <a
         *        href="https://developer.mozilla.org/en-US/docs/Web/HTTP/Headers/X-XSS-Protection">X-XSS-Protection</a>
         *        in the MDN Web Docs.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder xssProtection(ResponseHeadersPolicyXSSProtection xssProtection);

        /**
         * <p>
         * Determines whether CloudFront includes the <code>X-XSS-Protection</code> HTTP response header and the
         * header's value.
         * </p>
         * <p>
         * For more information about the <code>X-XSS-Protection</code> HTTP response header, see <a
         * href="https://developer.mozilla.org/en-US/docs/Web/HTTP/Headers/X-XSS-Protection">X-XSS-Protection</a> in the
         * MDN Web Docs.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link ResponseHeadersPolicyXSSProtection.Builder} avoiding the need to create one manually via
         * {@link ResponseHeadersPolicyXSSProtection#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link ResponseHeadersPolicyXSSProtection.Builder#build()} is called
         * immediately and its result is passed to {@link #xssProtection(ResponseHeadersPolicyXSSProtection)}.
         * 
         * @param xssProtection
         *        a consumer that will call methods on {@link ResponseHeadersPolicyXSSProtection.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #xssProtection(ResponseHeadersPolicyXSSProtection)
         */
        default Builder xssProtection(Consumer<ResponseHeadersPolicyXSSProtection.Builder> xssProtection) {
            return xssProtection(ResponseHeadersPolicyXSSProtection.builder().applyMutation(xssProtection).build());
        }

        /**
         * <p>
         * Determines whether CloudFront includes the <code>X-Frame-Options</code> HTTP response header and the header's
         * value.
         * </p>
         * <p>
         * For more information about the <code>X-Frame-Options</code> HTTP response header, see <a
         * href="https://developer.mozilla.org/en-US/docs/Web/HTTP/Headers/X-Frame-Options">X-Frame-Options</a> in the
         * MDN Web Docs.
         * </p>
         * 
         * @param frameOptions
         *        Determines whether CloudFront includes the <code>X-Frame-Options</code> HTTP response header and the
         *        header's value.</p>
         *        <p>
         *        For more information about the <code>X-Frame-Options</code> HTTP response header, see <a
         *        href="https://developer.mozilla.org/en-US/docs/Web/HTTP/Headers/X-Frame-Options">X-Frame-Options</a>
         *        in the MDN Web Docs.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder frameOptions(ResponseHeadersPolicyFrameOptions frameOptions);

        /**
         * <p>
         * Determines whether CloudFront includes the <code>X-Frame-Options</code> HTTP response header and the header's
         * value.
         * </p>
         * <p>
         * For more information about the <code>X-Frame-Options</code> HTTP response header, see <a
         * href="https://developer.mozilla.org/en-US/docs/Web/HTTP/Headers/X-Frame-Options">X-Frame-Options</a> in the
         * MDN Web Docs.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link ResponseHeadersPolicyFrameOptions.Builder} avoiding the need to create one manually via
         * {@link ResponseHeadersPolicyFrameOptions#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link ResponseHeadersPolicyFrameOptions.Builder#build()} is called
         * immediately and its result is passed to {@link #frameOptions(ResponseHeadersPolicyFrameOptions)}.
         * 
         * @param frameOptions
         *        a consumer that will call methods on {@link ResponseHeadersPolicyFrameOptions.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #frameOptions(ResponseHeadersPolicyFrameOptions)
         */
        default Builder frameOptions(Consumer<ResponseHeadersPolicyFrameOptions.Builder> frameOptions) {
            return frameOptions(ResponseHeadersPolicyFrameOptions.builder().applyMutation(frameOptions).build());
        }

        /**
         * <p>
         * Determines whether CloudFront includes the <code>Referrer-Policy</code> HTTP response header and the header's
         * value.
         * </p>
         * <p>
         * For more information about the <code>Referrer-Policy</code> HTTP response header, see <a
         * href="https://developer.mozilla.org/en-US/docs/Web/HTTP/Headers/Referrer-Policy">Referrer-Policy</a> in the
         * MDN Web Docs.
         * </p>
         * 
         * @param referrerPolicy
         *        Determines whether CloudFront includes the <code>Referrer-Policy</code> HTTP response header and the
         *        header's value.</p>
         *        <p>
         *        For more information about the <code>Referrer-Policy</code> HTTP response header, see <a
         *        href="https://developer.mozilla.org/en-US/docs/Web/HTTP/Headers/Referrer-Policy">Referrer-Policy</a>
         *        in the MDN Web Docs.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder referrerPolicy(ResponseHeadersPolicyReferrerPolicy referrerPolicy);

        /**
         * <p>
         * Determines whether CloudFront includes the <code>Referrer-Policy</code> HTTP response header and the header's
         * value.
         * </p>
         * <p>
         * For more information about the <code>Referrer-Policy</code> HTTP response header, see <a
         * href="https://developer.mozilla.org/en-US/docs/Web/HTTP/Headers/Referrer-Policy">Referrer-Policy</a> in the
         * MDN Web Docs.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link ResponseHeadersPolicyReferrerPolicy.Builder} avoiding the need to create one manually via
         * {@link ResponseHeadersPolicyReferrerPolicy#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link ResponseHeadersPolicyReferrerPolicy.Builder#build()} is called
         * immediately and its result is passed to {@link #referrerPolicy(ResponseHeadersPolicyReferrerPolicy)}.
         * 
         * @param referrerPolicy
         *        a consumer that will call methods on {@link ResponseHeadersPolicyReferrerPolicy.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #referrerPolicy(ResponseHeadersPolicyReferrerPolicy)
         */
        default Builder referrerPolicy(Consumer<ResponseHeadersPolicyReferrerPolicy.Builder> referrerPolicy) {
            return referrerPolicy(ResponseHeadersPolicyReferrerPolicy.builder().applyMutation(referrerPolicy).build());
        }

        /**
         * <p>
         * The policy directives and their values that CloudFront includes as values for the
         * <code>Content-Security-Policy</code> HTTP response header.
         * </p>
         * <p>
         * For more information about the <code>Content-Security-Policy</code> HTTP response header, see <a
         * href="https://developer.mozilla.org/en-US/docs/Web/HTTP/Headers/Content-Security-Policy"
         * >Content-Security-Policy</a> in the MDN Web Docs.
         * </p>
         * 
         * @param contentSecurityPolicy
         *        The policy directives and their values that CloudFront includes as values for the
         *        <code>Content-Security-Policy</code> HTTP response header.</p>
         *        <p>
         *        For more information about the <code>Content-Security-Policy</code> HTTP response header, see <a
         *        href="https://developer.mozilla.org/en-US/docs/Web/HTTP/Headers/Content-Security-Policy"
         *        >Content-Security-Policy</a> in the MDN Web Docs.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder contentSecurityPolicy(ResponseHeadersPolicyContentSecurityPolicy contentSecurityPolicy);

        /**
         * <p>
         * The policy directives and their values that CloudFront includes as values for the
         * <code>Content-Security-Policy</code> HTTP response header.
         * </p>
         * <p>
         * For more information about the <code>Content-Security-Policy</code> HTTP response header, see <a
         * href="https://developer.mozilla.org/en-US/docs/Web/HTTP/Headers/Content-Security-Policy"
         * >Content-Security-Policy</a> in the MDN Web Docs.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link ResponseHeadersPolicyContentSecurityPolicy.Builder} avoiding the need to create one manually via
         * {@link ResponseHeadersPolicyContentSecurityPolicy#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link ResponseHeadersPolicyContentSecurityPolicy.Builder#build()} is
         * called immediately and its result is passed to
         * {@link #contentSecurityPolicy(ResponseHeadersPolicyContentSecurityPolicy)}.
         * 
         * @param contentSecurityPolicy
         *        a consumer that will call methods on {@link ResponseHeadersPolicyContentSecurityPolicy.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #contentSecurityPolicy(ResponseHeadersPolicyContentSecurityPolicy)
         */
        default Builder contentSecurityPolicy(Consumer<ResponseHeadersPolicyContentSecurityPolicy.Builder> contentSecurityPolicy) {
            return contentSecurityPolicy(ResponseHeadersPolicyContentSecurityPolicy.builder()
                    .applyMutation(contentSecurityPolicy).build());
        }

        /**
         * <p>
         * Determines whether CloudFront includes the <code>X-Content-Type-Options</code> HTTP response header with its
         * value set to <code>nosniff</code>.
         * </p>
         * <p>
         * For more information about the <code>X-Content-Type-Options</code> HTTP response header, see <a
         * href="https://developer.mozilla.org/en-US/docs/Web/HTTP/Headers/X-Content-Type-Options"
         * >X-Content-Type-Options</a> in the MDN Web Docs.
         * </p>
         * 
         * @param contentTypeOptions
         *        Determines whether CloudFront includes the <code>X-Content-Type-Options</code> HTTP response header
         *        with its value set to <code>nosniff</code>.</p>
         *        <p>
         *        For more information about the <code>X-Content-Type-Options</code> HTTP response header, see <a
         *        href="https://developer.mozilla.org/en-US/docs/Web/HTTP/Headers/X-Content-Type-Options"
         *        >X-Content-Type-Options</a> in the MDN Web Docs.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder contentTypeOptions(ResponseHeadersPolicyContentTypeOptions contentTypeOptions);

        /**
         * <p>
         * Determines whether CloudFront includes the <code>X-Content-Type-Options</code> HTTP response header with its
         * value set to <code>nosniff</code>.
         * </p>
         * <p>
         * For more information about the <code>X-Content-Type-Options</code> HTTP response header, see <a
         * href="https://developer.mozilla.org/en-US/docs/Web/HTTP/Headers/X-Content-Type-Options"
         * >X-Content-Type-Options</a> in the MDN Web Docs.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link ResponseHeadersPolicyContentTypeOptions.Builder} avoiding the need to create one manually via
         * {@link ResponseHeadersPolicyContentTypeOptions#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link ResponseHeadersPolicyContentTypeOptions.Builder#build()} is
         * called immediately and its result is passed to
         * {@link #contentTypeOptions(ResponseHeadersPolicyContentTypeOptions)}.
         * 
         * @param contentTypeOptions
         *        a consumer that will call methods on {@link ResponseHeadersPolicyContentTypeOptions.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #contentTypeOptions(ResponseHeadersPolicyContentTypeOptions)
         */
        default Builder contentTypeOptions(Consumer<ResponseHeadersPolicyContentTypeOptions.Builder> contentTypeOptions) {
            return contentTypeOptions(ResponseHeadersPolicyContentTypeOptions.builder().applyMutation(contentTypeOptions).build());
        }

        /**
         * <p>
         * Determines whether CloudFront includes the <code>Strict-Transport-Security</code> HTTP response header and
         * the header's value.
         * </p>
         * <p>
         * For more information about the <code>Strict-Transport-Security</code> HTTP response header, see <a
         * href="https://developer.mozilla.org/en-US/docs/Web/HTTP/Headers/Strict-Transport-Security"
         * >Strict-Transport-Security</a> in the MDN Web Docs.
         * </p>
         * 
         * @param strictTransportSecurity
         *        Determines whether CloudFront includes the <code>Strict-Transport-Security</code> HTTP response header
         *        and the header's value.</p>
         *        <p>
         *        For more information about the <code>Strict-Transport-Security</code> HTTP response header, see <a
         *        href="https://developer.mozilla.org/en-US/docs/Web/HTTP/Headers/Strict-Transport-Security">Strict-
         *        Transport-Security</a> in the MDN Web Docs.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder strictTransportSecurity(ResponseHeadersPolicyStrictTransportSecurity strictTransportSecurity);

        /**
         * <p>
         * Determines whether CloudFront includes the <code>Strict-Transport-Security</code> HTTP response header and
         * the header's value.
         * </p>
         * <p>
         * For more information about the <code>Strict-Transport-Security</code> HTTP response header, see <a
         * href="https://developer.mozilla.org/en-US/docs/Web/HTTP/Headers/Strict-Transport-Security"
         * >Strict-Transport-Security</a> in the MDN Web Docs.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link ResponseHeadersPolicyStrictTransportSecurity.Builder} avoiding the need to create one manually via
         * {@link ResponseHeadersPolicyStrictTransportSecurity#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link ResponseHeadersPolicyStrictTransportSecurity.Builder#build()} is
         * called immediately and its result is passed to
         * {@link #strictTransportSecurity(ResponseHeadersPolicyStrictTransportSecurity)}.
         * 
         * @param strictTransportSecurity
         *        a consumer that will call methods on {@link ResponseHeadersPolicyStrictTransportSecurity.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #strictTransportSecurity(ResponseHeadersPolicyStrictTransportSecurity)
         */
        default Builder strictTransportSecurity(
                Consumer<ResponseHeadersPolicyStrictTransportSecurity.Builder> strictTransportSecurity) {
            return strictTransportSecurity(ResponseHeadersPolicyStrictTransportSecurity.builder()
                    .applyMutation(strictTransportSecurity).build());
        }
    }

    static final class BuilderImpl implements Builder {
        private ResponseHeadersPolicyXSSProtection xssProtection;

        private ResponseHeadersPolicyFrameOptions frameOptions;

        private ResponseHeadersPolicyReferrerPolicy referrerPolicy;

        private ResponseHeadersPolicyContentSecurityPolicy contentSecurityPolicy;

        private ResponseHeadersPolicyContentTypeOptions contentTypeOptions;

        private ResponseHeadersPolicyStrictTransportSecurity strictTransportSecurity;

        private BuilderImpl() {
        }

        private BuilderImpl(ResponseHeadersPolicySecurityHeadersConfig model) {
            xssProtection(model.xssProtection);
            frameOptions(model.frameOptions);
            referrerPolicy(model.referrerPolicy);
            contentSecurityPolicy(model.contentSecurityPolicy);
            contentTypeOptions(model.contentTypeOptions);
            strictTransportSecurity(model.strictTransportSecurity);
        }

        public final ResponseHeadersPolicyXSSProtection.Builder getXssProtection() {
            return xssProtection != null ? xssProtection.toBuilder() : null;
        }

        public final void setXssProtection(ResponseHeadersPolicyXSSProtection.BuilderImpl xssProtection) {
            this.xssProtection = xssProtection != null ? xssProtection.build() : null;
        }

        @Override
        public final Builder xssProtection(ResponseHeadersPolicyXSSProtection xssProtection) {
            this.xssProtection = xssProtection;
            return this;
        }

        public final ResponseHeadersPolicyFrameOptions.Builder getFrameOptions() {
            return frameOptions != null ? frameOptions.toBuilder() : null;
        }

        public final void setFrameOptions(ResponseHeadersPolicyFrameOptions.BuilderImpl frameOptions) {
            this.frameOptions = frameOptions != null ? frameOptions.build() : null;
        }

        @Override
        public final Builder frameOptions(ResponseHeadersPolicyFrameOptions frameOptions) {
            this.frameOptions = frameOptions;
            return this;
        }

        public final ResponseHeadersPolicyReferrerPolicy.Builder getReferrerPolicy() {
            return referrerPolicy != null ? referrerPolicy.toBuilder() : null;
        }

        public final void setReferrerPolicy(ResponseHeadersPolicyReferrerPolicy.BuilderImpl referrerPolicy) {
            this.referrerPolicy = referrerPolicy != null ? referrerPolicy.build() : null;
        }

        @Override
        public final Builder referrerPolicy(ResponseHeadersPolicyReferrerPolicy referrerPolicy) {
            this.referrerPolicy = referrerPolicy;
            return this;
        }

        public final ResponseHeadersPolicyContentSecurityPolicy.Builder getContentSecurityPolicy() {
            return contentSecurityPolicy != null ? contentSecurityPolicy.toBuilder() : null;
        }

        public final void setContentSecurityPolicy(ResponseHeadersPolicyContentSecurityPolicy.BuilderImpl contentSecurityPolicy) {
            this.contentSecurityPolicy = contentSecurityPolicy != null ? contentSecurityPolicy.build() : null;
        }

        @Override
        public final Builder contentSecurityPolicy(ResponseHeadersPolicyContentSecurityPolicy contentSecurityPolicy) {
            this.contentSecurityPolicy = contentSecurityPolicy;
            return this;
        }

        public final ResponseHeadersPolicyContentTypeOptions.Builder getContentTypeOptions() {
            return contentTypeOptions != null ? contentTypeOptions.toBuilder() : null;
        }

        public final void setContentTypeOptions(ResponseHeadersPolicyContentTypeOptions.BuilderImpl contentTypeOptions) {
            this.contentTypeOptions = contentTypeOptions != null ? contentTypeOptions.build() : null;
        }

        @Override
        public final Builder contentTypeOptions(ResponseHeadersPolicyContentTypeOptions contentTypeOptions) {
            this.contentTypeOptions = contentTypeOptions;
            return this;
        }

        public final ResponseHeadersPolicyStrictTransportSecurity.Builder getStrictTransportSecurity() {
            return strictTransportSecurity != null ? strictTransportSecurity.toBuilder() : null;
        }

        public final void setStrictTransportSecurity(
                ResponseHeadersPolicyStrictTransportSecurity.BuilderImpl strictTransportSecurity) {
            this.strictTransportSecurity = strictTransportSecurity != null ? strictTransportSecurity.build() : null;
        }

        @Override
        public final Builder strictTransportSecurity(ResponseHeadersPolicyStrictTransportSecurity strictTransportSecurity) {
            this.strictTransportSecurity = strictTransportSecurity;
            return this;
        }

        @Override
        public ResponseHeadersPolicySecurityHeadersConfig build() {
            return new ResponseHeadersPolicySecurityHeadersConfig(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
