/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.arczonalshift.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * A complex structure for a managed resource in an account.
 * </p>
 * <p>
 * A managed resource is a Network Load Balancer or Application Load Balancer that has been registered with Route 53 ARC
 * by Elastic Load Balancing. You can start a zonal shift in Route 53 ARC for a managed resource to temporarily move
 * traffic for the resource away from an Availability Zone in an AWS Region.
 * </p>
 * <note>
 * <p>
 * At this time, you can only start a zonal shift for Network Load Balancers and Application Load Balancers with
 * cross-zone load balancing turned off.
 * </p>
 * </note>
 */
@Generated("software.amazon.awssdk:codegen")
public final class ManagedResourceSummary implements SdkPojo, Serializable,
        ToCopyableBuilder<ManagedResourceSummary.Builder, ManagedResourceSummary> {
    private static final SdkField<String> ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("arn")
            .getter(getter(ManagedResourceSummary::arn)).setter(setter(Builder::arn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("arn").build()).build();

    private static final SdkField<List<String>> AVAILABILITY_ZONES_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("availabilityZones")
            .getter(getter(ManagedResourceSummary::availabilityZones))
            .setter(setter(Builder::availabilityZones))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("availabilityZones").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("name")
            .getter(getter(ManagedResourceSummary::name)).setter(setter(Builder::name))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("name").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(ARN_FIELD,
            AVAILABILITY_ZONES_FIELD, NAME_FIELD));

    private static final long serialVersionUID = 1L;

    private final String arn;

    private final List<String> availabilityZones;

    private final String name;

    private ManagedResourceSummary(BuilderImpl builder) {
        this.arn = builder.arn;
        this.availabilityZones = builder.availabilityZones;
        this.name = builder.name;
    }

    /**
     * <p>
     * The Amazon Resource Name (ARN) for the managed resource.
     * </p>
     * 
     * @return The Amazon Resource Name (ARN) for the managed resource.
     */
    public final String arn() {
        return arn;
    }

    /**
     * For responses, this returns true if the service returned a value for the AvailabilityZones property. This DOES
     * NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasAvailabilityZones() {
        return availabilityZones != null && !(availabilityZones instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The Availability Zones that a resource is deployed in.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasAvailabilityZones} method.
     * </p>
     * 
     * @return The Availability Zones that a resource is deployed in.
     */
    public final List<String> availabilityZones() {
        return availabilityZones;
    }

    /**
     * <p>
     * The name of the managed resource.
     * </p>
     * 
     * @return The name of the managed resource.
     */
    public final String name() {
        return name;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(arn());
        hashCode = 31 * hashCode + Objects.hashCode(hasAvailabilityZones() ? availabilityZones() : null);
        hashCode = 31 * hashCode + Objects.hashCode(name());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof ManagedResourceSummary)) {
            return false;
        }
        ManagedResourceSummary other = (ManagedResourceSummary) obj;
        return Objects.equals(arn(), other.arn()) && hasAvailabilityZones() == other.hasAvailabilityZones()
                && Objects.equals(availabilityZones(), other.availabilityZones()) && Objects.equals(name(), other.name());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("ManagedResourceSummary").add("Arn", arn())
                .add("AvailabilityZones", hasAvailabilityZones() ? availabilityZones() : null).add("Name", name()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "arn":
            return Optional.ofNullable(clazz.cast(arn()));
        case "availabilityZones":
            return Optional.ofNullable(clazz.cast(availabilityZones()));
        case "name":
            return Optional.ofNullable(clazz.cast(name()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<ManagedResourceSummary, T> g) {
        return obj -> g.apply((ManagedResourceSummary) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, ManagedResourceSummary> {
        /**
         * <p>
         * The Amazon Resource Name (ARN) for the managed resource.
         * </p>
         * 
         * @param arn
         *        The Amazon Resource Name (ARN) for the managed resource.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder arn(String arn);

        /**
         * <p>
         * The Availability Zones that a resource is deployed in.
         * </p>
         * 
         * @param availabilityZones
         *        The Availability Zones that a resource is deployed in.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder availabilityZones(Collection<String> availabilityZones);

        /**
         * <p>
         * The Availability Zones that a resource is deployed in.
         * </p>
         * 
         * @param availabilityZones
         *        The Availability Zones that a resource is deployed in.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder availabilityZones(String... availabilityZones);

        /**
         * <p>
         * The name of the managed resource.
         * </p>
         * 
         * @param name
         *        The name of the managed resource.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder name(String name);
    }

    static final class BuilderImpl implements Builder {
        private String arn;

        private List<String> availabilityZones = DefaultSdkAutoConstructList.getInstance();

        private String name;

        private BuilderImpl() {
        }

        private BuilderImpl(ManagedResourceSummary model) {
            arn(model.arn);
            availabilityZones(model.availabilityZones);
            name(model.name);
        }

        public final String getArn() {
            return arn;
        }

        public final void setArn(String arn) {
            this.arn = arn;
        }

        @Override
        public final Builder arn(String arn) {
            this.arn = arn;
            return this;
        }

        public final Collection<String> getAvailabilityZones() {
            if (availabilityZones instanceof SdkAutoConstructList) {
                return null;
            }
            return availabilityZones;
        }

        public final void setAvailabilityZones(Collection<String> availabilityZones) {
            this.availabilityZones = AvailabilityZonesCopier.copy(availabilityZones);
        }

        @Override
        public final Builder availabilityZones(Collection<String> availabilityZones) {
            this.availabilityZones = AvailabilityZonesCopier.copy(availabilityZones);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder availabilityZones(String... availabilityZones) {
            availabilityZones(Arrays.asList(availabilityZones));
            return this;
        }

        public final String getName() {
            return name;
        }

        public final void setName(String name) {
            this.name = name;
        }

        @Override
        public final Builder name(String name) {
            this.name = name;
            return this;
        }

        @Override
        public ManagedResourceSummary build() {
            return new ManagedResourceSummary(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
