/*
 * Copyright 2015-present the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      https://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.springframework.session.data.gemfire.config.annotation.web.http.support;

import java.util.Optional;

import javax.servlet.http.HttpSession;

import org.apache.geode.cache.GemFireCache;
import org.apache.geode.cache.query.Index;

import org.springframework.beans.BeansException;
import org.springframework.beans.factory.BeanFactory;
import org.springframework.beans.factory.BeanFactoryAware;
import org.springframework.beans.factory.BeanNameAware;
import org.springframework.beans.factory.FactoryBean;
import org.springframework.beans.factory.InitializingBean;
import org.springframework.data.gemfire.IndexFactoryBean;
import org.springframework.data.gemfire.util.RegionUtils;
import org.springframework.util.ObjectUtils;

/**
 * The SessionAttributesIndexFactoryBean class is a Spring {@link FactoryBean} that creates a Pivotal GemFire {@link Index}
 * on the session attributes of the {@link HttpSession} object.
 *
 * @author John Blum
 * @since 1.3.0
 * @see org.springframework.beans.factory.BeanFactoryAware
 * @see org.springframework.beans.factory.BeanNameAware
 * @see org.springframework.beans.factory.FactoryBean
 * @see org.springframework.beans.factory.InitializingBean
 * @see org.apache.geode.cache.query.Index
 */
public class SessionAttributesIndexFactoryBean
		implements FactoryBean<Index>, InitializingBean, BeanFactoryAware, BeanNameAware {

	protected static final String[] DEFAULT_INDEXABLE_SESSION_ATTRIBUTES = {};

	private BeanFactory beanFactory;

	private GemFireCache gemfireCache;

	private Index sessionAttributesIndex;

	private String beanName;
	private String regionName;

	private String[] indexableSessionAttributes;

	/**
	 * @inheritDoc
	 */
	public void afterPropertiesSet() throws Exception {

		if (isIndexableSessionAttributesConfigured()) {
			this.sessionAttributesIndex = newIndex();
		}
	}

	/**
	 * Determines whether any indexable Session attributes were configured for this {@link FactoryBean}.
	 *
	 * @return a boolean value indicating whether any indexable Session attributes were configured
	 * for this {@link FactoryBean}
	 * @see #setIndexableSessionAttributes(String[])
	 */
	protected boolean isIndexableSessionAttributesConfigured() {
		return !ObjectUtils.isEmpty(this.indexableSessionAttributes);
	}

	/**
	 * Constructs a Pivotal GemFire {@link Index} over the attributes of the {@link HttpSession}.
	 *
	 * @return a Pivotal GemFire {@link Index} over the {@link HttpSession} attributes.
	 * @throws Exception if an error occurs while initializing the Pivotal GemFire {@link Index}.
	 * @see org.springframework.data.gemfire.IndexFactoryBean
	 */
	protected Index newIndex() throws Exception {

		IndexFactoryBean indexFactory = new IndexFactoryBean();

		indexFactory.setBeanFactory(this.beanFactory);
		indexFactory.setBeanName(this.beanName);
		indexFactory.setCache(this.gemfireCache);
		indexFactory.setName("sessionAttributesIndex");
		indexFactory.setExpression(String.format("s.attributes[%1$s]",
			getIndexableSessionAttributesAsGemFireIndexExpression()));
		indexFactory.setFrom(String.format("%1$s s", RegionUtils.toRegionPath(this.regionName)));
		indexFactory.setOverride(true);
		indexFactory.afterPropertiesSet();

		return indexFactory.getObject();
	}

	/**
	 * Gets the names of all Session attributes that will be indexed by Pivotal GemFire as single, comma-delimited
	 * String value constituting the Index expression of the Index definition.
	 *
	 * @return a String composed of all the named Session attributes for which GemFire
	 * will create an Index as an Index definition expression. If the indexable Session
	 * attributes were not configured, then the wildcard ("*") is returned.
	 * @see org.apache.geode.cache.query.Index#getIndexedExpression()
	 */
	protected String getIndexableSessionAttributesAsGemFireIndexExpression() {

		StringBuilder builder = new StringBuilder();

		for (String sessionAttribute : getIndexableSessionAttributes()) {
			builder.append(builder.length() > 0 ? ", " : "");
			builder.append(String.format("'%s'", sessionAttribute));
		}

		String indexExpression = builder.toString();

		return indexExpression.isEmpty() ? "*" : indexExpression;
	}

	public Index getObject() {
		return this.sessionAttributesIndex;
	}

	@SuppressWarnings("unchecked")
	public Class<?> getObjectType() {
		return Optional.ofNullable(this.sessionAttributesIndex).map(Object::getClass).orElse((Class) Index.class);
	}

	public boolean isSingleton() {
		return true;
	}

	public void setBeanFactory(BeanFactory beanFactory) throws BeansException {
		this.beanFactory = beanFactory;
	}

	public void setBeanName(String beanName) {
		this.beanName = beanName;
	}

	public void setGemFireCache(GemFireCache gemfireCache) {
		this.gemfireCache = gemfireCache;
	}

	public void setIndexableSessionAttributes(String[] indexableSessionAttributes) {
		this.indexableSessionAttributes = indexableSessionAttributes;
	}

	protected String[] getIndexableSessionAttributes() {
		return Optional.ofNullable(this.indexableSessionAttributes).orElse(DEFAULT_INDEXABLE_SESSION_ATTRIBUTES);
	}

	public void setRegionName(String regionName) {
		this.regionName = regionName;
	}

	protected String getRegionName() {
		return this.regionName;
	}
}
