/*
 * Decompiled with CFR 0.152.
 */
package org.projectnessie.client.auth.oauth2;

import com.fasterxml.jackson.databind.ObjectMapper;
import java.io.Closeable;
import java.io.IOException;
import java.io.InputStream;
import java.nio.charset.StandardCharsets;
import java.time.Duration;
import java.time.Instant;
import java.util.Arrays;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.CompletionStage;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.Executor;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.TimeUnit;
import org.projectnessie.client.auth.oauth2.AccessToken;
import org.projectnessie.client.auth.oauth2.ClientCredentialsTokensResponse;
import org.projectnessie.client.auth.oauth2.ErrorResponse;
import org.projectnessie.client.auth.oauth2.ImmutableClientCredentialsTokensRequest;
import org.projectnessie.client.auth.oauth2.ImmutablePasswordTokensRequest;
import org.projectnessie.client.auth.oauth2.ImmutableRefreshTokensRequest;
import org.projectnessie.client.auth.oauth2.ImmutableTokensExchangeRequest;
import org.projectnessie.client.auth.oauth2.JwtToken;
import org.projectnessie.client.auth.oauth2.OAuth2Authenticator;
import org.projectnessie.client.auth.oauth2.OAuth2ClientParams;
import org.projectnessie.client.auth.oauth2.OAuth2Exception;
import org.projectnessie.client.auth.oauth2.PasswordTokensResponse;
import org.projectnessie.client.auth.oauth2.RefreshTokensResponse;
import org.projectnessie.client.auth.oauth2.Token;
import org.projectnessie.client.auth.oauth2.TokenTypeIdentifiers;
import org.projectnessie.client.auth.oauth2.Tokens;
import org.projectnessie.client.auth.oauth2.TokensExchangeResponse;
import org.projectnessie.client.http.HttpClient;
import org.projectnessie.client.http.HttpClientException;
import org.projectnessie.client.http.HttpResponse;
import org.projectnessie.client.http.ResponseContext;
import org.projectnessie.client.http.Status;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

class OAuth2Client
implements OAuth2Authenticator,
Closeable {
    private static final Logger LOGGER = LoggerFactory.getLogger(OAuth2Client.class);
    private final String grantType;
    private final String username;
    private final byte[] password;
    private final String scope;
    private final Duration defaultAccessTokenLifespan;
    private final Duration defaultRefreshTokenLifespan;
    private final Duration refreshSafetyWindow;
    private final boolean tokenExchangeEnabled;
    private final HttpClient httpClient;
    private final ScheduledExecutorService executor;
    private final boolean shouldCloseExecutor;
    private final ObjectMapper objectMapper;
    private final CompletableFuture<Void> started = new CompletableFuture();
    private volatile CompletionStage<Tokens> currentTokensStage;
    private volatile ScheduledFuture<?> tokenRefreshFuture;

    OAuth2Client(OAuth2ClientParams params) {
        this.grantType = params.getGrantType();
        this.username = params.getUsername().orElse(null);
        this.password = params.getPassword().map(s -> s.getBytes(StandardCharsets.UTF_8)).orElse(null);
        this.scope = params.getScope().orElse(null);
        this.defaultAccessTokenLifespan = params.getDefaultAccessTokenLifespan();
        this.defaultRefreshTokenLifespan = params.getDefaultRefreshTokenLifespan();
        this.refreshSafetyWindow = params.getRefreshSafetyWindow();
        this.tokenExchangeEnabled = params.getTokenExchangeEnabled();
        this.httpClient = params.getHttpClient().addResponseFilter(this::checkErrorResponse).build();
        this.executor = params.getExecutor().orElseGet(OAuth2Client::createDefaultExecutor);
        this.shouldCloseExecutor = !params.getExecutor().isPresent();
        this.objectMapper = params.getObjectMapper();
        this.currentTokensStage = ((CompletableFuture)this.started.thenApplyAsync(v -> this.fetchNewTokens(), (Executor)this.executor)).whenComplete(this::log);
        this.currentTokensStage.thenAccept(this::scheduleTokensRenewal);
    }

    private static ScheduledExecutorService createDefaultExecutor() {
        return Executors.newSingleThreadScheduledExecutor(runnable -> {
            Thread thread = Executors.defaultThreadFactory().newThread(runnable);
            thread.setDaemon(true);
            thread.setName("oauth2-client");
            return thread;
        });
    }

    @Override
    public AccessToken authenticate() {
        return this.getCurrentTokens().getAccessToken();
    }

    Tokens getCurrentTokens() {
        try {
            return this.currentTokensStage.toCompletableFuture().get();
        }
        catch (InterruptedException e) {
            Thread.currentThread().interrupt();
            throw new RuntimeException(e);
        }
        catch (ExecutionException e) {
            Throwable cause = e.getCause();
            if (cause instanceof RuntimeException) {
                throw (RuntimeException)cause;
            }
            if (cause instanceof Error) {
                throw (Error)cause;
            }
            throw new RuntimeException(cause);
        }
    }

    public void start() {
        this.started.complete(null);
    }

    @Override
    public void close() {
        try {
            this.currentTokensStage.toCompletableFuture().cancel(true);
            ScheduledFuture<?> tokenRefreshFuture = this.tokenRefreshFuture;
            if (tokenRefreshFuture != null) {
                tokenRefreshFuture.cancel(true);
            }
            if (this.shouldCloseExecutor && !this.executor.isShutdown()) {
                this.executor.shutdown();
                if (!this.executor.awaitTermination(10L, TimeUnit.SECONDS)) {
                    this.executor.shutdownNow();
                }
            }
            if (this.password != null) {
                Arrays.fill(this.password, (byte)0);
            }
        }
        catch (InterruptedException e) {
            Thread.currentThread().interrupt();
        }
        finally {
            this.tokenRefreshFuture = null;
        }
    }

    private void scheduleTokensRenewal(Tokens currentTokens) {
        Instant now = Instant.now();
        Instant accessExpirationTime = OAuth2Client.tokenExpirationTime(now, currentTokens.getAccessToken(), this.defaultAccessTokenLifespan);
        Instant refreshExpirationTime = OAuth2Client.tokenExpirationTime(now, currentTokens.getRefreshToken(), this.defaultRefreshTokenLifespan);
        Duration delay = OAuth2Client.nextDelay(now, accessExpirationTime, refreshExpirationTime, this.refreshSafetyWindow);
        LOGGER.debug("Scheduling token refresh in {}", (Object)delay);
        this.tokenRefreshFuture = this.executor.schedule(() -> {
            this.currentTokensStage = this.renewTokens();
            this.currentTokensStage.thenAccept(this::scheduleTokensRenewal);
        }, delay.toMillis(), TimeUnit.MILLISECONDS);
    }

    private CompletionStage<Tokens> renewTokens() {
        return this.currentTokensStage.thenApply(this::refreshTokens).exceptionally(error -> this.fetchNewTokens()).whenComplete(this::log);
    }

    private void log(Tokens tokens, Throwable error) {
        if (error != null) {
            LOGGER.error("Failed to renew tokens", error);
        } else {
            LOGGER.debug("Successfully renewed tokens");
        }
    }

    Tokens fetchNewTokens() {
        LOGGER.debug("Fetching new tokens");
        if (this.grantType.equals("client_credentials")) {
            ImmutableClientCredentialsTokensRequest body = ImmutableClientCredentialsTokensRequest.builder().scope(this.scope).build();
            HttpResponse httpResponse = this.httpClient.newRequest().postForm(body);
            return httpResponse.readEntity(ClientCredentialsTokensResponse.class);
        }
        ImmutablePasswordTokensRequest body = ImmutablePasswordTokensRequest.builder().username(this.username).password(new String(this.password, StandardCharsets.UTF_8)).scope(this.scope).build();
        HttpResponse httpResponse = this.httpClient.newRequest().postForm(body);
        return httpResponse.readEntity(PasswordTokensResponse.class);
    }

    Tokens refreshTokens(Tokens currentTokens) {
        if (currentTokens.getRefreshToken() == null) {
            return this.exchangeTokens(currentTokens);
        }
        if (this.isAboutToExpire(currentTokens.getRefreshToken())) {
            throw new MustFetchNewTokensException("Refresh token is about to expire");
        }
        LOGGER.debug("Refreshing tokens");
        ImmutableRefreshTokensRequest body = ImmutableRefreshTokensRequest.builder().refreshToken(currentTokens.getRefreshToken().getPayload()).scope(this.scope).build();
        HttpResponse httpResponse = this.httpClient.newRequest().postForm(body);
        return httpResponse.readEntity(RefreshTokensResponse.class);
    }

    Tokens exchangeTokens(Tokens currentToken) {
        if (!this.tokenExchangeEnabled) {
            throw new MustFetchNewTokensException("Token exchange is disabled");
        }
        LOGGER.debug("Exchanging tokens");
        ImmutableTokensExchangeRequest body = ImmutableTokensExchangeRequest.builder().subjectToken(currentToken.getAccessToken().getPayload()).subjectTokenType(TokenTypeIdentifiers.ACCESS_TOKEN).requestedTokenType(TokenTypeIdentifiers.REFRESH_TOKEN).scope(this.scope).build();
        HttpResponse httpResponse = this.httpClient.newRequest().postForm(body);
        return httpResponse.readEntity(TokensExchangeResponse.class);
    }

    private boolean isAboutToExpire(Token token) {
        Instant now = Instant.now();
        return OAuth2Client.tokenExpirationTime(now, token, this.defaultRefreshTokenLifespan).isBefore(now.plus(this.refreshSafetyWindow));
    }

    static Duration nextDelay(Instant now, Instant accessExpirationTime, Instant refreshExpirationTime, Duration refreshSafetyWindow) {
        Instant expirationTime = accessExpirationTime.isBefore(refreshExpirationTime) ? accessExpirationTime : refreshExpirationTime;
        Duration delay = Duration.between(now, expirationTime).minus(refreshSafetyWindow);
        if (delay.compareTo(OAuth2ClientParams.MIN_REFRESH_DELAY) < 0) {
            delay = OAuth2ClientParams.MIN_REFRESH_DELAY;
        }
        return delay;
    }

    static Instant tokenExpirationTime(Instant now, Token token, Duration defaultLifespan) {
        Instant expirationTime = null;
        if (token != null && (expirationTime = token.getExpirationTime()) == null) {
            try {
                JwtToken jwtToken = JwtToken.parse(token.getPayload());
                expirationTime = jwtToken.getExpirationTime();
            }
            catch (Exception exception) {
                // empty catch block
            }
        }
        if (expirationTime == null) {
            expirationTime = now.plus(defaultLifespan);
        }
        return expirationTime;
    }

    private void checkErrorResponse(ResponseContext responseContext) {
        try {
            Status status = responseContext.getResponseCode();
            if (status.getCode() >= 400) {
                if (!responseContext.isJsonCompatibleResponse()) {
                    throw OAuth2Client.genericError(status);
                }
                InputStream is = responseContext.getErrorStream();
                if (is != null) {
                    try {
                        ErrorResponse errorResponse = (ErrorResponse)this.objectMapper.readValue(is, ErrorResponse.class);
                        throw new OAuth2Exception(status, errorResponse);
                    }
                    catch (IOException ignored) {
                        throw OAuth2Client.genericError(status);
                    }
                }
            }
        }
        catch (RuntimeException e) {
            throw e;
        }
        catch (Exception e) {
            throw new HttpClientException(e);
        }
    }

    private static HttpClientException genericError(Status status) {
        return new HttpClientException("OAuth2 server replied with HTTP status code: " + status.getCode());
    }

    static class MustFetchNewTokensException
    extends RuntimeException {
        public MustFetchNewTokensException(String message) {
            super(message);
        }
    }
}

