/*
 * Copyright (c) 2002-2017 "Neo Technology,"
 * Network Engine for Objects in Lund AB [http://neotechnology.com]
 *
 * This file is part of Neo4j.
 *
 * Neo4j is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package org.neo4j.kernel.impl.util;

import java.util.Collections;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.CopyOnWriteArrayList;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.Executor;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Future;
import java.util.concurrent.RejectedExecutionException;
import java.util.concurrent.ScheduledThreadPoolExecutor;
import java.util.concurrent.ThreadFactory;
import java.util.concurrent.TimeUnit;

import org.neo4j.helpers.Exceptions;
import org.neo4j.kernel.lifecycle.LifecycleAdapter;

import static java.util.concurrent.Executors.newCachedThreadPool;
import static org.neo4j.helpers.NamedThreadFactory.daemon;
import static org.neo4j.kernel.impl.util.DebugUtil.trackTest;
import static org.neo4j.kernel.impl.util.JobScheduler.Group.NO_METADATA;

public class Neo4jJobScheduler extends LifecycleAdapter implements JobScheduler
{
    private ExecutorService globalPool;
    private ScheduledThreadPoolExecutor scheduledExecutor;

    // Contains JobHandles which hasn't been cancelled yet, this to be able to cancel those when shutting down
    // This Set is synchronized, which is fine because there are only a handful of jobs generally and only
    // added when starting a database.
    private final Set<JobHandle> jobs = Collections.synchronizedSet( new HashSet<>() );

    @Override
    public void init()
    {
        this.globalPool = newCachedThreadPool( daemon( "neo4j.Pooled" + trackTest() ) );
        this.scheduledExecutor = new ScheduledThreadPoolExecutor( 2, daemon( "neo4j.Scheduled" + trackTest() ) );
    }

    @Override
    public Executor executor( final Group group )
    {
        return job -> schedule( group, job );
    }

    @Override
    public ThreadFactory threadFactory( final Group group )
    {
        return job -> createNewThread( group, job, NO_METADATA );
    }

    @Override
    public JobHandle schedule( Group group, Runnable job )
    {
        return schedule( group, job, NO_METADATA );
    }

    @Override
    public JobHandle schedule( Group group, Runnable job, Map<String,String> metadata )
    {
        if ( globalPool == null )
        {
            throw new RejectedExecutionException( "Scheduler is not started" );
        }

        switch ( group.strategy() )
        {
        case POOLED:
            return register( new PooledJobHandle( this.globalPool.submit( job ) ) );
        case NEW_THREAD:
            Thread thread = createNewThread( group, job, metadata );
            thread.start();
            return new SingleThreadHandle( thread );
        default:
            throw new IllegalArgumentException( "Unsupported strategy for scheduling job: " + group.strategy() );
        }
    }

    private JobHandle register( PooledJobHandle pooledJobHandle )
    {
        jobs.add( pooledJobHandle );

        // Return a JobHandle which removes itself from this register,
        // otherwise functions like the supplied handle
        return new JobHandle()
        {
            @Override
            public void waitTermination() throws InterruptedException, ExecutionException
            {
                pooledJobHandle.waitTermination();
            }

            @Override
            public void cancel( boolean mayInterruptIfRunning )
            {
                pooledJobHandle.cancel( mayInterruptIfRunning );
                jobs.remove( pooledJobHandle );
            }

            @Override
            public void registerCancelListener( CancelListener listener )
            {
                pooledJobHandle.registerCancelListener( listener );
            }
        };
    }

    @Override
    public JobHandle scheduleRecurring( Group group, final Runnable runnable, long period, TimeUnit timeUnit )
    {
        return scheduleRecurring( group, runnable, 0, period, timeUnit );
    }

    @Override
    public JobHandle scheduleRecurring( Group group, final Runnable runnable, long initialDelay, long period,
                                        TimeUnit timeUnit )
    {
        switch ( group.strategy() )
        {
        case POOLED:
            return new PooledJobHandle( scheduledExecutor.scheduleAtFixedRate( runnable, initialDelay, period, timeUnit ) );
        default:
            throw new IllegalArgumentException( "Unsupported strategy to use for recurring jobs: " + group.strategy() );
        }
    }

    @Override
    public JobHandle schedule( Group group, final Runnable runnable, long initialDelay, TimeUnit timeUnit )
    {
        switch ( group.strategy() )
        {
        case POOLED:
            return new PooledJobHandle( scheduledExecutor.schedule( runnable, initialDelay, timeUnit ) );
        default:
            throw new IllegalArgumentException( "Unsupported strategy to use for delayed jobs: " + group.strategy() );
        }
    }

    @Override
    public void stop()
    {
    }

    @Override
    public void shutdown()
    {
        RuntimeException exception = null;
        try
        {
            // Cancel jobs which hasn't been cancelled already, this to avoid having to wait the full
            // max wait time and then just leave them.
            for ( JobHandle handle : jobs )
            {
                handle.cancel( true );
            }
            jobs.clear();

            shutdownPool( globalPool );
        }
        catch ( RuntimeException e )
        {
            exception = e;
        }
        finally
        {
            globalPool = null;
        }

        try
        {
            shutdownPool( scheduledExecutor );
        }
        catch ( RuntimeException e )
        {
            exception = Exceptions.chain( exception, e );
        }
        finally
        {
            scheduledExecutor = null;
        }

        if ( exception != null )
        {
            throw new RuntimeException( "Unable to shut down job scheduler properly.", exception );
        }
    }

    private void shutdownPool( ExecutorService pool )
    {
        if ( pool != null )
        {
            pool.shutdown();
            try
            {
                pool.awaitTermination( 30, TimeUnit.SECONDS );
            }
            catch ( InterruptedException e )
            {
                throw new RuntimeException( e );
            }
        }
    }

    /**
     * Used to spin up new threads for groups or access-patterns that don't use the pooled thread options.
     * The returned thread is not started, to allow users to modify it before setting it in motion.
     */
    private Thread createNewThread( Group group, Runnable job, Map<String,String> metadata )
    {
        Thread thread = new Thread( null, job, group.threadName( metadata ) );
        thread.setDaemon( true );
        return thread;
    }

    private static class PooledJobHandle implements JobHandle
    {
        private final Future<?> job;
        private final List<CancelListener> cancelListeners = new CopyOnWriteArrayList<>();

        PooledJobHandle( Future<?> job )
        {
            this.job = job;
        }

        @Override
        public void cancel( boolean mayInterruptIfRunning )
        {
            job.cancel( mayInterruptIfRunning );
            for ( CancelListener cancelListener : cancelListeners )
            {
                cancelListener.cancelled( mayInterruptIfRunning );
            }
        }

        @Override
        public void waitTermination() throws InterruptedException, ExecutionException
        {
            job.get();
        }

        @Override
        public void registerCancelListener( CancelListener listener )
        {
            cancelListeners.add( listener );
        }
    }

    private static class SingleThreadHandle implements JobHandle
    {
        private final Thread thread;

        SingleThreadHandle( Thread thread )
        {
            this.thread = thread;
        }

        @Override
        public void cancel( boolean mayInterruptIfRunning )
        {
            if ( mayInterruptIfRunning )
            {
                thread.interrupt();
            }
        }

        @Override
        public void waitTermination() throws InterruptedException
        {
            thread.join();
        }
    }
}
