package org.jboss.classpool.plugins.jbosscl;
/*
* JBoss, Home of Professional Open Source.
* Copyright 2006, Red Hat Middleware LLC, and individual contributors
* as indicated by the @author tags. See the copyright.txt file in the
* distribution for a full listing of individual contributors. 
*
* This is free software; you can redistribute it and/or modify it
* under the terms of the GNU Lesser General Public License as
* published by the Free Software Foundation; either version 2.1 of
* the License, or (at your option) any later version.
*
* This software is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
* Lesser General Public License for more details.
*
* You should have received a copy of the GNU Lesser General Public
* License along with this software; if not, write to the Free
* Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
* 02110-1301 USA, or see the FSF site: http://www.fsf.org.
*/ 


import java.lang.ref.WeakReference;
import java.util.List;
import java.util.Map;
import java.util.WeakHashMap;
import java.util.concurrent.CopyOnWriteArrayList;

import org.jboss.classloader.spi.ClassLoaderDomain;
import org.jboss.classloader.spi.ClassLoaderSystem;
import org.jboss.classloading.spi.dependency.ClassLoading;
import org.jboss.classloading.spi.dependency.Module;

/**
 * 
 * @author <a href="kabir.khan@jboss.com">Kabir Khan</a>
 * @version $Revision: 104481 $
 */
public class VFSClassLoaderDomainRegistry implements DomainRegistry
{
   //protected final static ClassLoaderDomain domain = new ClassLoaderDomain("NOT_USED_PLACEHOLDER");
   
   private static final VFSClassLoaderDomainRegistry INSTANCE = new VFSClassLoaderDomainRegistry();
   
   private ClassLoaderSystem system;
   
   private ClassLoaderDomain defaultDomain;
   
   /** classloader domains by their classloaders */
   protected Map<ClassLoader, WeakReference<ClassLoaderDomain>> classLoaderDomainsByLoader = new WeakHashMap<ClassLoader, WeakReference<ClassLoaderDomain>>();

   /** parent deployment unit classloaders indexed by children */
   private Map<ClassLoader, WeakReference<ClassLoader>> classLoaderUnitParents = new WeakHashMap<ClassLoader, WeakReference<ClassLoader>>(); 
   
   /** Modules by classloader */
   private Map<ClassLoader, WeakReference<Module>> classLoaderModules = new WeakHashMap<ClassLoader, WeakReference<Module>>();
   
   /** classloaders by module */
   private Map<Module, WeakReference<ClassLoader>> moduleClassLoaders = new WeakHashMap<Module, WeakReference<ClassLoader>>();

   protected Map<ClassLoaderDomain, Integer> classLoaderDomainReferenceCounts = new WeakHashMap<ClassLoaderDomain, Integer>();
   
   private final List<Callback> callbacks = new CopyOnWriteArrayList<Callback>(); 
   
   private VFSClassLoaderDomainRegistry() 
   {
      
   }
   
   public static final VFSClassLoaderDomainRegistry getInstance()
   {
      return INSTANCE;
   }
   
   /**
    * Needed for the tests, not expected to be needed in production
    */
   public synchronized void setSystem(ClassLoaderSystem system)
   {
      this.system = system;
   }
   
   /**
    * Needed for the tests, not expected to be needed in production
    */
   public synchronized void setDefaultDomain(ClassLoaderDomain domain)
   {
      this.defaultDomain = domain;
   }
   
   public synchronized ClassLoaderSystem getSystem()
   {
      if (system == null)
      {
         system = ClassLoaderSystem.getInstance();
      }
      return system;
   }
   
   public synchronized ClassLoaderDomain getDefaultDomain()
   {
      if (defaultDomain == null)
      {
         defaultDomain = getSystem().getDefaultDomain();
      }
      return defaultDomain;
   }

   public synchronized boolean initMapsForModule(Module module)
   {
      ClassLoader loader = ClassLoading.getClassLoaderForModule(module);
      ClassLoader parentUnitLoader = findParentUnitLoader(module);
      if (loader == parentUnitLoader)
      {
         throw new IllegalArgumentException("initMapsForLoader() should only be called if parentUnitLoader is different from loader");
      }
      ClassLoaderSystem system = getSystem();
      
      String domainName = module.getDeterminedDomainName();
      ClassLoaderDomain clDomain = system.getDomain(domainName);
      this.validateInitMaps(loader, clDomain, module, parentUnitLoader);
      
      boolean ret = false;
      if (!classLoaderDomainsByLoader.containsKey(loader))
      {
         Integer count = classLoaderDomainReferenceCounts.get(clDomain);
         int cnt = count == null ? 0 : count.intValue();
         classLoaderDomainReferenceCounts.put(clDomain, cnt);
         
         classLoaderDomainsByLoader.put(loader, new WeakReference<ClassLoaderDomain>(clDomain));
         classLoaderUnitParents.put(loader, new WeakReference<ClassLoader>(parentUnitLoader));
         classLoaderModules.put(loader, new WeakReference<Module>(module));
         moduleClassLoaders.put(module, new WeakReference<ClassLoader>(loader));
         ret = true;
         this.initMapsDone(loader, clDomain, module, parentUnitLoader);
      }
      
      
      return ret;
   }
   
   protected void validateInitMaps(ClassLoader loader, ClassLoaderDomain loaderDomain, Module module, ClassLoader parentUnitLoader)
   {
      if (callbacks.size() == 0)
         return;
      for (Callback callback : callbacks)
         callback.validateInitMaps(loader, loaderDomain, module, parentUnitLoader);
   }
   
   protected void initMapsDone(ClassLoader loader, ClassLoaderDomain loaderDomain, Module module, ClassLoader parentUnitLoader)
   {
      if (callbacks.size() == 0)
         return;
      for (Callback callback : callbacks)
         callback.initMapsDone(loader, loaderDomain, module, parentUnitLoader);
   }
   
   public synchronized void cleanupModule(Module module)
   {
      ClassLoader loader = this.getClassLoader(module);
      validateCleanupLoader(loader);
      WeakReference<ClassLoaderDomain> clDomainRef = classLoaderDomainsByLoader.remove(loader);
      ClassLoaderDomain clDomain = clDomainRef == null ? null : clDomainRef.get();
      int cnt = 0;
      if (clDomain != null)
      {
         Integer count =  classLoaderDomainReferenceCounts.get(clDomain);
         cnt = count == null ? 0 : count.intValue();
         if (cnt > 0)
         {
            cnt--;
         }
         if (cnt == 0)
         {
            classLoaderDomainReferenceCounts.remove(clDomain);
         }
         else
         {
            classLoaderDomainReferenceCounts.put(clDomain, ++cnt);
         }
         classLoaderUnitParents.remove(loader);
         classLoaderModules.remove(loader);
         moduleClassLoaders.remove(module);
      }
      cleanupLoaderDone(loader, clDomain, cnt);
   }

   protected void validateCleanupLoader(ClassLoader loader)
   {
      
   }
   
   protected void cleanupLoaderDone(ClassLoader loader, ClassLoaderDomain loaderDomain, int domainReferences)
   {
      if (callbacks.size() == 0)
         return;
      for (Callback callback : callbacks)
         callback.cleanupLoaderDone(loader, loaderDomain, domainReferences);
   }
   
   public synchronized ClassLoaderDomain getClassLoaderDomainForLoader(ClassLoader cl)
   {
      WeakReference<ClassLoaderDomain> clDomainRef = classLoaderDomainsByLoader.get(cl);
      if (clDomainRef != null)
      {
         return clDomainRef.get();
      }
      
      ClassLoader parent = SecurityActions.getParent(cl);
      if (parent != null)
      {
         ClassLoaderDomain domain = getClassLoaderDomainForLoader(parent);
         if (domain != null)
         {
            classLoaderDomainsByLoader.put(parent, new WeakReference<ClassLoaderDomain>(domain));
            return domain;
         }
      }
      return null;
   }
   
   public synchronized ClassLoader getParentUnitLoader(ClassLoader loader)
   {
      WeakReference<ClassLoader> parentRef = classLoaderUnitParents.get(loader);
      if (parentRef != null)
      {
         return parentRef.get();
      }
      return null;
   }
   
   public synchronized Module getModule(ClassLoader loader)
   {
      WeakReference<Module> moduleRef = classLoaderModules.get(loader);
      if (moduleRef != null)
      {
         return moduleRef.get();
      }
      return null;
   }
   
   public synchronized ClassLoader getClassLoader(Module module)
   {
      WeakReference<ClassLoader> loaderRef = moduleClassLoaders.get(module);
      if (loaderRef != null)
      {
         return loaderRef.get();
      }
      return null;
   }
   
   private ClassLoader findParentUnitLoader(Module module)
   {
      return ParentUnitLoaderFinders.getInstance().findParentUnitLoader(module);
   }
   
   public void addCallback(Callback callback)
   {
      callbacks.add(callback);
   }
   
   public void removeCallback(Callback callback)
   {
      callbacks.remove(callback);
   }
}