/*
* JBoss, Home of Professional Open Source.
* Copyright 2006, Red Hat Middleware LLC, and individual contributors
* as indicated by the @author tags. See the copyright.txt file in the
* distribution for a full listing of individual contributors. 
*
* This is free software; you can redistribute it and/or modify it
* under the terms of the GNU Lesser General Public License as
* published by the Free Software Foundation; either version 2.1 of
* the License, or (at your option) any later version.
*
* This software is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
* Lesser General Public License for more details.
*
* You should have received a copy of the GNU Lesser General Public
* License along with this software; if not, write to the Free
* Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
* 02110-1301 USA, or see the FSF site: http://www.fsf.org.
*/ 
package org.jboss.classpool.plugins.jbosscl;

import java.lang.reflect.Method;
import java.security.AccessController;
import java.security.PrivilegedAction;
import java.security.PrivilegedActionException;
import java.security.PrivilegedExceptionAction;

import org.jboss.classloader.spi.base.BaseClassLoaderSource;

/**
 * 
 * @author <a href="kabir.khan@jboss.com">Kabir Khan</a>
 * @version $Revision: 104426 $
 */
class SecurityActions
{
   interface GetParentAction
   {
      ClassLoader getParent(ClassLoader loader);
      
      GetParentAction NON_PRIVILEGED = new GetParentAction()
      {
         public ClassLoader getParent(ClassLoader loader)
         {
            return loader.getParent();
         }
      };
      
      GetParentAction PRIVILEGED = new GetParentAction()
      {
         public ClassLoader getParent(final ClassLoader loader)
         {
            try
            {
               return AccessController.doPrivileged(new PrivilegedExceptionAction<ClassLoader>()
               {
                  public ClassLoader run() throws Exception
                  {
                     return loader.getParent();
                  }
               });
            }
            catch (PrivilegedActionException e)
            {
               throw new RuntimeException(e.getException());
            }
         }
      };
   }

   public static ClassLoader getParent(ClassLoader loader)
   {
      if (System.getSecurityManager() == null)
      {
         return GetParentAction.NON_PRIVILEGED.getParent(loader);
      }
      else
      {
         return GetParentAction.PRIVILEGED.getParent(loader);
      }
   }
   
   private static final Method getClassLoader;

   static
   {
      getClassLoader = AccessController.doPrivileged(new PrivilegedAction<Method>()
      {
         public Method run()
         {
            try
            {
               Method method = BaseClassLoaderSource.class.getDeclaredMethod("getClassLoader");
               method.setAccessible(true);
               return method;
            }
            catch (NoSuchMethodException e)
            {
               throw new RuntimeException("Cannot get classloader from " + BaseClassLoaderSource.class.getName(), e);
            }
         }
      });
   }

   static ClassLoader getClassLoader(BaseClassLoaderSource clSource)
   {
      try
      {
         return (ClassLoader)getClassLoader.invoke(clSource);
      }
      catch (Exception e)
      {
         throw new RuntimeException(e);
      }
   }
   
   private interface LoadClassAction
   {
      Class<?> loadClass(ClassLoader loader, String name) throws ClassNotFoundException;
      
      LoadClassAction NON_PRIVILEGED = new LoadClassAction()
      {
         public Class<?> loadClass(ClassLoader loader, String name) throws ClassNotFoundException
         {
            return loader.loadClass(name);
         }
      };
      
      LoadClassAction PRIVILEGED = new LoadClassAction()
      {
         
         public Class<?> loadClass(final ClassLoader loader, final String name) throws ClassNotFoundException
         {
            try
            {
               return AccessController.doPrivileged(new PrivilegedExceptionAction<Class<?>>()
               {

                  public Class<?> run() throws Exception
                  {
                     return loader.loadClass(name);
                  }
               });
            }
            catch (PrivilegedActionException e)
            {
               if (e.getCause() instanceof RuntimeException)
                  throw new RuntimeException(e.getCause());
               if (e.getCause() instanceof ClassNotFoundException)
                  throw (ClassNotFoundException)e.getCause();
               throw new RuntimeException(e.getCause());
            }
         }
      };
   }
   
   
   static Class<?> loadClass(ClassLoader loader, String name) throws ClassNotFoundException
   {
      if (System.getSecurityManager() == null)
         return LoadClassAction.NON_PRIVILEGED.loadClass(loader, name);
      return LoadClassAction.PRIVILEGED.loadClass(loader, name);
   }
}
