/*
 * Copyright (c) 2012, 2018 Oracle and/or its affiliates. All rights reserved.
 *
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License v. 2.0, which is available at
 * http://www.eclipse.org/legal/epl-2.0.
 *
 * This Source Code may also be made available under the following Secondary
 * Licenses when the conditions for such availability set forth in the
 * Eclipse Public License v. 2.0 are satisfied: GNU General Public License,
 * version 2 with the GNU Classpath Exception, which is available at
 * https://www.gnu.org/software/classpath/license.html.
 *
 * SPDX-License-Identifier: EPL-2.0 OR GPL-2.0 WITH Classpath-exception-2.0
 */

package org.glassfish.cdi.hk2;

import java.lang.annotation.Annotation;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

import javax.enterprise.event.Observes;
import javax.enterprise.inject.spi.AfterBeanDiscovery;
import javax.enterprise.inject.spi.AfterDeploymentValidation;
import javax.enterprise.inject.spi.Extension;
import javax.enterprise.inject.spi.InjectionPoint;
import javax.enterprise.inject.spi.InjectionTarget;
import javax.enterprise.inject.spi.ProcessInjectionTarget;
import javax.inject.Singleton;

import org.glassfish.hk2.api.ActiveDescriptor;
import org.glassfish.hk2.api.DynamicConfiguration;
import org.glassfish.hk2.api.DynamicConfigurationService;
import org.glassfish.hk2.api.Injectee;
import org.glassfish.hk2.api.PerLookup;
import org.glassfish.hk2.api.ServiceLocator;

/**
 * A portable CDI extension, which is the touch-point for hk2 integration with CDI
 *
 * This extension is registered via META-INF/services/javax.enterprise.inject.spi.Extension
 *
 * @author jwells
 *
 */
public class HK2IntegrationExtension implements Extension {
    private final HashMap<Long, ActiveDescriptor<?>> foundWithHK2 = new HashMap<Long, ActiveDescriptor<?>>();
    private final ServiceLocator locator = HK2IntegrationUtilities.getApplicationServiceLocator();

    /**
     * Called by CDI, gathers up all of the injection points known to hk2.
     *
     * @param pit The injection target even from CDI
     */
    @SuppressWarnings("unused")
    private <T> void injectionTargetObserver(@Observes ProcessInjectionTarget<T> pit) {
        if (locator == null) return;

        InjectionTarget<?> injectionTarget = pit.getInjectionTarget();
        Set<InjectionPoint> injectionPoints = injectionTarget.getInjectionPoints();

        for (InjectionPoint injectionPoint : injectionPoints) {
            Injectee injectee = HK2IntegrationUtilities.convertInjectionPointToInjectee(injectionPoint);

            ActiveDescriptor<?> descriptor = locator.getInjecteeDescriptor(injectee);
            if (descriptor == null || descriptor.getServiceId() == null) continue;

            // using a map removes duplicates
            foundWithHK2.put(descriptor.getServiceId(), descriptor);
        }
    }

    /**
     * Called by CDI after going through all of the injection points. For each
     * service known to hk2, adds a CDI bean.
     *
     * @param abd This is used just to mark the type of the event
     */
    @SuppressWarnings({ "unused", "unchecked", "rawtypes" })
    private void afterDiscoveryObserver(@Observes AfterBeanDiscovery abd) {
        if (locator == null) return;

        HashSet<Class<? extends Annotation>> customScopes = new HashSet<Class<? extends Annotation>>();

        for (ActiveDescriptor<?> descriptor : foundWithHK2.values()) {
            abd.addBean(new HK2CDIBean(locator, descriptor));

            customScopes.add(descriptor.getScopeAnnotation());
        }

        customScopes.remove(PerLookup.class);
        customScopes.remove(Singleton.class);

        List<org.glassfish.hk2.api.Context> hk2Contexts = locator.getAllServices(org.glassfish.hk2.api.Context.class);

        for (org.glassfish.hk2.api.Context hk2Context : hk2Contexts) {
            if (!customScopes.contains(hk2Context.getScope())) {
                continue;
            }

            abd.addContext(new HK2ContextBridge(hk2Context));
        }

        foundWithHK2.clear();  // No need to keep these references around
    }

    /**
     * Called by CDI after it has been completely validated.  Will add the JIT resolver to HK2
     * with the BeanManager
     *
     * @param event This is just to mark the type of the event
     * @param manager The manager that will be used to get references
     */
    @SuppressWarnings("unused")
    private void afterDeploymentValidation(@Observes AfterDeploymentValidation event) {
        if (locator == null) return;

        DynamicConfigurationService dcs = locator.getService(DynamicConfigurationService.class);
        DynamicConfiguration config = dcs.createDynamicConfiguration();

        config.addActiveDescriptor(CDISecondChanceResolver.class);
        config.addActiveDescriptor(CDIContextBridge.class);

        config.commit();
    }

    public String toString() {
        return "HK2IntegrationExtension(" + locator + "," + System.identityHashCode(this) + ")";
    }
}
