/**
 * Copyright (c) 2013-2017 Lorenzo Bettini.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *   Lorenzo Bettini - Initial contribution and API
 */
package org.eclipse.xsemantics.dsl.validation;

import com.google.common.base.Objects;
import com.google.common.collect.Iterables;
import com.google.common.collect.ListMultimap;
import com.google.inject.Inject;
import java.util.Collection;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;
import org.eclipse.emf.common.util.EList;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.EStructuralFeature;
import org.eclipse.xsemantics.dsl.XsemanticsConstants;
import org.eclipse.xsemantics.dsl.typing.TupleType;
import org.eclipse.xsemantics.dsl.typing.XsemanticsTypeSystem;
import org.eclipse.xsemantics.dsl.util.XsemanticsNodeModelUtils;
import org.eclipse.xsemantics.dsl.util.XsemanticsUtils;
import org.eclipse.xsemantics.dsl.xsemantics.AbstractFieldDefinition;
import org.eclipse.xsemantics.dsl.xsemantics.AuxiliaryDescription;
import org.eclipse.xsemantics.dsl.xsemantics.AuxiliaryFunction;
import org.eclipse.xsemantics.dsl.xsemantics.CheckRule;
import org.eclipse.xsemantics.dsl.xsemantics.FieldDefinition;
import org.eclipse.xsemantics.dsl.xsemantics.Injected;
import org.eclipse.xsemantics.dsl.xsemantics.InputParameter;
import org.eclipse.xsemantics.dsl.xsemantics.JudgmentDescription;
import org.eclipse.xsemantics.dsl.xsemantics.JudgmentParameter;
import org.eclipse.xsemantics.dsl.xsemantics.OutputParameter;
import org.eclipse.xsemantics.dsl.xsemantics.Overrider;
import org.eclipse.xsemantics.dsl.xsemantics.ReferToJudgment;
import org.eclipse.xsemantics.dsl.xsemantics.Rule;
import org.eclipse.xsemantics.dsl.xsemantics.RuleConclusion;
import org.eclipse.xsemantics.dsl.xsemantics.RuleConclusionElement;
import org.eclipse.xsemantics.dsl.xsemantics.RuleInvocation;
import org.eclipse.xsemantics.dsl.xsemantics.RuleParameter;
import org.eclipse.xsemantics.dsl.xsemantics.UniqueByName;
import org.eclipse.xsemantics.dsl.xsemantics.XsemanticsPackage;
import org.eclipse.xsemantics.dsl.xsemantics.XsemanticsSystem;
import org.eclipse.xtext.EcoreUtil2;
import org.eclipse.xtext.common.types.JvmFormalParameter;
import org.eclipse.xtext.common.types.JvmIdentifiableElement;
import org.eclipse.xtext.common.types.JvmParameterizedTypeReference;
import org.eclipse.xtext.common.types.JvmTypeReference;
import org.eclipse.xtext.common.types.TypesPackage;
import org.eclipse.xtext.validation.Check;
import org.eclipse.xtext.validation.ValidationMessageAcceptor;
import org.eclipse.xtext.xbase.XAssignment;
import org.eclipse.xtext.xbase.XClosure;
import org.eclipse.xtext.xbase.XExpression;
import org.eclipse.xtext.xbase.XFeatureCall;
import org.eclipse.xtext.xbase.XReturnExpression;
import org.eclipse.xtext.xbase.XThrowExpression;
import org.eclipse.xtext.xbase.XVariableDeclaration;
import org.eclipse.xtext.xbase.XbasePackage;
import org.eclipse.xtext.xbase.lib.Extension;
import org.eclipse.xtext.xbase.lib.Functions.Function1;
import org.eclipse.xtext.xbase.lib.Functions.Function2;
import org.eclipse.xtext.xbase.lib.IterableExtensions;
import org.eclipse.xtext.xbase.typesystem.references.LightweightTypeReference;
import org.eclipse.xtext.xbase.typesystem.util.Multimaps2;
import org.eclipse.xtext.xbase.validation.IssueCodes;

/**
 * Custom validation rules.
 * 
 * see http://www.eclipse.org/Xtext/documentation.html#validation
 */
@SuppressWarnings("all")
public class XsemanticsValidator extends AbstractXsemanticsValidator {
  @Inject
  protected XsemanticsTypeSystem typeSystem;

  @Inject
  @Extension
  protected XsemanticsUtils _xsemanticsUtils;

  @Inject
  protected XsemanticsXExpressionHelper xExpressionHelper;

  @Inject
  protected XsemanticsNodeModelUtils nodeModelUtils;

  public static final int maxOfOutputParams = 3;

  protected boolean enableWarnings = true;

  public static final String PREFIX = "org.eclipse.xsemantics.dsl.validation.";

  public static final String DUPLICATE_JUDGMENT_DESCRIPTION_SYMBOLS = (XsemanticsValidator.PREFIX + "DuplicateJudgmentDescriptionSymbols");

  public static final String NO_JUDGMENT_DESCRIPTION = (XsemanticsValidator.PREFIX + "NoJudgmentDescription");

  public static final String NOT_SUBTYPE = (XsemanticsValidator.PREFIX + "NotSubtype");

  public static final String DUPLICATE_RULE_WITH_SAME_ARGUMENTS = (XsemanticsValidator.PREFIX + "DuplicateRulesWithSameArguments");

  public static final String DUPLICATE_AUXFUN_WITH_SAME_ARGUMENTS = (XsemanticsValidator.PREFIX + "DuplicateAuxFunWithSameArguments");

  public static final String MUST_OVERRIDE = (XsemanticsValidator.PREFIX + "MustOverride");

  public static final String DUPLICATE_NAME = (XsemanticsValidator.PREFIX + "DuplicateName");

  public static final String NOT_VALIDATOR = (XsemanticsValidator.PREFIX + "NotAbstractDeclarativeValidator");

  public static final String NOT_PARAMETER = (XsemanticsValidator.PREFIX + "NotParameter");

  public static final String NOT_VALID_OUTPUT_ARG = (XsemanticsValidator.PREFIX + "NotValidOutputArg");

  public static final String NOT_VALID_INPUT_ARG = (XsemanticsValidator.PREFIX + "NotValidInputArg");

  public static final String TOO_MANY_OUTPUT_PARAMS = (XsemanticsValidator.PREFIX + "TooManyOutputParams");

  public static final String NO_INPUT_PARAM = (XsemanticsValidator.PREFIX + "NoInputParam");

  public static final String ASSIGNMENT_TO_INPUT_PARAM = (XsemanticsValidator.PREFIX + "AssignmentToInputParam");

  public static final String NO_RULE_FOR_JUDGMENT_DESCRIPTION = (XsemanticsValidator.PREFIX + "NoRuleForJudgmentDescription");

  public static final String NO_AUXFUN_FOR_AUX_DESCRIPTION = (XsemanticsValidator.PREFIX + "NoAuxFunForAuxiliaryDescription");

  public static final String RETURN_NOT_ALLOWED = (XsemanticsValidator.PREFIX + "ReturnNotAllowed");

  public static final String THROW_NOT_ALLOWED = (XsemanticsValidator.PREFIX + "ThrowNotAllowed");

  public static final String NOT_VALID_SUPER_SYSTEM = (XsemanticsValidator.PREFIX + "NotValidSuperSystem");

  public static final String CYCLIC_HIERARCHY = (XsemanticsValidator.PREFIX + "CyclicHierarchy");

  public static final String EXTENDS_CANNOT_COEXIST_WITH_VALIDATOR_EXTENDS = (XsemanticsValidator.PREFIX + "ExtendsCannotCoexistWithValidatorExtends");

  public static final String OVERRIDE_WITHOUT_SYSTEM_EXTENDS = (XsemanticsValidator.PREFIX + "OverrideWithoutSystemExtends");

  public static final String NOTHING_TO_OVERRIDE = (XsemanticsValidator.PREFIX + "NothingToOverride");

  public static final String DUPLICATE_AUXILIARY_NAME = (XsemanticsValidator.PREFIX + "DuplicateAuxiliaryDescription");

  public static final String NO_AUXDESC_FOR_AUX_FUNCTION = (XsemanticsValidator.PREFIX + "NoAuxDescForAuxiliaryFunction");

  public static final String PARAMS_SIZE_DONT_MATCH = (XsemanticsValidator.PREFIX + "ParamsSizeDontMatch");

  public static final String ACCESS_TO_OUTPUT_PARAM_WITHIN_CLOSURE = (XsemanticsValidator.PREFIX + "AccessToOutputParamWithinClosure");

  public static final String RESERVED_VARIABLE_NAME = (XsemanticsValidator.PREFIX + "ReservedVariableName");

  public static final String FINAL_FIELD_NOT_INITIALIZED = (XsemanticsValidator.PREFIX + "FinalFieldNotInitialized");

  public static final String TOO_LITTLE_TYPE_INFORMATION = (XsemanticsValidator.PREFIX + "TooLittleTypeInformation");

  @Check
  @Override
  public void checkAssignment(final XAssignment assignment) {
    final JvmIdentifiableElement assignmentFeature = assignment.getFeature();
    if ((assignmentFeature instanceof JvmFormalParameter)) {
      boolean _isInputParam = this._xsemanticsUtils.isInputParam(((JvmFormalParameter)assignmentFeature));
      if (_isInputParam) {
        this.error("Assignment to input parameter", 
          XbasePackage.Literals.XASSIGNMENT__ASSIGNABLE, 
          ValidationMessageAcceptor.INSIGNIFICANT_INDEX, 
          XsemanticsValidator.ASSIGNMENT_TO_INPUT_PARAM);
      }
      return;
    }
    super.checkAssignment(assignment);
  }

  @Check
  @Override
  public void checkReturn(final XReturnExpression expr) {
    boolean _isContainedInAuxiliaryFunction = this.isContainedInAuxiliaryFunction(expr);
    boolean _not = (!_isContainedInAuxiliaryFunction);
    if (_not) {
      this.error("Return statements are not allowed here", expr, null, 
        XsemanticsValidator.RETURN_NOT_ALLOWED);
    }
  }

  @Check
  @Override
  public void checkVariableDeclaration(final XVariableDeclaration declaration) {
    super.checkVariableDeclaration(declaration);
    boolean _equals = XsemanticsConstants.PREVIOUS_FAILURE.equals(declaration.getName());
    if (_equals) {
      this.error((XsemanticsConstants.PREVIOUS_FAILURE + 
        " is a reserved name"), declaration, null, 
        XsemanticsValidator.RESERVED_VARIABLE_NAME);
    }
  }

  protected boolean isContainedInAuxiliaryFunction(final XExpression expr) {
    AuxiliaryFunction _containerOfType = EcoreUtil2.<AuxiliaryFunction>getContainerOfType(expr, AuxiliaryFunction.class);
    return (_containerOfType != null);
  }

  @Check
  public void checkThrow(final XThrowExpression expr) {
    this.error("Throw statements are not allowed here", expr, null, 
      XsemanticsValidator.THROW_NOT_ALLOWED);
  }

  @Override
  protected boolean isLocallyUsed(final EObject target, final EObject containerToFindUsage) {
    if ((containerToFindUsage instanceof RuleInvocation)) {
      return true;
    }
    return super.isLocallyUsed(target, containerToFindUsage);
  }

  @Override
  protected boolean isValueExpectedRecursive(final XExpression expr) {
    final boolean valueExpectedRecursive = super.isValueExpectedRecursive(expr);
    return (valueExpectedRecursive || this.xExpressionHelper.isXsemanticsXExpression(expr.eContainer()));
  }

  @Check
  public void checkJudgmentDescription(final JudgmentDescription judgmentDescription) {
    this.checkNumOfOutputParams(judgmentDescription);
    this.checkInputParams(judgmentDescription);
    this.checkJudgmentDescriptionRules(judgmentDescription);
  }

  public void checkJudgmentDescriptionRules(final JudgmentDescription judgmentDescription) {
    final Iterable<Rule> rulesForJudgmentDescription = this._xsemanticsUtils.rulesForJudgmentDescription(judgmentDescription);
    boolean _isEmpty = IterableExtensions.isEmpty(rulesForJudgmentDescription);
    if (_isEmpty) {
      if ((this.isEnableWarnings() && (!judgmentDescription.isOverride()))) {
        this.warning("No rule defined for the judgment description", 
          XsemanticsPackage.Literals.JUDGMENT_DESCRIPTION.getEIDAttribute(), 
          XsemanticsValidator.NO_RULE_FOR_JUDGMENT_DESCRIPTION);
      }
    } else {
      final EList<JudgmentParameter> judgmentParameters = judgmentDescription.getJudgmentParameters();
      for (final Rule rule : rulesForJudgmentDescription) {
        {
          final EList<RuleConclusionElement> conclusionElements = rule.getConclusion().getConclusionElements();
          final Iterator<JudgmentParameter> judgmentParametersIt = judgmentParameters.iterator();
          for (final RuleConclusionElement ruleConclusionElement : conclusionElements) {
            if (((!this._xsemanticsUtils.isOutputParameter(judgmentParametersIt.next())) && (!(ruleConclusionElement instanceof RuleParameter)))) {
              this.error("Must be a parameter, not an expression", ruleConclusionElement, 
                XsemanticsPackage.Literals.RULE_CONCLUSION_ELEMENT.getEIDAttribute(), 
                XsemanticsValidator.NOT_PARAMETER);
            }
          }
          this.checkRuleConformantToJudgmentDescription(rule, judgmentDescription);
        }
      }
    }
  }

  protected void checkNumOfOutputParams(final JudgmentDescription judgmentDescription) {
    int _size = this._xsemanticsUtils.outputJudgmentParameters(judgmentDescription).size();
    boolean _greaterThan = (_size > XsemanticsValidator.maxOfOutputParams);
    if (_greaterThan) {
      this.error((("No more than " + Integer.valueOf(XsemanticsValidator.maxOfOutputParams)) + " output parameters are handled at the moment"), 
        XsemanticsPackage.Literals.JUDGMENT_DESCRIPTION__JUDGMENT_PARAMETERS, 
        XsemanticsValidator.TOO_MANY_OUTPUT_PARAMS);
    }
  }

  protected void checkInputParams(final JudgmentDescription judgmentDescription) {
    final List<InputParameter> inputParams = this._xsemanticsUtils.inputParams(judgmentDescription);
    boolean _isEmpty = inputParams.isEmpty();
    if (_isEmpty) {
      this.error("No input parameter; at least one is needed", 
        XsemanticsPackage.Literals.JUDGMENT_DESCRIPTION__JUDGMENT_PARAMETERS, 
        XsemanticsValidator.NO_INPUT_PARAM);
    } else {
      this.<InputParameter>checkDuplicatesByName(inputParams, null, XsemanticsValidator.DUPLICATE_NAME);
    }
  }

  @Check
  public void checkRule(final Rule rule) {
    final RuleConclusion conclusion = rule.getConclusion();
    this.findJudgmentDescriptionOrError(rule, 
      conclusion.getJudgmentSymbol(), 
      conclusion.getRelationSymbols(), 
      XsemanticsPackage.Literals.RULE__CONCLUSION);
  }

  @Check
  public void checkRuleInvocation(final RuleInvocation ruleInvocation) {
    final JudgmentDescription judgmentDescription = this.checkRuleInvocationConformantToJudgmentDescription(ruleInvocation);
    if ((judgmentDescription != null)) {
      final EList<JudgmentParameter> judgmentParameters = judgmentDescription.getJudgmentParameters();
      final EList<XExpression> invocationExpressions = ruleInvocation.getExpressions();
      final Iterator<JudgmentParameter> judgmentParametersIt = judgmentParameters.iterator();
      for (final XExpression ruleInvocationExpression : invocationExpressions) {
        boolean _isOutputParameter = this._xsemanticsUtils.isOutputParameter(judgmentParametersIt.next());
        if (_isOutputParameter) {
          boolean _validOutputArgExpression = this._xsemanticsUtils.validOutputArgExpression(ruleInvocationExpression);
          boolean _not = (!_validOutputArgExpression);
          if (_not) {
            String _programText = this.nodeModelUtils.getProgramText(ruleInvocationExpression);
            String _plus = ("Not a valid argument for output parameter: " + _programText);
            this.error(_plus, ruleInvocationExpression, 
              null, 
              XsemanticsValidator.NOT_VALID_OUTPUT_ARG);
          }
        } else {
          boolean _validInputArgExpression = this._xsemanticsUtils.validInputArgExpression(ruleInvocationExpression);
          boolean _not_1 = (!_validInputArgExpression);
          if (_not_1) {
            String _programText_1 = this.nodeModelUtils.getProgramText(ruleInvocationExpression);
            String _plus_1 = ("Not a valid argument for input parameter: " + _programText_1);
            this.error(_plus_1, ruleInvocationExpression, 
              null, 
              XsemanticsValidator.NOT_VALID_INPUT_ARG);
          }
        }
      }
    }
  }

  @Check
  public void checkSystem(final XsemanticsSystem system) {
    final JvmParameterizedTypeReference validatorExtends = system.getValidatorExtends();
    if ((validatorExtends != null)) {
      boolean _isAbstractDeclarativeValidator = this.typeSystem.isAbstractDeclarativeValidator(validatorExtends, system);
      boolean _not = (!_isAbstractDeclarativeValidator);
      if (_not) {
        Object _nameOfTypes = this.getNameOfTypes(validatorExtends);
        String _plus = ("Not an AbstractDeclarativeValidator: " + _nameOfTypes);
        this.error(_plus, 
          XsemanticsPackage.Literals.XSEMANTICS_SYSTEM__VALIDATOR_EXTENDS, 
          XsemanticsValidator.NOT_VALIDATOR);
      }
    }
    final JvmParameterizedTypeReference superSystem = system.getSuperSystem();
    if ((superSystem != null)) {
      boolean _isValidSuperSystem = this.typeSystem.isValidSuperSystem(superSystem, system);
      boolean _not_1 = (!_isValidSuperSystem);
      if (_not_1) {
        Object _nameOfTypes_1 = this.getNameOfTypes(superSystem);
        String _plus_1 = ("Not an Xsemantics system: " + _nameOfTypes_1);
        this.error(_plus_1, 
          XsemanticsPackage.Literals.XSEMANTICS_SYSTEM__SUPER_SYSTEM, 
          XsemanticsValidator.NOT_VALID_SUPER_SYSTEM);
      }
      if ((validatorExtends != null)) {
        this.error("system \'extends\' cannot coexist with \'validatorExtends\'", 
          XsemanticsPackage.Literals.XSEMANTICS_SYSTEM__SUPER_SYSTEM, 
          XsemanticsValidator.EXTENDS_CANNOT_COEXIST_WITH_VALIDATOR_EXTENDS);
        this.error("system \'extends\' cannot coexist with \'validatorExtends\'", 
          XsemanticsPackage.Literals.XSEMANTICS_SYSTEM__VALIDATOR_EXTENDS, 
          XsemanticsValidator.EXTENDS_CANNOT_COEXIST_WITH_VALIDATOR_EXTENDS);
      }
    }
    final List<XsemanticsSystem> superSystems = this._xsemanticsUtils.allSuperSystemDefinitions(system);
    boolean _contains = superSystems.contains(system);
    if (_contains) {
      this.error("Cycle in extends relation", 
        XsemanticsPackage.Literals.XSEMANTICS_SYSTEM__SUPER_SYSTEM, 
        XsemanticsValidator.CYCLIC_HIERARCHY);
    }
    final XsemanticsSystem superSystemDefinition = this._xsemanticsUtils.superSystemDefinition(system);
    Iterable<JudgmentDescription> _allJudgments = null;
    if (superSystemDefinition!=null) {
      _allJudgments=this._xsemanticsUtils.allJudgments(superSystemDefinition);
    }
    final Iterable<JudgmentDescription> allSuperJudgments = _allJudgments;
    final Function2<JudgmentDescription, JudgmentDescription, Boolean> _function = (JudgmentDescription j1, JudgmentDescription j2) -> {
      return Boolean.valueOf(((Objects.equal(j1.getJudgmentSymbol(), j2.getJudgmentSymbol()) && 
        IterableExtensions.elementsEqual(j1.getRelationSymbols(), j2.getRelationSymbols())) && 
        this.typeSystem.equals(j1, j2)));
    };
    this.<JudgmentDescription>checkOverrides(system.getJudgmentDescriptions(), allSuperJudgments, _function, 
      "judgment");
    Iterable<AuxiliaryDescription> _allAuxiliaryDescriptions = null;
    if (superSystemDefinition!=null) {
      _allAuxiliaryDescriptions=this._xsemanticsUtils.allAuxiliaryDescriptions(superSystemDefinition);
    }
    final Iterable<AuxiliaryDescription> allSuperAuxiliaryDescriptions = _allAuxiliaryDescriptions;
    final Function2<AuxiliaryDescription, AuxiliaryDescription, Boolean> _function_1 = (AuxiliaryDescription a1, AuxiliaryDescription a2) -> {
      return Boolean.valueOf(this.typeSystem.equals(a1, a2));
    };
    this.<AuxiliaryDescription>checkOverrides(system.getAuxiliaryDescriptions(), allSuperAuxiliaryDescriptions, _function_1, 
      "auxiliary description");
    Iterable<CheckRule> _allCheckRules = null;
    if (superSystemDefinition!=null) {
      _allCheckRules=this._xsemanticsUtils.allCheckRules(superSystemDefinition);
    }
    final Iterable<CheckRule> allSuperCheckRules = _allCheckRules;
    final Function2<CheckRule, CheckRule, Boolean> _function_2 = (CheckRule r1, CheckRule r2) -> {
      return Boolean.valueOf(this.typeSystem.equals(
        r1.getElement().getParameter().getParameterType(), 
        r2.getElement().getParameter().getParameterType(), r1));
    };
    this.<CheckRule>checkOverrides(system.getCheckrules(), allSuperCheckRules, _function_2, 
      "checkrule");
    Iterable<Rule> _allRules = null;
    if (superSystemDefinition!=null) {
      _allRules=this._xsemanticsUtils.allRules(superSystemDefinition);
    }
    final Iterable<Rule> allSuperRules = _allRules;
    final Function2<Rule, Rule, Boolean> _function_3 = (Rule r1, Rule r2) -> {
      return Boolean.valueOf(((r1.getConclusion().getJudgmentSymbol().equals(r2.getConclusion().getJudgmentSymbol()) && 
        IterableExtensions.elementsEqual(r1.getConclusion().getRelationSymbols(), r2.getConclusion().getRelationSymbols())) && 
        this.typeSystem.equals(
          this.typeSystem.getInputTypes(r1), 
          this.typeSystem.getInputTypes(r2))));
    };
    this.<Rule>checkOverrides(system.getRules(), allSuperRules, _function_3, 
      "rule");
    EList<Rule> _rules = system.getRules();
    for (final Rule rule : _rules) {
      {
        final RuleConclusion conclusion = rule.getConclusion();
        final List<Rule> rulesOfTheSameKind = this._xsemanticsUtils.allRulesByJudgmentDescription(system, 
          conclusion.getJudgmentSymbol(), conclusion.getRelationSymbols());
        int _size = rulesOfTheSameKind.size();
        boolean _greaterThan = (_size > 1);
        if (_greaterThan) {
          final TupleType tupleType = this.typeSystem.getInputTypes(rule);
          for (final Rule rule2 : rulesOfTheSameKind) {
            if (((!Objects.equal(rule2, rule)) && (!rule.isOverride()))) {
              final TupleType tupleType2 = this.typeSystem.getInputTypes(rule2);
              boolean _equals = this.typeSystem.equals(tupleType, tupleType2);
              if (_equals) {
                String _tupleTypeRepresentation = this.tupleTypeRepresentation(tupleType);
                String _plus_2 = ("Duplicate rule of the same kind with parameters: " + _tupleTypeRepresentation);
                String _reportContainingSystemName = this.reportContainingSystemName(rule2);
                String _plus_3 = (_plus_2 + _reportContainingSystemName);
                this.error(_plus_3, rule, 
                  XsemanticsPackage.Literals.RULE__CONCLUSION, 
                  XsemanticsValidator.DUPLICATE_RULE_WITH_SAME_ARGUMENTS);
              }
            }
          }
        }
      }
    }
    EList<AbstractFieldDefinition> _fields = system.getFields();
    EList<JudgmentDescription> _judgmentDescriptions = system.getJudgmentDescriptions();
    Iterable<UniqueByName> _plus_2 = Iterables.<UniqueByName>concat(_fields, _judgmentDescriptions);
    EList<AuxiliaryDescription> _auxiliaryDescriptions = system.getAuxiliaryDescriptions();
    Iterable<UniqueByName> _plus_3 = Iterables.<UniqueByName>concat(_plus_2, _auxiliaryDescriptions);
    EList<Rule> _rules_1 = system.getRules();
    Iterable<UniqueByName> _plus_4 = Iterables.<UniqueByName>concat(_plus_3, _rules_1);
    EList<CheckRule> _checkrules = system.getCheckrules();
    final Iterable<UniqueByName> elements = Iterables.<UniqueByName>concat(_plus_4, _checkrules);
    this.<UniqueByName>checkDuplicatesByName(elements, null, XsemanticsValidator.DUPLICATE_NAME);
    final Function1<JudgmentDescription, String> _function_4 = (JudgmentDescription it) -> {
      return this.judgmentRepresentation(it.getJudgmentSymbol(), it.getRelationSymbols());
    };
    final Function2<String, JudgmentDescription, String> _function_5 = (String key, JudgmentDescription it) -> {
      String _name = it.eClass().getName();
      String _plus_5 = ((("Duplicate judgment symbols \'" + key) + "\' (") + _name);
      return (_plus_5 + ")");
    };
    this.<JudgmentDescription, String>checkDuplicates(system.getJudgmentDescriptions(), 
      XsemanticsPackage.Literals.JUDGMENT_DESCRIPTION__JUDGMENT_SYMBOL, 
      XsemanticsValidator.DUPLICATE_JUDGMENT_DESCRIPTION_SYMBOLS, _function_4, _function_5);
  }

  private <T extends UniqueByName> void checkDuplicatesByName(final Iterable<T> collection, final EStructuralFeature feature, final String issue) {
    final Function1<T, String> _function = (T it) -> {
      return it.getName();
    };
    final Function2<String, T, String> _function_1 = (String key, T it) -> {
      String _name = it.getName();
      String _plus = ("Duplicate name \'" + _name);
      String _plus_1 = (_plus + "\' (");
      String _name_1 = it.eClass().getName();
      String _plus_2 = (_plus_1 + _name_1);
      return (_plus_2 + ")");
    };
    this.<T, String>checkDuplicates(collection, feature, issue, _function, _function_1);
  }

  private <T extends EObject, K extends Object> void checkDuplicates(final Iterable<T> collection, final EStructuralFeature feature, final String issue, final Function1<? super T, ? extends K> keyComputer, final Function2<? super K, ? super T, ? extends String> errorMessageProvider) {
    boolean _isEmpty = IterableExtensions.isEmpty(collection);
    boolean _not = (!_isEmpty);
    if (_not) {
      final ListMultimap<K, T> map = this.<K, T>duplicatesMultimap();
      for (final T e : collection) {
        map.put(keyComputer.apply(e), e);
      }
      Set<Map.Entry<K, Collection<T>>> _entrySet = map.asMap().entrySet();
      for (final Map.Entry<K, Collection<T>> entry : _entrySet) {
        {
          final Collection<T> duplicates = entry.getValue();
          int _size = duplicates.size();
          boolean _greaterThan = (_size > 1);
          if (_greaterThan) {
            for (final T d : duplicates) {
              this.error(
                errorMessageProvider.apply(entry.getKey(), d), d, feature, issue);
            }
          }
        }
      }
    }
  }

  private <T extends Overrider> void checkOverrides(final Iterable<T> collection, final Iterable<T> superCollection, final Function2<? super T, ? super T, ? extends Boolean> conformanceComputer, final String kind) {
    if ((superCollection == null)) {
      final Function1<T, Boolean> _function = (T it) -> {
        return Boolean.valueOf(it.isOverride());
      };
      Iterable<T> _filter = IterableExtensions.<T>filter(collection, _function);
      for (final T j : _filter) {
        this.error(
          (("Cannot override " + kind) + " without system \'extends\'"), j, 
          null, 
          XsemanticsValidator.OVERRIDE_WITHOUT_SYSTEM_EXTENDS);
      }
    } else {
      final Function1<T, String> _function_1 = (T it) -> {
        return it.getName();
      };
      final Map<String, T> superMap = IterableExtensions.<String, T>toMap(superCollection, _function_1);
      for (final T j_1 : collection) {
        {
          final String name = j_1.getName();
          final T overridden = superMap.get(name);
          boolean _isOverride = j_1.isOverride();
          boolean _not = (!_isOverride);
          if (_not) {
            if ((overridden != null)) {
              String _reportContainingSystemName = this.reportContainingSystemName(overridden);
              String _plus = (((((kind + " \'") + name) + "\' must override ") + kind) + _reportContainingSystemName);
              this.error(_plus, j_1, 
                null, 
                XsemanticsValidator.MUST_OVERRIDE);
            }
          } else {
            if (((overridden == null) || (!(conformanceComputer.apply(j_1, overridden)).booleanValue()))) {
              this.error(((("No " + kind) + " to override: ") + name), j_1, 
                null, 
                XsemanticsValidator.NOTHING_TO_OVERRIDE);
            }
          }
        }
      }
    }
  }

  @Check
  public void checkAuxiliaryDescription(final AuxiliaryDescription aux) {
    boolean _isEmpty = aux.getParameters().isEmpty();
    if (_isEmpty) {
      this.error("No input parameter; at least one is needed", 
        XsemanticsPackage.Literals.AUXILIARY_DESCRIPTION__NAME, 
        XsemanticsValidator.NO_INPUT_PARAM);
    }
    final List<AuxiliaryFunction> functionsForAuxiliaryDescrition = this._xsemanticsUtils.functionsForAuxiliaryDescrition(aux);
    if (((this.isEnableWarnings() && functionsForAuxiliaryDescrition.isEmpty()) && (!aux.isOverride()))) {
      this.warning("No function defined for the auxiliary description", 
        XsemanticsPackage.Literals.AUXILIARY_DESCRIPTION.getEIDAttribute(), 
        XsemanticsValidator.NO_AUXFUN_FOR_AUX_DESCRIPTION);
    }
    int _size = functionsForAuxiliaryDescrition.size();
    boolean _greaterThan = (_size > 1);
    if (_greaterThan) {
      final Function1<AuxiliaryFunction, TupleType> _function = (AuxiliaryFunction it) -> {
        return this.typeSystem.getInputTypes(it);
      };
      final Function2<TupleType, AuxiliaryFunction, String> _function_1 = (TupleType key, AuxiliaryFunction it) -> {
        String _tupleTypeRepresentation = this.tupleTypeRepresentation(key);
        String _plus = ("Duplicate auxiliary function of the same kind with parameters: " + _tupleTypeRepresentation);
        String _reportContainingSystemName = this.reportContainingSystemName(it);
        return (_plus + _reportContainingSystemName);
      };
      this.<AuxiliaryFunction, TupleType>checkDuplicates(functionsForAuxiliaryDescrition, 
        XsemanticsPackage.Literals.AUXILIARY_FUNCTION__PARAMETERS, 
        XsemanticsValidator.DUPLICATE_AUXFUN_WITH_SAME_ARGUMENTS, _function, _function_1);
    }
  }

  @Check
  public void checkAuxiliaryFunctionHasAuxiliaryDescription(final AuxiliaryFunction aux) {
    final AuxiliaryDescription auxiliaryDescription = this._xsemanticsUtils.getAuxiliaryDescription(aux);
    if ((auxiliaryDescription == null)) {
      String _name = aux.getName();
      String _plus = ("No auxiliary description for auxiliary function \'" + _name);
      String _plus_1 = (_plus + "\'");
      this.error(_plus_1, 
        XsemanticsPackage.Literals.AUXILIARY_FUNCTION__NAME, 
        XsemanticsValidator.NO_AUXDESC_FOR_AUX_FUNCTION);
    } else {
      this.checkConformanceOfAuxiliaryFunction(aux, auxiliaryDescription);
    }
  }

  @Check
  public void checkOutputParamAccessWithinClosure(final XFeatureCall featureCall) {
    final JvmIdentifiableElement feature = featureCall.getFeature();
    if ((feature instanceof JvmFormalParameter)) {
      final EObject container = ((JvmFormalParameter)feature).eContainer();
      if ((container instanceof RuleParameter)) {
        if ((this._xsemanticsUtils.isOutputParam(((RuleParameter)container)) && this.insideClosure(featureCall))) {
          String _identifier = ((JvmFormalParameter)feature).getIdentifier();
          String _plus = ("Cannot refer to an output parameter " + _identifier);
          String _plus_1 = (_plus + " from within a closure");
          this.error(_plus_1, featureCall, null, 
            XsemanticsValidator.ACCESS_TO_OUTPUT_PARAM_WITHIN_CLOSURE);
        }
      }
      return;
    }
  }

  @Check
  protected void ensureNotPrimitive(final JvmTypeReference typeRef) {
    final LightweightTypeReference reference = this.toLightweightTypeReference(typeRef);
    boolean _isPrimitive = reference.isPrimitive();
    if (_isPrimitive) {
      final EObject container = typeRef.eContainer();
      if (((((container instanceof OutputParameter) || (container instanceof JvmFormalParameter)) || 
        (container instanceof Injected)) || (container instanceof AuxiliaryDescription))) {
        this.error("Primitives cannot be used in this context.", typeRef, null, 
          IssueCodes.INVALID_USE_OF_TYPE);
      }
    }
  }

  @Check
  public void checkFieldInitialization(final FieldDefinition f) {
    if (((!f.isWriteable()) && (f.getRight() == null))) {
      String _name = f.getName();
      String _plus = ("The final field " + _name);
      String _plus_1 = (_plus + " may not have been initialized");
      this.error(_plus_1, 
        XsemanticsPackage.Literals.ABSTRACT_FIELD_DEFINITION__NAME, 
        XsemanticsValidator.FINAL_FIELD_NOT_INITIALIZED);
    }
    if (((f.getType() == null) && (f.getRight() == null))) {
      String _name_1 = f.getName();
      String _plus_2 = ("The field " + _name_1);
      String _plus_3 = (_plus_2 + " needs an explicit type since there is no initialization expression to infer the type from.");
      this.error(_plus_3, f, XsemanticsPackage.Literals.ABSTRACT_FIELD_DEFINITION__NAME, 
        XsemanticsValidator.TOO_LITTLE_TYPE_INFORMATION);
    }
  }

  private boolean insideClosure(final XFeatureCall featureCall) {
    XClosure _containerOfType = EcoreUtil2.<XClosure>getContainerOfType(featureCall, XClosure.class);
    return (_containerOfType != null);
  }

  protected void checkConformanceOfAuxiliaryFunction(final AuxiliaryFunction aux, final AuxiliaryDescription auxiliaryDescription) {
    final EList<JvmFormalParameter> funParams = aux.getParameters();
    final EList<JvmFormalParameter> descParams = auxiliaryDescription.getParameters();
    int _size = funParams.size();
    int _size_1 = descParams.size();
    boolean _notEquals = (_size != _size_1);
    if (_notEquals) {
      int _size_2 = descParams.size();
      String _plus = ("expected " + Integer.valueOf(_size_2));
      String _plus_1 = (_plus + " parameter(s), but was ");
      int _size_3 = funParams.size();
      String _plus_2 = (_plus_1 + Integer.valueOf(_size_3));
      this.error(_plus_2, aux, 
        XsemanticsPackage.Literals.AUXILIARY_FUNCTION__PARAMETERS, 
        XsemanticsValidator.PARAMS_SIZE_DONT_MATCH);
    } else {
      final Iterator<JvmFormalParameter> funParamsIt = funParams.iterator();
      for (final JvmFormalParameter jvmFormalParameter : descParams) {
        {
          final JvmTypeReference expected = this.typeSystem.getType(jvmFormalParameter);
          final JvmFormalParameter funParam = funParamsIt.next();
          final JvmTypeReference actual = this.typeSystem.getType(funParam);
          boolean _isConformant = this.typeSystem.isConformant(expected, actual, funParam);
          boolean _not = (!_isConformant);
          if (_not) {
            Object _nameOfTypes = this.getNameOfTypes(actual);
            String _plus_3 = ("parameter type " + _nameOfTypes);
            String _plus_4 = (_plus_3 + " is not subtype of AuxiliaryDescription declared type ");
            Object _nameOfTypes_1 = this.getNameOfTypes(expected);
            String _plus_5 = (_plus_4 + _nameOfTypes_1);
            this.error(_plus_5, funParam, 
              TypesPackage.Literals.JVM_FORMAL_PARAMETER__PARAMETER_TYPE, 
              XsemanticsValidator.NOT_SUBTYPE);
          }
        }
      }
    }
  }

  protected String reportContainingSystemName(final EObject object) {
    String _name = this._xsemanticsUtils.containingSystem(object).getName();
    return (", in system: " + _name);
  }

  private void checkRuleConformantToJudgmentDescription(final Rule rule, final JudgmentDescription judgmentDescription) {
    final RuleConclusion conclusion = rule.getConclusion();
    this.checkConformanceAgainstJudgmentDescription(judgmentDescription, conclusion, 
      conclusion.getJudgmentSymbol(), 
      conclusion.getRelationSymbols(), 
      conclusion.getConclusionElements(), "Rule conclusion", 
      XsemanticsPackage.Literals.RULE__CONCLUSION, 
      XsemanticsPackage.Literals.RULE_CONCLUSION_ELEMENT.getEIDAttribute());
  }

  private JudgmentDescription checkRuleInvocationConformantToJudgmentDescription(final RuleInvocation ruleInvocation) {
    return this.checkConformanceAgainstJudgmentDescription(ruleInvocation, 
      ruleInvocation.getJudgmentSymbol(), 
      ruleInvocation.getRelationSymbols(), 
      ruleInvocation.getExpressions(), 
      "Rule invocation", 
      XsemanticsPackage.Literals.RULE_INVOCATION.getEIDAttribute(), 
      null);
  }

  private JudgmentDescription checkConformanceAgainstJudgmentDescription(final ReferToJudgment element, final String judgmentSymbol, final Iterable<String> relationSymbols, final Iterable<? extends EObject> elements, final String elementDescription, final EStructuralFeature elementFeature, final EStructuralFeature conformanceFeature) {
    final JudgmentDescription judgmentDescription = this.findJudgmentDescriptionOrError(element, judgmentSymbol, relationSymbols, elementFeature);
    this.checkConformanceAgainstJudgmentDescription(judgmentDescription, element, judgmentSymbol, relationSymbols, elements, elementDescription, elementFeature, conformanceFeature);
    return judgmentDescription;
  }

  private JudgmentDescription findJudgmentDescriptionOrError(final ReferToJudgment element, final String judgmentSymbol, final Iterable<String> relationSymbols, final EStructuralFeature elementFeature) {
    final JudgmentDescription judgmentDescription = this._xsemanticsUtils.getJudgmentDescription(element);
    if ((judgmentDescription == null)) {
      String _judgmentRepresentation = this.judgmentRepresentation(judgmentSymbol, relationSymbols);
      String _plus = ("No Judgment description for: " + _judgmentRepresentation);
      this.error(_plus, elementFeature, XsemanticsValidator.NO_JUDGMENT_DESCRIPTION);
    }
    return judgmentDescription;
  }

  private void checkConformanceAgainstJudgmentDescription(final JudgmentDescription judgmentDescription, final EObject element, final String judgmentSymbol, final Iterable<String> relationSymbols, final Iterable<? extends EObject> elements, final String elementDescription, final EStructuralFeature elementFeature, final EStructuralFeature conformanceFeature) {
    if ((judgmentDescription != null)) {
      final EList<JudgmentParameter> judgmentParameters = judgmentDescription.getJudgmentParameters();
      final Iterator<? extends EObject> elementsIt = elements.iterator();
      for (final JudgmentParameter judgmentParameter : judgmentParameters) {
        boolean _hasNext = elementsIt.hasNext();
        if (_hasNext) {
          this.checkConformance(judgmentParameter, elementsIt.next(), elementDescription, conformanceFeature);
        }
      }
    }
  }

  protected void checkConformance(final JudgmentParameter judgmentParameter, final EObject element, final String elementDescription, final EStructuralFeature feature) {
    final JvmTypeReference expected = this.typeSystem.getType(judgmentParameter);
    final JvmTypeReference actual = this.typeSystem.getType(element);
    boolean _isConformant = this.typeSystem.isConformant(expected, actual, element);
    boolean _not = (!_isConformant);
    if (_not) {
      Object _nameOfTypes = this.getNameOfTypes(actual);
      String _plus = ((elementDescription + " type ") + _nameOfTypes);
      String _plus_1 = (_plus + " is not subtype of JudgmentDescription declared type ");
      Object _nameOfTypes_1 = this.getNameOfTypes(expected);
      String _plus_2 = (_plus_1 + _nameOfTypes_1);
      this.error(_plus_2, element, feature, 
        XsemanticsValidator.NOT_SUBTYPE);
    }
  }

  protected String judgmentRepresentation(final String judgmentSymbol, final Iterable<String> relationSymbols) {
    String _join = IterableExtensions.join(relationSymbols, " ");
    return ((judgmentSymbol + " ") + _join);
  }

  protected String tupleTypeRepresentation(final TupleType tupleType) {
    final StringBuilder builder = new StringBuilder();
    final Iterator<JvmTypeReference> it = tupleType.iterator();
    while (it.hasNext()) {
      {
        builder.append(this.getNameOfTypes(it.next()));
        boolean _hasNext = it.hasNext();
        if (_hasNext) {
          builder.append(", ");
        }
      }
    }
    return builder.toString();
  }

  private Object getNameOfTypes(final JvmTypeReference typeRef) {
    String _xifexpression = null;
    if ((typeRef == null)) {
      _xifexpression = "<null>";
    } else {
      _xifexpression = typeRef.getSimpleName();
    }
    return _xifexpression;
  }

  public boolean isEnableWarnings() {
    return this.enableWarnings;
  }

  public void setEnableWarnings(final boolean enableWarnings) {
    this.enableWarnings = enableWarnings;
  }

  private <K extends Object, T extends Object> ListMultimap<K, T> duplicatesMultimap() {
    return Multimaps2.<K, T>newLinkedHashListMultimap();
  }
}
