/**
 * Copyright (c) 2013-2017 Lorenzo Bettini.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *   Lorenzo Bettini - Initial contribution and API
 */
package org.eclipse.xsemantics.dsl.util;

import com.google.common.base.Objects;
import com.google.common.collect.Iterables;
import com.google.common.collect.Lists;
import com.google.common.collect.Sets;
import com.google.inject.Inject;
import com.google.inject.Provider;
import com.google.inject.Singleton;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;
import org.eclipse.emf.common.util.EList;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.xsemantics.dsl.xsemantics.AuxiliaryDescription;
import org.eclipse.xsemantics.dsl.xsemantics.AuxiliaryFunction;
import org.eclipse.xsemantics.dsl.xsemantics.Cachable;
import org.eclipse.xsemantics.dsl.xsemantics.CachedClause;
import org.eclipse.xsemantics.dsl.xsemantics.CheckRule;
import org.eclipse.xsemantics.dsl.xsemantics.EntryPointsOption;
import org.eclipse.xsemantics.dsl.xsemantics.ExpressionInConclusion;
import org.eclipse.xsemantics.dsl.xsemantics.InputParameter;
import org.eclipse.xsemantics.dsl.xsemantics.JudgmentDescription;
import org.eclipse.xsemantics.dsl.xsemantics.JudgmentParameter;
import org.eclipse.xsemantics.dsl.xsemantics.OrExpression;
import org.eclipse.xsemantics.dsl.xsemantics.OutputParameter;
import org.eclipse.xsemantics.dsl.xsemantics.ReferToJudgment;
import org.eclipse.xsemantics.dsl.xsemantics.Rule;
import org.eclipse.xsemantics.dsl.xsemantics.RuleConclusionElement;
import org.eclipse.xsemantics.dsl.xsemantics.RuleInvocation;
import org.eclipse.xsemantics.dsl.xsemantics.RuleParameter;
import org.eclipse.xsemantics.dsl.xsemantics.XsemanticsFile;
import org.eclipse.xsemantics.dsl.xsemantics.XsemanticsSystem;
import org.eclipse.xtext.EcoreUtil2;
import org.eclipse.xtext.common.types.JvmFormalParameter;
import org.eclipse.xtext.common.types.JvmIdentifiableElement;
import org.eclipse.xtext.common.types.JvmParameterizedTypeReference;
import org.eclipse.xtext.common.types.JvmTypeReference;
import org.eclipse.xtext.xbase.XExpression;
import org.eclipse.xtext.xbase.XFeatureCall;
import org.eclipse.xtext.xbase.XVariableDeclaration;
import org.eclipse.xtext.xbase.jvmmodel.IJvmModelAssociations;
import org.eclipse.xtext.xbase.lib.Functions.Function1;
import org.eclipse.xtext.xbase.lib.IterableExtensions;
import org.eclipse.xtext.xbase.lib.ListExtensions;
import org.eclipse.xtext.xbase.lib.ObjectExtensions;
import org.eclipse.xtext.xbase.lib.Pair;
import org.eclipse.xtext.xbase.lib.Procedures.Procedure1;

@Singleton
@SuppressWarnings("all")
public class XsemanticsUtils {
  @Inject
  private IJvmModelAssociations associations;

  @Inject
  private XsemanticsDslCache cache;

  public XsemanticsSystem containingSystem(final EObject element) {
    EObject _head = IterableExtensions.<EObject>head(element.eResource().getContents());
    return ((XsemanticsFile) _head).getXsemanticsSystem();
  }

  public Rule containingRule(final EObject element) {
    return EcoreUtil2.<Rule>getContainerOfType(element, Rule.class);
  }

  public OrExpression containingOrExpression(final EObject element) {
    return EcoreUtil2.<OrExpression>getContainerOfType(element, OrExpression.class);
  }

  public JudgmentDescription getJudgmentDescription(final ReferToJudgment e) {
    boolean _isSet_judgment = e.isSet_judgment();
    boolean _not = (!_isSet_judgment);
    if (_not) {
      e.set_judgment(this.judgmentDescription(e, e.getJudgmentSymbol(), 
        e.getRelationSymbols()));
    }
    return e.get_judgment();
  }

  private JudgmentDescription judgmentDescription(final EObject object, final String judgmentSymbol, final Iterable<String> relationSymbols) {
    Map<Pair<String, EList<String>>, JudgmentDescription> _allJudgmentsRepresentations = this.allJudgmentsRepresentations(this.containingSystem(object));
    Pair<String, Iterable<String>> _mappedTo = Pair.<String, Iterable<String>>of(judgmentSymbol, relationSymbols);
    return _allJudgmentsRepresentations.get(_mappedTo);
  }

  public AuxiliaryDescription getAuxiliaryDescription(final AuxiliaryFunction fun) {
    AuxiliaryDescription _xblockexpression = null;
    {
      boolean _isSet_auxiliaryDescription = fun.isSet_auxiliaryDescription();
      boolean _not = (!_isSet_auxiliaryDescription);
      if (_not) {
        fun.set_auxiliaryDescription(this.auxiliaryDescriptionsByName(this.containingSystem(fun)).get(fun.getName()));
      }
      _xblockexpression = fun.get_auxiliaryDescription();
    }
    return _xblockexpression;
  }

  public Iterable<Rule> rulesForJudgmentDescription(final JudgmentDescription judgmentDescription) {
    return this.filterRulesByJudgmentDescription(this.containingSystem(judgmentDescription), judgmentDescription.getJudgmentSymbol(), judgmentDescription.getRelationSymbols());
  }

  public List<AuxiliaryFunction> functionsForAuxiliaryDescrition(final AuxiliaryDescription aux) {
    final Function1<AuxiliaryFunction, Boolean> _function = (AuxiliaryFunction it) -> {
      String _name = it.getName();
      String _name_1 = aux.getName();
      return Boolean.valueOf(Objects.equal(_name, _name_1));
    };
    return Lists.<AuxiliaryFunction>newArrayList(
      IterableExtensions.<AuxiliaryFunction>filter(this.containingSystem(aux).getAuxiliaryFunctions(), _function));
  }

  public Iterable<Rule> filterRulesByJudgmentDescription(final XsemanticsSystem ts, final String judgmentSymbol, final Iterable<String> relationSymbols) {
    return this.filterRulesByJudgmentDescription(ts.getRules(), judgmentSymbol, relationSymbols);
  }

  public Iterable<Rule> filterRulesByJudgmentDescription(final Iterable<Rule> rules, final String judgmentSymbol, final Iterable<String> relationSymbols) {
    final Function1<Rule, Boolean> _function = (Rule it) -> {
      return Boolean.valueOf((it.getConclusion().getJudgmentSymbol().equals(judgmentSymbol) && 
        IterableExtensions.elementsEqual(it.getConclusion().getRelationSymbols(), relationSymbols)));
    };
    return IterableExtensions.<Rule>filter(rules, _function);
  }

  public boolean isOutputParameter(final JudgmentParameter j) {
    return (j instanceof OutputParameter);
  }

  public List<OutputParameter> outputJudgmentParameters(final JudgmentDescription judgmentDescription) {
    return EcoreUtil2.<OutputParameter>typeSelect(judgmentDescription.getJudgmentParameters(), OutputParameter.class);
  }

  public List<RuleConclusionElement> outputConclusionElements(final Rule rule) {
    ArrayList<RuleConclusionElement> _xblockexpression = null;
    {
      final Iterator<JudgmentParameter> judgmentParameters = this.getJudgmentDescription(rule).getJudgmentParameters().iterator();
      final Function1<RuleConclusionElement, Boolean> _function = (RuleConclusionElement it) -> {
        return Boolean.valueOf(this.isOutputParameter(judgmentParameters.next()));
      };
      _xblockexpression = Lists.<RuleConclusionElement>newArrayList(
        IterableExtensions.<RuleConclusionElement>filter(rule.getConclusion().getConclusionElements(), _function));
    }
    return _xblockexpression;
  }

  public List<OutputParameter> outputParams(final RuleInvocation ruleInvocation) {
    return this.outputJudgmentParameters(this.getJudgmentDescription(ruleInvocation));
  }

  public List<InputParameter> inputParams(final JudgmentDescription judgmentDescription) {
    return EcoreUtil2.<InputParameter>typeSelect(judgmentDescription.getJudgmentParameters(), InputParameter.class);
  }

  public EList<RuleParameter> inputParams(final Rule rule) {
    EList<RuleParameter> _xblockexpression = null;
    {
      boolean _isSet_inputParams = rule.isSet_inputParams();
      boolean _not = (!_isSet_inputParams);
      if (_not) {
        final Iterator<JudgmentParameter> judgmentParameters = this.getJudgmentDescription(rule).getJudgmentParameters().iterator();
        EList<RuleParameter> __inputParams = rule.get_inputParams();
        final Function1<RuleConclusionElement, Boolean> _function = (RuleConclusionElement it) -> {
          boolean _isOutputParameter = this.isOutputParameter(judgmentParameters.next());
          return Boolean.valueOf((!_isOutputParameter));
        };
        List<RuleParameter> _list = IterableExtensions.<RuleParameter>toList(Iterables.<RuleParameter>filter(IterableExtensions.<RuleConclusionElement>filter(rule.getConclusion().getConclusionElements(), _function), RuleParameter.class));
        Iterables.<RuleParameter>addAll(__inputParams, _list);
      }
      _xblockexpression = rule.get_inputParams();
    }
    return _xblockexpression;
  }

  public boolean isInputParam(final RuleParameter ruleParameter) {
    return this.inputParams(this.containingRule(ruleParameter)).contains(ruleParameter);
  }

  public boolean isOutputParam(final RuleParameter ruleParameter) {
    return this.outputParams(this.containingRule(ruleParameter)).contains(ruleParameter);
  }

  public boolean isInputParam(final JvmFormalParameter jvmFormalParameter) {
    boolean _xblockexpression = false;
    {
      final RuleParameter ruleParameter = EcoreUtil2.<RuleParameter>getContainerOfType(jvmFormalParameter, RuleParameter.class);
      boolean _xifexpression = false;
      if ((ruleParameter != null)) {
        _xifexpression = this.isInputParam(ruleParameter);
      } else {
        _xifexpression = true;
      }
      _xblockexpression = _xifexpression;
    }
    return _xblockexpression;
  }

  public List<RuleParameter> outputParams(final Rule rule) {
    EList<RuleParameter> _xblockexpression = null;
    {
      boolean _isSet_outputParams = rule.isSet_outputParams();
      boolean _not = (!_isSet_outputParams);
      if (_not) {
        final JudgmentDescription judgmentDescription = this.getJudgmentDescription(rule);
        if (((judgmentDescription == null) || judgmentDescription.getJudgmentParameters().isEmpty())) {
          return Lists.<RuleParameter>newArrayList();
        }
        final Iterator<JudgmentParameter> judgmentParameters = judgmentDescription.getJudgmentParameters().iterator();
        EList<RuleParameter> __outputParams = rule.get_outputParams();
        final Function1<RuleConclusionElement, Boolean> _function = (RuleConclusionElement it) -> {
          return Boolean.valueOf(this.isOutputParameter(judgmentParameters.next()));
        };
        List<RuleParameter> _list = IterableExtensions.<RuleParameter>toList(Iterables.<RuleParameter>filter(IterableExtensions.<RuleConclusionElement>filter(rule.getConclusion().getConclusionElements(), _function), RuleParameter.class));
        Iterables.<RuleParameter>addAll(__outputParams, _list);
      }
      _xblockexpression = rule.get_outputParams();
    }
    return _xblockexpression;
  }

  public boolean hasOutputParams(final RuleInvocation ruleInvocation) {
    boolean _isEmpty = this.outputParams(ruleInvocation).isEmpty();
    return (!_isEmpty);
  }

  public List<ExpressionInConclusion> expressionsInConclusion(final Rule rule) {
    return EcoreUtil2.<ExpressionInConclusion>getAllContentsOfType(rule.getConclusion(), ExpressionInConclusion.class);
  }

  public ArrayList<XExpression> outputArgsExpressions(final RuleInvocation ruleInvocation) {
    ArrayList<XExpression> _xblockexpression = null;
    {
      final Iterator<JudgmentParameter> judgmentParameters = this.getJudgmentDescription(ruleInvocation).getJudgmentParameters().iterator();
      final Function1<XExpression, Boolean> _function = (XExpression it) -> {
        return Boolean.valueOf(this.isOutputParameter(judgmentParameters.next()));
      };
      _xblockexpression = Lists.<XExpression>newArrayList(
        IterableExtensions.<XExpression>filter(ruleInvocation.getExpressions(), _function));
    }
    return _xblockexpression;
  }

  public boolean validOutputArgExpression(final XExpression xexp) {
    if ((xexp instanceof XFeatureCall)) {
      final JvmIdentifiableElement feature = ((XFeatureCall)xexp).getFeature();
      if ((feature instanceof JvmFormalParameter)) {
        boolean _isInputParam = this.isInputParam(((JvmFormalParameter)feature));
        return (!_isInputParam);
      }
      return true;
    } else {
      if ((xexp instanceof XVariableDeclaration)) {
        return (((XVariableDeclaration)xexp).isWriteable() && (((XVariableDeclaration)xexp).getRight() == null));
      }
    }
    return false;
  }

  public boolean validInputArgExpression(final XExpression ruleInvocationExpression) {
    return (!(ruleInvocationExpression instanceof XVariableDeclaration));
  }

  public ArrayList<XExpression> inputArgsExpressions(final RuleInvocation ruleInvocation) {
    ArrayList<XExpression> _xblockexpression = null;
    {
      final Iterator<JudgmentParameter> judgmentParameters = this.getJudgmentDescription(ruleInvocation).getJudgmentParameters().iterator();
      final Function1<XExpression, Boolean> _function = (XExpression it) -> {
        boolean _isOutputParameter = this.isOutputParameter(judgmentParameters.next());
        return Boolean.valueOf((!_isOutputParameter));
      };
      _xblockexpression = Lists.<XExpression>newArrayList(
        IterableExtensions.<XExpression>filter(ruleInvocation.getExpressions(), _function));
    }
    return _xblockexpression;
  }

  public Iterable<JudgmentDescription> allJudgments(final XsemanticsSystem system) {
    Pair<String, EObject> _mappedTo = Pair.<String, EObject>of("allJudgments", system);
    final Provider<Iterable<JudgmentDescription>> _function = () -> {
      EList<JudgmentDescription> _judgmentDescriptions = system.getJudgmentDescriptions();
      final Function1<XsemanticsSystem, EList<JudgmentDescription>> _function_1 = (XsemanticsSystem it) -> {
        return it.getJudgmentDescriptions();
      };
      Iterable<JudgmentDescription> _flatten = Iterables.<JudgmentDescription>concat(ListExtensions.<XsemanticsSystem, EList<JudgmentDescription>>map(this.allSuperSystemDefinitions(system), _function_1));
      return Iterables.<JudgmentDescription>concat(_judgmentDescriptions, _flatten);
    };
    return this.cache.<Iterable<JudgmentDescription>>get(_mappedTo, _function);
  }

  public Iterable<Rule> allRules(final XsemanticsSystem system) {
    Pair<String, EObject> _mappedTo = Pair.<String, EObject>of("allRules", system);
    final Provider<Iterable<Rule>> _function = () -> {
      EList<Rule> _rules = system.getRules();
      final Function1<XsemanticsSystem, EList<Rule>> _function_1 = (XsemanticsSystem it) -> {
        return it.getRules();
      };
      Iterable<Rule> _flatten = Iterables.<Rule>concat(ListExtensions.<XsemanticsSystem, EList<Rule>>map(this.allSuperSystemDefinitions(system), _function_1));
      return Iterables.<Rule>concat(_rules, _flatten);
    };
    return this.cache.<Iterable<Rule>>get(_mappedTo, _function);
  }

  public List<Rule> allRulesByJudgmentDescription(final XsemanticsSystem ts, final String judgmentSymbol, final Iterable<String> relationSymbols) {
    Pair<String, Iterable<String>> _mappedTo = Pair.<String, Iterable<String>>of(judgmentSymbol, relationSymbols);
    Pair<XsemanticsSystem, Pair<String, Iterable<String>>> _mappedTo_1 = Pair.<XsemanticsSystem, Pair<String, Iterable<String>>>of(ts, _mappedTo);
    Pair<String, Pair<? extends EObject, ?>> _mappedTo_2 = Pair.<String, Pair<? extends EObject, ?>>of("allRulesByJudgmentDescription", _mappedTo_1);
    final Provider<List<Rule>> _function = () -> {
      return IterableExtensions.<Rule>toList(this.filterRulesByJudgmentDescription(this.allRules(ts), judgmentSymbol, relationSymbols));
    };
    return this.cache.<List<Rule>>get2(_mappedTo_2, _function);
  }

  public Iterable<CheckRule> allCheckRules(final XsemanticsSystem system) {
    Pair<String, EObject> _mappedTo = Pair.<String, EObject>of("allCheckRules", system);
    final Provider<Iterable<CheckRule>> _function = () -> {
      EList<CheckRule> _checkrules = system.getCheckrules();
      final Function1<XsemanticsSystem, EList<CheckRule>> _function_1 = (XsemanticsSystem it) -> {
        return it.getCheckrules();
      };
      Iterable<CheckRule> _flatten = Iterables.<CheckRule>concat(ListExtensions.<XsemanticsSystem, EList<CheckRule>>map(this.allSuperSystemDefinitions(system), _function_1));
      return Iterables.<CheckRule>concat(_checkrules, _flatten);
    };
    return this.cache.<Iterable<CheckRule>>get(_mappedTo, _function);
  }

  public Iterable<AuxiliaryDescription> allAuxiliaryDescriptions(final XsemanticsSystem system) {
    Pair<String, EObject> _mappedTo = Pair.<String, EObject>of("allAuxiliaryDescriptions", system);
    final Provider<Iterable<AuxiliaryDescription>> _function = () -> {
      EList<AuxiliaryDescription> _auxiliaryDescriptions = system.getAuxiliaryDescriptions();
      final Function1<XsemanticsSystem, EList<AuxiliaryDescription>> _function_1 = (XsemanticsSystem it) -> {
        return it.getAuxiliaryDescriptions();
      };
      Iterable<AuxiliaryDescription> _flatten = Iterables.<AuxiliaryDescription>concat(ListExtensions.<XsemanticsSystem, EList<AuxiliaryDescription>>map(this.allSuperSystemDefinitions(system), _function_1));
      return Iterables.<AuxiliaryDescription>concat(_auxiliaryDescriptions, _flatten);
    };
    return this.cache.<Iterable<AuxiliaryDescription>>get(_mappedTo, _function);
  }

  public List<XsemanticsSystem> allSuperSystemDefinitions(final XsemanticsSystem system) {
    Pair<String, EObject> _mappedTo = Pair.<String, EObject>of("allSuperSystemDefinitions", system);
    final Provider<List<XsemanticsSystem>> _function = () -> {
      return this.allSuperSystemDefinitionsInternal(system, Sets.<XsemanticsSystem>newHashSet());
    };
    return this.cache.<List<XsemanticsSystem>>get(_mappedTo, _function);
  }

  protected List<XsemanticsSystem> allSuperSystemDefinitionsInternal(final XsemanticsSystem system, final Set<XsemanticsSystem> visited) {
    ArrayList<XsemanticsSystem> _xblockexpression = null;
    {
      boolean _contains = visited.contains(system);
      if (_contains) {
        return Lists.<XsemanticsSystem>newArrayList();
      }
      visited.add(system);
      ArrayList<XsemanticsSystem> _newArrayList = Lists.<XsemanticsSystem>newArrayList();
      final Procedure1<ArrayList<XsemanticsSystem>> _function = (ArrayList<XsemanticsSystem> it) -> {
        final XsemanticsSystem superS = this.superSystemDefinition(system);
        if ((superS != null)) {
          it.add(superS);
          List<XsemanticsSystem> _allSuperSystemDefinitionsInternal = this.allSuperSystemDefinitionsInternal(superS, visited);
          Iterables.<XsemanticsSystem>addAll(it, _allSuperSystemDefinitionsInternal);
        }
      };
      _xblockexpression = ObjectExtensions.<ArrayList<XsemanticsSystem>>operator_doubleArrow(_newArrayList, _function);
    }
    return _xblockexpression;
  }

  public XsemanticsSystem superSystemDefinition(final XsemanticsSystem system) {
    JvmParameterizedTypeReference _superSystem = system.getSuperSystem();
    XsemanticsSystem _originalSystemDefinition = null;
    if (_superSystem!=null) {
      _originalSystemDefinition=this.originalSystemDefinition(_superSystem);
    }
    return _originalSystemDefinition;
  }

  public XsemanticsSystem originalSystemDefinition(final JvmTypeReference typeReference) {
    return IterableExtensions.<XsemanticsSystem>head(Iterables.<XsemanticsSystem>filter(this.associations.getSourceElements(typeReference.getType()), XsemanticsSystem.class));
  }

  public Map<Pair<String, EList<String>>, JudgmentDescription> allJudgmentsRepresentations(final XsemanticsSystem system) {
    Pair<String, EObject> _mappedTo = Pair.<String, EObject>of("allJudgmentsRepresentations", system);
    final Provider<Map<Pair<String, EList<String>>, JudgmentDescription>> _function = () -> {
      final Function1<JudgmentDescription, Pair<String, EList<String>>> _function_1 = (JudgmentDescription it) -> {
        String _judgmentSymbol = it.getJudgmentSymbol();
        EList<String> _relationSymbols = it.getRelationSymbols();
        return Pair.<String, EList<String>>of(_judgmentSymbol, _relationSymbols);
      };
      return IterableExtensions.<Pair<String, EList<String>>, JudgmentDescription>toMap(this.allJudgments(system), _function_1);
    };
    return this.cache.<Map<Pair<String, EList<String>>, JudgmentDescription>>get(_mappedTo, _function);
  }

  public Map<String, AuxiliaryDescription> auxiliaryDescriptionsByName(final XsemanticsSystem system) {
    Pair<String, EObject> _mappedTo = Pair.<String, EObject>of("auxiliaryDescriptionsByName", system);
    final Provider<Map<String, AuxiliaryDescription>> _function = () -> {
      final Function1<AuxiliaryDescription, String> _function_1 = (AuxiliaryDescription it) -> {
        return it.getName();
      };
      return IterableExtensions.<String, AuxiliaryDescription>toMap(system.getAuxiliaryDescriptions(), _function_1);
    };
    return this.cache.<Map<String, AuxiliaryDescription>>get(_mappedTo, _function);
  }

  public boolean cacheEntryPointMethods(final Cachable c) {
    CachedClause _cachedClause = c.getCachedClause();
    EntryPointsOption _entryPointsOption = null;
    if (_cachedClause!=null) {
      _entryPointsOption=_cachedClause.getEntryPointsOption();
    }
    return Objects.equal(_entryPointsOption, EntryPointsOption.DEFAULT);
  }

  public XExpression cacheCondition(final Cachable c) {
    CachedClause _cachedClause = c.getCachedClause();
    XExpression _condition = null;
    if (_cachedClause!=null) {
      _condition=_cachedClause.getCondition();
    }
    return _condition;
  }
}
