/**
 * Copyright (c) 2013-2017 Lorenzo Bettini.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *   Lorenzo Bettini - Initial contribution and API
 */
package org.eclipse.xsemantics.dsl.typing;

import com.google.common.base.Objects;
import com.google.inject.Inject;
import java.util.Iterator;
import org.eclipse.emf.common.util.EList;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.EStructuralFeature;
import org.eclipse.xsemantics.dsl.util.XsemanticsUtils;
import org.eclipse.xsemantics.dsl.xsemantics.AuxiliaryDescription;
import org.eclipse.xsemantics.dsl.xsemantics.AuxiliaryFunction;
import org.eclipse.xsemantics.dsl.xsemantics.JudgmentDescription;
import org.eclipse.xsemantics.dsl.xsemantics.JudgmentParameter;
import org.eclipse.xsemantics.dsl.xsemantics.Rule;
import org.eclipse.xsemantics.dsl.xsemantics.RuleInvocation;
import org.eclipse.xsemantics.dsl.xsemantics.RuleParameter;
import org.eclipse.xsemantics.runtime.RuleFailedException;
import org.eclipse.xsemantics.runtime.XsemanticsRuntimeSystem;
import org.eclipse.xtext.common.types.JvmFormalParameter;
import org.eclipse.xtext.common.types.JvmTypeReference;
import org.eclipse.xtext.common.types.util.TypeReferences;
import org.eclipse.xtext.validation.AbstractDeclarativeValidator;
import org.eclipse.xtext.xbase.XBinaryOperation;
import org.eclipse.xtext.xbase.XBooleanLiteral;
import org.eclipse.xtext.xbase.XExpression;
import org.eclipse.xtext.xbase.XFeatureCall;
import org.eclipse.xtext.xbase.XInstanceOfExpression;
import org.eclipse.xtext.xbase.XMemberFeatureCall;
import org.eclipse.xtext.xbase.XUnaryOperation;
import org.eclipse.xtext.xbase.lib.Extension;
import org.eclipse.xtext.xbase.typesystem.references.LightweightTypeReference;
import org.eclipse.xtext.xbase.typesystem.references.StandardTypeReferenceOwner;
import org.eclipse.xtext.xbase.typesystem.util.CommonTypeComputationServices;

@SuppressWarnings("all")
public class XsemanticsTypeSystem {
  @Inject
  private CommonTypeComputationServices services;

  @Inject
  private TypeReferences typeReferences;

  @Inject
  private XsemanticsTypeSystemGen xsemanticsTypeSystemGen;

  @Inject
  @Extension
  private XsemanticsUtils _xsemanticsUtils;

  public JvmTypeReference getType(final EObject element) {
    return this.xsemanticsTypeSystemGen.type(element).getValue();
  }

  public TupleType getInputTypes(final Rule rule) {
    final TupleType tupleType = new TupleType();
    EList<RuleParameter> _inputParams = this._xsemanticsUtils.inputParams(rule);
    for (final RuleParameter p : _inputParams) {
      tupleType.add(this.getType(p));
    }
    return tupleType;
  }

  public TupleType getInputTypes(final AuxiliaryFunction aux) {
    final TupleType tupleType = new TupleType();
    EList<JvmFormalParameter> _parameters = aux.getParameters();
    for (final JvmFormalParameter p : _parameters) {
      tupleType.add(this.getType(p));
    }
    return tupleType;
  }

  public boolean equals(final TupleType tupleType1, final TupleType tupleType2) {
    return tupleType1.equals(tupleType2);
  }

  public boolean equals(final JudgmentDescription j1, final JudgmentDescription j2) {
    int _size = j1.getJudgmentParameters().size();
    int _size_1 = j2.getJudgmentParameters().size();
    boolean _notEquals = (_size != _size_1);
    if (_notEquals) {
      return false;
    }
    final Iterator<JudgmentParameter> judgmentParametersIt = j1.getJudgmentParameters().iterator();
    EList<JudgmentParameter> _judgmentParameters = j2.getJudgmentParameters();
    for (final JudgmentParameter jParam2 : _judgmentParameters) {
      {
        final JudgmentParameter jParam1 = judgmentParametersIt.next();
        if (((!Objects.equal(jParam1.eClass(), jParam2.eClass())) || 
          (!this.equals(this.getType(jParam1), this.getType(jParam2), jParam1)))) {
          return false;
        }
      }
    }
    return true;
  }

  public boolean equals(final AuxiliaryDescription aux1, final AuxiliaryDescription aux2) {
    int _size = aux1.getParameters().size();
    int _size_1 = aux2.getParameters().size();
    boolean _notEquals = (_size != _size_1);
    if (_notEquals) {
      return false;
    }
    boolean _equals = this.equals(aux1.getType(), aux2.getType(), aux1);
    boolean _not = (!_equals);
    if (_not) {
      return false;
    }
    final Iterator<JvmFormalParameter> paramIt = aux1.getParameters().iterator();
    EList<JvmFormalParameter> _parameters = aux2.getParameters();
    for (final JvmFormalParameter param2 : _parameters) {
      {
        final JvmFormalParameter param1 = paramIt.next();
        if (((!Objects.equal(param1.eClass(), param2.eClass())) || 
          (!this.equals(this.getType(param1), this.getType(param2), param1)))) {
          return false;
        }
      }
    }
    return true;
  }

  public boolean equals(final JvmTypeReference t1, final JvmTypeReference t2, final EObject context) {
    if ((t1 == null)) {
      return (t2 == null);
    }
    if ((t2 == null)) {
      return (t1 == null);
    }
    return t1.getType().equals(t2.getType());
  }

  public boolean isConformant(final JvmTypeReference expected, final JvmTypeReference actual, final EObject context) {
    return this.isSubtype(actual, expected, context);
  }

  public boolean isEObject(final JvmTypeReference type, final EObject context) {
    return this.isConformant(
      this.typeReferences.getTypeForName(EObject.class, context), type, context);
  }

  public boolean isAbstractDeclarativeValidator(final JvmTypeReference type, final EObject context) {
    return this.isConformant(
      this.typeReferences.getTypeForName(AbstractDeclarativeValidator.class, context), type, context);
  }

  public boolean isEStructuralFeature(final JvmTypeReference type, final EObject context) {
    return this.isConformant(
      this.typeReferences.getTypeForName(
        EStructuralFeature.class, context), type, context);
  }

  public boolean isValidSuperSystem(final JvmTypeReference type, final EObject context) {
    return this.isConformant(
      this.typeReferences.getTypeForName(
        XsemanticsRuntimeSystem.class, context), type, context);
  }

  public boolean isBooleanPremise(final XExpression expression) {
    boolean _isExpressionToCheck = this.isExpressionToCheck(expression);
    boolean _not = (!_isExpressionToCheck);
    if (_not) {
      return false;
    } else {
      final JvmTypeReference booleanType = this.typeReferences.getTypeForName(Boolean.TYPE, expression);
      final JvmTypeReference operationType = this.getType(expression);
      return this.isSubtype(operationType, booleanType, expression);
    }
  }

  public boolean isExpressionToCheck(final XExpression expression) {
    return ((((((expression instanceof XFeatureCall) || (expression instanceof XBinaryOperation)) || (expression instanceof XUnaryOperation)) || (expression instanceof XMemberFeatureCall)) || (expression instanceof XBooleanLiteral)) || (expression instanceof XInstanceOfExpression));
  }

  public boolean isSubtype(final JvmTypeReference t1, final JvmTypeReference t2, final EObject context) {
    boolean _xblockexpression = false;
    {
      if (((t1 == null) || (t2 == null))) {
        return false;
      }
      final LightweightTypeReference type1 = this.toLightweightTypeReference(t1, context);
      final LightweightTypeReference type2 = this.toLightweightTypeReference(t2, context);
      _xblockexpression = type2.isAssignableFrom(type1);
    }
    return _xblockexpression;
  }

  public LightweightTypeReference toLightweightTypeReference(final JvmTypeReference typeRef, final EObject context) {
    return this.newTypeReferenceOwner(context).toLightweightTypeReference(typeRef);
  }

  protected StandardTypeReferenceOwner newTypeReferenceOwner(final EObject context) {
    return new StandardTypeReferenceOwner(this.services, context);
  }

  public boolean isPredicate(final JudgmentDescription j) {
    boolean _xblockexpression = false;
    {
      if ((j == null)) {
        return false;
      }
      _xblockexpression = this._xsemanticsUtils.outputJudgmentParameters(j).isEmpty();
    }
    return _xblockexpression;
  }

  public boolean isPredicate(final RuleInvocation ruleInvocation) {
    return this.isPredicate(this._xsemanticsUtils.getJudgmentDescription(ruleInvocation));
  }

  public JvmTypeReference ruleFailedExceptionType(final EObject o) {
    return this.typeReferences.getTypeForName(RuleFailedException.class, o);
  }
}
