/**
 * Copyright (c) 2013-2017 Lorenzo Bettini.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *   Lorenzo Bettini - Initial contribution and API
 */
package org.eclipse.xsemantics.dsl.typing;

import com.google.inject.Inject;
import com.google.inject.Singleton;
import java.util.Arrays;
import java.util.List;
import org.eclipse.emf.common.util.EList;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.EStructuralFeature;
import org.eclipse.xsemantics.dsl.XsemanticsConstants;
import org.eclipse.xsemantics.dsl.util.XsemanticsUtils;
import org.eclipse.xsemantics.dsl.xsemantics.AuxiliaryDescription;
import org.eclipse.xsemantics.dsl.xsemantics.AuxiliaryFunction;
import org.eclipse.xsemantics.dsl.xsemantics.CheckRule;
import org.eclipse.xsemantics.dsl.xsemantics.EmptyEnvironment;
import org.eclipse.xsemantics.dsl.xsemantics.EnvironmentComposition;
import org.eclipse.xsemantics.dsl.xsemantics.EnvironmentMapping;
import org.eclipse.xsemantics.dsl.xsemantics.ErrorSpecification;
import org.eclipse.xsemantics.dsl.xsemantics.Fail;
import org.eclipse.xsemantics.dsl.xsemantics.OrExpression;
import org.eclipse.xsemantics.dsl.xsemantics.Rule;
import org.eclipse.xsemantics.dsl.xsemantics.RuleInvocation;
import org.eclipse.xsemantics.dsl.xsemantics.RuleParameter;
import org.eclipse.xsemantics.dsl.xsemantics.RuleWithPremises;
import org.eclipse.xsemantics.runtime.RuleEnvironment;
import org.eclipse.xtext.common.types.JvmTypeReference;
import org.eclipse.xtext.xbase.XBlockExpression;
import org.eclipse.xtext.xbase.XExpression;
import org.eclipse.xtext.xbase.XVariableDeclaration;
import org.eclipse.xtext.xbase.XbaseFactory;
import org.eclipse.xtext.xbase.annotations.typesystem.XbaseWithAnnotationsTypeComputer;
import org.eclipse.xtext.xbase.lib.Extension;
import org.eclipse.xtext.xbase.lib.IterableExtensions;
import org.eclipse.xtext.xbase.lib.ObjectExtensions;
import org.eclipse.xtext.xbase.lib.Procedures.Procedure1;
import org.eclipse.xtext.xbase.typesystem.computation.ITypeComputationState;
import org.eclipse.xtext.xbase.typesystem.computation.ITypeExpectation;
import org.eclipse.xtext.xbase.typesystem.conformance.ConformanceHint;
import org.eclipse.xtext.xbase.typesystem.references.AnyTypeReference;
import org.eclipse.xtext.xbase.typesystem.references.ITypeReferenceOwner;
import org.eclipse.xtext.xbase.typesystem.references.LightweightTypeReference;

/**
 * Custom version of type computer for Custom XExpressions
 * introduced by Xsemantics
 */
@Singleton
@SuppressWarnings("all")
public class XsemanticsTypeComputer extends XbaseWithAnnotationsTypeComputer {
  @Inject
  @Extension
  private XsemanticsUtils _xsemanticsUtils;

  @Inject
  @Extension
  private XsemanticsTypeSystem _xsemanticsTypeSystem;

  @Override
  public void computeTypes(final XExpression expression, final ITypeComputationState state) {
    boolean _matched = false;
    if (expression instanceof RuleInvocation) {
      _matched=true;
      this._computeTypes(((RuleInvocation)expression), state);
    }
    if (!_matched) {
      if (expression instanceof OrExpression) {
        _matched=true;
        this._computeTypes(((OrExpression)expression), state);
      }
    }
    if (!_matched) {
      if (expression instanceof Fail) {
        _matched=true;
        this._computeTypes(((Fail)expression), state);
      }
    }
    if (!_matched) {
      if (expression instanceof ErrorSpecification) {
        _matched=true;
        this._computeTypes(((ErrorSpecification)expression), state);
      }
    }
    if (!_matched) {
      super.computeTypes(expression, state);
    }
  }

  @Override
  public void _computeTypes(final XBlockExpression b, final ITypeComputationState typeState) {
    ITypeComputationState state = typeState;
    if (((b.eContainer() instanceof Rule) || 
      (b.eContainer() instanceof CheckRule))) {
      state = state.withoutRootExpectation();
    }
    EObject _eContainer = b.eContainer();
    if ((_eContainer instanceof AuxiliaryFunction)) {
      EObject _eContainer_1 = b.eContainer();
      final AuxiliaryFunction aux = ((AuxiliaryFunction) _eContainer_1);
      AuxiliaryDescription _auxiliaryDescription = this._xsemanticsUtils.getAuxiliaryDescription(aux);
      JvmTypeReference _type = null;
      if (_auxiliaryDescription!=null) {
        _type=_auxiliaryDescription.getType();
      }
      final JvmTypeReference type = _type;
      if ((type == null)) {
        state = state.withExpectation(this.getPrimitiveVoid(state));
      }
    }
    EObject _eContainer_2 = b.eContainer();
    if ((_eContainer_2 instanceof RuleWithPremises)) {
      EObject _eContainer_3 = b.eContainer();
      final RuleWithPremises rule = ((RuleWithPremises) _eContainer_3);
      List<RuleParameter> _outputParams = this._xsemanticsUtils.outputParams(rule);
      for (final RuleParameter outputParam : _outputParams) {
        state.addLocalToCurrentScope(outputParam.getParameter());
      }
    }
    List<? extends ITypeExpectation> _expectations = state.getExpectations();
    for (final ITypeExpectation expectation : _expectations) {
      {
        final LightweightTypeReference expectedType = expectation.getExpectedType();
        if (((expectedType != null) && expectedType.isPrimitiveVoid())) {
          final EList<XExpression> expressions = b.getExpressions();
          boolean _isEmpty = expressions.isEmpty();
          boolean _not = (!_isEmpty);
          if (_not) {
            for (final XExpression expression : expressions) {
              {
                final ITypeComputationState expressionState = state.withoutExpectation();
                expressionState.computeTypes(expression);
                this.addVariableDeclarationsToScope(expression, state);
              }
            }
          }
          expectation.acceptActualType(this.getPrimitiveVoid(state), ConformanceHint.CHECKED, ConformanceHint.SUCCESS);
        } else {
          final EList<XExpression> expressions_1 = b.getExpressions();
          boolean _isEmpty_1 = expressions_1.isEmpty();
          boolean _not_1 = (!_isEmpty_1);
          if (_not_1) {
            int _size = expressions_1.size();
            int _minus = (_size - 1);
            List<XExpression> _subList = expressions_1.subList(0, _minus);
            for (final XExpression expression_1 : _subList) {
              {
                final ITypeComputationState expressionState = state.withoutExpectation();
                expressionState.computeTypes(expression_1);
                this.addVariableDeclarationsToScope(expression_1, state);
              }
            }
            final XExpression lastExpression = IterableExtensions.<XExpression>last(expressions_1);
            state.computeTypes(lastExpression);
            this.addVariableDeclarationsToScope(lastExpression, state);
          } else {
            ITypeReferenceOwner _referenceOwner = expectation.getReferenceOwner();
            AnyTypeReference _anyTypeReference = new AnyTypeReference(_referenceOwner);
            expectation.acceptActualType(_anyTypeReference, ConformanceHint.UNCHECKED);
          }
        }
      }
    }
  }

  protected void addVariableDeclarationsToScope(final XExpression e, final ITypeComputationState state) {
    boolean _matched = false;
    if (e instanceof XVariableDeclaration) {
      _matched=true;
      this.addLocalToCurrentScope(((XVariableDeclaration)e), state);
    }
    if (!_matched) {
      if (e instanceof RuleInvocation) {
        _matched=true;
        EList<XExpression> _expressions = ((RuleInvocation)e).getExpressions();
        for (final XExpression exp : _expressions) {
          if ((exp instanceof XVariableDeclaration)) {
            this.addLocalToCurrentScope(((XVariableDeclaration)exp), state);
          }
        }
      }
    }
  }

  protected void _computeTypes(final RuleInvocation e, final ITypeComputationState state) {
    EList<XExpression> _expressions = e.getExpressions();
    for (final XExpression expression : _expressions) {
      {
        final ITypeComputationState expressionState = state.withoutExpectation();
        expressionState.computeTypes(expression);
      }
    }
    this.handleEnvironmentSpecification(e.getEnvironment(), 
      state.withExpectation(this.getTypeForName(RuleEnvironment.class, state)));
    if ((this.hasTypeExpectations(state) && this._xsemanticsTypeSystem.isPredicate(e))) {
      state.acceptActualType(this.getTypeForName(Boolean.TYPE, state));
    } else {
      state.acceptActualType(this.getPrimitiveVoid(state));
    }
  }

  protected boolean hasTypeExpectations(final ITypeComputationState state) {
    boolean _isEmpty = state.getExpectations().isEmpty();
    return (!_isEmpty);
  }

  protected void _computeTypes(final OrExpression e, final ITypeComputationState state) {
    boolean firstBranch = true;
    EList<XExpression> _branches = e.getBranches();
    for (final XExpression b : _branches) {
      {
        ITypeComputationState s = state.withoutExpectation();
        if (((!firstBranch) && (this._xsemanticsUtils.containingOrExpression(e.eContainer()) == null))) {
          XVariableDeclaration _createXVariableDeclaration = XbaseFactory.eINSTANCE.createXVariableDeclaration();
          final Procedure1<XVariableDeclaration> _function = (XVariableDeclaration it) -> {
            it.setWriteable(false);
            it.setName(XsemanticsConstants.PREVIOUS_FAILURE);
          };
          final XVariableDeclaration implicitVar = ObjectExtensions.<XVariableDeclaration>operator_doubleArrow(_createXVariableDeclaration, _function);
          EList<EObject> _contents = e.eResource().getContents();
          _contents.add(implicitVar);
          s = s.assignType(implicitVar, 
            this._xsemanticsTypeSystem.toLightweightTypeReference(this._xsemanticsTypeSystem.ruleFailedExceptionType(e), e));
        }
        firstBranch = false;
        this.computeTypes(b, s);
      }
    }
    state.acceptActualType(this.getPrimitiveVoid(state));
  }

  protected void _handleEnvironmentSpecification(final EmptyEnvironment e, final ITypeComputationState state) {
    state.acceptActualType(this.getTypeForName(RuleEnvironment.class, state));
  }

  protected void _handleEnvironmentSpecification(final EnvironmentComposition e, final ITypeComputationState state) {
    this.handleEnvironmentSpecification(e.getCurrentEnvironment(), state);
    this.handleEnvironmentSpecification(e.getSubEnvironment(), state);
  }

  protected void _handleEnvironmentSpecification(final EnvironmentMapping e, final ITypeComputationState state) {
    state.withNonVoidExpectation().computeTypes(e.getKey());
    state.withNonVoidExpectation().computeTypes(e.getValue());
  }

  protected void _handleEnvironmentSpecification(final XExpression e, final ITypeComputationState state) {
    state.computeTypes(e);
  }

  protected void _computeTypes(final Fail e, final ITypeComputationState state) {
    XExpression _error = e.getError();
    if (_error!=null) {
      this.computeTypes(_error, state);
    }
    state.acceptActualType(this.getPrimitiveVoid(state));
  }

  protected void _computeTypes(final ErrorSpecification e, final ITypeComputationState state) {
    XExpression _error = e.getError();
    boolean _tripleNotEquals = (_error != null);
    if (_tripleNotEquals) {
      state.withExpectation(this.getTypeForName(String.class, state)).computeTypes(e.getError());
    }
    XExpression _source = e.getSource();
    boolean _tripleNotEquals_1 = (_source != null);
    if (_tripleNotEquals_1) {
      state.withExpectation(this.getTypeForName(EObject.class, state)).computeTypes(e.getSource());
    }
    XExpression _feature = e.getFeature();
    boolean _tripleNotEquals_2 = (_feature != null);
    if (_tripleNotEquals_2) {
      state.withExpectation(this.getTypeForName(EStructuralFeature.class, state)).computeTypes(e.getFeature());
    }
    XExpression _data = e.getData();
    boolean _tripleNotEquals_3 = (_data != null);
    if (_tripleNotEquals_3) {
      state.withNonVoidExpectation().computeTypes(e.getData());
    }
    state.acceptActualType(this.getPrimitiveVoid(state));
  }

  protected void handleEnvironmentSpecification(final XExpression e, final ITypeComputationState state) {
    if (e instanceof EmptyEnvironment) {
      _handleEnvironmentSpecification((EmptyEnvironment)e, state);
      return;
    } else if (e instanceof EnvironmentComposition) {
      _handleEnvironmentSpecification((EnvironmentComposition)e, state);
      return;
    } else if (e instanceof EnvironmentMapping) {
      _handleEnvironmentSpecification((EnvironmentMapping)e, state);
      return;
    } else if (e != null) {
      _handleEnvironmentSpecification(e, state);
      return;
    } else {
      throw new IllegalArgumentException("Unhandled parameter types: " +
        Arrays.<Object>asList(e, state).toString());
    }
  }
}
