/**
 * Copyright (c) 2013-2017 Lorenzo Bettini.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *   Lorenzo Bettini - Initial contribution and API
 */
package org.eclipse.xsemantics.dsl.jvmmodel;

import com.google.common.collect.Iterables;
import com.google.inject.Inject;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import org.eclipse.emf.common.util.EList;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.xsemantics.dsl.generator.UniqueNames;
import org.eclipse.xsemantics.dsl.generator.XsemanticsGeneratorExtensions;
import org.eclipse.xsemantics.dsl.typing.XsemanticsTypeSystem;
import org.eclipse.xsemantics.dsl.util.XsemanticsUtils;
import org.eclipse.xsemantics.dsl.xsemantics.AbstractFieldDefinition;
import org.eclipse.xsemantics.dsl.xsemantics.AuxiliaryDescription;
import org.eclipse.xsemantics.dsl.xsemantics.AuxiliaryFunction;
import org.eclipse.xsemantics.dsl.xsemantics.Cachable;
import org.eclipse.xsemantics.dsl.xsemantics.CachedClause;
import org.eclipse.xsemantics.dsl.xsemantics.CheckRule;
import org.eclipse.xsemantics.dsl.xsemantics.ExpressionInConclusion;
import org.eclipse.xsemantics.dsl.xsemantics.FieldDefinition;
import org.eclipse.xsemantics.dsl.xsemantics.InputParameter;
import org.eclipse.xsemantics.dsl.xsemantics.JudgmentDescription;
import org.eclipse.xsemantics.dsl.xsemantics.Named;
import org.eclipse.xsemantics.dsl.xsemantics.Overrider;
import org.eclipse.xsemantics.dsl.xsemantics.Rule;
import org.eclipse.xsemantics.dsl.xsemantics.RuleParameter;
import org.eclipse.xsemantics.dsl.xsemantics.RuleWithPremises;
import org.eclipse.xsemantics.dsl.xsemantics.XsemanticsSystem;
import org.eclipse.xsemantics.runtime.ErrorInformation;
import org.eclipse.xsemantics.runtime.RuleApplicationTrace;
import org.eclipse.xsemantics.runtime.RuleEnvironment;
import org.eclipse.xsemantics.runtime.RuleFailedException;
import org.eclipse.xsemantics.runtime.XsemanticsProvider;
import org.eclipse.xsemantics.runtime.XsemanticsRuntimeSystem;
import org.eclipse.xsemantics.runtime.validation.XsemanticsValidatorErrorGenerator;
import org.eclipse.xtend2.lib.StringConcatenation;
import org.eclipse.xtend2.lib.StringConcatenationClient;
import org.eclipse.xtext.common.types.JvmAnnotationReference;
import org.eclipse.xtext.common.types.JvmAnnotationTarget;
import org.eclipse.xtext.common.types.JvmConstructor;
import org.eclipse.xtext.common.types.JvmDeclaredType;
import org.eclipse.xtext.common.types.JvmExecutable;
import org.eclipse.xtext.common.types.JvmField;
import org.eclipse.xtext.common.types.JvmFormalParameter;
import org.eclipse.xtext.common.types.JvmGenericType;
import org.eclipse.xtext.common.types.JvmMember;
import org.eclipse.xtext.common.types.JvmOperation;
import org.eclipse.xtext.common.types.JvmParameterizedTypeReference;
import org.eclipse.xtext.common.types.JvmType;
import org.eclipse.xtext.common.types.JvmTypeReference;
import org.eclipse.xtext.common.types.JvmVisibility;
import org.eclipse.xtext.common.types.util.TypeReferences;
import org.eclipse.xtext.naming.QualifiedName;
import org.eclipse.xtext.util.PolymorphicDispatcher;
import org.eclipse.xtext.validation.AbstractDeclarativeValidator;
import org.eclipse.xtext.validation.Check;
import org.eclipse.xtext.xbase.XExpression;
import org.eclipse.xtext.xbase.annotations.xAnnotations.XAnnotation;
import org.eclipse.xtext.xbase.compiler.output.ITreeAppendable;
import org.eclipse.xtext.xbase.jvmmodel.AbstractModelInferrer;
import org.eclipse.xtext.xbase.jvmmodel.IJvmDeclaredTypeAcceptor;
import org.eclipse.xtext.xbase.jvmmodel.JvmTypesBuilder;
import org.eclipse.xtext.xbase.lib.CollectionLiterals;
import org.eclipse.xtext.xbase.lib.Extension;
import org.eclipse.xtext.xbase.lib.Functions.Function1;
import org.eclipse.xtext.xbase.lib.IterableExtensions;
import org.eclipse.xtext.xbase.lib.ListExtensions;
import org.eclipse.xtext.xbase.lib.ObjectExtensions;
import org.eclipse.xtext.xbase.lib.Procedures.Procedure1;

/**
 * <p>Infers a JVM model from the source model.</p>
 * 
 * <p>The JVM model should contain all elements that would appear in the Java code
 * which is generated from the source model. Other models link against the JVM model rather than the source model.</p>
 */
@SuppressWarnings("all")
public class XsemanticsJvmModelInferrer extends AbstractModelInferrer {
  /**
   * convenience API to build and initialize JVM types and their members.
   */
  @Inject
  @Extension
  private JvmTypesBuilder _jvmTypesBuilder;

  @Inject
  @Extension
  private XsemanticsGeneratorExtensions _xsemanticsGeneratorExtensions;

  @Inject
  @Extension
  private XsemanticsUtils _xsemanticsUtils;

  @Inject
  @Extension
  private TypeReferences _typeReferences;

  @Inject
  @Extension
  private XsemanticsTypeSystem typeSystem;

  /**
   * The dispatch method {@code infer} is called for each instance of the
   * given element's type that is contained in a resource.
   * 
   * @param element
   *            the model to create one or more
   *            {@link JvmDeclaredType declared
   *            types} from.
   * @param acceptor
   *            each created
   *            {@link JvmDeclaredType type}
   *            without a container should be passed to the acceptor in order
   *            get attached to the current resource. The acceptor's
   *            accept(..) method takes the constructed empty type for the
   *            pre-indexing phase. This one is further initialized in the
   *            indexing phase using the lambda you pass to accept as the last argument.
   * @param isPreIndexingPhase
   *            whether the method is called in a pre-indexing phase, i.e.
   *            when the global index is not yet fully updated. You must not
   *            rely on linking using the index if isPreIndexingPhase is
   *            <code>true</code>.
   */
  protected void _infer(final XsemanticsSystem ts, final IJvmDeclaredTypeAcceptor acceptor, final boolean isPreIndexingPhase) {
    QualifiedName _javaFullyQualifiedName = this._xsemanticsGeneratorExtensions.toJavaFullyQualifiedName(ts);
    boolean _tripleEquals = (_javaFullyQualifiedName == null);
    if (_tripleEquals) {
      return;
    }
    final JvmGenericType inferredClass = this._jvmTypesBuilder.toClass(ts, this._xsemanticsGeneratorExtensions.toJavaFullyQualifiedName(ts));
    final Procedure1<JvmGenericType> _function = (JvmGenericType it) -> {
      String _copyright = ts.getCopyright();
      boolean _tripleNotEquals = (_copyright != null);
      if (_tripleNotEquals) {
        this._jvmTypesBuilder.setFileHeader(it, ts.getCopyright());
      }
      this._jvmTypesBuilder.setDocumentation(it, this._jvmTypesBuilder.getDocumentation(ts));
      JvmParameterizedTypeReference _superSystem = ts.getSuperSystem();
      boolean _tripleNotEquals_1 = (_superSystem != null);
      if (_tripleNotEquals_1) {
        EList<JvmTypeReference> _superTypes = it.getSuperTypes();
        JvmTypeReference _cloneWithProxies = this._jvmTypesBuilder.cloneWithProxies(ts.getSuperSystem());
        this._jvmTypesBuilder.<JvmTypeReference>operator_add(_superTypes, _cloneWithProxies);
      } else {
        EList<JvmTypeReference> _superTypes_1 = it.getSuperTypes();
        JvmTypeReference _typeRef = this._typeReferenceBuilder.typeRef(XsemanticsRuntimeSystem.class);
        this._jvmTypesBuilder.<JvmTypeReference>operator_add(_superTypes_1, _typeRef);
      }
      EList<AuxiliaryDescription> _auxiliaryDescriptions = ts.getAuxiliaryDescriptions();
      for (final AuxiliaryDescription elem : _auxiliaryDescriptions) {
        EList<JvmMember> _members = it.getMembers();
        JvmField _genIssueField = this.genIssueField(elem);
        this._jvmTypesBuilder.<JvmField>operator_add(_members, _genIssueField);
      }
      EList<Rule> _rules = ts.getRules();
      for (final Rule elem_1 : _rules) {
        EList<JvmMember> _members_1 = it.getMembers();
        JvmField _genIssueField_1 = this.genIssueField(elem_1);
        this._jvmTypesBuilder.<JvmField>operator_add(_members_1, _genIssueField_1);
      }
      EList<AbstractFieldDefinition> _fields = ts.getFields();
      for (final AbstractFieldDefinition field : _fields) {
        EList<JvmMember> _members_2 = it.getMembers();
        final Procedure1<JvmField> _function_1 = (JvmField it_1) -> {
          this._jvmTypesBuilder.setDocumentation(it_1, this._jvmTypesBuilder.getDocumentation(field));
          boolean _isExtension = field.isExtension();
          if (_isExtension) {
            this.addExtensionAnnotation(it_1);
          }
          it_1.setVisibility(JvmVisibility.PRIVATE);
          if ((field instanceof FieldDefinition)) {
            this.translateAnnotations(it_1, ((FieldDefinition)field).getAnnotations());
            this._jvmTypesBuilder.setInitializer(it_1, ((FieldDefinition)field).getRight());
            boolean _isWriteable = ((FieldDefinition)field).isWriteable();
            boolean _not = (!_isWriteable);
            it_1.setFinal(_not);
          } else {
            this.addInjectAnnotation(it_1);
          }
        };
        JvmField _field = this._jvmTypesBuilder.toField(field, 
          field.getName(), 
          field.getType(), _function_1);
        this._jvmTypesBuilder.<JvmField>operator_add(_members_2, _field);
      }
      EList<AuxiliaryDescription> _auxiliaryDescriptions_1 = ts.getAuxiliaryDescriptions();
      for (final AuxiliaryDescription elem_2 : _auxiliaryDescriptions_1) {
        EList<JvmMember> _members_3 = it.getMembers();
        JvmField _genPolymorphicDispatcherField = this.genPolymorphicDispatcherField(elem_2);
        this._jvmTypesBuilder.<JvmField>operator_add(_members_3, _genPolymorphicDispatcherField);
      }
      EList<JudgmentDescription> _judgmentDescriptions = ts.getJudgmentDescriptions();
      for (final JudgmentDescription elem_3 : _judgmentDescriptions) {
        EList<JvmMember> _members_4 = it.getMembers();
        JvmField _genPolymorphicDispatcherField_1 = this.genPolymorphicDispatcherField(elem_3);
        this._jvmTypesBuilder.<JvmField>operator_add(_members_4, _genPolymorphicDispatcherField_1);
      }
      EList<JvmMember> _members_5 = it.getMembers();
      JvmConstructor _genConstructor = this.genConstructor(ts);
      this._jvmTypesBuilder.<JvmConstructor>operator_add(_members_5, _genConstructor);
      EList<JvmMember> _members_6 = it.getMembers();
      JvmOperation _genInit = this.genInit(ts);
      this._jvmTypesBuilder.<JvmOperation>operator_add(_members_6, _genInit);
      EList<AbstractFieldDefinition> _fields_1 = ts.getFields();
      for (final AbstractFieldDefinition field_1 : _fields_1) {
        {
          boolean addSetter = true;
          JvmTypeReference fieldType = field_1.getType();
          if ((field_1 instanceof FieldDefinition)) {
            XExpression _right = ((FieldDefinition)field_1).getRight();
            boolean _tripleNotEquals_2 = (_right != null);
            if (_tripleNotEquals_2) {
              fieldType = this._jvmTypesBuilder.inferredType(((FieldDefinition)field_1).getRight());
            }
          }
          if ((fieldType != null)) {
            EList<JvmMember> _members_7 = it.getMembers();
            JvmOperation _getter = this._jvmTypesBuilder.toGetter(field_1, field_1.getName(), fieldType);
            this._jvmTypesBuilder.<JvmOperation>operator_add(_members_7, _getter);
            if ((field_1 instanceof FieldDefinition)) {
              addSetter = ((FieldDefinition)field_1).isWriteable();
            }
            if (addSetter) {
              EList<JvmMember> _members_8 = it.getMembers();
              JvmOperation _setter = this._jvmTypesBuilder.toSetter(field_1, field_1.getName(), fieldType);
              this._jvmTypesBuilder.<JvmOperation>operator_add(_members_8, _setter);
            }
          }
        }
      }
      EList<AuxiliaryDescription> _auxiliaryDescriptions_2 = ts.getAuxiliaryDescriptions();
      for (final AuxiliaryDescription elem_4 : _auxiliaryDescriptions_2) {
        EList<JvmMember> _members_7 = it.getMembers();
        ArrayList<JvmOperation> _genEntryPointMethods = this.genEntryPointMethods(elem_4);
        this._jvmTypesBuilder.<JvmMember>operator_add(_members_7, _genEntryPointMethods);
      }
      EList<JudgmentDescription> _judgmentDescriptions_1 = ts.getJudgmentDescriptions();
      for (final JudgmentDescription elem_5 : _judgmentDescriptions_1) {
        {
          EList<JvmMember> _members_8 = it.getMembers();
          ArrayList<JvmOperation> _genEntryPointMethods_1 = this.genEntryPointMethods(elem_5);
          this._jvmTypesBuilder.<JvmMember>operator_add(_members_8, _genEntryPointMethods_1);
          EList<JvmMember> _members_9 = it.getMembers();
          ArrayList<JvmOperation> _genSucceededMethods = this.genSucceededMethods(elem_5);
          this._jvmTypesBuilder.<JvmMember>operator_add(_members_9, _genSucceededMethods);
        }
      }
      EList<CheckRule> _checkrules = ts.getCheckrules();
      for (final CheckRule r : _checkrules) {
        {
          EList<JvmMember> _members_8 = it.getMembers();
          ArrayList<JvmOperation> _compileCheckRuleMethods = this.compileCheckRuleMethods(r);
          this._jvmTypesBuilder.<JvmMember>operator_add(_members_8, _compileCheckRuleMethods);
          EList<JvmMember> _members_9 = it.getMembers();
          JvmOperation _compileInternalMethod = this.compileInternalMethod(r);
          this._jvmTypesBuilder.<JvmOperation>operator_add(_members_9, _compileInternalMethod);
        }
      }
      EList<AuxiliaryDescription> _auxiliaryDescriptions_3 = ts.getAuxiliaryDescriptions();
      for (final AuxiliaryDescription elem_6 : _auxiliaryDescriptions_3) {
        {
          EList<JvmMember> _members_8 = it.getMembers();
          JvmOperation _compileInternalMethod = this.compileInternalMethod(elem_6);
          this._jvmTypesBuilder.<JvmOperation>operator_add(_members_8, _compileInternalMethod);
          EList<JvmMember> _members_9 = it.getMembers();
          JvmOperation _compileThrowExceptionMethod = this.compileThrowExceptionMethod(elem_6);
          this._jvmTypesBuilder.<JvmOperation>operator_add(_members_9, _compileThrowExceptionMethod);
          final JvmOperation cacheConditionMethod = this.compileCacheConditionMethod(elem_6);
          if ((cacheConditionMethod != null)) {
            EList<JvmMember> _members_10 = it.getMembers();
            this._jvmTypesBuilder.<JvmOperation>operator_add(_members_10, cacheConditionMethod);
          }
        }
      }
      EList<JudgmentDescription> _judgmentDescriptions_2 = ts.getJudgmentDescriptions();
      for (final JudgmentDescription elem_7 : _judgmentDescriptions_2) {
        {
          EList<JvmMember> _members_8 = it.getMembers();
          JvmOperation _compileInternalMethod = this.compileInternalMethod(elem_7);
          this._jvmTypesBuilder.<JvmOperation>operator_add(_members_8, _compileInternalMethod);
          EList<JvmMember> _members_9 = it.getMembers();
          JvmOperation _compileThrowExceptionMethod = this.compileThrowExceptionMethod(elem_7);
          this._jvmTypesBuilder.<JvmOperation>operator_add(_members_9, _compileThrowExceptionMethod);
          final JvmOperation cacheConditionMethod = this.compileCacheConditionMethod(elem_7);
          if ((cacheConditionMethod != null)) {
            EList<JvmMember> _members_10 = it.getMembers();
            this._jvmTypesBuilder.<JvmOperation>operator_add(_members_10, cacheConditionMethod);
          }
        }
      }
      EList<AuxiliaryFunction> _auxiliaryFunctions = ts.getAuxiliaryFunctions();
      for (final AuxiliaryFunction aux : _auxiliaryFunctions) {
        AuxiliaryDescription _auxiliaryDescription = this._xsemanticsUtils.getAuxiliaryDescription(aux);
        boolean _tripleNotEquals_2 = (_auxiliaryDescription != null);
        if (_tripleNotEquals_2) {
          EList<JvmMember> _members_8 = it.getMembers();
          JvmOperation _compileImplMethod = this.compileImplMethod(aux);
          this._jvmTypesBuilder.<JvmOperation>operator_add(_members_8, _compileImplMethod);
          EList<JvmMember> _members_9 = it.getMembers();
          JvmOperation _compileApplyAuxiliaryFunction = this.compileApplyAuxiliaryFunction(aux);
          this._jvmTypesBuilder.<JvmOperation>operator_add(_members_9, _compileApplyAuxiliaryFunction);
        }
      }
      EList<Rule> _rules_1 = ts.getRules();
      for (final Rule rule : _rules_1) {
        JudgmentDescription _judgmentDescription = this._xsemanticsUtils.getJudgmentDescription(rule);
        boolean _tripleNotEquals_3 = (_judgmentDescription != null);
        if (_tripleNotEquals_3) {
          EList<JvmMember> _members_10 = it.getMembers();
          JvmOperation _compileImplMethod_1 = this.compileImplMethod(rule);
          this._jvmTypesBuilder.<JvmOperation>operator_add(_members_10, _compileImplMethod_1);
          EList<JvmMember> _members_11 = it.getMembers();
          JvmOperation _compileApplyMethod = this.compileApplyMethod(rule);
          this._jvmTypesBuilder.<JvmOperation>operator_add(_members_11, _compileApplyMethod);
          List<ExpressionInConclusion> _expressionsInConclusion = this._xsemanticsUtils.expressionsInConclusion(rule);
          for (final ExpressionInConclusion e : _expressionsInConclusion) {
            EList<JvmMember> _members_12 = it.getMembers();
            JvmOperation _compileExpressionInConclusionMethod = this.compileExpressionInConclusionMethod(e);
            this._jvmTypesBuilder.<JvmOperation>operator_add(_members_12, _compileExpressionInConclusionMethod);
          }
          XExpression _error = rule.getConclusion().getError();
          boolean _tripleNotEquals_4 = (_error != null);
          if (_tripleNotEquals_4) {
            EList<JvmMember> _members_13 = it.getMembers();
            JvmOperation _compileErrorSpecificationMethod = this.compileErrorSpecificationMethod(rule);
            this._jvmTypesBuilder.<JvmOperation>operator_add(_members_13, _compileErrorSpecificationMethod);
          }
        }
      }
    };
    acceptor.<JvmGenericType>accept(inferredClass, _function);
    final Procedure1<JvmGenericType> _function_1 = (JvmGenericType it) -> {
      String _copyright = ts.getCopyright();
      boolean _tripleNotEquals = (_copyright != null);
      if (_tripleNotEquals) {
        this._jvmTypesBuilder.setFileHeader(it, ts.getCopyright());
      }
      this._jvmTypesBuilder.setDocumentation(it, this._jvmTypesBuilder.getDocumentation(ts));
      XsemanticsSystem _superSystemDefinition = this._xsemanticsUtils.superSystemDefinition(ts);
      boolean _tripleNotEquals_1 = (_superSystemDefinition != null);
      if (_tripleNotEquals_1) {
        EList<JvmTypeReference> _superTypes = it.getSuperTypes();
        JvmTypeReference _typeRef = this._typeReferenceBuilder.typeRef(this._xsemanticsGeneratorExtensions.toValidatorJavaFullyQualifiedName(this._xsemanticsUtils.superSystemDefinition(ts)));
        this._jvmTypesBuilder.<JvmTypeReference>operator_add(_superTypes, _typeRef);
      } else {
        JvmParameterizedTypeReference _validatorExtends = ts.getValidatorExtends();
        boolean _tripleNotEquals_2 = (_validatorExtends != null);
        if (_tripleNotEquals_2) {
          EList<JvmTypeReference> _superTypes_1 = it.getSuperTypes();
          JvmTypeReference _cloneWithProxies = this._jvmTypesBuilder.cloneWithProxies(ts.getValidatorExtends());
          this._jvmTypesBuilder.<JvmTypeReference>operator_add(_superTypes_1, _cloneWithProxies);
        } else {
          EList<JvmTypeReference> _superTypes_2 = it.getSuperTypes();
          JvmTypeReference _typeRef_1 = this._typeReferenceBuilder.typeRef(AbstractDeclarativeValidator.class);
          this._jvmTypesBuilder.<JvmTypeReference>operator_add(_superTypes_2, _typeRef_1);
        }
      }
      JvmParameterizedTypeReference _superSystem = ts.getSuperSystem();
      boolean _tripleEquals_1 = (_superSystem == null);
      if (_tripleEquals_1) {
        EList<JvmMember> _members = it.getMembers();
        final Procedure1<JvmField> _function_2 = (JvmField it_1) -> {
          this.addInjectAnnotation(it_1);
          it_1.setVisibility(JvmVisibility.PROTECTED);
        };
        JvmField _field = this._jvmTypesBuilder.toField(ts, "errorGenerator", this._typeReferenceBuilder.typeRef(XsemanticsValidatorErrorGenerator.class), _function_2);
        this._jvmTypesBuilder.<JvmField>operator_add(_members, _field);
      }
      EList<JvmMember> _members_1 = it.getMembers();
      final Procedure1<JvmField> _function_3 = (JvmField it_1) -> {
        this.addInjectAnnotation(it_1);
        it_1.setVisibility(JvmVisibility.PROTECTED);
      };
      JvmField _field_1 = this._jvmTypesBuilder.toField(ts, "xsemanticsSystem", this._typeReferences.createTypeRef(inferredClass), _function_3);
      this._jvmTypesBuilder.<JvmField>operator_add(_members_1, _field_1);
      EList<JvmMember> _members_2 = it.getMembers();
      JvmOperation _getter = this._jvmTypesBuilder.toGetter(ts, "xsemanticsSystem", this._typeReferences.createTypeRef(inferredClass));
      final Procedure1<JvmOperation> _function_4 = (JvmOperation it_1) -> {
        it_1.setVisibility(JvmVisibility.PROTECTED);
        JvmParameterizedTypeReference _superSystem_1 = ts.getSuperSystem();
        boolean _tripleNotEquals_3 = (_superSystem_1 != null);
        if (_tripleNotEquals_3) {
          this.addOverrideAnnotation(it_1);
        }
      };
      JvmOperation _doubleArrow = ObjectExtensions.<JvmOperation>operator_doubleArrow(_getter, _function_4);
      this._jvmTypesBuilder.<JvmMember>operator_add(_members_2, _doubleArrow);
      EList<CheckRule> _checkrules = ts.getCheckrules();
      for (final CheckRule rule : _checkrules) {
        EList<JvmMember> _members_3 = it.getMembers();
        JvmOperation _compileValidatorCheckRuleMethod = this.compileValidatorCheckRuleMethod(rule);
        this._jvmTypesBuilder.<JvmOperation>operator_add(_members_3, _compileValidatorCheckRuleMethod);
      }
    };
    acceptor.<JvmGenericType>accept(this._jvmTypesBuilder.toClass(ts, this._xsemanticsGeneratorExtensions.toValidatorJavaFullyQualifiedName(ts)), _function_1);
  }

  public JvmField genIssueField(final Named e) {
    JvmField _xblockexpression = null;
    {
      final Procedure1<JvmField> _function = (JvmField it) -> {
        it.setVisibility(JvmVisibility.PUBLIC);
        it.setStatic(true);
        it.setFinal(true);
      };
      final JvmField issueField = this._jvmTypesBuilder.toField(e, 
        this._xsemanticsGeneratorExtensions.ruleIssueString(e), 
        this._typeReferenceBuilder.typeRef(String.class), _function);
      final Procedure1<ITreeAppendable> _function_1 = (ITreeAppendable it) -> {
        StringConcatenation _builder = new StringConcatenation();
        _builder.append("\"");
        String _javaFullyQualifiedName = this._xsemanticsGeneratorExtensions.toJavaFullyQualifiedName(e);
        _builder.append(_javaFullyQualifiedName);
        _builder.append("\"");
        it.append(_builder);
      };
      this._jvmTypesBuilder.setInitializer(issueField, _function_1);
      _xblockexpression = issueField;
    }
    return _xblockexpression;
  }

  public JvmConstructor genConstructor(final XsemanticsSystem ts) {
    final Procedure1<JvmConstructor> _function = (JvmConstructor it) -> {
      final Procedure1<ITreeAppendable> _function_1 = (ITreeAppendable it_1) -> {
        it_1.append("init();");
      };
      this._jvmTypesBuilder.setBody(it, _function_1);
    };
    return this._jvmTypesBuilder.toConstructor(ts, _function);
  }

  public JvmField genPolymorphicDispatcherField(final JudgmentDescription e) {
    return this._jvmTypesBuilder.toField(e, 
      this._xsemanticsGeneratorExtensions.polymorphicDispatcherField(e).toString(), 
      this.polymorphicDispatcherType(e));
  }

  public JvmField genPolymorphicDispatcherField(final AuxiliaryDescription e) {
    return this._jvmTypesBuilder.toField(e, 
      this._xsemanticsGeneratorExtensions.polymorphicDispatcherField(e).toString(), 
      this.polymorphicDispatcherType(e));
  }

  public JvmTypeReference polymorphicDispatcherType(final JudgmentDescription e) {
    return this._typeReferenceBuilder.typeRef(PolymorphicDispatcher.class, this._xsemanticsGeneratorExtensions.resultType(e));
  }

  public JvmTypeReference polymorphicDispatcherType(final AuxiliaryDescription e) {
    return this._typeReferenceBuilder.typeRef(PolymorphicDispatcher.class, this._xsemanticsGeneratorExtensions.resultType(e));
  }

  public JvmOperation genInit(final XsemanticsSystem ts) {
    final Procedure1<JvmOperation> _function = (JvmOperation it) -> {
      JvmParameterizedTypeReference _superSystem = ts.getSuperSystem();
      boolean _tripleNotEquals = (_superSystem != null);
      if (_tripleNotEquals) {
        this.addOverrideAnnotation(it);
      }
      final Procedure1<ITreeAppendable> _function_1 = (ITreeAppendable it_1) -> {
        JvmParameterizedTypeReference _superSystem_1 = ts.getSuperSystem();
        boolean _tripleNotEquals_1 = (_superSystem_1 != null);
        if (_tripleNotEquals_1) {
          StringConcatenation _builder = new StringConcatenation();
          _builder.append("super.init();");
          _builder.newLine();
          it_1.append(_builder);
        }
        final Function1<JudgmentDescription, CharSequence> _function_2 = (JudgmentDescription desc) -> {
          return this.genPolymorphicDispatcherInit(desc);
        };
        List<CharSequence> _map = ListExtensions.<JudgmentDescription, CharSequence>map(ts.getJudgmentDescriptions(), _function_2);
        final Function1<AuxiliaryDescription, CharSequence> _function_3 = (AuxiliaryDescription desc) -> {
          return this.genPolymorphicDispatcherInit(desc);
        };
        List<CharSequence> _map_1 = ListExtensions.<AuxiliaryDescription, CharSequence>map(ts.getAuxiliaryDescriptions(), _function_3);
        it_1.append(
          IterableExtensions.join(Iterables.<CharSequence>concat(_map, _map_1), "\n"));
      };
      this._jvmTypesBuilder.setBody(it, _function_1);
    };
    return this._jvmTypesBuilder.toMethod(ts, "init", this._typeReferences.getTypeForName(Void.TYPE, ts), _function);
  }

  public CharSequence genPolymorphicDispatcherInit(final JudgmentDescription judgmentDescription) {
    CharSequence _xblockexpression = null;
    {
      final String relationSymbols = this._xsemanticsGeneratorExtensions.relationSymbolsArgs(judgmentDescription);
      String _xifexpression = null;
      boolean _isEmpty = relationSymbols.isEmpty();
      boolean _not = (!_isEmpty);
      if (_not) {
        _xifexpression = (", " + relationSymbols);
      } else {
        _xifexpression = "";
      }
      final String relationSymbolArgs = _xifexpression;
      StringConcatenation _builder = new StringConcatenation();
      CharSequence _polymorphicDispatcherField = this._xsemanticsGeneratorExtensions.polymorphicDispatcherField(judgmentDescription);
      _builder.append(_polymorphicDispatcherField);
      _builder.append(" = ");
      CharSequence _polymorphicDispatcherBuildMethod = this._xsemanticsGeneratorExtensions.polymorphicDispatcherBuildMethod(judgmentDescription);
      _builder.append(_polymorphicDispatcherBuildMethod);
      _builder.append("(");
      _builder.newLineIfNotEmpty();
      _builder.append("\t");
      _builder.append("\"");
      CharSequence _polymorphicDispatcherImpl = this._xsemanticsGeneratorExtensions.polymorphicDispatcherImpl(judgmentDescription);
      _builder.append(_polymorphicDispatcherImpl, "\t");
      _builder.append("\", ");
      CharSequence _polymorphicDispatcherNumOfArgs = this._xsemanticsGeneratorExtensions.polymorphicDispatcherNumOfArgs(judgmentDescription);
      _builder.append(_polymorphicDispatcherNumOfArgs, "\t");
      _builder.append(", ");
      _builder.append("\"");
      String _escapeJavaStringChars = this._xsemanticsGeneratorExtensions.escapeJavaStringChars(judgmentDescription.getJudgmentSymbol());
      _builder.append(_escapeJavaStringChars, "\t");
      _builder.append("\"");
      _builder.append(relationSymbolArgs, "\t");
      _builder.append(");");
      _xblockexpression = _builder;
    }
    return _xblockexpression;
  }

  public CharSequence genPolymorphicDispatcherInit(final AuxiliaryDescription aux) {
    StringConcatenation _builder = new StringConcatenation();
    CharSequence _polymorphicDispatcherField = this._xsemanticsGeneratorExtensions.polymorphicDispatcherField(aux);
    _builder.append(_polymorphicDispatcherField);
    _builder.append(" = buildPolymorphicDispatcher(");
    _builder.newLineIfNotEmpty();
    _builder.append("\t");
    _builder.append("\"");
    CharSequence _polymorphicDispatcherImpl = this._xsemanticsGeneratorExtensions.polymorphicDispatcherImpl(aux);
    _builder.append(_polymorphicDispatcherImpl, "\t");
    _builder.append("\", ");
    CharSequence _polymorphicDispatcherNumOfArgs = this._xsemanticsGeneratorExtensions.polymorphicDispatcherNumOfArgs(aux);
    _builder.append(_polymorphicDispatcherNumOfArgs, "\t");
    _builder.append(");");
    return _builder;
  }

  public ArrayList<JvmOperation> genEntryPointMethods(final JudgmentDescription judgmentDescription) {
    ArrayList<JvmOperation> _xblockexpression = null;
    {
      final CharSequence methodName = this._xsemanticsGeneratorExtensions.entryPointMethodName(judgmentDescription);
      final JvmTypeReference resultType = this._xsemanticsGeneratorExtensions.resultType(judgmentDescription);
      final Procedure1<JvmOperation> _function = (JvmOperation it) -> {
        final String inputArgs = this._xsemanticsGeneratorExtensions.inputArgs(judgmentDescription);
        StringConcatenation _builder = new StringConcatenation();
        _builder.append("try {");
        _builder.newLine();
        _builder.append("\t");
        _builder.append("return ");
        CharSequence _entryPointInternalMethodName = this._xsemanticsGeneratorExtensions.entryPointInternalMethodName(judgmentDescription);
        _builder.append(_entryPointInternalMethodName, "\t");
        _builder.append("(");
        CharSequence _additionalArgs = this._xsemanticsGeneratorExtensions.additionalArgs();
        _builder.append(_additionalArgs, "\t");
        _builder.append(", ");
        _builder.append(inputArgs, "\t");
        _builder.append(");");
        _builder.newLineIfNotEmpty();
        _builder.append("} catch (");
        String _simpleName = Exception.class.getSimpleName();
        _builder.append(_simpleName);
        _builder.append(" ");
        CharSequence _exceptionVarName = this._xsemanticsGeneratorExtensions.exceptionVarName(judgmentDescription);
        _builder.append(_exceptionVarName);
        _builder.append(") {");
        _builder.newLineIfNotEmpty();
        _builder.append("\t");
        _builder.append("return resultForFailure");
        String _suffixStartingFrom2 = this._xsemanticsGeneratorExtensions.suffixStartingFrom2(judgmentDescription);
        _builder.append(_suffixStartingFrom2, "\t");
        _builder.append("(");
        CharSequence _exceptionVarName_1 = this._xsemanticsGeneratorExtensions.exceptionVarName(judgmentDescription);
        _builder.append(_exceptionVarName_1, "\t");
        _builder.append(");");
        _builder.newLineIfNotEmpty();
        _builder.append("}");
        final String entryPointInternalInvocation = _builder.toString();
        boolean _cacheEntryPointMethods = this._xsemanticsUtils.cacheEntryPointMethods(judgmentDescription);
        if (_cacheEntryPointMethods) {
          StringConcatenationClient _client = new StringConcatenationClient() {
            @Override
            protected void appendTo(StringConcatenationClient.TargetStringConcatenation _builder) {
              StringConcatenation _builder_1 = new StringConcatenation();
              CharSequence _environmentName = XsemanticsJvmModelInferrer.this._xsemanticsGeneratorExtensions.environmentName();
              _builder_1.append(_environmentName);
              _builder_1.append(", ");
              _builder_1.append(inputArgs);
              CharSequence _bodyForCache = XsemanticsJvmModelInferrer.this.bodyForCache(judgmentDescription, _builder_1, entryPointInternalInvocation);
              _builder.append(_bodyForCache);
              _builder.newLineIfNotEmpty();
              StringConcatenationClient _wrapInGetFromCache = XsemanticsJvmModelInferrer.this.wrapInGetFromCache(methodName, resultType, inputArgs, entryPointInternalInvocation);
              _builder.append(_wrapInGetFromCache);
            }
          };
          this._jvmTypesBuilder.setBody(it, _client);
        } else {
          final Procedure1<ITreeAppendable> _function_1 = (ITreeAppendable it_1) -> {
            it_1.append(entryPointInternalInvocation);
          };
          this._jvmTypesBuilder.setBody(it, _function_1);
        }
      };
      _xblockexpression = this.toEntryMethodsTriple(judgmentDescription, methodName.toString(), resultType, _function);
    }
    return _xblockexpression;
  }

  /**
   * @param judgmentDescription
   * @param name method name
   * @param resultType method return type
   * @param beforeInputParams can add parameters before the input parameters
   * @param mainBodyCreator handles the creation of the body of the third method,
   * 	which implements the real logic; the first two methods are only delegates with
   * 	default arguments.
   */
  public ArrayList<JvmOperation> toEntryMethodsTriple(final JudgmentDescription judgmentDescription, final String name, final JvmTypeReference resultType, final Procedure1<? super JvmOperation> mainBodyCreator) {
    ArrayList<JvmOperation> _xblockexpression = null;
    {
      final ArrayList<JvmOperation> entryPointMethods = CollectionLiterals.<JvmOperation>newArrayList();
      final String inputArgs = this._xsemanticsGeneratorExtensions.inputArgs(judgmentDescription);
      final Procedure1<JvmOperation> _function = (JvmOperation it) -> {
      };
      final Procedure1<JvmOperation> _function_1 = (JvmOperation it) -> {
        StringConcatenationClient _client = new StringConcatenationClient() {
          @Override
          protected void appendTo(StringConcatenationClient.TargetStringConcatenation _builder) {
            _builder.append("return ");
            _builder.append(name);
            _builder.append("(new ");
            _builder.append(RuleEnvironment.class);
            _builder.append("(), null, ");
            _builder.append(inputArgs);
            _builder.append(");");
          }
        };
        this._jvmTypesBuilder.setBody(it, _client);
      };
      JvmOperation _entryMethodCommon = this.toEntryMethodCommon(judgmentDescription, name, resultType, _function, _function_1);
      entryPointMethods.add(_entryMethodCommon);
      final Procedure1<JvmOperation> _function_2 = (JvmOperation it) -> {
        EList<JvmFormalParameter> _parameters = it.getParameters();
        JvmFormalParameter _environmentParam = this.environmentParam(judgmentDescription);
        this._jvmTypesBuilder.<JvmFormalParameter>operator_add(_parameters, _environmentParam);
      };
      final Procedure1<JvmOperation> _function_3 = (JvmOperation it) -> {
        StringConcatenationClient _client = new StringConcatenationClient() {
          @Override
          protected void appendTo(StringConcatenationClient.TargetStringConcatenation _builder) {
            _builder.append("return ");
            _builder.append(name);
            _builder.append("(");
            CharSequence _environmentName = XsemanticsJvmModelInferrer.this._xsemanticsGeneratorExtensions.environmentName();
            _builder.append(_environmentName);
            _builder.append(", null, ");
            _builder.append(inputArgs);
            _builder.append(");");
          }
        };
        this._jvmTypesBuilder.setBody(it, _client);
      };
      JvmOperation _entryMethodCommon_1 = this.toEntryMethodCommon(judgmentDescription, name, resultType, _function_2, _function_3);
      entryPointMethods.add(_entryMethodCommon_1);
      final Procedure1<JvmOperation> _function_4 = (JvmOperation it) -> {
        EList<JvmFormalParameter> _parameters = it.getParameters();
        JvmFormalParameter _environmentParam = this.environmentParam(judgmentDescription);
        this._jvmTypesBuilder.<JvmFormalParameter>operator_add(_parameters, _environmentParam);
        EList<JvmFormalParameter> _parameters_1 = it.getParameters();
        JvmFormalParameter _ruleApplicationTraceParam = this.ruleApplicationTraceParam(judgmentDescription);
        this._jvmTypesBuilder.<JvmFormalParameter>operator_add(_parameters_1, _ruleApplicationTraceParam);
      };
      JvmOperation _entryMethodCommon_2 = this.toEntryMethodCommon(judgmentDescription, name, resultType, _function_4, mainBodyCreator);
      entryPointMethods.add(_entryMethodCommon_2);
      _xblockexpression = entryPointMethods;
    }
    return _xblockexpression;
  }

  /**
   * @param judgmentDescription
   * @param name method name
   * @param resultType method return type
   * @param beforeInputParams can add parameters before the input parameters
   * @param bodyCreator handles the creation of the body of the method
   */
  protected JvmOperation toEntryMethodCommon(final JudgmentDescription judgmentDescription, final String name, final JvmTypeReference resultType, final Procedure1<? super JvmOperation> beforeInputParams, final Procedure1<? super JvmOperation> bodyCreator) {
    final Procedure1<JvmOperation> _function = (JvmOperation it) -> {
      this.addOverrideAnnotation(it, judgmentDescription);
      beforeInputParams.apply(it);
      EList<JvmFormalParameter> _parameters = it.getParameters();
      List<JvmFormalParameter> _inputParameters = this.inputParameters(judgmentDescription);
      this._jvmTypesBuilder.<JvmFormalParameter>operator_add(_parameters, _inputParameters);
      bodyCreator.apply(it);
    };
    return this._jvmTypesBuilder.toMethod(judgmentDescription, name, resultType, _function);
  }

  private CharSequence bodyForCache(final Cachable c, final CharSequence args, final CharSequence entryPointInternalInvocation) {
    CharSequence _xblockexpression = null;
    {
      XExpression _cacheCondition = this._xsemanticsUtils.cacheCondition(c);
      final boolean hasCacheCondition = (_cacheCondition != null);
      StringConcatenation _builder = new StringConcatenation();
      {
        if (hasCacheCondition) {
          _builder.append("if (!");
          CharSequence _cacheConditionMethod = this._xsemanticsGeneratorExtensions.cacheConditionMethod(c);
          _builder.append(_cacheConditionMethod);
          _builder.append("(");
          _builder.append(args);
          _builder.append("))");
          _builder.newLineIfNotEmpty();
          _builder.append("\t");
          _builder.append(entryPointInternalInvocation, "\t");
          _builder.newLineIfNotEmpty();
        }
      }
      _xblockexpression = _builder;
    }
    return _xblockexpression;
  }

  private StringConcatenationClient wrapInGetFromCache(final CharSequence methodName, final JvmTypeReference resultType, final String inputArgs, final CharSequence providerBody) {
    StringConcatenationClient _xblockexpression = null;
    {
      final CharSequence envArg = this._xsemanticsGeneratorExtensions.environmentName();
      _xblockexpression = this.wrapInGetFromCacheCommon(methodName, resultType, envArg, envArg, inputArgs, providerBody);
    }
    return _xblockexpression;
  }

  private StringConcatenationClient wrapInGetFromCache2(final CharSequence methodName, final JvmTypeReference resultType, final String inputArgs, final CharSequence providerBody) {
    StringConcatenation _builder = new StringConcatenation();
    _builder.append("(");
    String _simpleName = RuleEnvironment.class.getSimpleName();
    _builder.append(_simpleName);
    _builder.append(")null");
    return this.wrapInGetFromCacheCommon(methodName, resultType, _builder, "null", inputArgs, providerBody);
  }

  private StringConcatenationClient wrapInGetFromCacheCommon(final CharSequence methodName, final JvmTypeReference resultType, final CharSequence getFromCacheEnvArg, final CharSequence providerEnvArg, final String inputArgs, final CharSequence providerBody) {
    StringConcatenationClient _client = new StringConcatenationClient() {
      @Override
      protected void appendTo(StringConcatenationClient.TargetStringConcatenation _builder) {
        _builder.append("return getFromCache(\"");
        _builder.append(methodName);
        _builder.append("\", ");
        _builder.append(getFromCacheEnvArg);
        _builder.append(", ");
        CharSequence _ruleApplicationTraceName = XsemanticsJvmModelInferrer.this._xsemanticsGeneratorExtensions.ruleApplicationTraceName();
        _builder.append(_ruleApplicationTraceName);
        _builder.append(",");
        _builder.newLineIfNotEmpty();
        _builder.append("\t");
        _builder.append("new ");
        _builder.append(XsemanticsProvider.class, "\t");
        _builder.append("<");
        _builder.append(resultType, "\t");
        _builder.append(">(");
        _builder.append(providerEnvArg, "\t");
        _builder.append(", ");
        CharSequence _ruleApplicationTraceName_1 = XsemanticsJvmModelInferrer.this._xsemanticsGeneratorExtensions.ruleApplicationTraceName();
        _builder.append(_ruleApplicationTraceName_1, "\t");
        _builder.append(") {");
        _builder.newLineIfNotEmpty();
        _builder.append("\t\t");
        _builder.append("public ");
        _builder.append(resultType, "\t\t");
        _builder.append(" doGet() {");
        _builder.newLineIfNotEmpty();
        _builder.append("\t\t\t");
        _builder.append(providerBody, "\t\t\t");
        _builder.newLineIfNotEmpty();
        _builder.append("\t\t");
        _builder.append("}");
        _builder.newLine();
        _builder.append("\t");
        _builder.append("}, ");
        _builder.append(inputArgs, "\t");
        _builder.append(");");
      }
    };
    return _client;
  }

  public ArrayList<JvmOperation> genSucceededMethods(final JudgmentDescription judgmentDescription) {
    ArrayList<JvmOperation> _xblockexpression = null;
    {
      boolean _isPredicate = this.typeSystem.isPredicate(judgmentDescription);
      boolean _not = (!_isPredicate);
      if (_not) {
        return CollectionLiterals.<JvmOperation>newArrayList();
      }
      final Procedure1<JvmOperation> _function = (JvmOperation it) -> {
        StringConcatenationClient _client = new StringConcatenationClient() {
          @Override
          protected void appendTo(StringConcatenationClient.TargetStringConcatenation _builder) {
            _builder.append("try {");
            _builder.newLine();
            _builder.append("\t");
            CharSequence _entryPointInternalMethodName = XsemanticsJvmModelInferrer.this._xsemanticsGeneratorExtensions.entryPointInternalMethodName(judgmentDescription);
            _builder.append(_entryPointInternalMethodName, "\t");
            _builder.append("(");
            CharSequence _additionalArgs = XsemanticsJvmModelInferrer.this._xsemanticsGeneratorExtensions.additionalArgs();
            _builder.append(_additionalArgs, "\t");
            _builder.append(", ");
            String _inputArgs = XsemanticsJvmModelInferrer.this._xsemanticsGeneratorExtensions.inputArgs(judgmentDescription);
            _builder.append(_inputArgs, "\t");
            _builder.append(");");
            _builder.newLineIfNotEmpty();
            _builder.append("\t");
            _builder.append("return true;");
            _builder.newLine();
            _builder.append("} catch (");
            _builder.append(Exception.class);
            _builder.append(" ");
            CharSequence _exceptionVarName = XsemanticsJvmModelInferrer.this._xsemanticsGeneratorExtensions.exceptionVarName(judgmentDescription);
            _builder.append(_exceptionVarName);
            _builder.append(") {");
            _builder.newLineIfNotEmpty();
            _builder.append("\t");
            _builder.append("return false;");
            _builder.newLine();
            _builder.append("}");
          }
        };
        this._jvmTypesBuilder.setBody(it, _client);
      };
      _xblockexpression = this.toEntryMethodsTriple(judgmentDescription, this._xsemanticsGeneratorExtensions.succeededMethodName(judgmentDescription).toString(), this._typeReferenceBuilder.typeRef(Boolean.class), _function);
    }
    return _xblockexpression;
  }

  public ArrayList<JvmOperation> genEntryPointMethods(final AuxiliaryDescription aux) {
    ArrayList<JvmOperation> _xblockexpression = null;
    {
      final ArrayList<JvmOperation> entryPointMethods = CollectionLiterals.<JvmOperation>newArrayList();
      final Procedure1<JvmOperation> _function = (JvmOperation it) -> {
        EList<JvmTypeReference> _exceptions = it.getExceptions();
        JvmTypeReference _ruleFailedExceptionType = this.ruleFailedExceptionType();
        this._jvmTypesBuilder.<JvmTypeReference>operator_add(_exceptions, _ruleFailedExceptionType);
        this.addOverrideAnnotation(it, aux);
        EList<JvmFormalParameter> _parameters = it.getParameters();
        List<JvmFormalParameter> _inputParameters = this.inputParameters(aux);
        this._jvmTypesBuilder.<JvmFormalParameter>operator_add(_parameters, _inputParameters);
        StringConcatenationClient _client = new StringConcatenationClient() {
          @Override
          protected void appendTo(StringConcatenationClient.TargetStringConcatenation _builder) {
            _builder.append("return ");
            CharSequence _entryPointMethodName = XsemanticsJvmModelInferrer.this._xsemanticsGeneratorExtensions.entryPointMethodName(aux);
            _builder.append(_entryPointMethodName);
            _builder.append("(null, ");
            String _inputArgs = XsemanticsJvmModelInferrer.this._xsemanticsGeneratorExtensions.inputArgs(aux);
            _builder.append(_inputArgs);
            _builder.append(");");
          }
        };
        this._jvmTypesBuilder.setBody(it, _client);
      };
      JvmOperation _method = this._jvmTypesBuilder.toMethod(aux, 
        this._xsemanticsGeneratorExtensions.entryPointMethodName(aux).toString(), 
        this._xsemanticsGeneratorExtensions.resultType(aux), _function);
      entryPointMethods.add(_method);
      final Procedure1<JvmOperation> _function_1 = (JvmOperation it) -> {
        EList<JvmTypeReference> _exceptions = it.getExceptions();
        JvmTypeReference _ruleFailedExceptionType = this.ruleFailedExceptionType();
        this._jvmTypesBuilder.<JvmTypeReference>operator_add(_exceptions, _ruleFailedExceptionType);
        this.addOverrideAnnotation(it, aux);
        EList<JvmFormalParameter> _parameters = it.getParameters();
        JvmFormalParameter _ruleApplicationTraceParam = this.ruleApplicationTraceParam(aux);
        this._jvmTypesBuilder.<JvmFormalParameter>operator_add(_parameters, _ruleApplicationTraceParam);
        EList<JvmFormalParameter> _parameters_1 = it.getParameters();
        List<JvmFormalParameter> _inputParameters = this.inputParameters(aux);
        this._jvmTypesBuilder.<JvmFormalParameter>operator_add(_parameters_1, _inputParameters);
        StringConcatenationClient _client = new StringConcatenationClient() {
          @Override
          protected void appendTo(StringConcatenationClient.TargetStringConcatenation _builder) {
            _builder.append("try {");
            _builder.newLine();
            _builder.append("\t");
            _builder.append("return ");
            CharSequence _entryPointInternalMethodName = XsemanticsJvmModelInferrer.this._xsemanticsGeneratorExtensions.entryPointInternalMethodName(aux);
            _builder.append(_entryPointInternalMethodName, "\t");
            _builder.append("(");
            CharSequence _ruleApplicationTraceName = XsemanticsJvmModelInferrer.this._xsemanticsGeneratorExtensions.ruleApplicationTraceName();
            _builder.append(_ruleApplicationTraceName, "\t");
            _builder.append(", ");
            String _inputArgs = XsemanticsJvmModelInferrer.this._xsemanticsGeneratorExtensions.inputArgs(aux);
            _builder.append(_inputArgs, "\t");
            _builder.append(");");
            _builder.newLineIfNotEmpty();
            _builder.append("} catch (");
            _builder.append(Exception.class);
            _builder.append(" ");
            CharSequence _exceptionVarName = XsemanticsJvmModelInferrer.this._xsemanticsGeneratorExtensions.exceptionVarName(aux);
            _builder.append(_exceptionVarName);
            _builder.append(") {");
            _builder.newLineIfNotEmpty();
            _builder.append("\t");
            _builder.append("throw extractRuleFailedException(");
            CharSequence _exceptionVarName_1 = XsemanticsJvmModelInferrer.this._xsemanticsGeneratorExtensions.exceptionVarName(aux);
            _builder.append(_exceptionVarName_1, "\t");
            _builder.append(");");
            _builder.newLineIfNotEmpty();
            _builder.append("}");
          }
        };
        this._jvmTypesBuilder.setBody(it, _client);
      };
      JvmOperation _method_1 = this._jvmTypesBuilder.toMethod(aux, 
        this._xsemanticsGeneratorExtensions.entryPointMethodName(aux).toString(), 
        this._xsemanticsGeneratorExtensions.resultType(aux), _function_1);
      entryPointMethods.add(_method_1);
      _xblockexpression = entryPointMethods;
    }
    return _xblockexpression;
  }

  public List<JvmFormalParameter> inputParameters(final JudgmentDescription judgmentDescription) {
    List<JvmFormalParameter> _xblockexpression = null;
    {
      final UniqueNames names = new UniqueNames();
      final Function1<InputParameter, JvmFormalParameter> _function = (InputParameter it) -> {
        return this._jvmTypesBuilder.toParameter(it, 
          names.createName(this._xsemanticsGeneratorExtensions.inputParameterName(it)), 
          it.getParameter().getParameterType());
      };
      _xblockexpression = ListExtensions.<InputParameter, JvmFormalParameter>map(this._xsemanticsUtils.inputParams(judgmentDescription), _function);
    }
    return _xblockexpression;
  }

  public List<JvmFormalParameter> inputParameters(final AuxiliaryDescription aux) {
    final Function1<JvmFormalParameter, JvmFormalParameter> _function = (JvmFormalParameter it) -> {
      return this._jvmTypesBuilder.toParameter(it, 
        it.getName(), 
        it.getParameterType());
    };
    return ListExtensions.<JvmFormalParameter, JvmFormalParameter>map(aux.getParameters(), _function);
  }

  public List<JvmFormalParameter> inputParameters(final AuxiliaryFunction aux) {
    final Function1<JvmFormalParameter, JvmFormalParameter> _function = (JvmFormalParameter it) -> {
      return this._jvmTypesBuilder.toParameter(it, 
        it.getName(), 
        it.getParameterType());
    };
    return ListExtensions.<JvmFormalParameter, JvmFormalParameter>map(aux.getParameters(), _function);
  }

  public JvmFormalParameter environmentParam(final JudgmentDescription e) {
    return this._jvmTypesBuilder.toParameter(e, 
      this._xsemanticsGeneratorExtensions.environmentName().toString(), 
      this.environmentType());
  }

  public JvmFormalParameter ruleApplicationTraceParam(final EObject e) {
    return this._jvmTypesBuilder.toParameter(e, 
      this._xsemanticsGeneratorExtensions.ruleApplicationTraceName().toString(), 
      this._typeReferenceBuilder.typeRef(RuleApplicationTrace.class));
  }

  public JvmOperation compileThrowExceptionMethod(final JudgmentDescription judgmentDescription) {
    JvmOperation _xblockexpression = null;
    {
      final XExpression errorSpecification = judgmentDescription.getError();
      final Procedure1<JvmOperation> _function = (JvmOperation it) -> {
        this.addProtectedAndOverride(it, judgmentDescription);
        EList<JvmTypeReference> _exceptions = it.getExceptions();
        JvmTypeReference _ruleFailedExceptionType = this.ruleFailedExceptionType();
        this._jvmTypesBuilder.<JvmTypeReference>operator_add(_exceptions, _ruleFailedExceptionType);
        EList<JvmFormalParameter> _parameters = it.getParameters();
        JvmFormalParameter _parameter = this._jvmTypesBuilder.toParameter(judgmentDescription, "_error", 
          this._typeReferenceBuilder.typeRef(String.class));
        this._jvmTypesBuilder.<JvmFormalParameter>operator_add(_parameters, _parameter);
        EList<JvmFormalParameter> _parameters_1 = it.getParameters();
        JvmFormalParameter _parameter_1 = this._jvmTypesBuilder.toParameter(judgmentDescription, "_issue", 
          this._typeReferenceBuilder.typeRef(String.class));
        this._jvmTypesBuilder.<JvmFormalParameter>operator_add(_parameters_1, _parameter_1);
        EList<JvmFormalParameter> _parameters_2 = it.getParameters();
        JvmFormalParameter _parameter_2 = this._jvmTypesBuilder.toParameter(judgmentDescription, "_ex", 
          this._xsemanticsGeneratorExtensions.exceptionType(judgmentDescription));
        this._jvmTypesBuilder.<JvmFormalParameter>operator_add(_parameters_2, _parameter_2);
        EList<JvmFormalParameter> _parameters_3 = it.getParameters();
        List<JvmFormalParameter> _inputParameters = this.inputParameters(judgmentDescription);
        this._jvmTypesBuilder.<JvmFormalParameter>operator_add(_parameters_3, _inputParameters);
        EList<JvmFormalParameter> _parameters_4 = it.getParameters();
        JvmFormalParameter _parameter_3 = this._jvmTypesBuilder.toParameter(judgmentDescription, "_errorInformations", 
          this._jvmTypesBuilder.addArrayTypeDimension(this._typeReferenceBuilder.typeRef(ErrorInformation.class)));
        this._jvmTypesBuilder.<JvmFormalParameter>operator_add(_parameters_4, _parameter_3);
        this.compileErrorSpecification(it, errorSpecification);
      };
      _xblockexpression = this._jvmTypesBuilder.toMethod(judgmentDescription, 
        this._xsemanticsGeneratorExtensions.throwExceptionMethod(judgmentDescription).toString(), 
        this._typeReferences.getTypeForName(Void.TYPE, judgmentDescription), _function);
    }
    return _xblockexpression;
  }

  public void compileErrorSpecification(final JvmOperation it, final XExpression errorSpecification) {
    if ((errorSpecification != null)) {
      this._jvmTypesBuilder.setBody(it, errorSpecification);
    } else {
      StringConcatenationClient _client = new StringConcatenationClient() {
        @Override
        protected void appendTo(StringConcatenationClient.TargetStringConcatenation _builder) {
          CharSequence _throwRuleFailedExceptionMethod = XsemanticsJvmModelInferrer.this._xsemanticsGeneratorExtensions.throwRuleFailedExceptionMethod();
          _builder.append(_throwRuleFailedExceptionMethod);
          _builder.append("(_error, _issue, _ex, _errorInformations);");
        }
      };
      this._jvmTypesBuilder.setBody(it, _client);
    }
  }

  public JvmOperation compileCacheConditionMethod(final Cachable cachable) {
    JvmOperation _xblockexpression = null;
    {
      final XExpression condition = this._xsemanticsUtils.cacheCondition(cachable);
      if ((condition == null)) {
        return null;
      }
      final Procedure1<JvmOperation> _function = (JvmOperation it) -> {
        it.setVisibility(JvmVisibility.PROTECTED);
        if ((cachable instanceof JudgmentDescription)) {
          EList<JvmFormalParameter> _parameters = it.getParameters();
          JvmFormalParameter _parameter = this._jvmTypesBuilder.toParameter(cachable, "environment", 
            this._typeReferenceBuilder.typeRef(RuleEnvironment.class));
          this._jvmTypesBuilder.<JvmFormalParameter>operator_add(_parameters, _parameter);
          EList<JvmFormalParameter> _parameters_1 = it.getParameters();
          List<JvmFormalParameter> _inputParameters = this.inputParameters(((JudgmentDescription)cachable));
          this._jvmTypesBuilder.<JvmFormalParameter>operator_add(_parameters_1, _inputParameters);
        } else {
          if ((cachable instanceof AuxiliaryDescription)) {
            EList<JvmFormalParameter> _parameters_2 = it.getParameters();
            List<JvmFormalParameter> _inputParameters_1 = this.inputParameters(((AuxiliaryDescription)cachable));
            this._jvmTypesBuilder.<JvmFormalParameter>operator_add(_parameters_2, _inputParameters_1);
          }
        }
        this._jvmTypesBuilder.setBody(it, condition);
      };
      _xblockexpression = this._jvmTypesBuilder.toMethod(cachable, 
        this._xsemanticsGeneratorExtensions.cacheConditionMethod(cachable).toString(), 
        this._typeReferences.getTypeForName(Boolean.class, cachable), _function);
    }
    return _xblockexpression;
  }

  public JvmOperation compileThrowExceptionMethod(final AuxiliaryDescription aux) {
    JvmOperation _xblockexpression = null;
    {
      final XExpression errorSpecification = aux.getError();
      final Procedure1<JvmOperation> _function = (JvmOperation it) -> {
        it.setVisibility(JvmVisibility.PROTECTED);
        EList<JvmTypeReference> _exceptions = it.getExceptions();
        JvmTypeReference _ruleFailedExceptionType = this.ruleFailedExceptionType();
        this._jvmTypesBuilder.<JvmTypeReference>operator_add(_exceptions, _ruleFailedExceptionType);
        EList<JvmFormalParameter> _parameters = it.getParameters();
        JvmFormalParameter _parameter = this._jvmTypesBuilder.toParameter(aux, "_error", this._typeReferenceBuilder.typeRef(String.class));
        this._jvmTypesBuilder.<JvmFormalParameter>operator_add(_parameters, _parameter);
        EList<JvmFormalParameter> _parameters_1 = it.getParameters();
        JvmFormalParameter _parameter_1 = this._jvmTypesBuilder.toParameter(aux, "_issue", this._typeReferenceBuilder.typeRef(String.class));
        this._jvmTypesBuilder.<JvmFormalParameter>operator_add(_parameters_1, _parameter_1);
        EList<JvmFormalParameter> _parameters_2 = it.getParameters();
        JvmFormalParameter _parameter_2 = this._jvmTypesBuilder.toParameter(aux, "_ex", this._xsemanticsGeneratorExtensions.exceptionType(aux));
        this._jvmTypesBuilder.<JvmFormalParameter>operator_add(_parameters_2, _parameter_2);
        EList<JvmFormalParameter> _parameters_3 = it.getParameters();
        List<JvmFormalParameter> _inputParameters = this.inputParameters(aux);
        this._jvmTypesBuilder.<JvmFormalParameter>operator_add(_parameters_3, _inputParameters);
        EList<JvmFormalParameter> _parameters_4 = it.getParameters();
        JvmFormalParameter _parameter_3 = this._jvmTypesBuilder.toParameter(aux, "_errorInformations", 
          this._jvmTypesBuilder.addArrayTypeDimension(this._typeReferenceBuilder.typeRef(ErrorInformation.class)));
        this._jvmTypesBuilder.<JvmFormalParameter>operator_add(_parameters_4, _parameter_3);
        this.compileErrorSpecification(it, errorSpecification);
      };
      _xblockexpression = this._jvmTypesBuilder.toMethod(aux, 
        this._xsemanticsGeneratorExtensions.throwExceptionMethod(aux).toString(), 
        this._typeReferences.getTypeForName(Void.TYPE, aux), _function);
    }
    return _xblockexpression;
  }

  public JvmOperation compileInternalMethod(final JudgmentDescription judgmentDescription) {
    JvmOperation _xblockexpression = null;
    {
      final String methodName = this._xsemanticsGeneratorExtensions.entryPointInternalMethodName(judgmentDescription).toString();
      final JvmTypeReference resultType = this._xsemanticsGeneratorExtensions.resultType(judgmentDescription);
      final Procedure1<JvmOperation> _function = (JvmOperation it) -> {
        this.addProtectedAndOverride(it, judgmentDescription);
        EList<JvmFormalParameter> _parameters = it.getParameters();
        JvmFormalParameter _environmentParam = this.environmentParam(judgmentDescription);
        this._jvmTypesBuilder.<JvmFormalParameter>operator_add(_parameters, _environmentParam);
        EList<JvmFormalParameter> _parameters_1 = it.getParameters();
        JvmFormalParameter _ruleApplicationTraceParam = this.ruleApplicationTraceParam(judgmentDescription);
        this._jvmTypesBuilder.<JvmFormalParameter>operator_add(_parameters_1, _ruleApplicationTraceParam);
        EList<JvmFormalParameter> _parameters_2 = it.getParameters();
        List<JvmFormalParameter> _inputParameters = this.inputParameters(judgmentDescription);
        this._jvmTypesBuilder.<JvmFormalParameter>operator_add(_parameters_2, _inputParameters);
        final String inputArgs = this._xsemanticsGeneratorExtensions.inputArgs(judgmentDescription);
        final CharSequence exceptionName = this._xsemanticsGeneratorExtensions.exceptionVarName(judgmentDescription);
        CharSequence _polymorphicDispatcherField = this._xsemanticsGeneratorExtensions.polymorphicDispatcherField(judgmentDescription);
        CharSequence _additionalArgs = this._xsemanticsGeneratorExtensions.additionalArgs();
        StringConcatenation _builder = new StringConcatenation();
        _builder.append("sneakyThrowRuleFailedException(");
        _builder.append(exceptionName);
        _builder.append(");");
        _builder.newLineIfNotEmpty();
        _builder.append("return null;");
        final CharSequence usePolymorphicDispatch = this.polymorphicInvokeTryCacth(_polymorphicDispatcherField, _additionalArgs, inputArgs, exceptionName, _builder);
        CachedClause _cachedClause = judgmentDescription.getCachedClause();
        boolean _tripleNotEquals = (_cachedClause != null);
        if (_tripleNotEquals) {
          StringConcatenationClient _client = new StringConcatenationClient() {
            @Override
            protected void appendTo(StringConcatenationClient.TargetStringConcatenation _builder) {
              StringConcatenation _builder_1 = new StringConcatenation();
              CharSequence _environmentName = XsemanticsJvmModelInferrer.this._xsemanticsGeneratorExtensions.environmentName();
              _builder_1.append(_environmentName);
              _builder_1.append(", ");
              _builder_1.append(inputArgs);
              CharSequence _bodyForCache = XsemanticsJvmModelInferrer.this.bodyForCache(judgmentDescription, _builder_1, usePolymorphicDispatch);
              _builder.append(_bodyForCache);
              _builder.newLineIfNotEmpty();
              StringConcatenationClient _wrapInGetFromCache = XsemanticsJvmModelInferrer.this.wrapInGetFromCache(methodName, resultType, inputArgs, usePolymorphicDispatch);
              _builder.append(_wrapInGetFromCache);
            }
          };
          this._jvmTypesBuilder.setBody(it, _client);
        } else {
          final Procedure1<ITreeAppendable> _function_1 = (ITreeAppendable it_1) -> {
            it_1.append(usePolymorphicDispatch);
          };
          this._jvmTypesBuilder.setBody(it, _function_1);
        }
      };
      _xblockexpression = this._jvmTypesBuilder.toMethod(judgmentDescription, methodName, resultType, _function);
    }
    return _xblockexpression;
  }

  public JvmOperation compileInternalMethod(final AuxiliaryDescription aux) {
    JvmOperation _xblockexpression = null;
    {
      final String methodName = this._xsemanticsGeneratorExtensions.entryPointInternalMethodName(aux).toString();
      final JvmTypeReference resultType = this._xsemanticsGeneratorExtensions.resultType(aux);
      final Procedure1<JvmOperation> _function = (JvmOperation it) -> {
        this.addProtectedAndOverride(it, aux);
        EList<JvmFormalParameter> _parameters = it.getParameters();
        JvmFormalParameter _ruleApplicationTraceParam = this.ruleApplicationTraceParam(aux);
        this._jvmTypesBuilder.<JvmFormalParameter>operator_add(_parameters, _ruleApplicationTraceParam);
        EList<JvmFormalParameter> _parameters_1 = aux.getParameters();
        for (final JvmFormalParameter p : _parameters_1) {
          EList<JvmFormalParameter> _parameters_2 = it.getParameters();
          JvmFormalParameter _parameter = this._jvmTypesBuilder.toParameter(p, p.getName(), p.getParameterType());
          this._jvmTypesBuilder.<JvmFormalParameter>operator_add(_parameters_2, _parameter);
        }
        final boolean isBoolean = this.isBoolean(aux.getType());
        final CharSequence exceptionName = this._xsemanticsGeneratorExtensions.exceptionVarName(aux);
        final String inputArgs = this._xsemanticsGeneratorExtensions.inputArgs(aux);
        CharSequence _polymorphicDispatcherField = this._xsemanticsGeneratorExtensions.polymorphicDispatcherField(aux);
        CharSequence _ruleApplicationTraceName = this._xsemanticsGeneratorExtensions.ruleApplicationTraceName();
        StringConcatenation _builder = new StringConcatenation();
        {
          if (isBoolean) {
            _builder.append("return false;");
            _builder.newLine();
          } else {
            _builder.append("sneakyThrowRuleFailedException(");
            _builder.append(exceptionName);
            _builder.append(");");
            _builder.newLineIfNotEmpty();
            _builder.append("return ");
            {
              JvmTypeReference _type = aux.getType();
              boolean _tripleEquals = (_type == null);
              if (_tripleEquals) {
                _builder.append("false");
              } else {
                _builder.append("null");
              }
            }
            _builder.append(";");
            _builder.newLineIfNotEmpty();
          }
        }
        final CharSequence invokePolymorphicDispatcher = this.polymorphicInvokeTryCacth(_polymorphicDispatcherField, _ruleApplicationTraceName, inputArgs, exceptionName, _builder);
        CachedClause _cachedClause = aux.getCachedClause();
        boolean _tripleNotEquals = (_cachedClause != null);
        if (_tripleNotEquals) {
          StringConcatenationClient _client = new StringConcatenationClient() {
            @Override
            protected void appendTo(StringConcatenationClient.TargetStringConcatenation _builder) {
              CharSequence _bodyForCache = XsemanticsJvmModelInferrer.this.bodyForCache(aux, inputArgs, invokePolymorphicDispatcher);
              _builder.append(_bodyForCache);
              _builder.newLineIfNotEmpty();
              StringConcatenationClient _wrapInGetFromCache2 = XsemanticsJvmModelInferrer.this.wrapInGetFromCache2(methodName, resultType, inputArgs, invokePolymorphicDispatcher);
              _builder.append(_wrapInGetFromCache2);
            }
          };
          this._jvmTypesBuilder.setBody(it, _client);
        } else {
          final Procedure1<ITreeAppendable> _function_1 = (ITreeAppendable it_1) -> {
            it_1.append(invokePolymorphicDispatcher);
          };
          this._jvmTypesBuilder.setBody(it, _function_1);
        }
      };
      _xblockexpression = this._jvmTypesBuilder.toMethod(aux, methodName, resultType, _function);
    }
    return _xblockexpression;
  }

  private CharSequence polymorphicInvokeTryCacth(final CharSequence polymorphicDispatcherField, final CharSequence additionalArgs, final CharSequence inputArgs, final CharSequence exceptionName, final CharSequence catchBlock) {
    StringConcatenation _builder = new StringConcatenation();
    _builder.append("try {");
    _builder.newLine();
    _builder.append("\t");
    _builder.append("checkParamsNotNull(");
    _builder.append(inputArgs, "\t");
    _builder.append(");");
    _builder.newLineIfNotEmpty();
    _builder.append("\t");
    _builder.append("return ");
    _builder.append(polymorphicDispatcherField, "\t");
    _builder.append(".invoke(");
    _builder.append(additionalArgs, "\t");
    _builder.append(", ");
    _builder.append(inputArgs, "\t");
    _builder.append(");");
    _builder.newLineIfNotEmpty();
    _builder.append("} catch (");
    String _simpleName = Exception.class.getSimpleName();
    _builder.append(_simpleName);
    _builder.append(" ");
    _builder.append(exceptionName);
    _builder.append(") {");
    _builder.newLineIfNotEmpty();
    _builder.append("\t");
    _builder.append(catchBlock, "\t");
    _builder.newLineIfNotEmpty();
    _builder.append("}");
    return _builder;
  }

  public JvmTypeReference ruleFailedExceptionType() {
    return this._typeReferenceBuilder.typeRef(RuleFailedException.class);
  }

  public JvmTypeReference environmentType() {
    return this._typeReferenceBuilder.typeRef(RuleEnvironment.class);
  }

  public JvmOperation compileImplMethod(final Rule rule) {
    JvmOperation _xblockexpression = null;
    {
      final JudgmentDescription judgment = this._xsemanticsUtils.getJudgmentDescription(rule);
      final JvmTypeReference resultType = this._xsemanticsGeneratorExtensions.resultType(judgment);
      StringConcatenation _builder = new StringConcatenation();
      CharSequence _polymorphicDispatcherImpl = this._xsemanticsGeneratorExtensions.polymorphicDispatcherImpl(judgment);
      _builder.append(_polymorphicDispatcherImpl);
      final Procedure1<JvmOperation> _function = (JvmOperation it) -> {
        this.setupMethodForRule(it, rule);
        final String exceptionVarName = this._xsemanticsGeneratorExtensions.exceptionVarName(rule);
        final String inputArgs = this._xsemanticsGeneratorExtensions.inputParameterNames(rule);
        final CharSequence applyRuleName = this._xsemanticsGeneratorExtensions.applyRuleName(rule);
        CharSequence _additionalArgsForRule = this._xsemanticsGeneratorExtensions.additionalArgsForRule(rule);
        String _traceStringForRule = this._xsemanticsGeneratorExtensions.traceStringForRule(rule);
        StringConcatenationClient _client = new StringConcatenationClient() {
          @Override
          protected void appendTo(StringConcatenationClient.TargetStringConcatenation _builder) {
            {
              XExpression _error = rule.getConclusion().getError();
              boolean _tripleNotEquals = (_error != null);
              if (_tripleNotEquals) {
                CharSequence _throwExceptionMethod = XsemanticsJvmModelInferrer.this._xsemanticsGeneratorExtensions.throwExceptionMethod(rule);
                _builder.append(_throwExceptionMethod);
                _builder.append("(");
                _builder.append(exceptionVarName);
                _builder.append(", ");
                _builder.append(inputArgs);
                _builder.append(");");
                _builder.newLineIfNotEmpty();
              } else {
                CharSequence _throwExceptionMethod_1 = XsemanticsJvmModelInferrer.this._xsemanticsGeneratorExtensions.throwExceptionMethod(judgment);
                _builder.append(_throwExceptionMethod_1);
                _builder.append("(");
                String _errorForRule = XsemanticsJvmModelInferrer.this._xsemanticsGeneratorExtensions.errorForRule(rule);
                _builder.append(_errorForRule);
                _builder.append(",");
                _builder.newLineIfNotEmpty();
                _builder.append("\t");
                String _ruleIssueString = XsemanticsJvmModelInferrer.this._xsemanticsGeneratorExtensions.ruleIssueString(rule);
                _builder.append(_ruleIssueString, "\t");
                _builder.append(",");
                _builder.newLineIfNotEmpty();
                _builder.append("\t");
                _builder.append("e_");
                _builder.append(applyRuleName, "\t");
                _builder.append(", ");
                _builder.append(inputArgs, "\t");
                StringConcatenationClient _errorInformationArgs = XsemanticsJvmModelInferrer.this.errorInformationArgs(rule);
                _builder.append(_errorInformationArgs, "\t");
                _builder.append(");");
                _builder.newLineIfNotEmpty();
              }
            }
            _builder.append("return null;");
            _builder.newLine();
          }
        };
        this._jvmTypesBuilder.setBody(it, this.bodyForImplMethod(resultType, applyRuleName, _additionalArgsForRule, inputArgs, _traceStringForRule, exceptionVarName, _client));
      };
      _xblockexpression = this._jvmTypesBuilder.toMethod(rule, 
        _builder.toString(), 
        this._xsemanticsGeneratorExtensions.resultType(judgment), _function);
    }
    return _xblockexpression;
  }

  private void setupMethodForRule(final JvmOperation it, final Rule rule) {
    this.addProtectedAndOverride(it, rule);
    EList<JvmTypeReference> _exceptions = it.getExceptions();
    JvmTypeReference _ruleFailedExceptionType = this.ruleFailedExceptionType();
    this._jvmTypesBuilder.<JvmTypeReference>operator_add(_exceptions, _ruleFailedExceptionType);
    EList<JvmFormalParameter> _parameters = it.getParameters();
    JvmFormalParameter _paramForEnvironment = this.paramForEnvironment(rule);
    this._jvmTypesBuilder.<JvmFormalParameter>operator_add(_parameters, _paramForEnvironment);
    EList<JvmFormalParameter> _parameters_1 = it.getParameters();
    JvmFormalParameter _ruleApplicationTraceParam = this.ruleApplicationTraceParam(rule);
    this._jvmTypesBuilder.<JvmFormalParameter>operator_add(_parameters_1, _ruleApplicationTraceParam);
    EList<JvmFormalParameter> _parameters_2 = it.getParameters();
    List<JvmFormalParameter> _inputParameters = this.inputParameters(rule);
    this._jvmTypesBuilder.<JvmFormalParameter>operator_add(_parameters_2, _inputParameters);
  }

  public JvmOperation compileImplMethod(final AuxiliaryFunction aux) {
    JvmOperation _xblockexpression = null;
    {
      final JvmTypeReference resultType = this._xsemanticsGeneratorExtensions.resultType(aux);
      final boolean isBoolean = this.isBoolean(resultType);
      final AuxiliaryDescription auxiliaryDescription = this._xsemanticsUtils.getAuxiliaryDescription(aux);
      StringConcatenation _builder = new StringConcatenation();
      CharSequence _polymorphicDispatcherImpl = this._xsemanticsGeneratorExtensions.polymorphicDispatcherImpl(auxiliaryDescription);
      _builder.append(_polymorphicDispatcherImpl);
      final Procedure1<JvmOperation> _function = (JvmOperation it) -> {
        it.setVisibility(JvmVisibility.PROTECTED);
        EList<JvmTypeReference> _exceptions = it.getExceptions();
        JvmTypeReference _ruleFailedExceptionType = this.ruleFailedExceptionType();
        this._jvmTypesBuilder.<JvmTypeReference>operator_add(_exceptions, _ruleFailedExceptionType);
        EList<JvmFormalParameter> _parameters = it.getParameters();
        JvmFormalParameter _ruleApplicationTraceParam = this.ruleApplicationTraceParam(auxiliaryDescription);
        this._jvmTypesBuilder.<JvmFormalParameter>operator_add(_parameters, _ruleApplicationTraceParam);
        EList<JvmFormalParameter> _parameters_1 = it.getParameters();
        List<JvmFormalParameter> _inputParameters = this.inputParameters(aux);
        this._jvmTypesBuilder.<JvmFormalParameter>operator_add(_parameters_1, _inputParameters);
        final String inputArgs = this._xsemanticsGeneratorExtensions.inputParameterNames(aux);
        final CharSequence applyName = this._xsemanticsGeneratorExtensions.applyAuxFunName(aux);
        final String exceptionVarName = this._xsemanticsGeneratorExtensions.exceptionVarName(aux);
        CharSequence _ruleApplicationSubtraceName = this._xsemanticsGeneratorExtensions.ruleApplicationSubtraceName();
        String _traceStringForAuxiliaryFun = this._xsemanticsGeneratorExtensions.traceStringForAuxiliaryFun(aux);
        StringConcatenationClient _client = new StringConcatenationClient() {
          @Override
          protected void appendTo(StringConcatenationClient.TargetStringConcatenation _builder) {
            CharSequence _throwExceptionMethod = XsemanticsJvmModelInferrer.this._xsemanticsGeneratorExtensions.throwExceptionMethod(auxiliaryDescription);
            _builder.append(_throwExceptionMethod);
            _builder.append("(");
            String _errorForAuxiliaryFun = XsemanticsJvmModelInferrer.this._xsemanticsGeneratorExtensions.errorForAuxiliaryFun(aux);
            _builder.append(_errorForAuxiliaryFun);
            _builder.append(",");
            _builder.newLineIfNotEmpty();
            _builder.append("\t");
            String _ruleIssueString = XsemanticsJvmModelInferrer.this._xsemanticsGeneratorExtensions.ruleIssueString(auxiliaryDescription);
            _builder.append(_ruleIssueString, "\t");
            _builder.append(",");
            _builder.newLineIfNotEmpty();
            _builder.append("\t");
            _builder.append(exceptionVarName, "\t");
            _builder.append(", ");
            _builder.append(inputArgs, "\t");
            StringConcatenationClient _errorInformationArgs = XsemanticsJvmModelInferrer.this.errorInformationArgs(aux);
            _builder.append(_errorInformationArgs, "\t");
            _builder.append(");");
            _builder.newLineIfNotEmpty();
            _builder.append("return ");
            {
              if (isBoolean) {
                _builder.append("false");
              } else {
                _builder.append("null");
              }
            }
            _builder.append(";");
            _builder.newLineIfNotEmpty();
          }
        };
        this._jvmTypesBuilder.setBody(it, this.bodyForImplMethod(resultType, applyName, _ruleApplicationSubtraceName, inputArgs, _traceStringForAuxiliaryFun, exceptionVarName, _client));
      };
      _xblockexpression = this._jvmTypesBuilder.toMethod(aux, 
        _builder.toString(), resultType, _function);
    }
    return _xblockexpression;
  }

  private StringConcatenationClient bodyForImplMethod(final JvmTypeReference resultType, final CharSequence applyRuleName, final CharSequence additionalArgs, final CharSequence inputArgs, final CharSequence traceString, final CharSequence exceptionVarName, final StringConcatenationClient catchBlock) {
    StringConcatenationClient _client = new StringConcatenationClient() {
      @Override
      protected void appendTo(StringConcatenationClient.TargetStringConcatenation _builder) {
        _builder.append("try {");
        _builder.newLine();
        _builder.append("\t");
        _builder.append("final ");
        _builder.append(RuleApplicationTrace.class, "\t");
        _builder.append(" ");
        CharSequence _ruleApplicationSubtraceName = XsemanticsJvmModelInferrer.this._xsemanticsGeneratorExtensions.ruleApplicationSubtraceName();
        _builder.append(_ruleApplicationSubtraceName, "\t");
        _builder.append(" = ");
        CharSequence _newTraceMethod = XsemanticsJvmModelInferrer.this._xsemanticsGeneratorExtensions.newTraceMethod(XsemanticsJvmModelInferrer.this._xsemanticsGeneratorExtensions.ruleApplicationTraceName());
        _builder.append(_newTraceMethod, "\t");
        _builder.append(";");
        _builder.newLineIfNotEmpty();
        _builder.append("\t");
        _builder.append("final ");
        _builder.append(resultType, "\t");
        _builder.append(" ");
        CharSequence _resultVariableForTrace = XsemanticsJvmModelInferrer.this._xsemanticsGeneratorExtensions.resultVariableForTrace();
        _builder.append(_resultVariableForTrace, "\t");
        _builder.append(" = ");
        _builder.append(applyRuleName, "\t");
        _builder.append("(");
        _builder.append(additionalArgs, "\t");
        _builder.append(", ");
        _builder.append(inputArgs, "\t");
        _builder.append(");");
        _builder.newLineIfNotEmpty();
        _builder.append("\t");
        StringConcatenationClient _addToTraceMethod = XsemanticsJvmModelInferrer.this._xsemanticsGeneratorExtensions.addToTraceMethod(XsemanticsJvmModelInferrer.this._xsemanticsGeneratorExtensions.ruleApplicationTraceName(), traceString);
        _builder.append(_addToTraceMethod, "\t");
        _builder.append(";");
        _builder.newLineIfNotEmpty();
        _builder.append("\t");
        CharSequence _addAsSubtraceMethod = XsemanticsJvmModelInferrer.this._xsemanticsGeneratorExtensions.addAsSubtraceMethod(XsemanticsJvmModelInferrer.this._xsemanticsGeneratorExtensions.ruleApplicationTraceName(), XsemanticsJvmModelInferrer.this._xsemanticsGeneratorExtensions.ruleApplicationSubtraceName());
        _builder.append(_addAsSubtraceMethod, "\t");
        _builder.append(";");
        _builder.newLineIfNotEmpty();
        _builder.append("\t");
        _builder.append("return ");
        CharSequence _resultVariableForTrace_1 = XsemanticsJvmModelInferrer.this._xsemanticsGeneratorExtensions.resultVariableForTrace();
        _builder.append(_resultVariableForTrace_1, "\t");
        _builder.append(";");
        _builder.newLineIfNotEmpty();
        _builder.append("} catch (");
        _builder.append(Exception.class);
        _builder.append(" ");
        _builder.append(exceptionVarName);
        _builder.append(") {");
        _builder.newLineIfNotEmpty();
        _builder.append("\t");
        _builder.append(catchBlock, "\t");
        _builder.newLineIfNotEmpty();
        _builder.append("}");
        _builder.newLine();
      }
    };
    return _client;
  }

  public StringConcatenationClient errorInformationArgs(final Rule rule) {
    final Function1<RuleParameter, String> _function = (RuleParameter it) -> {
      return it.getParameter().getName();
    };
    return this.errorInformationArgs(IterableExtensions.<RuleParameter, String>map(this._xsemanticsGeneratorExtensions.inputEObjectParams(rule), _function));
  }

  public StringConcatenationClient errorInformationArgs(final AuxiliaryFunction aux) {
    final Function1<JvmFormalParameter, String> _function = (JvmFormalParameter it) -> {
      return it.getName();
    };
    return this.errorInformationArgs(IterableExtensions.<JvmFormalParameter, String>map(this._xsemanticsGeneratorExtensions.inputEObjectParams(aux), _function));
  }

  public StringConcatenationClient errorInformationArgs(final Iterable<String> names) {
    StringConcatenationClient _client = new StringConcatenationClient() {
      @Override
      protected void appendTo(StringConcatenationClient.TargetStringConcatenation _builder) {
        _builder.append(", new ");
        _builder.append(ErrorInformation.class);
        _builder.append("[] {");
        {
          boolean _hasElements = false;
          for(final String name : names) {
            if (!_hasElements) {
              _hasElements = true;
            } else {
              _builder.appendImmediate(", ", "");
            }
            _builder.append("new ");
            _builder.append(ErrorInformation.class);
            _builder.append("(");
            _builder.append(name);
            _builder.append(")");
          }
        }
        _builder.append("}");
      }
    };
    return _client;
  }

  public JvmOperation compileApplyMethod(final Rule rule) {
    final Procedure1<JvmOperation> _function = (JvmOperation it) -> {
      this.setupMethodForRule(it, rule);
      this.assignBody(it, rule);
    };
    return this._jvmTypesBuilder.toMethod(rule, 
      this._xsemanticsGeneratorExtensions.applyRuleName(rule).toString(), 
      this._xsemanticsGeneratorExtensions.resultType(this._xsemanticsUtils.getJudgmentDescription(rule)), _function);
  }

  protected void _assignBody(final JvmExecutable logicalContainer, final Rule rule) {
    final Procedure1<ITreeAppendable> _function = (ITreeAppendable it) -> {
      this._xsemanticsGeneratorExtensions.declareVariablesForOutputParams(rule, it);
      this._xsemanticsGeneratorExtensions.compileReturnResult(rule, this._xsemanticsGeneratorExtensions.resultType(this._xsemanticsUtils.getJudgmentDescription(rule)), it);
    };
    this._jvmTypesBuilder.setBody(logicalContainer, _function);
  }

  protected void _assignBody(final JvmExecutable logicalContainer, final RuleWithPremises rule) {
    this._jvmTypesBuilder.setBody(logicalContainer, rule.getPremises());
  }

  public JvmOperation compileExpressionInConclusionMethod(final ExpressionInConclusion e) {
    final Procedure1<JvmOperation> _function = (JvmOperation it) -> {
      it.setVisibility(JvmVisibility.PRIVATE);
      final Rule rule = this._xsemanticsUtils.containingRule(e);
      EList<JvmTypeReference> _exceptions = it.getExceptions();
      JvmTypeReference _ruleFailedExceptionType = this.ruleFailedExceptionType();
      this._jvmTypesBuilder.<JvmTypeReference>operator_add(_exceptions, _ruleFailedExceptionType);
      EList<JvmFormalParameter> _parameters = it.getParameters();
      JvmFormalParameter _paramForEnvironment = this.paramForEnvironment(rule);
      this._jvmTypesBuilder.<JvmFormalParameter>operator_add(_parameters, _paramForEnvironment);
      EList<JvmFormalParameter> _parameters_1 = it.getParameters();
      List<JvmFormalParameter> _inputParameters = this.inputParameters(rule);
      this._jvmTypesBuilder.<JvmFormalParameter>operator_add(_parameters_1, _inputParameters);
      this._jvmTypesBuilder.setBody(it, e.getExpression());
    };
    return this._jvmTypesBuilder.toMethod(e, 
      this._xsemanticsGeneratorExtensions.expressionInConclusionMethodName(e), 
      this._jvmTypesBuilder.inferredType(e.getExpression()), _function);
  }

  public JvmOperation compileErrorSpecificationMethod(final Rule rule) {
    JvmOperation _xblockexpression = null;
    {
      final XExpression errSpec = rule.getConclusion().getError();
      final Procedure1<JvmOperation> _function = (JvmOperation it) -> {
        it.setVisibility(JvmVisibility.PRIVATE);
        EList<JvmTypeReference> _exceptions = it.getExceptions();
        JvmTypeReference _ruleFailedExceptionType = this.ruleFailedExceptionType();
        this._jvmTypesBuilder.<JvmTypeReference>operator_add(_exceptions, _ruleFailedExceptionType);
        EList<JvmFormalParameter> _parameters = it.getParameters();
        JvmFormalParameter _parameter = this._jvmTypesBuilder.toParameter(rule, this._xsemanticsGeneratorExtensions.exceptionVarName(rule), 
          this._xsemanticsGeneratorExtensions.exceptionType(rule));
        this._jvmTypesBuilder.<JvmFormalParameter>operator_add(_parameters, _parameter);
        EList<JvmFormalParameter> _parameters_1 = it.getParameters();
        List<JvmFormalParameter> _inputParameters = this.inputParameters(rule);
        this._jvmTypesBuilder.<JvmFormalParameter>operator_add(_parameters_1, _inputParameters);
        this._jvmTypesBuilder.setBody(it, errSpec);
      };
      _xblockexpression = this._jvmTypesBuilder.toMethod(errSpec, 
        this._xsemanticsGeneratorExtensions.throwExceptionMethod(rule).toString(), 
        this._typeReferences.getTypeForName(Void.TYPE, rule), _function);
    }
    return _xblockexpression;
  }

  public JvmOperation compileApplyAuxiliaryFunction(final AuxiliaryFunction auxfun) {
    final Procedure1<JvmOperation> _function = (JvmOperation it) -> {
      it.setVisibility(JvmVisibility.PROTECTED);
      EList<JvmTypeReference> _exceptions = it.getExceptions();
      JvmTypeReference _ruleFailedExceptionType = this.ruleFailedExceptionType();
      this._jvmTypesBuilder.<JvmTypeReference>operator_add(_exceptions, _ruleFailedExceptionType);
      EList<JvmFormalParameter> _parameters = it.getParameters();
      JvmFormalParameter _ruleApplicationTraceParam = this.ruleApplicationTraceParam(auxfun);
      this._jvmTypesBuilder.<JvmFormalParameter>operator_add(_parameters, _ruleApplicationTraceParam);
      EList<JvmFormalParameter> _parameters_1 = it.getParameters();
      List<JvmFormalParameter> _inputParameters = this.inputParameters(auxfun);
      this._jvmTypesBuilder.<JvmFormalParameter>operator_add(_parameters_1, _inputParameters);
      this._jvmTypesBuilder.setBody(it, auxfun.getBody());
    };
    return this._jvmTypesBuilder.toMethod(auxfun, 
      this._xsemanticsGeneratorExtensions.applyAuxFunName(auxfun).toString(), 
      this._xsemanticsGeneratorExtensions.resultType(auxfun), _function);
  }

  public ArrayList<JvmOperation> compileCheckRuleMethods(final CheckRule rule) {
    ArrayList<JvmOperation> _xblockexpression = null;
    {
      final ArrayList<JvmOperation> checkMethods = CollectionLiterals.<JvmOperation>newArrayList();
      final String methodName = this._xsemanticsGeneratorExtensions.methodName(rule);
      final JvmTypeReference resultType = this._xsemanticsGeneratorExtensions.resultType(rule);
      StringConcatenation _builder = new StringConcatenation();
      _builder.append(methodName);
      final Procedure1<JvmOperation> _function = (JvmOperation it) -> {
        this.setupMethodForCheckRule(it, rule);
        StringConcatenationClient _client = new StringConcatenationClient() {
          @Override
          protected void appendTo(StringConcatenationClient.TargetStringConcatenation _builder) {
            _builder.append("return ");
            _builder.append(methodName);
            _builder.append("(null, ");
            String _name = rule.getElement().getParameter().getName();
            _builder.append(_name);
            _builder.append(");");
          }
        };
        this._jvmTypesBuilder.setBody(it, _client);
      };
      JvmOperation _method = this._jvmTypesBuilder.toMethod(rule, _builder.toString(), resultType, _function);
      checkMethods.add(_method);
      StringConcatenation _builder_1 = new StringConcatenation();
      _builder_1.append(methodName);
      final Procedure1<JvmOperation> _function_1 = (JvmOperation it) -> {
        EList<JvmFormalParameter> _parameters = it.getParameters();
        JvmFormalParameter _ruleApplicationTraceParam = this.ruleApplicationTraceParam(rule);
        this._jvmTypesBuilder.<JvmFormalParameter>operator_add(_parameters, _ruleApplicationTraceParam);
        this.setupMethodForCheckRule(it, rule);
        final CharSequence exceptionName = this._xsemanticsGeneratorExtensions.exceptionVarName(rule);
        StringConcatenationClient _client = new StringConcatenationClient() {
          @Override
          protected void appendTo(StringConcatenationClient.TargetStringConcatenation _builder) {
            _builder.append("try {");
            _builder.newLine();
            _builder.append("\t");
            _builder.append("return ");
            _builder.append(methodName, "\t");
            _builder.append("Internal(");
            String _string = XsemanticsJvmModelInferrer.this._xsemanticsGeneratorExtensions.ruleApplicationTraceName().toString();
            _builder.append(_string, "\t");
            _builder.append(", ");
            String _name = rule.getElement().getParameter().getName();
            _builder.append(_name, "\t");
            _builder.append(");");
            _builder.newLineIfNotEmpty();
            _builder.append("} catch (");
            _builder.append(Exception.class);
            _builder.append(" ");
            _builder.append(exceptionName);
            _builder.append(") {");
            _builder.newLineIfNotEmpty();
            _builder.append("\t");
            _builder.append("return resultForFailure(");
            _builder.append(exceptionName, "\t");
            _builder.append(");");
            _builder.newLineIfNotEmpty();
            _builder.append("}");
          }
        };
        this._jvmTypesBuilder.setBody(it, _client);
      };
      JvmOperation _method_1 = this._jvmTypesBuilder.toMethod(rule, _builder_1.toString(), resultType, _function_1);
      checkMethods.add(_method_1);
      _xblockexpression = checkMethods;
    }
    return _xblockexpression;
  }

  public JvmOperation compileValidatorCheckRuleMethod(final CheckRule rule) {
    JvmOperation _xblockexpression = null;
    {
      final String methodName = this._xsemanticsGeneratorExtensions.methodName(rule);
      StringConcatenation _builder = new StringConcatenation();
      _builder.append(methodName);
      final Procedure1<JvmOperation> _function = (JvmOperation it) -> {
        this.setupMethodForCheckRule(it, rule);
        EList<JvmAnnotationReference> _annotations = it.getAnnotations();
        JvmAnnotationReference _annotationRef = this._annotationTypesBuilder.annotationRef(Check.class);
        this._jvmTypesBuilder.<JvmAnnotationReference>operator_add(_annotations, _annotationRef);
        StringConcatenationClient _client = new StringConcatenationClient() {
          @Override
          protected void appendTo(StringConcatenationClient.TargetStringConcatenation _builder) {
            _builder.append("errorGenerator.generateErrors(this,");
            _builder.newLine();
            _builder.append("\t");
            _builder.append("getXsemanticsSystem().");
            _builder.append(methodName, "\t");
            _builder.append("(");
            String _name = rule.getElement().getParameter().getName();
            _builder.append(_name, "\t");
            _builder.append("),");
            _builder.newLineIfNotEmpty();
            _builder.append("\t\t");
            String _name_1 = rule.getElement().getParameter().getName();
            _builder.append(_name_1, "\t\t");
            _builder.append(");");
          }
        };
        this._jvmTypesBuilder.setBody(it, _client);
      };
      _xblockexpression = this._jvmTypesBuilder.toMethod(rule, _builder.toString(), 
        this._typeReferences.getTypeForName(Void.TYPE, rule), _function);
    }
    return _xblockexpression;
  }

  public JvmOperation compileInternalMethod(final CheckRule rule) {
    JvmOperation _xblockexpression = null;
    {
      final String methodName = this._xsemanticsGeneratorExtensions.methodName(rule);
      StringConcatenation _builder = new StringConcatenation();
      _builder.append(methodName);
      _builder.append("Internal");
      final Procedure1<JvmOperation> _function = (JvmOperation it) -> {
        it.setVisibility(JvmVisibility.PROTECTED);
        EList<JvmTypeReference> _exceptions = it.getExceptions();
        JvmTypeReference _ruleFailedExceptionType = this.ruleFailedExceptionType();
        this._jvmTypesBuilder.<JvmTypeReference>operator_add(_exceptions, _ruleFailedExceptionType);
        EList<JvmFormalParameter> _parameters = it.getParameters();
        JvmFormalParameter _ruleApplicationTraceParam = this.ruleApplicationTraceParam(rule);
        this._jvmTypesBuilder.<JvmFormalParameter>operator_add(_parameters, _ruleApplicationTraceParam);
        this.setupMethodForCheckRule(it, rule);
        this._jvmTypesBuilder.setBody(it, rule.getPremises());
      };
      _xblockexpression = this._jvmTypesBuilder.toMethod(rule, _builder.toString(), 
        this._xsemanticsGeneratorExtensions.resultType(rule), _function);
    }
    return _xblockexpression;
  }

  private boolean setupMethodForCheckRule(final JvmOperation it, final CheckRule rule) {
    boolean _xblockexpression = false;
    {
      this.addOverrideAnnotation(it, rule);
      EList<JvmFormalParameter> _parameters = it.getParameters();
      JvmFormalParameter _parameter = this._jvmTypesBuilder.toParameter(rule.getElement().getParameter(), rule.getElement().getParameter().getName(), 
        rule.getElement().getParameter().getParameterType());
      _xblockexpression = this._jvmTypesBuilder.<JvmFormalParameter>operator_add(_parameters, _parameter);
    }
    return _xblockexpression;
  }

  public JvmFormalParameter paramForEnvironment(final Rule rule) {
    return this._jvmTypesBuilder.toParameter(rule, this._xsemanticsGeneratorExtensions.ruleEnvName(rule), this._typeReferenceBuilder.typeRef(RuleEnvironment.class));
  }

  public List<JvmFormalParameter> inputParameters(final Rule rule) {
    final Function1<RuleParameter, JvmFormalParameter> _function = (RuleParameter it) -> {
      return this._jvmTypesBuilder.toParameter(it, 
        it.getParameter().getName(), 
        it.getParameter().getParameterType());
    };
    return ListExtensions.<RuleParameter, JvmFormalParameter>map(this._xsemanticsUtils.inputParams(rule), _function);
  }

  public boolean isBoolean(final JvmTypeReference typeRef) {
    boolean _and = false;
    if (!(typeRef != null)) {
      _and = false;
    } else {
      boolean _xblockexpression = false;
      {
        final String identifier = typeRef.getType().getIdentifier();
        _xblockexpression = ("boolean".equals(identifier) || 
          "java.lang.Boolean".equals(identifier));
      }
      _and = _xblockexpression;
    }
    return _and;
  }

  private boolean addProtectedAndOverride(final JvmOperation it, final Overrider overrider) {
    boolean _xblockexpression = false;
    {
      it.setVisibility(JvmVisibility.PROTECTED);
      _xblockexpression = this.addOverrideAnnotation(it, overrider);
    }
    return _xblockexpression;
  }

  private boolean addOverrideAnnotation(final JvmAnnotationTarget it, final Overrider overrider) {
    boolean _xifexpression = false;
    boolean _isOverride = overrider.isOverride();
    if (_isOverride) {
      _xifexpression = this.addOverrideAnnotation(it);
    }
    return _xifexpression;
  }

  private boolean addOverrideAnnotation(final JvmAnnotationTarget it) {
    EList<JvmAnnotationReference> _annotations = it.getAnnotations();
    JvmAnnotationReference _annotationRef = this._annotationTypesBuilder.annotationRef(Override.class);
    return this._jvmTypesBuilder.<JvmAnnotationReference>operator_add(_annotations, _annotationRef);
  }

  private boolean addInjectAnnotation(final JvmAnnotationTarget it) {
    EList<JvmAnnotationReference> _annotations = it.getAnnotations();
    JvmAnnotationReference _annotationRef = this._annotationTypesBuilder.annotationRef(Inject.class);
    return this._jvmTypesBuilder.<JvmAnnotationReference>operator_add(_annotations, _annotationRef);
  }

  private boolean addExtensionAnnotation(final JvmAnnotationTarget it) {
    EList<JvmAnnotationReference> _annotations = it.getAnnotations();
    JvmAnnotationReference _annotationRef = this._annotationTypesBuilder.annotationRef(Extension.class);
    return this._jvmTypesBuilder.<JvmAnnotationReference>operator_add(_annotations, _annotationRef);
  }

  private void translateAnnotations(final JvmAnnotationTarget target, final List<XAnnotation> annotations) {
    final Function1<XAnnotation, Boolean> _function = (XAnnotation it) -> {
      JvmType _annotationType = it.getAnnotationType();
      return Boolean.valueOf((_annotationType != null));
    };
    this._jvmTypesBuilder.addAnnotations(target, IterableExtensions.<XAnnotation>filter(IterableExtensions.<XAnnotation>filterNull(annotations), _function));
  }

  public void infer(final EObject ts, final IJvmDeclaredTypeAcceptor acceptor, final boolean isPreIndexingPhase) {
    if (ts instanceof XsemanticsSystem) {
      _infer((XsemanticsSystem)ts, acceptor, isPreIndexingPhase);
      return;
    } else if (ts != null) {
      _infer(ts, acceptor, isPreIndexingPhase);
      return;
    } else {
      throw new IllegalArgumentException("Unhandled parameter types: " +
        Arrays.<Object>asList(ts, acceptor, isPreIndexingPhase).toString());
    }
  }

  public void assignBody(final JvmExecutable logicalContainer, final Rule rule) {
    if (rule instanceof RuleWithPremises) {
      _assignBody(logicalContainer, (RuleWithPremises)rule);
      return;
    } else if (rule != null) {
      _assignBody(logicalContainer, rule);
      return;
    } else {
      throw new IllegalArgumentException("Unhandled parameter types: " +
        Arrays.<Object>asList(logicalContainer, rule).toString());
    }
  }
}
