/**
 * Copyright (c) 2013-2017 Lorenzo Bettini.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *   Lorenzo Bettini - Initial contribution and API
 */
package org.eclipse.xsemantics.dsl.generator;

import com.google.common.base.Objects;
import com.google.inject.Inject;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Iterator;
import java.util.List;
import java.util.Set;
import org.eclipse.emf.common.util.EList;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.EStructuralFeature;
import org.eclipse.xsemantics.dsl.XsemanticsConstants;
import org.eclipse.xsemantics.dsl.typing.XsemanticsTypeSystem;
import org.eclipse.xsemantics.dsl.util.XsemanticsNodeModelUtils;
import org.eclipse.xsemantics.dsl.util.XsemanticsUtils;
import org.eclipse.xsemantics.dsl.xsemantics.AuxiliaryDescription;
import org.eclipse.xsemantics.dsl.xsemantics.AuxiliaryFunction;
import org.eclipse.xsemantics.dsl.xsemantics.CachedClause;
import org.eclipse.xsemantics.dsl.xsemantics.CheckRule;
import org.eclipse.xsemantics.dsl.xsemantics.Description;
import org.eclipse.xsemantics.dsl.xsemantics.EmptyEnvironment;
import org.eclipse.xsemantics.dsl.xsemantics.EnvironmentComposition;
import org.eclipse.xsemantics.dsl.xsemantics.EnvironmentMapping;
import org.eclipse.xsemantics.dsl.xsemantics.ErrorSpecification;
import org.eclipse.xsemantics.dsl.xsemantics.Fail;
import org.eclipse.xsemantics.dsl.xsemantics.JudgmentDescription;
import org.eclipse.xsemantics.dsl.xsemantics.OrExpression;
import org.eclipse.xsemantics.dsl.xsemantics.ReferToJudgment;
import org.eclipse.xsemantics.dsl.xsemantics.Rule;
import org.eclipse.xsemantics.dsl.xsemantics.RuleConclusion;
import org.eclipse.xsemantics.dsl.xsemantics.RuleInvocation;
import org.eclipse.xsemantics.dsl.xsemantics.RuleWithPremises;
import org.eclipse.xtend2.lib.StringConcatenation;
import org.eclipse.xtext.EcoreUtil2;
import org.eclipse.xtext.common.types.JvmExecutable;
import org.eclipse.xtext.common.types.JvmIdentifiableElement;
import org.eclipse.xtext.common.types.JvmOperation;
import org.eclipse.xtext.common.types.JvmTypeReference;
import org.eclipse.xtext.common.types.util.TypeReferences;
import org.eclipse.xtext.xbase.XAbstractFeatureCall;
import org.eclipse.xtext.xbase.XBlockExpression;
import org.eclipse.xtext.xbase.XClosure;
import org.eclipse.xtext.xbase.XExpression;
import org.eclipse.xtext.xbase.XMemberFeatureCall;
import org.eclipse.xtext.xbase.XVariableDeclaration;
import org.eclipse.xtext.xbase.XbasePackage;
import org.eclipse.xtext.xbase.compiler.Later;
import org.eclipse.xtext.xbase.compiler.XbaseCompiler;
import org.eclipse.xtext.xbase.compiler.output.ITreeAppendable;
import org.eclipse.xtext.xbase.lib.Extension;
import org.eclipse.xtext.xbase.typesystem.references.LightweightTypeReference;

@SuppressWarnings("all")
public class XsemanticsXbaseCompiler extends XbaseCompiler {
  @Inject
  @Extension
  private XsemanticsUtils _xsemanticsUtils;

  @Inject
  @Extension
  private XsemanticsGeneratorExtensions _xsemanticsGeneratorExtensions;

  @Inject
  @Extension
  private XsemanticsNodeModelUtils _xsemanticsNodeModelUtils;

  @Inject
  @Extension
  private XsemanticsTypeSystem typeSystem;

  @Inject
  private TypeReferences typeReferences;

  @Override
  public ITreeAppendable compile(final XExpression obj, final ITreeAppendable appendable, final JvmTypeReference expectedReturnType, final Set<JvmTypeReference> declaredExceptions) {
    final EObject rule = obj.eContainer();
    boolean _matched = false;
    if (rule instanceof RuleWithPremises) {
      _matched=true;
      this._xsemanticsGeneratorExtensions.declareVariablesForOutputParams(((Rule)rule), appendable);
      this.compileRuleBody(((RuleWithPremises)rule), this._xsemanticsGeneratorExtensions.resultType(this._xsemanticsUtils.getJudgmentDescription(((ReferToJudgment)rule))), appendable);
      return appendable;
    }
    if (!_matched) {
      if (rule instanceof CheckRule) {
        _matched=true;
        this.compilePremises(((CheckRule)rule), appendable);
        boolean _isEmpty = appendable.toString().isEmpty();
        boolean _not = (!_isEmpty);
        if (_not) {
          appendable.newLine();
        }
        appendable.append("return new ");
        this._xsemanticsGeneratorExtensions.resultType(((CheckRule)rule), appendable);
        appendable.append("(true);");
        return appendable;
      }
    }
    if (!_matched) {
      if (rule instanceof AuxiliaryFunction) {
        _matched=true;
        JvmTypeReference _type = this._xsemanticsUtils.getAuxiliaryDescription(((AuxiliaryFunction)rule)).getType();
        boolean _tripleNotEquals = (_type != null);
        if (_tripleNotEquals) {
          return super.compile(obj, appendable, expectedReturnType, declaredExceptions);
        }
        this.internalToJavaStatement(obj, appendable, false);
        appendable.newLine().append("return true;");
        return appendable;
      }
    }
    if ((obj instanceof ErrorSpecification)) {
      final String error = this.compileErrorOfErrorSpecification(((ErrorSpecification)obj), appendable);
      final String source = this.compileSourceOfErrorSpecification(((ErrorSpecification)obj), appendable);
      final String feature = this.compileFeatureOfErrorSpecification(((ErrorSpecification)obj), appendable);
      final String data = this.compileDataOfErrorSpecification(((ErrorSpecification)obj), appendable);
      appendable.newLine();
      this.compileFinalPartOfThrowExceptionMethod(((ErrorSpecification)obj).eContainer(), appendable, error, source, feature, data);
      return appendable;
    }
    return super.compile(obj, appendable, expectedReturnType, declaredExceptions);
  }

  public void compileRuleBody(final RuleWithPremises rule, final JvmTypeReference resultType, final ITreeAppendable result) {
    this.compilePremises(rule, result);
    this._xsemanticsGeneratorExtensions.compileReturnResult(rule, resultType, result);
  }

  public void compilePremises(final RuleWithPremises rule, final ITreeAppendable result) {
    this.toJavaStatement(rule.getPremises(), result, false);
  }

  public void compilePremises(final CheckRule rule, final ITreeAppendable result) {
    this.toJavaStatement(rule.getPremises(), result, false);
  }

  protected ITreeAppendable _compileFinalPartOfThrowExceptionMethod(final Description desc, final ITreeAppendable a, final String error, final String source, final String feature, final String data) {
    ITreeAppendable _xblockexpression = null;
    {
      StringConcatenation _builder = new StringConcatenation();
      CharSequence _throwRuleFailedExceptionMethod = this._xsemanticsGeneratorExtensions.throwRuleFailedExceptionMethod();
      _builder.append(_throwRuleFailedExceptionMethod);
      _builder.append("(");
      _builder.append(error);
      _builder.append(",");
      _builder.newLineIfNotEmpty();
      _builder.append("\t");
      _builder.append("_issue, _ex, new ");
      a.append(_builder);
      this.serialize(this._xsemanticsGeneratorExtensions.errorInformationType(desc), desc, a);
      _xblockexpression = this.compileArgsForErrorInformationConstructor(a, source, feature, data);
    }
    return _xblockexpression;
  }

  protected ITreeAppendable _compileFinalPartOfThrowExceptionMethod(final RuleConclusion ruleConclusion, final ITreeAppendable a, final String error, final String source, final String feature, final String data) {
    ITreeAppendable _xblockexpression = null;
    {
      final Rule rule = this._xsemanticsUtils.containingRule(ruleConclusion);
      StringConcatenation _builder = new StringConcatenation();
      CharSequence _throwRuleFailedExceptionMethod = this._xsemanticsGeneratorExtensions.throwRuleFailedExceptionMethod();
      _builder.append(_throwRuleFailedExceptionMethod);
      _builder.append("(");
      _builder.append(error);
      _builder.append(",");
      _builder.newLineIfNotEmpty();
      _builder.append("\t");
      String _ruleIssueString = this._xsemanticsGeneratorExtensions.ruleIssueString(rule);
      _builder.append(_ruleIssueString, "\t");
      _builder.append(", e_");
      CharSequence _applyRuleName = this._xsemanticsGeneratorExtensions.applyRuleName(rule);
      _builder.append(_applyRuleName, "\t");
      _builder.append(", new ");
      a.append(_builder);
      this.serialize(this._xsemanticsGeneratorExtensions.errorInformationType(rule), rule, a);
      _xblockexpression = this.compileArgsForErrorInformationConstructor(a, source, feature, data);
    }
    return _xblockexpression;
  }

  private ITreeAppendable compileArgsForErrorInformationConstructor(final ITreeAppendable a, final String source, final String feature, final String data) {
    ITreeAppendable _xifexpression = null;
    boolean _isEmpty = data.isEmpty();
    if (_isEmpty) {
      StringConcatenation _builder = new StringConcatenation();
      _builder.append("(");
      _builder.append(source);
      _builder.append(", ");
      _builder.append(feature);
      _builder.append("));");
      _xifexpression = a.append(_builder);
    } else {
      StringConcatenation _builder_1 = new StringConcatenation();
      _builder_1.append("(");
      _builder_1.append(source);
      _builder_1.append(", ");
      _builder_1.append(feature);
      _builder_1.append(", ");
      _builder_1.append(data);
      _builder_1.append("));");
      _xifexpression = a.append(_builder_1);
    }
    return _xifexpression;
  }

  public String compileErrorOfErrorSpecification(final ErrorSpecification errorSpecification, final ITreeAppendable b) {
    return this.compileAndAssignToLocalVariable(
      errorSpecification.getError(), b, 
      this.typeReferences.getTypeForName(String.class, errorSpecification), "error");
  }

  public String compileSourceOfErrorSpecification(final ErrorSpecification errorSpecification, final ITreeAppendable b) {
    return this.compileAndAssignToLocalVariable(
      errorSpecification.getSource(), b, 
      this.typeReferences.getTypeForName(EObject.class, errorSpecification), "source");
  }

  public String compileFeatureOfErrorSpecification(final ErrorSpecification errorSpecification, final ITreeAppendable b) {
    return this.compileAndAssignToLocalVariable(
      errorSpecification.getFeature(), b, 
      this.typeReferences.getTypeForName(EStructuralFeature.class, errorSpecification), "feature");
  }

  public String compileDataOfErrorSpecification(final ErrorSpecification errorSpecification, final ITreeAppendable b) {
    XExpression _data = errorSpecification.getData();
    boolean _tripleEquals = (_data == null);
    if (_tripleEquals) {
      return "";
    }
    return this.compileAndAssignToLocalVariable(
      errorSpecification.getData(), b, 
      this.typeReferences.getTypeForName(Object.class, errorSpecification), "data");
  }

  protected String compileAndAssignToLocalVariable(final XExpression expression, final ITreeAppendable b, final JvmTypeReference expectedType, final String proposedVariable) {
    if ((expression == null)) {
      return "null";
    }
    this.toJavaStatement(expression, b, true);
    final Object syntheticObject = new Object();
    final String varName = b.declareSyntheticVariable(syntheticObject, proposedVariable);
    b.append("\n");
    this.serialize(expectedType, expression, b);
    b.append(" ").append(varName).append(" = ");
    this.toJavaExpression(expression, b);
    b.append(";");
    return b.getName(syntheticObject);
  }

  /**
   * We need to wrap boolean expressions with try-catch-throw RuleFailedException,
   * since in Xsemantics a boolean expression is an assertion
   */
  @Override
  protected void _toJavaStatement(final XBlockExpression expr, final ITreeAppendable b, final boolean isReferenced) {
    if ((this.insideClosure(expr) || this.insideCachedCondition(expr))) {
      super._toJavaStatement(expr, b, isReferenced);
    } else {
      boolean _isEmpty = expr.getExpressions().isEmpty();
      if (_isEmpty) {
        return;
      }
      int _size = expr.getExpressions().size();
      boolean _equals = (_size == 1);
      if (_equals) {
        this.compileBooleanXExpression(expr.getExpressions().get(0), b, isReferenced);
        return;
      }
      if (isReferenced) {
        this.declareSyntheticVariable(expr, b);
      }
      final boolean needsBraces = isReferenced;
      if (needsBraces) {
        b.newLine().append("{").increaseIndentation();
        b.openPseudoScope();
      }
      final EList<XExpression> expressions = expr.getExpressions();
      int i = 0;
      for (final XExpression ex : expressions) {
        {
          boolean hasToBeReferenced = isReferenced;
          int _size_1 = expressions.size();
          int _minus = (_size_1 - 1);
          boolean _lessThan = (i < _minus);
          if (_lessThan) {
            hasToBeReferenced = false;
          }
          this.compileBooleanXExpression(ex, b, hasToBeReferenced);
          if (hasToBeReferenced) {
            b.append("\n").append(this.getVarName(expr, b)).append(" = (");
            this.internalToConvertedExpression(ex, b, null);
            b.append(");");
          }
          i = (i + 1);
        }
      }
      if (needsBraces) {
        b.closeScope();
        b.decreaseIndentation().newLine().append("}");
      }
    }
  }

  /**
   * If it's boolean, wraps in an if with throw RuleFailedException
   * 
   * @param expression
   * @param b
   * @param hasToBeReferenced
   */
  public void compileBooleanXExpression(final XExpression expression, final ITreeAppendable b, final boolean toBeReferenced) {
    boolean hasToBeReferenced = toBeReferenced;
    if ((expression instanceof XBlockExpression)) {
      this.internalToJavaStatement(expression, b, hasToBeReferenced);
      return;
    }
    final boolean isBoolean = this.typeSystem.isBooleanPremise(expression);
    if (isBoolean) {
      hasToBeReferenced = true;
    }
    this.internalToJavaStatement(expression, b, hasToBeReferenced);
    if (isBoolean) {
      this.generateCommentWithOriginalCode(expression, b);
      this.newLine(b);
      b.append("if (!");
      this.toJavaExpression(expression, b);
      b.append(") {");
      b.increaseIndentation();
      this.newLine(b);
      this.throwNewRuleFailedException(expression, b);
      this.closeBracket(b);
    }
  }

  @Override
  protected boolean isVariableDeclarationRequired(final XExpression expr, final ITreeAppendable b, final boolean recursive) {
    boolean _isBooleanPremise = this.typeSystem.isBooleanPremise(expr);
    if (_isBooleanPremise) {
      final EObject container = expr.eContainer();
      if ((container instanceof XBlockExpression)) {
        final EList<XExpression> siblings = ((XBlockExpression)container).getExpressions();
        int _size = siblings.size();
        int _minus = (_size - 1);
        XExpression _get = siblings.get(_minus);
        boolean _equals = Objects.equal(_get, expr);
        if (_equals) {
          EObject _eContainer = ((XBlockExpression)container).eContainer();
          if ((_eContainer instanceof AuxiliaryFunction)) {
            return true;
          }
          return false;
        }
      }
    }
    return super.isVariableDeclarationRequired(expr, b, recursive);
  }

  /**
   * When we invoke an auxiliary function we must also add the
   * trace argument.
   */
  @Override
  protected void appendFeatureCall(final XAbstractFeatureCall call, final ITreeAppendable b) {
    final JvmIdentifiableElement feature = call.getFeature();
    final AuxiliaryDescription auxiliaryDescription = this._xsemanticsGeneratorExtensions.associatedAuxiliaryDescription(feature);
    if ((auxiliaryDescription == null)) {
      super.appendFeatureCall(call, b);
      return;
    }
    final CharSequence name = this._xsemanticsGeneratorExtensions.entryPointInternalMethodName(auxiliaryDescription);
    b.trace(call, XbasePackage.Literals.XABSTRACT_FEATURE_CALL__FEATURE, 0).append(name);
    if ((feature instanceof JvmExecutable)) {
      b.append("(");
      b.append(this._xsemanticsGeneratorExtensions.ruleApplicationTraceName());
      final EList<XExpression> arguments = call.getActualArguments();
      boolean _isEmpty = arguments.isEmpty();
      boolean _not = (!_isEmpty);
      if (_not) {
        b.append(", ");
        XExpression _xifexpression = null;
        if ((call instanceof XMemberFeatureCall)) {
          _xifexpression = ((XMemberFeatureCall)call).getMemberCallTarget();
        } else {
          _xifexpression = null;
        }
        final XExpression receiver = _xifexpression;
        final boolean shouldBreakFirstArgument = ((receiver == null) || (!Objects.equal(arguments.get(0), receiver)));
        this.appendArguments(arguments, b, shouldBreakFirstArgument);
      }
      b.append(")");
    }
  }

  public void throwNewRuleFailedException(final XExpression expression, final ITreeAppendable b) {
    b.append(this._xsemanticsGeneratorExtensions.sneakyThrowRuleFailedException());
    b.append("(");
    this.generateStringWithOriginalCode(expression, b);
    b.append(");");
  }

  protected void _doInternalToJavaStatement(final XExpression e, final ITreeAppendable b, final boolean isReferenced) {
    super.doInternalToJavaStatement(e, b, isReferenced);
  }

  protected void _internalToConvertedExpression(final XExpression obj, final ITreeAppendable appendable) {
    super.internalToConvertedExpression(obj, appendable);
  }

  protected void _doInternalToJavaStatement(final OrExpression orExpression, final ITreeAppendable b, final boolean isReferenced) {
    this.generateCommentWithOriginalCode(orExpression, b);
    this.openBracket(b);
    boolean _previousFailureMustBeDeclared = this.previousFailureMustBeDeclared(orExpression);
    if (_previousFailureMustBeDeclared) {
      this.declarePreviousFailureVariable(orExpression, b);
    }
    final XExpression left = orExpression.getBranches().get(0);
    final XExpression right = orExpression.getBranches().get(1);
    this.tryStmnt(b);
    this.compileBooleanXExpression(left, b, false);
    this.catchStmnt(b, orExpression);
    this.compileBooleanXExpression(right, b, false);
    this.closeBracket(b);
    this.closeBracket(b);
  }

  private ITreeAppendable declarePreviousFailureVariable(final OrExpression orExpression, final ITreeAppendable b) {
    ITreeAppendable _xblockexpression = null;
    {
      b.newLine();
      b.declareVariable(orExpression, XsemanticsConstants.PREVIOUS_FAILURE);
      b.append(this.typeSystem.ruleFailedExceptionType(orExpression).getType());
      _xblockexpression = b.append(((" " + XsemanticsConstants.PREVIOUS_FAILURE) + " = null;"));
    }
    return _xblockexpression;
  }

  private boolean previousFailureMustBeDeclared(final OrExpression e) {
    EObject container = e.eContainer();
    while ((container != null)) {
      {
        if ((container instanceof OrExpression)) {
          return false;
        }
        if ((container instanceof XClosure)) {
          return true;
        }
        container = container.eContainer();
      }
    }
    return true;
  }

  /**
   * To respect the semantics of previousFailureMustBeDeclared we must check
   * whether it's contained in an OrExpression: when we're inside an OrExpression
   * we compile the lambda into an anonymous class, so that redeclaring
   * "previousFailure" local variable does not generate invalid Java code
   */
  @Override
  protected boolean canCompileToJavaLambda(final XClosure closure, final LightweightTypeReference typeRef, final JvmOperation operation) {
    return ((EcoreUtil2.<OrExpression>getContainerOfType(closure, OrExpression.class) == null) && 
      super.canCompileToJavaLambda(closure, typeRef, operation));
  }

  protected void _doInternalToJavaStatement(final Fail fail, final ITreeAppendable b, final boolean isReference) {
    this.generateCommentWithOriginalCode(fail, b);
    final XExpression errorSpecification = fail.getError();
    if ((errorSpecification == null)) {
      this.newLine(b);
      b.append("throwForExplicitFail();");
    } else {
      this.toJavaStatement(errorSpecification, b, isReference);
    }
  }

  protected void _doInternalToJavaStatement(final ErrorSpecification errorSpecification, final ITreeAppendable b, final boolean isReference) {
    final String errorMessageVar = this.compileErrorOfErrorSpecification(errorSpecification, b);
    final String sourceVar = this.compileSourceOfErrorSpecification(errorSpecification, b);
    final String featureVar = this.compileFeatureOfErrorSpecification(errorSpecification, b);
    final String dataVar = this.compileDataOfErrorSpecification(errorSpecification, b);
    this.newLine(b);
    b.append("throwForExplicitFail(");
    b.append(errorMessageVar);
    this.comma(b);
    b.append("new ");
    b.append(this._xsemanticsGeneratorExtensions.errorInformationType(errorSpecification).getType());
    b.append("(");
    b.append(sourceVar);
    this.comma(b);
    b.append(featureVar);
    boolean _isEmpty = dataVar.isEmpty();
    boolean _not = (!_isEmpty);
    if (_not) {
      this.comma(b);
      b.append(dataVar);
    }
    b.append(")");
    b.append(");");
  }

  protected void _doInternalToJavaStatement(final RuleInvocation ruleInvocation, final ITreeAppendable b, final boolean isReferenced) {
    this.generateCommentWithOriginalCode(ruleInvocation, b);
    final JudgmentDescription judgmentDescription = this._xsemanticsUtils.getJudgmentDescription(ruleInvocation);
    final EList<XExpression> ruleInvocationExpressions = ruleInvocation.getExpressions();
    this.ruleInvocationExpressionsToJavaStatements(b, ruleInvocationExpressions);
    this.generateEnvironmentSpecificationAsStatements(
      ruleInvocation.getEnvironment(), b);
    if (isReferenced) {
      final Later _function = (ITreeAppendable it) -> {
        it.append(this._xsemanticsGeneratorExtensions.succeededMethodName(judgmentDescription).toString());
        it.append("(");
        this.generateEnvironmentSpecificationAsExpression(
          ruleInvocation.getEnvironment(), it);
        this.comma(it);
        it.append(this._xsemanticsGeneratorExtensions.additionalArgsForRuleInvocation(ruleInvocation).toString());
        this.comma(it);
        this.ruleInvocationArgumentsToJavaExpressions(it, ruleInvocation);
        it.append(")");
      };
      this.declareFreshLocalVariable(ruleInvocation, b, _function);
    } else {
      final boolean hasOutputParams = this._xsemanticsUtils.hasOutputParams(ruleInvocation);
      this.newLine(b);
      String resultVariable = "";
      if (hasOutputParams) {
        this._xsemanticsGeneratorExtensions.resultType(judgmentDescription, b);
        this.space(b);
        resultVariable = this.generateResultVariable(ruleInvocation, b);
        this.assign(b);
      }
      b.append(this._xsemanticsGeneratorExtensions.entryPointInternalMethodName(judgmentDescription).toString());
      b.append("(");
      this.generateEnvironmentSpecificationAsExpression(
        ruleInvocation.getEnvironment(), b);
      this.comma(b);
      b.append(this._xsemanticsGeneratorExtensions.additionalArgsForRuleInvocation(ruleInvocation).toString());
      this.comma(b);
      this.ruleInvocationArgumentsToJavaExpressions(b, ruleInvocation);
      b.append(");");
      if (hasOutputParams) {
        this.reassignResults(b, ruleInvocation, resultVariable);
      }
    }
  }

  protected void ruleInvocationExpressionsToJavaStatements(final ITreeAppendable b, final List<XExpression> ruleInvocationExpressions) {
    for (final XExpression ruleInvocationExpression : ruleInvocationExpressions) {
      this.toJavaStatement(ruleInvocationExpression, b, true);
    }
  }

  protected void generateEnvironmentSpecificationAsStatements(final XExpression environmentSpecification, final ITreeAppendable b) {
    if ((environmentSpecification instanceof EmptyEnvironment)) {
      return;
    }
    if ((environmentSpecification instanceof EnvironmentMapping)) {
      this.toJavaStatement(((EnvironmentMapping)environmentSpecification).getKey(), b, true);
      this.toJavaStatement(((EnvironmentMapping)environmentSpecification).getValue(), b, true);
    } else {
      if ((environmentSpecification instanceof EnvironmentComposition)) {
        this.generateEnvironmentSpecificationAsStatements(
          ((EnvironmentComposition)environmentSpecification).getCurrentEnvironment(), b);
        this.generateEnvironmentSpecificationAsStatements(
          ((EnvironmentComposition)environmentSpecification).getSubEnvironment(), b);
      } else {
        this.toJavaStatement(environmentSpecification, b, true);
      }
    }
  }

  public void generateEnvironmentSpecificationAsExpression(final XExpression environmentSpecification, final ITreeAppendable b) {
    if ((environmentSpecification instanceof EnvironmentMapping)) {
      b.append(this._xsemanticsGeneratorExtensions.environmentEntryInvocation());
      b.append("(");
      this.toJavaExpression(((EnvironmentMapping)environmentSpecification).getKey(), b);
      this.comma(b);
      this.toJavaExpression(((EnvironmentMapping)environmentSpecification).getValue(), b);
      b.append(")");
    } else {
      if ((environmentSpecification instanceof EnvironmentComposition)) {
        b.append(this._xsemanticsGeneratorExtensions.environmentCompositionInvocation());
        b.append("(");
        b.increaseIndentation();
        this.newLine(b);
        this.generateEnvironmentSpecificationAsExpression(
          ((EnvironmentComposition)environmentSpecification).getCurrentEnvironment(), b);
        this.comma(b);
        this.generateEnvironmentSpecificationAsExpression(
          ((EnvironmentComposition)environmentSpecification).getSubEnvironment(), b);
        b.decreaseIndentation();
        this.newLine(b);
        b.append(")");
      } else {
        this.toJavaExpression(environmentSpecification, b);
      }
    }
  }

  protected String generateResultVariable(final RuleInvocation ruleInvocation, final ITreeAppendable b) {
    final String declareResultVariable = this.declareResultVariable(ruleInvocation, b);
    b.append(declareResultVariable);
    return declareResultVariable;
  }

  public String declareResultVariable(final RuleInvocation ruleInvocation, final ITreeAppendable b) {
    return b.declareSyntheticVariable(ruleInvocation, "result");
  }

  protected void ruleInvocationArgumentsToJavaExpressions(final ITreeAppendable b, final RuleInvocation ruleInvocation) {
    this.ruleInvocationArgumentsToJavaExpressions(b, 
      this._xsemanticsUtils.inputArgsExpressions(ruleInvocation));
  }

  protected void ruleInvocationArgumentsToJavaExpressions(final ITreeAppendable b, final Iterable<XExpression> inputArgsExpressions) {
    final Iterator<XExpression> expIt = inputArgsExpressions.iterator();
    while (expIt.hasNext()) {
      {
        this.toJavaExpression(expIt.next(), b);
        boolean _hasNext = expIt.hasNext();
        if (_hasNext) {
          this.comma(b);
        }
      }
    }
  }

  protected void reassignResults(final ITreeAppendable b, final RuleInvocation ruleInvocation, final String resultVariable) {
    final ArrayList<XExpression> expressions = this._xsemanticsUtils.outputArgsExpressions(ruleInvocation);
    this.newLine(b);
    final Iterator<String> getMethods = XsemanticsGeneratorExtensions.getResultGetMethods().iterator();
    for (final XExpression expression : expressions) {
      {
        final JvmTypeReference expressionType = this.typeSystem.getType(expression);
        final String getMethod = getMethods.next();
        b.append("checkAssignableTo");
        b.append("(");
        b.append(((resultVariable + ".") + getMethod));
        this.comma(b);
        this.generateJavaClassReference(expressionType, expression, b);
        b.append(");");
        this.newLine(b);
        if ((expression instanceof XVariableDeclaration)) {
          b.append(b.getName(expression));
        } else {
          this.toJavaExpression(expression, b);
        }
        this.assign(b);
        b.append("(");
        this.serialize(expressionType, expression, b);
        b.append(")");
        this.space(b);
        b.append(((resultVariable + ".") + getMethod));
        b.append(";");
        this.newLine(b);
      }
    }
  }

  protected void _internalToConvertedExpression(final RuleInvocation ruleInvocation, final ITreeAppendable b) {
    b.append(this.getVarName(ruleInvocation, b));
  }

  protected void _internalToConvertedExpression(final EmptyEnvironment e, final ITreeAppendable b) {
    b.append(this._xsemanticsGeneratorExtensions.emptyEnvironmentInvocation());
  }

  public void generateCommentWithOriginalCode(final EObject modelElement, final ITreeAppendable b) {
    b.append("\n").append("/* ").append(this._xsemanticsNodeModelUtils.getProgramText(modelElement)).append(" */");
  }

  public void generateStringWithOriginalCode(final EObject modelElement, final ITreeAppendable b) {
    b.append("\"").append(this._xsemanticsGeneratorExtensions.javaString(this._xsemanticsNodeModelUtils.getProgramText(modelElement))).append("\"");
  }

  public void comma(final ITreeAppendable b) {
    b.append(", ");
  }

  public void newLine(final ITreeAppendable b) {
    b.append("\n");
  }

  public void space(final ITreeAppendable b) {
    b.append(" ");
  }

  public void assign(final ITreeAppendable b) {
    this.space(b);
    b.append("=");
    this.space(b);
  }

  public void openBracket(final ITreeAppendable b) {
    this.newLine(b);
    b.append("{");
    b.increaseIndentation();
  }

  public void closeBracket(final ITreeAppendable b) {
    b.decreaseIndentation();
    this.newLine(b);
    b.append("}");
  }

  public void tryStmnt(final ITreeAppendable b) {
    this.newLine(b);
    b.append("try {");
    b.increaseIndentation();
  }

  /**
   * Also declares a RuleFailedException variable for the passed expressions
   * 
   * @param b
   * @param expression
   * @return
   */
  public String catchStmnt(final ITreeAppendable b, final XExpression expression) {
    b.decreaseIndentation();
    this.newLine(b);
    b.append("} catch (");
    b.append(this._xsemanticsGeneratorExtensions.exceptionType(expression).getType());
    b.append(" ");
    final String declareExceptionVariable = this.declareExceptionVariable(expression, b);
    b.append(declareExceptionVariable);
    b.append(") {");
    b.increaseIndentation();
    b.newLine();
    b.append((((XsemanticsConstants.PREVIOUS_FAILURE + " = extractRuleFailedException(") + declareExceptionVariable) + ");"));
    return declareExceptionVariable;
  }

  public String declareExceptionVariable(final XExpression expression, final ITreeAppendable b) {
    return b.declareSyntheticVariable(expression, "e");
  }

  public void generateJavaClassReference(final JvmTypeReference expressionType, final XExpression expression, final ITreeAppendable b) {
    b.append(expressionType.getType());
    b.append(".class");
  }

  public boolean insideClosure(final XExpression expr) {
    EObject _eContainer = expr.eContainer();
    return (_eContainer instanceof XClosure);
  }

  public boolean insideCachedCondition(final XExpression expr) {
    CachedClause _containerOfType = EcoreUtil2.<CachedClause>getContainerOfType(expr, CachedClause.class);
    return (_containerOfType != null);
  }

  public ITreeAppendable compileFinalPartOfThrowExceptionMethod(final EObject desc, final ITreeAppendable a, final String error, final String source, final String feature, final String data) {
    if (desc instanceof Description) {
      return _compileFinalPartOfThrowExceptionMethod((Description)desc, a, error, source, feature, data);
    } else if (desc instanceof RuleConclusion) {
      return _compileFinalPartOfThrowExceptionMethod((RuleConclusion)desc, a, error, source, feature, data);
    } else {
      throw new IllegalArgumentException("Unhandled parameter types: " +
        Arrays.<Object>asList(desc, a, error, source, feature, data).toString());
    }
  }

  public void doInternalToJavaStatement(final XExpression errorSpecification, final ITreeAppendable b, final boolean isReference) {
    if (errorSpecification instanceof ErrorSpecification) {
      _doInternalToJavaStatement((ErrorSpecification)errorSpecification, b, isReference);
      return;
    } else if (errorSpecification instanceof Fail) {
      _doInternalToJavaStatement((Fail)errorSpecification, b, isReference);
      return;
    } else if (errorSpecification instanceof OrExpression) {
      _doInternalToJavaStatement((OrExpression)errorSpecification, b, isReference);
      return;
    } else if (errorSpecification instanceof RuleInvocation) {
      _doInternalToJavaStatement((RuleInvocation)errorSpecification, b, isReference);
      return;
    } else if (errorSpecification != null) {
      _doInternalToJavaStatement(errorSpecification, b, isReference);
      return;
    } else {
      throw new IllegalArgumentException("Unhandled parameter types: " +
        Arrays.<Object>asList(errorSpecification, b, isReference).toString());
    }
  }

  public void internalToConvertedExpression(final XExpression e, final ITreeAppendable b) {
    if (e instanceof EmptyEnvironment) {
      _internalToConvertedExpression((EmptyEnvironment)e, b);
      return;
    } else if (e instanceof RuleInvocation) {
      _internalToConvertedExpression((RuleInvocation)e, b);
      return;
    } else if (e != null) {
      _internalToConvertedExpression(e, b);
      return;
    } else {
      throw new IllegalArgumentException("Unhandled parameter types: " +
        Arrays.<Object>asList(e, b).toString());
    }
  }
}
