/**
 * Copyright (c) 2013-2017 Lorenzo Bettini.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *   Lorenzo Bettini - Initial contribution and API
 */
package org.eclipse.xsemantics.dsl.generator;

import com.google.inject.Inject;
import com.google.inject.Provider;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import org.eclipse.emf.common.util.EList;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.xsemantics.dsl.typing.XsemanticsTypeSystem;
import org.eclipse.xsemantics.dsl.util.XsemanticsUtils;
import org.eclipse.xsemantics.dsl.xsemantics.AuxiliaryDescription;
import org.eclipse.xsemantics.dsl.xsemantics.AuxiliaryFunction;
import org.eclipse.xsemantics.dsl.xsemantics.Cachable;
import org.eclipse.xsemantics.dsl.xsemantics.CheckRule;
import org.eclipse.xsemantics.dsl.xsemantics.Description;
import org.eclipse.xsemantics.dsl.xsemantics.ExpressionInConclusion;
import org.eclipse.xsemantics.dsl.xsemantics.InputParameter;
import org.eclipse.xsemantics.dsl.xsemantics.JudgmentDescription;
import org.eclipse.xsemantics.dsl.xsemantics.JudgmentParameter;
import org.eclipse.xsemantics.dsl.xsemantics.Named;
import org.eclipse.xsemantics.dsl.xsemantics.OutputParameter;
import org.eclipse.xsemantics.dsl.xsemantics.Rule;
import org.eclipse.xsemantics.dsl.xsemantics.RuleConclusionElement;
import org.eclipse.xsemantics.dsl.xsemantics.RuleInvocation;
import org.eclipse.xsemantics.dsl.xsemantics.RuleParameter;
import org.eclipse.xsemantics.dsl.xsemantics.UniqueByName;
import org.eclipse.xsemantics.dsl.xsemantics.XsemanticsSystem;
import org.eclipse.xsemantics.runtime.ErrorInformation;
import org.eclipse.xsemantics.runtime.Result;
import org.eclipse.xsemantics.runtime.Result2;
import org.eclipse.xsemantics.runtime.Result3;
import org.eclipse.xtend2.lib.StringConcatenation;
import org.eclipse.xtend2.lib.StringConcatenationClient;
import org.eclipse.xtext.common.types.JvmFormalParameter;
import org.eclipse.xtext.common.types.JvmIdentifiableElement;
import org.eclipse.xtext.common.types.JvmTypeReference;
import org.eclipse.xtext.common.types.util.TypeReferences;
import org.eclipse.xtext.naming.IQualifiedNameProvider;
import org.eclipse.xtext.naming.QualifiedName;
import org.eclipse.xtext.util.Strings;
import org.eclipse.xtext.xbase.compiler.IAppendable;
import org.eclipse.xtext.xbase.compiler.TypeReferenceSerializer;
import org.eclipse.xtext.xbase.compiler.output.ITreeAppendable;
import org.eclipse.xtext.xbase.jvmmodel.IJvmModelAssociations;
import org.eclipse.xtext.xbase.lib.CollectionLiterals;
import org.eclipse.xtext.xbase.lib.Conversions;
import org.eclipse.xtext.xbase.lib.Extension;
import org.eclipse.xtext.xbase.lib.Functions.Function1;
import org.eclipse.xtext.xbase.lib.IterableExtensions;
import org.eclipse.xtext.xbase.lib.ListExtensions;

@SuppressWarnings("all")
public class XsemanticsGeneratorExtensions {
  @Inject
  @Extension
  private IQualifiedNameProvider _iQualifiedNameProvider;

  @Inject
  @Extension
  private XsemanticsUtils _xsemanticsUtils;

  @Inject
  @Extension
  private XsemanticsTypeSystem typeSystem;

  @Inject
  @Extension
  private TypeReferenceSerializer _typeReferenceSerializer;

  @Inject
  @Extension
  private TypeReferences _typeReferences;

  @Inject
  protected IJvmModelAssociations associations;

  protected static final ArrayList<String> RESULT_GET_METHODS = CollectionLiterals.<String>newArrayList(
    "getFirst()", "getSecond()", "getThird()");

  public static ArrayList<String> getResultGetMethods() {
    return XsemanticsGeneratorExtensions.RESULT_GET_METHODS;
  }

  public QualifiedName toJavaFullyQualifiedName(final XsemanticsSystem ts) {
    return this._iQualifiedNameProvider.getFullyQualifiedName(ts);
  }

  public String toJavaFullyQualifiedName(final Named e) {
    String _xblockexpression = null;
    {
      final QualifiedName fQN = this.toJavaFullyQualifiedName(this._xsemanticsUtils.containingSystem(e));
      String _xifexpression = null;
      if (((fQN != null) && (fQN.getSegmentCount() > 1))) {
        QualifiedName _skipLast = fQN.skipLast(1);
        String _plus = (_skipLast + ".");
        String _javaClassName = this.toJavaClassName(e);
        _xifexpression = (_plus + _javaClassName);
      } else {
        _xifexpression = this.toJavaClassName(e);
      }
      _xblockexpression = _xifexpression;
    }
    return _xblockexpression;
  }

  public String toJavaClassName(final Named e) {
    return Strings.toFirstUpper(e.getName());
  }

  public String toValidatorPackage(final XsemanticsSystem ts) {
    String _xblockexpression = null;
    {
      final QualifiedName typeSystemPackage = this._iQualifiedNameProvider.getFullyQualifiedName(ts);
      String _xifexpression = null;
      if (((typeSystemPackage != null) && (typeSystemPackage.getSegmentCount() > 1))) {
        QualifiedName _skipLast = typeSystemPackage.skipLast(1);
        String _plus = (_skipLast + ".");
        _xifexpression = (_plus + "validation");
      } else {
        _xifexpression = "";
      }
      _xblockexpression = _xifexpression;
    }
    return _xblockexpression;
  }

  public String toValidatorJavaClassName(final XsemanticsSystem ts) {
    QualifiedName _fullyQualifiedName = this._iQualifiedNameProvider.getFullyQualifiedName(ts);
    String _lastSegment = null;
    if (_fullyQualifiedName!=null) {
      _lastSegment=_fullyQualifiedName.getLastSegment();
    }
    return (_lastSegment + "Validator");
  }

  public String toValidatorJavaFullyQualifiedName(final XsemanticsSystem ts) {
    String _xblockexpression = null;
    {
      final String validatorPackage = this.toValidatorPackage(ts);
      String _xifexpression = null;
      int _length = validatorPackage.length();
      boolean _greaterThan = (_length > 0);
      if (_greaterThan) {
        String _validatorJavaClassName = this.toValidatorJavaClassName(ts);
        _xifexpression = ((validatorPackage + ".") + _validatorJavaClassName);
      } else {
        _xifexpression = this.toValidatorJavaClassName(ts);
      }
      _xblockexpression = _xifexpression;
    }
    return _xblockexpression;
  }

  public String ruleIssueString(final Named e) {
    return e.getName().toUpperCase();
  }

  public CharSequence polymorphicDispatcherField(final UniqueByName e) {
    StringConcatenation _builder = new StringConcatenation();
    String _name = e.getName();
    _builder.append(_name);
    _builder.append("Dispatcher");
    return _builder;
  }

  public String relationSymbolsArgs(final JudgmentDescription judgmentDescription) {
    final Function1<String, String> _function = (String it) -> {
      StringConcatenation _builder = new StringConcatenation();
      _builder.append("\"");
      String _escapeJavaStringChars = this.escapeJavaStringChars(it);
      _builder.append(_escapeJavaStringChars);
      _builder.append("\"");
      return _builder.toString();
    };
    return IterableExtensions.join(ListExtensions.<String, String>map(judgmentDescription.getRelationSymbols(), _function), ", ");
  }

  public CharSequence polymorphicDispatcherImpl(final JudgmentDescription judgmentDescription) {
    StringConcatenation _builder = new StringConcatenation();
    String _name = judgmentDescription.getName();
    _builder.append(_name);
    _builder.append("Impl");
    return _builder;
  }

  public CharSequence polymorphicDispatcherBuildMethod(final JudgmentDescription judgmentDescription) {
    StringConcatenation _builder = new StringConcatenation();
    _builder.append("buildPolymorphicDispatcher");
    int _polymorphicDispatcherBuildMethodSuffix = this.polymorphicDispatcherBuildMethodSuffix(judgmentDescription);
    _builder.append(_polymorphicDispatcherBuildMethodSuffix);
    return _builder;
  }

  public int polymorphicDispatcherBuildMethodSuffix(final JudgmentDescription judgmentDescription) {
    int _xifexpression = (int) 0;
    int _size = this._xsemanticsUtils.outputJudgmentParameters(judgmentDescription).size();
    boolean _equals = (_size == 0);
    if (_equals) {
      _xifexpression = 1;
    } else {
      _xifexpression = this._xsemanticsUtils.outputJudgmentParameters(judgmentDescription).size();
    }
    return _xifexpression;
  }

  public CharSequence polymorphicDispatcherImpl(final AuxiliaryDescription aux) {
    StringConcatenation _builder = new StringConcatenation();
    String _name = aux.getName();
    _builder.append(_name);
    _builder.append("Impl");
    return _builder;
  }

  public CharSequence polymorphicDispatcherNumOfArgs(final JudgmentDescription judgmentDescription) {
    StringConcatenation _builder = new StringConcatenation();
    int _size = this._xsemanticsUtils.inputParams(judgmentDescription).size();
    int _plus = (_size + 2);
    _builder.append(_plus);
    return _builder;
  }

  public CharSequence polymorphicDispatcherNumOfArgs(final AuxiliaryDescription aux) {
    StringConcatenation _builder = new StringConcatenation();
    int _size = aux.getParameters().size();
    int _plus = (_size + 1);
    _builder.append(_plus);
    return _builder;
  }

  public CharSequence entryPointMethodName(final Description desc) {
    StringConcatenation _builder = new StringConcatenation();
    String _name = desc.getName();
    _builder.append(_name);
    return _builder;
  }

  public CharSequence succeededMethodName(final JudgmentDescription judgmentDescription) {
    StringConcatenation _builder = new StringConcatenation();
    String _name = judgmentDescription.getName();
    _builder.append(_name);
    _builder.append("Succeeded");
    return _builder;
  }

  public String inputArgs(final JudgmentDescription judgmentDescription) {
    String _xblockexpression = null;
    {
      final UniqueNames names = new UniqueNames();
      final Function1<InputParameter, String> _function = (InputParameter it) -> {
        StringConcatenation _builder = new StringConcatenation();
        String _createName = names.createName(this.inputParameterName(it));
        _builder.append(_createName);
        return _builder.toString();
      };
      _xblockexpression = IterableExtensions.join(ListExtensions.<InputParameter, String>map(this._xsemanticsUtils.inputParams(judgmentDescription), _function), ", ");
    }
    return _xblockexpression;
  }

  public String inputArgs(final AuxiliaryDescription aux) {
    final Function1<JvmFormalParameter, String> _function = (JvmFormalParameter it) -> {
      return it.getName();
    };
    return IterableExtensions.join(ListExtensions.<JvmFormalParameter, String>map(aux.getParameters(), _function), ", ");
  }

  public String inputParameterName(final InputParameter param) {
    return param.getParameter().getName();
  }

  public CharSequence environmentName() {
    StringConcatenation _builder = new StringConcatenation();
    _builder.append("_environment_");
    return _builder;
  }

  public CharSequence entryPointInternalMethodName(final JudgmentDescription judgmentDescription) {
    StringConcatenation _builder = new StringConcatenation();
    CharSequence _entryPointMethodName = this.entryPointMethodName(judgmentDescription);
    _builder.append(_entryPointMethodName);
    _builder.append("Internal");
    return _builder;
  }

  public CharSequence entryPointInternalMethodName(final AuxiliaryDescription aux) {
    StringConcatenation _builder = new StringConcatenation();
    CharSequence _entryPointMethodName = this.entryPointMethodName(aux);
    _builder.append(_entryPointMethodName);
    _builder.append("Internal");
    return _builder;
  }

  public CharSequence additionalArgs() {
    StringConcatenation _builder = new StringConcatenation();
    CharSequence _environmentName = this.environmentName();
    _builder.append(_environmentName);
    _builder.append(", ");
    CharSequence _ruleApplicationTraceName = this.ruleApplicationTraceName();
    _builder.append(_ruleApplicationTraceName);
    return _builder;
  }

  public CharSequence ruleApplicationTraceName() {
    StringConcatenation _builder = new StringConcatenation();
    _builder.append("_trace_");
    return _builder;
  }

  public CharSequence ruleApplicationSubtraceName() {
    StringConcatenation _builder = new StringConcatenation();
    _builder.append("_subtrace_");
    return _builder;
  }

  public CharSequence exceptionVarName(final UniqueByName e) {
    StringConcatenation _builder = new StringConcatenation();
    _builder.append("_e_");
    String _name = e.getName();
    _builder.append(_name);
    return _builder;
  }

  public String exceptionVarName(final Rule rule) {
    StringConcatenation _builder = new StringConcatenation();
    _builder.append("e_");
    CharSequence _applyRuleName = this.applyRuleName(rule);
    _builder.append(_applyRuleName);
    return _builder.toString();
  }

  public String exceptionVarName(final AuxiliaryFunction aux) {
    StringConcatenation _builder = new StringConcatenation();
    _builder.append("e_");
    CharSequence _applyAuxFunName = this.applyAuxFunName(aux);
    _builder.append(_applyAuxFunName);
    return _builder.toString();
  }

  public String suffixStartingFrom2(final JudgmentDescription judgmentDescription) {
    String _xblockexpression = null;
    {
      final int numOfOutputParams = this._xsemanticsUtils.outputJudgmentParameters(judgmentDescription).size();
      String _xifexpression = null;
      if ((numOfOutputParams > 1)) {
        _xifexpression = ("" + Integer.valueOf(numOfOutputParams));
      } else {
        _xifexpression = "";
      }
      _xblockexpression = _xifexpression;
    }
    return _xblockexpression;
  }

  public CharSequence throwExceptionMethod(final UniqueByName e) {
    StringConcatenation _builder = new StringConcatenation();
    String _firstLower = Strings.toFirstLower(e.getName());
    _builder.append(_firstLower);
    _builder.append("ThrowException");
    return _builder;
  }

  public CharSequence cacheConditionMethod(final Cachable c) {
    StringConcatenation _builder = new StringConcatenation();
    String _firstLower = Strings.toFirstLower(c.getName());
    _builder.append(_firstLower);
    _builder.append("CacheCondition");
    return _builder;
  }

  public CharSequence throwRuleFailedExceptionMethod() {
    StringConcatenation _builder = new StringConcatenation();
    _builder.append("throwRuleFailedException");
    return _builder;
  }

  public CharSequence newTraceMethod(final CharSequence trace) {
    StringConcatenation _builder = new StringConcatenation();
    _builder.append("newTrace(");
    _builder.append(trace);
    _builder.append(")");
    return _builder;
  }

  public CharSequence applyRuleName(final Rule rule) {
    StringConcatenation _builder = new StringConcatenation();
    _builder.append("applyRule");
    String _javaClassName = this.toJavaClassName(rule);
    _builder.append(_javaClassName);
    return _builder;
  }

  public CharSequence applyAuxFunName(final AuxiliaryFunction aux) {
    StringConcatenation _builder = new StringConcatenation();
    _builder.append("applyAuxFun");
    String _javaClassName = this.toJavaClassName(aux);
    _builder.append(_javaClassName);
    return _builder;
  }

  public CharSequence additionalArgsForRule(final Rule rule) {
    StringConcatenation _builder = new StringConcatenation();
    String _ruleEnvName = this.ruleEnvName(rule);
    _builder.append(_ruleEnvName);
    _builder.append(", ");
    CharSequence _ruleApplicationSubtraceName = this.ruleApplicationSubtraceName();
    _builder.append(_ruleApplicationSubtraceName);
    return _builder;
  }

  public CharSequence additionalArgsForRuleInvocation(final RuleInvocation ruleInvocation) {
    StringConcatenation _builder = new StringConcatenation();
    CharSequence _ruleApplicationTraceName = this.ruleApplicationTraceName();
    _builder.append(_ruleApplicationTraceName);
    return _builder;
  }

  public String ruleEnvName(final Rule rule) {
    return rule.getConclusion().getEnvironment().getName();
  }

  public StringConcatenationClient addToTraceMethod(final CharSequence trace, final CharSequence toAdd) {
    StringConcatenationClient _client = new StringConcatenationClient() {
      @Override
      protected void appendTo(StringConcatenationClient.TargetStringConcatenation _builder) {
        _builder.append("addToTrace(");
        _builder.append(trace);
        _builder.append(", new ");
        _builder.append(Provider.class);
        _builder.append("<Object>() {");
        _builder.newLineIfNotEmpty();
        _builder.append("\t");
        _builder.append("public Object get() {");
        _builder.newLine();
        _builder.append("\t\t");
        _builder.append("return ");
        _builder.append(toAdd, "\t\t");
        _builder.append(";");
        _builder.newLineIfNotEmpty();
        _builder.append("\t");
        _builder.append("}");
        _builder.newLine();
        _builder.append("})");
      }
    };
    return _client;
  }

  public CharSequence addAsSubtraceMethod(final CharSequence trace, final CharSequence subtrace) {
    StringConcatenation _builder = new StringConcatenation();
    _builder.append("addAsSubtrace(");
    _builder.append(trace);
    _builder.append(", ");
    _builder.append(subtrace);
    _builder.append(")");
    return _builder;
  }

  public String traceStringForRule(final Rule rule) {
    String _xblockexpression = null;
    {
      final Iterator<String> getMethods = XsemanticsGeneratorExtensions.getResultGetMethods().iterator();
      final Function1<RuleConclusionElement, CharSequence> _function = (RuleConclusionElement it) -> {
        StringConcatenation _builder = new StringConcatenation();
        CharSequence _resultVariableForTrace = this.resultVariableForTrace();
        _builder.append(_resultVariableForTrace);
        _builder.append(".");
        String _next = getMethods.next();
        _builder.append(_next);
        return this.wrapInStringRepr(_builder);
      };
      final Function1<RuleConclusionElement, CharSequence> _function_1 = (RuleConclusionElement it) -> {
        return this.wrapInStringRepr(this.ruleConclusionInputParamForError(it));
      };
      _xblockexpression = this.stringForRule(rule, _function, _function_1);
    }
    return _xblockexpression;
  }

  public String stringForRule(final Rule rule, final Function1<? super RuleConclusionElement, ? extends CharSequence> forOutput, final Function1<? super RuleConclusionElement, ? extends CharSequence> forInput) {
    String _xblockexpression = null;
    {
      StringConcatenation _builder = new StringConcatenation();
      CharSequence _ruleNameInvocation = this.ruleNameInvocation(rule.getName());
      _builder.append(_ruleNameInvocation);
      _builder.append(" + ");
      CharSequence _wrapInStringReprForEnv = this.wrapInStringReprForEnv(this.ruleEnvName(rule));
      _builder.append(_wrapInStringReprForEnv);
      _builder.append(" + \" ");
      String _judgmentSymbol = rule.getConclusion().getJudgmentSymbol();
      _builder.append(_judgmentSymbol);
      _builder.append(" \"");
      final StringBuffer buffer = new StringBuffer(_builder);
      final Iterator<JudgmentParameter> judgmentParameters = this._xsemanticsUtils.getJudgmentDescription(rule).getJudgmentParameters().iterator();
      final Iterator<String> relationSymbols = rule.getConclusion().getRelationSymbols().iterator();
      EList<RuleConclusionElement> _conclusionElements = rule.getConclusion().getConclusionElements();
      for (final RuleConclusionElement e : _conclusionElements) {
        {
          buffer.append(" + ");
          boolean _isOutputParameter = this._xsemanticsUtils.isOutputParameter(judgmentParameters.next());
          if (_isOutputParameter) {
            buffer.append(forOutput.apply(e));
          } else {
            buffer.append(forInput.apply(e));
          }
          boolean _hasNext = relationSymbols.hasNext();
          if (_hasNext) {
            StringConcatenation _builder_1 = new StringConcatenation();
            _builder_1.append(" ");
            _builder_1.append("+ \" ");
            String _escapeJavaStringChars = this.escapeJavaStringChars(relationSymbols.next());
            _builder_1.append(_escapeJavaStringChars, " ");
            _builder_1.append(" \"");
            buffer.append(_builder_1);
          }
        }
      }
      _xblockexpression = buffer.toString();
    }
    return _xblockexpression;
  }

  public String traceStringForAuxiliaryFun(final AuxiliaryFunction aux) {
    String _errorForAuxiliaryFun = this.errorForAuxiliaryFun(aux);
    StringConcatenation _builder = new StringConcatenation();
    _builder.append(" ");
    _builder.append("+ \" = \" + ");
    String _plus = (_errorForAuxiliaryFun + _builder);
    CharSequence _wrapInStringRepr = this.wrapInStringRepr(this.resultVariableForTrace());
    return (_plus + _wrapInStringRepr);
  }

  public CharSequence ruleNameInvocation(final String ruleName) {
    StringConcatenation _builder = new StringConcatenation();
    _builder.append("ruleName(\"");
    _builder.append(ruleName);
    _builder.append("\")");
    return _builder;
  }

  public CharSequence auxFunNameInvocation(final String ruleName) {
    StringConcatenation _builder = new StringConcatenation();
    _builder.append("auxFunName(\"");
    _builder.append(ruleName);
    _builder.append("\")");
    return _builder;
  }

  public CharSequence wrapInStringReprForEnv(final CharSequence s) {
    StringConcatenation _builder = new StringConcatenation();
    CharSequence _stringRepresentationForEnv = this.stringRepresentationForEnv();
    _builder.append(_stringRepresentationForEnv);
    _builder.append("(");
    _builder.append(s);
    _builder.append(")");
    return _builder;
  }

  public CharSequence stringRepresentationForEnv() {
    StringConcatenation _builder = new StringConcatenation();
    _builder.append("stringRepForEnv");
    return _builder;
  }

  public CharSequence stringRepresentation() {
    StringConcatenation _builder = new StringConcatenation();
    _builder.append("stringRep");
    return _builder;
  }

  public CharSequence wrapInStringRepr(final CharSequence s) {
    StringConcatenation _builder = new StringConcatenation();
    CharSequence _stringRepresentation = this.stringRepresentation();
    _builder.append(_stringRepresentation);
    _builder.append("(");
    _builder.append(s);
    _builder.append(")");
    return _builder;
  }

  public CharSequence resultVariableForTrace() {
    StringConcatenation _builder = new StringConcatenation();
    _builder.append("_result_");
    return _builder;
  }

  public String sneakyThrowRuleFailedException() {
    return "sneakyThrowRuleFailedException";
  }

  public String ruleConclusionInputParamForError(final RuleConclusionElement element) {
    if ((element instanceof RuleParameter)) {
      return ((RuleParameter)element).getParameter().getName();
    } else {
      return "";
    }
  }

  public String inputParameterNames(final Rule rule) {
    final Function1<RuleParameter, String> _function = (RuleParameter it) -> {
      return it.getParameter().getName();
    };
    return IterableExtensions.join(ListExtensions.<RuleParameter, String>map(this._xsemanticsUtils.inputParams(rule), _function), ", ");
  }

  public String inputParameterNames(final AuxiliaryFunction aux) {
    final Function1<JvmFormalParameter, String> _function = (JvmFormalParameter it) -> {
      return it.getName();
    };
    return IterableExtensions.join(ListExtensions.<JvmFormalParameter, String>map(aux.getParameters(), _function), ", ");
  }

  public String errorForRule(final Rule rule) {
    final Function1<RuleConclusionElement, CharSequence> _function = (RuleConclusionElement it) -> {
      StringConcatenation _builder = new StringConcatenation();
      _builder.append("\"");
      String _ruleConclusionOutputParamForError = this.ruleConclusionOutputParamForError(it);
      _builder.append(_ruleConclusionOutputParamForError);
      _builder.append("\"");
      return _builder.toString();
    };
    final Function1<RuleConclusionElement, CharSequence> _function_1 = (RuleConclusionElement it) -> {
      return this.wrapInStringRepr(this.ruleConclusionInputParamForError(it));
    };
    return this.stringForRule(rule, _function, _function_1);
  }

  public String errorForAuxiliaryFun(final AuxiliaryFunction aux) {
    CharSequence _auxFunNameInvocation = this.auxFunNameInvocation(this._xsemanticsUtils.getAuxiliaryDescription(aux).getName());
    StringConcatenation _builder = new StringConcatenation();
    _builder.append(" ");
    _builder.append("+ \"(\" + ");
    final Function1<JvmFormalParameter, CharSequence> _function = (JvmFormalParameter it) -> {
      return this.wrapInStringRepr(it.getName());
    };
    List<CharSequence> _map = ListExtensions.<JvmFormalParameter, CharSequence>map(aux.getParameters(), _function);
    StringConcatenation _builder_1 = new StringConcatenation();
    _builder_1.append(" ");
    _builder_1.append("+ \", \" + ");
    String _join = IterableExtensions.join(_map, _builder_1);
    _builder.append(_join, " ");
    _builder.append("+ \")\"");
    return (_auxFunNameInvocation + _builder.toString());
  }

  public String ruleConclusionOutputParamForError(final RuleConclusionElement element) {
    String _switchResult = null;
    boolean _matched = false;
    if (element instanceof RuleParameter) {
      _matched=true;
      _switchResult = ((RuleParameter)element).getParameter().getParameterType().getSimpleName();
    }
    if (!_matched) {
      if (element instanceof ExpressionInConclusion) {
        _matched=true;
        _switchResult = this.typeSystem.getType(element).getSimpleName();
      }
    }
    return _switchResult;
  }

  public String javaString(final String s) {
    return Strings.convertToJavaString(s);
  }

  public void resultType(final JudgmentDescription judgmentDescription, final IAppendable b) {
    this._typeReferenceSerializer.serialize(this.resultType(judgmentDescription), judgmentDescription, b);
  }

  public JvmTypeReference resultType(final JudgmentDescription e) {
    final ArrayList<JvmTypeReference> resultTypeArguments = this.resultJvmTypeReferences(e);
    JvmTypeReference resultT = null;
    int _size = resultTypeArguments.size();
    boolean _equals = (_size == 1);
    if (_equals) {
      resultT = this._typeReferences.getTypeForName(Result.class, e, 
        resultTypeArguments.get(0));
    } else {
      int _size_1 = resultTypeArguments.size();
      boolean _equals_1 = (_size_1 == 2);
      if (_equals_1) {
        resultT = this._typeReferences.getTypeForName(Result2.class, e, 
          resultTypeArguments.get(0), 
          resultTypeArguments.get(1));
      } else {
        int _size_2 = resultTypeArguments.size();
        boolean _equals_2 = (_size_2 == 3);
        if (_equals_2) {
          resultT = this._typeReferences.getTypeForName(Result3.class, e, 
            resultTypeArguments.get(0), 
            resultTypeArguments.get(1), 
            resultTypeArguments.get(2));
        } else {
          resultT = this._typeReferences.getTypeForName(Result.class, e);
        }
      }
    }
    return resultT;
  }

  public JvmTypeReference resultType(final AuxiliaryDescription e) {
    return this.typeSystem.getType(e);
  }

  public JvmTypeReference resultType(final AuxiliaryFunction e) {
    return this.typeSystem.getType(this._xsemanticsUtils.getAuxiliaryDescription(e));
  }

  public ArrayList<JvmTypeReference> resultJvmTypeReferences(final JudgmentDescription e) {
    ArrayList<JvmTypeReference> _xblockexpression = null;
    {
      final List<OutputParameter> outputParams = this._xsemanticsUtils.outputJudgmentParameters(e);
      ArrayList<JvmTypeReference> _xifexpression = null;
      int _size = outputParams.size();
      boolean _equals = (_size == 0);
      if (_equals) {
        _xifexpression = CollectionLiterals.<JvmTypeReference>newArrayList(this._typeReferences.getTypeForName(Boolean.class, e));
      } else {
        final Function1<OutputParameter, Boolean> _function = (OutputParameter it) -> {
          return Boolean.valueOf(((it.getJvmTypeReference() != null) && (it.getJvmTypeReference().getType() != null)));
        };
        final Function1<OutputParameter, JvmTypeReference> _function_1 = (OutputParameter it) -> {
          return it.getJvmTypeReference();
        };
        _xifexpression = CollectionLiterals.<JvmTypeReference>newArrayList(
          ((JvmTypeReference[])Conversions.unwrapArray(IterableExtensions.<OutputParameter, JvmTypeReference>map(IterableExtensions.<OutputParameter>filter(outputParams, _function), _function_1), JvmTypeReference.class)));
      }
      _xblockexpression = _xifexpression;
    }
    return _xblockexpression;
  }

  public void resultType(final CheckRule checkRule, final IAppendable b) {
    this._typeReferenceSerializer.serialize(this.resultType(checkRule), checkRule, b);
  }

  public JvmTypeReference resultType(final CheckRule checkRule) {
    return this._typeReferences.getTypeForName(Result.class, checkRule, 
      this._typeReferences.getTypeForName(Boolean.class, checkRule));
  }

  public JvmTypeReference errorInformationType(final EObject o) {
    return this._typeReferences.getTypeForName(ErrorInformation.class, o);
  }

  public JvmTypeReference exceptionType(final EObject o) {
    return this._typeReferences.getTypeForName(Exception.class, o);
  }

  public String emptyEnvironmentInvocation() {
    return "emptyEnvironment()";
  }

  public String environmentEntryInvocation() {
    return "environmentEntry";
  }

  public String environmentCompositionInvocation() {
    return "environmentComposition";
  }

  public String methodName(final CheckRule rule) {
    return Strings.toFirstLower(rule.getName());
  }

  public AuxiliaryDescription associatedAuxiliaryDescription(final JvmIdentifiableElement e) {
    AuxiliaryDescription _xblockexpression = null;
    {
      final EObject associated = this.associations.getPrimarySourceElement(e);
      AuxiliaryDescription _xifexpression = null;
      if ((associated instanceof AuxiliaryDescription)) {
        _xifexpression = ((AuxiliaryDescription)associated);
      } else {
        _xifexpression = null;
      }
      _xblockexpression = _xifexpression;
    }
    return _xblockexpression;
  }

  public void declareVariablesForOutputParams(final Rule rule, final ITreeAppendable appendable) {
    List<RuleParameter> _outputParams = this._xsemanticsUtils.outputParams(rule);
    for (final RuleParameter p : _outputParams) {
      {
        appendable.append("\n");
        this.declareVariableForOutputParam(p, appendable);
      }
    }
  }

  public ITreeAppendable declareVariableForOutputParam(final RuleParameter ruleParam, final ITreeAppendable appendable) {
    ITreeAppendable _xblockexpression = null;
    {
      final String outputVarName = appendable.declareVariable(ruleParam.getParameter(), ruleParam.getParameter().getSimpleName());
      final ITreeAppendable childAppendable = appendable.trace(ruleParam.getParameter(), true);
      this._typeReferenceSerializer.serialize(ruleParam.getParameter().getParameterType(), ruleParam.getParameter(), childAppendable);
      _xblockexpression = childAppendable.append(((" " + outputVarName) + " = null; // output parameter"));
    }
    return _xblockexpression;
  }

  public String expressionInConclusionMethodName(final ExpressionInConclusion e) {
    String _xblockexpression = null;
    {
      final Rule containingRule = this._xsemanticsUtils.containingRule(e);
      CharSequence _applyRuleName = this.applyRuleName(containingRule);
      String _plus = ("_" + _applyRuleName);
      String _plus_1 = (_plus + 
        "_");
      int _indexOf = containingRule.getConclusion().getConclusionElements().indexOf(e);
      _xblockexpression = (_plus_1 + Integer.valueOf(_indexOf));
    }
    return _xblockexpression;
  }

  public ITreeAppendable compileReturnResult(final Rule rule, final JvmTypeReference resultType, final ITreeAppendable result) {
    ITreeAppendable _xblockexpression = null;
    {
      final List<RuleConclusionElement> expressions = this._xsemanticsUtils.outputConclusionElements(rule);
      result.append("\n");
      result.append("return new ");
      this._typeReferenceSerializer.serialize(resultType, rule, result);
      result.append("(");
      int _size = expressions.size();
      boolean _equals = (_size == 0);
      if (_equals) {
        result.append("true");
      } else {
        final Iterator<RuleConclusionElement> iterator = expressions.iterator();
        while (iterator.hasNext()) {
          {
            final RuleConclusionElement elem = iterator.next();
            boolean _matched = false;
            if (elem instanceof RuleParameter) {
              _matched=true;
              result.append(result.getName(((RuleParameter)elem).getParameter()));
            }
            if (!_matched) {
              if (elem instanceof ExpressionInConclusion) {
                _matched=true;
                final String inputParams = this.inputParameterNames(rule);
                String _expressionInConclusionMethodName = this.expressionInConclusionMethodName(((ExpressionInConclusion)elem));
                String _plus = (_expressionInConclusionMethodName + 
                  "(");
                String _ruleEnvName = this.ruleEnvName(rule);
                String _plus_1 = (_plus + _ruleEnvName);
                String _plus_2 = (_plus_1 + 
                  ", ");
                String _plus_3 = (_plus_2 + inputParams);
                String _plus_4 = (_plus_3 + ")");
                result.append(_plus_4);
              }
            }
            boolean _hasNext = iterator.hasNext();
            if (_hasNext) {
              result.append(", ");
            }
          }
        }
      }
      _xblockexpression = result.append(");");
    }
    return _xblockexpression;
  }

  public String escapeJavaStringChars(final String s) {
    return Strings.convertToJavaString(s);
  }

  public Iterable<RuleParameter> inputEObjectParams(final Rule rule) {
    final Function1<RuleParameter, Boolean> _function = (RuleParameter it) -> {
      return Boolean.valueOf(this.typeSystem.isEObject(it.getParameter().getParameterType(), rule));
    };
    return IterableExtensions.<RuleParameter>filter(this._xsemanticsUtils.inputParams(rule), _function);
  }

  public Iterable<JvmFormalParameter> inputEObjectParams(final AuxiliaryFunction aux) {
    final Function1<JvmFormalParameter, Boolean> _function = (JvmFormalParameter it) -> {
      return Boolean.valueOf(this.typeSystem.isEObject(it.getParameterType(), aux));
    };
    return IterableExtensions.<JvmFormalParameter>filter(aux.getParameters(), _function);
  }
}
