/*
 * Decompiled with CFR 0.152.
 */
package org.apache.solr.common.cloud;

import java.io.IOException;
import java.lang.invoke.MethodHandles;
import java.nio.file.FileVisitResult;
import java.nio.file.FileVisitor;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.SimpleFileVisitor;
import java.nio.file.attribute.BasicFileAttributes;
import java.nio.file.attribute.FileAttribute;
import java.util.List;
import java.util.Locale;
import java.util.regex.Pattern;
import org.apache.solr.client.solrj.SolrServerException;
import org.apache.solr.common.cloud.SolrZkClient;
import org.apache.solr.common.cloud.ZkConfigManager;
import org.apache.solr.common.cloud.ZkCopier;
import org.apache.zookeeper.KeeperException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class ZkMaintenanceUtils {
    private static final Logger log = LoggerFactory.getLogger(MethodHandles.lookup().lookupClass());

    private ZkMaintenanceUtils() {
    }

    public static String listZnode(SolrZkClient zkClient, String path, Boolean recurse) throws KeeperException, InterruptedException, SolrServerException {
        String root = path;
        if (path.toLowerCase(Locale.ROOT).startsWith("zk:")) {
            root = path.substring(3);
        }
        if (!root.equals("/") && root.endsWith("/")) {
            root = root.substring(0, root.length() - 1);
        }
        StringBuilder sb = new StringBuilder();
        if (!recurse.booleanValue()) {
            for (String node : zkClient.getChildren(root, null, true)) {
                if (node.equals("zookeeper")) continue;
                sb.append(node).append(System.lineSeparator());
            }
            return sb.toString();
        }
        ZkMaintenanceUtils.traverseZkTree(zkClient, root, VISIT_ORDER.VISIT_PRE, znode -> {
            if (znode.startsWith("/zookeeper")) {
                return;
            }
            int iPos = znode.lastIndexOf("/");
            if (iPos > 0) {
                for (int idx = 0; idx < iPos; ++idx) {
                    sb.append(" ");
                }
                sb.append(znode.substring(iPos + 1)).append(System.lineSeparator());
            } else {
                sb.append(znode).append(System.lineSeparator());
            }
        });
        return sb.toString();
    }

    public static void zkTransfer(SolrZkClient zkClient, String src, Boolean srcIsZk, String dst, Boolean dstIsZk, Boolean recurse) throws SolrServerException, KeeperException, InterruptedException, IOException {
        if (!srcIsZk.booleanValue() && !dstIsZk.booleanValue()) {
            throw new SolrServerException("One or both of source or destination must specify ZK nodes.");
        }
        if (!recurse.booleanValue()) {
            if (srcIsZk.booleanValue()) {
                if (zkClient.getChildren(src, null, true).size() != 0) {
                    throw new SolrServerException("Zookeeper node " + src + " has children and recurse is false");
                }
            } else if (Files.isDirectory(Paths.get(src, new String[0]), new LinkOption[0])) {
                throw new SolrServerException("Local path " + Paths.get(src, new String[0]).toAbsolutePath() + " is a directory and recurse is false");
            }
        }
        if (!srcIsZk.booleanValue() && !dstIsZk.booleanValue()) {
            throw new SolrServerException("At least one of the source and dest parameters must be prefixed with 'zk:' ");
        }
        dst = ZkMaintenanceUtils.normalizeDest(src, dst);
        if (srcIsZk.booleanValue() && dstIsZk.booleanValue()) {
            ZkMaintenanceUtils.traverseZkTree(zkClient, src, VISIT_ORDER.VISIT_PRE, new ZkCopier(zkClient, src, dst));
            return;
        }
        if (dstIsZk.booleanValue()) {
            ZkMaintenanceUtils.uploadToZK(zkClient, Paths.get(src, new String[0]), dst, null);
            return;
        }
        if (zkClient.getChildren(src, null, true).size() > 0) {
            ZkMaintenanceUtils.downloadFromZK(zkClient, src, Paths.get(dst, new String[0]));
            return;
        }
        if (Files.isDirectory(Paths.get(dst, new String[0]), new LinkOption[0])) {
            if (!dst.endsWith("/")) {
                dst = dst + "/";
            }
            dst = ZkMaintenanceUtils.normalizeDest(src, dst);
        }
        byte[] data = zkClient.getData(src, null, null, true);
        Path filename = Paths.get(dst, new String[0]);
        Files.createDirectories(filename.getParent(), new FileAttribute[0]);
        log.info("Writing file {}", (Object)filename);
        Files.write(filename, data, new OpenOption[0]);
    }

    private static String normalizeDest(String srcName, String dstName) {
        if (dstName.endsWith("/")) {
            int pos = srcName.lastIndexOf("/");
            dstName = pos < 0 ? dstName + srcName : dstName + srcName.substring(pos + 1);
        } else if (dstName.equals(".")) {
            dstName = Paths.get(".", new String[0]).normalize().toAbsolutePath().toString();
        }
        return dstName;
    }

    public static void moveZnode(SolrZkClient zkClient, String src, String dst) throws SolrServerException, KeeperException, InterruptedException {
        String destName = ZkMaintenanceUtils.normalizeDest(src, dst);
        if (zkClient.getChildren(src, null, true).size() == 0) {
            zkClient.makePath(destName, false, true);
            zkClient.setData(destName, zkClient.getData(src, null, null, true), true);
        } else {
            ZkMaintenanceUtils.traverseZkTree(zkClient, src, VISIT_ORDER.VISIT_PRE, new ZkCopier(zkClient, src, destName));
        }
        ZkMaintenanceUtils.checkAllZnodesThere(zkClient, src, destName);
        ZkMaintenanceUtils.clean(zkClient, src);
    }

    private static void checkAllZnodesThere(SolrZkClient zkClient, String src, String dst) throws KeeperException, InterruptedException, SolrServerException {
        for (String node : zkClient.getChildren(src, null, true)) {
            if (!zkClient.exists(dst + "/" + node, true).booleanValue()) {
                throw new SolrServerException("mv command did not move node " + dst + "/" + node + " source left intact");
            }
            ZkMaintenanceUtils.checkAllZnodesThere(zkClient, src + "/" + node, dst + "/" + node);
        }
    }

    public static void downConfig(SolrZkClient zkClient, String confName, Path confPath) throws IOException {
        ZkConfigManager manager = new ZkConfigManager(zkClient);
        manager.downloadConfigDir(confName, confPath);
    }

    public static void upConfig(SolrZkClient zkClient, Path confPath, String confName) throws IOException {
        ZkConfigManager manager = new ZkConfigManager(zkClient);
        manager.uploadConfigDir(confPath, confName);
    }

    public static void clean(SolrZkClient zkClient, String path) throws InterruptedException, KeeperException {
        ZkMaintenanceUtils.traverseZkTree(zkClient, path, VISIT_ORDER.VISIT_POST, znode -> {
            try {
                if (!znode.equals("/")) {
                    try {
                        zkClient.delete(znode, -1, true);
                    }
                    catch (KeeperException.NotEmptyException e) {
                        ZkMaintenanceUtils.clean(zkClient, znode);
                    }
                }
            }
            catch (KeeperException.NoNodeException r) {
                return;
            }
        });
    }

    public static void uploadToZK(final SolrZkClient zkClient, final Path rootPath, final String zkPath, final Pattern filenameExclusions) throws IOException {
        if (!Files.exists(rootPath, new LinkOption[0])) {
            throw new IOException("Path " + rootPath + " does not exist");
        }
        Files.walkFileTree(rootPath, (FileVisitor<? super Path>)new SimpleFileVisitor<Path>(){

            @Override
            public FileVisitResult visitFile(Path file, BasicFileAttributes attrs) throws IOException {
                String filename = file.getFileName().toString();
                if (filenameExclusions != null && filenameExclusions.matcher(filename).matches()) {
                    log.info("uploadToZK skipping '{}' due to filenameExclusions '{}'", (Object)filename, (Object)filenameExclusions);
                    return FileVisitResult.CONTINUE;
                }
                String zkNode = ZkMaintenanceUtils.createZkNodeName(zkPath, rootPath, file);
                try {
                    zkClient.makePath(zkNode, file.toFile(), false, true);
                }
                catch (InterruptedException | KeeperException e) {
                    throw new IOException("Error uploading file " + file.toString() + " to zookeeper path " + zkNode, SolrZkClient.checkInterrupted(e));
                }
                return FileVisitResult.CONTINUE;
            }

            @Override
            public FileVisitResult preVisitDirectory(Path dir, BasicFileAttributes attrs) throws IOException {
                return dir.getFileName().toString().startsWith(".") ? FileVisitResult.SKIP_SUBTREE : FileVisitResult.CONTINUE;
            }
        });
    }

    public static void downloadFromZK(SolrZkClient zkClient, String zkPath, Path dir) throws IOException {
        try {
            List<String> files = zkClient.getChildren(zkPath, null, true);
            Files.createDirectories(dir, new FileAttribute[0]);
            for (String file : files) {
                List<String> children = zkClient.getChildren(zkPath + "/" + file, null, true);
                if (children.size() == 0) {
                    byte[] data = zkClient.getData(zkPath + "/" + file, null, null, true);
                    Path filename = dir.resolve(file);
                    log.info("Writing file {}", (Object)filename);
                    Files.write(filename, data, new OpenOption[0]);
                    continue;
                }
                ZkMaintenanceUtils.downloadFromZK(zkClient, zkPath + "/" + file, dir.resolve(file));
            }
        }
        catch (InterruptedException | KeeperException e) {
            throw new IOException("Error downloading files from zookeeper path " + zkPath + " to " + dir.toString(), SolrZkClient.checkInterrupted(e));
        }
    }

    public static void traverseZkTree(SolrZkClient zkClient, String path, VISIT_ORDER visitOrder, ZkVisitor visitor) throws InterruptedException, KeeperException {
        List<String> children;
        if (visitOrder == VISIT_ORDER.VISIT_PRE) {
            visitor.visit(path);
        }
        try {
            children = zkClient.getChildren(path, null, true);
        }
        catch (KeeperException.NoNodeException r) {
            return;
        }
        for (String string : children) {
            if (path.equals("/") && string.equals("zookeeper") || path.startsWith("/zookeeper")) continue;
            if (path.equals("/")) {
                ZkMaintenanceUtils.traverseZkTree(zkClient, path + string, visitOrder, visitor);
                continue;
            }
            ZkMaintenanceUtils.traverseZkTree(zkClient, path + "/" + string, visitOrder, visitor);
        }
        if (visitOrder == VISIT_ORDER.VISIT_POST) {
            visitor.visit(path);
        }
    }

    public static String createZkNodeName(String zkRoot, Path root, Path file) {
        String relativePath = root.relativize(file).toString();
        String separator = root.getFileSystem().getSeparator();
        if ("\\".equals(separator)) {
            relativePath = relativePath.replaceAll("\\\\", "/");
        }
        if (relativePath.length() == 0) {
            return zkRoot;
        }
        return zkRoot + "/" + relativePath;
    }

    public static enum VISIT_ORDER {
        VISIT_PRE,
        VISIT_POST;

    }

    @FunctionalInterface
    public static interface ZkVisitor {
        public void visit(String var1) throws InterruptedException, KeeperException;
    }
}

