/*
 * Decompiled with CFR 0.152.
 */
package org.apache.kafka.common.security.authenticator;

import java.io.IOException;
import java.nio.ByteBuffer;
import java.security.Principal;
import java.security.PrivilegedActionException;
import java.security.PrivilegedExceptionAction;
import java.util.Arrays;
import java.util.Iterator;
import java.util.Map;
import java.util.Set;
import javax.security.auth.Subject;
import javax.security.sasl.Sasl;
import javax.security.sasl.SaslClient;
import javax.security.sasl.SaslException;
import org.apache.kafka.clients.NetworkClient;
import org.apache.kafka.common.KafkaException;
import org.apache.kafka.common.errors.IllegalSaslStateException;
import org.apache.kafka.common.errors.SaslAuthenticationException;
import org.apache.kafka.common.errors.UnsupportedSaslMechanismException;
import org.apache.kafka.common.network.Authenticator;
import org.apache.kafka.common.network.NetworkReceive;
import org.apache.kafka.common.network.NetworkSend;
import org.apache.kafka.common.network.Send;
import org.apache.kafka.common.network.TransportLayer;
import org.apache.kafka.common.protocol.ApiKeys;
import org.apache.kafka.common.protocol.Errors;
import org.apache.kafka.common.protocol.types.SchemaException;
import org.apache.kafka.common.requests.AbstractResponse;
import org.apache.kafka.common.requests.ApiVersionsRequest;
import org.apache.kafka.common.requests.ApiVersionsResponse;
import org.apache.kafka.common.requests.RequestHeader;
import org.apache.kafka.common.requests.SaslAuthenticateRequest;
import org.apache.kafka.common.requests.SaslAuthenticateResponse;
import org.apache.kafka.common.requests.SaslHandshakeRequest;
import org.apache.kafka.common.requests.SaslHandshakeResponse;
import org.apache.kafka.common.security.auth.AuthenticateCallbackHandler;
import org.apache.kafka.common.security.auth.KafkaPrincipal;
import org.apache.kafka.common.utils.Utils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class SaslClientAuthenticator
implements Authenticator {
    private static final Logger LOG = LoggerFactory.getLogger(SaslClientAuthenticator.class);
    private static final short DISABLE_KAFKA_SASL_AUTHENTICATE_HEADER = -1;
    private final Subject subject;
    private final String servicePrincipal;
    private final String host;
    private final String node;
    private final String mechanism;
    private final TransportLayer transportLayer;
    private final SaslClient saslClient;
    private final Map<String, ?> configs;
    private final String clientPrincipalName;
    private final AuthenticateCallbackHandler callbackHandler;
    private NetworkReceive netInBuffer;
    private Send netOutBuffer;
    private SaslState saslState;
    private SaslState pendingSaslState;
    private int correlationId;
    private RequestHeader currentRequestHeader;
    private short saslAuthenticateVersion;

    public SaslClientAuthenticator(Map<String, ?> configs, AuthenticateCallbackHandler callbackHandler, String node, Subject subject, String servicePrincipal, String host, String mechanism, boolean handshakeRequestEnable, TransportLayer transportLayer) throws IOException {
        this.node = node;
        this.subject = subject;
        this.callbackHandler = callbackHandler;
        this.host = host;
        this.servicePrincipal = servicePrincipal;
        this.mechanism = mechanism;
        this.correlationId = -1;
        this.transportLayer = transportLayer;
        this.configs = configs;
        this.saslAuthenticateVersion = (short)-1;
        try {
            this.setSaslState(handshakeRequestEnable ? SaslState.SEND_APIVERSIONS_REQUEST : SaslState.INITIAL);
            this.clientPrincipalName = mechanism.equals("GSSAPI") ? SaslClientAuthenticator.firstPrincipal(subject) : null;
            this.saslClient = this.createSaslClient();
        }
        catch (Exception e) {
            throw new SaslAuthenticationException("Failed to configure SaslClientAuthenticator", e);
        }
    }

    private SaslClient createSaslClient() {
        try {
            return Subject.doAs(this.subject, new PrivilegedExceptionAction<SaslClient>(){

                @Override
                public SaslClient run() throws SaslException {
                    Object[] mechs = new String[]{SaslClientAuthenticator.this.mechanism};
                    LOG.debug("Creating SaslClient: client={};service={};serviceHostname={};mechs={}", SaslClientAuthenticator.this.clientPrincipalName, SaslClientAuthenticator.this.servicePrincipal, SaslClientAuthenticator.this.host, Arrays.toString(mechs));
                    return Sasl.createSaslClient((String[])mechs, SaslClientAuthenticator.this.clientPrincipalName, SaslClientAuthenticator.this.servicePrincipal, SaslClientAuthenticator.this.host, SaslClientAuthenticator.this.configs, SaslClientAuthenticator.this.callbackHandler);
                }
            });
        }
        catch (PrivilegedActionException e) {
            throw new SaslAuthenticationException("Failed to create SaslClient with mechanism " + this.mechanism, e.getCause());
        }
    }

    @Override
    public void authenticate() throws IOException {
        short saslHandshakeVersion = 0;
        if (this.netOutBuffer != null && !this.flushNetOutBufferAndUpdateInterestOps()) {
            return;
        }
        switch (this.saslState) {
            case SEND_APIVERSIONS_REQUEST: {
                ApiVersionsRequest apiVersionsRequest = new ApiVersionsRequest(0);
                this.send(apiVersionsRequest.toSend(this.node, this.nextRequestHeader(ApiKeys.API_VERSIONS, apiVersionsRequest.version())));
                this.setSaslState(SaslState.RECEIVE_APIVERSIONS_RESPONSE);
                break;
            }
            case RECEIVE_APIVERSIONS_RESPONSE: {
                ApiVersionsResponse apiVersionsResponse = (ApiVersionsResponse)this.receiveKafkaResponse();
                if (apiVersionsResponse == null) break;
                saslHandshakeVersion = apiVersionsResponse.apiVersion((short)ApiKeys.SASL_HANDSHAKE.id).maxVersion;
                ApiVersionsResponse.ApiVersion authenticateVersion = apiVersionsResponse.apiVersion(ApiKeys.SASL_AUTHENTICATE.id);
                if (authenticateVersion != null) {
                    this.saslAuthenticateVersion((short)Math.min(authenticateVersion.maxVersion, ApiKeys.SASL_AUTHENTICATE.latestVersion()));
                }
                this.setSaslState(SaslState.SEND_HANDSHAKE_REQUEST);
            }
            case SEND_HANDSHAKE_REQUEST: {
                SaslHandshakeRequest handshakeRequest = this.createSaslHandshakeRequest(saslHandshakeVersion);
                this.send(handshakeRequest.toSend(this.node, this.nextRequestHeader(ApiKeys.SASL_HANDSHAKE, handshakeRequest.version())));
                this.setSaslState(SaslState.RECEIVE_HANDSHAKE_RESPONSE);
                break;
            }
            case RECEIVE_HANDSHAKE_RESPONSE: {
                SaslHandshakeResponse handshakeResponse = (SaslHandshakeResponse)this.receiveKafkaResponse();
                if (handshakeResponse == null) break;
                this.handleSaslHandshakeResponse(handshakeResponse);
                this.setSaslState(SaslState.INITIAL);
            }
            case INITIAL: {
                this.sendSaslClientToken(new byte[0], true);
                this.setSaslState(SaslState.INTERMEDIATE);
                break;
            }
            case INTERMEDIATE: {
                boolean noResponsesPending;
                byte[] serverToken = this.receiveToken();
                boolean bl = noResponsesPending = serverToken != null && !this.sendSaslClientToken(serverToken, false);
                if (!this.saslClient.isComplete()) break;
                if (this.saslAuthenticateVersion == -1 || noResponsesPending) {
                    this.setSaslState(SaslState.COMPLETE);
                    break;
                }
                this.setSaslState(SaslState.CLIENT_COMPLETE);
                break;
            }
            case CLIENT_COMPLETE: {
                byte[] serverResponse = this.receiveToken();
                if (serverResponse == null) break;
                this.setSaslState(SaslState.COMPLETE);
                break;
            }
            case COMPLETE: {
                break;
            }
            case FAILED: {
                throw new IllegalStateException("SASL handshake has already failed");
            }
        }
    }

    private RequestHeader nextRequestHeader(ApiKeys apiKey, short version) {
        String clientId = (String)this.configs.get("client.id");
        this.currentRequestHeader = new RequestHeader(apiKey, version, clientId, this.correlationId++);
        return this.currentRequestHeader;
    }

    protected SaslHandshakeRequest createSaslHandshakeRequest(short version) {
        return new SaslHandshakeRequest.Builder(this.mechanism).build(version);
    }

    protected void saslAuthenticateVersion(short version) {
        this.saslAuthenticateVersion = version;
    }

    private void setSaslState(SaslState saslState) {
        if (this.netOutBuffer != null && !this.netOutBuffer.completed()) {
            this.pendingSaslState = saslState;
        } else {
            this.pendingSaslState = null;
            this.saslState = saslState;
            LOG.debug("Set SASL client state to {}", (Object)saslState);
            if (saslState == SaslState.COMPLETE) {
                this.transportLayer.removeInterestOps(4);
            }
        }
    }

    private boolean sendSaslClientToken(byte[] serverToken, boolean isInitial) throws IOException {
        byte[] saslToken;
        if (!this.saslClient.isComplete() && (saslToken = this.createSaslToken(serverToken, isInitial)) != null) {
            ByteBuffer tokenBuf = ByteBuffer.wrap(saslToken);
            if (this.saslAuthenticateVersion != -1) {
                SaslAuthenticateRequest request = new SaslAuthenticateRequest.Builder(tokenBuf).build(this.saslAuthenticateVersion);
                tokenBuf = request.serialize(this.nextRequestHeader(ApiKeys.SASL_AUTHENTICATE, this.saslAuthenticateVersion));
            }
            this.send(new NetworkSend(this.node, tokenBuf));
            return true;
        }
        return false;
    }

    private void send(Send send) throws IOException {
        try {
            this.netOutBuffer = send;
            this.flushNetOutBufferAndUpdateInterestOps();
        }
        catch (IOException e) {
            this.setSaslState(SaslState.FAILED);
            throw e;
        }
    }

    private boolean flushNetOutBufferAndUpdateInterestOps() throws IOException {
        boolean flushedCompletely = this.flushNetOutBuffer();
        if (flushedCompletely) {
            this.transportLayer.removeInterestOps(4);
            if (this.pendingSaslState != null) {
                this.setSaslState(this.pendingSaslState);
            }
        } else {
            this.transportLayer.addInterestOps(4);
        }
        return flushedCompletely;
    }

    private byte[] receiveResponseOrToken() throws IOException {
        if (this.netInBuffer == null) {
            this.netInBuffer = new NetworkReceive(this.node);
        }
        this.netInBuffer.readFrom(this.transportLayer);
        byte[] serverPacket = null;
        if (this.netInBuffer.complete()) {
            this.netInBuffer.payload().rewind();
            serverPacket = new byte[this.netInBuffer.payload().remaining()];
            this.netInBuffer.payload().get(serverPacket, 0, serverPacket.length);
            this.netInBuffer = null;
        }
        return serverPacket;
    }

    @Override
    public KafkaPrincipal principal() {
        return new KafkaPrincipal("User", this.clientPrincipalName);
    }

    @Override
    public boolean complete() {
        return this.saslState == SaslState.COMPLETE;
    }

    @Override
    public void close() throws IOException {
        if (this.saslClient != null) {
            this.saslClient.dispose();
        }
    }

    private byte[] receiveToken() throws IOException {
        if (this.saslAuthenticateVersion == -1) {
            return this.receiveResponseOrToken();
        }
        SaslAuthenticateResponse response = (SaslAuthenticateResponse)this.receiveKafkaResponse();
        if (response != null) {
            Errors error = response.error();
            if (error != Errors.NONE) {
                this.setSaslState(SaslState.FAILED);
                String errMsg = response.errorMessage();
                throw errMsg == null ? error.exception() : error.exception(errMsg);
            }
            return Utils.readBytes(response.saslAuthBytes());
        }
        return null;
    }

    private byte[] createSaslToken(final byte[] saslToken, boolean isInitial) throws SaslException {
        if (saslToken == null) {
            throw new IllegalSaslStateException("Error authenticating with the Kafka Broker: received a `null` saslToken.");
        }
        try {
            if (isInitial && !this.saslClient.hasInitialResponse()) {
                return saslToken;
            }
            return Subject.doAs(this.subject, new PrivilegedExceptionAction<byte[]>(){

                @Override
                public byte[] run() throws SaslException {
                    return SaslClientAuthenticator.this.saslClient.evaluateChallenge(saslToken);
                }
            });
        }
        catch (PrivilegedActionException e) {
            String error = "An error: (" + e + ") occurred when evaluating SASL token received from the Kafka Broker.";
            String unknownServerErrorText = "(Mechanism level: Server not found in Kerberos database (7) - UNKNOWN_SERVER)";
            if (e.toString().contains("(Mechanism level: Server not found in Kerberos database (7) - UNKNOWN_SERVER)")) {
                error = error + " This may be caused by Java's being unable to resolve the Kafka Broker's hostname correctly. You may want to try to adding '-Dsun.net.spi.nameservice.provider.1=dns,sun' to your client's JVMFLAGS environment. Users must configure FQDN of kafka brokers when authenticating using SASL and `socketChannel.socket().getInetAddress().getHostName()` must match the hostname in `principal/hostname@realm`";
            }
            error = error + " Kafka Client will go to AUTHENTICATION_FAILED state.";
            throw new SaslAuthenticationException(error, e.getCause());
        }
    }

    private boolean flushNetOutBuffer() throws IOException {
        if (!this.netOutBuffer.completed()) {
            this.netOutBuffer.writeTo(this.transportLayer);
        }
        return this.netOutBuffer.completed();
    }

    private AbstractResponse receiveKafkaResponse() throws IOException {
        try {
            byte[] responseBytes = this.receiveResponseOrToken();
            if (responseBytes == null) {
                return null;
            }
            AbstractResponse response = NetworkClient.parseResponse(ByteBuffer.wrap(responseBytes), this.currentRequestHeader);
            this.currentRequestHeader = null;
            return response;
        }
        catch (IllegalArgumentException | SchemaException e) {
            LOG.debug("Invalid SASL mechanism response, server may be expecting only GSSAPI tokens");
            this.setSaslState(SaslState.FAILED);
            throw new IllegalSaslStateException("Invalid SASL mechanism response, server may be expecting a different protocol", e);
        }
    }

    private void handleSaslHandshakeResponse(SaslHandshakeResponse response) {
        Errors error = response.error();
        if (error != Errors.NONE) {
            this.setSaslState(SaslState.FAILED);
        }
        switch (error) {
            case NONE: {
                break;
            }
            case UNSUPPORTED_SASL_MECHANISM: {
                throw new UnsupportedSaslMechanismException(String.format("Client SASL mechanism '%s' not enabled in the server, enabled mechanisms are %s", this.mechanism, response.enabledMechanisms()));
            }
            case ILLEGAL_SASL_STATE: {
                throw new IllegalSaslStateException(String.format("Unexpected handshake request with client mechanism %s, enabled mechanisms are %s", this.mechanism, response.enabledMechanisms()));
            }
            default: {
                throw new IllegalSaslStateException(String.format("Unknown error code %s, client mechanism is %s, enabled mechanisms are %s", new Object[]{response.error(), this.mechanism, response.enabledMechanisms()}));
            }
        }
    }

    static final String firstPrincipal(Subject subject) {
        Set<Principal> principals;
        Set<Principal> set = principals = subject.getPrincipals();
        synchronized (set) {
            Iterator<Principal> iterator2 = principals.iterator();
            if (iterator2.hasNext()) {
                return iterator2.next().getName();
            }
            throw new KafkaException("Principal could not be determined from Subject, this may be a transient failure due to Kerberos re-login");
        }
    }

    public static enum SaslState {
        SEND_APIVERSIONS_REQUEST,
        RECEIVE_APIVERSIONS_RESPONSE,
        SEND_HANDSHAKE_REQUEST,
        RECEIVE_HANDSHAKE_RESPONSE,
        INITIAL,
        INTERMEDIATE,
        CLIENT_COMPLETE,
        COMPLETE,
        FAILED;

    }
}

