/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.fs.s3;

import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.net.URI;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.concurrent.TimeUnit;
import org.apache.hadoop.classification.InterfaceAudience;
import org.apache.hadoop.classification.InterfaceStability;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.FSDataInputStream;
import org.apache.hadoop.fs.FSDataOutputStream;
import org.apache.hadoop.fs.FileAlreadyExistsException;
import org.apache.hadoop.fs.FileStatus;
import org.apache.hadoop.fs.FileSystem;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.fs.permission.FsPermission;
import org.apache.hadoop.fs.s3.Block;
import org.apache.hadoop.fs.s3.FileSystemStore;
import org.apache.hadoop.fs.s3.INode;
import org.apache.hadoop.fs.s3.Jets3tFileSystemStore;
import org.apache.hadoop.fs.s3.S3Exception;
import org.apache.hadoop.fs.s3.S3InputStream;
import org.apache.hadoop.fs.s3.S3OutputStream;
import org.apache.hadoop.io.retry.RetryPolicies;
import org.apache.hadoop.io.retry.RetryPolicy;
import org.apache.hadoop.io.retry.RetryProxy;
import org.apache.hadoop.util.Progressable;

@InterfaceAudience.Public
@InterfaceStability.Stable
public class S3FileSystem
extends FileSystem {
    private URI uri;
    private FileSystemStore store;
    private Path workingDir;

    public S3FileSystem() {
    }

    public S3FileSystem(FileSystemStore store) {
        this.store = store;
    }

    public String getScheme() {
        return "s3";
    }

    public URI getUri() {
        return this.uri;
    }

    public void initialize(URI uri, Configuration conf) throws IOException {
        super.initialize(uri, conf);
        if (this.store == null) {
            this.store = S3FileSystem.createDefaultStore(conf);
        }
        this.store.initialize(uri, conf);
        this.setConf(conf);
        this.uri = URI.create(uri.getScheme() + "://" + uri.getAuthority());
        this.workingDir = new Path("/user", System.getProperty("user.name")).makeQualified((FileSystem)this);
    }

    private static FileSystemStore createDefaultStore(Configuration conf) {
        Jets3tFileSystemStore store = new Jets3tFileSystemStore();
        RetryPolicy basePolicy = RetryPolicies.retryUpToMaximumCountWithFixedSleep((int)conf.getInt("fs.s3.maxRetries", 4), (long)conf.getLong("fs.s3.sleepTimeSeconds", 10L), (TimeUnit)TimeUnit.SECONDS);
        HashMap<Class, RetryPolicy> exceptionToPolicyMap = new HashMap<Class, RetryPolicy>();
        exceptionToPolicyMap.put(IOException.class, basePolicy);
        exceptionToPolicyMap.put(S3Exception.class, basePolicy);
        RetryPolicy methodPolicy = RetryPolicies.retryByException((RetryPolicy)RetryPolicies.TRY_ONCE_THEN_FAIL, exceptionToPolicyMap);
        HashMap<String, RetryPolicy> methodNameToPolicyMap = new HashMap<String, RetryPolicy>();
        methodNameToPolicyMap.put("storeBlock", methodPolicy);
        methodNameToPolicyMap.put("retrieveBlock", methodPolicy);
        return (FileSystemStore)RetryProxy.create(FileSystemStore.class, (Object)store, methodNameToPolicyMap);
    }

    public Path getWorkingDirectory() {
        return this.workingDir;
    }

    public void setWorkingDirectory(Path dir) {
        this.workingDir = this.makeAbsolute(dir);
    }

    private Path makeAbsolute(Path path) {
        if (path.isAbsolute()) {
            return path;
        }
        return new Path(this.workingDir, path);
    }

    public boolean mkdirs(Path path, FsPermission permission) throws IOException {
        Path absolutePath = this.makeAbsolute(path);
        ArrayList<Path> paths = new ArrayList<Path>();
        do {
            paths.add(0, absolutePath);
        } while ((absolutePath = absolutePath.getParent()) != null);
        boolean result = true;
        for (Path p : paths) {
            result &= this.mkdir(p);
        }
        return result;
    }

    private boolean mkdir(Path path) throws IOException {
        Path absolutePath = this.makeAbsolute(path);
        INode inode = this.store.retrieveINode(absolutePath);
        if (inode == null) {
            this.store.storeINode(absolutePath, INode.DIRECTORY_INODE);
        } else if (inode.isFile()) {
            throw new IOException(String.format("Can't make directory for path %s since it is a file.", absolutePath));
        }
        return true;
    }

    public boolean isFile(Path path) throws IOException {
        INode inode = this.store.retrieveINode(this.makeAbsolute(path));
        if (inode == null) {
            return false;
        }
        return inode.isFile();
    }

    private INode checkFile(Path path) throws IOException {
        INode inode = this.store.retrieveINode(this.makeAbsolute(path));
        if (inode == null) {
            throw new IOException("No such file.");
        }
        if (inode.isDirectory()) {
            throw new IOException("Path " + path + " is a directory.");
        }
        return inode;
    }

    public FileStatus[] listStatus(Path f) throws IOException {
        Path absolutePath = this.makeAbsolute(f);
        INode inode = this.store.retrieveINode(absolutePath);
        if (inode == null) {
            throw new FileNotFoundException("File " + f + " does not exist.");
        }
        if (inode.isFile()) {
            return new FileStatus[]{new S3FileStatus(f.makeQualified((FileSystem)this), inode)};
        }
        ArrayList<FileStatus> ret = new ArrayList<FileStatus>();
        for (Path p : this.store.listSubPaths(absolutePath)) {
            ret.add(this.getFileStatus(p.makeQualified((FileSystem)this)));
        }
        return ret.toArray(new FileStatus[0]);
    }

    public FSDataOutputStream append(Path f, int bufferSize, Progressable progress) throws IOException {
        throw new IOException("Not supported");
    }

    /*
     * Enabled force condition propagation
     * Lifted jumps to return sites
     */
    public FSDataOutputStream create(Path file, FsPermission permission, boolean overwrite, int bufferSize, short replication, long blockSize, Progressable progress) throws IOException {
        INode inode = this.store.retrieveINode(this.makeAbsolute(file));
        if (inode != null) {
            if (!overwrite) throw new FileAlreadyExistsException("File already exists: " + file);
            this.delete(file, true);
            return new FSDataOutputStream((OutputStream)new S3OutputStream(this.getConf(), this.store, this.makeAbsolute(file), blockSize, progress, bufferSize), this.statistics);
        } else {
            Path parent = file.getParent();
            if (parent == null || this.mkdirs(parent)) return new FSDataOutputStream((OutputStream)new S3OutputStream(this.getConf(), this.store, this.makeAbsolute(file), blockSize, progress, bufferSize), this.statistics);
            throw new IOException("Mkdirs failed to create " + parent.toString());
        }
    }

    public FSDataInputStream open(Path path, int bufferSize) throws IOException {
        INode inode = this.checkFile(path);
        return new FSDataInputStream((InputStream)((Object)new S3InputStream(this.getConf(), this.store, inode, this.statistics)));
    }

    public boolean rename(Path src, Path dst) throws IOException {
        INode dstParentINode;
        Path absoluteSrc = this.makeAbsolute(src);
        INode srcINode = this.store.retrieveINode(absoluteSrc);
        if (srcINode == null) {
            return false;
        }
        Path absoluteDst = this.makeAbsolute(dst);
        INode dstINode = this.store.retrieveINode(absoluteDst);
        if (dstINode != null && dstINode.isDirectory()) {
            absoluteDst = new Path(absoluteDst, absoluteSrc.getName());
            dstINode = this.store.retrieveINode(absoluteDst);
        }
        if (dstINode != null) {
            return false;
        }
        Path dstParent = absoluteDst.getParent();
        if (dstParent != null && ((dstParentINode = this.store.retrieveINode(dstParent)) == null || dstParentINode.isFile())) {
            return false;
        }
        return this.renameRecursive(absoluteSrc, absoluteDst);
    }

    private boolean renameRecursive(Path src, Path dst) throws IOException {
        INode srcINode = this.store.retrieveINode(src);
        this.store.storeINode(dst, srcINode);
        this.store.deleteINode(src);
        if (srcINode.isDirectory()) {
            for (Path oldSrc : this.store.listDeepSubPaths(src)) {
                INode inode = this.store.retrieveINode(oldSrc);
                if (inode == null) {
                    return false;
                }
                String oldSrcPath = oldSrc.toUri().getPath();
                String srcPath = src.toUri().getPath();
                String dstPath = dst.toUri().getPath();
                Path newDst = new Path(oldSrcPath.replaceFirst(srcPath, dstPath));
                this.store.storeINode(newDst, inode);
                this.store.deleteINode(oldSrc);
            }
        }
        return true;
    }

    public boolean delete(Path path, boolean recursive) throws IOException {
        Path absolutePath = this.makeAbsolute(path);
        INode inode = this.store.retrieveINode(absolutePath);
        if (inode == null) {
            return false;
        }
        if (inode.isFile()) {
            this.store.deleteINode(absolutePath);
            for (Block block : inode.getBlocks()) {
                this.store.deleteBlock(block);
            }
        } else {
            FileStatus[] contents = null;
            try {
                contents = this.listStatus(absolutePath);
            }
            catch (FileNotFoundException fnfe) {
                return false;
            }
            if (contents.length != 0 && !recursive) {
                throw new IOException("Directory " + path.toString() + " is not empty.");
            }
            for (FileStatus p : contents) {
                if (this.delete(p.getPath(), recursive)) continue;
                return false;
            }
            this.store.deleteINode(absolutePath);
        }
        return true;
    }

    public FileStatus getFileStatus(Path f) throws IOException {
        INode inode = this.store.retrieveINode(this.makeAbsolute(f));
        if (inode == null) {
            throw new FileNotFoundException(f + ": No such file or directory.");
        }
        return new S3FileStatus(f.makeQualified((FileSystem)this), inode);
    }

    public long getDefaultBlockSize() {
        return this.getConf().getLong("fs.s3.block.size", 0x4000000L);
    }

    public String getCanonicalServiceName() {
        return null;
    }

    void dump() throws IOException {
        this.store.dump();
    }

    void purge() throws IOException {
        this.store.purge();
    }

    private static class S3FileStatus
    extends FileStatus {
        S3FileStatus(Path f, INode inode) throws IOException {
            super(S3FileStatus.findLength(inode), inode.isDirectory(), 1, S3FileStatus.findBlocksize(inode), 0L, f);
        }

        private static long findLength(INode inode) {
            if (!inode.isDirectory()) {
                long length = 0L;
                for (Block block : inode.getBlocks()) {
                    length += block.getLength();
                }
                return length;
            }
            return 0L;
        }

        private static long findBlocksize(INode inode) {
            Block[] ret = inode.getBlocks();
            return ret == null ? 0L : ret[0].getLength();
        }
    }
}

