/*
 * BSD-style license; for more info see http://pmd.sourceforge.net/license.html
 */

package net.sourceforge.pmd.util.fxdesigner.util.codearea;

import java.util.Collection;
import java.util.Collections;
import java.util.EnumSet;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.stream.Collectors;

import org.apache.commons.lang3.StringUtils;
import org.checkerframework.checker.nullness.qual.NonNull;
import org.checkerframework.checker.nullness.qual.Nullable;
import org.fxmisc.richtext.model.StyleSpans;

import net.sourceforge.pmd.lang.ast.Node;
import net.sourceforge.pmd.util.fxdesigner.util.RichRunnable;
import net.sourceforge.pmd.util.fxdesigner.util.codearea.HighlightLayerCodeArea.LayerId;

import javafx.application.Platform;
import javafx.beans.NamedArg;
import javafx.scene.control.IndexRange;


/**
 * Code area that can manipulate different layers of styling independently,
 * in addition to syntax highlighting. Layers are identified by a {@link LayerId},
 * which are listed in an enum.
 *
 * @param <K> Enum type listing the layer ids to use
 * @author Clément Fournier
 * @since 6.5.0
 */
public class HighlightLayerCodeArea<K extends Enum<K> & LayerId> extends SyntaxHighlightingCodeArea {


    /** Contains the highlighting layers. */
    private final Map<K, StyleLayer> layersById;


    /**
     * Builds a new code area with the given enum type as layer id provider.
     * Constants of the enum will identify layers of the code area.
     *
     * @param idEnum Enum type
     */
    // the annotation lets the value be passed from FXML
    public HighlightLayerCodeArea(@NamedArg("idEnum") Class<K> idEnum) {
        super();

        this.layersById = EnumSet.allOf(idEnum)
                                 .stream()
                                 .collect(Collectors.toConcurrentMap(id -> id, id -> new StyleLayer()));
    }


    /**
     * Styles some nodes in a given layer and updates the visual appearance of the area.
     *
     * <p>Each layer has its own style class, that is assigned to the nodes
     * that belong to it.
     *
     * @param nodes      Nodes to style
     * @param layerId    Id of the layer in which to save the node highlight
     * @param resetLayer Whether to replace the contents of the layer with the
     *                   styling for these nodes, or just add them.
     */
    // TODO there's no reason to only be able to style nodes, in fact, this causes problem
    // to highlight errors that are not bound to a node, eg parsing errors
    // We'll need to abstract away NodeStyleSpan
    public void styleNodes(Collection<? extends Node> nodes, K layerId, boolean resetLayer) {
        updateStyling(styleNodesUpdate(nodes, layerId, resetLayer));
    }


    public RichRunnable styleNodesUpdate(Collection<? extends Node> nodes, K layerId, boolean resetLayer) {
        Objects.requireNonNull(nodes, "Pass an empty collection to represent absence, not null!");

        if (nodes.isEmpty() && resetLayer) {
            return layersById.get(layerId)::clearStyles;
        }

        List<NodeStyleSpan> wrappedNodes = nodes.stream().map(n -> NodeStyleSpan.fromNode(n, this)).collect(Collectors.toList());

        UniformStyleCollection collection = new UniformStyleCollection(Collections.singleton(layerId.getStyleClass()), wrappedNodes);

        return () -> layersById.get(layerId).styleNodes(resetLayer, collection);
    }


    /**
     * Applies the given update and applies the styling to the code area.
     * We use a closure parameter to encapsulate the application of the
     * update inside the restyling procedure, and mostly to make obvious
     * that each update needs restyling, and each restyling needs an update.
     *
     * @param update Update to carry out
     */
    public void updateStyling(Runnable update) {
        Platform.runLater(() -> {
            update.run();
            try {
                this.setStyleSpans(0, recomputePainting());
            } catch (Exception e) {
                // we ignore these particular exceptions because they're
                // commonly thrown when the text is being edited while
                // the layering algorithm runs, and it doesn't matter
                if ("StyleSpan's length cannot be negative".equals(e.getMessage())
                    || StringUtils.contains(e.getMessage(), "is not a valid range within")) {
                    return;
                }
                throw new RuntimeException("Unhandled error while recomputing the styling", e);
            }
        });
    }


    /**
     * Clears all style layers from their contents, including syntax highlighting.
     */
    public void clearStyleLayers() {
        updateStyling(() -> {
            layersById.values().forEach(StyleLayer::clearStyles);
            clearSyntaxHighlighting();
        });
    }


    /**
     * Clears a style layer.
     *
     * @param id layer id.
     */
    public void clearStyleLayer(K id) {
        updateStyling(layersById.get(id)::clearStyles);
    }


    /**
     * Recomputes a single style spans from the syntax highlighting layer and nodes to highlight.
     */
    private StyleSpans<Collection<String>> recomputePainting() {

        List<StyleSpans<Collection<String>>> allSpans = layersById.values().stream()
                                                                  .flatMap(layer -> layer.getCollections().stream())
                                                                  .filter(c -> !c.isEmpty())
                                                                  .map(UniformStyleCollection::toSpans)
                                                                  .collect(Collectors.toList());

        ensureSyntaxUpToDate();

        if (allSpans.isEmpty()) {
            return syntaxHighlight.getOrElse(emptySpan());
        }

        if (syntaxHighlight.getOpt().map(StyleSpans::length).filter(l -> l != getLength()).isPresent()) {
            // This is only executed if the text has changed (we use the length as an approximation)
            // This makes the highlighting much more resilient to staccato code changes,
            // which previously would have overlaid an outdated syntax highlighting layer on the
            // up-to-date node highlights, making the highlighting twitch briefly before the
            // asynchronous syntax highlight catches up
            updateSyntaxHighlightingSynchronously();
        }

        syntaxHighlight.ifPresent(allSpans::add);

        final StyleSpans<Collection<String>> base = allSpans.get(0);

        return allSpans.stream()
                       .filter(spans -> spans != base)
                       .filter(spans -> spans.length() <= getLength())
                       .reduce(base, (accumulator, elt) -> accumulator.overlay(elt, SyntaxHighlightingCodeArea::additiveOverlay));


    }


    /**
     * Attempts to preserve the other layers when syntax highlighting changes. The result
     * misplaces some style classes, which is undesirable, but covered up by the subsequent
     * parsing update.
     */
    @Override
    @NonNull
    protected final StyleSpans<Collection<String>> styleSyntaxHighlightChange(final @Nullable StyleSpans<Collection<String>> oldSyntax,
                                                                              final @Nullable StyleSpans<Collection<String>> newSyntax) {

        StyleSpans<Collection<String>> currentSpans = getStyleSpans(new IndexRange(0, getLength()));
        StyleSpans<Collection<String>> base = Optional.ofNullable(oldSyntax).map(s -> subtract(currentSpans, s)).orElse(currentSpans);

        return Optional.ofNullable(newSyntax)
                       .map(s -> base.overlay(s, SyntaxHighlightingCodeArea::additiveOverlay))
                       .orElse(base)
                       .subView(0, getLength());
    }


    /** Subtracts the second argument from the first. */
    private static StyleSpans<Collection<String>> subtract(StyleSpans<Collection<String>> base, StyleSpans<Collection<String>> diff) {
        return base.overlay(diff, (style1, style2) -> {
            if (style2.isEmpty()) {
                return style1;
            }
            Set<String> styles = new HashSet<>(style1);
            styles.removeAll(style2);
            return styles;
        });
    }

    /** Identifier for a highlighting layer. */
    public interface LayerId {
        /**
         * Returns the style class associated with that layer.
         * Nodes styled in that layer will have this style class.
         *
         * @return The style class
         */
        String getStyleClass();
    }
}
