/* Generic definitions */
/* Assertions (useful to generate conditional code) */
/* Current type and class (and size, if applicable) */
/* Value methods */
/* Interfaces (keys) */
/* Interfaces (values) */
/* Abstract implementations (keys) */
/* Abstract implementations (values) */
/* Static containers (keys) */
/* Static containers (values) */
/* Implementations */
/* Synchronized wrappers */
/* Unmodifiable wrappers */
/* Other wrappers */
/* Methods (keys) */
/* Methods (values) */
/* Methods (keys/values) */
/* Methods that have special names depending on keys (but the special names depend on values) */
/* Equality */
/* Object/Reference-only definitions (keys) */
/* Primitive-type-only definitions (keys) */
/* Object/Reference-only definitions (values) */
/* Primitive-type-only definitions (values) */
/*		 
 * Copyright (C) 2002-2014 Sebastiano Vigna 
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License. 
 */
package it.unimi.dsi.fastutil.longs;
import it.unimi.dsi.fastutil.Hash;
import it.unimi.dsi.fastutil.HashCommon;
import it.unimi.dsi.fastutil.booleans.BooleanArrays;
import static it.unimi.dsi.fastutil.HashCommon.arraySize;
import static it.unimi.dsi.fastutil.HashCommon.maxFill;
import java.util.Map;
import java.util.NoSuchElementException;
import it.unimi.dsi.fastutil.doubles.DoubleCollection;
import it.unimi.dsi.fastutil.doubles.AbstractDoubleCollection;
import it.unimi.dsi.fastutil.doubles.DoubleIterator;
import java.util.Comparator;
import it.unimi.dsi.fastutil.doubles.DoubleListIterator;
import it.unimi.dsi.fastutil.objects.AbstractObjectSortedSet;
import it.unimi.dsi.fastutil.objects.ObjectListIterator;
import it.unimi.dsi.fastutil.objects.ObjectBidirectionalIterator;
import it.unimi.dsi.fastutil.objects.ObjectSortedSet;
/**  A type-specific linked hash map with with a fast, small-footprint implementation.
 *
 * <P>Instances of this class use a hash table to represent a map. The table is
 * enlarged as needed by doubling its size when new entries are created, but it is <em>never</em> made
 * smaller (even on a {@link #clear()}). A family of {@linkplain #trim() trimming
 * methods} lets you control the size of the table; this is particularly useful
 * if you reuse instances of this class.
 *
 * <P>Iterators generated by this map will enumerate pairs in the same order in which they
 * have been added to the map (addition of pairs whose key is already present 
 * in the set does not change the iteration order). Note that this order has nothing in common with the natural
 * order of the keys. The order is kept by means of a doubly linked list, represented
 * <i>via</i> an array of longs parallel to the table.
 *
 * <P>This class implements the interface of a sorted map, so to allow easy
 * access of the iteration order: for instance, you can get the first key
 * in iteration order with {@link #firstKey()} without having to create an
 * iterator; however, this class partially violates the {@link java.util.SortedMap}
 * contract because all submap methods throw an exception and {@link
 * #comparator()} returns always <code>null</code>.
 *
 * <p>Additional methods, such as <code>getAndMoveToFirst()</code>, make it easy
 * to use instances of this class as a cache (e.g., with LRU policy).
 *
 * <P>The iterators provided by the views of this class using are type-specific
 * {@linkplain java.util.ListIterator list iterators}, and can be started at any
 * element <em>which is a key of the map</em>, or
 * a {@link NoSuchElementException} exception will be thrown.
 * If, however, the provided element is not the first or last key in the
 * set, the first access to the list index will require linear time, as in the worst case
 * the entire key set must be scanned in iteration order to retrieve the positional
 * index of the starting key. If you use just the methods of a type-specific {@link it.unimi.dsi.fastutil.BidirectionalIterator},
 * however, all operations will be performed in constant time.
 *
 * @see Hash
 * @see HashCommon
 */
public class Long2DoubleLinkedOpenHashMap extends AbstractLong2DoubleSortedMap implements java.io.Serializable, Cloneable, Hash {
    private static final long serialVersionUID = 0L;
 private static final boolean ASSERTS = false;
 /** The array of keys. */
 protected transient long key[];
 /** The array of values. */
 protected transient double value[];
 /** The array telling whether a position is used. */
 protected transient boolean used[];
 /** The acceptable load factor. */
 protected final float f;
 /** The current table size. */
 protected transient int n;
 /** Threshold after which we rehash. It must be the table size times {@link #f}. */
 protected transient int maxFill;
 /** The mask for wrapping a position counter. */
 protected transient int mask;
 /** Number of entries in the set. */
 protected int size;
 /** Cached set of entries. */
 protected transient volatile FastSortedEntrySet entries;
 /** Cached set of keys. */
 protected transient volatile LongSortedSet keys;
 /** Cached collection of values. */
 protected transient volatile DoubleCollection values;
 /** The index of the first entry in iteration order. It is valid iff {@link #size} is nonzero; otherwise, it contains -1. */
 protected transient int first = -1;
 /** The index of the last entry in iteration order. It is valid iff {@link #size} is nonzero; otherwise, it contains -1. */
 protected transient int last = -1;
 /** For each entry, the next and the previous entry in iteration order,
     * stored as <code>((prev & 0xFFFFFFFFL) << 32) | (next & 0xFFFFFFFFL)</code>.
     * The first entry contains predecessor -1, and the last entry 
     * contains successor -1. */
     protected transient long link[];
     /* Macros for transforming the bi-directional long link. Return values are 32-bit int indexes.
      * SET_UPPER and SET_LOWER do a masked assignment as described at
      * http://www-graphics.stanford.edu/~seander/bithacks.html#MaskedMerge
      */
 /** Creates a new hash map.
	 *
	 * <p>The actual table size will be the least power of two greater than <code>expected</code>/<code>f</code>.
	 *
	 * @param expected the expected number of elements in the hash set. 
	 * @param f the load factor.
	 */
 @SuppressWarnings("unchecked")
 public Long2DoubleLinkedOpenHashMap( final int expected, final float f ) {
  if ( f <= 0 || f > 1 ) throw new IllegalArgumentException( "Load factor must be greater than 0 and smaller than or equal to 1" );
  if ( expected < 0 ) throw new IllegalArgumentException( "The expected number of elements must be nonnegative" );
  this.f = f;
  n = arraySize( expected, f );
  mask = n - 1;
  maxFill = maxFill( n, f );
  key = new long[ n ];
  value = new double[ n ];
  used = new boolean[ n ];
  link = new long[ n ];
 }
 /** Creates a new hash map with {@link Hash#DEFAULT_LOAD_FACTOR} as load factor.
	 *
	 * @param expected the expected number of elements in the hash map.
	 */
 public Long2DoubleLinkedOpenHashMap( final int expected ) {
  this( expected, DEFAULT_LOAD_FACTOR );
 }
 /** Creates a new hash map with initial expected {@link Hash#DEFAULT_INITIAL_SIZE} entries
	 * and {@link Hash#DEFAULT_LOAD_FACTOR} as load factor.
	 */
 public Long2DoubleLinkedOpenHashMap() {
  this( DEFAULT_INITIAL_SIZE, DEFAULT_LOAD_FACTOR );
 }
 /** Creates a new hash map copying a given one.
	 *
	 * @param m a {@link Map} to be copied into the new hash map. 
	 * @param f the load factor.
	 */
 public Long2DoubleLinkedOpenHashMap( final Map<? extends Long, ? extends Double> m, final float f ) {
  this( m.size(), f );
  putAll( m );
 }
 /** Creates a new hash map with {@link Hash#DEFAULT_LOAD_FACTOR} as load factor copying a given one.
	 *
	 * @param m a {@link Map} to be copied into the new hash map. 
	 */
 public Long2DoubleLinkedOpenHashMap( final Map<? extends Long, ? extends Double> m ) {
  this( m, DEFAULT_LOAD_FACTOR );
 }
 /** Creates a new hash map copying a given type-specific one.
	 *
	 * @param m a type-specific map to be copied into the new hash map. 
	 * @param f the load factor.
	 */
 public Long2DoubleLinkedOpenHashMap( final Long2DoubleMap m, final float f ) {
  this( m.size(), f );
  putAll( m );
 }
 /** Creates a new hash map with {@link Hash#DEFAULT_LOAD_FACTOR} as load factor copying a given type-specific one.
	 *
	 * @param m a type-specific map to be copied into the new hash map. 
	 */
 public Long2DoubleLinkedOpenHashMap( final Long2DoubleMap m ) {
  this( m, DEFAULT_LOAD_FACTOR );
 }
 /** Creates a new hash map using the elements of two parallel arrays.
	 *
	 * @param k the array of keys of the new hash map.
	 * @param v the array of corresponding values in the new hash map.
	 * @param f the load factor.
	 * @throws IllegalArgumentException if <code>k</code> and <code>v</code> have different lengths.
	 */
 public Long2DoubleLinkedOpenHashMap( final long[] k, final double v[], final float f ) {
  this( k.length, f );
  if ( k.length != v.length ) throw new IllegalArgumentException( "The key array and the value array have different lengths (" + k.length + " and " + v.length + ")" );
  for( int i = 0; i < k.length; i++ ) this.put( k[ i ], v[ i ] );
 }
 /** Creates a new hash map with {@link Hash#DEFAULT_LOAD_FACTOR} as load factor using the elements of two parallel arrays.
	 *
	 * @param k the array of keys of the new hash map.
	 * @param v the array of corresponding values in the new hash map.
	 * @throws IllegalArgumentException if <code>k</code> and <code>v</code> have different lengths.
	 */
 public Long2DoubleLinkedOpenHashMap( final long[] k, final double v[] ) {
  this( k, v, DEFAULT_LOAD_FACTOR );
 }
 /*
	 * The following methods implements some basic building blocks used by
	 * all accessors. They are (and should be maintained) identical to those used in OpenHashSet.drv.
	 */
 public double put(final long k, final double v) {
  // The starting point.
  int pos = (int)it.unimi.dsi.fastutil.HashCommon.murmurHash3( (k) ^ mask ) & mask;
  // There's always an unused entry.
  while( used[ pos ] ) {
   if ( ( (key[ pos ]) == (k) ) ) {
    final double oldValue = value[ pos ];
    value[ pos ] = v;
    return oldValue;
   }
   pos = ( pos + 1 ) & mask;
  }
  used[ pos ] = true;
  key[ pos ] = k;
  value[ pos ] = v;
  if ( size == 0 ) {
   first = last = pos;
   // Special case of SET_UPPER_LOWER( link[ pos ], -1, -1 );
   link[ pos ] = -1L;
  }
  else {
   link[ last ] ^= ( ( link[ last ] ^ ( pos & 0xFFFFFFFFL ) ) & 0xFFFFFFFFL );
   link[ pos ] = ( ( last & 0xFFFFFFFFL ) << 32 ) | ( -1 & 0xFFFFFFFFL );
   last = pos;
  }
  if ( size++ >= maxFill ) rehash( arraySize( size + 1, f ) );
  if ( ASSERTS ) checkTable();
  return defRetValue;
 }
 public Double put( final Long ok, final Double ov ) {
  final double v = ((ov).doubleValue());
  final long k = ((ok).longValue());
  // The starting point.
  int pos = (int)it.unimi.dsi.fastutil.HashCommon.murmurHash3( (k) ^ mask ) & mask;
  // There's always an unused entry.
  while( used[ pos ] ) {
   if ( ( (key[ pos ]) == (k) ) ) {
    final Double oldValue = (Double.valueOf(value[ pos ]));
    value[ pos ] = v;
    return oldValue;
   }
   pos = ( pos + 1 ) & mask;
  }
  used[ pos ] = true;
  key[ pos ] = k;
  value[ pos ] = v;
  if ( size == 0 ) {
   first = last = pos;
   // Special case of SET_UPPER_LOWER( link[ pos ], -1, -1 );
   link[ pos ] = -1L;
  }
  else {
   link[ last ] ^= ( ( link[ last ] ^ ( pos & 0xFFFFFFFFL ) ) & 0xFFFFFFFFL );
   link[ pos ] = ( ( last & 0xFFFFFFFFL ) << 32 ) | ( -1 & 0xFFFFFFFFL );
   last = pos;
  }
  if ( size++ >= maxFill ) rehash( arraySize( size + 1, f ) );
  if ( ASSERTS ) checkTable();
  return (null);
 }
 /** Adds an increment to value currently associated with a key.
	 *
	 * @param k the key.
	 * @param incr the increment.
	 * @return the old value, or the {@linkplain #defaultReturnValue() default return value} if no value was present for the given key.
	 * @deprecated use <code>addTo()</code> instead; having the same name of a {@link java.util.Set} method turned out to be a recipe for disaster.
	 */
 @Deprecated
 public double add(final long k, final double incr) {
  return addTo( k, incr );
 }
 /** Adds an increment to value currently associated with a key.
	 *
	 * <P>Note that this method respects the {@linkplain #defaultReturnValue() default return value} semantics: when
	 * called with a key that does not currently appears in the map, the key
	 * will be associated with the default return value plus
	 * the given increment.
	 *
	 * @param k the key.
	 * @param incr the increment.
	 * @return the old value, or the {@linkplain #defaultReturnValue() default return value} if no value was present for the given key.
	 */
 public double addTo(final long k, final double incr) {
  // The starting point.
  int pos = (int)it.unimi.dsi.fastutil.HashCommon.murmurHash3( (k) ^ mask ) & mask;
  // There's always an unused entry.
  while( used[ pos ] ) {
   if ( ( (key[ pos ]) == (k) ) ) {
    final double oldValue = value[ pos ];
    value[ pos ] += incr;
    return oldValue;
   }
   pos = ( pos + 1 ) & mask;
  }
  used[ pos ] = true;
  key[ pos ] = k;
  value[ pos ] = defRetValue + incr;
  if ( size == 0 ) {
   first = last = pos;
   // Special case of SET_UPPER_LOWER( link[ pos ], -1, -1 );
   link[ pos ] = -1L;
  }
  else {
   link[ last ] ^= ( ( link[ last ] ^ ( pos & 0xFFFFFFFFL ) ) & 0xFFFFFFFFL );
   link[ pos ] = ( ( last & 0xFFFFFFFFL ) << 32 ) | ( -1 & 0xFFFFFFFFL );
   last = pos;
  }
  if ( size++ >= maxFill ) rehash( arraySize( size + 1, f ) );
  if ( ASSERTS ) checkTable();
  return defRetValue;
 }
 /** Shifts left entries with the specified hash code, starting at the specified position,
	 * and empties the resulting free entry.
	 *
	 * @param pos a starting position.
	 * @return the position cleared by the shifting process.
	 */
 protected final int shiftKeys( int pos ) {
  // Shift entries with the same hash.
  int last, slot;
  for(;;) {
   pos = ( ( last = pos ) + 1 ) & mask;
   while( used[ pos ] ) {
    slot = (int)it.unimi.dsi.fastutil.HashCommon.murmurHash3( (key[ pos ]) ^ mask ) & mask;
    if ( last <= pos ? last >= slot || slot > pos : last >= slot && slot > pos ) break;
    pos = ( pos + 1 ) & mask;
   }
   if ( ! used[ pos ] ) break;
   key[ last ] = key[ pos ];
   value[ last ] = value[ pos ];
   fixPointers( pos, last );
  }
  used[ last ] = false;
  return last;
 }
 @SuppressWarnings("unchecked")
 public double remove( final long k ) {
  // The starting point.
  int pos = (int)it.unimi.dsi.fastutil.HashCommon.murmurHash3( (k) ^ mask ) & mask;
  // There's always an unused entry.
  while( used[ pos ] ) {
   if ( ( (key[ pos ]) == (k) ) ) {
    size--;
    fixPointers( pos );
    final double v = value[ pos ];
    shiftKeys( pos );
    return v;
   }
   pos = ( pos + 1 ) & mask;
  }
  return defRetValue;
 }
 @SuppressWarnings("unchecked")
 public Double remove( final Object ok ) {
  final long k = ((((Long)(ok)).longValue()));
  // The starting point.
  int pos = (int)it.unimi.dsi.fastutil.HashCommon.murmurHash3( (k) ^ mask ) & mask;
  // There's always an unused entry.
  while( used[ pos ] ) {
   if ( ( (key[ pos ]) == (k) ) ) {
    size--;
    fixPointers( pos );
    final double v = value[ pos ];
    shiftKeys( pos );
    return (Double.valueOf(v));
   }
   pos = ( pos + 1 ) & mask;
  }
  return (null);
 }
 /** Removes the mapping associated with the first key in iteration order.
	 * @return the value previously associated with the first key in iteration order.
	 * @throws NoSuchElementException is this map is empty.
	 */
 public double removeFirstDouble() {
  if ( size == 0 ) throw new NoSuchElementException();
  --size;
  final int pos = first;
  // Abbreviated version of fixPointers(pos)
  first = (int) link[ pos ];
  if ( 0 <= first ) {
   // Special case of SET_PREV( link[ first ], -1 )
   link[ first ] |= (-1 & 0xFFFFFFFFL) << 32;
  }
  final double v = value[ pos ];
  shiftKeys( pos );
  return v;
 }
 /** Removes the mapping associated with the last key in iteration order.
	 * @return the value previously associated with the last key in iteration order.
	 * @throws NoSuchElementException is this map is empty.
	 */
 public double removeLastDouble() {
  if ( size == 0 ) throw new NoSuchElementException();
  --size;
  final int pos = last;
  // Abbreviated version of fixPointers(pos)
  last = (int) ( link[ pos ] >>> 32 );
  if ( 0 <= last ) {
   // Special case of SET_NEXT( link[ last ], -1 )
   link[ last ] |= -1 & 0xFFFFFFFFL;
  }
  final double v = value[ pos ];
  shiftKeys( pos );
  return v;
 }
 private void moveIndexToFirst( final int i ) {
  if ( size == 1 || first == i ) return;
  if ( last == i ) {
   last = (int) ( link[ i ] >>> 32 );
   // Special case of SET_NEXT( link[ last ], -1 );
   link[ last ] |= -1 & 0xFFFFFFFFL;
  }
  else {
   final long linki = link[ i ];
   final int prev = (int) ( linki >>> 32 );
   final int next = (int) linki;
   link[ prev ] ^= ( ( link[ prev ] ^ ( linki & 0xFFFFFFFFL ) ) & 0xFFFFFFFFL );
   link[ next ] ^= ( ( link[ next ] ^ ( linki & 0xFFFFFFFF00000000L ) ) & 0xFFFFFFFF00000000L );
  }
  link[ first ] ^= ( ( link[ first ] ^ ( ( i & 0xFFFFFFFFL ) << 32 ) ) & 0xFFFFFFFF00000000L );
  link[ i ] = ( ( -1 & 0xFFFFFFFFL ) << 32 ) | ( first & 0xFFFFFFFFL );
  first = i;
 }
 private void moveIndexToLast( final int i ) {
  if ( size == 1 || last == i ) return;
  if ( first == i ) {
   first = (int) link[ i ];
   // Special case of SET_PREV( link[ first ], -1 );
   link[ first ] |= (-1 & 0xFFFFFFFFL) << 32;
  }
  else {
   final long linki = link[ i ];
   final int prev = (int) ( linki >>> 32 );
   final int next = (int) linki;
   link[ prev ] ^= ( ( link[ prev ] ^ ( linki & 0xFFFFFFFFL ) ) & 0xFFFFFFFFL );
   link[ next ] ^= ( ( link[ next ] ^ ( linki & 0xFFFFFFFF00000000L ) ) & 0xFFFFFFFF00000000L );
  }
  link[ last ] ^= ( ( link[ last ] ^ ( i & 0xFFFFFFFFL ) ) & 0xFFFFFFFFL );
  link[ i ] = ( ( last & 0xFFFFFFFFL ) << 32 ) | ( -1 & 0xFFFFFFFFL );
  last = i;
 }
 /** Returns the value to which the given key is mapped; if the key is present, it is moved to the first position of the iteration order.
	 *
	 * @param k the key.
	 * @return the corresponding value, or the {@linkplain #defaultReturnValue() default return value} if no value was present for the given key.
	 */
 public double getAndMoveToFirst( final long k ) {
  final long key[] = this.key;
  final boolean used[] = this.used;
  final int mask = this.mask;
  // The starting point.
  int pos = (int)it.unimi.dsi.fastutil.HashCommon.murmurHash3( (k) ^ mask ) & mask;
  // There's always an unused entry.
  while( used[ pos ] ) {
   if( ( (k) == (key[ pos ]) ) ) {
    moveIndexToFirst( pos );
    return value[ pos ];
   }
   pos = ( pos + 1 ) & mask;
  }
  return defRetValue;
 }
 /** Returns the value to which the given key is mapped; if the key is present, it is moved to the last position of the iteration order.
	 *
	 * @param k the key.
	 * @return the corresponding value, or the {@linkplain #defaultReturnValue() default return value} if no value was present for the given key.
	 */
 public double getAndMoveToLast( final long k ) {
  final long key[] = this.key;
  final boolean used[] = this.used;
  final int mask = this.mask;
  // The starting point.
  int pos = (int)it.unimi.dsi.fastutil.HashCommon.murmurHash3( (k) ^ mask ) & mask;
  // There's always an unused entry.
  while( used[ pos ] ) {
   if( ( (k) == (key[ pos ]) ) ) {
    moveIndexToLast( pos );
    return value[ pos ];
   }
   pos = ( pos + 1 ) & mask;
  }
  return defRetValue;
 }
 /** Adds a pair to the map; if the key is already present, it is moved to the first position of the iteration order.
	 *
	 * @param k the key.
	 * @param v the value.
	 * @return the old value, or the {@linkplain #defaultReturnValue() default return value} if no value was present for the given key.
	 */
 public double putAndMoveToFirst( final long k, final double v ) {
  final long key[] = this.key;
  final boolean used[] = this.used;
  final int mask = this.mask;
  // The starting point.
  int pos = (int)it.unimi.dsi.fastutil.HashCommon.murmurHash3( (k) ^ mask ) & mask;
  // There's always an unused entry.
  while( used[ pos ] ) {
   if ( ( (k) == (key[ pos ]) ) ) {
    final double oldValue = value[ pos ];
    value[ pos ] = v;
    moveIndexToFirst( pos );
    return oldValue;
   }
   pos = ( pos + 1 ) & mask;
  }
  used[ pos ] = true;
  key[ pos ] = k;
  value[ pos ] = v;
  if ( size == 0 ) {
   first = last = pos;
   // Special case of SET_UPPER_LOWER( link[ pos ], -1, -1 );
   link[ pos ] = -1L;
  }
  else {
   link[ first ] ^= ( ( link[ first ] ^ ( ( pos & 0xFFFFFFFFL ) << 32 ) ) & 0xFFFFFFFF00000000L );
   link[ pos ] = ( ( -1 & 0xFFFFFFFFL ) << 32 ) | ( first & 0xFFFFFFFFL );
   first = pos;
  }
  if ( size++ >= maxFill ) rehash( arraySize( size, f ) );
  if ( ASSERTS ) checkTable();
  return defRetValue;
 }
 /** Adds a pair to the map; if the key is already present, it is moved to the last position of the iteration order.
	 *
	 * @param k the key.
	 * @param v the value.
	 * @return the old value, or the {@linkplain #defaultReturnValue() default return value} if no value was present for the given key.
	 */
 public double putAndMoveToLast( final long k, final double v ) {
  final long key[] = this.key;
  final boolean used[] = this.used;
  final int mask = this.mask;
  // The starting point.
  int pos = (int)it.unimi.dsi.fastutil.HashCommon.murmurHash3( (k) ^ mask ) & mask;
  // There's always an unused entry.
  while( used[ pos ] ) {
   if ( ( (k) == (key[ pos ]) ) ) {
    final double oldValue = value[ pos ];
    value[ pos ] = v;
    moveIndexToLast( pos );
    return oldValue;
   }
   pos = ( pos + 1 ) & mask;
  }
  used[ pos ] = true;
  key[ pos ] = k;
  value[ pos ] = v;
  if ( size == 0 ) {
   first = last = pos;
   // Special case of SET_UPPER_LOWER( link[ pos ], -1, -1 );
   link[ pos ] = -1L;
  }
  else {
   link[ last ] ^= ( ( link[ last ] ^ ( pos & 0xFFFFFFFFL ) ) & 0xFFFFFFFFL );
   link[ pos ] = ( ( last & 0xFFFFFFFFL ) << 32 ) | ( -1 & 0xFFFFFFFFL );
   last = pos;
  }
  if ( size++ >= maxFill ) rehash( arraySize( size, f ) );
  if ( ASSERTS ) checkTable();
  return defRetValue;
 }
 public Double get( final Long ok ) {
  final long k = ((ok).longValue());
  // The starting point.
  int pos = (int)it.unimi.dsi.fastutil.HashCommon.murmurHash3( ( k) ^ mask ) & mask;
  // There's always an unused entry.
  while( used[ pos ] ) {
   if ( ( (key[ pos ]) == ( k) ) ) return (Double.valueOf(value[ pos ]));
   pos = ( pos + 1 ) & mask;
  }
  return (null);
 }
 @SuppressWarnings("unchecked")
 public double get( final long k ) {
  // The starting point.
  int pos = (int)it.unimi.dsi.fastutil.HashCommon.murmurHash3( (k) ^ mask ) & mask;
  // There's always an unused entry.
  while( used[ pos ] ) {
   if ( ( (key[ pos ]) == (k) ) ) return value[ pos ];
   pos = ( pos + 1 ) & mask;
  }
  return defRetValue;
 }
 @SuppressWarnings("unchecked")
 public boolean containsKey( final long k ) {
  // The starting point.
  int pos = (int)it.unimi.dsi.fastutil.HashCommon.murmurHash3( (k) ^ mask ) & mask;
  // There's always an unused entry.
  while( used[ pos ] ) {
   if ( ( (key[ pos ]) == (k) ) ) return true;
   pos = ( pos + 1 ) & mask;
  }
  return false;
 }
 public boolean containsValue( final double v ) {
  final double value[] = this.value;
  final boolean used[] = this.used;
  for( int i = n; i-- != 0; ) if ( used[ i ] && ( (value[ i ]) == (v) ) ) return true;
  return false;
 }
 /* Removes all elements from this map.
	 *
	 * <P>To increase object reuse, this method does not change the table size.
	 * If you want to reduce the table size, you must use {@link #trim()}.
	 *
	 */
 public void clear() {
  if ( size == 0 ) return;
  size = 0;
  BooleanArrays.fill( used, false );
  // We null all object entries so that the garbage collector can do its work.
  first = last = -1;
 }
 public int size() {
  return size;
 }
 public boolean isEmpty() {
  return size == 0;
 }
 /** A no-op for backward compatibility.
	 * 
	 * @param growthFactor unused.
	 * @deprecated Since <code>fastutil</code> 6.1.0, hash tables are doubled when they are too full.
	 */
 @Deprecated
 public void growthFactor( int growthFactor ) {}
 /** Gets the growth factor (2).
	 *
	 * @return the growth factor of this set, which is fixed (2).
	 * @see #growthFactor(int)
	 * @deprecated Since <code>fastutil</code> 6.1.0, hash tables are doubled when they are too full.
	 */
 @Deprecated
 public int growthFactor() {
  return 16;
 }
 /** The entry class for a hash map does not record key and value, but
	 * rather the position in the hash table of the corresponding entry. This
	 * is necessary so that calls to {@link java.util.Map.Entry#setValue(Object)} are reflected in
	 * the map */
 private final class MapEntry implements Long2DoubleMap.Entry , Map.Entry<Long, Double> {
  // The table index this entry refers to, or -1 if this entry has been deleted.
  private int index;
  MapEntry( final int index ) {
   this.index = index;
  }
  public Long getKey() {
   return (Long.valueOf(key[ index ]));
  }
  public long getLongKey() {
      return key[ index ];
  }
  public Double getValue() {
   return (Double.valueOf(value[ index ]));
  }
  public double getDoubleValue() {
   return value[ index ];
  }
  public double setValue( final double v ) {
   final double oldValue = value[ index ];
   value[ index ] = v;
   return oldValue;
  }
  public Double setValue( final Double v ) {
   return (Double.valueOf(setValue( ((v).doubleValue()) )));
  }
  @SuppressWarnings("unchecked")
  public boolean equals( final Object o ) {
   if (!(o instanceof Map.Entry)) return false;
   Map.Entry<Long, Double> e = (Map.Entry<Long, Double>)o;
   return ( (key[ index ]) == (((e.getKey()).longValue())) ) && ( (value[ index ]) == (((e.getValue()).doubleValue())) );
  }
  public int hashCode() {
   return it.unimi.dsi.fastutil.HashCommon.long2int(key[ index ]) ^ it.unimi.dsi.fastutil.HashCommon.double2int(value[ index ]);
  }
  public String toString() {
   return key[ index ] + "=>" + value[ index ];
  }
 }
 /** Modifies the {@link #link} vector so that the given entry is removed.
	 * This method will complete in constant time.
	 *
	 * @param i the index of an entry. 
	 */
 protected void fixPointers( final int i ) {
  if ( size == 0 ) {
   first = last = -1;
   return;
  }
  if ( first == i ) {
   first = (int) link[ i ];
   if (0 <= first) {
    // Special case of SET_PREV( link[ first ], -1 )
    link[ first ] |= (-1 & 0xFFFFFFFFL) << 32;
   }
   return;
  }
  if ( last == i ) {
   last = (int) ( link[ i ] >>> 32 );
   if (0 <= last) {
    // Special case of SET_NEXT( link[ last ], -1 )
    link[ last ] |= -1 & 0xFFFFFFFFL;
   }
   return;
  }
  final long linki = link[ i ];
  final int prev = (int) ( linki >>> 32 );
  final int next = (int) linki;
  link[ prev ] ^= ( ( link[ prev ] ^ ( linki & 0xFFFFFFFFL ) ) & 0xFFFFFFFFL );
  link[ next ] ^= ( ( link[ next ] ^ ( linki & 0xFFFFFFFF00000000L ) ) & 0xFFFFFFFF00000000L );
 }
 /** Modifies the {@link #link} vector for a shift from s to d.
	 * <P>This method will complete in constant time.
	 *
	 * @param s the source position.
	 * @param d the destination position.
	 */
 protected void fixPointers( int s, int d ) {
  if ( size == 1 ) {
   first = last = d;
   // Special case of SET_UPPER_LOWER( link[ d ], -1, -1 )
   link[ d ] = -1L;
   return;
  }
  if ( first == s ) {
   first = d;
   link[ (int) link[ s ] ] ^= ( ( link[ (int) link[ s ] ] ^ ( ( d & 0xFFFFFFFFL ) << 32 ) ) & 0xFFFFFFFF00000000L );
   link[ d ] = link[ s ];
   return;
  }
  if ( last == s ) {
   last = d;
   link[ (int) ( link[ s ] >>> 32 )] ^= ( ( link[ (int) ( link[ s ] >>> 32 )] ^ ( d & 0xFFFFFFFFL ) ) & 0xFFFFFFFFL );
   link[ d ] = link[ s ];
   return;
  }
  final long links = link[ s ];
  final int prev = (int) ( links >>> 32 );
  final int next = (int) links;
  link[ prev ] ^= ( ( link[ prev ] ^ ( d & 0xFFFFFFFFL ) ) & 0xFFFFFFFFL );
  link[ next ] ^= ( ( link[ next ] ^ ( ( d & 0xFFFFFFFFL ) << 32 ) ) & 0xFFFFFFFF00000000L );
  link[ d ] = links;
 }
 /** Returns the first key of this map in iteration order.
	 *
	 * @return the first key in iteration order.
	 */
 public long firstLongKey() {
  if ( size == 0 ) throw new NoSuchElementException();
  return key[ first ];
 }
 /** Returns the last key of this map in iteration order.
	 *
	 * @return the last key in iteration order.
	 */
 public long lastLongKey() {
  if ( size == 0 ) throw new NoSuchElementException();
  return key[ last ];
 }
 public LongComparator comparator() { return null; }
 public Long2DoubleSortedMap tailMap( long from ) { throw new UnsupportedOperationException(); }
 public Long2DoubleSortedMap headMap( long to ) { throw new UnsupportedOperationException(); }
 public Long2DoubleSortedMap subMap( long from, long to ) { throw new UnsupportedOperationException(); }
 /** A list iterator over a linked map.
	 *
	 * <P>This class provides a list iterator over a linked hash map. The constructor runs in constant time.
	 */
 private class MapIterator {
  /** The entry that will be returned by the next call to {@link java.util.ListIterator#previous()} (or <code>null</code> if no previous entry exists). */
  int prev = -1;
  /** The entry that will be returned by the next call to {@link java.util.ListIterator#next()} (or <code>null</code> if no next entry exists). */
  int next = -1;
  /** The last entry that was returned (or -1 if we did not iterate or used {@link java.util.Iterator#remove()}). */
  int curr = -1;
  /** The current index (in the sense of a {@link java.util.ListIterator}). Note that this value is not meaningful when this iterator has been created using the nonempty constructor.*/
  int index = -1;
  private MapIterator() {
   next = first;
   index = 0;
  }
  private MapIterator( final long from ) {
   if ( ( (key[ last ]) == (from) ) ) {
    prev = last;
    index = size;
   }
   else {
    // The starting point.
    int pos = (int)it.unimi.dsi.fastutil.HashCommon.murmurHash3( (from) ^ mask ) & mask;
    // There's always an unused entry.
    while( used[ pos ] ) {
     if ( ( (key[ pos ]) == (from) ) ) {
      // Note: no valid index known.
      next = (int) link[ pos ];
      prev = pos;
      return;
     }
     pos = ( pos + 1 ) & mask;
    }
    throw new NoSuchElementException( "The key " + from + " does not belong to this map." );
   }
  }
  public boolean hasNext() { return next != -1; }
  public boolean hasPrevious() { return prev != -1; }
  private final void ensureIndexKnown() {
   if ( index >= 0 ) return;
   if ( prev == -1 ) {
    index = 0;
    return;
   }
   if ( next == -1 ) {
    index = size;
    return;
   }
   int pos = first;
   index = 1;
   while( pos != prev ) {
    pos = (int) link[ pos ];
    index++;
   }
  }
  public int nextIndex() {
   ensureIndexKnown();
   return index;
  }
  public int previousIndex() {
   ensureIndexKnown();
   return index - 1;
  }
  public int nextEntry() {
   if ( ! hasNext() ) return size();
   curr = next;
   next = (int) link[ curr ];
   prev = curr;
   if ( index >= 0 ) index++;
   return curr;
  }
  public int previousEntry() {
   if ( ! hasPrevious() ) return -1;
   curr = prev;
   prev = (int) ( link[ curr ] >>> 32 );
   next = curr;
   if ( index >= 0 ) index--;
   return curr;
  }
  @SuppressWarnings("unchecked")
  public void remove() {
   ensureIndexKnown();
   if ( curr == -1 ) throw new IllegalStateException();
   if ( curr == prev ) {
    /* If the last operation was a next(), we are removing an entry that preceeds
				   the current index, and thus we must decrement it. */
    index--;
    prev = (int) ( link[ curr ] >>> 32 );
   }
   else
    next = (int) link[ curr ];
   size--;
   /* Now we manually fix the pointers. Because of our knowledge of next
			   and prev, this is going to be faster than calling fixPointers(). */
   if ( prev == -1 ) first = next;
   else
    link[ prev ] ^= ( ( link[ prev ] ^ ( next & 0xFFFFFFFFL ) ) & 0xFFFFFFFFL );
   if ( next == -1 ) last = prev;
   else
    link[ next ] ^= ( ( link[ next ] ^ ( ( prev & 0xFFFFFFFFL ) << 32 ) ) & 0xFFFFFFFF00000000L );
   int last, slot, pos = curr;
   // We have to horribly duplicate the shiftKeys() code because we need to update next/prev.			
   for(;;) {
    pos = ( ( last = pos ) + 1 ) & mask;
    while( used[ pos ] ) {
     slot = (int)it.unimi.dsi.fastutil.HashCommon.murmurHash3( (key[ pos ]) ^ mask ) & mask;
     if ( last <= pos ? last >= slot || slot > pos : last >= slot && slot > pos ) break;
     pos = ( pos + 1 ) & mask;
    }
    if ( ! used[ pos ] ) break;
    key[ last ] = key[ pos ];
    value[ last ] = value[ pos ];
    if ( next == pos ) next = last;
    if ( prev == pos ) prev = last;
    fixPointers( pos, last );
   }
   used[ last ] = false;
   curr = -1;
  }
  public int skip( final int n ) {
   int i = n;
   while( i-- != 0 && hasNext() ) nextEntry();
   return n - i - 1;
  }
  public int back( final int n ) {
   int i = n;
   while( i-- != 0 && hasPrevious() ) previousEntry();
   return n - i - 1;
  }
 }
 private class EntryIterator extends MapIterator implements ObjectListIterator<Long2DoubleMap.Entry > {
  private MapEntry entry;
  public EntryIterator() {}
  public EntryIterator( long from ) {
   super( from );
  }
  public MapEntry next() {
   return entry = new MapEntry( nextEntry() );
  }
  public MapEntry previous() {
   return entry = new MapEntry( previousEntry() );
  }
  @Override
  public void remove() {
   super.remove();
   entry.index = -1; // You cannot use a deleted entry.
  }
  public void set( Long2DoubleMap.Entry ok ) { throw new UnsupportedOperationException(); }
  public void add( Long2DoubleMap.Entry ok ) { throw new UnsupportedOperationException(); }
 }
 private class FastEntryIterator extends MapIterator implements ObjectListIterator<Long2DoubleMap.Entry > {
  final BasicEntry entry = new BasicEntry ( ((long)0), (0) );
  public FastEntryIterator() {}
  public FastEntryIterator( long from ) {
   super( from );
  }
  public BasicEntry next() {
   final int e = nextEntry();
   entry.key = key[ e ];
   entry.value = value[ e ];
   return entry;
  }
  public BasicEntry previous() {
   final int e = previousEntry();
   entry.key = key[ e ];
   entry.value = value[ e ];
   return entry;
  }
  public void set( Long2DoubleMap.Entry ok ) { throw new UnsupportedOperationException(); }
  public void add( Long2DoubleMap.Entry ok ) { throw new UnsupportedOperationException(); }
 }
 private final class MapEntrySet extends AbstractObjectSortedSet<Long2DoubleMap.Entry > implements FastSortedEntrySet {
  public ObjectBidirectionalIterator<Long2DoubleMap.Entry > iterator() {
   return new EntryIterator();
  }
  public Comparator<? super Long2DoubleMap.Entry > comparator() { return null; }
  public ObjectSortedSet<Long2DoubleMap.Entry > subSet( Long2DoubleMap.Entry fromElement, Long2DoubleMap.Entry toElement) { throw new UnsupportedOperationException(); }
  public ObjectSortedSet<Long2DoubleMap.Entry > headSet( Long2DoubleMap.Entry toElement ) { throw new UnsupportedOperationException(); }
  public ObjectSortedSet<Long2DoubleMap.Entry > tailSet( Long2DoubleMap.Entry fromElement ) { throw new UnsupportedOperationException(); }
  public Long2DoubleMap.Entry first() {
   if ( size == 0 ) throw new NoSuchElementException();
   return new MapEntry( Long2DoubleLinkedOpenHashMap.this.first );
  }
  public Long2DoubleMap.Entry last() {
   if ( size == 0 ) throw new NoSuchElementException();
   return new MapEntry( Long2DoubleLinkedOpenHashMap.this.last );
  }
  @SuppressWarnings("unchecked")
  public boolean contains( final Object o ) {
   if ( !( o instanceof Map.Entry ) ) return false;
   final Map.Entry<Long, Double> e = (Map.Entry<Long, Double>)o;
   final long k = ((e.getKey()).longValue());
   // The starting point.
   int pos = (int)it.unimi.dsi.fastutil.HashCommon.murmurHash3( (k) ^ mask ) & mask;
   // There's always an unused entry.
   while( used[ pos ] ) {
    if ( ( (key[ pos ]) == (k) ) ) return ( (value[ pos ]) == (((e.getValue()).doubleValue())) );
    pos = ( pos + 1 ) & mask;
   }
   return false;
  }
  @SuppressWarnings("unchecked")
  public boolean remove( final Object o ) {
   if ( !( o instanceof Map.Entry ) ) return false;
   final Map.Entry<Long, Double> e = (Map.Entry<Long, Double>)o;
   final long k = ((e.getKey()).longValue());
   // The starting point.
   int pos = (int)it.unimi.dsi.fastutil.HashCommon.murmurHash3( (k) ^ mask ) & mask;
   // There's always an unused entry.
   while( used[ pos ] ) {
    if ( ( (key[ pos ]) == (k) ) ) {
     Long2DoubleLinkedOpenHashMap.this.remove( e.getKey() );
     return true;
    }
    pos = ( pos + 1 ) & mask;
   }
   return false;
  }
  public int size() {
   return size;
  }
  public void clear() {
   Long2DoubleLinkedOpenHashMap.this.clear();
  }
  public ObjectBidirectionalIterator<Long2DoubleMap.Entry > iterator( final Long2DoubleMap.Entry from ) {
   return new EntryIterator( ((from.getKey()).longValue()) );
  }
  public ObjectBidirectionalIterator<Long2DoubleMap.Entry > fastIterator() {
   return new FastEntryIterator();
  }
  public ObjectBidirectionalIterator<Long2DoubleMap.Entry > fastIterator( final Long2DoubleMap.Entry from ) {
   return new FastEntryIterator( ((from.getKey()).longValue()) );
  }
 }
 public FastSortedEntrySet long2DoubleEntrySet() {
  if ( entries == null ) entries = new MapEntrySet();
  return entries;
 }
 /** An iterator on keys.
	 *
	 * <P>We simply override the {@link java.util.ListIterator#next()}/{@link java.util.ListIterator#previous()} methods
	 * (and possibly their type-specific counterparts) so that they return keys
	 * instead of entries.
	 */
 private final class KeyIterator extends MapIterator implements LongListIterator {
  public KeyIterator( final long k ) { super( k ); }
  public long previousLong() { return key[ previousEntry() ]; }
  public void set( long k ) { throw new UnsupportedOperationException(); }
  public void add( long k ) { throw new UnsupportedOperationException(); }
  public Long previous() { return (Long.valueOf(key[ previousEntry() ])); }
  public void set( Long ok ) { throw new UnsupportedOperationException(); }
  public void add( Long ok ) { throw new UnsupportedOperationException(); }
  public KeyIterator() { super(); }
  public long nextLong() { return key[ nextEntry() ]; }
  public Long next() { return (Long.valueOf(key[ nextEntry() ])); }
 }
 private final class KeySet extends AbstractLongSortedSet {
  public LongListIterator iterator( final long from ) {
   return new KeyIterator( from );
  }
  public LongListIterator iterator() {
   return new KeyIterator();
  }
  public int size() {
   return size;
  }
  public boolean contains( long k ) {
   return containsKey( k );
  }
  public boolean remove( long k ) {
   final int oldSize = size;
   Long2DoubleLinkedOpenHashMap.this.remove( k );
   return size != oldSize;
  }
  public void clear() {
   Long2DoubleLinkedOpenHashMap.this.clear();
  }
  public long firstLong() {
   if ( size == 0 ) throw new NoSuchElementException();
   return key[ first ];
  }
  public long lastLong() {
   if ( size == 0 ) throw new NoSuchElementException();
   return key[ last ];
  }
  public LongComparator comparator() { return null; }
  final public LongSortedSet tailSet( long from ) { throw new UnsupportedOperationException(); }
  final public LongSortedSet headSet( long to ) { throw new UnsupportedOperationException(); }
  final public LongSortedSet subSet( long from, long to ) { throw new UnsupportedOperationException(); }
 }
 public LongSortedSet keySet() {
  if ( keys == null ) keys = new KeySet();
  return keys;
 }
 /** An iterator on values.
	 *
	 * <P>We simply override the {@link java.util.ListIterator#next()}/{@link java.util.ListIterator#previous()} methods
	 * (and possibly their type-specific counterparts) so that they return values
	 * instead of entries.
	 */
 private final class ValueIterator extends MapIterator implements DoubleListIterator {
  public double previousDouble() { return value[ previousEntry() ]; }
  public Double previous() { return (Double.valueOf(value[ previousEntry() ])); }
  public void set( Double ok ) { throw new UnsupportedOperationException(); }
  public void add( Double ok ) { throw new UnsupportedOperationException(); }
  public void set( double v ) { throw new UnsupportedOperationException(); }
  public void add( double v ) { throw new UnsupportedOperationException(); }
  public ValueIterator() { super(); }
  public double nextDouble() { return value[ nextEntry() ]; }
  public Double next() { return (Double.valueOf(value[ nextEntry() ])); }
 }
 public DoubleCollection values() {
  if ( values == null ) values = new AbstractDoubleCollection () {
    public DoubleIterator iterator() {
     return new ValueIterator();
    }
    public int size() {
     return size;
    }
    public boolean contains( double v ) {
     return containsValue( v );
    }
    public void clear() {
     Long2DoubleLinkedOpenHashMap.this.clear();
    }
   };
  return values;
 }
 /** A no-op for backward compatibility. The kind of tables implemented by
	 * this class never need rehashing.
	 *
	 * <P>If you need to reduce the table size to fit exactly
	 * this set, use {@link #trim()}.
	 *
	 * @return true.
	 * @see #trim()
	 * @deprecated A no-op.
	 */
 @Deprecated
 public boolean rehash() {
  return true;
 }
 /** Rehashes the map, making the table as small as possible.
	 * 
	 * <P>This method rehashes the table to the smallest size satisfying the
	 * load factor. It can be used when the set will not be changed anymore, so
	 * to optimize access speed and size.
	 *
	 * <P>If the table size is already the minimum possible, this method
	 * does nothing. 
	 *
	 * @return true if there was enough memory to trim the map.
	 * @see #trim(int)
	 */
 public boolean trim() {
  final int l = arraySize( size, f );
  if ( l >= n ) return true;
  try {
   rehash( l );
  }
  catch(OutOfMemoryError cantDoIt) { return false; }
  return true;
 }
 /** Rehashes this map if the table is too large.
	 * 
	 * <P>Let <var>N</var> be the smallest table size that can hold
	 * <code>max(n,{@link #size()})</code> entries, still satisfying the load factor. If the current
	 * table size is smaller than or equal to <var>N</var>, this method does
	 * nothing. Otherwise, it rehashes this map in a table of size
	 * <var>N</var>.
	 *
	 * <P>This method is useful when reusing maps.  {@linkplain #clear() Clearing a
	 * map} leaves the table size untouched. If you are reusing a map
	 * many times, you can call this method with a typical
	 * size to avoid keeping around a very large table just
	 * because of a few large transient maps.
	 *
	 * @param n the threshold for the trimming.
	 * @return true if there was enough memory to trim the map.
	 * @see #trim()
	 */
 public boolean trim( final int n ) {
  final int l = HashCommon.nextPowerOfTwo( (int)Math.ceil( n / f ) );
  if ( this.n <= l ) return true;
  try {
   rehash( l );
  }
  catch( OutOfMemoryError cantDoIt ) { return false; }
  return true;
 }
 /** Rehashes the map.
	 *
	 * <P>This method implements the basic rehashing strategy, and may be
	 * overriden by subclasses implementing different rehashing strategies (e.g.,
	 * disk-based rehashing). However, you should not override this method
	 * unless you understand the internal workings of this class.
	 *
	 * @param newN the new size
	 */
 @SuppressWarnings("unchecked")
 protected void rehash( final int newN ) {
  int i = first, prev = -1, newPrev = -1, t, pos;
  long k;
  final long key[] = this.key;
  final double value[] = this.value;
  final int mask = newN - 1; // Note that this is used by the hashing macro
  final long newKey[] = new long[ newN ];
  final double newValue[] = new double[newN];
  final boolean newUsed[] = new boolean[ newN ];
  final long link[] = this.link;
  final long newLink[] = new long[ newN ];
  first = -1;
  for( int j = size; j-- != 0; ) {
   k = key[ i ];
   pos = (int)it.unimi.dsi.fastutil.HashCommon.murmurHash3( (k) ^ mask ) & mask;
   while ( newUsed[ pos ] ) pos = ( pos + 1 ) & mask;
   newUsed[ pos ] = true;
   newKey[ pos ] = k;
   newValue[ pos ] = value[ i ];
   if ( prev != -1 ) {
    newLink[ newPrev ] ^= ( ( newLink[ newPrev ] ^ ( pos & 0xFFFFFFFFL ) ) & 0xFFFFFFFFL );
    newLink[ pos ] ^= ( ( newLink[ pos ] ^ ( ( newPrev & 0xFFFFFFFFL ) << 32 ) ) & 0xFFFFFFFF00000000L );
    newPrev = pos;
   }
   else {
    newPrev = first = pos;
    // Special case of SET(newLink[ pos ], -1, -1);
    newLink[ pos ] = -1L;
   }
   t = i;
   i = (int) link[ i ];
   prev = t;
  }
  n = newN;
  this.mask = mask;
  maxFill = maxFill( n, f );
  this.key = newKey;
  this.value = newValue;
  this.used = newUsed;
  this.link = newLink;
  this.last = newPrev;
  if ( newPrev != -1 )
   // Special case of SET_NEXT( newLink[ newPrev ], -1 );
   newLink[ newPrev ] |= -1 & 0xFFFFFFFFL;
 }
 /** Returns a deep copy of this map. 
	 *
	 * <P>This method performs a deep copy of this hash map; the data stored in the
	 * map, however, is not cloned. Note that this makes a difference only for object keys.
	 *
	 *  @return a deep copy of this map.
	 */
 @SuppressWarnings("unchecked")
 public Long2DoubleLinkedOpenHashMap clone() {
  Long2DoubleLinkedOpenHashMap c;
  try {
   c = (Long2DoubleLinkedOpenHashMap )super.clone();
  }
  catch(CloneNotSupportedException cantHappen) {
   throw new InternalError();
  }
  c.keys = null;
  c.values = null;
  c.entries = null;
  c.key = key.clone();
  c.value = value.clone();
  c.used = used.clone();
  c.link = link.clone();
  return c;
 }
 /** Returns a hash code for this map.
	 *
	 * This method overrides the generic method provided by the superclass. 
	 * Since <code>equals()</code> is not overriden, it is important
	 * that the value returned by this method is the same value as
	 * the one returned by the overriden method.
	 *
	 * @return a hash code for this map.
	 */
 public int hashCode() {
  int h = 0;
  for( int j = size, i = 0, t = 0; j-- != 0; ) {
   while( ! used[ i ] ) i++;
    t = it.unimi.dsi.fastutil.HashCommon.long2int(key[ i ]);
    t ^= it.unimi.dsi.fastutil.HashCommon.double2int(value[ i ]);
   h += t;
   i++;
  }
  return h;
 }
 private void writeObject(java.io.ObjectOutputStream s) throws java.io.IOException {
  final long key[] = this.key;
  final double value[] = this.value;
  final MapIterator i = new MapIterator();
  s.defaultWriteObject();
  for( int j = size, e; j-- != 0; ) {
   e = i.nextEntry();
   s.writeLong( key[ e ] );
   s.writeDouble( value[ e ] );
  }
 }
 @SuppressWarnings("unchecked")
 private void readObject(java.io.ObjectInputStream s) throws java.io.IOException, ClassNotFoundException {
  s.defaultReadObject();
  n = arraySize( size, f );
  maxFill = maxFill( n, f );
  mask = n - 1;
  final long key[] = this.key = new long[ n ];
  final double value[] = this.value = new double[ n ];
  final boolean used[] = this.used = new boolean[ n ];
  final long link[] = this.link = new long[ n ];
  int prev = -1;
  first = last = -1;
  long k;
  double v;
  for( int i = size, pos = 0; i-- != 0; ) {
   k = s.readLong();
   v = s.readDouble();
   pos = (int)it.unimi.dsi.fastutil.HashCommon.murmurHash3( (k) ^ mask ) & mask;
   while ( used[ pos ] ) pos = ( pos + 1 ) & mask;
   used[ pos ] = true;
   key[ pos ] = k;
   value[ pos ] = v;
   if ( first != -1 ) {
    link[ prev ] ^= ( ( link[ prev ] ^ ( pos & 0xFFFFFFFFL ) ) & 0xFFFFFFFFL );
    link[ pos ] ^= ( ( link[ pos ] ^ ( ( prev & 0xFFFFFFFFL ) << 32 ) ) & 0xFFFFFFFF00000000L );
    prev = pos;
   }
   else {
    prev = first = pos;
    // Special case of SET_PREV( newLink[ pos ], -1 );
    link[ pos ] |= (-1L & 0xFFFFFFFFL) << 32;
   }
  }
  last = prev;
  if ( prev != -1 )
   // Special case of SET_NEXT( link[ prev ], -1 );
   link[ prev ] |= -1 & 0xFFFFFFFFL;
  if ( ASSERTS ) checkTable();
 }
 private void checkTable() {}
}
