/*
 * Decompiled with CFR 0.152.
 */
package org.apache.pinot.core.transport;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import com.google.common.util.concurrent.FutureCallback;
import com.google.common.util.concurrent.Futures;
import com.google.common.util.concurrent.ListenableFuture;
import com.google.common.util.concurrent.MoreExecutors;
import io.netty.buffer.ByteBuf;
import io.netty.buffer.Unpooled;
import io.netty.channel.ChannelHandler;
import io.netty.channel.ChannelHandlerContext;
import io.netty.channel.SimpleChannelInboundHandler;
import io.netty.handler.ssl.SslHandshakeCompletionEvent;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.CancellationException;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.Executor;
import java.util.concurrent.Future;
import java.util.concurrent.TimeUnit;
import javax.annotation.Nullable;
import org.apache.pinot.common.datatable.DataTable;
import org.apache.pinot.common.exception.QueryException;
import org.apache.pinot.common.metrics.AbstractMetrics;
import org.apache.pinot.common.metrics.ServerMeter;
import org.apache.pinot.common.metrics.ServerMetrics;
import org.apache.pinot.common.metrics.ServerQueryPhase;
import org.apache.pinot.common.metrics.ServerTimer;
import org.apache.pinot.common.request.InstanceRequest;
import org.apache.pinot.common.response.ProcessingException;
import org.apache.pinot.core.common.datatable.DataTableBuilderFactory;
import org.apache.pinot.core.query.request.ServerQueryRequest;
import org.apache.pinot.core.query.scheduler.QueryScheduler;
import org.apache.pinot.server.access.AccessControl;
import org.apache.pinot.spi.env.PinotConfiguration;
import org.apache.pinot.spi.utils.BytesUtils;
import org.apache.thrift.TBase;
import org.apache.thrift.TDeserializer;
import org.apache.thrift.TException;
import org.apache.thrift.protocol.TCompactProtocol;
import org.apache.thrift.protocol.TProtocolFactory;
import org.apache.thrift.transport.TTransportException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@ChannelHandler.Sharable
public class InstanceRequestHandler
extends SimpleChannelInboundHandler<ByteBuf> {
    private static final Logger LOGGER = LoggerFactory.getLogger(InstanceRequestHandler.class);
    private static final int SLOW_QUERY_LATENCY_THRESHOLD_MS = 100;
    private final String _instanceName;
    private final ThreadLocal<TDeserializer> _deserializer;
    private final QueryScheduler _queryScheduler;
    private final ServerMetrics _serverMetrics;
    private final AccessControl _accessControl;
    private final Map<String, Future<byte[]>> _queryFuturesById;

    public InstanceRequestHandler(String instanceName, PinotConfiguration config, QueryScheduler queryScheduler, ServerMetrics serverMetrics, AccessControl accessControl) {
        this._instanceName = instanceName;
        this._queryScheduler = queryScheduler;
        this._serverMetrics = serverMetrics;
        this._accessControl = accessControl;
        this._deserializer = ThreadLocal.withInitial(() -> {
            try {
                return new TDeserializer((TProtocolFactory)new TCompactProtocol.Factory());
            }
            catch (TTransportException e) {
                throw new RuntimeException("Failed to initialize Thrift Deserializer", e);
            }
        });
        if (Boolean.parseBoolean(config.getProperty("pinot.server.enable.query.cancellation"))) {
            this._queryFuturesById = new ConcurrentHashMap<String, Future<byte[]>>();
            LOGGER.info("Enable query cancellation");
        } else {
            this._queryFuturesById = null;
        }
    }

    public void userEventTriggered(ChannelHandlerContext ctx, Object evt) throws Exception {
        super.userEventTriggered(ctx, evt);
        if (evt instanceof SslHandshakeCompletionEvent && !this._accessControl.isAuthorizedChannel(ctx)) {
            ctx.disconnect();
            LOGGER.error("Exception while processing instance request: Unauthorized access to pinot-server");
        }
    }

    protected void channelRead0(ChannelHandlerContext ctx, ByteBuf msg) {
        long queryArrivalTimeMs = 0L;
        InstanceRequest instanceRequest = null;
        byte[] requestBytes = null;
        String tableNameWithType = null;
        try {
            int requestSize = msg.readableBytes();
            instanceRequest = new InstanceRequest();
            requestBytes = new byte[requestSize];
            queryArrivalTimeMs = System.currentTimeMillis();
            this._serverMetrics.addMeteredGlobalValue((AbstractMetrics.Meter)ServerMeter.QUERIES, 1L);
            this._serverMetrics.addMeteredGlobalValue((AbstractMetrics.Meter)ServerMeter.NETTY_CONNECTION_BYTES_RECEIVED, (long)requestSize);
            msg.readBytes(requestBytes);
            this._deserializer.get().deserialize((TBase)instanceRequest, requestBytes);
            ServerQueryRequest queryRequest = new ServerQueryRequest(instanceRequest, this._serverMetrics, queryArrivalTimeMs);
            queryRequest.getTimerContext().startNewPhaseTimer(ServerQueryPhase.REQUEST_DESERIALIZATION, queryArrivalTimeMs).stopAndRecord();
            tableNameWithType = queryRequest.getTableNameWithType();
            this.submitQuery(queryRequest, ctx, tableNameWithType, queryArrivalTimeMs, instanceRequest);
        }
        catch (Exception e) {
            if (e instanceof TException) {
                this._serverMetrics.addMeteredGlobalValue((AbstractMetrics.Meter)ServerMeter.REQUEST_DESERIALIZATION_EXCEPTIONS, 1L);
            }
            String hexString = requestBytes != null ? BytesUtils.toHexString(requestBytes) : "";
            long requestId = instanceRequest != null ? instanceRequest.getRequestId() : 0L;
            LOGGER.error("Exception while processing instance request: {}", (Object)hexString, (Object)e);
            this.sendErrorResponse(ctx, requestId, tableNameWithType, queryArrivalTimeMs, DataTableBuilderFactory.getEmptyDataTable(), e);
        }
    }

    @VisibleForTesting
    void submitQuery(ServerQueryRequest queryRequest, ChannelHandlerContext ctx, String tableNameWithType, long queryArrivalTimeMs, InstanceRequest instanceRequest) {
        ListenableFuture<byte[]> future = this._queryScheduler.submit(queryRequest);
        if (this._queryFuturesById != null) {
            String queryId = queryRequest.getQueryId();
            if (LOGGER.isDebugEnabled()) {
                LOGGER.debug("Keep track of running query: {}", (Object)queryId);
            }
            this._queryFuturesById.put(queryId, (Future<byte[]>)future);
        }
        Futures.addCallback(future, this.createCallback(ctx, tableNameWithType, queryArrivalTimeMs, instanceRequest, queryRequest), (Executor)MoreExecutors.directExecutor());
    }

    private FutureCallback<byte[]> createCallback(final ChannelHandlerContext ctx, final String tableNameWithType, final long queryArrivalTimeMs, final InstanceRequest instanceRequest, final ServerQueryRequest queryRequest) {
        return new FutureCallback<byte[]>(){

            public void onSuccess(@Nullable byte[] responseBytes) {
                if (InstanceRequestHandler.this._queryFuturesById != null) {
                    String queryId = queryRequest.getQueryId();
                    if (LOGGER.isDebugEnabled()) {
                        LOGGER.debug("Remove track of running query: {} on success", (Object)queryId);
                    }
                    InstanceRequestHandler.this._queryFuturesById.remove(queryId);
                }
                if (responseBytes != null) {
                    InstanceRequestHandler.this.sendResponse(ctx, queryRequest.getTableNameWithType(), queryArrivalTimeMs, responseBytes);
                } else {
                    InstanceRequestHandler.this.sendErrorResponse(ctx, queryRequest.getRequestId(), tableNameWithType, queryArrivalTimeMs, DataTableBuilderFactory.getEmptyDataTable(), new Exception("Null query response."));
                }
            }

            public void onFailure(Throwable t) {
                Exception e;
                if (InstanceRequestHandler.this._queryFuturesById != null) {
                    String queryId = queryRequest.getQueryId();
                    if (LOGGER.isDebugEnabled()) {
                        LOGGER.debug("Remove track of running query: {} on failure", (Object)queryId);
                    }
                    InstanceRequestHandler.this._queryFuturesById.remove(queryId);
                }
                if (t instanceof Exception) {
                    e = (Exception)t;
                    if (e instanceof CancellationException) {
                        LOGGER.info("Query: {} got cancelled", (Object)queryRequest.getQueryId());
                    } else {
                        LOGGER.error("Exception while processing instance request", (Throwable)e);
                    }
                } else {
                    LOGGER.error("Error while processing instance request", t);
                    e = new Exception(t);
                }
                InstanceRequestHandler.this.sendErrorResponse(ctx, instanceRequest.getRequestId(), tableNameWithType, queryArrivalTimeMs, DataTableBuilderFactory.getEmptyDataTable(), e);
            }
        };
    }

    public void exceptionCaught(ChannelHandlerContext ctx, Throwable cause) {
        String message = "Unhandled Exception in " + ((Object)((Object)this)).getClass().getCanonicalName();
        LOGGER.error(message, cause);
        this.sendErrorResponse(ctx, 0L, null, System.currentTimeMillis(), DataTableBuilderFactory.getEmptyDataTable(), new Exception(message, cause));
    }

    public boolean cancelQuery(String queryId) {
        Preconditions.checkState((this._queryFuturesById != null ? 1 : 0) != 0, (Object)"Query cancellation is not enabled on server");
        Future<byte[]> future = this._queryFuturesById.get(queryId);
        if (future == null) {
            return false;
        }
        boolean done = future.isDone();
        if (!done) {
            future.cancel(true);
        }
        if (LOGGER.isDebugEnabled()) {
            LOGGER.debug("Cancelled query: {} that's done: {}", (Object)queryId, (Object)done);
        }
        return true;
    }

    public Set<String> getRunningQueryIds() {
        Preconditions.checkState((this._queryFuturesById != null ? 1 : 0) != 0, (Object)"Query cancellation is not enabled on server");
        return new HashSet<String>(this._queryFuturesById.keySet());
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void sendErrorResponse(ChannelHandlerContext ctx, long requestId, String tableNameWithType, long queryArrivalTimeMs, DataTable dataTable, Exception e) {
        boolean cancelled = e instanceof CancellationException;
        try {
            Map dataTableMetadata = dataTable.getMetadata();
            dataTableMetadata.put(DataTable.MetadataKey.REQUEST_ID.getName(), Long.toString(requestId));
            if (cancelled) {
                dataTable.addException(QueryException.getException((ProcessingException)QueryException.QUERY_CANCELLATION_ERROR, (String)("Query cancelled on: " + this._instanceName + e)));
            } else {
                dataTable.addException(QueryException.getException((ProcessingException)QueryException.QUERY_EXECUTION_ERROR, (Throwable)e));
            }
            byte[] serializedDataTable = dataTable.toBytes();
            this.sendResponse(ctx, tableNameWithType, queryArrivalTimeMs, serializedDataTable);
        }
        catch (Exception exception) {
            LOGGER.error("Exception while sending query processing error to Broker.", (Throwable)exception);
        }
        finally {
            if (!cancelled) {
                LOGGER.error("Query processing error: ", (Throwable)e);
            }
            this._serverMetrics.addMeteredGlobalValue((AbstractMetrics.Meter)ServerMeter.QUERY_EXECUTION_EXCEPTIONS, 1L);
        }
    }

    private void sendResponse(ChannelHandlerContext ctx, String tableNameWithType, long queryArrivalTimeMs, byte[] serializedDataTable) {
        long sendResponseStartTimeMs = System.currentTimeMillis();
        int queryProcessingTimeMs = (int)(sendResponseStartTimeMs - queryArrivalTimeMs);
        ctx.writeAndFlush((Object)Unpooled.wrappedBuffer((byte[])serializedDataTable)).addListener(f -> {
            long sendResponseEndTimeMs = System.currentTimeMillis();
            int sendResponseLatencyMs = (int)(sendResponseEndTimeMs - sendResponseStartTimeMs);
            this._serverMetrics.addMeteredGlobalValue((AbstractMetrics.Meter)ServerMeter.NETTY_CONNECTION_RESPONSES_SENT, 1L);
            this._serverMetrics.addMeteredGlobalValue((AbstractMetrics.Meter)ServerMeter.NETTY_CONNECTION_BYTES_SENT, (long)serializedDataTable.length);
            this._serverMetrics.addTimedTableValue(tableNameWithType, (AbstractMetrics.Timer)ServerTimer.NETTY_CONNECTION_SEND_RESPONSE_LATENCY, (long)sendResponseLatencyMs, TimeUnit.MILLISECONDS);
            int totalQueryTimeMs = (int)(sendResponseEndTimeMs - queryArrivalTimeMs);
            if (totalQueryTimeMs > 100) {
                LOGGER.info("Slow query: request handler processing time: {}, send response latency: {}, total time to handle request: {}", new Object[]{queryProcessingTimeMs, sendResponseLatencyMs, totalQueryTimeMs});
            }
        });
    }
}

