/*
 * Decompiled with CFR 0.152.
 */
package org.apache.pinot.segment.local.segment.index.converter;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import java.io.File;
import java.io.FilenameFilter;
import java.io.IOException;
import java.nio.file.CopyOption;
import java.nio.file.Files;
import java.nio.file.attribute.FileAttribute;
import java.nio.file.attribute.PosixFilePermission;
import java.util.EnumSet;
import java.util.HashMap;
import java.util.Set;
import org.apache.commons.configuration.ConfigurationException;
import org.apache.commons.configuration.PropertiesConfiguration;
import org.apache.commons.io.FileUtils;
import org.apache.pinot.segment.spi.converter.SegmentFormatConverter;
import org.apache.pinot.segment.spi.creator.SegmentVersion;
import org.apache.pinot.segment.spi.index.metadata.SegmentMetadataImpl;
import org.apache.pinot.segment.spi.loader.SegmentDirectoryLoaderContext;
import org.apache.pinot.segment.spi.loader.SegmentDirectoryLoaderRegistry;
import org.apache.pinot.segment.spi.memory.PinotDataBuffer;
import org.apache.pinot.segment.spi.store.ColumnIndexType;
import org.apache.pinot.segment.spi.store.SegmentDirectory;
import org.apache.pinot.segment.spi.store.SegmentDirectoryPaths;
import org.apache.pinot.spi.env.CommonsConfigurationUtils;
import org.apache.pinot.spi.env.PinotConfiguration;
import org.apache.pinot.spi.utils.ReadMode;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class SegmentV1V2ToV3FormatConverter
implements SegmentFormatConverter {
    private static final Logger LOGGER = LoggerFactory.getLogger(SegmentV1V2ToV3FormatConverter.class);
    private static final String V3_TEMP_DIR_SUFFIX = ".v3.tmp";

    public void convert(File v2SegmentDirectory) throws Exception {
        Preconditions.checkNotNull((Object)v2SegmentDirectory, (Object)"Segment directory should not be null");
        Preconditions.checkState((v2SegmentDirectory.exists() && v2SegmentDirectory.isDirectory() ? 1 : 0) != 0, (Object)("Segment directory: " + v2SegmentDirectory.toString() + " must exist and should be a directory"));
        LOGGER.info("Converting segment: {} to v3 format", (Object)v2SegmentDirectory);
        SegmentMetadataImpl v2Metadata = new SegmentMetadataImpl(v2SegmentDirectory);
        SegmentVersion oldVersion = v2Metadata.getVersion();
        Preconditions.checkState((oldVersion != SegmentVersion.v3 ? 1 : 0) != 0, (String)"Segment {} is already in v3 format but at wrong path", (Object)v2Metadata.getName());
        Preconditions.checkArgument((oldVersion == SegmentVersion.v1 || oldVersion == SegmentVersion.v2 ? 1 : 0) != 0, (String)"Can not convert segment version: {} at path: {} ", (Object)oldVersion, (Object)v2SegmentDirectory);
        this.deleteStaleConversionDirectories(v2SegmentDirectory);
        File v3TempDirectory = this.v3ConversionTempDirectory(v2SegmentDirectory);
        this.setDirectoryPermissions(v3TempDirectory);
        this.createMetadataFile(v2SegmentDirectory, v3TempDirectory);
        this.copyCreationMetadataIfExists(v2SegmentDirectory, v3TempDirectory);
        this.copyLuceneTextIndexIfExists(v2SegmentDirectory, v3TempDirectory);
        this.copyIndexData(v2SegmentDirectory, v2Metadata, v3TempDirectory);
        File newLocation = SegmentDirectoryPaths.segmentDirectoryFor((File)v2SegmentDirectory, (SegmentVersion)SegmentVersion.v3);
        LOGGER.info("v3 segment location for segment: {} is {}", (Object)v2Metadata.getName(), (Object)newLocation);
        v3TempDirectory.renameTo(newLocation);
        this.deleteV2Files(v2SegmentDirectory);
    }

    private void deleteV2Files(File v2SegmentDirectory) throws IOException {
        LOGGER.info("Deleting files in v1 segment directory: {}", (Object)v2SegmentDirectory);
        File[] files = v2SegmentDirectory.listFiles();
        if (files == null) {
            LOGGER.error("v1 segment directory: {}  returned null list of files", (Object)v2SegmentDirectory);
            return;
        }
        for (File file : files) {
            if (file.isFile() && file.exists()) {
                FileUtils.deleteQuietly((File)file);
            }
            if (!file.isDirectory() || !file.getName().endsWith(".lucene.index")) continue;
            FileUtils.deleteDirectory((File)file);
        }
    }

    @VisibleForTesting
    public File v3ConversionTempDirectory(File v2SegmentDirectory) throws IOException {
        File v3TempDirectory = Files.createTempDirectory(v2SegmentDirectory.toPath(), v2SegmentDirectory.getName() + V3_TEMP_DIR_SUFFIX, new FileAttribute[0]).toFile();
        return v3TempDirectory;
    }

    private void setDirectoryPermissions(File v3Directory) throws IOException {
        EnumSet<PosixFilePermission[]> permissions = EnumSet.of(PosixFilePermission.OWNER_READ, new PosixFilePermission[]{PosixFilePermission.OWNER_WRITE, PosixFilePermission.OWNER_EXECUTE, PosixFilePermission.GROUP_READ, PosixFilePermission.GROUP_WRITE, PosixFilePermission.GROUP_EXECUTE, PosixFilePermission.OTHERS_READ, PosixFilePermission.OTHERS_EXECUTE});
        try {
            Files.setPosixFilePermissions(v3Directory.toPath(), permissions);
        }
        catch (UnsupportedOperationException ex) {
            LOGGER.error("unsupported non-posix filesystem permissions setting");
        }
    }

    private void copyIndexData(File v2Directory, SegmentMetadataImpl v2Metadata, File v3Directory) throws Exception {
        HashMap<String, String> props = new HashMap<String, String>();
        props.put("readMode", ReadMode.mmap.toString());
        PinotConfiguration configuration = new PinotConfiguration(props);
        try (SegmentDirectory v2Segment = SegmentDirectoryLoaderRegistry.getDefaultSegmentDirectoryLoader().load(v2Directory.toURI(), new SegmentDirectoryLoaderContext.Builder().setSegmentName(v2Metadata.getName()).setSegmentDirectoryConfigs(configuration).build());
             SegmentDirectory v3Segment = SegmentDirectoryLoaderRegistry.getDefaultSegmentDirectoryLoader().load(v3Directory.toURI(), new SegmentDirectoryLoaderContext.Builder().setSegmentName(v2Metadata.getName()).setSegmentDirectoryConfigs(configuration).build());){
            Set allColumns = v2Metadata.getAllColumns();
            try (SegmentDirectory.Reader v2DataReader = v2Segment.createReader();
                 SegmentDirectory.Writer v3DataWriter = v3Segment.createWriter();){
                for (String column : allColumns) {
                    this.copyIndexIfExists(v2DataReader, v3DataWriter, column, ColumnIndexType.DICTIONARY);
                    this.copyIndexIfExists(v2DataReader, v3DataWriter, column, ColumnIndexType.FORWARD_INDEX);
                    this.copyIndexIfExists(v2DataReader, v3DataWriter, column, ColumnIndexType.NULLVALUE_VECTOR);
                }
                for (String column : allColumns) {
                    this.copyIndexIfExists(v2DataReader, v3DataWriter, column, ColumnIndexType.INVERTED_INDEX);
                    this.copyIndexIfExists(v2DataReader, v3DataWriter, column, ColumnIndexType.FST_INDEX);
                    this.copyIndexIfExists(v2DataReader, v3DataWriter, column, ColumnIndexType.JSON_INDEX);
                    this.copyIndexIfExists(v2DataReader, v3DataWriter, column, ColumnIndexType.H3_INDEX);
                    this.copyIndexIfExists(v2DataReader, v3DataWriter, column, ColumnIndexType.RANGE_INDEX);
                    this.copyIndexIfExists(v2DataReader, v3DataWriter, column, ColumnIndexType.BLOOM_FILTER);
                }
                v3DataWriter.save();
            }
        }
        this.copyStarTreeV2(v2Directory, v3Directory);
    }

    private void copyIndexIfExists(SegmentDirectory.Reader reader, SegmentDirectory.Writer writer, String column, ColumnIndexType indexType) throws IOException {
        if (reader.hasIndexFor(column, indexType)) {
            this.readCopyBuffers(reader, writer, column, indexType);
        }
    }

    private void copyStarTreeV2(File src, File dest) throws IOException {
        File indexFile = new File(src, "star_tree_index");
        if (indexFile.exists()) {
            FileUtils.copyFile((File)indexFile, (File)new File(dest, "star_tree_index"));
            FileUtils.copyFile((File)new File(src, "star_tree_index_map"), (File)new File(dest, "star_tree_index_map"));
        }
    }

    private void readCopyBuffers(SegmentDirectory.Reader reader, SegmentDirectory.Writer writer, String column, ColumnIndexType indexType) throws IOException {
        PinotDataBuffer oldBuffer = reader.getIndexFor(column, indexType);
        long oldBufferSize = oldBuffer.size();
        PinotDataBuffer newBuffer = writer.newIndexFor(column, indexType, oldBufferSize);
        oldBuffer.copyTo(0L, newBuffer, 0L, oldBufferSize);
    }

    private void createMetadataFile(File currentDir, File v3Dir) throws ConfigurationException {
        File v2MetadataFile = new File(currentDir, "metadata.properties");
        File v3MetadataFile = new File(v3Dir, "metadata.properties");
        PropertiesConfiguration properties = CommonsConfigurationUtils.fromFile((File)v2MetadataFile);
        properties.setProperty("segment.index.version", (Object)SegmentVersion.v3.toString());
        properties.save(v3MetadataFile);
    }

    private void copyCreationMetadataIfExists(File currentDir, File v3Dir) throws IOException {
        File v2CreationFile = new File(currentDir, "creation.meta");
        if (v2CreationFile.exists()) {
            File v3CreationFile = new File(v3Dir, "creation.meta");
            Files.copy(v2CreationFile.toPath(), v3CreationFile.toPath(), new CopyOption[0]);
        }
    }

    private void copyLuceneTextIndexIfExists(File segmentDirectory, File v3Dir) throws IOException {
        File[] textIndexDocIdMappingFiles;
        File[] textIndexFiles;
        final String suffix = ".lucene.index";
        for (File textIndexFile : textIndexFiles = segmentDirectory.listFiles(new FilenameFilter(){

            @Override
            public boolean accept(File dir, String name) {
                return name.endsWith(suffix);
            }
        })) {
            File[] indexFiles = textIndexFile.listFiles();
            File v3LuceneIndexDir = new File(v3Dir, textIndexFile.getName());
            v3LuceneIndexDir.mkdir();
            for (File indexFile : indexFiles) {
                File v3LuceneIndexFile = new File(v3LuceneIndexDir, indexFile.getName());
                Files.copy(indexFile.toPath(), v3LuceneIndexFile.toPath(), new CopyOption[0]);
            }
        }
        final String docIDFileSuffix = ".lucene.mapping";
        for (File docIdMappingFile : textIndexDocIdMappingFiles = segmentDirectory.listFiles(new FilenameFilter(){

            @Override
            public boolean accept(File dir, String name) {
                return name.endsWith(docIDFileSuffix);
            }
        })) {
            File v3DocIdMappingFile = new File(v3Dir, docIdMappingFile.getName());
            Files.copy(docIdMappingFile.toPath(), v3DocIdMappingFile.toPath(), new CopyOption[0]);
        }
    }

    private void deleteStaleConversionDirectories(File segmentDirectory) {
        File[] files;
        final String prefix = segmentDirectory.getName() + V3_TEMP_DIR_SUFFIX;
        for (File file : files = segmentDirectory.listFiles(new FilenameFilter(){

            @Override
            public boolean accept(File dir, String name) {
                return name.startsWith(prefix);
            }
        })) {
            LOGGER.info("Deleting stale v3 directory: {}", (Object)file);
            FileUtils.deleteQuietly((File)file);
        }
    }

    public static void main(String[] args) throws Exception {
        if (args.length < 1) {
            System.err.println("Usage: $0 <table directory with segments>");
            System.exit(1);
        }
        File tableDirectory = new File(args[0]);
        Preconditions.checkState((boolean)tableDirectory.exists(), (String)"Directory: {} does not exist", (Object)tableDirectory);
        Preconditions.checkState((boolean)tableDirectory.isDirectory(), (String)"Path: {} is not a directory", (Object)tableDirectory);
        File[] files = tableDirectory.listFiles();
        SegmentV1V2ToV3FormatConverter converter = new SegmentV1V2ToV3FormatConverter();
        for (File file : files) {
            if (!file.isDirectory()) {
                System.out.println("Path: " + file + " is not a directory. Skipping...");
                continue;
            }
            long startTimeNano = System.nanoTime();
            converter.convert(file);
            long endTimeNano = System.nanoTime();
            long latency = (endTimeNano - startTimeNano) / 1000000L;
            System.out.println("Converting segment: " + file + " took " + latency + " milliseconds");
        }
    }
}

