/*
 * Decompiled with CFR 0.152.
 */
package org.apache.pinot.core.transport.server.routing.stats;

import com.google.common.base.Preconditions;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ThreadPoolExecutor;
import org.apache.commons.lang3.tuple.ImmutablePair;
import org.apache.commons.lang3.tuple.Pair;
import org.apache.pinot.core.transport.server.routing.stats.ServerRoutingStatsEntry;
import org.apache.pinot.spi.env.PinotConfiguration;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class ServerRoutingStatsManager {
    private static final Logger LOGGER = LoggerFactory.getLogger(ServerRoutingStatsManager.class);
    private final PinotConfiguration _config;
    private volatile boolean _isEnabled;
    private ConcurrentHashMap<String, ServerRoutingStatsEntry> _serverQueryStatsMap;
    private ExecutorService _executorService;
    private ScheduledExecutorService _periodicTaskExecutor;
    private double _alpha;
    private long _autoDecayWindowMs;
    private long _warmupDurationMs;
    private double _avgInitializationVal;
    private int _hybridScoreExponent;

    public ServerRoutingStatsManager(PinotConfiguration pinotConfig) {
        this._config = pinotConfig;
    }

    public void init() {
        this._isEnabled = this._config.getProperty("pinot.broker.adaptive.server.selector.enable.stats.collection", false);
        if (!this._isEnabled) {
            LOGGER.info("Server stats collection for Adaptive Server Selection is not enabled.");
            return;
        }
        LOGGER.info("Initializing ServerRoutingStatsManager for Adaptive Server Selection.");
        this._alpha = this._config.getProperty("pinot.broker.adaptive.server.selector.ewma.alpha", 0.666);
        this._autoDecayWindowMs = this._config.getProperty("pinot.broker.adaptive.server.selector.autodecay.window.ms", 10000L);
        this._warmupDurationMs = this._config.getProperty("pinot.broker.adaptive.server.selector.warmup.duration", 0L);
        this._avgInitializationVal = this._config.getProperty("pinot.broker.adaptive.server.selector.avg.initialization.val", 1.0);
        this._hybridScoreExponent = this._config.getProperty("pinot.broker.adaptive.server.selector.hybrid.score.exponent", 3);
        int threadPoolSize = this._config.getProperty("pinot.broker.adaptive.server.selector.stats.manager.threadpool.size", 2);
        this._executorService = Executors.newFixedThreadPool(threadPoolSize);
        this._periodicTaskExecutor = Executors.newSingleThreadScheduledExecutor();
        this._serverQueryStatsMap = new ConcurrentHashMap();
    }

    public boolean isEnabled() {
        return this._isEnabled;
    }

    public void shutDown() {
        if (!this._isEnabled) {
            return;
        }
        LOGGER.info("Shutting down ServerRoutingStatsManager.");
        this._isEnabled = false;
        this._executorService.shutdownNow();
    }

    public int getQueueSize() {
        if (!this._isEnabled) {
            return 0;
        }
        ThreadPoolExecutor tpe = (ThreadPoolExecutor)this._executorService;
        return tpe.getQueue().size();
    }

    public long getCompletedTaskCount() {
        if (!this._isEnabled) {
            return 0L;
        }
        ThreadPoolExecutor tpe = (ThreadPoolExecutor)this._executorService;
        return tpe.getCompletedTaskCount();
    }

    public void recordStatsAfterQuerySubmission(long requestId, String serverInstanceId) {
        if (!this._isEnabled) {
            return;
        }
        this._executorService.execute(() -> {
            try {
                this.updateStatsAfterQuerySubmission(serverInstanceId);
            }
            catch (Exception e) {
                LOGGER.error("Exception caught while updating stats. requestId={}, exception={}", (Object)requestId, (Object)e);
            }
        });
    }

    private void updateStatsAfterQuerySubmission(String serverInstanceId) {
        ServerRoutingStatsEntry stats = this._serverQueryStatsMap.computeIfAbsent(serverInstanceId, k -> new ServerRoutingStatsEntry(serverInstanceId, this._alpha, this._autoDecayWindowMs, this._warmupDurationMs, this._avgInitializationVal, this._hybridScoreExponent, this._periodicTaskExecutor));
        try {
            stats.getServerWriteLock().lock();
            stats.updateNumInFlightRequestsForQuerySubmission();
        }
        finally {
            stats.getServerWriteLock().unlock();
        }
    }

    public void recordStatsUponResponseArrival(long requestId, String serverInstanceId, long latency) {
        if (!this._isEnabled) {
            return;
        }
        this._executorService.execute(() -> {
            try {
                this.updateStatsUponResponseArrival(serverInstanceId, latency);
            }
            catch (Exception e) {
                LOGGER.error("Exception caught while updating stats. requestId={}, exception={}", (Object)requestId, (Object)e);
            }
        });
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void updateStatsUponResponseArrival(String serverInstanceId, long latencyMs) {
        ServerRoutingStatsEntry stats = this._serverQueryStatsMap.computeIfAbsent(serverInstanceId, k -> new ServerRoutingStatsEntry(serverInstanceId, this._alpha, this._autoDecayWindowMs, this._warmupDurationMs, this._avgInitializationVal, this._hybridScoreExponent, this._periodicTaskExecutor));
        try {
            stats.getServerWriteLock().lock();
            stats.updateNumInFlightRequestsForResponseArrival();
            if ((double)latencyMs >= 0.0) {
                stats.updateLatency(latencyMs);
            }
        }
        finally {
            stats.getServerWriteLock().unlock();
        }
    }

    public String getServerRoutingStatsStr() {
        if (!this._isEnabled) {
            return "";
        }
        StringBuilder stringBuilder = new StringBuilder("(Server=NumInFlightRequests,NumInFlightRequestsEMA,LatencyEMA,Score)");
        for (Map.Entry<String, ServerRoutingStatsEntry> entry : this._serverQueryStatsMap.entrySet()) {
            String server = entry.getKey();
            Preconditions.checkState((entry.getValue() != null ? 1 : 0) != 0, (Object)"Server stats is null");
            ServerRoutingStatsEntry stats = entry.getValue();
            stats.getServerReadLock().lock();
            Integer numInFlightRequests = stats.getNumInFlightRequests();
            Double numInFlightRequestsEMA = stats.getInFlightRequestsEMA();
            Double latencyEMA = stats.getLatencyEMA();
            Double score = stats.computeHybridScore();
            stats.getServerReadLock().unlock();
            stringBuilder.append(";").append(server).append("=").append(numInFlightRequests.toString()).append(",").append(numInFlightRequestsEMA.toString()).append(",").append(latencyEMA.toString()).append(",").append(score.toString());
        }
        return stringBuilder.toString();
    }

    public List<Pair<String, Integer>> fetchNumInFlightRequestsForAllServers() {
        ArrayList<Pair<String, Integer>> response = new ArrayList<Pair<String, Integer>>();
        if (!this._isEnabled) {
            return response;
        }
        for (Map.Entry<String, ServerRoutingStatsEntry> entry : this._serverQueryStatsMap.entrySet()) {
            String server = entry.getKey();
            Preconditions.checkState((entry.getValue() != null ? 1 : 0) != 0, (Object)"Server stats is null");
            ServerRoutingStatsEntry stats = entry.getValue();
            stats.getServerReadLock().lock();
            int numInFlightRequests = stats.getNumInFlightRequests();
            stats.getServerReadLock().unlock();
            response.add((Pair<String, Integer>)new ImmutablePair((Object)server, (Object)numInFlightRequests));
        }
        return response;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public Integer fetchNumInFlightRequestsForServer(String server) {
        if (!this._isEnabled) {
            return null;
        }
        ServerRoutingStatsEntry stats = this._serverQueryStatsMap.get(server);
        if (stats == null) {
            return null;
        }
        try {
            stats.getServerReadLock().lock();
            Integer n = stats.getNumInFlightRequests();
            return n;
        }
        finally {
            stats.getServerReadLock().unlock();
        }
    }

    public List<Pair<String, Double>> fetchEMALatencyForAllServers() {
        ArrayList<Pair<String, Double>> response = new ArrayList<Pair<String, Double>>();
        if (!this._isEnabled) {
            return response;
        }
        for (Map.Entry<String, ServerRoutingStatsEntry> entry : this._serverQueryStatsMap.entrySet()) {
            String server = entry.getKey();
            Preconditions.checkState((entry.getValue() != null ? 1 : 0) != 0, (Object)"Server stats is null");
            ServerRoutingStatsEntry stats = entry.getValue();
            stats.getServerReadLock().lock();
            double latency = stats.getLatencyEMA();
            stats.getServerReadLock().unlock();
            response.add((Pair<String, Double>)new ImmutablePair((Object)server, (Object)latency));
        }
        return response;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public Double fetchEMALatencyForServer(String server) {
        if (!this._isEnabled) {
            return null;
        }
        ServerRoutingStatsEntry stats = this._serverQueryStatsMap.get(server);
        if (stats == null) {
            return null;
        }
        try {
            stats.getServerReadLock().lock();
            Double d = stats.getLatencyEMA();
            return d;
        }
        finally {
            stats.getServerReadLock().unlock();
        }
    }

    public List<Pair<String, Double>> fetchHybridScoreForAllServers() {
        ArrayList<Pair<String, Double>> response = new ArrayList<Pair<String, Double>>();
        if (!this._isEnabled) {
            return response;
        }
        for (Map.Entry<String, ServerRoutingStatsEntry> entry : this._serverQueryStatsMap.entrySet()) {
            String server = entry.getKey();
            Preconditions.checkState((entry.getValue() != null ? 1 : 0) != 0, (Object)"Server stats is null");
            ServerRoutingStatsEntry stats = entry.getValue();
            stats.getServerReadLock().lock();
            double score = stats.computeHybridScore();
            stats.getServerReadLock().unlock();
            response.add((Pair<String, Double>)new ImmutablePair((Object)server, (Object)score));
        }
        return response;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public Double fetchHybridScoreForServer(String server) {
        if (!this._isEnabled) {
            return null;
        }
        ServerRoutingStatsEntry stats = this._serverQueryStatsMap.get(server);
        if (stats == null) {
            return null;
        }
        try {
            stats.getServerReadLock().lock();
            Double d = stats.computeHybridScore();
            return d;
        }
        finally {
            stats.getServerReadLock().unlock();
        }
    }
}

