/*
 * Decompiled with CFR 0.152.
 */
package io.trino.plugin.pinot.query;

import com.google.common.base.Preconditions;
import com.google.common.collect.ImmutableMap;
import io.trino.plugin.pinot.PinotColumnHandle;
import io.trino.spi.connector.ColumnHandle;
import java.text.DecimalFormat;
import java.text.DecimalFormatSymbols;
import java.util.Locale;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.stream.Collectors;
import org.apache.commons.codec.binary.Hex;
import org.apache.pinot.common.request.Expression;
import org.apache.pinot.common.request.ExpressionType;
import org.apache.pinot.common.request.Function;
import org.apache.pinot.common.request.Identifier;
import org.apache.pinot.common.request.Literal;
import org.apache.pinot.common.request.PinotQuery;

public class FilterToPinotSqlConverter {
    private static final Map<String, String> BINARY_OPERATORS = ImmutableMap.builder().put((Object)"equals", (Object)"=").put((Object)"not_equals", (Object)"!=").put((Object)"greater_than", (Object)">").put((Object)"less_than", (Object)"<").put((Object)"greater_than_or_equal", (Object)">=").put((Object)"less_than_or_equal", (Object)"<=").put((Object)"plus", (Object)"+").put((Object)"minus", (Object)"-").put((Object)"times", (Object)"*").put((Object)"divide", (Object)"/").build();
    private static final ThreadLocal<DecimalFormat> doubleFormatter = ThreadLocal.withInitial(() -> {
        DecimalFormat decimalFormat = new DecimalFormat("0", new DecimalFormatSymbols(Locale.US));
        decimalFormat.setMaximumFractionDigits(340);
        return decimalFormat;
    });
    private final Map<String, ColumnHandle> columnHandles;

    private FilterToPinotSqlConverter(Map<String, ColumnHandle> columnHandles) {
        this.columnHandles = Objects.requireNonNull(columnHandles, "columnHandles is null");
    }

    public static String convertFilter(PinotQuery pinotQuery, Map<String, ColumnHandle> columnHandles) {
        return new FilterToPinotSqlConverter(columnHandles).formatExpression(pinotQuery.getFilterExpression());
    }

    private String formatExpression(Expression expression) {
        switch (expression.getType()) {
            case FUNCTION: {
                return this.formatFunction(expression.getFunctionCall());
            }
            case LITERAL: {
                return this.formatLiteral(expression.getLiteral());
            }
            case IDENTIFIER: {
                return this.formatIdentifier(expression.getIdentifier());
            }
        }
        throw new UnsupportedOperationException(String.format("Unknown type: '%s'", expression.getType()));
    }

    private String formatFunction(Function functionCall) {
        String binaryOperator = BINARY_OPERATORS.get(functionCall.getOperator().toLowerCase(Locale.ENGLISH));
        if (binaryOperator != null) {
            return this.formatEqualsMinusZero(functionCall).orElse(String.format("(%s) %s (%s)", this.formatExpression((Expression)functionCall.getOperands().get(0)), binaryOperator, this.formatExpression((Expression)functionCall.getOperands().get(1))));
        }
        if (functionCall.getOperator().equalsIgnoreCase("cast")) {
            Preconditions.checkState((functionCall.getOperands().size() == 2 ? 1 : 0) != 0, (Object)"Unexpected size for cast operator");
            return String.format("CAST(%s AS %s)", this.formatExpression((Expression)functionCall.getOperands().get(0)), ((Expression)functionCall.getOperands().get(1)).getLiteral().getStringValue());
        }
        if (functionCall.getOperator().equalsIgnoreCase("in") || functionCall.getOperator().equalsIgnoreCase("not_in")) {
            return this.formatInClause(functionCall);
        }
        if (functionCall.getOperator().equalsIgnoreCase("case")) {
            return this.formatCaseStatement(functionCall);
        }
        return functionCall.getOperator() + "(" + functionCall.getOperands().stream().map(this::formatExpression).collect(Collectors.joining(", ")) + ")";
    }

    private Optional<String> formatEqualsMinusZero(Function functionCall) {
        if (!functionCall.getOperator().equalsIgnoreCase("equals")) {
            return Optional.empty();
        }
        Expression left = (Expression)functionCall.getOperands().get(0);
        if (left.getType() != ExpressionType.FUNCTION || !left.getFunctionCall().getOperator().equalsIgnoreCase("minus")) {
            return Optional.empty();
        }
        Expression right = (Expression)functionCall.getOperands().get(1);
        if (right.getType() != ExpressionType.LITERAL || !this.formatLiteral(right.getLiteral()).equals("0")) {
            return Optional.empty();
        }
        Function minus = left.getFunctionCall();
        return Optional.of(String.format("(%s) = (%s)", this.formatExpression((Expression)minus.getOperands().get(0)), this.formatExpression((Expression)minus.getOperands().get(1))));
    }

    private String formatInClause(Function functionCall) {
        Preconditions.checkState((functionCall.getOperator().equalsIgnoreCase("in") || functionCall.getOperator().equalsIgnoreCase("not_in") ? 1 : 0) != 0, (String)"Unexpected operator '%s'", (Object)functionCall.getOperator());
        Preconditions.checkState((functionCall.getOperands().size() > 1 ? 1 : 0) != 0, (Object)"Unexpected expression");
        String operator = functionCall.getOperator().equalsIgnoreCase("in") ? "IN" : "NOT IN";
        return String.format("%s %s (%s)", this.formatExpression((Expression)functionCall.getOperands().get(0)), operator, functionCall.getOperands().subList(1, functionCall.getOperands().size()).stream().map(this::formatExpression).collect(Collectors.joining(", ")));
    }

    private String formatCaseStatement(Function functionCall) {
        Preconditions.checkState((boolean)functionCall.getOperator().equalsIgnoreCase("case"), (String)"Unexpected operator '%s'", (Object)functionCall.getOperator());
        Preconditions.checkState((functionCall.getOperands().size() >= 2 ? 1 : 0) != 0, (Object)"Unexpected expression");
        int whenStatements = functionCall.getOperands().size() / 2;
        StringBuilder builder = new StringBuilder("CASE ");
        builder.append("WHEN ").append(this.formatExpression((Expression)functionCall.getOperands().get(0))).append(" THEN ").append(this.formatExpression((Expression)functionCall.getOperands().get(whenStatements)));
        for (int index = 1; index < whenStatements; ++index) {
            builder.append(" ").append("WHEN ").append(this.formatExpression((Expression)functionCall.getOperands().get(index))).append(" THEN ").append(this.formatExpression((Expression)functionCall.getOperands().get(index + whenStatements)));
        }
        if (functionCall.getOperands().size() % 2 != 0) {
            builder.append(" ELSE " + this.formatExpression((Expression)functionCall.getOperands().get(functionCall.getOperands().size() - 1)));
        }
        return builder.append(" END").toString();
    }

    private String formatLiteral(Literal literal) {
        if (!literal.isSet()) {
            return "null";
        }
        switch ((Literal._Fields)literal.getSetField()) {
            case LONG_VALUE: {
                return String.valueOf(literal.getLongValue());
            }
            case INT_VALUE: {
                return String.valueOf(literal.getIntValue());
            }
            case BOOL_VALUE: {
                return String.valueOf(literal.getBoolValue());
            }
            case STRING_VALUE: {
                return String.format("'%s'", literal.getStringValue().replaceAll("'", "''"));
            }
            case BYTE_VALUE: {
                return String.valueOf(literal.getByteValue());
            }
            case BINARY_VALUE: {
                return Hex.encodeHexString((byte[])literal.getBinaryValue());
            }
            case DOUBLE_VALUE: {
                return doubleFormatter.get().format(literal.getDoubleValue());
            }
            case SHORT_VALUE: {
                return String.valueOf(literal.getShortValue());
            }
        }
        throw new UnsupportedOperationException(String.format("Unknown literal type: '%s'", literal.getSetField()));
    }

    private String formatIdentifier(Identifier identifier) {
        PinotColumnHandle pinotColumnHandle = (PinotColumnHandle)Objects.requireNonNull(this.columnHandles.get(identifier.getName()), "Column not found");
        Preconditions.checkArgument((!pinotColumnHandle.getColumnName().contains("\"") ? 1 : 0) != 0, (String)"Column name contains double quotes: '%s'", (Object)pinotColumnHandle.getColumnName());
        return String.format("\"%s\"", pinotColumnHandle.getColumnName());
    }
}

