/*
 * Decompiled with CFR 0.152.
 */
package org.apache.pinot.core.segment.store;

import com.google.common.base.Preconditions;
import java.io.BufferedWriter;
import java.io.File;
import java.io.FileWriter;
import java.io.FilenameFilter;
import java.io.IOException;
import java.io.PrintWriter;
import java.nio.ByteOrder;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.SortedMap;
import java.util.TreeMap;
import org.apache.commons.configuration.Configuration;
import org.apache.commons.configuration.ConfigurationException;
import org.apache.commons.configuration.PropertiesConfiguration;
import org.apache.pinot.common.segment.ReadMode;
import org.apache.pinot.core.segment.index.metadata.SegmentMetadataImpl;
import org.apache.pinot.core.segment.memory.PinotDataBuffer;
import org.apache.pinot.core.segment.store.ColumnIndexDirectory;
import org.apache.pinot.core.segment.store.ColumnIndexType;
import org.apache.pinot.core.segment.store.IndexEntry;
import org.apache.pinot.core.segment.store.IndexKey;
import org.apache.pinot.spi.env.CommonsConfigurationUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

class SingleFileIndexDirectory
extends ColumnIndexDirectory {
    private static Logger LOGGER = LoggerFactory.getLogger(SingleFileIndexDirectory.class);
    private static final String DEFAULT_INDEX_FILE_NAME = "columns.psf";
    private static final String INDEX_MAP_FILE = "index_map";
    private static final long MAGIC_MARKER = -2401053088876085587L;
    private static final int MAGIC_MARKER_SIZE_BYTES = 8;
    private static final String MAP_KEY_SEPARATOR = ".";
    private static final String MAP_KEY_NAME_START_OFFSET = "startOffset";
    private static final String MAP_KEY_NAME_SIZE = "size";
    private static final int MAX_ALLOCATION_SIZE = 0x7D000000;
    private File indexFile;
    private Map<IndexKey, IndexEntry> columnEntries;
    private List<PinotDataBuffer> allocBuffers;

    public SingleFileIndexDirectory(File segmentDirectory, SegmentMetadataImpl metadata, ReadMode readMode) throws IOException, ConfigurationException {
        super(segmentDirectory, metadata, readMode);
        this.indexFile = new File(segmentDirectory, DEFAULT_INDEX_FILE_NAME);
        if (!this.indexFile.exists()) {
            this.indexFile.createNewFile();
        }
        this.columnEntries = new HashMap<IndexKey, IndexEntry>(metadata.getAllColumns().size());
        this.allocBuffers = new ArrayList<PinotDataBuffer>();
        this.load();
    }

    @Override
    public PinotDataBuffer getBuffer(String column, ColumnIndexType type) throws IOException {
        return this.checkAndGetIndexBuffer(column, type);
    }

    @Override
    public PinotDataBuffer newBuffer(String column, ColumnIndexType type, long sizeBytes) throws IOException {
        return this.allocNewBufferInternal(column, type, sizeBytes, type.name().toLowerCase() + ".create");
    }

    @Override
    public boolean hasIndexFor(String column, ColumnIndexType type) {
        if (type == ColumnIndexType.TEXT_INDEX) {
            return this.hasTextIndex(column);
        }
        IndexKey key = new IndexKey(column, type);
        return this.columnEntries.containsKey(key);
    }

    private boolean hasTextIndex(final String column) {
        final String suffix = ".lucene.index";
        File[] textIndexFiles = this.segmentDirectory.listFiles(new FilenameFilter(){

            @Override
            public boolean accept(File dir, String name) {
                return name.equals(column + suffix);
            }
        });
        if (textIndexFiles.length > 0) {
            Preconditions.checkState((textIndexFiles.length == 1 ? 1 : 0) != 0, (Object)("Illegal number of text index directories for columns " + column + " segment directory " + this.segmentDirectory.getAbsolutePath()));
            return true;
        }
        return false;
    }

    private PinotDataBuffer checkAndGetIndexBuffer(String column, ColumnIndexType type) {
        IndexKey key = new IndexKey(column, type);
        IndexEntry entry = this.columnEntries.get(key);
        if (entry == null || entry.buffer == null) {
            throw new RuntimeException("Could not find index for column: " + column + ", type: " + (Object)((Object)type) + ", segment: " + this.segmentDirectory.toString());
        }
        return entry.buffer;
    }

    private PinotDataBuffer allocNewBufferInternal(String column, ColumnIndexType indexType, long size, String context) throws IOException {
        IndexKey key = new IndexKey(column, indexType);
        this.checkKeyNotPresent(key);
        String allocContext = this.allocationContext(key) + context;
        IndexEntry entry = new IndexEntry(key);
        entry.startOffset = this.indexFile.length();
        entry.size = size + 8L;
        PinotDataBuffer appendBuffer = PinotDataBuffer.mapFile(this.indexFile, false, entry.startOffset, entry.size, ByteOrder.BIG_ENDIAN, allocContext);
        LOGGER.debug("Allotted buffer for key: {}, startOffset: {}, size: {}", new Object[]{key, entry.startOffset, entry.size});
        appendBuffer.putLong(0, -2401053088876085587L);
        this.allocBuffers.add(appendBuffer);
        entry.buffer = appendBuffer.view(8L, entry.size);
        this.columnEntries.put(key, entry);
        this.persistIndexMap(entry);
        return entry.buffer;
    }

    private void checkKeyNotPresent(IndexKey key) {
        if (this.columnEntries.containsKey(key)) {
            throw new RuntimeException("Attempt to re-create an existing index for key: " + key.toString() + ", for segmentDirectory: " + this.segmentDirectory.getAbsolutePath());
        }
    }

    private void validateMagicMarker(PinotDataBuffer buffer, int startOffset) {
        long actualMarkerValue = buffer.getLong(startOffset);
        if (actualMarkerValue != -2401053088876085587L) {
            LOGGER.error("Missing magic marker in index file: {} at position: {}", (Object)this.indexFile, (Object)startOffset);
            throw new RuntimeException("Inconsistent data read. Index data file " + this.indexFile.toString() + " is possibly corrupted");
        }
    }

    private void load() throws IOException, ConfigurationException {
        this.loadMap();
        this.mapBufferEntries();
    }

    private void loadMap() throws ConfigurationException {
        File mapFile = new File(this.segmentDirectory, INDEX_MAP_FILE);
        PropertiesConfiguration mapConfig = CommonsConfigurationUtils.fromFile((File)mapFile);
        for (String string : CommonsConfigurationUtils.getKeys((Configuration)mapConfig)) {
            int lastSeparatorPos = string.lastIndexOf(MAP_KEY_SEPARATOR);
            Preconditions.checkState((lastSeparatorPos != -1 ? 1 : 0) != 0, (Object)("Key separator not found: " + string + ", segment: " + this.segmentDirectory));
            String propertyName = string.substring(lastSeparatorPos + 1);
            int indexSeparatorPos = string.lastIndexOf(MAP_KEY_SEPARATOR, lastSeparatorPos - 1);
            Preconditions.checkState((indexSeparatorPos != -1 ? 1 : 0) != 0, (Object)("Index separator not found: " + string + " , segment: " + this.segmentDirectory));
            String indexName = string.substring(indexSeparatorPos + 1, lastSeparatorPos);
            String columnName = string.substring(0, indexSeparatorPos);
            IndexKey indexKey = new IndexKey(columnName, ColumnIndexType.getValue(indexName));
            IndexEntry entry = this.columnEntries.get(indexKey);
            if (entry == null) {
                entry = new IndexEntry(indexKey);
                this.columnEntries.put(indexKey, entry);
            }
            if (propertyName.equals(MAP_KEY_NAME_START_OFFSET)) {
                entry.startOffset = mapConfig.getLong(string);
                continue;
            }
            if (propertyName.equals(MAP_KEY_NAME_SIZE)) {
                entry.size = mapConfig.getLong(string);
                continue;
            }
            throw new ConfigurationException("Invalid map file key: " + string + ", segmentDirectory: " + this.segmentDirectory.toString());
        }
        for (Map.Entry entry : this.columnEntries.entrySet()) {
            IndexEntry entry2 = (IndexEntry)entry.getValue();
            if (entry2.size >= 0L && entry2.startOffset >= 0L) continue;
            throw new ConfigurationException("Invalid map entry for key: " + ((IndexKey)entry.getKey()).toString() + ", segment: " + this.segmentDirectory.toString());
        }
    }

    private void mapBufferEntries() throws IOException {
        TreeMap<Long, IndexEntry> indexStartMap = new TreeMap<Long, IndexEntry>();
        for (Map.Entry<IndexKey, IndexEntry> columnEntry : this.columnEntries.entrySet()) {
            long startOffset = columnEntry.getValue().startOffset;
            indexStartMap.put(startOffset, columnEntry.getValue());
        }
        long runningSize = 0L;
        ArrayList<Long> offsetAccum = new ArrayList<Long>();
        for (Map.Entry offsetEntry : indexStartMap.entrySet()) {
            IndexEntry entry = (IndexEntry)offsetEntry.getValue();
            if ((runningSize += entry.size) >= 0x7D000000L) {
                this.mapAndSliceFile(indexStartMap, offsetAccum, (Long)offsetEntry.getKey());
                runningSize = entry.size;
                offsetAccum.clear();
            }
            offsetAccum.add((Long)offsetEntry.getKey());
        }
        if (offsetAccum.size() > 0) {
            this.mapAndSliceFile(indexStartMap, offsetAccum, (Long)offsetAccum.get(0) + runningSize);
        }
    }

    private void mapAndSliceFile(SortedMap<Long, IndexEntry> startOffsets, List<Long> offsetAccum, long endOffset) throws IOException {
        Preconditions.checkNotNull(startOffsets);
        Preconditions.checkNotNull(offsetAccum);
        Preconditions.checkArgument((offsetAccum.size() >= 1 ? 1 : 0) != 0);
        long fromFilePos = offsetAccum.get(0);
        long size = endOffset - fromFilePos;
        String context = this.allocationContext(this.indexFile, "single_file_index.rw.." + String.valueOf(fromFilePos) + MAP_KEY_SEPARATOR + String.valueOf(size));
        PinotDataBuffer buffer = this.readMode == ReadMode.heap ? PinotDataBuffer.loadFile(this.indexFile, fromFilePos, size, ByteOrder.BIG_ENDIAN, context) : PinotDataBuffer.mapFile(this.indexFile, true, fromFilePos, size, ByteOrder.BIG_ENDIAN, context);
        this.allocBuffers.add(buffer);
        int prevSlicePoint = 0;
        for (Long fileOffset : offsetAccum) {
            PinotDataBuffer viewBuffer;
            IndexEntry entry = (IndexEntry)startOffsets.get(fileOffset);
            int endSlicePoint = prevSlicePoint + (int)entry.size;
            this.validateMagicMarker(buffer, prevSlicePoint);
            entry.buffer = viewBuffer = buffer.view(prevSlicePoint + 8, endSlicePoint);
            prevSlicePoint = endSlicePoint;
        }
    }

    private void persistIndexMap(IndexEntry entry) throws IOException {
        File mapFile = new File(this.segmentDirectory, INDEX_MAP_FILE);
        try (PrintWriter writer = new PrintWriter(new BufferedWriter(new FileWriter(mapFile, true)));){
            String startKey = this.getKey(entry.key.name, entry.key.type.getIndexName(), true);
            StringBuilder sb = new StringBuilder();
            sb.append(startKey).append(" = ").append(entry.startOffset);
            writer.println(sb.toString());
            String endKey = this.getKey(entry.key.name, entry.key.type.getIndexName(), false);
            sb = new StringBuilder();
            sb.append(endKey).append(" = ").append(entry.size);
            writer.println(sb.toString());
        }
    }

    private String getKey(String column, String indexName, boolean isStartOffset) {
        return column + MAP_KEY_SEPARATOR + indexName + MAP_KEY_SEPARATOR + (isStartOffset ? MAP_KEY_NAME_START_OFFSET : MAP_KEY_NAME_SIZE);
    }

    private String allocationContext(IndexKey key) {
        return this.getClass().getSimpleName() + key.toString();
    }

    @Override
    public void close() throws IOException {
        for (PinotDataBuffer buf : this.allocBuffers) {
            buf.close();
        }
        this.columnEntries.clear();
        this.allocBuffers.clear();
    }

    @Override
    public void removeIndex(String columnName, ColumnIndexType indexType) {
        throw new UnsupportedOperationException("Index removal is not supported for single file index format. Requested colum: " + columnName + " indexType: " + (Object)((Object)indexType));
    }

    @Override
    public boolean isIndexRemovalSupported() {
        return false;
    }

    public String toString() {
        return this.segmentDirectory.toString() + "/" + this.indexFile.toString();
    }
}

