/*
 * Decompiled with CFR 0.152.
 */
package org.apache.pinot.core.segment.store;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import java.io.File;
import java.io.IOException;
import java.nio.ByteOrder;
import java.util.HashMap;
import java.util.Map;
import org.apache.pinot.common.segment.ReadMode;
import org.apache.pinot.core.segment.index.metadata.SegmentMetadataImpl;
import org.apache.pinot.core.segment.memory.PinotDataBuffer;
import org.apache.pinot.core.segment.store.ColumnIndexDirectory;
import org.apache.pinot.core.segment.store.ColumnIndexType;
import org.apache.pinot.core.segment.store.IndexKey;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

class FilePerIndexDirectory
extends ColumnIndexDirectory {
    private static Logger LOGGER = LoggerFactory.getLogger(FilePerIndexDirectory.class);
    private Map<IndexKey, PinotDataBuffer> indexBuffers = new HashMap<IndexKey, PinotDataBuffer>();

    protected FilePerIndexDirectory(File segmentDirectory, SegmentMetadataImpl metadata, ReadMode readMode) {
        super(segmentDirectory, metadata, readMode);
    }

    @Override
    public PinotDataBuffer getBuffer(String column, ColumnIndexType type) throws IOException {
        IndexKey key = new IndexKey(column, type);
        return this.getReadBufferFor(key);
    }

    @Override
    public PinotDataBuffer newBuffer(String column, ColumnIndexType type, long sizeBytes) throws IOException {
        IndexKey key = new IndexKey(column, type);
        return this.getWriteBufferFor(key, sizeBytes);
    }

    @Override
    public boolean hasIndexFor(String column, ColumnIndexType type) {
        File indexFile = this.getFileFor(column, type);
        return indexFile.exists();
    }

    @Override
    public void close() throws IOException {
        for (PinotDataBuffer dataBuffer : this.indexBuffers.values()) {
            dataBuffer.close();
        }
    }

    @Override
    public void removeIndex(String columnName, ColumnIndexType indexType) {
        File indexFile = this.getFileFor(columnName, indexType);
        indexFile.delete();
    }

    @Override
    public boolean isIndexRemovalSupported() {
        return true;
    }

    private PinotDataBuffer getReadBufferFor(IndexKey key) throws IOException {
        if (this.indexBuffers.containsKey(key)) {
            return this.indexBuffers.get(key);
        }
        File file = this.getFileFor(key.name, key.type);
        if (!file.exists()) {
            throw new RuntimeException("Could not find index for column: " + key.name + ", type: " + (Object)((Object)key.type) + ", segment: " + this.segmentDirectory.toString());
        }
        PinotDataBuffer buffer = this.mapForReads(file, key.type.toString() + ".reader");
        this.indexBuffers.put(key, buffer);
        return buffer;
    }

    private PinotDataBuffer getWriteBufferFor(IndexKey key, long sizeBytes) throws IOException {
        if (this.indexBuffers.containsKey(key)) {
            return this.indexBuffers.get(key);
        }
        File filename = this.getFileFor(key.name, key.type);
        PinotDataBuffer buffer = this.mapForWrites(filename, sizeBytes, key.type.toString() + ".writer");
        this.indexBuffers.put(key, buffer);
        return buffer;
    }

    @VisibleForTesting
    File getFileFor(String column, ColumnIndexType indexType) {
        String filename;
        switch (indexType) {
            case DICTIONARY: {
                filename = this.metadata.getDictionaryFileName(column);
                break;
            }
            case FORWARD_INDEX: {
                filename = this.metadata.getForwardIndexFileName(column);
                break;
            }
            case INVERTED_INDEX: {
                filename = this.metadata.getBitmapInvertedIndexFileName(column);
                break;
            }
            case RANGE_INDEX: {
                filename = this.metadata.getBitmapRangeIndexFileName(column);
                break;
            }
            case BLOOM_FILTER: {
                filename = this.metadata.getBloomFilterFileName(column);
                break;
            }
            case NULLVALUE_VECTOR: {
                filename = this.metadata.getNullValueVectorFileName(column);
                break;
            }
            case TEXT_INDEX: {
                filename = column + ".lucene.index";
                break;
            }
            default: {
                throw new UnsupportedOperationException("Unknown index type: " + indexType.toString());
            }
        }
        return new File(this.segmentDirectory, filename);
    }

    private PinotDataBuffer mapForWrites(File file, long sizeBytes, String context) throws IOException {
        Preconditions.checkNotNull((Object)file);
        Preconditions.checkArgument((sizeBytes >= 0L && sizeBytes < Integer.MAX_VALUE ? 1 : 0) != 0, (Object)("File size must be less than 2GB, file: " + file));
        Preconditions.checkState((!file.exists() ? 1 : 0) != 0, (Object)("File: " + file + " already exists"));
        String allocationContext = this.allocationContext(file, context);
        return PinotDataBuffer.mapFile(file, false, 0L, sizeBytes, ByteOrder.BIG_ENDIAN, allocationContext);
    }

    private PinotDataBuffer mapForReads(File file, String context) throws IOException {
        Preconditions.checkNotNull((Object)file);
        Preconditions.checkNotNull((Object)context);
        Preconditions.checkArgument((boolean)file.exists(), (Object)("File: " + file + " must exist"));
        Preconditions.checkArgument((boolean)file.isFile(), (Object)("File: " + file + " must be a regular file"));
        String allocationContext = this.allocationContext(file, context);
        if (this.readMode == ReadMode.heap) {
            return PinotDataBuffer.loadFile(file, 0L, file.length(), ByteOrder.BIG_ENDIAN, allocationContext);
        }
        return PinotDataBuffer.mapFile(file, true, 0L, file.length(), ByteOrder.BIG_ENDIAN, allocationContext);
    }
}

